/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.ItemCollectionMetricsMarshaller;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about item collections, if any, that were affected by the operation. <code>ItemCollectionMetrics</code>
 * is only returned if the request asked for it. If the table does not have any local secondary indexes, this
 * information is not returned in the response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ItemCollectionMetrics implements StructuredPojo,
        ToCopyableBuilder<ItemCollectionMetrics.Builder, ItemCollectionMetrics> {
    private final Map<String, AttributeValue> itemCollectionKey;

    private final List<Double> sizeEstimateRangeGB;

    private ItemCollectionMetrics(BuilderImpl builder) {
        this.itemCollectionKey = builder.itemCollectionKey;
        this.sizeEstimateRangeGB = builder.sizeEstimateRangeGB;
    }

    /**
     * <p>
     * The partition key value of the item collection. This value is the same as the partition key value of the item.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The partition key value of the item collection. This value is the same as the partition key value of the
     *         item.
     */
    public Map<String, AttributeValue> itemCollectionKey() {
        return itemCollectionKey;
    }

    /**
     * <p>
     * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound and
     * an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the size of
     * all attributes projected into all of the local secondary indexes on that table. Use this estimate to measure
     * whether a local secondary index is approaching its size limit.
     * </p>
     * <p>
     * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the estimate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower
     *         bound and an upper bound for the estimate. The estimate includes the size of all the items in the table,
     *         plus the size of all attributes projected into all of the local secondary indexes on that table. Use this
     *         estimate to measure whether a local secondary index is approaching its size limit.</p>
     *         <p>
     *         The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
     *         estimate.
     */
    public List<Double> sizeEstimateRangeGB() {
        return sizeEstimateRangeGB;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(itemCollectionKey());
        hashCode = 31 * hashCode + Objects.hashCode(sizeEstimateRangeGB());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ItemCollectionMetrics)) {
            return false;
        }
        ItemCollectionMetrics other = (ItemCollectionMetrics) obj;
        return Objects.equals(itemCollectionKey(), other.itemCollectionKey())
                && Objects.equals(sizeEstimateRangeGB(), other.sizeEstimateRangeGB());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (itemCollectionKey() != null) {
            sb.append("ItemCollectionKey: ").append(itemCollectionKey()).append(",");
        }
        if (sizeEstimateRangeGB() != null) {
            sb.append("SizeEstimateRangeGB: ").append(sizeEstimateRangeGB()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ItemCollectionKey":
            return Optional.of(clazz.cast(itemCollectionKey()));
        case "SizeEstimateRangeGB":
            return Optional.of(clazz.cast(sizeEstimateRangeGB()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        ItemCollectionMetricsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, ItemCollectionMetrics> {
        /**
         * <p>
         * The partition key value of the item collection. This value is the same as the partition key value of the
         * item.
         * </p>
         * 
         * @param itemCollectionKey
         *        The partition key value of the item collection. This value is the same as the partition key value of
         *        the item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCollectionKey(Map<String, AttributeValue> itemCollectionKey);

        /**
         * <p>
         * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound
         * and an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the
         * size of all attributes projected into all of the local secondary indexes on that table. Use this estimate to
         * measure whether a local secondary index is approaching its size limit.
         * </p>
         * <p>
         * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
         * estimate.
         * </p>
         * 
         * @param sizeEstimateRangeGB
         *        An estimate of item collection size, in gigabytes. This value is a two-element array containing a
         *        lower bound and an upper bound for the estimate. The estimate includes the size of all the items in
         *        the table, plus the size of all attributes projected into all of the local secondary indexes on that
         *        table. Use this estimate to measure whether a local secondary index is approaching its size limit.</p>
         *        <p>
         *        The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of
         *        the estimate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB);

        /**
         * <p>
         * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound
         * and an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the
         * size of all attributes projected into all of the local secondary indexes on that table. Use this estimate to
         * measure whether a local secondary index is approaching its size limit.
         * </p>
         * <p>
         * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
         * estimate.
         * </p>
         * 
         * @param sizeEstimateRangeGB
         *        An estimate of item collection size, in gigabytes. This value is a two-element array containing a
         *        lower bound and an upper bound for the estimate. The estimate includes the size of all the items in
         *        the table, plus the size of all attributes projected into all of the local secondary indexes on that
         *        table. Use this estimate to measure whether a local secondary index is approaching its size limit.</p>
         *        <p>
         *        The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of
         *        the estimate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeEstimateRangeGB(Double... sizeEstimateRangeGB);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, AttributeValue> itemCollectionKey;

        private List<Double> sizeEstimateRangeGB;

        private BuilderImpl() {
        }

        private BuilderImpl(ItemCollectionMetrics model) {
            itemCollectionKey(model.itemCollectionKey);
            sizeEstimateRangeGB(model.sizeEstimateRangeGB);
        }

        public final Map<String, AttributeValue.Builder> getItemCollectionKey() {
            return itemCollectionKey != null ? CollectionUtils.mapValues(itemCollectionKey, AttributeValue::toBuilder) : null;
        }

        @Override
        public final Builder itemCollectionKey(Map<String, AttributeValue> itemCollectionKey) {
            this.itemCollectionKey = ItemCollectionKeyAttributeMapCopier.copy(itemCollectionKey);
            return this;
        }

        public final void setItemCollectionKey(Map<String, AttributeValue.BuilderImpl> itemCollectionKey) {
            this.itemCollectionKey = ItemCollectionKeyAttributeMapCopier.copyFromBuilder(itemCollectionKey);
        }

        public final Collection<Double> getSizeEstimateRangeGB() {
            return sizeEstimateRangeGB;
        }

        @Override
        public final Builder sizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB) {
            this.sizeEstimateRangeGB = ItemCollectionSizeEstimateRangeCopier.copy(sizeEstimateRangeGB);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sizeEstimateRangeGB(Double... sizeEstimateRangeGB) {
            sizeEstimateRangeGB(Arrays.asList(sizeEstimateRangeGB));
            return this;
        }

        public final void setSizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB) {
            this.sizeEstimateRangeGB = ItemCollectionSizeEstimateRangeCopier.copy(sizeEstimateRangeGB);
        }

        @Override
        public ItemCollectionMetrics build() {
            return new ItemCollectionMetrics(this);
        }
    }
}
