/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.services.dynamodb.transform.CreateGlobalSecondaryIndexActionMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a new global secondary index to be added to an existing table.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class CreateGlobalSecondaryIndexAction implements StructuredPojo,
        ToCopyableBuilder<CreateGlobalSecondaryIndexAction.Builder, CreateGlobalSecondaryIndexAction> {
    private final String indexName;

    private final List<KeySchemaElement> keySchema;

    private final Projection projection;

    private final ProvisionedThroughput provisionedThroughput;

    private CreateGlobalSecondaryIndexAction(BuilderImpl builder) {
        this.indexName = builder.indexName;
        this.keySchema = builder.keySchema;
        this.projection = builder.projection;
        this.provisionedThroughput = builder.provisionedThroughput;
    }

    /**
     * <p>
     * The name of the global secondary index to be created.
     * </p>
     * 
     * @return The name of the global secondary index to be created.
     */
    public String indexName() {
        return indexName;
    }

    /**
     * <p>
     * The key schema for the global secondary index.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The key schema for the global secondary index.
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * Represents attributes that are copied (projected) from the table into an index. These are in addition to the
     * primary key attributes and index key attributes, which are automatically projected.
     * </p>
     * 
     * @return Represents attributes that are copied (projected) from the table into an index. These are in addition to
     *         the primary key attributes and index key attributes, which are automatically projected.
     */
    public Projection projection() {
        return projection;
    }

    /**
     * <p>
     * Represents the provisioned throughput settings for the specified global secondary index.
     * </p>
     * <p>
     * For current minimum and maximum provisioned throughput values, see <a
     * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the <i>Amazon
     * DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return Represents the provisioned throughput settings for the specified global secondary index.</p>
     *         <p>
     *         For current minimum and maximum provisioned throughput values, see <a
     *         href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
     *         <i>Amazon DynamoDB Developer Guide</i>.
     */
    public ProvisionedThroughput provisionedThroughput() {
        return provisionedThroughput;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(indexName());
        hashCode = 31 * hashCode + Objects.hashCode(keySchema());
        hashCode = 31 * hashCode + Objects.hashCode(projection());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughput());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateGlobalSecondaryIndexAction)) {
            return false;
        }
        CreateGlobalSecondaryIndexAction other = (CreateGlobalSecondaryIndexAction) obj;
        return Objects.equals(indexName(), other.indexName()) && Objects.equals(keySchema(), other.keySchema())
                && Objects.equals(projection(), other.projection())
                && Objects.equals(provisionedThroughput(), other.provisionedThroughput());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateGlobalSecondaryIndexAction").add("IndexName", indexName()).add("KeySchema", keySchema())
                .add("Projection", projection()).add("ProvisionedThroughput", provisionedThroughput()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IndexName":
            return Optional.of(clazz.cast(indexName()));
        case "KeySchema":
            return Optional.of(clazz.cast(keySchema()));
        case "Projection":
            return Optional.of(clazz.cast(projection()));
        case "ProvisionedThroughput":
            return Optional.of(clazz.cast(provisionedThroughput()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        CreateGlobalSecondaryIndexActionMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, CreateGlobalSecondaryIndexAction> {
        /**
         * <p>
         * The name of the global secondary index to be created.
         * </p>
         * 
         * @param indexName
         *        The name of the global secondary index to be created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexName(String indexName);

        /**
         * <p>
         * The key schema for the global secondary index.
         * </p>
         * 
         * @param keySchema
         *        The key schema for the global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The key schema for the global secondary index.
         * </p>
         * 
         * @param keySchema
         *        The key schema for the global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into an index. These are in addition to the
         * primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * 
         * @param projection
         *        Represents attributes that are copied (projected) from the table into an index. These are in addition
         *        to the primary key attributes and index key attributes, which are automatically projected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projection(Projection projection);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into an index. These are in addition to the
         * primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * This is a convenience that creates an instance of the {@link Projection.Builder} avoiding the need to create
         * one manually via {@link Projection#builder()}.
         *
         * When the {@link Consumer} completes, {@link Projection.Builder#build()} is called immediately and its result
         * is passed to {@link #projection(Projection)}.
         * 
         * @param projection
         *        a consumer that will call methods on {@link Projection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projection(Projection)
         */
        default Builder projection(Consumer<Projection.Builder> projection) {
            return projection(Projection.builder().apply(projection).build());
        }

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * <p>
         * For current minimum and maximum provisioned throughput values, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         * <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param provisionedThroughput
         *        Represents the provisioned throughput settings for the specified global secondary index.</p>
         *        <p>
         *        For current minimum and maximum provisioned throughput values, see <a
         *        href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         *        <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput);

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * <p>
         * For current minimum and maximum provisioned throughput values, see <a
         * href="http://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Limits</a> in the
         * <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link ProvisionedThroughput.Builder} avoiding the need
         * to create one manually via {@link ProvisionedThroughput#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedThroughput.Builder#build()} is called immediately and
         * its result is passed to {@link #provisionedThroughput(ProvisionedThroughput)}.
         * 
         * @param provisionedThroughput
         *        a consumer that will call methods on {@link ProvisionedThroughput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughput(ProvisionedThroughput)
         */
        default Builder provisionedThroughput(Consumer<ProvisionedThroughput.Builder> provisionedThroughput) {
            return provisionedThroughput(ProvisionedThroughput.builder().apply(provisionedThroughput).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String indexName;

        private List<KeySchemaElement> keySchema;

        private Projection projection;

        private ProvisionedThroughput provisionedThroughput;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateGlobalSecondaryIndexAction model) {
            indexName(model.indexName);
            keySchema(model.keySchema);
            projection(model.projection);
            provisionedThroughput(model.provisionedThroughput);
        }

        public final String getIndexName() {
            return indexName;
        }

        @Override
        public final Builder indexName(String indexName) {
            this.indexName = indexName;
            return this;
        }

        public final void setIndexName(String indexName) {
            this.indexName = indexName;
        }

        public final Collection<KeySchemaElement.Builder> getKeySchema() {
            return keySchema != null ? keySchema.stream().map(KeySchemaElement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        public final Projection.Builder getProjection() {
            return projection != null ? projection.toBuilder() : null;
        }

        @Override
        public final Builder projection(Projection projection) {
            this.projection = projection;
            return this;
        }

        public final void setProjection(Projection.BuilderImpl projection) {
            this.projection = projection != null ? projection.build() : null;
        }

        public final ProvisionedThroughput.Builder getProvisionedThroughput() {
            return provisionedThroughput != null ? provisionedThroughput.toBuilder() : null;
        }

        @Override
        public final Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        public final void setProvisionedThroughput(ProvisionedThroughput.BuilderImpl provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput != null ? provisionedThroughput.build() : null;
        }

        @Override
        public CreateGlobalSecondaryIndexAction build() {
            return new CreateGlobalSecondaryIndexAction(this);
        }
    }
}
