/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.core.protocol.ProtocolMarshaller;
import software.amazon.awssdk.core.protocol.StructuredPojo;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.services.dynamodb.transform.SourceTableDetailsMarshaller;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of the table when the backup was created.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SourceTableDetails implements StructuredPojo,
        ToCopyableBuilder<SourceTableDetails.Builder, SourceTableDetails> {
    private final String tableName;

    private final String tableId;

    private final String tableArn;

    private final Long tableSizeBytes;

    private final List<KeySchemaElement> keySchema;

    private final Instant tableCreationDateTime;

    private final ProvisionedThroughput provisionedThroughput;

    private final Long itemCount;

    private SourceTableDetails(BuilderImpl builder) {
        this.tableName = builder.tableName;
        this.tableId = builder.tableId;
        this.tableArn = builder.tableArn;
        this.tableSizeBytes = builder.tableSizeBytes;
        this.keySchema = builder.keySchema;
        this.tableCreationDateTime = builder.tableCreationDateTime;
        this.provisionedThroughput = builder.provisionedThroughput;
        this.itemCount = builder.itemCount;
    }

    /**
     * <p>
     * The name of the table for which the backup was created.
     * </p>
     * 
     * @return The name of the table for which the backup was created.
     */
    public String tableName() {
        return tableName;
    }

    /**
     * <p>
     * Unique identifier for the table for which the backup was created.
     * </p>
     * 
     * @return Unique identifier for the table for which the backup was created.
     */
    public String tableId() {
        return tableId;
    }

    /**
     * <p>
     * ARN of the table for which backup was created.
     * </p>
     * 
     * @return ARN of the table for which backup was created.
     */
    public String tableArn() {
        return tableArn;
    }

    /**
     * <p>
     * Size of the table in bytes. Please note this is an approximate value.
     * </p>
     * 
     * @return Size of the table in bytes. Please note this is an approximate value.
     */
    public Long tableSizeBytes() {
        return tableSizeBytes;
    }

    /**
     * <p>
     * Schema of the table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Schema of the table.
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * Time when the source table was created.
     * </p>
     * 
     * @return Time when the source table was created.
     */
    public Instant tableCreationDateTime() {
        return tableCreationDateTime;
    }

    /**
     * <p>
     * Read IOPs and Write IOPS on the table when the backup was created.
     * </p>
     * 
     * @return Read IOPs and Write IOPS on the table when the backup was created.
     */
    public ProvisionedThroughput provisionedThroughput() {
        return provisionedThroughput;
    }

    /**
     * <p>
     * Number of items in the table. Please note this is an approximate value.
     * </p>
     * 
     * @return Number of items in the table. Please note this is an approximate value.
     */
    public Long itemCount() {
        return itemCount;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(tableId());
        hashCode = 31 * hashCode + Objects.hashCode(tableArn());
        hashCode = 31 * hashCode + Objects.hashCode(tableSizeBytes());
        hashCode = 31 * hashCode + Objects.hashCode(keySchema());
        hashCode = 31 * hashCode + Objects.hashCode(tableCreationDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SourceTableDetails)) {
            return false;
        }
        SourceTableDetails other = (SourceTableDetails) obj;
        return Objects.equals(tableName(), other.tableName()) && Objects.equals(tableId(), other.tableId())
                && Objects.equals(tableArn(), other.tableArn()) && Objects.equals(tableSizeBytes(), other.tableSizeBytes())
                && Objects.equals(keySchema(), other.keySchema())
                && Objects.equals(tableCreationDateTime(), other.tableCreationDateTime())
                && Objects.equals(provisionedThroughput(), other.provisionedThroughput())
                && Objects.equals(itemCount(), other.itemCount());
    }

    @Override
    public String toString() {
        return ToString.builder("SourceTableDetails").add("TableName", tableName()).add("TableId", tableId())
                .add("TableArn", tableArn()).add("TableSizeBytes", tableSizeBytes()).add("KeySchema", keySchema())
                .add("TableCreationDateTime", tableCreationDateTime()).add("ProvisionedThroughput", provisionedThroughput())
                .add("ItemCount", itemCount()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "TableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "TableId":
            return Optional.ofNullable(clazz.cast(tableId()));
        case "TableArn":
            return Optional.ofNullable(clazz.cast(tableArn()));
        case "TableSizeBytes":
            return Optional.ofNullable(clazz.cast(tableSizeBytes()));
        case "KeySchema":
            return Optional.ofNullable(clazz.cast(keySchema()));
        case "TableCreationDateTime":
            return Optional.ofNullable(clazz.cast(tableCreationDateTime()));
        case "ProvisionedThroughput":
            return Optional.ofNullable(clazz.cast(provisionedThroughput()));
        case "ItemCount":
            return Optional.ofNullable(clazz.cast(itemCount()));
        default:
            return Optional.empty();
        }
    }

    @SdkInternalApi
    @Override
    public void marshall(ProtocolMarshaller protocolMarshaller) {
        SourceTableDetailsMarshaller.getInstance().marshall(this, protocolMarshaller);
    }

    public interface Builder extends CopyableBuilder<Builder, SourceTableDetails> {
        /**
         * <p>
         * The name of the table for which the backup was created.
         * </p>
         * 
         * @param tableName
         *        The name of the table for which the backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * Unique identifier for the table for which the backup was created.
         * </p>
         * 
         * @param tableId
         *        Unique identifier for the table for which the backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableId(String tableId);

        /**
         * <p>
         * ARN of the table for which backup was created.
         * </p>
         * 
         * @param tableArn
         *        ARN of the table for which backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableArn(String tableArn);

        /**
         * <p>
         * Size of the table in bytes. Please note this is an approximate value.
         * </p>
         * 
         * @param tableSizeBytes
         *        Size of the table in bytes. Please note this is an approximate value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableSizeBytes(Long tableSizeBytes);

        /**
         * <p>
         * Schema of the table.
         * </p>
         * 
         * @param keySchema
         *        Schema of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * Schema of the table.
         * </p>
         * 
         * @param keySchema
         *        Schema of the table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * Schema of the table.
         * </p>
         * This is a convenience that creates an instance of the {@link List<KeySchemaElement>.Builder} avoiding the
         * need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * Time when the source table was created.
         * </p>
         * 
         * @param tableCreationDateTime
         *        Time when the source table was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableCreationDateTime(Instant tableCreationDateTime);

        /**
         * <p>
         * Read IOPs and Write IOPS on the table when the backup was created.
         * </p>
         * 
         * @param provisionedThroughput
         *        Read IOPs and Write IOPS on the table when the backup was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput);

        /**
         * <p>
         * Read IOPs and Write IOPS on the table when the backup was created.
         * </p>
         * This is a convenience that creates an instance of the {@link ProvisionedThroughput.Builder} avoiding the need
         * to create one manually via {@link ProvisionedThroughput#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedThroughput.Builder#build()} is called immediately and
         * its result is passed to {@link #provisionedThroughput(ProvisionedThroughput)}.
         * 
         * @param provisionedThroughput
         *        a consumer that will call methods on {@link ProvisionedThroughput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughput(ProvisionedThroughput)
         */
        default Builder provisionedThroughput(Consumer<ProvisionedThroughput.Builder> provisionedThroughput) {
            return provisionedThroughput(ProvisionedThroughput.builder().applyMutation(provisionedThroughput).build());
        }

        /**
         * <p>
         * Number of items in the table. Please note this is an approximate value.
         * </p>
         * 
         * @param itemCount
         *        Number of items in the table. Please note this is an approximate value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Long itemCount);
    }

    static final class BuilderImpl implements Builder {
        private String tableName;

        private String tableId;

        private String tableArn;

        private Long tableSizeBytes;

        private List<KeySchemaElement> keySchema = DefaultSdkAutoConstructList.getInstance();

        private Instant tableCreationDateTime;

        private ProvisionedThroughput provisionedThroughput;

        private Long itemCount;

        private BuilderImpl() {
        }

        private BuilderImpl(SourceTableDetails model) {
            tableName(model.tableName);
            tableId(model.tableId);
            tableArn(model.tableArn);
            tableSizeBytes(model.tableSizeBytes);
            keySchema(model.keySchema);
            tableCreationDateTime(model.tableCreationDateTime);
            provisionedThroughput(model.provisionedThroughput);
            itemCount(model.itemCount);
        }

        public final String getTableName() {
            return tableName;
        }

        @Override
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        public final String getTableId() {
            return tableId;
        }

        @Override
        public final Builder tableId(String tableId) {
            this.tableId = tableId;
            return this;
        }

        public final void setTableId(String tableId) {
            this.tableId = tableId;
        }

        public final String getTableArn() {
            return tableArn;
        }

        @Override
        public final Builder tableArn(String tableArn) {
            this.tableArn = tableArn;
            return this;
        }

        public final void setTableArn(String tableArn) {
            this.tableArn = tableArn;
        }

        public final Long getTableSizeBytes() {
            return tableSizeBytes;
        }

        @Override
        public final Builder tableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
            return this;
        }

        public final void setTableSizeBytes(Long tableSizeBytes) {
            this.tableSizeBytes = tableSizeBytes;
        }

        public final Collection<KeySchemaElement.Builder> getKeySchema() {
            return keySchema != null ? keySchema.stream().map(KeySchemaElement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        public final Instant getTableCreationDateTime() {
            return tableCreationDateTime;
        }

        @Override
        public final Builder tableCreationDateTime(Instant tableCreationDateTime) {
            this.tableCreationDateTime = tableCreationDateTime;
            return this;
        }

        public final void setTableCreationDateTime(Instant tableCreationDateTime) {
            this.tableCreationDateTime = tableCreationDateTime;
        }

        public final ProvisionedThroughput.Builder getProvisionedThroughput() {
            return provisionedThroughput != null ? provisionedThroughput.toBuilder() : null;
        }

        @Override
        public final Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        public final void setProvisionedThroughput(ProvisionedThroughput.BuilderImpl provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput != null ? provisionedThroughput.build() : null;
        }

        public final Long getItemCount() {
            return itemCount;
        }

        @Override
        public final Builder itemCount(Long itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final void setItemCount(Long itemCount) {
            this.itemCount = itemCount;
        }

        @Override
        public SourceTableDetails build() {
            return new SourceTableDetails(this);
        }
    }
}
