/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the properties of a local secondary index.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LocalSecondaryIndex implements SdkPojo, Serializable,
        ToCopyableBuilder<LocalSecondaryIndex.Builder, LocalSecondaryIndex> {
    private static final SdkField<String> INDEX_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(LocalSecondaryIndex::indexName)).setter(setter(Builder::indexName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexName").build()).build();

    private static final SdkField<List<KeySchemaElement>> KEY_SCHEMA_FIELD = SdkField
            .<List<KeySchemaElement>> builder(MarshallingType.LIST)
            .getter(getter(LocalSecondaryIndex::keySchema))
            .setter(setter(Builder::keySchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySchema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeySchemaElement> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeySchemaElement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Projection> PROJECTION_FIELD = SdkField.<Projection> builder(MarshallingType.SDK_POJO)
            .getter(getter(LocalSecondaryIndex::projection)).setter(setter(Builder::projection)).constructor(Projection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Projection").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INDEX_NAME_FIELD,
            KEY_SCHEMA_FIELD, PROJECTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String indexName;

    private final List<KeySchemaElement> keySchema;

    private final Projection projection;

    private LocalSecondaryIndex(BuilderImpl builder) {
        this.indexName = builder.indexName;
        this.keySchema = builder.keySchema;
        this.projection = builder.projection;
    }

    /**
     * <p>
     * The name of the local secondary index. The name must be unique among all other indexes on this table.
     * </p>
     * 
     * @return The name of the local secondary index. The name must be unique among all other indexes on this table.
     */
    public String indexName() {
        return indexName;
    }

    /**
     * Returns true if the KeySchema property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasKeySchema() {
        return keySchema != null && !(keySchema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The complete key schema for the local secondary index, consisting of one or more pairs of attribute names and key
     * types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasKeySchema()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The complete key schema for the local secondary index, consisting of one or more pairs of attribute names
     *         and key types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     */
    public List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * Represents attributes that are copied (projected) from the table into the local secondary index. These are in
     * addition to the primary key attributes and index key attributes, which are automatically projected.
     * </p>
     * 
     * @return Represents attributes that are copied (projected) from the table into the local secondary index. These
     *         are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected.
     */
    public Projection projection() {
        return projection;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(indexName());
        hashCode = 31 * hashCode + Objects.hashCode(keySchema());
        hashCode = 31 * hashCode + Objects.hashCode(projection());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LocalSecondaryIndex)) {
            return false;
        }
        LocalSecondaryIndex other = (LocalSecondaryIndex) obj;
        return Objects.equals(indexName(), other.indexName()) && Objects.equals(keySchema(), other.keySchema())
                && Objects.equals(projection(), other.projection());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LocalSecondaryIndex").add("IndexName", indexName()).add("KeySchema", keySchema())
                .add("Projection", projection()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IndexName":
            return Optional.ofNullable(clazz.cast(indexName()));
        case "KeySchema":
            return Optional.ofNullable(clazz.cast(keySchema()));
        case "Projection":
            return Optional.ofNullable(clazz.cast(projection()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LocalSecondaryIndex, T> g) {
        return obj -> g.apply((LocalSecondaryIndex) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LocalSecondaryIndex> {
        /**
         * <p>
         * The name of the local secondary index. The name must be unique among all other indexes on this table.
         * </p>
         * 
         * @param indexName
         *        The name of the local secondary index. The name must be unique among all other indexes on this table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexName(String indexName);

        /**
         * <p>
         * The complete key schema for the local secondary index, consisting of one or more pairs of attribute names and
         * key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for the local secondary index, consisting of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The complete key schema for the local secondary index, consisting of one or more pairs of attribute names and
         * key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for the local secondary index, consisting of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The complete key schema for the local secondary index, consisting of one or more pairs of attribute names and
         * key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note> This is a convenience that creates an instance of the {@link List<KeySchemaElement>.Builder} avoiding
         * the need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the local secondary index. These are in
         * addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * 
         * @param projection
         *        Represents attributes that are copied (projected) from the table into the local secondary index. These
         *        are in addition to the primary key attributes and index key attributes, which are automatically
         *        projected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projection(Projection projection);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the local secondary index. These are in
         * addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * This is a convenience that creates an instance of the {@link Projection.Builder} avoiding the need to create
         * one manually via {@link Projection#builder()}.
         *
         * When the {@link Consumer} completes, {@link Projection.Builder#build()} is called immediately and its result
         * is passed to {@link #projection(Projection)}.
         * 
         * @param projection
         *        a consumer that will call methods on {@link Projection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projection(Projection)
         */
        default Builder projection(Consumer<Projection.Builder> projection) {
            return projection(Projection.builder().applyMutation(projection).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String indexName;

        private List<KeySchemaElement> keySchema = DefaultSdkAutoConstructList.getInstance();

        private Projection projection;

        private BuilderImpl() {
        }

        private BuilderImpl(LocalSecondaryIndex model) {
            indexName(model.indexName);
            keySchema(model.keySchema);
            projection(model.projection);
        }

        public final String getIndexName() {
            return indexName;
        }

        @Override
        public final Builder indexName(String indexName) {
            this.indexName = indexName;
            return this;
        }

        public final void setIndexName(String indexName) {
            this.indexName = indexName;
        }

        public final Collection<KeySchemaElement.Builder> getKeySchema() {
            return keySchema != null ? keySchema.stream().map(KeySchemaElement::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        public final Projection.Builder getProjection() {
            return projection != null ? projection.toBuilder() : null;
        }

        @Override
        public final Builder projection(Projection projection) {
            this.projection = projection;
            return this;
        }

        public final void setProjection(Projection.BuilderImpl projection) {
            this.projection = projection != null ? projection.build() : null;
        }

        @Override
        public LocalSecondaryIndex build() {
            return new LocalSecondaryIndex(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
