/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents attributes that are copied (projected) from the table into an index. These are in addition to the primary
 * key attributes and index key attributes, which are automatically projected.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Projection implements SdkPojo, Serializable, ToCopyableBuilder<Projection.Builder, Projection> {
    private static final SdkField<String> PROJECTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ProjectionType").getter(getter(Projection::projectionTypeAsString))
            .setter(setter(Builder::projectionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProjectionType").build()).build();

    private static final SdkField<List<String>> NON_KEY_ATTRIBUTES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NonKeyAttributes")
            .getter(getter(Projection::nonKeyAttributes))
            .setter(setter(Builder::nonKeyAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NonKeyAttributes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PROJECTION_TYPE_FIELD,
            NON_KEY_ATTRIBUTES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String projectionType;

    private final List<String> nonKeyAttributes;

    private Projection(BuilderImpl builder) {
        this.projectionType = builder.projectionType;
        this.nonKeyAttributes = builder.nonKeyAttributes;
    }

    /**
     * <p>
     * The set of attributes that are projected into the index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes is in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #projectionType}
     * will return {@link ProjectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #projectionTypeAsString}.
     * </p>
     * 
     * @return The set of attributes that are projected into the index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes is in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     * @see ProjectionType
     */
    public ProjectionType projectionType() {
        return ProjectionType.fromValue(projectionType);
    }

    /**
     * <p>
     * The set of attributes that are projected into the index:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of projected
     * attributes is in <code>NonKeyAttributes</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ALL</code> - All of the table attributes are projected into the index.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #projectionType}
     * will return {@link ProjectionType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #projectionTypeAsString}.
     * </p>
     * 
     * @return The set of attributes that are projected into the index:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
     *         projected attributes is in <code>NonKeyAttributes</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ALL</code> - All of the table attributes are projected into the index.
     *         </p>
     *         </li>
     * @see ProjectionType
     */
    public String projectionTypeAsString() {
        return projectionType;
    }

    /**
     * Returns true if the NonKeyAttributes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasNonKeyAttributes() {
        return nonKeyAttributes != null && !(nonKeyAttributes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents the non-key attribute names which will be projected into the index.
     * </p>
     * <p>
     * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
     * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this counts
     * as two distinct attributes when determining the total.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNonKeyAttributes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Represents the non-key attribute names which will be projected into the index.</p>
     *         <p>
     *         For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
     *         local secondary indexes, must not exceed 20. If you project the same attribute into two different
     *         indexes, this counts as two distinct attributes when determining the total.
     */
    public List<String> nonKeyAttributes() {
        return nonKeyAttributes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(projectionTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(nonKeyAttributes());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Projection)) {
            return false;
        }
        Projection other = (Projection) obj;
        return Objects.equals(projectionTypeAsString(), other.projectionTypeAsString())
                && Objects.equals(nonKeyAttributes(), other.nonKeyAttributes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Projection").add("ProjectionType", projectionTypeAsString())
                .add("NonKeyAttributes", nonKeyAttributes()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ProjectionType":
            return Optional.ofNullable(clazz.cast(projectionTypeAsString()));
        case "NonKeyAttributes":
            return Optional.ofNullable(clazz.cast(nonKeyAttributes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Projection, T> g) {
        return obj -> g.apply((Projection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Projection> {
        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes is in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes is in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(String projectionType);

        /**
         * <p>
         * The set of attributes that are projected into the index:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         * projected attributes is in <code>NonKeyAttributes</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ALL</code> - All of the table attributes are projected into the index.
         * </p>
         * </li>
         * </ul>
         * 
         * @param projectionType
         *        The set of attributes that are projected into the index:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - Only the index and primary keys are projected into the index.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>INCLUDE</code> - Only the specified table attributes are projected into the index. The list of
         *        projected attributes is in <code>NonKeyAttributes</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ALL</code> - All of the table attributes are projected into the index.
         *        </p>
         *        </li>
         * @see ProjectionType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ProjectionType
         */
        Builder projectionType(ProjectionType projectionType);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
         * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
         * counts as two distinct attributes when determining the total.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
         *        local secondary indexes, must not exceed 20. If you project the same attribute into two different
         *        indexes, this counts as two distinct attributes when determining the total.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(Collection<String> nonKeyAttributes);

        /**
         * <p>
         * Represents the non-key attribute names which will be projected into the index.
         * </p>
         * <p>
         * For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the local
         * secondary indexes, must not exceed 20. If you project the same attribute into two different indexes, this
         * counts as two distinct attributes when determining the total.
         * </p>
         * 
         * @param nonKeyAttributes
         *        Represents the non-key attribute names which will be projected into the index.</p>
         *        <p>
         *        For local secondary indexes, the total count of <code>NonKeyAttributes</code> summed across all of the
         *        local secondary indexes, must not exceed 20. If you project the same attribute into two different
         *        indexes, this counts as two distinct attributes when determining the total.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nonKeyAttributes(String... nonKeyAttributes);
    }

    static final class BuilderImpl implements Builder {
        private String projectionType;

        private List<String> nonKeyAttributes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Projection model) {
            projectionType(model.projectionType);
            nonKeyAttributes(model.nonKeyAttributes);
        }

        public final String getProjectionType() {
            return projectionType;
        }

        @Override
        public final Builder projectionType(String projectionType) {
            this.projectionType = projectionType;
            return this;
        }

        @Override
        public final Builder projectionType(ProjectionType projectionType) {
            this.projectionType(projectionType == null ? null : projectionType.toString());
            return this;
        }

        public final void setProjectionType(String projectionType) {
            this.projectionType = projectionType;
        }

        public final Collection<String> getNonKeyAttributes() {
            return nonKeyAttributes;
        }

        @Override
        public final Builder nonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder nonKeyAttributes(String... nonKeyAttributes) {
            nonKeyAttributes(Arrays.asList(nonKeyAttributes));
            return this;
        }

        public final void setNonKeyAttributes(Collection<String> nonKeyAttributes) {
            this.nonKeyAttributes = NonKeyAttributeNameListCopier.copy(nonKeyAttributes);
        }

        @Override
        public Projection build() {
            return new Projection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
