/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the output of a <code>GetRecords</code> operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetRecordsResponse extends DynamoDbStreamsResponse implements
        ToCopyableBuilder<GetRecordsResponse.Builder, GetRecordsResponse> {
    private static final SdkField<List<Record>> RECORDS_FIELD = SdkField
            .<List<Record>> builder(MarshallingType.LIST)
            .memberName("Records")
            .getter(getter(GetRecordsResponse::records))
            .setter(setter(Builder::records))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Records").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Record> builder(MarshallingType.SDK_POJO)
                                            .constructor(Record::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_SHARD_ITERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextShardIterator").getter(getter(GetRecordsResponse::nextShardIterator))
            .setter(setter(Builder::nextShardIterator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextShardIterator").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RECORDS_FIELD,
            NEXT_SHARD_ITERATOR_FIELD));

    private final List<Record> records;

    private final String nextShardIterator;

    private GetRecordsResponse(BuilderImpl builder) {
        super(builder);
        this.records = builder.records;
        this.nextShardIterator = builder.nextShardIterator;
    }

    /**
     * Returns true if the Records property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRecords() {
        return records != null && !(records instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The stream records from the shard, which were retrieved using the shard iterator.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecords()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The stream records from the shard, which were retrieved using the shard iterator.
     */
    public List<Record> records() {
        return records;
    }

    /**
     * <p>
     * The next position in the shard from which to start sequentially reading stream records. If set to
     * <code>null</code>, the shard has been closed and the requested iterator will not return any more data.
     * </p>
     * 
     * @return The next position in the shard from which to start sequentially reading stream records. If set to
     *         <code>null</code>, the shard has been closed and the requested iterator will not return any more data.
     */
    public String nextShardIterator() {
        return nextShardIterator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(hasRecords() ? records() : null);
        hashCode = 31 * hashCode + Objects.hashCode(nextShardIterator());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetRecordsResponse)) {
            return false;
        }
        GetRecordsResponse other = (GetRecordsResponse) obj;
        return hasRecords() == other.hasRecords() && Objects.equals(records(), other.records())
                && Objects.equals(nextShardIterator(), other.nextShardIterator());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("GetRecordsResponse").add("Records", hasRecords() ? records() : null)
                .add("NextShardIterator", nextShardIterator()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Records":
            return Optional.ofNullable(clazz.cast(records()));
        case "NextShardIterator":
            return Optional.ofNullable(clazz.cast(nextShardIterator()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GetRecordsResponse, T> g) {
        return obj -> g.apply((GetRecordsResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DynamoDbStreamsResponse.Builder, SdkPojo, CopyableBuilder<Builder, GetRecordsResponse> {
        /**
         * <p>
         * The stream records from the shard, which were retrieved using the shard iterator.
         * </p>
         * 
         * @param records
         *        The stream records from the shard, which were retrieved using the shard iterator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder records(Collection<Record> records);

        /**
         * <p>
         * The stream records from the shard, which were retrieved using the shard iterator.
         * </p>
         * 
         * @param records
         *        The stream records from the shard, which were retrieved using the shard iterator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder records(Record... records);

        /**
         * <p>
         * The stream records from the shard, which were retrieved using the shard iterator.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Record>.Builder} avoiding the need to
         * create one manually via {@link List<Record>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Record>.Builder#build()} is called immediately and its
         * result is passed to {@link #records(List<Record>)}.
         * 
         * @param records
         *        a consumer that will call methods on {@link List<Record>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #records(List<Record>)
         */
        Builder records(Consumer<Record.Builder>... records);

        /**
         * <p>
         * The next position in the shard from which to start sequentially reading stream records. If set to
         * <code>null</code>, the shard has been closed and the requested iterator will not return any more data.
         * </p>
         * 
         * @param nextShardIterator
         *        The next position in the shard from which to start sequentially reading stream records. If set to
         *        <code>null</code>, the shard has been closed and the requested iterator will not return any more data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextShardIterator(String nextShardIterator);
    }

    static final class BuilderImpl extends DynamoDbStreamsResponse.BuilderImpl implements Builder {
        private List<Record> records = DefaultSdkAutoConstructList.getInstance();

        private String nextShardIterator;

        private BuilderImpl() {
        }

        private BuilderImpl(GetRecordsResponse model) {
            super(model);
            records(model.records);
            nextShardIterator(model.nextShardIterator);
        }

        public final Collection<Record.Builder> getRecords() {
            if (records instanceof SdkAutoConstructList) {
                return null;
            }
            return records != null ? records.stream().map(Record::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder records(Collection<Record> records) {
            this.records = RecordListCopier.copy(records);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder records(Record... records) {
            records(Arrays.asList(records));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder records(Consumer<Record.Builder>... records) {
            records(Stream.of(records).map(c -> Record.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRecords(Collection<Record.BuilderImpl> records) {
            this.records = RecordListCopier.copyFromBuilder(records);
        }

        public final String getNextShardIterator() {
            return nextShardIterator;
        }

        @Override
        public final Builder nextShardIterator(String nextShardIterator) {
            this.nextShardIterator = nextShardIterator;
            return this;
        }

        public final void setNextShardIterator(String nextShardIterator) {
            this.nextShardIterator = nextShardIterator;
        }

        @Override
        public GetRecordsResponse build() {
            return new GetRecordsResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
