/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.dynamodb.model.BackupInUseException;
import software.amazon.awssdk.services.dynamodb.model.BackupNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.BatchGetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchGetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemResponse;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.ContinuousBackupsUnavailableException;
import software.amazon.awssdk.services.dynamodb.model.CreateBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.CreateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.CreateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableSettingsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeLimitsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeLimitsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableReplicaAutoScalingResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveResponse;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbException;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbRequest;
import software.amazon.awssdk.services.dynamodb.model.ExportConflictException;
import software.amazon.awssdk.services.dynamodb.model.ExportNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ExportTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.ExportTableToPointInTimeResponse;
import software.amazon.awssdk.services.dynamodb.model.GetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.GetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.GlobalTableAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.GlobalTableNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.dynamodb.model.IndexNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.InternalServerErrorException;
import software.amazon.awssdk.services.dynamodb.model.InvalidExportTimeException;
import software.amazon.awssdk.services.dynamodb.model.InvalidRestoreTimeException;
import software.amazon.awssdk.services.dynamodb.model.ItemCollectionSizeLimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.LimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ListBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListExportsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListExportsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListGlobalTablesRequest;
import software.amazon.awssdk.services.dynamodb.model.ListGlobalTablesResponse;
import software.amazon.awssdk.services.dynamodb.model.ListTablesRequest;
import software.amazon.awssdk.services.dynamodb.model.ListTablesResponse;
import software.amazon.awssdk.services.dynamodb.model.ListTagsOfResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.ListTagsOfResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.PointInTimeRecoveryUnavailableException;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.dynamodb.model.PutItemRequest;
import software.amazon.awssdk.services.dynamodb.model.PutItemResponse;
import software.amazon.awssdk.services.dynamodb.model.QueryRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryResponse;
import software.amazon.awssdk.services.dynamodb.model.ReplicaAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.ReplicaNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.RequestLimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ResourceInUseException;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableFromBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableFromBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableToPointInTimeResponse;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanResponse;
import software.amazon.awssdk.services.dynamodb.model.TableAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.TableInUseException;
import software.amazon.awssdk.services.dynamodb.model.TableNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.TagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.TagResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactGetItemsRequest;
import software.amazon.awssdk.services.dynamodb.model.TransactGetItemsResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactWriteItemsRequest;
import software.amazon.awssdk.services.dynamodb.model.TransactWriteItemsResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactionCanceledException;
import software.amazon.awssdk.services.dynamodb.model.TransactionConflictException;
import software.amazon.awssdk.services.dynamodb.model.TransactionInProgressException;
import software.amazon.awssdk.services.dynamodb.model.UntagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.UntagResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContinuousBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableSettingsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableReplicaAutoScalingResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveResponse;
import software.amazon.awssdk.services.dynamodb.paginators.BatchGetItemPublisher;
import software.amazon.awssdk.services.dynamodb.paginators.ListContributorInsightsPublisher;
import software.amazon.awssdk.services.dynamodb.paginators.ListExportsPublisher;
import software.amazon.awssdk.services.dynamodb.paginators.ListTablesPublisher;
import software.amazon.awssdk.services.dynamodb.paginators.QueryPublisher;
import software.amazon.awssdk.services.dynamodb.paginators.ScanPublisher;
import software.amazon.awssdk.services.dynamodb.transform.BatchGetItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.BatchWriteItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeContinuousBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeExportRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeGlobalTableSettingsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeLimitsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTableReplicaAutoScalingRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTimeToLiveRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ExportTableToPointInTimeRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.GetItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListExportsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListGlobalTablesRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListTagsOfResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.PutItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.RestoreTableFromBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.RestoreTableToPointInTimeRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ScanRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TransactGetItemsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TransactWriteItemsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateContinuousBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateGlobalTableSettingsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTableReplicaAutoScalingRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTimeToLiveRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.waiters.DynamoDbAsyncWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DynamoDbAsyncClient}.
 *
 * @see DynamoDbAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDynamoDbAsyncClient implements DynamoDbAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDynamoDbAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    private final ScheduledExecutorService executorService;

    protected DefaultDynamoDbAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(DynamoDbAsyncEndpointDiscoveryCacheLoader
                    .create(this));
        }
        this.executorService = clientConfiguration.option(SdkClientOption.SCHEDULED_EXECUTOR_SERVICE);
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * The <code>BatchGetItem</code> operation returns the attributes of one or more items from one or more tables. You
     * identify requested items by primary key.
     * </p>
     * <p>
     * A single operation can retrieve up to 16 MB of data, which can contain as many as 100 items.
     * <code>BatchGetItem</code> returns a partial result if the response size limit is exceeded, the table's
     * provisioned throughput is exceeded, or an internal processing failure occurs. If a partial result is returned,
     * the operation returns a value for <code>UnprocessedKeys</code>. You can use this value to retry the operation
     * starting with the next item to get.
     * </p>
     * <important>
     * <p>
     * If you request more than 100 items, <code>BatchGetItem</code> returns a <code>ValidationException</code> with the
     * message "Too many items requested for the BatchGetItem call."
     * </p>
     * </important>
     * <p>
     * For example, if you ask to retrieve 100 items, but each individual item is 300 KB in size, the system returns 52
     * items (so as not to exceed the 16 MB limit). It also returns an appropriate <code>UnprocessedKeys</code> value so
     * you can get the next page of results. If desired, your application can include its own logic to assemble the
     * pages of results into one dataset.
     * </p>
     * <p>
     * If <i>none</i> of the items can be processed due to insufficient provisioned throughput on all of the tables in
     * the request, then <code>BatchGetItem</code> returns a <code>ProvisionedThroughputExceededException</code>. If
     * <i>at least one</i> of the items is successfully processed, then <code>BatchGetItem</code> completes
     * successfully, while returning the keys of the unread items in <code>UnprocessedKeys</code>.
     * </p>
     * <important>
     * <p>
     * If DynamoDB returns any unprocessed items, you should retry the batch operation on those items. However, <i>we
     * strongly recommend that you use an exponential backoff algorithm</i>. If you retry the batch operation
     * immediately, the underlying read or write requests can still fail due to throttling on the individual tables. If
     * you delay the batch operation using exponential backoff, the individual requests in the batch are much more
     * likely to succeed.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#BatchOperations">Batch
     * Operations and Error Handling</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * By default, <code>BatchGetItem</code> performs eventually consistent reads on every table in the request. If you
     * want strongly consistent reads instead, you can set <code>ConsistentRead</code> to <code>true</code> for any or
     * all tables.
     * </p>
     * <p>
     * In order to minimize response latency, <code>BatchGetItem</code> retrieves items in parallel.
     * </p>
     * <p>
     * When designing your application, keep in mind that DynamoDB does not return items in any particular order. To
     * help parse the response by item, include the primary key values for the items in your request in the
     * <code>ProjectionExpression</code> parameter.
     * </p>
     * <p>
     * If a requested item does not exist, it is not returned in the result. Requests for nonexistent items consume the
     * minimum read capacity units according to the type of read. For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithTables.html#CapacityUnitCalculations"
     * >Working with Tables</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param batchGetItemRequest
     *        Represents the input of a <code>BatchGetItem</code> operation.
     * @return A Java Future containing the result of the BatchGetItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.BatchGetItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchGetItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchGetItemResponse> batchGetItem(BatchGetItemRequest batchGetItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchGetItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    BatchGetItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<BatchGetItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchGetItemRequest, BatchGetItemResponse>()
                            .withOperationName("BatchGetItem").withMarshaller(new BatchGetItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(batchGetItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchGetItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<BatchGetItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>BatchGetItem</code> operation returns the attributes of one or more items from one or more tables. You
     * identify requested items by primary key.
     * </p>
     * <p>
     * A single operation can retrieve up to 16 MB of data, which can contain as many as 100 items.
     * <code>BatchGetItem</code> returns a partial result if the response size limit is exceeded, the table's
     * provisioned throughput is exceeded, or an internal processing failure occurs. If a partial result is returned,
     * the operation returns a value for <code>UnprocessedKeys</code>. You can use this value to retry the operation
     * starting with the next item to get.
     * </p>
     * <important>
     * <p>
     * If you request more than 100 items, <code>BatchGetItem</code> returns a <code>ValidationException</code> with the
     * message "Too many items requested for the BatchGetItem call."
     * </p>
     * </important>
     * <p>
     * For example, if you ask to retrieve 100 items, but each individual item is 300 KB in size, the system returns 52
     * items (so as not to exceed the 16 MB limit). It also returns an appropriate <code>UnprocessedKeys</code> value so
     * you can get the next page of results. If desired, your application can include its own logic to assemble the
     * pages of results into one dataset.
     * </p>
     * <p>
     * If <i>none</i> of the items can be processed due to insufficient provisioned throughput on all of the tables in
     * the request, then <code>BatchGetItem</code> returns a <code>ProvisionedThroughputExceededException</code>. If
     * <i>at least one</i> of the items is successfully processed, then <code>BatchGetItem</code> completes
     * successfully, while returning the keys of the unread items in <code>UnprocessedKeys</code>.
     * </p>
     * <important>
     * <p>
     * If DynamoDB returns any unprocessed items, you should retry the batch operation on those items. However, <i>we
     * strongly recommend that you use an exponential backoff algorithm</i>. If you retry the batch operation
     * immediately, the underlying read or write requests can still fail due to throttling on the individual tables. If
     * you delay the batch operation using exponential backoff, the individual requests in the batch are much more
     * likely to succeed.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#BatchOperations">Batch
     * Operations and Error Handling</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * By default, <code>BatchGetItem</code> performs eventually consistent reads on every table in the request. If you
     * want strongly consistent reads instead, you can set <code>ConsistentRead</code> to <code>true</code> for any or
     * all tables.
     * </p>
     * <p>
     * In order to minimize response latency, <code>BatchGetItem</code> retrieves items in parallel.
     * </p>
     * <p>
     * When designing your application, keep in mind that DynamoDB does not return items in any particular order. To
     * help parse the response by item, include the primary key values for the items in your request in the
     * <code>ProjectionExpression</code> parameter.
     * </p>
     * <p>
     * If a requested item does not exist, it is not returned in the result. Requests for nonexistent items consume the
     * minimum read capacity units according to the type of read. For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithTables.html#CapacityUnitCalculations"
     * >Working with Tables</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #batchGetItem(software.amazon.awssdk.services.dynamodb.model.BatchGetItemRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.BatchGetItemPublisher publisher = client.batchGetItemPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.BatchGetItemPublisher publisher = client.batchGetItemPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.BatchGetItemResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.BatchGetItemResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #batchGetItem(software.amazon.awssdk.services.dynamodb.model.BatchGetItemRequest)} operation.</b>
     * </p>
     *
     * @param batchGetItemRequest
     *        Represents the input of a <code>BatchGetItem</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.BatchGetItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchGetItem" target="_top">AWS API
     *      Documentation</a>
     */
    public BatchGetItemPublisher batchGetItemPaginator(BatchGetItemRequest batchGetItemRequest) {
        return new BatchGetItemPublisher(this, applyPaginatorUserAgent(batchGetItemRequest));
    }

    /**
     * <p>
     * The <code>BatchWriteItem</code> operation puts or deletes multiple items in one or more tables. A single call to
     * <code>BatchWriteItem</code> can write up to 16 MB of data, which can comprise as many as 25 put or delete
     * requests. Individual items to be written can be as large as 400 KB.
     * </p>
     * <note>
     * <p>
     * <code>BatchWriteItem</code> cannot update items. To update items, use the <code>UpdateItem</code> action.
     * </p>
     * </note>
     * <p>
     * The individual <code>PutItem</code> and <code>DeleteItem</code> operations specified in
     * <code>BatchWriteItem</code> are atomic; however <code>BatchWriteItem</code> as a whole is not. If any requested
     * operations fail because the table's provisioned throughput is exceeded or an internal processing failure occurs,
     * the failed operations are returned in the <code>UnprocessedItems</code> response parameter. You can investigate
     * and optionally resend the requests. Typically, you would call <code>BatchWriteItem</code> in a loop. Each
     * iteration would check for unprocessed items and submit a new <code>BatchWriteItem</code> request with those
     * unprocessed items until all items have been processed.
     * </p>
     * <p>
     * If <i>none</i> of the items can be processed due to insufficient provisioned throughput on all of the tables in
     * the request, then <code>BatchWriteItem</code> returns a <code>ProvisionedThroughputExceededException</code>.
     * </p>
     * <important>
     * <p>
     * If DynamoDB returns any unprocessed items, you should retry the batch operation on those items. However, <i>we
     * strongly recommend that you use an exponential backoff algorithm</i>. If you retry the batch operation
     * immediately, the underlying read or write requests can still fail due to throttling on the individual tables. If
     * you delay the batch operation using exponential backoff, the individual requests in the batch are much more
     * likely to succeed.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#Programming.Errors.BatchOperations"
     * >Batch Operations and Error Handling</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * With <code>BatchWriteItem</code>, you can efficiently write or delete large amounts of data, such as from Amazon
     * EMR, or copy data from another database into DynamoDB. In order to improve performance with these large-scale
     * operations, <code>BatchWriteItem</code> does not behave in the same way as individual <code>PutItem</code> and
     * <code>DeleteItem</code> calls would. For example, you cannot specify conditions on individual put and delete
     * requests, and <code>BatchWriteItem</code> does not return deleted items in the response.
     * </p>
     * <p>
     * If you use a programming language that supports concurrency, you can use threads to write items in parallel. Your
     * application must include the necessary logic to manage the threads. With languages that don't support threading,
     * you must update or delete the specified items one at a time. In both situations, <code>BatchWriteItem</code>
     * performs the specified put and delete operations in parallel, giving you the power of the thread pool approach
     * without having to introduce complexity into your application.
     * </p>
     * <p>
     * Parallel processing reduces latency, but each specified put and delete request consumes the same number of write
     * capacity units whether it is processed in parallel or not. Delete operations on nonexistent items consume one
     * write capacity unit.
     * </p>
     * <p>
     * If one or more of the following is true, DynamoDB rejects the entire batch write operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more tables specified in the <code>BatchWriteItem</code> request does not exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * Primary key attributes specified on an item in the request do not match those in the corresponding table's
     * primary key schema.
     * </p>
     * </li>
     * <li>
     * <p>
     * You try to perform multiple operations on the same item in the same <code>BatchWriteItem</code> request. For
     * example, you cannot put and delete the same item in the same <code>BatchWriteItem</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your request contains at least two items with identical hash and range keys (which essentially is two put
     * operations).
     * </p>
     * </li>
     * <li>
     * <p>
     * There are more than 25 requests in the batch.
     * </p>
     * </li>
     * <li>
     * <p>
     * Any individual item in a batch exceeds 400 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The total request size exceeds 16 MB.
     * </p>
     * </li>
     * </ul>
     *
     * @param batchWriteItemRequest
     *        Represents the input of a <code>BatchWriteItem</code> operation.
     * @return A Java Future containing the result of the BatchWriteItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>ItemCollectionSizeLimitExceededException An item collection is too large. This exception is only
     *         returned for tables that have one or more local secondary indexes.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.BatchWriteItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchWriteItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<BatchWriteItemResponse> batchWriteItem(BatchWriteItemRequest batchWriteItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchWriteItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchWriteItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<BatchWriteItemResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, BatchWriteItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<BatchWriteItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<BatchWriteItemRequest, BatchWriteItemResponse>()
                            .withOperationName("BatchWriteItem")
                            .withMarshaller(new BatchWriteItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(batchWriteItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = batchWriteItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<BatchWriteItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a backup for an existing table.
     * </p>
     * <p>
     * Each time you create an on-demand backup, the entire table data is backed up. There is no limit to the number of
     * on-demand backups that can be taken.
     * </p>
     * <p>
     * When you create an on-demand backup, a time marker of the request is cataloged, and the backup is created
     * asynchronously, by applying all changes until the time of the request to the last full table snapshot. Backup
     * requests are processed instantaneously and become available for restore within minutes.
     * </p>
     * <p>
     * You can call <code>CreateBackup</code> at a maximum rate of 50 times per second.
     * </p>
     * <p>
     * All backups in DynamoDB work without consuming any provisioned throughput on the table.
     * </p>
     * <p>
     * If you submit a backup request on 2018-12-14 at 14:25:00, the backup is guaranteed to contain all data committed
     * to the table up to 14:24:00, and data committed after 14:26:00 will not be. The backup might contain data
     * modifications made between 14:24:00 and 14:26:00. On-demand backup does not support causal consistency.
     * </p>
     * <p>
     * Along with data, the following are also included on the backups:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Global secondary indexes (GSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Local secondary indexes (LSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Streams
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned read and write capacity
     * </p>
     * </li>
     * </ul>
     *
     * @param createBackupRequest
     * @return A Java Future containing the result of the CreateBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>TableInUseException A target table with the specified name is either being created or deleted.</li>
     *         <li>ContinuousBackupsUnavailableException Backups have not yet been enabled for this table.</li>
     *         <li>BackupInUseException There is another ongoing conflicting backup control plane operation on the
     *         table. The backup is either being created, deleted or restored to a table.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.CreateBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateBackupResponse> createBackup(CreateBackupRequest createBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                            .withOperationName("CreateBackup").withMarshaller(new CreateBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createBackupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createBackupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a global table from an existing table. A global table creates a replication relationship between two or
     * more DynamoDB tables with the same table name in the provided Regions.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V1.html">Version
     * 2017.11.29</a> of global tables.
     * </p>
     * </note>
     * <p>
     * If you want to add a new replica table to a global table, each of the following conditions must be true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The table must have the same primary key as all of the other replicas.
     * </p>
     * </li>
     * <li>
     * <p>
     * The table must have the same name as all of the other replicas.
     * </p>
     * </li>
     * <li>
     * <p>
     * The table must have DynamoDB Streams enabled, with the stream containing both the new and the old images of the
     * item.
     * </p>
     * </li>
     * <li>
     * <p>
     * None of the replica tables in the global table can contain any data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If global secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The global secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If local secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The local secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The local secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * Write capacity settings should be set consistently across your replica tables and secondary indexes. DynamoDB
     * strongly recommends enabling auto scaling to manage the write capacity settings for all of your global tables
     * replicas and indexes.
     * </p>
     * <p>
     * If you prefer to manage write capacity settings manually, you should provision equal replicated write capacity
     * units to your replica tables. You should also provision equal replicated write capacity units to matching
     * secondary indexes across your global table.
     * </p>
     * </important>
     *
     * @param createGlobalTableRequest
     * @return A Java Future containing the result of the CreateGlobalTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>GlobalTableAlreadyExistsException The specified global table already exists.</li>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.CreateGlobalTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateGlobalTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateGlobalTableResponse> createGlobalTable(CreateGlobalTableRequest createGlobalTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGlobalTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateGlobalTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateGlobalTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateGlobalTableRequest, CreateGlobalTableResponse>()
                            .withOperationName("CreateGlobalTable")
                            .withMarshaller(new CreateGlobalTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createGlobalTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createGlobalTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateGlobalTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to your account. In an AWS account, table names must be
     * unique within each Region. That is, you can have two tables with same name if you create the tables in different
     * Regions.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. Upon receiving a <code>CreateTable</code> request,
     * DynamoDB immediately returns a response with a <code>TableStatus</code> of <code>CREATING</code>. After the table
     * is created, DynamoDB sets the <code>TableStatus</code> to <code>ACTIVE</code>. You can perform read and write
     * operations only on an <code>ACTIVE</code> table.
     * </p>
     * <p>
     * You can optionally define secondary indexes on the new table, as part of the <code>CreateTable</code> operation.
     * If you want to create multiple tables with secondary indexes on them, you must create the tables sequentially.
     * Only one table with secondary indexes can be in the <code>CREATING</code> state at any given time.
     * </p>
     * <p>
     * You can use the <code>DescribeTable</code> action to check the table status.
     * </p>
     *
     * @param createTableRequest
     *        Represents the input of a <code>CreateTable</code> operation.
     * @return A Java Future containing the result of the CreateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.CreateTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreateTableResponse> createTable(CreateTableRequest createTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    CreateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<CreateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                            .withOperationName("CreateTable").withMarshaller(new CreateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(createTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes an existing backup of a table.
     * </p>
     * <p>
     * You can call <code>DeleteBackup</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param deleteBackupRequest
     * @return A Java Future containing the result of the DeleteBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BackupNotFoundException Backup not found for the given BackupARN.</li>
     *         <li>BackupInUseException There is another ongoing conflicting backup control plane operation on the
     *         table. The backup is either being created, deleted or restored to a table.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DeleteBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteBackupResponse> deleteBackup(DeleteBackupRequest deleteBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                            .withOperationName("DeleteBackup").withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteBackupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteBackupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a single item in a table by primary key. You can perform a conditional delete operation that deletes the
     * item if it exists, or if it has an expected attribute value.
     * </p>
     * <p>
     * In addition to deleting an item, you can also return the item's attribute values in the same operation, using the
     * <code>ReturnValues</code> parameter.
     * </p>
     * <p>
     * Unless you specify conditions, the <code>DeleteItem</code> is an idempotent operation; running it multiple times
     * on the same item or attribute does <i>not</i> result in an error response.
     * </p>
     * <p>
     * Conditional deletes are useful for deleting items only if specific conditions are met. If those conditions are
     * met, DynamoDB performs the delete. Otherwise, the item is not deleted.
     * </p>
     *
     * @param deleteItemRequest
     *        Represents the input of a <code>DeleteItem</code> operation.
     * @return A Java Future containing the result of the DeleteItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConditionalCheckFailedException A condition specified in the operation could not be evaluated.</li>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>ItemCollectionSizeLimitExceededException An item collection is too large. This exception is only
     *         returned for tables that have one or more local secondary indexes.</li>
     *         <li>TransactionConflictException Operation was rejected because there is an ongoing transaction for the
     *         item.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DeleteItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteItemResponse> deleteItem(DeleteItemRequest deleteItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteItemRequest, DeleteItemResponse>().withOperationName("DeleteItem")
                            .withMarshaller(new DeleteItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its items. After a <code>DeleteTable</code>
     * request, the specified table is in the <code>DELETING</code> state until DynamoDB completes the deletion. If the
     * table is in the <code>ACTIVE</code> state, you can delete it. If a table is in <code>CREATING</code> or
     * <code>UPDATING</code> states, then DynamoDB returns a <code>ResourceInUseException</code>. If the specified table
     * does not exist, DynamoDB returns a <code>ResourceNotFoundException</code>. If table is already in the
     * <code>DELETING</code> state, no error is returned.
     * </p>
     * <note>
     * <p>
     * DynamoDB might continue to accept data read and write operations, such as <code>GetItem</code> and
     * <code>PutItem</code>, on a table in the <code>DELETING</code> state until the table deletion is complete.
     * </p>
     * </note>
     * <p>
     * When you delete a table, any indexes on that table are also deleted.
     * </p>
     * <p>
     * If you have DynamoDB Streams enabled on the table, then the corresponding stream on that table goes into the
     * <code>DISABLED</code> state, and the stream is automatically deleted after 24 hours.
     * </p>
     * <p>
     * Use the <code>DescribeTable</code> action to check the status of the table.
     * </p>
     *
     * @param deleteTableRequest
     *        Represents the input of a <code>DeleteTable</code> operation.
     * @return A Java Future containing the result of the DeleteTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DeleteTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteTableResponse> deleteTable(DeleteTableRequest deleteTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeleteTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DeleteTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                            .withOperationName("DeleteTable").withMarshaller(new DeleteTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(deleteTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeleteTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing backup of a table.
     * </p>
     * <p>
     * You can call <code>DescribeBackup</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param describeBackupRequest
     * @return A Java Future containing the result of the DescribeBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>BackupNotFoundException Backup not found for the given BackupARN.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeBackupResponse> describeBackup(DescribeBackupRequest describeBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeBackupRequest, DescribeBackupResponse>()
                            .withOperationName("DescribeBackup")
                            .withMarshaller(new DescribeBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeBackupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeBackupRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Checks the status of continuous backups and point in time recovery on the specified table. Continuous backups are
     * <code>ENABLED</code> on all tables at table creation. If point in time recovery is enabled,
     * <code>PointInTimeRecoveryStatus</code> will be set to ENABLED.
     * </p>
     * <p>
     * After continuous backups and point in time recovery are enabled, you can restore to any point in time within
     * <code>EarliestRestorableDateTime</code> and <code>LatestRestorableDateTime</code>.
     * </p>
     * <p>
     * <code>LatestRestorableDateTime</code> is typically 5 minutes before the current time. You can restore your table
     * to any point in time during the last 35 days.
     * </p>
     * <p>
     * You can call <code>DescribeContinuousBackups</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param describeContinuousBackupsRequest
     * @return A Java Future containing the result of the DescribeContinuousBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeContinuousBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeContinuousBackups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeContinuousBackupsResponse> describeContinuousBackups(
            DescribeContinuousBackupsRequest describeContinuousBackupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContinuousBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContinuousBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeContinuousBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeContinuousBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeContinuousBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeContinuousBackupsRequest, DescribeContinuousBackupsResponse>()
                            .withOperationName("DescribeContinuousBackups")
                            .withMarshaller(new DescribeContinuousBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeContinuousBackupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeContinuousBackupsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeContinuousBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about contributor insights, for a given table or global secondary index.
     * </p>
     *
     * @param describeContributorInsightsRequest
     * @return A Java Future containing the result of the DescribeContributorInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeContributorInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeContributorInsightsResponse> describeContributorInsights(
            DescribeContributorInsightsRequest describeContributorInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContributorInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeContributorInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeContributorInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeContributorInsightsRequest, DescribeContributorInsightsResponse>()
                            .withOperationName("DescribeContributorInsights")
                            .withMarshaller(new DescribeContributorInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeContributorInsightsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeContributorInsightsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeContributorInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the regional endpoint information.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return A Java Future containing the result of the DescribeEndpoints operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeEndpoints
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeEndpoints" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeEndpointsResponse> describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeEndpointsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeEndpointsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                            .withOperationName("DescribeEndpoints")
                            .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeEndpointsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeEndpointsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeEndpointsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes an existing table export.
     * </p>
     *
     * @param describeExportRequest
     * @return A Java Future containing the result of the DescribeExport operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ExportNotFoundException The specified export was not found.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeExport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeExport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeExportResponse> describeExport(DescribeExportRequest describeExportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeExportResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeExportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeExportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeExportRequest, DescribeExportResponse>()
                            .withOperationName("DescribeExport")
                            .withMarshaller(new DescribeExportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeExportRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeExportRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeExportResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the specified global table.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V1.html">Version
     * 2017.11.29</a> of global tables. If you are using global tables <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html">Version
     * 2019.11.21</a> you can use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_DescribeTable.html">DescribeTable</a>
     * instead.
     * </p>
     * </note>
     *
     * @param describeGlobalTableRequest
     * @return A Java Future containing the result of the DescribeGlobalTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>GlobalTableNotFoundException The specified global table does not exist.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeGlobalTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeGlobalTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGlobalTableResponse> describeGlobalTable(
            DescribeGlobalTableRequest describeGlobalTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGlobalTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGlobalTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeGlobalTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGlobalTableRequest, DescribeGlobalTableResponse>()
                            .withOperationName("DescribeGlobalTable")
                            .withMarshaller(new DescribeGlobalTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeGlobalTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGlobalTableRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeGlobalTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes Region-specific settings for a global table.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V1.html">Version
     * 2017.11.29</a> of global tables.
     * </p>
     * </note>
     *
     * @param describeGlobalTableSettingsRequest
     * @return A Java Future containing the result of the DescribeGlobalTableSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GlobalTableNotFoundException The specified global table does not exist.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeGlobalTableSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeGlobalTableSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeGlobalTableSettingsResponse> describeGlobalTableSettings(
            DescribeGlobalTableSettingsRequest describeGlobalTableSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGlobalTableSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGlobalTableSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeGlobalTableSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeGlobalTableSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeGlobalTableSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeGlobalTableSettingsRequest, DescribeGlobalTableSettingsResponse>()
                            .withOperationName("DescribeGlobalTableSettings")
                            .withMarshaller(new DescribeGlobalTableSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeGlobalTableSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeGlobalTableSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeGlobalTableSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the current provisioned-capacity quotas for your AWS account in a Region, both for the Region as a whole
     * and for any one DynamoDB table that you create there.
     * </p>
     * <p>
     * When you establish an AWS account, the account has initial quotas on the maximum read capacity units and write
     * capacity units that you can provision across all of your DynamoDB tables in a given Region. Also, there are
     * per-table quotas that apply when you create a table there. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Service, Account, and Table
     * Quotas</a> page in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Although you can increase these quotas by filing a case at <a
     * href="https://console.aws.amazon.com/support/home#/">AWS Support Center</a>, obtaining the increase is not
     * instantaneous. The <code>DescribeLimits</code> action lets you write code to compare the capacity you are
     * currently using to those quotas imposed by your account so that you have enough time to apply for an increase
     * before you hit a quota.
     * </p>
     * <p>
     * For example, you could use one of the AWS SDKs to do the following:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Call <code>DescribeLimits</code> for a particular Region to obtain your current account quotas on provisioned
     * capacity there.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create a variable to hold the aggregate read capacity units provisioned for all your tables in that Region, and
     * one to hold the aggregate write capacity units. Zero them both.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call <code>ListTables</code> to obtain a list of all your DynamoDB tables.
     * </p>
     * </li>
     * <li>
     * <p>
     * For each table name listed by <code>ListTables</code>, do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Call <code>DescribeTable</code> with the table name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the data returned by <code>DescribeTable</code> to add the read capacity units and write capacity units
     * provisioned for the table itself to your variables.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the table has one or more global secondary indexes (GSIs), loop over these GSIs and add their provisioned
     * capacity values to your variables as well.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Report the account quotas for that Region returned by <code>DescribeLimits</code>, along with the total current
     * provisioned capacity levels you have calculated.
     * </p>
     * </li>
     * </ol>
     * <p>
     * This will let you see whether you are getting close to your account-level quotas.
     * </p>
     * <p>
     * The per-table quotas apply only when you are creating a new table. They restrict the sum of the provisioned
     * capacity of the new table itself and all its global secondary indexes.
     * </p>
     * <p>
     * For existing tables and their GSIs, DynamoDB doesn't let you increase provisioned capacity extremely rapidly, but
     * the only quota that applies is that the aggregate provisioned capacity over all your tables and GSIs cannot
     * exceed either of the per-account quotas.
     * </p>
     * <note>
     * <p>
     * <code>DescribeLimits</code> should only be called periodically. You can expect throttling errors if you call it
     * more than once in a minute.
     * </p>
     * </note>
     * <p>
     * The <code>DescribeLimits</code> Request element has no content.
     * </p>
     *
     * @param describeLimitsRequest
     *        Represents the input of a <code>DescribeLimits</code> operation. Has no content.
     * @return A Java Future containing the result of the DescribeLimits operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeLimits
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeLimitsResponse> describeLimits(DescribeLimitsRequest describeLimitsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLimits");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeLimitsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeLimitsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeLimitsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeLimitsRequest, DescribeLimitsResponse>()
                            .withOperationName("DescribeLimits")
                            .withMarshaller(new DescribeLimitsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeLimitsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeLimitsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeLimitsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns information about the table, including the current status of the table, when it was created, the primary
     * key schema, and any indexes on the table.
     * </p>
     * <note>
     * <p>
     * If you issue a <code>DescribeTable</code> request immediately after a <code>CreateTable</code> request, DynamoDB
     * might return a <code>ResourceNotFoundException</code>. This is because <code>DescribeTable</code> uses an
     * eventually consistent query, and the metadata for your table might not be available at that moment. Wait for a
     * few seconds, and then try the <code>DescribeTable</code> request again.
     * </p>
     * </note>
     *
     * @param describeTableRequest
     *        Represents the input of a <code>DescribeTable</code> operation.
     * @return A Java Future containing the result of the DescribeTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTableResponse> describeTable(DescribeTableRequest describeTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DescribeTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                            .withOperationName("DescribeTable")
                            .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Describes auto scaling settings across replicas of the global table at once.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html">Version
     * 2019.11.21</a> of global tables.
     * </p>
     * </note>
     *
     * @param describeTableReplicaAutoScalingRequest
     * @return A Java Future containing the result of the DescribeTableReplicaAutoScaling operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeTableReplicaAutoScaling
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTableReplicaAutoScaling"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTableReplicaAutoScalingResponse> describeTableReplicaAutoScaling(
            DescribeTableReplicaAutoScalingRequest describeTableReplicaAutoScalingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTableReplicaAutoScalingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTableReplicaAutoScaling");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTableReplicaAutoScalingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTableReplicaAutoScalingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeTableReplicaAutoScalingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTableReplicaAutoScalingRequest, DescribeTableReplicaAutoScalingResponse>()
                            .withOperationName("DescribeTableReplicaAutoScaling")
                            .withMarshaller(new DescribeTableReplicaAutoScalingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeTableReplicaAutoScalingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTableReplicaAutoScalingRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeTableReplicaAutoScalingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Gives a description of the Time to Live (TTL) status on the specified table.
     * </p>
     *
     * @param describeTimeToLiveRequest
     * @return A Java Future containing the result of the DescribeTimeToLive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.DescribeTimeToLive
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTimeToLive" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeTimeToLiveResponse> describeTimeToLive(DescribeTimeToLiveRequest describeTimeToLiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTimeToLiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTimeToLive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeTimeToLiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeTimeToLiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<DescribeTimeToLiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeTimeToLiveRequest, DescribeTimeToLiveResponse>()
                            .withOperationName("DescribeTimeToLive")
                            .withMarshaller(new DescribeTimeToLiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(describeTimeToLiveRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeTimeToLiveRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeTimeToLiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports table data to an S3 bucket. The table must have point in time recovery enabled, and you can export data
     * from any time within the point in time recovery window.
     * </p>
     *
     * @param exportTableToPointInTimeRequest
     * @return A Java Future containing the result of the ExportTableToPointInTime operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>PointInTimeRecoveryUnavailableException Point in time recovery has not yet been enabled for this
     *         source table.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InvalidExportTimeException The specified <code>ExportTime</code> is outside of the point in time
     *         recovery window.</li>
     *         <li>ExportConflictException There was a conflict when writing to the specified S3 bucket.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ExportTableToPointInTime
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ExportTableToPointInTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportTableToPointInTimeResponse> exportTableToPointInTime(
            ExportTableToPointInTimeRequest exportTableToPointInTimeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportTableToPointInTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportTableToPointInTime");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportTableToPointInTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ExportTableToPointInTimeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportTableToPointInTimeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportTableToPointInTimeRequest, ExportTableToPointInTimeResponse>()
                            .withOperationName("ExportTableToPointInTime")
                            .withMarshaller(new ExportTableToPointInTimeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportTableToPointInTimeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportTableToPointInTimeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ExportTableToPointInTimeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>GetItem</code> operation returns a set of attributes for the item with the given primary key. If there
     * is no matching item, <code>GetItem</code> does not return any data and there will be no <code>Item</code> element
     * in the response.
     * </p>
     * <p>
     * <code>GetItem</code> provides an eventually consistent read by default. If your application requires a strongly
     * consistent read, set <code>ConsistentRead</code> to <code>true</code>. Although a strongly consistent read might
     * take more time than an eventually consistent read, it always returns the last updated value.
     * </p>
     *
     * @param getItemRequest
     *        Represents the input of a <code>GetItem</code> operation.
     * @return A Java Future containing the result of the GetItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.GetItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/GetItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetItemResponse> getItem(GetItemRequest getItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<GetItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetItemRequest, GetItemResponse>().withOperationName("GetItem")
                            .withMarshaller(new GetItemRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .discoveredEndpoint(cachedEndpoint).withInput(getItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List backups associated with an AWS account. To list backups for a given table, specify <code>TableName</code>.
     * <code>ListBackups</code> returns a paginated list of results with at most 1 MB worth of items in a page. You can
     * also specify a maximum number of entries to be returned in a page.
     * </p>
     * <p>
     * In the request, start time is inclusive, but end time is exclusive. Note that these boundaries are for the time
     * at which the original backup was requested.
     * </p>
     * <p>
     * You can call <code>ListBackups</code> a maximum of five times per second.
     * </p>
     *
     * @param listBackupsRequest
     * @return A Java Future containing the result of the ListBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListBackupsResponse> listBackups(ListBackupsRequest listBackupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListBackupsRequest, ListBackupsResponse>()
                            .withOperationName("ListBackups").withMarshaller(new ListBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listBackupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listBackupsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ContributorInsightsSummary for a table and all its global secondary indexes.
     * </p>
     *
     * @param listContributorInsightsRequest
     * @return A Java Future containing the result of the ListContributorInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListContributorInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListContributorInsightsResponse> listContributorInsights(
            ListContributorInsightsRequest listContributorInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContributorInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListContributorInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListContributorInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListContributorInsightsRequest, ListContributorInsightsResponse>()
                            .withOperationName("ListContributorInsights")
                            .withMarshaller(new ListContributorInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listContributorInsightsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listContributorInsightsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListContributorInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of ContributorInsightsSummary for a table and all its global secondary indexes.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listContributorInsights(software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListContributorInsightsPublisher publisher = client.listContributorInsightsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListContributorInsightsPublisher publisher = client.listContributorInsightsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listContributorInsights(software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listContributorInsightsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListContributorInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    public ListContributorInsightsPublisher listContributorInsightsPaginator(
            ListContributorInsightsRequest listContributorInsightsRequest) {
        return new ListContributorInsightsPublisher(this, applyPaginatorUserAgent(listContributorInsightsRequest));
    }

    /**
     * <p>
     * Lists completed exports within the past 90 days.
     * </p>
     *
     * @param listExportsRequest
     * @return A Java Future containing the result of the ListExports operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListExports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListExports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListExportsResponse> listExports(ListExportsRequest listExportsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExports");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListExportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListExportsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListExportsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListExportsRequest, ListExportsResponse>()
                            .withOperationName("ListExports").withMarshaller(new ListExportsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listExportsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listExportsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListExportsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists completed exports within the past 90 days.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listExports(software.amazon.awssdk.services.dynamodb.model.ListExportsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListExportsPublisher publisher = client.listExportsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListExportsPublisher publisher = client.listExportsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.ListExportsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.ListExportsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listExports(software.amazon.awssdk.services.dynamodb.model.ListExportsRequest)} operation.</b>
     * </p>
     *
     * @param listExportsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListExports
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListExports" target="_top">AWS API
     *      Documentation</a>
     */
    public ListExportsPublisher listExportsPaginator(ListExportsRequest listExportsRequest) {
        return new ListExportsPublisher(this, applyPaginatorUserAgent(listExportsRequest));
    }

    /**
     * <p>
     * Lists all global tables that have a replica in the specified Region.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V1.html">Version
     * 2017.11.29</a> of global tables.
     * </p>
     * </note>
     *
     * @param listGlobalTablesRequest
     * @return A Java Future containing the result of the ListGlobalTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListGlobalTables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListGlobalTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListGlobalTablesResponse> listGlobalTables(ListGlobalTablesRequest listGlobalTablesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGlobalTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGlobalTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListGlobalTablesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListGlobalTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListGlobalTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListGlobalTablesRequest, ListGlobalTablesResponse>()
                            .withOperationName("ListGlobalTables")
                            .withMarshaller(new ListGlobalTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listGlobalTablesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listGlobalTablesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListGlobalTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of table names associated with the current account and endpoint. The output from
     * <code>ListTables</code> is paginated, with each page returning a maximum of 100 table names.
     * </p>
     *
     * @param listTablesRequest
     *        Represents the input of a <code>ListTables</code> operation.
     * @return A Java Future containing the result of the ListTables operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListTables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTablesResponse> listTables(ListTablesRequest listTablesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTablesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListTablesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listTablesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTablesRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTablesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns an array of table names associated with the current account and endpoint. The output from
     * <code>ListTables</code> is paginated, with each page returning a maximum of 100 table names.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTables(software.amazon.awssdk.services.dynamodb.model.ListTablesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ListTablesPublisher publisher = client.listTablesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.ListTablesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.ListTablesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTables(software.amazon.awssdk.services.dynamodb.model.ListTablesRequest)} operation.</b>
     * </p>
     *
     * @param listTablesRequest
     *        Represents the input of a <code>ListTables</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListTables
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTablesPublisher listTablesPaginator(ListTablesRequest listTablesRequest) {
        return new ListTablesPublisher(this, applyPaginatorUserAgent(listTablesRequest));
    }

    /**
     * <p>
     * List all tags on an Amazon DynamoDB resource. You can call ListTagsOfResource up to 10 times per second, per
     * account.
     * </p>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param listTagsOfResourceRequest
     * @return A Java Future containing the result of the ListTagsOfResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.ListTagsOfResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListTagsOfResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsOfResourceResponse> listTagsOfResource(ListTagsOfResourceRequest listTagsOfResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsOfResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsOfResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsOfResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsOfResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ListTagsOfResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsOfResourceRequest, ListTagsOfResourceResponse>()
                            .withOperationName("ListTagsOfResource")
                            .withMarshaller(new ListTagsOfResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(listTagsOfResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsOfResourceRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListTagsOfResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a new item, or replaces an old item with a new item. If an item that has the same primary key as the new
     * item already exists in the specified table, the new item completely replaces the existing item. You can perform a
     * conditional put operation (add a new item if one with the specified primary key doesn't exist), or replace an
     * existing item if it has certain attribute values. You can return the item's attribute values in the same
     * operation, using the <code>ReturnValues</code> parameter.
     * </p>
     * <important>
     * <p>
     * This topic provides general information about the <code>PutItem</code> API.
     * </p>
     * <p>
     * For information on how to call the <code>PutItem</code> API using the AWS SDK in specific languages, see the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/aws-cli/dynamodb-2012-08-10/PutItem"> PutItem in the AWS Command Line
     * Interface</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/DotNetSDKV3/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * .NET</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/SdkForCpp/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * C++</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/SdkForGoV1/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * Go</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/SdkForJava/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * Java</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/AWSJavaScriptSDK/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK
     * for JavaScript</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/SdkForPHPV3/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for PHP
     * V3</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/boto3/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * Python</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a href="http://docs.aws.amazon.com/goto/SdkForRubyV2/dynamodb-2012-08-10/PutItem"> PutItem in the AWS SDK for
     * Ruby V2</a>
     * </p>
     * </li>
     * </ul>
     * </important>
     * <p>
     * When you add an item, the primary key attributes are the only required attributes. Attribute values cannot be
     * null.
     * </p>
     * <p>
     * Empty String and Binary attribute values are allowed. Attribute values of type String and Binary must have a
     * length greater than zero if the attribute is used as a key attribute for a table or index. Set type attributes
     * cannot be empty.
     * </p>
     * <p>
     * Invalid Requests with empty values will be rejected with a <code>ValidationException</code> exception.
     * </p>
     * <note>
     * <p>
     * To prevent a new item from replacing an existing item, use a conditional expression that contains the
     * <code>attribute_not_exists</code> function with the name of the attribute being used as the partition key for the
     * table. Since every record must contain that attribute, the <code>attribute_not_exists</code> function will only
     * succeed if no matching item exists.
     * </p>
     * </note>
     * <p>
     * For more information about <code>PutItem</code>, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithItems.html">Working with
     * Items</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param putItemRequest
     *        Represents the input of a <code>PutItem</code> operation.
     * @return A Java Future containing the result of the PutItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConditionalCheckFailedException A condition specified in the operation could not be evaluated.</li>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>ItemCollectionSizeLimitExceededException An item collection is too large. This exception is only
     *         returned for tables that have one or more local secondary indexes.</li>
     *         <li>TransactionConflictException Operation was rejected because there is an ongoing transaction for the
     *         item.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.PutItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/PutItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutItemResponse> putItem(PutItemRequest putItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<PutItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutItemRequest, PutItemResponse>().withOperationName("PutItem")
                            .withMarshaller(new PutItemRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .discoveredEndpoint(cachedEndpoint).withInput(putItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Query</code> operation finds items based on primary key values. You can query any table or secondary
     * index that has a composite primary key (a partition key and a sort key).
     * </p>
     * <p>
     * Use the <code>KeyConditionExpression</code> parameter to provide a specific value for the partition key. The
     * <code>Query</code> operation will return all of the items from the table or index with that partition key value.
     * You can optionally narrow the scope of the <code>Query</code> operation by specifying a sort key value and a
     * comparison operator in <code>KeyConditionExpression</code>. To further refine the <code>Query</code> results, you
     * can optionally provide a <code>FilterExpression</code>. A <code>FilterExpression</code> determines which items
     * within the results should be returned to you. All of the other results are discarded.
     * </p>
     * <p>
     * A <code>Query</code> operation always returns a result set. If no matching items are found, the result set will
     * be empty. Queries that do not return results consume the minimum number of read capacity units for that type of
     * read operation.
     * </p>
     * <note>
     * <p>
     * DynamoDB calculates the number of read capacity units consumed based on item size, not on the amount of data that
     * is returned to an application. The number of capacity units consumed will be the same whether you request all of
     * the attributes (the default behavior) or just some of them (using a projection expression). The number will also
     * be the same whether or not you use a <code>FilterExpression</code>.
     * </p>
     * </note>
     * <p>
     * <code>Query</code> results are always sorted by the sort key value. If the data type of the sort key is Number,
     * the results are returned in numeric order; otherwise, the results are returned in order of UTF-8 bytes. By
     * default, the sort order is ascending. To reverse the order, set the <code>ScanIndexForward</code> parameter to
     * false.
     * </p>
     * <p>
     * A single <code>Query</code> operation will read up to the maximum number of items set (if using the
     * <code>Limit</code> parameter) or a maximum of 1 MB of data and then apply any filtering to the results using
     * <code>FilterExpression</code>. If <code>LastEvaluatedKey</code> is present in the response, you will need to
     * paginate the result set. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Query.html#Query.Pagination">Paginating
     * the Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>FilterExpression</code> is applied after a <code>Query</code> finishes, but before the results are
     * returned. A <code>FilterExpression</code> cannot contain partition key or sort key attributes. You need to
     * specify those attributes in the <code>KeyConditionExpression</code>.
     * </p>
     * <note>
     * <p>
     * A <code>Query</code> operation can return an empty result set and a <code>LastEvaluatedKey</code> if all the
     * items read for the page of results are filtered out.
     * </p>
     * </note>
     * <p>
     * You can query a table, a local secondary index, or a global secondary index. For a query on a table or on a local
     * secondary index, you can set the <code>ConsistentRead</code> parameter to <code>true</code> and obtain a strongly
     * consistent result. Global secondary indexes support eventually consistent reads only, so do not specify
     * <code>ConsistentRead</code> when querying a global secondary index.
     * </p>
     *
     * @param queryRequest
     *        Represents the input of a <code>Query</code> operation.
     * @return A Java Future containing the result of the Query operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.Query
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<QueryResponse> query(QueryRequest queryRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    QueryResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<QueryResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                            .withMarshaller(new QueryRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .discoveredEndpoint(cachedEndpoint).withInput(queryRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = queryRequest.overrideConfiguration().orElse(null);
            CompletableFuture<QueryResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Query</code> operation finds items based on primary key values. You can query any table or secondary
     * index that has a composite primary key (a partition key and a sort key).
     * </p>
     * <p>
     * Use the <code>KeyConditionExpression</code> parameter to provide a specific value for the partition key. The
     * <code>Query</code> operation will return all of the items from the table or index with that partition key value.
     * You can optionally narrow the scope of the <code>Query</code> operation by specifying a sort key value and a
     * comparison operator in <code>KeyConditionExpression</code>. To further refine the <code>Query</code> results, you
     * can optionally provide a <code>FilterExpression</code>. A <code>FilterExpression</code> determines which items
     * within the results should be returned to you. All of the other results are discarded.
     * </p>
     * <p>
     * A <code>Query</code> operation always returns a result set. If no matching items are found, the result set will
     * be empty. Queries that do not return results consume the minimum number of read capacity units for that type of
     * read operation.
     * </p>
     * <note>
     * <p>
     * DynamoDB calculates the number of read capacity units consumed based on item size, not on the amount of data that
     * is returned to an application. The number of capacity units consumed will be the same whether you request all of
     * the attributes (the default behavior) or just some of them (using a projection expression). The number will also
     * be the same whether or not you use a <code>FilterExpression</code>.
     * </p>
     * </note>
     * <p>
     * <code>Query</code> results are always sorted by the sort key value. If the data type of the sort key is Number,
     * the results are returned in numeric order; otherwise, the results are returned in order of UTF-8 bytes. By
     * default, the sort order is ascending. To reverse the order, set the <code>ScanIndexForward</code> parameter to
     * false.
     * </p>
     * <p>
     * A single <code>Query</code> operation will read up to the maximum number of items set (if using the
     * <code>Limit</code> parameter) or a maximum of 1 MB of data and then apply any filtering to the results using
     * <code>FilterExpression</code>. If <code>LastEvaluatedKey</code> is present in the response, you will need to
     * paginate the result set. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Query.html#Query.Pagination">Paginating
     * the Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>FilterExpression</code> is applied after a <code>Query</code> finishes, but before the results are
     * returned. A <code>FilterExpression</code> cannot contain partition key or sort key attributes. You need to
     * specify those attributes in the <code>KeyConditionExpression</code>.
     * </p>
     * <note>
     * <p>
     * A <code>Query</code> operation can return an empty result set and a <code>LastEvaluatedKey</code> if all the
     * items read for the page of results are filtered out.
     * </p>
     * </note>
     * <p>
     * You can query a table, a local secondary index, or a global secondary index. For a query on a table or on a local
     * secondary index, you can set the <code>ConsistentRead</code> parameter to <code>true</code> and obtain a strongly
     * consistent result. Global secondary indexes support eventually consistent reads only, so do not specify
     * <code>ConsistentRead</code> when querying a global secondary index.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #query(software.amazon.awssdk.services.dynamodb.model.QueryRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.QueryPublisher publisher = client.queryPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.QueryPublisher publisher = client.queryPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.QueryResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.QueryResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #query(software.amazon.awssdk.services.dynamodb.model.QueryRequest)} operation.</b>
     * </p>
     *
     * @param queryRequest
     *        Represents the input of a <code>Query</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.Query
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Query" target="_top">AWS API
     *      Documentation</a>
     */
    public QueryPublisher queryPaginator(QueryRequest queryRequest) {
        return new QueryPublisher(this, applyPaginatorUserAgent(queryRequest));
    }

    /**
     * <p>
     * Creates a new table from an existing backup. Any number of users can execute up to 4 concurrent restores (any
     * type of restore) in a given account.
     * </p>
     * <p>
     * You can call <code>RestoreTableFromBackup</code> at a maximum rate of 10 times per second.
     * </p>
     * <p>
     * You must manually set up the following on the restored table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto scaling policies
     * </p>
     * </li>
     * <li>
     * <p>
     * IAM policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Time to Live (TTL) settings
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableFromBackupRequest
     * @return A Java Future containing the result of the RestoreTableFromBackup operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableAlreadyExistsException A target table with the specified name already exists.</li>
     *         <li>TableInUseException A target table with the specified name is either being created or deleted.</li>
     *         <li>BackupNotFoundException Backup not found for the given BackupARN.</li>
     *         <li>BackupInUseException There is another ongoing conflicting backup control plane operation on the
     *         table. The backup is either being created, deleted or restored to a table.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.RestoreTableFromBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/RestoreTableFromBackup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreTableFromBackupResponse> restoreTableFromBackup(
            RestoreTableFromBackupRequest restoreTableFromBackupRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTableFromBackup");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreTableFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreTableFromBackupResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<RestoreTableFromBackupResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreTableFromBackupRequest, RestoreTableFromBackupResponse>()
                            .withOperationName("RestoreTableFromBackup")
                            .withMarshaller(new RestoreTableFromBackupRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(restoreTableFromBackupRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = restoreTableFromBackupRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<RestoreTableFromBackupResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores the specified table to the specified point in time within <code>EarliestRestorableDateTime</code> and
     * <code>LatestRestorableDateTime</code>. You can restore your table to any point in time during the last 35 days.
     * Any number of users can execute up to 4 concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, DynamoDB restores your table data to the state based on the
     * selected date and time (day:hour:minute:second) to a new table.
     * </p>
     * <p>
     * Along with data, the following are also included on the new restored table using point in time recovery:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Global secondary indexes (GSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Local secondary indexes (LSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned read and write capacity
     * </p>
     * </li>
     * <li>
     * <p>
     * Encryption settings
     * </p>
     * <important>
     * <p>
     * All these settings come from the current settings of the source table at the time of restore.
     * </p>
     * </important></li>
     * </ul>
     * <p>
     * You must manually set up the following on the restored table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto scaling policies
     * </p>
     * </li>
     * <li>
     * <p>
     * IAM policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Time to Live (TTL) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point in time recovery settings
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableToPointInTimeRequest
     * @return A Java Future containing the result of the RestoreTableToPointInTime operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableAlreadyExistsException A target table with the specified name already exists.</li>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>TableInUseException A target table with the specified name is either being created or deleted.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InvalidRestoreTimeException An invalid restore time was specified. RestoreDateTime must be between
     *         EarliestRestorableDateTime and LatestRestorableDateTime.</li>
     *         <li>PointInTimeRecoveryUnavailableException Point in time recovery has not yet been enabled for this
     *         source table.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.RestoreTableToPointInTime
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/RestoreTableToPointInTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreTableToPointInTimeResponse> restoreTableToPointInTime(
            RestoreTableToPointInTimeRequest restoreTableToPointInTimeRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableToPointInTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTableToPointInTime");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreTableToPointInTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreTableToPointInTimeResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<RestoreTableToPointInTimeResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreTableToPointInTimeRequest, RestoreTableToPointInTimeResponse>()
                            .withOperationName("RestoreTableToPointInTime")
                            .withMarshaller(new RestoreTableToPointInTimeRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(restoreTableToPointInTimeRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = restoreTableToPointInTimeRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RestoreTableToPointInTimeResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Scan</code> operation returns one or more items and item attributes by accessing every item in a table
     * or a secondary index. To have DynamoDB return fewer items, you can provide a <code>FilterExpression</code>
     * operation.
     * </p>
     * <p>
     * If the total number of scanned items exceeds the maximum dataset size limit of 1 MB, the scan stops and results
     * are returned to the user as a <code>LastEvaluatedKey</code> value to continue the scan in a subsequent operation.
     * The results also include the number of items exceeding the limit. A scan can result in no table data meeting the
     * filter criteria.
     * </p>
     * <p>
     * A single <code>Scan</code> operation reads up to the maximum number of items set (if using the <code>Limit</code>
     * parameter) or a maximum of 1 MB of data and then apply any filtering to the results using
     * <code>FilterExpression</code>. If <code>LastEvaluatedKey</code> is present in the response, you need to paginate
     * the result set. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.Pagination">Paginating the
     * Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>Scan</code> operations proceed sequentially; however, for faster performance on a large table or secondary
     * index, applications can request a parallel <code>Scan</code> operation by providing the <code>Segment</code> and
     * <code>TotalSegments</code> parameters. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.ParallelScan">Parallel
     * Scan</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>Scan</code> uses eventually consistent reads when accessing the data in a table; therefore, the result set
     * might not include the changes to data in the table immediately before the operation began. If you need a
     * consistent copy of the data, as of the time that the <code>Scan</code> begins, you can set the
     * <code>ConsistentRead</code> parameter to <code>true</code>.
     * </p>
     *
     * @param scanRequest
     *        Represents the input of a <code>Scan</code> operation.
     * @return A Java Future containing the result of the Scan operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.Scan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Scan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ScanResponse> scan(ScanRequest scanRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, scanRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Scan");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ScanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ScanResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<ScanResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ScanRequest, ScanResponse>().withOperationName("Scan")
                            .withMarshaller(new ScanRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .discoveredEndpoint(cachedEndpoint).withInput(scanRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = scanRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ScanResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>Scan</code> operation returns one or more items and item attributes by accessing every item in a table
     * or a secondary index. To have DynamoDB return fewer items, you can provide a <code>FilterExpression</code>
     * operation.
     * </p>
     * <p>
     * If the total number of scanned items exceeds the maximum dataset size limit of 1 MB, the scan stops and results
     * are returned to the user as a <code>LastEvaluatedKey</code> value to continue the scan in a subsequent operation.
     * The results also include the number of items exceeding the limit. A scan can result in no table data meeting the
     * filter criteria.
     * </p>
     * <p>
     * A single <code>Scan</code> operation reads up to the maximum number of items set (if using the <code>Limit</code>
     * parameter) or a maximum of 1 MB of data and then apply any filtering to the results using
     * <code>FilterExpression</code>. If <code>LastEvaluatedKey</code> is present in the response, you need to paginate
     * the result set. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.Pagination">Paginating the
     * Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>Scan</code> operations proceed sequentially; however, for faster performance on a large table or secondary
     * index, applications can request a parallel <code>Scan</code> operation by providing the <code>Segment</code> and
     * <code>TotalSegments</code> parameters. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.ParallelScan">Parallel
     * Scan</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>Scan</code> uses eventually consistent reads when accessing the data in a table; therefore, the result set
     * might not include the changes to data in the table immediately before the operation began. If you need a
     * consistent copy of the data, as of the time that the <code>Scan</code> begins, you can set the
     * <code>ConsistentRead</code> parameter to <code>true</code>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #scan(software.amazon.awssdk.services.dynamodb.model.ScanRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ScanPublisher publisher = client.scanPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.dynamodb.paginators.ScanPublisher publisher = client.scanPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.dynamodb.model.ScanResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.dynamodb.model.ScanResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of Limit won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #scan(software.amazon.awssdk.services.dynamodb.model.ScanRequest)} operation.</b>
     * </p>
     *
     * @param scanRequest
     *        Represents the input of a <code>Scan</code> operation.
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.Scan
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Scan" target="_top">AWS API
     *      Documentation</a>
     */
    public ScanPublisher scanPaginator(ScanRequest scanRequest) {
        return new ScanPublisher(this, applyPaginatorUserAgent(scanRequest));
    }

    /**
     * <p>
     * Associate a set of tags with an Amazon DynamoDB resource. You can then activate these user-defined tags so that
     * they appear on the Billing and Cost Management console for cost allocation tracking. You can call TagResource up
     * to five times per second, per account.
     * </p>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(tagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>TransactGetItems</code> is a synchronous operation that atomically retrieves multiple items from one or
     * more tables (but not from indexes) in a single account and Region. A <code>TransactGetItems</code> call can
     * contain up to 25 <code>TransactGetItem</code> objects, each of which contains a <code>Get</code> structure that
     * specifies an item to retrieve from a table in the account and Region. A call to <code>TransactGetItems</code>
     * cannot retrieve items from tables in more than one AWS account or Region. The aggregate size of the items in the
     * transaction cannot exceed 4 MB.
     * </p>
     * <p>
     * DynamoDB rejects the entire <code>TransactGetItems</code> request if any of the following is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A conflicting operation is in the process of updating an item to be read.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is insufficient provisioned capacity for the transaction to be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is a user error, such as an invalid data format.
     * </p>
     * </li>
     * <li>
     * <p>
     * The aggregate size of the items in the transaction cannot exceed 4 MB.
     * </p>
     * </li>
     * </ul>
     *
     * @param transactGetItemsRequest
     * @return A Java Future containing the result of the TransactGetItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>TransactionCanceledException The entire transaction request was canceled.</p>
     *         <p>
     *         DynamoDB cancels a <code>TransactWriteItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A condition in one of the condition expressions is not met.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactWriteItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than one action in the <code>TransactWriteItems</code> operation targets the same item.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An item size becomes too large (larger than 400 KB), or a local secondary index (LSI) becomes too large,
     *         or a similar validation error occurs because of changes made by the transaction.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         DynamoDB cancels a <code>TransactGetItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactGetItems</code> operation that conflicts with a concurrent
     *         <code>PutItem</code>, <code>UpdateItem</code>, <code>DeleteItem</code> or <code>TransactWriteItems</code>
     *         request. In this case the <code>TransactGetItems</code> operation fails with a
     *         <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactGetItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If using Java, DynamoDB lists the cancellation reasons on the <code>CancellationReasons</code> property.
     *         This property is not set for other languages. Transaction cancellation reasons are ordered in the order
     *         of requested items, if an item has no error it will have <code>NONE</code> code and <code>Null</code>
     *         message.
     *         </p>
     *         </note>
     *         <p>
     *         Cancellation reason codes and possible error messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         No Errors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>NONE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: <code>null</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Conditional Check Failed:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ConditionalCheckFailed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: The conditional request failed.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Item Collection Size Limit Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ItemCollectionSizeLimitExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Collection size exceeded.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Transaction Conflict:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>TransactionConflict</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Transaction is ongoing for the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned Throughput Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for the table was exceeded. Consider increasing your
     *         provisioning level with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This Message is received when provisioned throughput is exceeded is on a provisioned DynamoDB table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for one or more global secondary indexes of the table was
     *         exceeded. Consider increasing your provisioning level for the under-provisioned global secondary indexes
     *         with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when provisioned throughput is exceeded is on a provisioned GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Throttling Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ThrottlingError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity of your table or index. DynamoDB is automatically scaling your
     *         table or index so please try again shortly. If exceptions persist, check if you have a hot key:
     *         https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand table as DynamoDB is automatically
     *         scaling the table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity for one or more global secondary indexes. DynamoDB is
     *         automatically scaling your index so please try again shortly.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when when writes get throttled on an On-Demand GSI as DynamoDB is automatically
     *         scaling the GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Validation Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ValidationError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more parameter values were invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key beyond allowed size limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key to unsupported type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An operand in the update expression has an incorrect data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Item size to update has exceeded the maximum allowed size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Number overflow. Attempting to store a number with magnitude larger than supported range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Type mismatch for attribute to update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Nesting Levels have exceeded supported limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The document path provided in the update expression is invalid for update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The provided expression refers to an attribute that does not exist in the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li></li>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.TransactGetItems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TransactGetItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TransactGetItemsResponse> transactGetItems(TransactGetItemsRequest transactGetItemsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transactGetItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransactGetItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TransactGetItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TransactGetItemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<TransactGetItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TransactGetItemsRequest, TransactGetItemsResponse>()
                            .withOperationName("TransactGetItems")
                            .withMarshaller(new TransactGetItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(transactGetItemsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = transactGetItemsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<TransactGetItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>TransactWriteItems</code> is a synchronous write operation that groups up to 25 action requests. These
     * actions can target items in different tables, but not in different AWS accounts or Regions, and no two actions
     * can target the same item. For example, you cannot both <code>ConditionCheck</code> and <code>Update</code> the
     * same item. The aggregate size of the items in the transaction cannot exceed 4 MB.
     * </p>
     * <p>
     * The actions are completed atomically so that either all of them succeed, or all of them fail. They are defined by
     * the following objects:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Put</code>  &#x97;   Initiates a <code>PutItem</code> operation to write a new item. This structure
     * specifies the primary key of the item to be written, the name of the table to write it in, an optional condition
     * expression that must be satisfied for the write to succeed, a list of the item's attributes, and a field
     * indicating whether to retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Update</code>  &#x97;   Initiates an <code>UpdateItem</code> operation to update an existing item. This
     * structure specifies the primary key of the item to be updated, the name of the table where it resides, an
     * optional condition expression that must be satisfied for the update to succeed, an expression that defines one or
     * more attributes to be updated, and a field indicating whether to retrieve the item's attributes if the condition
     * is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Delete</code>  &#x97;   Initiates a <code>DeleteItem</code> operation to delete an existing item. This
     * structure specifies the primary key of the item to be deleted, the name of the table where it resides, an
     * optional condition expression that must be satisfied for the deletion to succeed, and a field indicating whether
     * to retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConditionCheck</code>  &#x97;   Applies a condition to an item that is not being modified by the
     * transaction. This structure specifies the primary key of the item to be checked, the name of the table where it
     * resides, a condition expression that must be satisfied for the transaction to succeed, and a field indicating
     * whether to retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * </ul>
     * <p>
     * DynamoDB rejects the entire <code>TransactWriteItems</code> request if any of the following is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A condition in one of the condition expressions is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * An ongoing operation is in the process of updating the same item.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is insufficient provisioned capacity for the transaction to be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * An item size becomes too large (bigger than 400 KB), a local secondary index (LSI) becomes too large, or a
     * similar validation error occurs because of changes made by the transaction.
     * </p>
     * </li>
     * <li>
     * <p>
     * The aggregate size of the items in the transaction exceeds 4 MB.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is a user error, such as an invalid data format.
     * </p>
     * </li>
     * </ul>
     *
     * @param transactWriteItemsRequest
     * @return A Java Future containing the result of the TransactWriteItems operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>TransactionCanceledException The entire transaction request was canceled.</p>
     *         <p>
     *         DynamoDB cancels a <code>TransactWriteItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A condition in one of the condition expressions is not met.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactWriteItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than one action in the <code>TransactWriteItems</code> operation targets the same item.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An item size becomes too large (larger than 400 KB), or a local secondary index (LSI) becomes too large,
     *         or a similar validation error occurs because of changes made by the transaction.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         DynamoDB cancels a <code>TransactGetItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactGetItems</code> operation that conflicts with a concurrent
     *         <code>PutItem</code>, <code>UpdateItem</code>, <code>DeleteItem</code> or <code>TransactWriteItems</code>
     *         request. In this case the <code>TransactGetItems</code> operation fails with a
     *         <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactGetItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If using Java, DynamoDB lists the cancellation reasons on the <code>CancellationReasons</code> property.
     *         This property is not set for other languages. Transaction cancellation reasons are ordered in the order
     *         of requested items, if an item has no error it will have <code>NONE</code> code and <code>Null</code>
     *         message.
     *         </p>
     *         </note>
     *         <p>
     *         Cancellation reason codes and possible error messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         No Errors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>NONE</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: <code>null</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Conditional Check Failed:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ConditionalCheckFailed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: The conditional request failed.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Item Collection Size Limit Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ItemCollectionSizeLimitExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Collection size exceeded.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Transaction Conflict:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>TransactionConflict</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Transaction is ongoing for the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned Throughput Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for the table was exceeded. Consider increasing your
     *         provisioning level with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This Message is received when provisioned throughput is exceeded is on a provisioned DynamoDB table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for one or more global secondary indexes of the table was
     *         exceeded. Consider increasing your provisioning level for the under-provisioned global secondary indexes
     *         with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when provisioned throughput is exceeded is on a provisioned GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Throttling Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ThrottlingError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity of your table or index. DynamoDB is automatically scaling your
     *         table or index so please try again shortly. If exceptions persist, check if you have a hot key:
     *         https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand table as DynamoDB is automatically
     *         scaling the table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity for one or more global secondary indexes. DynamoDB is
     *         automatically scaling your index so please try again shortly.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when when writes get throttled on an On-Demand GSI as DynamoDB is automatically
     *         scaling the GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Validation Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ValidationError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more parameter values were invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key beyond allowed size limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key to unsupported type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An operand in the update expression has an incorrect data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Item size to update has exceeded the maximum allowed size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Number overflow. Attempting to store a number with magnitude larger than supported range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Type mismatch for attribute to update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Nesting Levels have exceeded supported limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The document path provided in the update expression is invalid for update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The provided expression refers to an attribute that does not exist in the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li></li>
     *         <li>TransactionInProgressException The transaction with the given request token is already in progress.</li>
     *         <li>IdempotentParameterMismatchException DynamoDB rejected the request because you retried a request with
     *         a different payload but with an idempotent token that was already used.</li>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.TransactWriteItems
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TransactWriteItems" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<TransactWriteItemsResponse> transactWriteItems(TransactWriteItemsRequest transactWriteItemsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transactWriteItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransactWriteItems");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TransactWriteItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TransactWriteItemsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<TransactWriteItemsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TransactWriteItemsRequest, TransactWriteItemsResponse>()
                            .withOperationName("TransactWriteItems")
                            .withMarshaller(new TransactWriteItemsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(transactWriteItemsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = transactWriteItemsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<TransactWriteItemsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes the association of tags from an Amazon DynamoDB resource. You can call <code>UntagResource</code> up to
     * five times per second, per account.
     * </p>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource")
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(untagResourceRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagResourceRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * <code>UpdateContinuousBackups</code> enables or disables point in time recovery for the specified table. A
     * successful <code>UpdateContinuousBackups</code> call returns the current
     * <code>ContinuousBackupsDescription</code>. Continuous backups are <code>ENABLED</code> on all tables at table
     * creation. If point in time recovery is enabled, <code>PointInTimeRecoveryStatus</code> will be set to ENABLED.
     * </p>
     * <p>
     * Once continuous backups and point in time recovery are enabled, you can restore to any point in time within
     * <code>EarliestRestorableDateTime</code> and <code>LatestRestorableDateTime</code>.
     * </p>
     * <p>
     * <code>LatestRestorableDateTime</code> is typically 5 minutes before the current time. You can restore your table
     * to any point in time during the last 35 days.
     * </p>
     *
     * @param updateContinuousBackupsRequest
     * @return A Java Future containing the result of the UpdateContinuousBackups operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>ContinuousBackupsUnavailableException Backups have not yet been enabled for this table.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateContinuousBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateContinuousBackups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContinuousBackupsResponse> updateContinuousBackups(
            UpdateContinuousBackupsRequest updateContinuousBackupsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContinuousBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContinuousBackups");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContinuousBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateContinuousBackupsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateContinuousBackupsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContinuousBackupsRequest, UpdateContinuousBackupsResponse>()
                            .withOperationName("UpdateContinuousBackups")
                            .withMarshaller(new UpdateContinuousBackupsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateContinuousBackupsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateContinuousBackupsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateContinuousBackupsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status for contributor insights for a specific table or index.
     * </p>
     *
     * @param updateContributorInsightsRequest
     * @return A Java Future containing the result of the UpdateContributorInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateContributorInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateContributorInsightsResponse> updateContributorInsights(
            UpdateContributorInsightsRequest updateContributorInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContributorInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateContributorInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateContributorInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateContributorInsightsRequest, UpdateContributorInsightsResponse>()
                            .withOperationName("UpdateContributorInsights")
                            .withMarshaller(new UpdateContributorInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateContributorInsightsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateContributorInsightsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateContributorInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds or removes replicas in the specified global table. The global table must already exist to be able to use
     * this operation. Any replica to be added must be empty, have the same name as the global table, have the same key
     * schema, have DynamoDB Streams enabled, and have the same provisioned and maximum write capacity units.
     * </p>
     * <note>
     * <p>
     * Although you can use <code>UpdateGlobalTable</code> to add replicas and remove replicas in a single request, for
     * simplicity we recommend that you issue separate requests for adding or removing replicas.
     * </p>
     * </note>
     * <p>
     * If global secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The global secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same provisioned and maximum write capacity units.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGlobalTableRequest
     * @return A Java Future containing the result of the UpdateGlobalTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>GlobalTableNotFoundException The specified global table does not exist.</li>
     *         <li>ReplicaAlreadyExistsException The specified replica is already part of the global table.</li>
     *         <li>ReplicaNotFoundException The specified replica is no longer part of the global table.</li>
     *         <li>TableNotFoundException A source table with the name <code>TableName</code> does not currently exist
     *         within the subscriber's account.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateGlobalTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateGlobalTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGlobalTableResponse> updateGlobalTable(UpdateGlobalTableRequest updateGlobalTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGlobalTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateGlobalTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGlobalTableRequest, UpdateGlobalTableResponse>()
                            .withOperationName("UpdateGlobalTable")
                            .withMarshaller(new UpdateGlobalTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateGlobalTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGlobalTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateGlobalTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates settings for a global table.
     * </p>
     *
     * @param updateGlobalTableSettingsRequest
     * @return A Java Future containing the result of the UpdateGlobalTableSettings operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>GlobalTableNotFoundException The specified global table does not exist.</li>
     *         <li>ReplicaNotFoundException The specified replica is no longer part of the global table.</li>
     *         <li>IndexNotFoundException The operation tried to access a nonexistent index.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateGlobalTableSettings
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateGlobalTableSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateGlobalTableSettingsResponse> updateGlobalTableSettings(
            UpdateGlobalTableSettingsRequest updateGlobalTableSettingsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalTableSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalTableSettings");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateGlobalTableSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateGlobalTableSettingsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateGlobalTableSettingsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateGlobalTableSettingsRequest, UpdateGlobalTableSettingsResponse>()
                            .withOperationName("UpdateGlobalTableSettings")
                            .withMarshaller(new UpdateGlobalTableSettingsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateGlobalTableSettingsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateGlobalTableSettingsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateGlobalTableSettingsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Edits an existing item's attributes, or adds a new item to the table if it does not already exist. You can put,
     * delete, or add attribute values. You can also perform a conditional update on an existing item (insert a new
     * attribute name-value pair if it doesn't exist, or replace an existing name-value pair if it has certain expected
     * attribute values).
     * </p>
     * <p>
     * You can also return the item's attribute values in the same <code>UpdateItem</code> operation using the
     * <code>ReturnValues</code> parameter.
     * </p>
     *
     * @param updateItemRequest
     *        Represents the input of an <code>UpdateItem</code> operation.
     * @return A Java Future containing the result of the UpdateItem operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConditionalCheckFailedException A condition specified in the operation could not be evaluated.</li>
     *         <li>ProvisionedThroughputExceededException Your request rate is too high. The AWS SDKs for DynamoDB
     *         automatically retry requests that receive this exception. Your request is eventually successful, unless
     *         your retry queue is too large to finish. Reduce the frequency of requests and use exponential backoff.
     *         For more information, go to <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>ItemCollectionSizeLimitExceededException An item collection is too large. This exception is only
     *         returned for tables that have one or more local secondary indexes.</li>
     *         <li>TransactionConflictException Operation was rejected because there is an ongoing transaction for the
     *         item.</li>
     *         <li>RequestLimitExceededException Throughput exceeds the current throughput quota for your account.
     *         Please contact AWS Support at <a href="https://aws.amazon.com/support">AWS Support</a> to request a quota
     *         increase.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateItem
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateItemResponse> updateItem(UpdateItemRequest updateItemRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateItem");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateItemResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateItemResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateItemRequest, UpdateItemResponse>().withOperationName("UpdateItem")
                            .withMarshaller(new UpdateItemRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateItemRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateItemRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateItemResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given
     * table.
     * </p>
     * <p>
     * You can only perform one of the following operations at once:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Modify the provisioned throughput settings of the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Enable or disable DynamoDB Streams on the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove a global secondary index from the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create a new global secondary index on the table. After the index begins backfilling, you can use
     * <code>UpdateTable</code> to perform other operations.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>UpdateTable</code> is an asynchronous operation; while it is executing, the table status changes from
     * <code>ACTIVE</code> to <code>UPDATING</code>. While it is <code>UPDATING</code>, you cannot issue another
     * <code>UpdateTable</code> request. When the table returns to the <code>ACTIVE</code> state, the
     * <code>UpdateTable</code> operation is complete.
     * </p>
     *
     * @param updateTableRequest
     *        Represents the input of an <code>UpdateTable</code> operation.
     * @return A Java Future containing the result of the UpdateTable operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateTable
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableResponse> updateTable(UpdateTableRequest updateTableRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UpdateTableResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateTableResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                            .withOperationName("UpdateTable").withMarshaller(new UpdateTableRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateTableRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTableRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateTableResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates auto scaling settings on your global tables at once.
     * </p>
     * <note>
     * <p>
     * This operation only applies to <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.V2.html">Version
     * 2019.11.21</a> of global tables.
     * </p>
     * </note>
     *
     * @param updateTableReplicaAutoScalingRequest
     * @return A Java Future containing the result of the UpdateTableReplicaAutoScaling operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateTableReplicaAutoScaling
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTableReplicaAutoScaling"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTableReplicaAutoScalingResponse> updateTableReplicaAutoScaling(
            UpdateTableReplicaAutoScalingRequest updateTableReplicaAutoScalingRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateTableReplicaAutoScalingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTableReplicaAutoScaling");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTableReplicaAutoScalingResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTableReplicaAutoScalingResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateTableReplicaAutoScalingResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTableReplicaAutoScalingRequest, UpdateTableReplicaAutoScalingResponse>()
                            .withOperationName("UpdateTableReplicaAutoScaling")
                            .withMarshaller(new UpdateTableReplicaAutoScalingRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateTableReplicaAutoScalingRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTableReplicaAutoScalingRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateTableReplicaAutoScalingResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * The <code>UpdateTimeToLive</code> method enables or disables Time to Live (TTL) for the specified table. A
     * successful <code>UpdateTimeToLive</code> call returns the current <code>TimeToLiveSpecification</code>. It can
     * take up to one hour for the change to fully process. Any additional <code>UpdateTimeToLive</code> calls for the
     * same table during this one hour duration result in a <code>ValidationException</code>.
     * </p>
     * <p>
     * TTL compares the current time in epoch time format to the time stored in the TTL attribute of an item. If the
     * epoch time value stored in the attribute is less than the current time, the item is marked as expired and
     * subsequently deleted.
     * </p>
     * <note>
     * <p>
     * The epoch time format is the number of seconds elapsed since 12:00:00 AM January 1, 1970 UTC.
     * </p>
     * </note>
     * <p>
     * DynamoDB deletes expired items on a best-effort basis to ensure availability of throughput for other data
     * operations.
     * </p>
     * <important>
     * <p>
     * DynamoDB typically deletes expired items within two days of expiration. The exact duration within which an item
     * gets deleted after expiration is specific to the nature of the workload. Items that have expired and not been
     * deleted will still show up in reads, queries, and scans.
     * </p>
     * </important>
     * <p>
     * As items are deleted, they are removed from any local secondary index and global secondary index immediately in
     * the same eventually consistent way as a standard delete operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/TTL.html">Time To Live</a> in the Amazon
     * DynamoDB Developer Guide.
     * </p>
     *
     * @param updateTimeToLiveRequest
     *        Represents the input of an <code>UpdateTimeToLive</code> operation.
     * @return A Java Future containing the result of the UpdateTimeToLive operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceInUseException The operation conflicts with the resource's availability. For example, you
     *         attempted to recreate an existing table, or tried to delete a table currently in the
     *         <code>CREATING</code> state.</li>
     *         <li>ResourceNotFoundException The operation tried to access a nonexistent table or index. The resource
     *         might not be specified correctly, or its status might not be <code>ACTIVE</code>.</li>
     *         <li>LimitExceededException There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         Up to 50 simultaneous table operations are allowed per account. These operations include
     *         <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         The only exception is when you are creating a table with one or more secondary indexes. You can have up
     *         to 25 such requests running at a time; however, if the table or index specifications are complex,
     *         DynamoDB might temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         There is a soft account quota of 256 tables.</li>
     *         <li>InternalServerErrorException An error occurred on the server side.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DynamoDbException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DynamoDbAsyncClient.UpdateTimeToLive
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTimeToLive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateTimeToLiveResponse> updateTimeToLive(UpdateTimeToLiveRequest updateTimeToLiveRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTimeToLiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTimeToLive");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateTimeToLiveResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateTimeToLiveResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);
            boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
            boolean endpointOverridden = clientConfiguration.option(SdkClientOption.ENDPOINT_OVERRIDDEN) == Boolean.TRUE;
            URI cachedEndpoint = null;
            if (endpointDiscoveryEnabled) {

                String key = clientConfiguration.option(AwsClientOption.CREDENTIALS_PROVIDER).resolveCredentials().accessKeyId();
                EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                        .defaultEndpoint(clientConfiguration.option(SdkClientOption.ENDPOINT)).build();
                cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
            }

            CompletableFuture<UpdateTimeToLiveResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateTimeToLiveRequest, UpdateTimeToLiveResponse>()
                            .withOperationName("UpdateTimeToLive")
                            .withMarshaller(new UpdateTimeToLiveRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).discoveredEndpoint(cachedEndpoint)
                            .withInput(updateTimeToLiveRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateTimeToLiveRequest.overrideConfiguration().orElse(null);
            CompletableFuture<UpdateTimeToLiveResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DynamoDbException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestLimitExceeded")
                                .exceptionBuilderSupplier(RequestLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlobalTableAlreadyExistsException")
                                .exceptionBuilderSupplier(GlobalTableAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                                .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlobalTableNotFoundException")
                                .exceptionBuilderSupplier(GlobalTableNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ItemCollectionSizeLimitExceededException")
                                .exceptionBuilderSupplier(ItemCollectionSizeLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicaNotFoundException")
                                .exceptionBuilderSupplier(ReplicaNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupInUseException")
                                .exceptionBuilderSupplier(BackupInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContinuousBackupsUnavailableException")
                                .exceptionBuilderSupplier(ContinuousBackupsUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExportNotFoundException")
                                .exceptionBuilderSupplier(ExportNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionInProgressException")
                                .exceptionBuilderSupplier(TransactionInProgressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableInUseException")
                                .exceptionBuilderSupplier(TableInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                                .exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PointInTimeRecoveryUnavailableException")
                                .exceptionBuilderSupplier(PointInTimeRecoveryUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableAlreadyExistsException")
                                .exceptionBuilderSupplier(TableAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExportConflictException")
                                .exceptionBuilderSupplier(ExportConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionConflictException")
                                .exceptionBuilderSupplier(TransactionConflictException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRestoreTimeException")
                                .exceptionBuilderSupplier(InvalidRestoreTimeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicaAlreadyExistsException")
                                .exceptionBuilderSupplier(ReplicaAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupNotFoundException")
                                .exceptionBuilderSupplier(BackupNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IndexNotFoundException")
                                .exceptionBuilderSupplier(IndexNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableNotFoundException")
                                .exceptionBuilderSupplier(TableNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionCanceledException")
                                .exceptionBuilderSupplier(TransactionCanceledException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExportTimeException")
                                .exceptionBuilderSupplier(InvalidExportTimeException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DynamoDbRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    @Override
    public DynamoDbAsyncWaiter waiter() {
        return DynamoDbAsyncWaiter.builder().client(this).scheduledExecutorService(executorService).build();
    }
}
