/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class ExecuteStatementRequest extends DynamoDbRequest implements
        ToCopyableBuilder<ExecuteStatementRequest.Builder, ExecuteStatementRequest> {
    private static final SdkField<String> STATEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Statement").getter(getter(ExecuteStatementRequest::statement)).setter(setter(Builder::statement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statement").build()).build();

    private static final SdkField<List<AttributeValue>> PARAMETERS_FIELD = SdkField
            .<List<AttributeValue>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(ExecuteStatementRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> CONSISTENT_READ_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ConsistentRead").getter(getter(ExecuteStatementRequest::consistentRead))
            .setter(setter(Builder::consistentRead))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsistentRead").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("NextToken").getter(getter(ExecuteStatementRequest::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATEMENT_FIELD,
            PARAMETERS_FIELD, CONSISTENT_READ_FIELD, NEXT_TOKEN_FIELD));

    private final String statement;

    private final List<AttributeValue> parameters;

    private final Boolean consistentRead;

    private final String nextToken;

    private ExecuteStatementRequest(BuilderImpl builder) {
        super(builder);
        this.statement = builder.statement;
        this.parameters = builder.parameters;
        this.consistentRead = builder.consistentRead;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * The PartiQL statement representing the operation to run.
     * </p>
     * 
     * @return The PartiQL statement representing the operation to run.
     */
    public final String statement() {
        return statement;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parameters for the PartiQL statement, if any.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The parameters for the PartiQL statement, if any.
     */
    public final List<AttributeValue> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The consistency of a read operation. If set to <code>true</code>, then a strongly consistent read is used;
     * otherwise, an eventually consistent read is used.
     * </p>
     * 
     * @return The consistency of a read operation. If set to <code>true</code>, then a strongly consistent read is
     *         used; otherwise, an eventually consistent read is used.
     */
    public final Boolean consistentRead() {
        return consistentRead;
    }

    /**
     * <p>
     * Set this value to get remaining results, if <code>NextToken</code> was returned in the statement response.
     * </p>
     * 
     * @return Set this value to get remaining results, if <code>NextToken</code> was returned in the statement
     *         response.
     */
    public final String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(statement());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(consistentRead());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ExecuteStatementRequest)) {
            return false;
        }
        ExecuteStatementRequest other = (ExecuteStatementRequest) obj;
        return Objects.equals(statement(), other.statement()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(consistentRead(), other.consistentRead())
                && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ExecuteStatementRequest").add("Statement", statement())
                .add("Parameters", hasParameters() ? parameters() : null).add("ConsistentRead", consistentRead())
                .add("NextToken", nextToken()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Statement":
            return Optional.ofNullable(clazz.cast(statement()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "ConsistentRead":
            return Optional.ofNullable(clazz.cast(consistentRead()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ExecuteStatementRequest, T> g) {
        return obj -> g.apply((ExecuteStatementRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends DynamoDbRequest.Builder, SdkPojo, CopyableBuilder<Builder, ExecuteStatementRequest> {
        /**
         * <p>
         * The PartiQL statement representing the operation to run.
         * </p>
         * 
         * @param statement
         *        The PartiQL statement representing the operation to run.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statement(String statement);

        /**
         * <p>
         * The parameters for the PartiQL statement, if any.
         * </p>
         * 
         * @param parameters
         *        The parameters for the PartiQL statement, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<AttributeValue> parameters);

        /**
         * <p>
         * The parameters for the PartiQL statement, if any.
         * </p>
         * 
         * @param parameters
         *        The parameters for the PartiQL statement, if any.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(AttributeValue... parameters);

        /**
         * <p>
         * The parameters for the PartiQL statement, if any.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeValue>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeValue>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeValue>.Builder#build()} is called immediately and
         * its result is passed to {@link #parameters(List<AttributeValue>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<AttributeValue>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<AttributeValue>)
         */
        Builder parameters(Consumer<AttributeValue.Builder>... parameters);

        /**
         * <p>
         * The consistency of a read operation. If set to <code>true</code>, then a strongly consistent read is used;
         * otherwise, an eventually consistent read is used.
         * </p>
         * 
         * @param consistentRead
         *        The consistency of a read operation. If set to <code>true</code>, then a strongly consistent read is
         *        used; otherwise, an eventually consistent read is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consistentRead(Boolean consistentRead);

        /**
         * <p>
         * Set this value to get remaining results, if <code>NextToken</code> was returned in the statement response.
         * </p>
         * 
         * @param nextToken
         *        Set this value to get remaining results, if <code>NextToken</code> was returned in the statement
         *        response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends DynamoDbRequest.BuilderImpl implements Builder {
        private String statement;

        private List<AttributeValue> parameters = DefaultSdkAutoConstructList.getInstance();

        private Boolean consistentRead;

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(ExecuteStatementRequest model) {
            super(model);
            statement(model.statement);
            parameters(model.parameters);
            consistentRead(model.consistentRead);
            nextToken(model.nextToken);
        }

        public final String getStatement() {
            return statement;
        }

        @Override
        public final Builder statement(String statement) {
            this.statement = statement;
            return this;
        }

        public final void setStatement(String statement) {
            this.statement = statement;
        }

        public final Collection<AttributeValue.Builder> getParameters() {
            if (parameters instanceof SdkAutoConstructList) {
                return null;
            }
            return parameters != null ? parameters.stream().map(AttributeValue::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<AttributeValue> parameters) {
            this.parameters = PreparedStatementParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(AttributeValue... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<AttributeValue.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> AttributeValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<AttributeValue.BuilderImpl> parameters) {
            this.parameters = PreparedStatementParametersCopier.copyFromBuilder(parameters);
        }

        public final Boolean getConsistentRead() {
            return consistentRead;
        }

        @Override
        public final Builder consistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
            return this;
        }

        public final void setConsistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public ExecuteStatementRequest build() {
            return new ExecuteStatementRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
