/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * For the <code>UpdateItem</code> operation, represents the attributes to be modified, the action to perform on each,
 * and the new value for each.
 * </p>
 * <note>
 * <p>
 * You cannot use <code>UpdateItem</code> to update any primary key attributes. Instead, you will need to delete the
 * item, and then use <code>PutItem</code> to create a new item with new attributes.
 * </p>
 * </note>
 * <p>
 * Attribute values cannot be null; string and binary type attributes must have lengths greater than zero; and set type
 * attributes must not be empty. Requests with empty values will be rejected with a <code>ValidationException</code>
 * exception.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AttributeValueUpdate implements SdkPojo, Serializable,
        ToCopyableBuilder<AttributeValueUpdate.Builder, AttributeValueUpdate> {
    private static final SdkField<AttributeValue> VALUE_FIELD = SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
            .memberName("Value").getter(getter(AttributeValueUpdate::value)).setter(setter(Builder::value))
            .constructor(AttributeValue::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Value").build()).build();

    private static final SdkField<String> ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Action")
            .getter(getter(AttributeValueUpdate::actionAsString)).setter(setter(Builder::action))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Action").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VALUE_FIELD, ACTION_FIELD));

    private static final long serialVersionUID = 1L;

    private final AttributeValue value;

    private final String action;

    private AttributeValueUpdate(BuilderImpl builder) {
        this.value = builder.value;
        this.action = builder.action;
    }

    /**
     * <p>
     * Represents the data for an attribute.
     * </p>
     * <p>
     * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
     * itself.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
     * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * 
     * @return Represents the data for an attribute.</p>
     *         <p>
     *         Each attribute value is described as a name-value pair. The name is the data type, and the value is the
     *         data itself.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
     *         >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     */
    public final AttributeValue value() {
        return value;
    }

    /**
     * <p>
     * Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>, and
     * <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the table.
     * </p>
     * <p>
     * <b>If an item with the specified <i>Key</i> is found in the table:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is replaced by
     * the new value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item. The data
     * type of the specified value must match the existing value's data type.
     * </p>
     * <p>
     * If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example, if the
     * attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified <code>[a,c]</code>,
     * then the final attribute value would be <code>[b]</code>. Specifying an empty set is an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added to the
     * item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type of the
     * attribute:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the existing attribute is a number, and if <code>Value</code> is also a number, then the <code>Value</code> is
     * mathematically added to the existing attribute. If <code>Value</code> is a negative number, then it is subtracted
     * from the existing attribute.
     * </p>
     * <note>
     * <p>
     * If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist before the
     * update, DynamoDB uses 0 as the initial value.
     * </p>
     * <p>
     * In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement an
     * attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For example, suppose
     * that the item you want to update does not yet have an attribute named <i>itemcount</i>, but you decide to
     * <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it currently does not exist.
     * DynamoDB will create the <i>itemcount</i> attribute, set its initial value to <code>0</code>, and finally add
     * <code>3</code> to it. The result will be a new <i>itemcount</i> attribute in the item, with a value of
     * <code>3</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * If the existing data type is a set, and if the <code>Value</code> is also a set, then the <code>Value</code> is
     * added to the existing set. (This is a <i>set</i> operation, not mathematical addition.) For example, if the
     * attribute value was the set <code>[1,2]</code>, and the <code>ADD</code> action specified <code>[3]</code>, then
     * the final attribute value would be <code>[1,2,3]</code>. An error occurs if an Add action is specified for a set
     * attribute and the attribute type specified does not match the existing set type.
     * </p>
     * <p>
     * Both sets must have the same primitive data type. For example, if the existing data type is a set of strings, the
     * <code>Value</code> must also be a set of strings. The same holds true for number sets and binary sets.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This action is only valid for an existing attribute whose data type is number or is a set. Do not use
     * <code>ADD</code> for any other data types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>If no item with the specified <i>Key</i> is found:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE</code> - Nothing happens; there is no attribute to delete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers) for the
     * attribute value. The only data types allowed are number and number set; no other data types can be specified.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link AttributeAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>,
     *         and <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the
     *         table.</p>
     *         <p>
     *         <b>If an item with the specified <i>Key</i> is found in the table:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is
     *         replaced by the new value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item.
     *         The data type of the specified value must match the existing value's data type.
     *         </p>
     *         <p>
     *         If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example,
     *         if the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified
     *         <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an empty set is
     *         an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added
     *         to the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type
     *         of the attribute:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the existing attribute is a number, and if <code>Value</code> is also a number, then the
     *         <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a negative
     *         number, then it is subtracted from the existing attribute.
     *         </p>
     *         <note>
     *         <p>
     *         If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist
     *         before the update, DynamoDB uses 0 as the initial value.
     *         </p>
     *         <p>
     *         In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement
     *         an attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For
     *         example, suppose that the item you want to update does not yet have an attribute named <i>itemcount</i>,
     *         but you decide to <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it
     *         currently does not exist. DynamoDB will create the <i>itemcount</i> attribute, set its initial value to
     *         <code>0</code>, and finally add <code>3</code> to it. The result will be a new <i>itemcount</i> attribute
     *         in the item, with a value of <code>3</code>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         If the existing data type is a set, and if the <code>Value</code> is also a set, then the
     *         <code>Value</code> is added to the existing set. (This is a <i>set</i> operation, not mathematical
     *         addition.) For example, if the attribute value was the set <code>[1,2]</code>, and the <code>ADD</code>
     *         action specified <code>[3]</code>, then the final attribute value would be <code>[1,2,3]</code>. An error
     *         occurs if an Add action is specified for a set attribute and the attribute type specified does not match
     *         the existing set type.
     *         </p>
     *         <p>
     *         Both sets must have the same primitive data type. For example, if the existing data type is a set of
     *         strings, the <code>Value</code> must also be a set of strings. The same holds true for number sets and
     *         binary sets.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This action is only valid for an existing attribute whose data type is number or is a set. Do not use
     *         <code>ADD</code> for any other data types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>If no item with the specified <i>Key</i> is found:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the
     *         attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE</code> - Nothing happens; there is no attribute to delete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers)
     *         for the attribute value. The only data types allowed are number and number set; no other data types can
     *         be specified.
     *         </p>
     *         </li>
     * @see AttributeAction
     */
    public final AttributeAction action() {
        return AttributeAction.fromValue(action);
    }

    /**
     * <p>
     * Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>, and
     * <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the table.
     * </p>
     * <p>
     * <b>If an item with the specified <i>Key</i> is found in the table:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is replaced by
     * the new value.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item. The data
     * type of the specified value must match the existing value's data type.
     * </p>
     * <p>
     * If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example, if the
     * attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified <code>[a,c]</code>,
     * then the final attribute value would be <code>[b]</code>. Specifying an empty set is an error.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added to the
     * item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type of the
     * attribute:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the existing attribute is a number, and if <code>Value</code> is also a number, then the <code>Value</code> is
     * mathematically added to the existing attribute. If <code>Value</code> is a negative number, then it is subtracted
     * from the existing attribute.
     * </p>
     * <note>
     * <p>
     * If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist before the
     * update, DynamoDB uses 0 as the initial value.
     * </p>
     * <p>
     * In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement an
     * attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For example, suppose
     * that the item you want to update does not yet have an attribute named <i>itemcount</i>, but you decide to
     * <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it currently does not exist.
     * DynamoDB will create the <i>itemcount</i> attribute, set its initial value to <code>0</code>, and finally add
     * <code>3</code> to it. The result will be a new <i>itemcount</i> attribute in the item, with a value of
     * <code>3</code>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * If the existing data type is a set, and if the <code>Value</code> is also a set, then the <code>Value</code> is
     * added to the existing set. (This is a <i>set</i> operation, not mathematical addition.) For example, if the
     * attribute value was the set <code>[1,2]</code>, and the <code>ADD</code> action specified <code>[3]</code>, then
     * the final attribute value would be <code>[1,2,3]</code>. An error occurs if an Add action is specified for a set
     * attribute and the attribute type specified does not match the existing set type.
     * </p>
     * <p>
     * Both sets must have the same primitive data type. For example, if the existing data type is a set of strings, the
     * <code>Value</code> must also be a set of strings. The same holds true for number sets and binary sets.
     * </p>
     * </li>
     * </ul>
     * <p>
     * This action is only valid for an existing attribute whose data type is number or is a set. Do not use
     * <code>ADD</code> for any other data types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <b>If no item with the specified <i>Key</i> is found:</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the attribute.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETE</code> - Nothing happens; there is no attribute to delete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers) for the
     * attribute value. The only data types allowed are number and number set; no other data types can be specified.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #action} will
     * return {@link AttributeAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #actionAsString}.
     * </p>
     * 
     * @return Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>,
     *         and <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the
     *         table.</p>
     *         <p>
     *         <b>If an item with the specified <i>Key</i> is found in the table:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is
     *         replaced by the new value.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item.
     *         The data type of the specified value must match the existing value's data type.
     *         </p>
     *         <p>
     *         If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example,
     *         if the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified
     *         <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an empty set is
     *         an error.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added
     *         to the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type
     *         of the attribute:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If the existing attribute is a number, and if <code>Value</code> is also a number, then the
     *         <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a negative
     *         number, then it is subtracted from the existing attribute.
     *         </p>
     *         <note>
     *         <p>
     *         If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist
     *         before the update, DynamoDB uses 0 as the initial value.
     *         </p>
     *         <p>
     *         In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement
     *         an attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For
     *         example, suppose that the item you want to update does not yet have an attribute named <i>itemcount</i>,
     *         but you decide to <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it
     *         currently does not exist. DynamoDB will create the <i>itemcount</i> attribute, set its initial value to
     *         <code>0</code>, and finally add <code>3</code> to it. The result will be a new <i>itemcount</i> attribute
     *         in the item, with a value of <code>3</code>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         If the existing data type is a set, and if the <code>Value</code> is also a set, then the
     *         <code>Value</code> is added to the existing set. (This is a <i>set</i> operation, not mathematical
     *         addition.) For example, if the attribute value was the set <code>[1,2]</code>, and the <code>ADD</code>
     *         action specified <code>[3]</code>, then the final attribute value would be <code>[1,2,3]</code>. An error
     *         occurs if an Add action is specified for a set attribute and the attribute type specified does not match
     *         the existing set type.
     *         </p>
     *         <p>
     *         Both sets must have the same primitive data type. For example, if the existing data type is a set of
     *         strings, the <code>Value</code> must also be a set of strings. The same holds true for number sets and
     *         binary sets.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         This action is only valid for an existing attribute whose data type is number or is a set. Do not use
     *         <code>ADD</code> for any other data types.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         <b>If no item with the specified <i>Key</i> is found:</b>
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the
     *         attribute.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETE</code> - Nothing happens; there is no attribute to delete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers)
     *         for the attribute value. The only data types allowed are number and number set; no other data types can
     *         be specified.
     *         </p>
     *         </li>
     * @see AttributeAction
     */
    public final String actionAsString() {
        return action;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(value());
        hashCode = 31 * hashCode + Objects.hashCode(actionAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AttributeValueUpdate)) {
            return false;
        }
        AttributeValueUpdate other = (AttributeValueUpdate) obj;
        return Objects.equals(value(), other.value()) && Objects.equals(actionAsString(), other.actionAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AttributeValueUpdate").add("Value", value()).add("Action", actionAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Value":
            return Optional.ofNullable(clazz.cast(value()));
        case "Action":
            return Optional.ofNullable(clazz.cast(actionAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AttributeValueUpdate, T> g) {
        return obj -> g.apply((AttributeValueUpdate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AttributeValueUpdate> {
        /**
         * <p>
         * Represents the data for an attribute.
         * </p>
         * <p>
         * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
         * itself.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param value
         *        Represents the data for an attribute.</p>
         *        <p>
         *        Each attribute value is described as a name-value pair. The name is the data type, and the value is
         *        the data itself.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         *        >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(AttributeValue value);

        /**
         * <p>
         * Represents the data for an attribute.
         * </p>
         * <p>
         * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data
         * itself.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
         * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * This is a convenience that creates an instance of the {@link AttributeValue.Builder} avoiding the need to
         * create one manually via {@link AttributeValue#builder()}.
         *
         * When the {@link Consumer} completes, {@link AttributeValue.Builder#build()} is called immediately and its
         * result is passed to {@link #value(AttributeValue)}.
         * 
         * @param value
         *        a consumer that will call methods on {@link AttributeValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #value(AttributeValue)
         */
        default Builder value(Consumer<AttributeValue.Builder> value) {
            return value(AttributeValue.builder().applyMutation(value).build());
        }

        /**
         * <p>
         * Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>, and
         * <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the table.
         * </p>
         * <p>
         * <b>If an item with the specified <i>Key</i> is found in the table:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is replaced
         * by the new value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item. The
         * data type of the specified value must match the existing value's data type.
         * </p>
         * <p>
         * If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example, if
         * the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified
         * <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an empty set is an
         * error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added to
         * the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type of the
         * attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the existing attribute is a number, and if <code>Value</code> is also a number, then the
         * <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a negative
         * number, then it is subtracted from the existing attribute.
         * </p>
         * <note>
         * <p>
         * If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist before
         * the update, DynamoDB uses 0 as the initial value.
         * </p>
         * <p>
         * In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement an
         * attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For example,
         * suppose that the item you want to update does not yet have an attribute named <i>itemcount</i>, but you
         * decide to <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it currently does
         * not exist. DynamoDB will create the <i>itemcount</i> attribute, set its initial value to <code>0</code>, and
         * finally add <code>3</code> to it. The result will be a new <i>itemcount</i> attribute in the item, with a
         * value of <code>3</code>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * If the existing data type is a set, and if the <code>Value</code> is also a set, then the <code>Value</code>
         * is added to the existing set. (This is a <i>set</i> operation, not mathematical addition.) For example, if
         * the attribute value was the set <code>[1,2]</code>, and the <code>ADD</code> action specified
         * <code>[3]</code>, then the final attribute value would be <code>[1,2,3]</code>. An error occurs if an Add
         * action is specified for a set attribute and the attribute type specified does not match the existing set
         * type.
         * </p>
         * <p>
         * Both sets must have the same primitive data type. For example, if the existing data type is a set of strings,
         * the <code>Value</code> must also be a set of strings. The same holds true for number sets and binary sets.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This action is only valid for an existing attribute whose data type is number or is a set. Do not use
         * <code>ADD</code> for any other data types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>If no item with the specified <i>Key</i> is found:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE</code> - Nothing happens; there is no attribute to delete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers) for
         * the attribute value. The only data types allowed are number and number set; no other data types can be
         * specified.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>,
         *        and <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the
         *        table.</p>
         *        <p>
         *        <b>If an item with the specified <i>Key</i> is found in the table:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is
         *        replaced by the new value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item.
         *        The data type of the specified value must match the existing value's data type.
         *        </p>
         *        <p>
         *        If a <i>set</i> of values is specified, then those values are subtracted from the old set. For
         *        example, if the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action
         *        specified <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an
         *        empty set is an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are
         *        added to the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the
         *        data type of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the existing attribute is a number, and if <code>Value</code> is also a number, then the
         *        <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a
         *        negative number, then it is subtracted from the existing attribute.
         *        </p>
         *        <note>
         *        <p>
         *        If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist
         *        before the update, DynamoDB uses 0 as the initial value.
         *        </p>
         *        <p>
         *        In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or
         *        decrement an attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial
         *        value. For example, suppose that the item you want to update does not yet have an attribute named
         *        <i>itemcount</i>, but you decide to <code>ADD</code> the number <code>3</code> to this attribute
         *        anyway, even though it currently does not exist. DynamoDB will create the <i>itemcount</i> attribute,
         *        set its initial value to <code>0</code>, and finally add <code>3</code> to it. The result will be a
         *        new <i>itemcount</i> attribute in the item, with a value of <code>3</code>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        If the existing data type is a set, and if the <code>Value</code> is also a set, then the
         *        <code>Value</code> is added to the existing set. (This is a <i>set</i> operation, not mathematical
         *        addition.) For example, if the attribute value was the set <code>[1,2]</code>, and the
         *        <code>ADD</code> action specified <code>[3]</code>, then the final attribute value would be
         *        <code>[1,2,3]</code>. An error occurs if an Add action is specified for a set attribute and the
         *        attribute type specified does not match the existing set type.
         *        </p>
         *        <p>
         *        Both sets must have the same primitive data type. For example, if the existing data type is a set of
         *        strings, the <code>Value</code> must also be a set of strings. The same holds true for number sets and
         *        binary sets.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This action is only valid for an existing attribute whose data type is number or is a set. Do not use
         *        <code>ADD</code> for any other data types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>If no item with the specified <i>Key</i> is found:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the
         *        attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE</code> - Nothing happens; there is no attribute to delete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of
         *        numbers) for the attribute value. The only data types allowed are number and number set; no other data
         *        types can be specified.
         *        </p>
         *        </li>
         * @see AttributeAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttributeAction
         */
        Builder action(String action);

        /**
         * <p>
         * Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>, and
         * <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the table.
         * </p>
         * <p>
         * <b>If an item with the specified <i>Key</i> is found in the table:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is replaced
         * by the new value.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item. The
         * data type of the specified value must match the existing value's data type.
         * </p>
         * <p>
         * If a <i>set</i> of values is specified, then those values are subtracted from the old set. For example, if
         * the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action specified
         * <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an empty set is an
         * error.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are added to
         * the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the data type of the
         * attribute:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If the existing attribute is a number, and if <code>Value</code> is also a number, then the
         * <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a negative
         * number, then it is subtracted from the existing attribute.
         * </p>
         * <note>
         * <p>
         * If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist before
         * the update, DynamoDB uses 0 as the initial value.
         * </p>
         * <p>
         * In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or decrement an
         * attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial value. For example,
         * suppose that the item you want to update does not yet have an attribute named <i>itemcount</i>, but you
         * decide to <code>ADD</code> the number <code>3</code> to this attribute anyway, even though it currently does
         * not exist. DynamoDB will create the <i>itemcount</i> attribute, set its initial value to <code>0</code>, and
         * finally add <code>3</code> to it. The result will be a new <i>itemcount</i> attribute in the item, with a
         * value of <code>3</code>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * If the existing data type is a set, and if the <code>Value</code> is also a set, then the <code>Value</code>
         * is added to the existing set. (This is a <i>set</i> operation, not mathematical addition.) For example, if
         * the attribute value was the set <code>[1,2]</code>, and the <code>ADD</code> action specified
         * <code>[3]</code>, then the final attribute value would be <code>[1,2,3]</code>. An error occurs if an Add
         * action is specified for a set attribute and the attribute type specified does not match the existing set
         * type.
         * </p>
         * <p>
         * Both sets must have the same primitive data type. For example, if the existing data type is a set of strings,
         * the <code>Value</code> must also be a set of strings. The same holds true for number sets and binary sets.
         * </p>
         * </li>
         * </ul>
         * <p>
         * This action is only valid for an existing attribute whose data type is number or is a set. Do not use
         * <code>ADD</code> for any other data types.
         * </p>
         * </li>
         * </ul>
         * <p>
         * <b>If no item with the specified <i>Key</i> is found:</b>
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the attribute.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETE</code> - Nothing happens; there is no attribute to delete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of numbers) for
         * the attribute value. The only data types allowed are number and number set; no other data types can be
         * specified.
         * </p>
         * </li>
         * </ul>
         * 
         * @param action
         *        Specifies how to perform the update. Valid values are <code>PUT</code> (default), <code>DELETE</code>,
         *        and <code>ADD</code>. The behavior depends on whether the specified primary key already exists in the
         *        table.</p>
         *        <p>
         *        <b>If an item with the specified <i>Key</i> is found in the table:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PUT</code> - Adds the specified attribute to the item. If the attribute already exists, it is
         *        replaced by the new value.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE</code> - If no value is specified, the attribute and its value are removed from the item.
         *        The data type of the specified value must match the existing value's data type.
         *        </p>
         *        <p>
         *        If a <i>set</i> of values is specified, then those values are subtracted from the old set. For
         *        example, if the attribute value was the set <code>[a,b,c]</code> and the <code>DELETE</code> action
         *        specified <code>[a,c]</code>, then the final attribute value would be <code>[b]</code>. Specifying an
         *        empty set is an error.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADD</code> - If the attribute does not already exist, then the attribute and its values are
         *        added to the item. If the attribute does exist, then the behavior of <code>ADD</code> depends on the
         *        data type of the attribute:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If the existing attribute is a number, and if <code>Value</code> is also a number, then the
         *        <code>Value</code> is mathematically added to the existing attribute. If <code>Value</code> is a
         *        negative number, then it is subtracted from the existing attribute.
         *        </p>
         *        <note>
         *        <p>
         *        If you use <code>ADD</code> to increment or decrement a number value for an item that doesn't exist
         *        before the update, DynamoDB uses 0 as the initial value.
         *        </p>
         *        <p>
         *        In addition, if you use <code>ADD</code> to update an existing item, and intend to increment or
         *        decrement an attribute value which does not yet exist, DynamoDB uses <code>0</code> as the initial
         *        value. For example, suppose that the item you want to update does not yet have an attribute named
         *        <i>itemcount</i>, but you decide to <code>ADD</code> the number <code>3</code> to this attribute
         *        anyway, even though it currently does not exist. DynamoDB will create the <i>itemcount</i> attribute,
         *        set its initial value to <code>0</code>, and finally add <code>3</code> to it. The result will be a
         *        new <i>itemcount</i> attribute in the item, with a value of <code>3</code>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        If the existing data type is a set, and if the <code>Value</code> is also a set, then the
         *        <code>Value</code> is added to the existing set. (This is a <i>set</i> operation, not mathematical
         *        addition.) For example, if the attribute value was the set <code>[1,2]</code>, and the
         *        <code>ADD</code> action specified <code>[3]</code>, then the final attribute value would be
         *        <code>[1,2,3]</code>. An error occurs if an Add action is specified for a set attribute and the
         *        attribute type specified does not match the existing set type.
         *        </p>
         *        <p>
         *        Both sets must have the same primitive data type. For example, if the existing data type is a set of
         *        strings, the <code>Value</code> must also be a set of strings. The same holds true for number sets and
         *        binary sets.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        This action is only valid for an existing attribute whose data type is number or is a set. Do not use
         *        <code>ADD</code> for any other data types.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        <b>If no item with the specified <i>Key</i> is found:</b>
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PUT</code> - DynamoDB creates a new item with the specified primary key, and then adds the
         *        attribute.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETE</code> - Nothing happens; there is no attribute to delete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ADD</code> - DynamoDB creates an item with the supplied primary key and number (or set of
         *        numbers) for the attribute value. The only data types allowed are number and number set; no other data
         *        types can be specified.
         *        </p>
         *        </li>
         * @see AttributeAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AttributeAction
         */
        Builder action(AttributeAction action);
    }

    static final class BuilderImpl implements Builder {
        private AttributeValue value;

        private String action;

        private BuilderImpl() {
        }

        private BuilderImpl(AttributeValueUpdate model) {
            value(model.value);
            action(model.action);
        }

        public final AttributeValue.Builder getValue() {
            return value != null ? value.toBuilder() : null;
        }

        public final void setValue(AttributeValue.BuilderImpl value) {
            this.value = value != null ? value.build() : null;
        }

        @Override
        @Transient
        public final Builder value(AttributeValue value) {
            this.value = value;
            return this;
        }

        public final String getAction() {
            return action;
        }

        public final void setAction(String action) {
            this.action = action;
        }

        @Override
        @Transient
        public final Builder action(String action) {
            this.action = action;
            return this;
        }

        @Override
        @Transient
        public final Builder action(AttributeAction action) {
            this.action(action == null ? null : action.toString());
            return this;
        }

        @Override
        public AttributeValueUpdate build() {
            return new AttributeValueUpdate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
