/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the selection criteria for a <code>Query</code> or <code>Scan</code> operation:
 * </p>
 * <ul>
 * <li>
 * <p>
 * For a <code>Query</code> operation, <code>Condition</code> is used for specifying the <code>KeyConditions</code> to
 * use when querying a table or an index. For <code>KeyConditions</code>, only the following comparison operators are
 * supported:
 * </p>
 * <p>
 * <code>EQ | LE | LT | GE | GT | BEGINS_WITH | BETWEEN</code>
 * </p>
 * <p>
 * <code>Condition</code> is also used in a <code>QueryFilter</code>, which evaluates the query results and returns only
 * the desired values.
 * </p>
 * </li>
 * <li>
 * <p>
 * For a <code>Scan</code> operation, <code>Condition</code> is used in a <code>ScanFilter</code>, which evaluates the
 * scan results and returns only the desired values.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Condition implements SdkPojo, Serializable, ToCopyableBuilder<Condition.Builder, Condition> {
    private static final SdkField<List<AttributeValue>> ATTRIBUTE_VALUE_LIST_FIELD = SdkField
            .<List<AttributeValue>> builder(MarshallingType.LIST)
            .memberName("AttributeValueList")
            .getter(getter(Condition::attributeValueList))
            .setter(setter(Builder::attributeValueList))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttributeValueList").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> COMPARISON_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ComparisonOperator").getter(getter(Condition::comparisonOperatorAsString))
            .setter(setter(Builder::comparisonOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ComparisonOperator").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ATTRIBUTE_VALUE_LIST_FIELD,
            COMPARISON_OPERATOR_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<AttributeValue> attributeValueList;

    private final String comparisonOperator;

    private Condition(BuilderImpl builder) {
        this.attributeValueList = builder.attributeValueList;
        this.comparisonOperator = builder.comparisonOperator;
    }

    /**
     * Returns true if the AttributeValueList property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasAttributeValueList() {
        return attributeValueList != null && !(attributeValueList instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more values to evaluate against the supplied attribute. The number of values in the list depends on the
     * <code>ComparisonOperator</code> being used.
     * </p>
     * <p>
     * For type Number, value comparisons are numeric.
     * </p>
     * <p>
     * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
     * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>. For a
     * list of code values, see <a
     * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://en.wikipedia
     * .org/wiki/ASCII#ASCII_printable_characters</a>.
     * </p>
     * <p>
     * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAttributeValueList()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more values to evaluate against the supplied attribute. The number of values in the list depends
     *         on the <code>ComparisonOperator</code> being used.</p>
     *         <p>
     *         For type Number, value comparisons are numeric.
     *         </p>
     *         <p>
     *         String value comparisons for greater than, equals, or less than are based on ASCII character code values.
     *         For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
     *         <code>B</code>. For a list of code values, see <a
     *         href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
     *         >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
     *         </p>
     *         <p>
     *         For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
     */
    public final List<AttributeValue> attributeValueList() {
        return attributeValueList;
    }

    /**
     * <p>
     * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
     * </p>
     * <p>
     * The following comparison operators are available:
     * </p>
     * <p>
     * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     * </p>
     * <p>
     * The following are descriptions of each comparison operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> element of a
     * different type than the one provided in the request, the value does not match. For example,
     * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     * <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, Binary,
     * String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a different type
     * than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not
     * equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LE</code> : Less than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LT</code> : Less than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
     * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than the
     * one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GE</code> : Greater than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GT</code> : Greater than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not relevant to
     * the <code>NOT_NULL</code> comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean <code>false</code>.
     * This is because the attribute "<code>a</code>" exists; its data type is not relevant to the <code>NULL</code>
     * comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is of type String, then the operator checks
     * for a substring match. If the target attribute of the comparison is of type Binary, then the operator looks for a
     * subsequence of the target that matches the input. If the target attribute of the comparison is a set ("
     * <code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it finds an
     * exact match with any member of the set.
     * </p>
     * <p>
     * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is a String, then the operator checks for
     * the absence of a substring match. If the target attribute of the comparison is Binary, then the operator checks
     * for the absence of a subsequence of the target that matches the input. If the target attribute of the comparison
     * is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it
     * <i>does not</i> find an exact match with any member of the set.
     * </p>
     * <p>
     * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BEGINS_WITH</code> : Checks for a prefix.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary (not a
     * Number or a set type). The target attribute of the comparison must be of type String or Binary (not a Number or a
     * set type).
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>IN</code> : Checks for matching elements in a list.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
     * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of the
     * input are equal to the item attribute, the expression evaluates to true.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
     * </p>
     * <p>
     * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type, either
     * String, Number, or Binary (not a set type). A target attribute matches if the target value is greater than, or
     * equal to, the first element and less than, or equal to, the second element. If an item contains an
     * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does not
     * match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
     * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html">Legacy
     * Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
     *         <p>
     *         The following comparison operators are available:
     *         </p>
     *         <p>
     *         <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     *         </p>
     *         <p>
     *         The following are descriptions of each comparison operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
     *         element of a different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LE</code> : Less than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LT</code> : Less than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
     *         type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code>
     *         does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GE</code> : Greater than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GT</code> : Greater than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     *         <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     *         <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
     *         relevant to the <code>NOT_NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the nonexistence of an attribute, not its data type. If the data type of
     *         attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
     *         <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant
     *         to the <code>NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
     *         operator checks for a substring match. If the target attribute of the comparison is of type Binary, then
     *         the operator looks for a subsequence of the target that matches the input. If the target attribute of the
     *         comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator
     *         evaluates to true if it finds an exact match with any member of the set.
     *         </p>
     *         <p>
     *         CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the
     *         operator checks for the absence of a substring match. If the target attribute of the comparison is
     *         Binary, then the operator checks for the absence of a subsequence of the target that matches the input.
     *         If the target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "
     *         <code>BS</code>"), then the operator evaluates to true if it <i>does not</i> find an exact match with any
     *         member of the set.
     *         </p>
     *         <p>
     *         NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BEGINS_WITH</code> : Checks for a prefix.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
     *         (not a Number or a set type). The target attribute of the comparison must be of type String or Binary
     *         (not a Number or a set type).
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>IN</code> : Checks for matching elements in a list.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
     *         String, Number, or Binary. These attributes are compared against an existing attribute of an item. If any
     *         elements of the input are equal to the item attribute, the expression evaluates to true.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
     *         value.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
     *         either String, Number, or Binary (not a set type). A target attribute matches if the target value is
     *         greater than, or equal to, the first element and less than, or equal to, the second element. If an item
     *         contains an <code>AttributeValue</code> element of a different type than the one provided in the request,
     *         the value does not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>.
     *         Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
     *         >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @see ComparisonOperator
     */
    public final ComparisonOperator comparisonOperator() {
        return ComparisonOperator.fromValue(comparisonOperator);
    }

    /**
     * <p>
     * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
     * </p>
     * <p>
     * The following comparison operators are available:
     * </p>
     * <p>
     * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     * </p>
     * <p>
     * The following are descriptions of each comparison operator.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> element of a
     * different type than the one provided in the request, the value does not match. For example,
     * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     * <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, Binary,
     * String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a different type
     * than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not
     * equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LE</code> : Less than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>LT</code> : Less than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
     * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than the
     * one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GE</code> : Greater than or equal.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>GT</code> : Greater than.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
     * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
     * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not relevant to
     * the <code>NOT_NULL</code> comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types, including
     * lists and maps.
     * </p>
     * <note>
     * <p>
     * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
     * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean <code>false</code>.
     * This is because the attribute "<code>a</code>" exists; its data type is not relevant to the <code>NULL</code>
     * comparison operator.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is of type String, then the operator checks
     * for a substring match. If the target attribute of the comparison is of type Binary, then the operator looks for a
     * subsequence of the target that matches the input. If the target attribute of the comparison is a set ("
     * <code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it finds an
     * exact match with any member of the set.
     * </p>
     * <p>
     * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String, Number,
     * or Binary (not a set type). If the target attribute of the comparison is a String, then the operator checks for
     * the absence of a substring match. If the target attribute of the comparison is Binary, then the operator checks
     * for the absence of a subsequence of the target that matches the input. If the target attribute of the comparison
     * is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it
     * <i>does not</i> find an exact match with any member of the set.
     * </p>
     * <p>
     * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BEGINS_WITH</code> : Checks for a prefix.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary (not a
     * Number or a set type). The target attribute of the comparison must be of type String or Binary (not a Number or a
     * set type).
     * </p>
     * <p/></li>
     * <li>
     * <p>
     * <code>IN</code> : Checks for matching elements in a list.
     * </p>
     * <p>
     * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
     * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of the
     * input are equal to the item attribute, the expression evaluates to true.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
     * </p>
     * <p>
     * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type, either
     * String, Number, or Binary (not a set type). A target attribute matches if the target value is greater than, or
     * equal to, the first element and less than, or equal to, the second element. If an item contains an
     * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does not
     * match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
     * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html">Legacy
     * Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #comparisonOperator} will return {@link ComparisonOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #comparisonOperatorAsString}.
     * </p>
     * 
     * @return A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
     *         <p>
     *         The following comparison operators are available:
     *         </p>
     *         <p>
     *         <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
     *         </p>
     *         <p>
     *         The following are descriptions of each comparison operator.
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
     *         element of a different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LE</code> : Less than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>LT</code> : Less than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
     *         or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
     *         type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code>
     *         does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
     *         <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GE</code> : Greater than or equal.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>GT</code> : Greater than.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
     *         different type than the one provided in the request, the value does not match. For example,
     *         <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
     *         compare to <code>{"NS":["6", "2", "1"]}</code>.
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
     *         <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
     *         <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
     *         relevant to the <code>NOT_NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
     *         including lists and maps.
     *         </p>
     *         <note>
     *         <p>
     *         This operator tests for the nonexistence of an attribute, not its data type. If the data type of
     *         attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
     *         <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant
     *         to the <code>NULL</code> comparison operator.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
     *         operator checks for a substring match. If the target attribute of the comparison is of type Binary, then
     *         the operator looks for a subsequence of the target that matches the input. If the target attribute of the
     *         comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator
     *         evaluates to true if it finds an exact match with any member of the set.
     *         </p>
     *         <p>
     *         CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
     *         Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the
     *         operator checks for the absence of a substring match. If the target attribute of the comparison is
     *         Binary, then the operator checks for the absence of a subsequence of the target that matches the input.
     *         If the target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "
     *         <code>BS</code>"), then the operator evaluates to true if it <i>does not</i> find an exact match with any
     *         member of the set.
     *         </p>
     *         <p>
     *         NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
     *         " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BEGINS_WITH</code> : Checks for a prefix.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
     *         (not a Number or a set type). The target attribute of the comparison must be of type String or Binary
     *         (not a Number or a set type).
     *         </p>
     *         <p/></li>
     *         <li>
     *         <p>
     *         <code>IN</code> : Checks for matching elements in a list.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
     *         String, Number, or Binary. These attributes are compared against an existing attribute of an item. If any
     *         elements of the input are equal to the item attribute, the expression evaluates to true.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
     *         value.
     *         </p>
     *         <p>
     *         <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
     *         either String, Number, or Binary (not a set type). A target attribute matches if the target value is
     *         greater than, or equal to, the first element and less than, or equal to, the second element. If an item
     *         contains an <code>AttributeValue</code> element of a different type than the one provided in the request,
     *         the value does not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>.
     *         Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
     *         href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
     *         >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @see ComparisonOperator
     */
    public final String comparisonOperatorAsString() {
        return comparisonOperator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasAttributeValueList() ? attributeValueList() : null);
        hashCode = 31 * hashCode + Objects.hashCode(comparisonOperatorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Condition)) {
            return false;
        }
        Condition other = (Condition) obj;
        return hasAttributeValueList() == other.hasAttributeValueList()
                && Objects.equals(attributeValueList(), other.attributeValueList())
                && Objects.equals(comparisonOperatorAsString(), other.comparisonOperatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Condition").add("AttributeValueList", hasAttributeValueList() ? attributeValueList() : null)
                .add("ComparisonOperator", comparisonOperatorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AttributeValueList":
            return Optional.ofNullable(clazz.cast(attributeValueList()));
        case "ComparisonOperator":
            return Optional.ofNullable(clazz.cast(comparisonOperatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Condition, T> g) {
        return obj -> g.apply((Condition) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Condition> {
        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(Collection<AttributeValue> attributeValueList);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * 
         * @param attributeValueList
         *        One or more values to evaluate against the supplied attribute. The number of values in the list
         *        depends on the <code>ComparisonOperator</code> being used.</p>
         *        <p>
         *        For type Number, value comparisons are numeric.
         *        </p>
         *        <p>
         *        String value comparisons for greater than, equals, or less than are based on ASCII character code
         *        values. For example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than
         *        <code>B</code>. For a list of code values, see <a
         *        href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters"
         *        >http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         *        </p>
         *        <p>
         *        For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder attributeValueList(AttributeValue... attributeValueList);

        /**
         * <p>
         * One or more values to evaluate against the supplied attribute. The number of values in the list depends on
         * the <code>ComparisonOperator</code> being used.
         * </p>
         * <p>
         * For type Number, value comparisons are numeric.
         * </p>
         * <p>
         * String value comparisons for greater than, equals, or less than are based on ASCII character code values. For
         * example, <code>a</code> is greater than <code>A</code>, and <code>a</code> is greater than <code>B</code>.
         * For a list of code values, see <a
         * href="http://en.wikipedia.org/wiki/ASCII#ASCII_printable_characters">http://
         * en.wikipedia.org/wiki/ASCII#ASCII_printable_characters</a>.
         * </p>
         * <p>
         * For Binary, DynamoDB treats each byte of the binary data as unsigned when it compares binary values.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeValue>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeValue>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeValue>.Builder#build()} is called immediately and
         * its result is passed to {@link #attributeValueList(List<AttributeValue>)}.
         * 
         * @param attributeValueList
         *        a consumer that will call methods on {@link List<AttributeValue>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #attributeValueList(List<AttributeValue>)
         */
        Builder attributeValueList(Consumer<AttributeValue.Builder>... attributeValueList);

        /**
         * <p>
         * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         * >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         *        href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         *        >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(String comparisonOperator);

        /**
         * <p>
         * A comparator for evaluating attributes. For example, equals, greater than, less than, etc.
         * </p>
         * <p>
         * The following comparison operators are available:
         * </p>
         * <p>
         * <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         * </p>
         * <p>
         * The following are descriptions of each comparison operator.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code>
         * element of a different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and maps.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number,
         * Binary, String Set, Number Set, or Binary Set. If an item contains an <code>AttributeValue</code> of a
         * different type than the one provided in the request, the value does not match. For example,
         * <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not equal
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LE</code> : Less than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>LT</code> : Less than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String, Number, or
         * Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different type than
         * the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does not equal
         * <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GE</code> : Greater than or equal.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>GT</code> : Greater than.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a different
         * type than the one provided in the request, the value does not match. For example, <code>{"S":"6"}</code> does
         * not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         * <code>{"NS":["6", "2", "1"]}</code>.
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the existence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a Boolean
         * <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type is not
         * relevant to the <code>NOT_NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         * including lists and maps.
         * </p>
         * <note>
         * <p>
         * This operator tests for the nonexistence of an attribute, not its data type. If the data type of attribute "
         * <code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a Boolean
         * <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is not relevant to
         * the <code>NULL</code> comparison operator.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is of type String, then the
         * operator checks for a substring match. If the target attribute of the comparison is of type Binary, then the
         * operator looks for a subsequence of the target that matches the input. If the target attribute of the
         * comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the operator evaluates
         * to true if it finds an exact match with any member of the set.
         * </p>
         * <p>
         * CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type String,
         * Number, or Binary (not a set type). If the target attribute of the comparison is a String, then the operator
         * checks for the absence of a substring match. If the target attribute of the comparison is Binary, then the
         * operator checks for the absence of a subsequence of the target that matches the input. If the target
         * attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>"), then the
         * operator evaluates to true if it <i>does not</i> find an exact match with any member of the set.
         * </p>
         * <p>
         * NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         * " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BEGINS_WITH</code> : Checks for a prefix.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or Binary
         * (not a Number or a set type). The target attribute of the comparison must be of type String or Binary (not a
         * Number or a set type).
         * </p>
         * <p/></li>
         * <li>
         * <p>
         * <code>IN</code> : Checks for matching elements in a list.
         * </p>
         * <p>
         * <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type String,
         * Number, or Binary. These attributes are compared against an existing attribute of an item. If any elements of
         * the input are equal to the item attribute, the expression evaluates to true.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second value.
         * </p>
         * <p>
         * <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same type,
         * either String, Number, or Binary (not a set type). A target attribute matches if the target value is greater
         * than, or equal to, the first element and less than, or equal to, the second element. If an item contains an
         * <code>AttributeValue</code> element of a different type than the one provided in the request, the value does
         * not match. For example, <code>{"S":"6"}</code> does not compare to <code>{"N":"6"}</code>. Also,
         * <code>{"N":"6"}</code> does not compare to <code>{"NS":["6", "2", "1"]}</code>
         * </p>
         * </li>
         * </ul>
         * <p>
         * For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         * >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * </p>
         * 
         * @param comparisonOperator
         *        A comparator for evaluating attributes. For example, equals, greater than, less than, etc.</p>
         *        <p>
         *        The following comparison operators are available:
         *        </p>
         *        <p>
         *        <code>EQ | NE | LE | LT | GE | GT | NOT_NULL | NULL | CONTAINS | NOT_CONTAINS | BEGINS_WITH | IN | BETWEEN</code>
         *        </p>
         *        <p>
         *        The following are descriptions of each comparison operator.
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>EQ</code> : Equal. <code>EQ</code> is supported for all data types, including lists and maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> element of a different type than the one provided in the request, the
         *        value does not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NE</code> : Not equal. <code>NE</code> is supported for all data types, including lists and
         *        maps.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, Binary, String Set, Number Set, or Binary Set. If an item contains an
         *        <code>AttributeValue</code> of a different type than the one provided in the request, the value does
         *        not match. For example, <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also,
         *        <code>{"N":"6"}</code> does not equal <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LE</code> : Less than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>LT</code> : Less than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String,
         *        Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element of a
         *        different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GE</code> : Greater than or equal.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>GT</code> : Greater than.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If an item contains an <code>AttributeValue</code> element
         *        of a different type than the one provided in the request, the value does not match. For example,
         *        <code>{"S":"6"}</code> does not equal <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not
         *        compare to <code>{"NS":["6", "2", "1"]}</code>.
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>NOT_NULL</code> : The attribute exists. <code>NOT_NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the existence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NOT_NULL</code>, the result is a
         *        Boolean <code>true</code>. This result is because the attribute "<code>a</code>" exists; its data type
         *        is not relevant to the <code>NOT_NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>NULL</code> : The attribute does not exist. <code>NULL</code> is supported for all data types,
         *        including lists and maps.
         *        </p>
         *        <note>
         *        <p>
         *        This operator tests for the nonexistence of an attribute, not its data type. If the data type of
         *        attribute "<code>a</code>" is null, and you evaluate it using <code>NULL</code>, the result is a
         *        Boolean <code>false</code>. This is because the attribute "<code>a</code>" exists; its data type is
         *        not relevant to the <code>NULL</code> comparison operator.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>CONTAINS</code> : Checks for a subsequence, or value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is of type
         *        String, then the operator checks for a substring match. If the target attribute of the comparison is
         *        of type Binary, then the operator looks for a subsequence of the target that matches the input. If the
         *        target attribute of the comparison is a set ("<code>SS</code>", "<code>NS</code>", or "<code>BS</code>
         *        "), then the operator evaluates to true if it finds an exact match with any member of the set.
         *        </p>
         *        <p>
         *        CONTAINS is supported for lists: When evaluating "<code>a CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NOT_CONTAINS</code> : Checks for absence of a subsequence, or absence of a value in a set.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> element of type
         *        String, Number, or Binary (not a set type). If the target attribute of the comparison is a String,
         *        then the operator checks for the absence of a substring match. If the target attribute of the
         *        comparison is Binary, then the operator checks for the absence of a subsequence of the target that
         *        matches the input. If the target attribute of the comparison is a set ("<code>SS</code>", "
         *        <code>NS</code>", or "<code>BS</code>"), then the operator evaluates to true if it <i>does not</i>
         *        find an exact match with any member of the set.
         *        </p>
         *        <p>
         *        NOT_CONTAINS is supported for lists: When evaluating "<code>a NOT CONTAINS b</code>", "<code>a</code>
         *        " can be a list; however, "<code>b</code>" cannot be a set, a map, or a list.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BEGINS_WITH</code> : Checks for a prefix.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain only one <code>AttributeValue</code> of type String or
         *        Binary (not a Number or a set type). The target attribute of the comparison must be of type String or
         *        Binary (not a Number or a set type).
         *        </p>
         *        <p/></li>
         *        <li>
         *        <p>
         *        <code>IN</code> : Checks for matching elements in a list.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> can contain one or more <code>AttributeValue</code> elements of type
         *        String, Number, or Binary. These attributes are compared against an existing attribute of an item. If
         *        any elements of the input are equal to the item attribute, the expression evaluates to true.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BETWEEN</code> : Greater than or equal to the first value, and less than or equal to the second
         *        value.
         *        </p>
         *        <p>
         *        <code>AttributeValueList</code> must contain two <code>AttributeValue</code> elements of the same
         *        type, either String, Number, or Binary (not a set type). A target attribute matches if the target
         *        value is greater than, or equal to, the first element and less than, or equal to, the second element.
         *        If an item contains an <code>AttributeValue</code> element of a different type than the one provided
         *        in the request, the value does not match. For example, <code>{"S":"6"}</code> does not compare to
         *        <code>{"N":"6"}</code>. Also, <code>{"N":"6"}</code> does not compare to
         *        <code>{"NS":["6", "2", "1"]}</code>
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For usage examples of <code>AttributeValueList</code> and <code>ComparisonOperator</code>, see <a
         *        href=
         *        "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/LegacyConditionalParameters.html"
         *        >Legacy Conditional Parameters</a> in the <i>Amazon DynamoDB Developer Guide</i>.
         * @see ComparisonOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ComparisonOperator
         */
        Builder comparisonOperator(ComparisonOperator comparisonOperator);
    }

    static final class BuilderImpl implements Builder {
        private List<AttributeValue> attributeValueList = DefaultSdkAutoConstructList.getInstance();

        private String comparisonOperator;

        private BuilderImpl() {
        }

        private BuilderImpl(Condition model) {
            attributeValueList(model.attributeValueList);
            comparisonOperator(model.comparisonOperator);
        }

        public final List<AttributeValue.Builder> getAttributeValueList() {
            List<AttributeValue.Builder> result = AttributeValueListCopier.copyToBuilder(this.attributeValueList);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAttributeValueList(Collection<AttributeValue.BuilderImpl> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copyFromBuilder(attributeValueList);
        }

        @Override
        @Transient
        public final Builder attributeValueList(Collection<AttributeValue> attributeValueList) {
            this.attributeValueList = AttributeValueListCopier.copy(attributeValueList);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributeValueList(AttributeValue... attributeValueList) {
            attributeValueList(Arrays.asList(attributeValueList));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder attributeValueList(Consumer<AttributeValue.Builder>... attributeValueList) {
            attributeValueList(Stream.of(attributeValueList).map(c -> AttributeValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getComparisonOperator() {
            return comparisonOperator;
        }

        public final void setComparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
        }

        @Override
        @Transient
        public final Builder comparisonOperator(String comparisonOperator) {
            this.comparisonOperator = comparisonOperator;
            return this;
        }

        @Override
        @Transient
        public final Builder comparisonOperator(ComparisonOperator comparisonOperator) {
            this.comparisonOperator(comparisonOperator == null ? null : comparisonOperator.toString());
            return this;
        }

        @Override
        public Condition build() {
            return new Condition(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
