/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about item collections, if any, that were affected by the operation. <code>ItemCollectionMetrics</code>
 * is only returned if the request asked for it. If the table does not have any local secondary indexes, this
 * information is not returned in the response.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ItemCollectionMetrics implements SdkPojo, Serializable,
        ToCopyableBuilder<ItemCollectionMetrics.Builder, ItemCollectionMetrics> {
    private static final SdkField<Map<String, AttributeValue>> ITEM_COLLECTION_KEY_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .memberName("ItemCollectionKey")
            .getter(getter(ItemCollectionMetrics::itemCollectionKey))
            .setter(setter(Builder::itemCollectionKey))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ItemCollectionKey").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<Double>> SIZE_ESTIMATE_RANGE_GB_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("SizeEstimateRangeGB")
            .getter(getter(ItemCollectionMetrics::sizeEstimateRangeGB))
            .setter(setter(Builder::sizeEstimateRangeGB))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SizeEstimateRangeGB").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ITEM_COLLECTION_KEY_FIELD,
            SIZE_ESTIMATE_RANGE_GB_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, AttributeValue> itemCollectionKey;

    private final List<Double> sizeEstimateRangeGB;

    private ItemCollectionMetrics(BuilderImpl builder) {
        this.itemCollectionKey = builder.itemCollectionKey;
        this.sizeEstimateRangeGB = builder.sizeEstimateRangeGB;
    }

    /**
     * Returns true if the ItemCollectionKey property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasItemCollectionKey() {
        return itemCollectionKey != null && !(itemCollectionKey instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * The partition key value of the item collection. This value is the same as the partition key value of the item.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasItemCollectionKey()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The partition key value of the item collection. This value is the same as the partition key value of the
     *         item.
     */
    public final Map<String, AttributeValue> itemCollectionKey() {
        return itemCollectionKey;
    }

    /**
     * Returns true if the SizeEstimateRangeGB property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasSizeEstimateRangeGB() {
        return sizeEstimateRangeGB != null && !(sizeEstimateRangeGB instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound and
     * an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the size of
     * all attributes projected into all of the local secondary indexes on that table. Use this estimate to measure
     * whether a local secondary index is approaching its size limit.
     * </p>
     * <p>
     * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the estimate.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSizeEstimateRangeGB()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower
     *         bound and an upper bound for the estimate. The estimate includes the size of all the items in the table,
     *         plus the size of all attributes projected into all of the local secondary indexes on that table. Use this
     *         estimate to measure whether a local secondary index is approaching its size limit.</p>
     *         <p>
     *         The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
     *         estimate.
     */
    public final List<Double> sizeEstimateRangeGB() {
        return sizeEstimateRangeGB;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasItemCollectionKey() ? itemCollectionKey() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasSizeEstimateRangeGB() ? sizeEstimateRangeGB() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ItemCollectionMetrics)) {
            return false;
        }
        ItemCollectionMetrics other = (ItemCollectionMetrics) obj;
        return hasItemCollectionKey() == other.hasItemCollectionKey()
                && Objects.equals(itemCollectionKey(), other.itemCollectionKey())
                && hasSizeEstimateRangeGB() == other.hasSizeEstimateRangeGB()
                && Objects.equals(sizeEstimateRangeGB(), other.sizeEstimateRangeGB());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ItemCollectionMetrics")
                .add("ItemCollectionKey", hasItemCollectionKey() ? itemCollectionKey() : null)
                .add("SizeEstimateRangeGB", hasSizeEstimateRangeGB() ? sizeEstimateRangeGB() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ItemCollectionKey":
            return Optional.ofNullable(clazz.cast(itemCollectionKey()));
        case "SizeEstimateRangeGB":
            return Optional.ofNullable(clazz.cast(sizeEstimateRangeGB()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ItemCollectionMetrics, T> g) {
        return obj -> g.apply((ItemCollectionMetrics) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ItemCollectionMetrics> {
        /**
         * <p>
         * The partition key value of the item collection. This value is the same as the partition key value of the
         * item.
         * </p>
         * 
         * @param itemCollectionKey
         *        The partition key value of the item collection. This value is the same as the partition key value of
         *        the item.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCollectionKey(Map<String, AttributeValue> itemCollectionKey);

        /**
         * <p>
         * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound
         * and an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the
         * size of all attributes projected into all of the local secondary indexes on that table. Use this estimate to
         * measure whether a local secondary index is approaching its size limit.
         * </p>
         * <p>
         * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
         * estimate.
         * </p>
         * 
         * @param sizeEstimateRangeGB
         *        An estimate of item collection size, in gigabytes. This value is a two-element array containing a
         *        lower bound and an upper bound for the estimate. The estimate includes the size of all the items in
         *        the table, plus the size of all attributes projected into all of the local secondary indexes on that
         *        table. Use this estimate to measure whether a local secondary index is approaching its size limit.</p>
         *        <p>
         *        The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of
         *        the estimate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB);

        /**
         * <p>
         * An estimate of item collection size, in gigabytes. This value is a two-element array containing a lower bound
         * and an upper bound for the estimate. The estimate includes the size of all the items in the table, plus the
         * size of all attributes projected into all of the local secondary indexes on that table. Use this estimate to
         * measure whether a local secondary index is approaching its size limit.
         * </p>
         * <p>
         * The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of the
         * estimate.
         * </p>
         * 
         * @param sizeEstimateRangeGB
         *        An estimate of item collection size, in gigabytes. This value is a two-element array containing a
         *        lower bound and an upper bound for the estimate. The estimate includes the size of all the items in
         *        the table, plus the size of all attributes projected into all of the local secondary indexes on that
         *        table. Use this estimate to measure whether a local secondary index is approaching its size limit.</p>
         *        <p>
         *        The estimate is subject to change over time; therefore, do not rely on the precision or accuracy of
         *        the estimate.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sizeEstimateRangeGB(Double... sizeEstimateRangeGB);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, AttributeValue> itemCollectionKey = DefaultSdkAutoConstructMap.getInstance();

        private List<Double> sizeEstimateRangeGB = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ItemCollectionMetrics model) {
            itemCollectionKey(model.itemCollectionKey);
            sizeEstimateRangeGB(model.sizeEstimateRangeGB);
        }

        public final Map<String, AttributeValue.Builder> getItemCollectionKey() {
            Map<String, AttributeValue.Builder> result = ItemCollectionKeyAttributeMapCopier
                    .copyToBuilder(this.itemCollectionKey);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setItemCollectionKey(Map<String, AttributeValue.BuilderImpl> itemCollectionKey) {
            this.itemCollectionKey = ItemCollectionKeyAttributeMapCopier.copyFromBuilder(itemCollectionKey);
        }

        @Override
        @Transient
        public final Builder itemCollectionKey(Map<String, AttributeValue> itemCollectionKey) {
            this.itemCollectionKey = ItemCollectionKeyAttributeMapCopier.copy(itemCollectionKey);
            return this;
        }

        public final Collection<Double> getSizeEstimateRangeGB() {
            if (sizeEstimateRangeGB instanceof SdkAutoConstructList) {
                return null;
            }
            return sizeEstimateRangeGB;
        }

        public final void setSizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB) {
            this.sizeEstimateRangeGB = ItemCollectionSizeEstimateRangeCopier.copy(sizeEstimateRangeGB);
        }

        @Override
        @Transient
        public final Builder sizeEstimateRangeGB(Collection<Double> sizeEstimateRangeGB) {
            this.sizeEstimateRangeGB = ItemCollectionSizeEstimateRangeCopier.copy(sizeEstimateRangeGB);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder sizeEstimateRangeGB(Double... sizeEstimateRangeGB) {
            sizeEstimateRangeGB(Arrays.asList(sizeEstimateRangeGB));
            return this;
        }

        @Override
        public ItemCollectionMetrics build() {
            return new ItemCollectionMetrics(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
