/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A PartiQL batch statement request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class BatchStatementRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<BatchStatementRequest.Builder, BatchStatementRequest> {
    private static final SdkField<String> STATEMENT_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Statement").getter(getter(BatchStatementRequest::statement)).setter(setter(Builder::statement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statement").build()).build();

    private static final SdkField<List<AttributeValue>> PARAMETERS_FIELD = SdkField
            .<List<AttributeValue>> builder(MarshallingType.LIST)
            .memberName("Parameters")
            .getter(getter(BatchStatementRequest::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> CONSISTENT_READ_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ConsistentRead").getter(getter(BatchStatementRequest::consistentRead))
            .setter(setter(Builder::consistentRead))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ConsistentRead").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STATEMENT_FIELD,
            PARAMETERS_FIELD, CONSISTENT_READ_FIELD));

    private static final long serialVersionUID = 1L;

    private final String statement;

    private final List<AttributeValue> parameters;

    private final Boolean consistentRead;

    private BatchStatementRequest(BuilderImpl builder) {
        this.statement = builder.statement;
        this.parameters = builder.parameters;
        this.consistentRead = builder.consistentRead;
    }

    /**
     * <p>
     * A valid PartiQL statement.
     * </p>
     * 
     * @return A valid PartiQL statement.
     */
    public final String statement() {
        return statement;
    }

    /**
     * For responses, this returns true if the service returned a value for the Parameters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The parameters associated with a PartiQL statement in the batch request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasParameters} method.
     * </p>
     * 
     * @return The parameters associated with a PartiQL statement in the batch request.
     */
    public final List<AttributeValue> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The read consistency of the PartiQL batch request.
     * </p>
     * 
     * @return The read consistency of the PartiQL batch request.
     */
    public final Boolean consistentRead() {
        return consistentRead;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(statement());
        hashCode = 31 * hashCode + Objects.hashCode(hasParameters() ? parameters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(consistentRead());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof BatchStatementRequest)) {
            return false;
        }
        BatchStatementRequest other = (BatchStatementRequest) obj;
        return Objects.equals(statement(), other.statement()) && hasParameters() == other.hasParameters()
                && Objects.equals(parameters(), other.parameters()) && Objects.equals(consistentRead(), other.consistentRead());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("BatchStatementRequest").add("Statement", statement())
                .add("Parameters", hasParameters() ? parameters() : null).add("ConsistentRead", consistentRead()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Statement":
            return Optional.ofNullable(clazz.cast(statement()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "ConsistentRead":
            return Optional.ofNullable(clazz.cast(consistentRead()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<BatchStatementRequest, T> g) {
        return obj -> g.apply((BatchStatementRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, BatchStatementRequest> {
        /**
         * <p>
         * A valid PartiQL statement.
         * </p>
         * 
         * @param statement
         *        A valid PartiQL statement.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statement(String statement);

        /**
         * <p>
         * The parameters associated with a PartiQL statement in the batch request.
         * </p>
         * 
         * @param parameters
         *        The parameters associated with a PartiQL statement in the batch request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<AttributeValue> parameters);

        /**
         * <p>
         * The parameters associated with a PartiQL statement in the batch request.
         * </p>
         * 
         * @param parameters
         *        The parameters associated with a PartiQL statement in the batch request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(AttributeValue... parameters);

        /**
         * <p>
         * The parameters associated with a PartiQL statement in the batch request.
         * </p>
         * This is a convenience that creates an instance of the {@link List<AttributeValue>.Builder} avoiding the need
         * to create one manually via {@link List<AttributeValue>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<AttributeValue>.Builder#build()} is called immediately and
         * its result is passed to {@link #parameters(List<AttributeValue>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<AttributeValue>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<AttributeValue>)
         */
        Builder parameters(Consumer<AttributeValue.Builder>... parameters);

        /**
         * <p>
         * The read consistency of the PartiQL batch request.
         * </p>
         * 
         * @param consistentRead
         *        The read consistency of the PartiQL batch request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder consistentRead(Boolean consistentRead);
    }

    static final class BuilderImpl implements Builder {
        private String statement;

        private List<AttributeValue> parameters = DefaultSdkAutoConstructList.getInstance();

        private Boolean consistentRead;

        private BuilderImpl() {
        }

        private BuilderImpl(BatchStatementRequest model) {
            statement(model.statement);
            parameters(model.parameters);
            consistentRead(model.consistentRead);
        }

        public final String getStatement() {
            return statement;
        }

        public final void setStatement(String statement) {
            this.statement = statement;
        }

        @Override
        @Transient
        public final Builder statement(String statement) {
            this.statement = statement;
            return this;
        }

        public final List<AttributeValue.Builder> getParameters() {
            List<AttributeValue.Builder> result = PreparedStatementParametersCopier.copyToBuilder(this.parameters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setParameters(Collection<AttributeValue.BuilderImpl> parameters) {
            this.parameters = PreparedStatementParametersCopier.copyFromBuilder(parameters);
        }

        @Override
        @Transient
        public final Builder parameters(Collection<AttributeValue> parameters) {
            this.parameters = PreparedStatementParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder parameters(AttributeValue... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder parameters(Consumer<AttributeValue.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> AttributeValue.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getConsistentRead() {
            return consistentRead;
        }

        public final void setConsistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
        }

        @Override
        @Transient
        public final Builder consistentRead(Boolean consistentRead) {
            this.consistentRead = consistentRead;
            return this;
        }

        @Override
        public BatchStatementRequest build() {
            return new BatchStatementRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
