/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the properties of a global secondary index for the table when the backup was created.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GlobalSecondaryIndexInfo implements SdkPojo, Serializable,
        ToCopyableBuilder<GlobalSecondaryIndexInfo.Builder, GlobalSecondaryIndexInfo> {
    private static final SdkField<String> INDEX_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IndexName").getter(getter(GlobalSecondaryIndexInfo::indexName)).setter(setter(Builder::indexName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IndexName").build()).build();

    private static final SdkField<List<KeySchemaElement>> KEY_SCHEMA_FIELD = SdkField
            .<List<KeySchemaElement>> builder(MarshallingType.LIST)
            .memberName("KeySchema")
            .getter(getter(GlobalSecondaryIndexInfo::keySchema))
            .setter(setter(Builder::keySchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySchema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeySchemaElement> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeySchemaElement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Projection> PROJECTION_FIELD = SdkField.<Projection> builder(MarshallingType.SDK_POJO)
            .memberName("Projection").getter(getter(GlobalSecondaryIndexInfo::projection)).setter(setter(Builder::projection))
            .constructor(Projection::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Projection").build()).build();

    private static final SdkField<ProvisionedThroughput> PROVISIONED_THROUGHPUT_FIELD = SdkField
            .<ProvisionedThroughput> builder(MarshallingType.SDK_POJO).memberName("ProvisionedThroughput")
            .getter(getter(GlobalSecondaryIndexInfo::provisionedThroughput)).setter(setter(Builder::provisionedThroughput))
            .constructor(ProvisionedThroughput::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedThroughput").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INDEX_NAME_FIELD,
            KEY_SCHEMA_FIELD, PROJECTION_FIELD, PROVISIONED_THROUGHPUT_FIELD));

    private static final long serialVersionUID = 1L;

    private final String indexName;

    private final List<KeySchemaElement> keySchema;

    private final Projection projection;

    private final ProvisionedThroughput provisionedThroughput;

    private GlobalSecondaryIndexInfo(BuilderImpl builder) {
        this.indexName = builder.indexName;
        this.keySchema = builder.keySchema;
        this.projection = builder.projection;
        this.provisionedThroughput = builder.provisionedThroughput;
    }

    /**
     * <p>
     * The name of the global secondary index.
     * </p>
     * 
     * @return The name of the global secondary index.
     */
    public final String indexName() {
        return indexName;
    }

    /**
     * For responses, this returns true if the service returned a value for the KeySchema property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasKeySchema() {
        return keySchema != null && !(keySchema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names and
     * key types:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>HASH</code> - partition key
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RANGE</code> - sort key
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives from
     * DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based on their
     * partition key values.
     * </p>
     * <p>
     * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from the
     * way DynamoDB stores items with the same partition key physically close together, in sorted order by the sort key
     * value.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasKeySchema} method.
     * </p>
     * 
     * @return The complete key schema for a global secondary index, which consists of one or more pairs of attribute
     *         names and key types:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>HASH</code> - partition key
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>RANGE</code> - sort key
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
     *         derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
     *         partitions, based on their partition key values.
     *         </p>
     *         <p>
     *         The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives
     *         from the way DynamoDB stores items with the same partition key physically close together, in sorted order
     *         by the sort key value.
     *         </p>
     */
    public final List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * <p>
     * Represents attributes that are copied (projected) from the table into the global secondary index. These are in
     * addition to the primary key attributes and index key attributes, which are automatically projected.
     * </p>
     * 
     * @return Represents attributes that are copied (projected) from the table into the global secondary index. These
     *         are in addition to the primary key attributes and index key attributes, which are automatically
     *         projected.
     */
    public final Projection projection() {
        return projection;
    }

    /**
     * <p>
     * Represents the provisioned throughput settings for the specified global secondary index.
     * </p>
     * 
     * @return Represents the provisioned throughput settings for the specified global secondary index.
     */
    public final ProvisionedThroughput provisionedThroughput() {
        return provisionedThroughput;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(indexName());
        hashCode = 31 * hashCode + Objects.hashCode(hasKeySchema() ? keySchema() : null);
        hashCode = 31 * hashCode + Objects.hashCode(projection());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughput());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GlobalSecondaryIndexInfo)) {
            return false;
        }
        GlobalSecondaryIndexInfo other = (GlobalSecondaryIndexInfo) obj;
        return Objects.equals(indexName(), other.indexName()) && hasKeySchema() == other.hasKeySchema()
                && Objects.equals(keySchema(), other.keySchema()) && Objects.equals(projection(), other.projection())
                && Objects.equals(provisionedThroughput(), other.provisionedThroughput());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GlobalSecondaryIndexInfo").add("IndexName", indexName())
                .add("KeySchema", hasKeySchema() ? keySchema() : null).add("Projection", projection())
                .add("ProvisionedThroughput", provisionedThroughput()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IndexName":
            return Optional.ofNullable(clazz.cast(indexName()));
        case "KeySchema":
            return Optional.ofNullable(clazz.cast(keySchema()));
        case "Projection":
            return Optional.ofNullable(clazz.cast(projection()));
        case "ProvisionedThroughput":
            return Optional.ofNullable(clazz.cast(provisionedThroughput()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<GlobalSecondaryIndexInfo, T> g) {
        return obj -> g.apply((GlobalSecondaryIndexInfo) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GlobalSecondaryIndexInfo> {
        /**
         * <p>
         * The name of the global secondary index.
         * </p>
         * 
         * @param indexName
         *        The name of the global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder indexName(String indexName);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for a global secondary index, which consists of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note>
         * 
         * @param keySchema
         *        The complete key schema for a global secondary index, which consists of one or more pairs of attribute
         *        names and key types:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>HASH</code> - partition key
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>RANGE</code> - sort key
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute"
         *        derives from DynamoDB's usage of an internal hash function to evenly distribute data items across
         *        partitions, based on their partition key values.
         *        </p>
         *        <p>
         *        The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute"
         *        derives from the way DynamoDB stores items with the same partition key physically close together, in
         *        sorted order by the sort key value.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The complete key schema for a global secondary index, which consists of one or more pairs of attribute names
         * and key types:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>HASH</code> - partition key
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>RANGE</code> - sort key
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The partition key of an item is also known as its <i>hash attribute</i>. The term "hash attribute" derives
         * from DynamoDB's usage of an internal hash function to evenly distribute data items across partitions, based
         * on their partition key values.
         * </p>
         * <p>
         * The sort key of an item is also known as its <i>range attribute</i>. The term "range attribute" derives from
         * the way DynamoDB stores items with the same partition key physically close together, in sorted order by the
         * sort key value.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link List<KeySchemaElement>.Builder}
         * avoiding the need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the global secondary index. These are
         * in addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * 
         * @param projection
         *        Represents attributes that are copied (projected) from the table into the global secondary index.
         *        These are in addition to the primary key attributes and index key attributes, which are automatically
         *        projected.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projection(Projection projection);

        /**
         * <p>
         * Represents attributes that are copied (projected) from the table into the global secondary index. These are
         * in addition to the primary key attributes and index key attributes, which are automatically projected.
         * </p>
         * This is a convenience method that creates an instance of the {@link Projection.Builder} avoiding the need to
         * create one manually via {@link Projection#builder()}.
         *
         * When the {@link Consumer} completes, {@link Projection.Builder#build()} is called immediately and its result
         * is passed to {@link #projection(Projection)}.
         * 
         * @param projection
         *        a consumer that will call methods on {@link Projection.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projection(Projection)
         */
        default Builder projection(Consumer<Projection.Builder> projection) {
            return projection(Projection.builder().applyMutation(projection).build());
        }

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * 
         * @param provisionedThroughput
         *        Represents the provisioned throughput settings for the specified global secondary index.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput);

        /**
         * <p>
         * Represents the provisioned throughput settings for the specified global secondary index.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProvisionedThroughput.Builder} avoiding
         * the need to create one manually via {@link ProvisionedThroughput#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedThroughput.Builder#build()} is called immediately and
         * its result is passed to {@link #provisionedThroughput(ProvisionedThroughput)}.
         * 
         * @param provisionedThroughput
         *        a consumer that will call methods on {@link ProvisionedThroughput.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughput(ProvisionedThroughput)
         */
        default Builder provisionedThroughput(Consumer<ProvisionedThroughput.Builder> provisionedThroughput) {
            return provisionedThroughput(ProvisionedThroughput.builder().applyMutation(provisionedThroughput).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String indexName;

        private List<KeySchemaElement> keySchema = DefaultSdkAutoConstructList.getInstance();

        private Projection projection;

        private ProvisionedThroughput provisionedThroughput;

        private BuilderImpl() {
        }

        private BuilderImpl(GlobalSecondaryIndexInfo model) {
            indexName(model.indexName);
            keySchema(model.keySchema);
            projection(model.projection);
            provisionedThroughput(model.provisionedThroughput);
        }

        public final String getIndexName() {
            return indexName;
        }

        public final void setIndexName(String indexName) {
            this.indexName = indexName;
        }

        @Override
        @Transient
        public final Builder indexName(String indexName) {
            this.indexName = indexName;
            return this;
        }

        public final List<KeySchemaElement.Builder> getKeySchema() {
            List<KeySchemaElement.Builder> result = KeySchemaCopier.copyToBuilder(this.keySchema);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        @Override
        @Transient
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Projection.Builder getProjection() {
            return projection != null ? projection.toBuilder() : null;
        }

        public final void setProjection(Projection.BuilderImpl projection) {
            this.projection = projection != null ? projection.build() : null;
        }

        @Override
        @Transient
        public final Builder projection(Projection projection) {
            this.projection = projection;
            return this;
        }

        public final ProvisionedThroughput.Builder getProvisionedThroughput() {
            return provisionedThroughput != null ? provisionedThroughput.toBuilder() : null;
        }

        public final void setProvisionedThroughput(ProvisionedThroughput.BuilderImpl provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput != null ? provisionedThroughput.build() : null;
        }

        @Override
        @Transient
        public final Builder provisionedThroughput(ProvisionedThroughput provisionedThroughput) {
            this.provisionedThroughput = provisionedThroughput;
            return this;
        }

        @Override
        public GlobalSecondaryIndexInfo build() {
            return new GlobalSecondaryIndexInfo(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
