/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents all of the data describing a particular stream.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class StreamDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<StreamDescription.Builder, StreamDescription> {
    private static final SdkField<String> STREAM_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamArn").getter(getter(StreamDescription::streamArn)).setter(setter(Builder::streamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamArn").build()).build();

    private static final SdkField<String> STREAM_LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamLabel").getter(getter(StreamDescription::streamLabel)).setter(setter(Builder::streamLabel))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamLabel").build()).build();

    private static final SdkField<String> STREAM_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamStatus").getter(getter(StreamDescription::streamStatusAsString))
            .setter(setter(Builder::streamStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamStatus").build()).build();

    private static final SdkField<String> STREAM_VIEW_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StreamViewType").getter(getter(StreamDescription::streamViewTypeAsString))
            .setter(setter(Builder::streamViewType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StreamViewType").build()).build();

    private static final SdkField<Instant> CREATION_REQUEST_DATE_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreationRequestDateTime").getter(getter(StreamDescription::creationRequestDateTime))
            .setter(setter(Builder::creationRequestDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationRequestDateTime").build())
            .build();

    private static final SdkField<String> TABLE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TableName").getter(getter(StreamDescription::tableName)).setter(setter(Builder::tableName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TableName").build()).build();

    private static final SdkField<List<KeySchemaElement>> KEY_SCHEMA_FIELD = SdkField
            .<List<KeySchemaElement>> builder(MarshallingType.LIST)
            .memberName("KeySchema")
            .getter(getter(StreamDescription::keySchema))
            .setter(setter(Builder::keySchema))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KeySchema").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<KeySchemaElement> builder(MarshallingType.SDK_POJO)
                                            .constructor(KeySchemaElement::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Shard>> SHARDS_FIELD = SdkField
            .<List<Shard>> builder(MarshallingType.LIST)
            .memberName("Shards")
            .getter(getter(StreamDescription::shards))
            .setter(setter(Builder::shards))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Shards").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Shard> builder(MarshallingType.SDK_POJO)
                                            .constructor(Shard::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LAST_EVALUATED_SHARD_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LastEvaluatedShardId").getter(getter(StreamDescription::lastEvaluatedShardId))
            .setter(setter(Builder::lastEvaluatedShardId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastEvaluatedShardId").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(STREAM_ARN_FIELD,
            STREAM_LABEL_FIELD, STREAM_STATUS_FIELD, STREAM_VIEW_TYPE_FIELD, CREATION_REQUEST_DATE_TIME_FIELD, TABLE_NAME_FIELD,
            KEY_SCHEMA_FIELD, SHARDS_FIELD, LAST_EVALUATED_SHARD_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String streamArn;

    private final String streamLabel;

    private final String streamStatus;

    private final String streamViewType;

    private final Instant creationRequestDateTime;

    private final String tableName;

    private final List<KeySchemaElement> keySchema;

    private final List<Shard> shards;

    private final String lastEvaluatedShardId;

    private StreamDescription(BuilderImpl builder) {
        this.streamArn = builder.streamArn;
        this.streamLabel = builder.streamLabel;
        this.streamStatus = builder.streamStatus;
        this.streamViewType = builder.streamViewType;
        this.creationRequestDateTime = builder.creationRequestDateTime;
        this.tableName = builder.tableName;
        this.keySchema = builder.keySchema;
        this.shards = builder.shards;
        this.lastEvaluatedShardId = builder.lastEvaluatedShardId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) for the stream.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) for the stream.
     */
    public final String streamArn() {
        return streamArn;
    }

    /**
     * <p>
     * A timestamp, in ISO 8601 format, for this stream.
     * </p>
     * <p>
     * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible that a
     * stream from another table might have the same timestamp. However, the combination of the following three elements
     * is guaranteed to be unique:
     * </p>
     * <ul>
     * <li>
     * <p>
     * the AWS customer ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * the table name
     * </p>
     * </li>
     * <li>
     * <p>
     * the <code>StreamLabel</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return A timestamp, in ISO 8601 format, for this stream.</p>
     *         <p>
     *         Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
     *         possible that a stream from another table might have the same timestamp. However, the combination of the
     *         following three elements is guaranteed to be unique:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         the AWS customer ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the table name
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         the <code>StreamLabel</code>
     *         </p>
     *         </li>
     */
    public final String streamLabel() {
        return streamLabel;
    }

    /**
     * <p>
     * Indicates the current status of the stream:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - the stream is enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> - the stream is disabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return Indicates the current status of the stream:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - the stream is enabled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> - the stream is disabled.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public final StreamStatus streamStatus() {
        return StreamStatus.fromValue(streamStatus);
    }

    /**
     * <p>
     * Indicates the current status of the stream:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ENABLED</code> - the stream is enabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> - the stream is disabled.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamStatus} will
     * return {@link StreamStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #streamStatusAsString}.
     * </p>
     * 
     * @return Indicates the current status of the stream:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ENABLED</code> - the stream is enabled.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DISABLED</code> - the stream is disabled.
     *         </p>
     *         </li>
     * @see StreamStatus
     */
    public final String streamStatusAsString() {
        return streamStatus;
    }

    /**
     * <p>
     * Indicates the format of the records within this stream:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamViewType}
     * will return {@link StreamViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #streamViewTypeAsString}.
     * </p>
     * 
     * @return Indicates the format of the records within this stream:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
     *         </p>
     *         </li>
     * @see StreamViewType
     */
    public final StreamViewType streamViewType() {
        return StreamViewType.fromValue(streamViewType);
    }

    /**
     * <p>
     * Indicates the format of the records within this stream:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #streamViewType}
     * will return {@link StreamViewType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #streamViewTypeAsString}.
     * </p>
     * 
     * @return Indicates the format of the records within this stream:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
     *         </p>
     *         </li>
     * @see StreamViewType
     */
    public final String streamViewTypeAsString() {
        return streamViewType;
    }

    /**
     * <p>
     * The date and time when the request to create this stream was issued.
     * </p>
     * 
     * @return The date and time when the request to create this stream was issued.
     */
    public final Instant creationRequestDateTime() {
        return creationRequestDateTime;
    }

    /**
     * <p>
     * The DynamoDB table with which the stream is associated.
     * </p>
     * 
     * @return The DynamoDB table with which the stream is associated.
     */
    public final String tableName() {
        return tableName;
    }

    /**
     * For responses, this returns true if the service returned a value for the KeySchema property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasKeySchema() {
        return keySchema != null && !(keySchema instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key attribute(s) of the stream's DynamoDB table.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasKeySchema} method.
     * </p>
     * 
     * @return The key attribute(s) of the stream's DynamoDB table.
     */
    public final List<KeySchemaElement> keySchema() {
        return keySchema;
    }

    /**
     * For responses, this returns true if the service returned a value for the Shards property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasShards() {
        return shards != null && !(shards instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The shards that comprise the stream.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasShards} method.
     * </p>
     * 
     * @return The shards that comprise the stream.
     */
    public final List<Shard> shards() {
        return shards;
    }

    /**
     * <p>
     * The shard ID of the item where the operation stopped, inclusive of the previous result set. Use this value to
     * start a new operation, excluding this value in the new request.
     * </p>
     * <p>
     * If <code>LastEvaluatedShardId</code> is empty, then the "last page" of results has been processed and there is
     * currently no more data to be retrieved.
     * </p>
     * <p>
     * If <code>LastEvaluatedShardId</code> is not empty, it does not necessarily mean that there is more data in the
     * result set. The only way to know when you have reached the end of the result set is when
     * <code>LastEvaluatedShardId</code> is empty.
     * </p>
     * 
     * @return The shard ID of the item where the operation stopped, inclusive of the previous result set. Use this
     *         value to start a new operation, excluding this value in the new request.</p>
     *         <p>
     *         If <code>LastEvaluatedShardId</code> is empty, then the "last page" of results has been processed and
     *         there is currently no more data to be retrieved.
     *         </p>
     *         <p>
     *         If <code>LastEvaluatedShardId</code> is not empty, it does not necessarily mean that there is more data
     *         in the result set. The only way to know when you have reached the end of the result set is when
     *         <code>LastEvaluatedShardId</code> is empty.
     */
    public final String lastEvaluatedShardId() {
        return lastEvaluatedShardId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(streamArn());
        hashCode = 31 * hashCode + Objects.hashCode(streamLabel());
        hashCode = 31 * hashCode + Objects.hashCode(streamStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(streamViewTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(creationRequestDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(tableName());
        hashCode = 31 * hashCode + Objects.hashCode(hasKeySchema() ? keySchema() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasShards() ? shards() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastEvaluatedShardId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof StreamDescription)) {
            return false;
        }
        StreamDescription other = (StreamDescription) obj;
        return Objects.equals(streamArn(), other.streamArn()) && Objects.equals(streamLabel(), other.streamLabel())
                && Objects.equals(streamStatusAsString(), other.streamStatusAsString())
                && Objects.equals(streamViewTypeAsString(), other.streamViewTypeAsString())
                && Objects.equals(creationRequestDateTime(), other.creationRequestDateTime())
                && Objects.equals(tableName(), other.tableName()) && hasKeySchema() == other.hasKeySchema()
                && Objects.equals(keySchema(), other.keySchema()) && hasShards() == other.hasShards()
                && Objects.equals(shards(), other.shards())
                && Objects.equals(lastEvaluatedShardId(), other.lastEvaluatedShardId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("StreamDescription").add("StreamArn", streamArn()).add("StreamLabel", streamLabel())
                .add("StreamStatus", streamStatusAsString()).add("StreamViewType", streamViewTypeAsString())
                .add("CreationRequestDateTime", creationRequestDateTime()).add("TableName", tableName())
                .add("KeySchema", hasKeySchema() ? keySchema() : null).add("Shards", hasShards() ? shards() : null)
                .add("LastEvaluatedShardId", lastEvaluatedShardId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "StreamArn":
            return Optional.ofNullable(clazz.cast(streamArn()));
        case "StreamLabel":
            return Optional.ofNullable(clazz.cast(streamLabel()));
        case "StreamStatus":
            return Optional.ofNullable(clazz.cast(streamStatusAsString()));
        case "StreamViewType":
            return Optional.ofNullable(clazz.cast(streamViewTypeAsString()));
        case "CreationRequestDateTime":
            return Optional.ofNullable(clazz.cast(creationRequestDateTime()));
        case "TableName":
            return Optional.ofNullable(clazz.cast(tableName()));
        case "KeySchema":
            return Optional.ofNullable(clazz.cast(keySchema()));
        case "Shards":
            return Optional.ofNullable(clazz.cast(shards()));
        case "LastEvaluatedShardId":
            return Optional.ofNullable(clazz.cast(lastEvaluatedShardId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<StreamDescription, T> g) {
        return obj -> g.apply((StreamDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, StreamDescription> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) for the stream.
         * </p>
         * 
         * @param streamArn
         *        The Amazon Resource Name (ARN) for the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamArn(String streamArn);

        /**
         * <p>
         * A timestamp, in ISO 8601 format, for this stream.
         * </p>
         * <p>
         * Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is possible
         * that a stream from another table might have the same timestamp. However, the combination of the following
         * three elements is guaranteed to be unique:
         * </p>
         * <ul>
         * <li>
         * <p>
         * the AWS customer ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * the table name
         * </p>
         * </li>
         * <li>
         * <p>
         * the <code>StreamLabel</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamLabel
         *        A timestamp, in ISO 8601 format, for this stream.</p>
         *        <p>
         *        Note that <code>LatestStreamLabel</code> is not a unique identifier for the stream, because it is
         *        possible that a stream from another table might have the same timestamp. However, the combination of
         *        the following three elements is guaranteed to be unique:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        the AWS customer ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the table name
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        the <code>StreamLabel</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder streamLabel(String streamLabel);

        /**
         * <p>
         * Indicates the current status of the stream:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - the stream is enabled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> - the stream is disabled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        Indicates the current status of the stream:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - the stream is enabled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> - the stream is disabled.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(String streamStatus);

        /**
         * <p>
         * Indicates the current status of the stream:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ENABLED</code> - the stream is enabled.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DISABLED</code> - the stream is disabled.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamStatus
         *        Indicates the current status of the stream:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ENABLING</code> - Streams is currently being enabled on the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ENABLED</code> - the stream is enabled.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLING</code> - Streams is currently being disabled on the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DISABLED</code> - the stream is disabled.
         *        </p>
         *        </li>
         * @see StreamStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamStatus
         */
        Builder streamStatus(StreamStatus streamStatus);

        /**
         * <p>
         * Indicates the format of the records within this stream:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamViewType
         *        Indicates the format of the records within this stream:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
         *        </p>
         *        </li>
         * @see StreamViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamViewType
         */
        Builder streamViewType(String streamViewType);

        /**
         * <p>
         * Indicates the format of the records within this stream:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
         * </p>
         * </li>
         * </ul>
         * 
         * @param streamViewType
         *        Indicates the format of the records within this stream:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>KEYS_ONLY</code> - only the key attributes of items that were modified in the DynamoDB table.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_IMAGE</code> - entire items from the table, as they appeared after they were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>OLD_IMAGE</code> - entire items from the table, as they appeared before they were modified.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>NEW_AND_OLD_IMAGES</code> - both the new and the old images of the items from the table.
         *        </p>
         *        </li>
         * @see StreamViewType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StreamViewType
         */
        Builder streamViewType(StreamViewType streamViewType);

        /**
         * <p>
         * The date and time when the request to create this stream was issued.
         * </p>
         * 
         * @param creationRequestDateTime
         *        The date and time when the request to create this stream was issued.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationRequestDateTime(Instant creationRequestDateTime);

        /**
         * <p>
         * The DynamoDB table with which the stream is associated.
         * </p>
         * 
         * @param tableName
         *        The DynamoDB table with which the stream is associated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tableName(String tableName);

        /**
         * <p>
         * The key attribute(s) of the stream's DynamoDB table.
         * </p>
         * 
         * @param keySchema
         *        The key attribute(s) of the stream's DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(Collection<KeySchemaElement> keySchema);

        /**
         * <p>
         * The key attribute(s) of the stream's DynamoDB table.
         * </p>
         * 
         * @param keySchema
         *        The key attribute(s) of the stream's DynamoDB table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder keySchema(KeySchemaElement... keySchema);

        /**
         * <p>
         * The key attribute(s) of the stream's DynamoDB table.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<KeySchemaElement>.Builder} avoiding
         * the need to create one manually via {@link List<KeySchemaElement>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<KeySchemaElement>.Builder#build()} is called immediately and
         * its result is passed to {@link #keySchema(List<KeySchemaElement>)}.
         * 
         * @param keySchema
         *        a consumer that will call methods on {@link List<KeySchemaElement>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #keySchema(List<KeySchemaElement>)
         */
        Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema);

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * 
         * @param shards
         *        The shards that comprise the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Collection<Shard> shards);

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * 
         * @param shards
         *        The shards that comprise the stream.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shards(Shard... shards);

        /**
         * <p>
         * The shards that comprise the stream.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<Shard>.Builder} avoiding the need to
         * create one manually via {@link List<Shard>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Shard>.Builder#build()} is called immediately and its result
         * is passed to {@link #shards(List<Shard>)}.
         * 
         * @param shards
         *        a consumer that will call methods on {@link List<Shard>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #shards(List<Shard>)
         */
        Builder shards(Consumer<Shard.Builder>... shards);

        /**
         * <p>
         * The shard ID of the item where the operation stopped, inclusive of the previous result set. Use this value to
         * start a new operation, excluding this value in the new request.
         * </p>
         * <p>
         * If <code>LastEvaluatedShardId</code> is empty, then the "last page" of results has been processed and there
         * is currently no more data to be retrieved.
         * </p>
         * <p>
         * If <code>LastEvaluatedShardId</code> is not empty, it does not necessarily mean that there is more data in
         * the result set. The only way to know when you have reached the end of the result set is when
         * <code>LastEvaluatedShardId</code> is empty.
         * </p>
         * 
         * @param lastEvaluatedShardId
         *        The shard ID of the item where the operation stopped, inclusive of the previous result set. Use this
         *        value to start a new operation, excluding this value in the new request.</p>
         *        <p>
         *        If <code>LastEvaluatedShardId</code> is empty, then the "last page" of results has been processed and
         *        there is currently no more data to be retrieved.
         *        </p>
         *        <p>
         *        If <code>LastEvaluatedShardId</code> is not empty, it does not necessarily mean that there is more
         *        data in the result set. The only way to know when you have reached the end of the result set is when
         *        <code>LastEvaluatedShardId</code> is empty.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastEvaluatedShardId(String lastEvaluatedShardId);
    }

    static final class BuilderImpl implements Builder {
        private String streamArn;

        private String streamLabel;

        private String streamStatus;

        private String streamViewType;

        private Instant creationRequestDateTime;

        private String tableName;

        private List<KeySchemaElement> keySchema = DefaultSdkAutoConstructList.getInstance();

        private List<Shard> shards = DefaultSdkAutoConstructList.getInstance();

        private String lastEvaluatedShardId;

        private BuilderImpl() {
        }

        private BuilderImpl(StreamDescription model) {
            streamArn(model.streamArn);
            streamLabel(model.streamLabel);
            streamStatus(model.streamStatus);
            streamViewType(model.streamViewType);
            creationRequestDateTime(model.creationRequestDateTime);
            tableName(model.tableName);
            keySchema(model.keySchema);
            shards(model.shards);
            lastEvaluatedShardId(model.lastEvaluatedShardId);
        }

        public final String getStreamArn() {
            return streamArn;
        }

        public final void setStreamArn(String streamArn) {
            this.streamArn = streamArn;
        }

        @Override
        @Transient
        public final Builder streamArn(String streamArn) {
            this.streamArn = streamArn;
            return this;
        }

        public final String getStreamLabel() {
            return streamLabel;
        }

        public final void setStreamLabel(String streamLabel) {
            this.streamLabel = streamLabel;
        }

        @Override
        @Transient
        public final Builder streamLabel(String streamLabel) {
            this.streamLabel = streamLabel;
            return this;
        }

        public final String getStreamStatus() {
            return streamStatus;
        }

        public final void setStreamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
        }

        @Override
        @Transient
        public final Builder streamStatus(String streamStatus) {
            this.streamStatus = streamStatus;
            return this;
        }

        @Override
        @Transient
        public final Builder streamStatus(StreamStatus streamStatus) {
            this.streamStatus(streamStatus == null ? null : streamStatus.toString());
            return this;
        }

        public final String getStreamViewType() {
            return streamViewType;
        }

        public final void setStreamViewType(String streamViewType) {
            this.streamViewType = streamViewType;
        }

        @Override
        @Transient
        public final Builder streamViewType(String streamViewType) {
            this.streamViewType = streamViewType;
            return this;
        }

        @Override
        @Transient
        public final Builder streamViewType(StreamViewType streamViewType) {
            this.streamViewType(streamViewType == null ? null : streamViewType.toString());
            return this;
        }

        public final Instant getCreationRequestDateTime() {
            return creationRequestDateTime;
        }

        public final void setCreationRequestDateTime(Instant creationRequestDateTime) {
            this.creationRequestDateTime = creationRequestDateTime;
        }

        @Override
        @Transient
        public final Builder creationRequestDateTime(Instant creationRequestDateTime) {
            this.creationRequestDateTime = creationRequestDateTime;
            return this;
        }

        public final String getTableName() {
            return tableName;
        }

        public final void setTableName(String tableName) {
            this.tableName = tableName;
        }

        @Override
        @Transient
        public final Builder tableName(String tableName) {
            this.tableName = tableName;
            return this;
        }

        public final List<KeySchemaElement.Builder> getKeySchema() {
            List<KeySchemaElement.Builder> result = KeySchemaCopier.copyToBuilder(this.keySchema);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setKeySchema(Collection<KeySchemaElement.BuilderImpl> keySchema) {
            this.keySchema = KeySchemaCopier.copyFromBuilder(keySchema);
        }

        @Override
        @Transient
        public final Builder keySchema(Collection<KeySchemaElement> keySchema) {
            this.keySchema = KeySchemaCopier.copy(keySchema);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder keySchema(KeySchemaElement... keySchema) {
            keySchema(Arrays.asList(keySchema));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder keySchema(Consumer<KeySchemaElement.Builder>... keySchema) {
            keySchema(Stream.of(keySchema).map(c -> KeySchemaElement.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Shard.Builder> getShards() {
            List<Shard.Builder> result = ShardDescriptionListCopier.copyToBuilder(this.shards);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setShards(Collection<Shard.BuilderImpl> shards) {
            this.shards = ShardDescriptionListCopier.copyFromBuilder(shards);
        }

        @Override
        @Transient
        public final Builder shards(Collection<Shard> shards) {
            this.shards = ShardDescriptionListCopier.copy(shards);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder shards(Shard... shards) {
            shards(Arrays.asList(shards));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder shards(Consumer<Shard.Builder>... shards) {
            shards(Stream.of(shards).map(c -> Shard.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLastEvaluatedShardId() {
            return lastEvaluatedShardId;
        }

        public final void setLastEvaluatedShardId(String lastEvaluatedShardId) {
            this.lastEvaluatedShardId = lastEvaluatedShardId;
        }

        @Override
        @Transient
        public final Builder lastEvaluatedShardId(String lastEvaluatedShardId) {
            this.lastEvaluatedShardId = lastEvaluatedShardId;
            return this;
        }

        @Override
        public StreamDescription build() {
            return new StreamDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
