/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkBytes;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the data for an attribute.
 * </p>
 * <p>
 * Each attribute value is described as a name-value pair. The name is the data type, and the value is the data itself.
 * </p>
 * <p>
 * For more information, see <a href=
 * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes"
 * >Data Types</a> in the <i>Amazon DynamoDB Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AttributeValue implements SdkPojo, Serializable, ToCopyableBuilder<AttributeValue.Builder, AttributeValue> {
    private static final SdkField<String> S_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("S")
            .getter(getter(AttributeValue::s)).setter(setter(Builder::s))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S").build()).build();

    private static final SdkField<String> N_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("N")
            .getter(getter(AttributeValue::n)).setter(setter(Builder::n))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("N").build()).build();

    private static final SdkField<SdkBytes> B_FIELD = SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES).memberName("B")
            .getter(getter(AttributeValue::b)).setter(setter(Builder::b))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("B").build()).build();

    private static final SdkField<List<String>> SS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SS")
            .getter(getter(AttributeValue::ss))
            .setter(setter(Builder::ss))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SS").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> NS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NS")
            .getter(getter(AttributeValue::ns))
            .setter(setter(Builder::ns))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NS").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<SdkBytes>> BS_FIELD = SdkField
            .<List<SdkBytes>> builder(MarshallingType.LIST)
            .memberName("BS")
            .getter(getter(AttributeValue::bs))
            .setter(setter(Builder::bs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BS").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SdkBytes> builder(MarshallingType.SDK_BYTES)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Map<String, AttributeValue>> M_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .memberName("M")
            .getter(getter(AttributeValue::m))
            .setter(setter(Builder::m))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("M").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<List<AttributeValue>> L_FIELD = SdkField
            .<List<AttributeValue>> builder(MarshallingType.LIST)
            .memberName("L")
            .getter(getter(AttributeValue::l))
            .setter(setter(Builder::l))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("L").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> BOOL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN).memberName("BOOL")
            .getter(getter(AttributeValue::bool)).setter(setter(Builder::bool))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BOOL").build()).build();

    private static final SdkField<Boolean> NUL_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN).memberName("NUL")
            .getter(getter(AttributeValue::nul)).setter(setter(Builder::nul))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NULL").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(S_FIELD, N_FIELD, B_FIELD,
            SS_FIELD, NS_FIELD, BS_FIELD, M_FIELD, L_FIELD, BOOL_FIELD, NUL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String s;

    private final String n;

    private final SdkBytes b;

    private final List<String> ss;

    private final List<String> ns;

    private final List<SdkBytes> bs;

    private final Map<String, AttributeValue> m;

    private final List<AttributeValue> l;

    private final Boolean bool;

    private final Boolean nul;

    private final Type type;

    private AttributeValue(BuilderImpl builder) {
        this.s = builder.s;
        this.n = builder.n;
        this.b = builder.b;
        this.ss = builder.ss;
        this.ns = builder.ns;
        this.bs = builder.bs;
        this.m = builder.m;
        this.l = builder.l;
        this.bool = builder.bool;
        this.nul = builder.nul;
        this.type = builder.type;
    }

    /**
     * <p>
     * An attribute of type String. For example:
     * </p>
     * <p>
     * <code>"S": "Hello"</code>
     * </p>
     * 
     * @return An attribute of type String. For example:</p>
     *         <p>
     *         <code>"S": "Hello"</code>
     */
    public final String s() {
        return s;
    }

    /**
     * <p>
     * An attribute of type Number. For example:
     * </p>
     * <p>
     * <code>"N": "123.45"</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * 
     * @return An attribute of type Number. For example:</p>
     *         <p>
     *         <code>"N": "123.45"</code>
     *         </p>
     *         <p>
     *         Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
     *         and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public final String n() {
        return n;
    }

    /**
     * <p>
     * An attribute of type Binary. For example:
     * </p>
     * <p>
     * <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     * </p>
     * 
     * @return An attribute of type Binary. For example:</p>
     *         <p>
     *         <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     */
    public final SdkBytes b() {
        return b;
    }

    /**
     * For responses, this returns true if the service returned a value for the SS property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasSs() {
        return ss != null && !(ss instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An attribute of type String Set. For example:
     * </p>
     * <p>
     * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSs} method.
     * </p>
     * 
     * @return An attribute of type String Set. For example:</p>
     *         <p>
     *         <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     */
    public final List<String> ss() {
        return ss;
    }

    /**
     * For responses, this returns true if the service returned a value for the NS property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasNs() {
        return ns != null && !(ns instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An attribute of type Number Set. For example:
     * </p>
     * <p>
     * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNs} method.
     * </p>
     * 
     * @return An attribute of type Number Set. For example:</p>
     *         <p>
     *         <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     *         </p>
     *         <p>
     *         Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
     *         and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public final List<String> ns() {
        return ns;
    }

    /**
     * For responses, this returns true if the service returned a value for the BS property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasBs() {
        return bs != null && !(bs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An attribute of type Binary Set. For example:
     * </p>
     * <p>
     * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasBs} method.
     * </p>
     * 
     * @return An attribute of type Binary Set. For example:</p>
     *         <p>
     *         <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     */
    public final List<SdkBytes> bs() {
        return bs;
    }

    /**
     * For responses, this returns true if the service returned a value for the M property. This DOES NOT check that the
     * value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasM() {
        return m != null && !(m instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * An attribute of type Map. For example:
     * </p>
     * <p>
     * <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasM} method.
     * </p>
     * 
     * @return An attribute of type Map. For example:</p>
     *         <p>
     *         <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     */
    public final Map<String, AttributeValue> m() {
        return m;
    }

    /**
     * For responses, this returns true if the service returned a value for the L property. This DOES NOT check that the
     * value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasL() {
        return l != null && !(l instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An attribute of type List. For example:
     * </p>
     * <p>
     * <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasL} method.
     * </p>
     * 
     * @return An attribute of type List. For example:</p>
     *         <p>
     *         <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
     */
    public final List<AttributeValue> l() {
        return l;
    }

    /**
     * <p>
     * An attribute of type Boolean. For example:
     * </p>
     * <p>
     * <code>"BOOL": true</code>
     * </p>
     * 
     * @return An attribute of type Boolean. For example:</p>
     *         <p>
     *         <code>"BOOL": true</code>
     */
    public final Boolean bool() {
        return bool;
    }

    /**
     * <p>
     * An attribute of type Null. For example:
     * </p>
     * <p>
     * <code>"NULL": true</code>
     * </p>
     * 
     * @return An attribute of type Null. For example:</p>
     *         <p>
     *         <code>"NULL": true</code>
     */
    public final Boolean nul() {
        return nul;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(s());
        hashCode = 31 * hashCode + Objects.hashCode(n());
        hashCode = 31 * hashCode + Objects.hashCode(b());
        hashCode = 31 * hashCode + Objects.hashCode(hasSs() ? ss() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasNs() ? ns() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasBs() ? bs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasM() ? m() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasL() ? l() : null);
        hashCode = 31 * hashCode + Objects.hashCode(bool());
        hashCode = 31 * hashCode + Objects.hashCode(nul());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AttributeValue)) {
            return false;
        }
        AttributeValue other = (AttributeValue) obj;
        return Objects.equals(s(), other.s()) && Objects.equals(n(), other.n()) && Objects.equals(b(), other.b())
                && hasSs() == other.hasSs() && Objects.equals(ss(), other.ss()) && hasNs() == other.hasNs()
                && Objects.equals(ns(), other.ns()) && hasBs() == other.hasBs() && Objects.equals(bs(), other.bs())
                && hasM() == other.hasM() && Objects.equals(m(), other.m()) && hasL() == other.hasL()
                && Objects.equals(l(), other.l()) && Objects.equals(bool(), other.bool()) && Objects.equals(nul(), other.nul());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AttributeValue").add("S", s()).add("N", n()).add("B", b()).add("SS", hasSs() ? ss() : null)
                .add("NS", hasNs() ? ns() : null).add("BS", hasBs() ? bs() : null).add("M", hasM() ? m() : null)
                .add("L", hasL() ? l() : null).add("BOOL", bool()).add("NUL", nul()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "S":
            return Optional.ofNullable(clazz.cast(s()));
        case "N":
            return Optional.ofNullable(clazz.cast(n()));
        case "B":
            return Optional.ofNullable(clazz.cast(b()));
        case "SS":
            return Optional.ofNullable(clazz.cast(ss()));
        case "NS":
            return Optional.ofNullable(clazz.cast(ns()));
        case "BS":
            return Optional.ofNullable(clazz.cast(bs()));
        case "M":
            return Optional.ofNullable(clazz.cast(m()));
        case "L":
            return Optional.ofNullable(clazz.cast(l()));
        case "BOOL":
            return Optional.ofNullable(clazz.cast(bool()));
        case "NUL":
            return Optional.ofNullable(clazz.cast(nul()));
        default:
            return Optional.empty();
        }
    }

    /**
     * Create an instance of this class with {@link #s()} initialized to the given value.
     *
     * <p>
     * An attribute of type String. For example:
     * </p>
     * <p>
     * <code>"S": "Hello"</code>
     * </p>
     * 
     * @param s
     *        An attribute of type String. For example:</p>
     *        <p>
     *        <code>"S": "Hello"</code>
     */
    public static AttributeValue fromS(String s) {
        return builder().s(s).build();
    }

    /**
     * Create an instance of this class with {@link #n()} initialized to the given value.
     *
     * <p>
     * An attribute of type Number. For example:
     * </p>
     * <p>
     * <code>"N": "123.45"</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * 
     * @param n
     *        An attribute of type Number. For example:</p>
     *        <p>
     *        <code>"N": "123.45"</code>
     *        </p>
     *        <p>
     *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     *        libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public static AttributeValue fromN(String n) {
        return builder().n(n).build();
    }

    /**
     * Create an instance of this class with {@link #b()} initialized to the given value.
     *
     * <p>
     * An attribute of type Binary. For example:
     * </p>
     * <p>
     * <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     * </p>
     * 
     * @param b
     *        An attribute of type Binary. For example:</p>
     *        <p>
     *        <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
     */
    public static AttributeValue fromB(SdkBytes b) {
        return builder().b(b).build();
    }

    /**
     * Create an instance of this class with {@link #ss()} initialized to the given value.
     *
     * <p>
     * An attribute of type String Set. For example:
     * </p>
     * <p>
     * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     * </p>
     * 
     * @param ss
     *        An attribute of type String Set. For example:</p>
     *        <p>
     *        <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
     */
    public static AttributeValue fromSs(List<String> ss) {
        return builder().ss(ss).build();
    }

    /**
     * Create an instance of this class with {@link #ns()} initialized to the given value.
     *
     * <p>
     * An attribute of type Number Set. For example:
     * </p>
     * <p>
     * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     * </p>
     * <p>
     * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     * </p>
     * 
     * @param ns
     *        An attribute of type Number Set. For example:</p>
     *        <p>
     *        <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
     *        </p>
     *        <p>
     *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
     *        libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    public static AttributeValue fromNs(List<String> ns) {
        return builder().ns(ns).build();
    }

    /**
     * Create an instance of this class with {@link #bs()} initialized to the given value.
     *
     * <p>
     * An attribute of type Binary Set. For example:
     * </p>
     * <p>
     * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     * </p>
     * 
     * @param bs
     *        An attribute of type Binary Set. For example:</p>
     *        <p>
     *        <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
     */
    public static AttributeValue fromBs(List<SdkBytes> bs) {
        return builder().bs(bs).build();
    }

    /**
     * Create an instance of this class with {@link #m()} initialized to the given value.
     *
     * <p>
     * An attribute of type Map. For example:
     * </p>
     * <p>
     * <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     * </p>
     * 
     * @param m
     *        An attribute of type Map. For example:</p>
     *        <p>
     *        <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
     */
    public static AttributeValue fromM(Map<String, AttributeValue> m) {
        return builder().m(m).build();
    }

    /**
     * Create an instance of this class with {@link #l()} initialized to the given value.
     *
     * <p>
     * An attribute of type List. For example:
     * </p>
     * <p>
     * <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
     * </p>
     * 
     * @param l
     *        An attribute of type List. For example:</p>
     *        <p>
     *        <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
     */
    public static AttributeValue fromL(List<AttributeValue> l) {
        return builder().l(l).build();
    }

    /**
     * Create an instance of this class with {@link #bool()} initialized to the given value.
     *
     * <p>
     * An attribute of type Boolean. For example:
     * </p>
     * <p>
     * <code>"BOOL": true</code>
     * </p>
     * 
     * @param bool
     *        An attribute of type Boolean. For example:</p>
     *        <p>
     *        <code>"BOOL": true</code>
     */
    public static AttributeValue fromBool(Boolean bool) {
        return builder().bool(bool).build();
    }

    /**
     * Create an instance of this class with {@link #nul()} initialized to the given value.
     *
     * <p>
     * An attribute of type Null. For example:
     * </p>
     * <p>
     * <code>"NULL": true</code>
     * </p>
     * 
     * @param nul
     *        An attribute of type Null. For example:</p>
     *        <p>
     *        <code>"NULL": true</code>
     */
    public static AttributeValue fromNul(Boolean nul) {
        return builder().nul(nul).build();
    }

    /**
     * Retrieve an enum value representing which member of this object is populated.
     *
     * When this class is returned in a service response, this will be {@link Type#UNKNOWN_TO_SDK_VERSION} if the
     * service returned a member that is only known to a newer SDK version.
     *
     * When this class is created directly in your code, this will be {@link Type#UNKNOWN_TO_SDK_VERSION} if zero
     * members are set, and {@code null} if more than one member is set.
     */
    public Type type() {
        return type;
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AttributeValue, T> g) {
        return obj -> g.apply((AttributeValue) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AttributeValue> {
        /**
         * <p>
         * An attribute of type String. For example:
         * </p>
         * <p>
         * <code>"S": "Hello"</code>
         * </p>
         * 
         * @param s
         *        An attribute of type String. For example:</p>
         *        <p>
         *        <code>"S": "Hello"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s(String s);

        /**
         * <p>
         * An attribute of type Number. For example:
         * </p>
         * <p>
         * <code>"N": "123.45"</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param n
         *        An attribute of type Number. For example:</p>
         *        <p>
         *        <code>"N": "123.45"</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder n(String n);

        /**
         * <p>
         * An attribute of type Binary. For example:
         * </p>
         * <p>
         * <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
         * </p>
         * 
         * @param b
         *        An attribute of type Binary. For example:</p>
         *        <p>
         *        <code>"B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder b(SdkBytes b);

        /**
         * <p>
         * An attribute of type String Set. For example:
         * </p>
         * <p>
         * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * </p>
         * 
         * @param ss
         *        An attribute of type String Set. For example:</p>
         *        <p>
         *        <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ss(Collection<String> ss);

        /**
         * <p>
         * An attribute of type String Set. For example:
         * </p>
         * <p>
         * <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * </p>
         * 
         * @param ss
         *        An attribute of type String Set. For example:</p>
         *        <p>
         *        <code>"SS": ["Giraffe", "Hippo" ,"Zebra"]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ss(String... ss);

        /**
         * <p>
         * An attribute of type Number Set. For example:
         * </p>
         * <p>
         * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param ns
         *        An attribute of type Number Set. For example:</p>
         *        <p>
         *        <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ns(Collection<String> ns);

        /**
         * <p>
         * An attribute of type Number Set. For example:
         * </p>
         * <p>
         * <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         * </p>
         * <p>
         * Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages and
         * libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * </p>
         * 
         * @param ns
         *        An attribute of type Number Set. For example:</p>
         *        <p>
         *        <code>"NS": ["42.2", "-19", "7.5", "3.14"]</code>
         *        </p>
         *        <p>
         *        Numbers are sent across the network to DynamoDB as strings, to maximize compatibility across languages
         *        and libraries. However, DynamoDB treats them as number type attributes for mathematical operations.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ns(String... ns);

        /**
         * <p>
         * An attribute of type Binary Set. For example:
         * </p>
         * <p>
         * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * </p>
         * 
         * @param bs
         *        An attribute of type Binary Set. For example:</p>
         *        <p>
         *        <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bs(Collection<SdkBytes> bs);

        /**
         * <p>
         * An attribute of type Binary Set. For example:
         * </p>
         * <p>
         * <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * </p>
         * 
         * @param bs
         *        An attribute of type Binary Set. For example:</p>
         *        <p>
         *        <code>"BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bs(SdkBytes... bs);

        /**
         * <p>
         * An attribute of type Map. For example:
         * </p>
         * <p>
         * <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
         * </p>
         * 
         * @param m
         *        An attribute of type Map. For example:</p>
         *        <p>
         *        <code>"M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder m(Map<String, AttributeValue> m);

        /**
         * <p>
         * An attribute of type List. For example:
         * </p>
         * <p>
         * <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
         * </p>
         * 
         * @param l
         *        An attribute of type List. For example:</p>
         *        <p>
         *        <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder l(Collection<AttributeValue> l);

        /**
         * <p>
         * An attribute of type List. For example:
         * </p>
         * <p>
         * <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
         * </p>
         * 
         * @param l
         *        An attribute of type List. For example:</p>
         *        <p>
         *        <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder l(AttributeValue... l);

        /**
         * <p>
         * An attribute of type List. For example:
         * </p>
         * <p>
         * <code>"L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N": "3.14159"}]</code>
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dynamodb.model.AttributeValue.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.dynamodb.model.AttributeValue#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.dynamodb.model.AttributeValue.Builder#build()} is called immediately
         * and its result is passed to {@link #l(List<AttributeValue>)}.
         * 
         * @param l
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.dynamodb.model.AttributeValue.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #l(java.util.Collection<AttributeValue>)
         */
        Builder l(Consumer<Builder>... l);

        /**
         * <p>
         * An attribute of type Boolean. For example:
         * </p>
         * <p>
         * <code>"BOOL": true</code>
         * </p>
         * 
         * @param bool
         *        An attribute of type Boolean. For example:</p>
         *        <p>
         *        <code>"BOOL": true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bool(Boolean bool);

        /**
         * <p>
         * An attribute of type Null. For example:
         * </p>
         * <p>
         * <code>"NULL": true</code>
         * </p>
         * 
         * @param nul
         *        An attribute of type Null. For example:</p>
         *        <p>
         *        <code>"NULL": true</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nul(Boolean nul);
    }

    static final class BuilderImpl implements Builder {
        private String s;

        private String n;

        private SdkBytes b;

        private List<String> ss = DefaultSdkAutoConstructList.getInstance();

        private List<String> ns = DefaultSdkAutoConstructList.getInstance();

        private List<SdkBytes> bs = DefaultSdkAutoConstructList.getInstance();

        private Map<String, AttributeValue> m = DefaultSdkAutoConstructMap.getInstance();

        private List<AttributeValue> l = DefaultSdkAutoConstructList.getInstance();

        private Boolean bool;

        private Boolean nul;

        private Type type = Type.UNKNOWN_TO_SDK_VERSION;

        private Set<Type> setTypes = EnumSet.noneOf(Type.class);

        private BuilderImpl() {
        }

        private BuilderImpl(AttributeValue model) {
            s(model.s);
            n(model.n);
            b(model.b);
            ss(model.ss);
            ns(model.ns);
            bs(model.bs);
            m(model.m);
            l(model.l);
            bool(model.bool);
            nul(model.nul);
        }

        public final String getS() {
            return s;
        }

        public final void setS(String s) {
            Object oldValue = this.s;
            this.s = s;
            handleUnionValueChange(Type.S, oldValue, this.s);
        }

        @Override
        public final Builder s(String s) {
            Object oldValue = this.s;
            this.s = s;
            handleUnionValueChange(Type.S, oldValue, this.s);
            return this;
        }

        public final String getN() {
            return n;
        }

        public final void setN(String n) {
            Object oldValue = this.n;
            this.n = n;
            handleUnionValueChange(Type.N, oldValue, this.n);
        }

        @Override
        public final Builder n(String n) {
            Object oldValue = this.n;
            this.n = n;
            handleUnionValueChange(Type.N, oldValue, this.n);
            return this;
        }

        public final ByteBuffer getB() {
            return b == null ? null : b.asByteBuffer();
        }

        public final void setB(ByteBuffer b) {
            b(b == null ? null : SdkBytes.fromByteBuffer(b));
        }

        @Override
        public final Builder b(SdkBytes b) {
            Object oldValue = this.b;
            this.b = b;
            handleUnionValueChange(Type.B, oldValue, this.b);
            return this;
        }

        public final Collection<String> getSs() {
            if (ss instanceof SdkAutoConstructList) {
                return null;
            }
            return ss;
        }

        public final void setSs(Collection<String> ss) {
            Object oldValue = this.ss;
            this.ss = StringSetAttributeValueCopier.copy(ss);
            handleUnionValueChange(Type.SS, oldValue, this.ss);
        }

        @Override
        public final Builder ss(Collection<String> ss) {
            Object oldValue = this.ss;
            this.ss = StringSetAttributeValueCopier.copy(ss);
            handleUnionValueChange(Type.SS, oldValue, this.ss);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ss(String... ss) {
            ss(Arrays.asList(ss));
            return this;
        }

        public final Collection<String> getNs() {
            if (ns instanceof SdkAutoConstructList) {
                return null;
            }
            return ns;
        }

        public final void setNs(Collection<String> ns) {
            Object oldValue = this.ns;
            this.ns = NumberSetAttributeValueCopier.copy(ns);
            handleUnionValueChange(Type.NS, oldValue, this.ns);
        }

        @Override
        public final Builder ns(Collection<String> ns) {
            Object oldValue = this.ns;
            this.ns = NumberSetAttributeValueCopier.copy(ns);
            handleUnionValueChange(Type.NS, oldValue, this.ns);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ns(String... ns) {
            ns(Arrays.asList(ns));
            return this;
        }

        public final List<ByteBuffer> getBs() {
            if (bs instanceof SdkAutoConstructList) {
                return null;
            }
            return bs == null ? null : bs.stream().map(SdkBytes::asByteBuffer).collect(Collectors.toList());
        }

        public final void setBs(Collection<ByteBuffer> bs) {
            bs(bs == null ? null : bs.stream().map(SdkBytes::fromByteBuffer).collect(Collectors.toList()));
        }

        @Override
        public final Builder bs(Collection<SdkBytes> bs) {
            Object oldValue = this.bs;
            this.bs = BinarySetAttributeValueCopier.copy(bs);
            handleUnionValueChange(Type.BS, oldValue, this.bs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder bs(SdkBytes... bs) {
            bs(Arrays.asList(bs));
            return this;
        }

        public final Map<String, Builder> getM() {
            Map<String, Builder> result = MapAttributeValueCopier.copyToBuilder(this.m);
            if (result instanceof SdkAutoConstructMap) {
                return null;
            }
            return result;
        }

        public final void setM(Map<String, BuilderImpl> m) {
            Object oldValue = this.m;
            this.m = MapAttributeValueCopier.copyFromBuilder(m);
            handleUnionValueChange(Type.M, oldValue, this.m);
        }

        @Override
        public final Builder m(Map<String, AttributeValue> m) {
            Object oldValue = this.m;
            this.m = MapAttributeValueCopier.copy(m);
            handleUnionValueChange(Type.M, oldValue, this.m);
            return this;
        }

        public final List<Builder> getL() {
            List<Builder> result = ListAttributeValueCopier.copyToBuilder(this.l);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setL(Collection<BuilderImpl> l) {
            Object oldValue = this.l;
            this.l = ListAttributeValueCopier.copyFromBuilder(l);
            handleUnionValueChange(Type.L, oldValue, this.l);
        }

        @Override
        public final Builder l(Collection<AttributeValue> l) {
            Object oldValue = this.l;
            this.l = ListAttributeValueCopier.copy(l);
            handleUnionValueChange(Type.L, oldValue, this.l);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder l(AttributeValue... l) {
            l(Arrays.asList(l));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder l(Consumer<Builder>... l) {
            l(Stream.of(l).map(c -> AttributeValue.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getBool() {
            return bool;
        }

        public final void setBool(Boolean bool) {
            Object oldValue = this.bool;
            this.bool = bool;
            handleUnionValueChange(Type.BOOL, oldValue, this.bool);
        }

        @Override
        public final Builder bool(Boolean bool) {
            Object oldValue = this.bool;
            this.bool = bool;
            handleUnionValueChange(Type.BOOL, oldValue, this.bool);
            return this;
        }

        public final Boolean getNul() {
            return nul;
        }

        public final void setNul(Boolean nul) {
            Object oldValue = this.nul;
            this.nul = nul;
            handleUnionValueChange(Type.NUL, oldValue, this.nul);
        }

        @Override
        public final Builder nul(Boolean nul) {
            Object oldValue = this.nul;
            this.nul = nul;
            handleUnionValueChange(Type.NUL, oldValue, this.nul);
            return this;
        }

        @Override
        public AttributeValue build() {
            return new AttributeValue(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        private final void handleUnionValueChange(Type type, Object oldValue, Object newValue) {
            if (this.type == type || oldValue == newValue) {
                return;
            }
            if (newValue == null || newValue instanceof SdkAutoConstructList || newValue instanceof SdkAutoConstructMap) {
                setTypes.remove(type);
            } else if (oldValue == null || oldValue instanceof SdkAutoConstructList || oldValue instanceof SdkAutoConstructMap) {
                setTypes.add(type);
            }
            if (setTypes.size() == 1) {
                this.type = setTypes.iterator().next();
            } else if (setTypes.isEmpty()) {
                this.type = Type.UNKNOWN_TO_SDK_VERSION;
            } else {
                this.type = null;
            }
        }
    }

    /**
     * @see AttributeValue#type()
     */
    public enum Type {
        S,

        N,

        B,

        SS,

        NS,

        BS,

        M,

        L,

        BOOL,

        NUL,

        UNKNOWN_TO_SDK_VERSION
    }
}
