/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.endpoints.internal;

import java.net.URI;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.endpoints.AwsEndpointAttribute;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.endpoints.Endpoint;
import software.amazon.awssdk.services.dynamodb.endpoints.DynamoDbEndpointParams;
import software.amazon.awssdk.services.dynamodb.endpoints.DynamoDbEndpointProvider;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;
import software.amazon.awssdk.utils.MapUtils;
import software.amazon.awssdk.utils.Validate;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
public final class DefaultDynamoDbEndpointProvider implements DynamoDbEndpointProvider {
    private static final Logger LOG = Logger.loggerFor(DefaultDynamoDbEndpointProvider.class);

    private static final EndpointRuleset ENDPOINT_RULE_SET = ruleSet();

    private final EndpointAuthSchemeStrategy endpointAuthSchemeStrategy;

    public DefaultDynamoDbEndpointProvider() {
        EndpointAuthSchemeStrategyFactory endpointAuthSchemeStrategyFactory = new DefaultEndpointAuthSchemeStrategyFactory();
        this.endpointAuthSchemeStrategy = endpointAuthSchemeStrategyFactory.endpointAuthSchemeStrategy();
    }

    @Override
    public CompletableFuture<Endpoint> resolveEndpoint(DynamoDbEndpointParams endpointParams) {
        Validate.notNull(endpointParams.useDualStack(), "Parameter 'UseDualStack' must not be null");
        Validate.notNull(endpointParams.useFips(), "Parameter 'UseFIPS' must not be null");
        Value res = new DefaultRuleEngine().evaluate(ENDPOINT_RULE_SET, toIdentifierValueMap(endpointParams));
        try {
            return CompletableFuture.completedFuture(valueAsEndpointOrThrow(res));
        } catch (Exception error) {
            return CompletableFutureUtils.failedFuture(error);
        }
    }

    private static Map<Identifier, Value> toIdentifierValueMap(DynamoDbEndpointParams params) {
        Map<Identifier, Value> paramsMap = new HashMap<>();
        if (params.region() != null) {
            paramsMap.put(Identifier.of("Region"), Value.fromStr(params.region().id()));
        }
        if (params.useDualStack() != null) {
            paramsMap.put(Identifier.of("UseDualStack"), Value.fromBool(params.useDualStack()));
        }
        if (params.useFips() != null) {
            paramsMap.put(Identifier.of("UseFIPS"), Value.fromBool(params.useFips()));
        }
        if (params.endpoint() != null) {
            paramsMap.put(Identifier.of("Endpoint"), Value.fromStr(params.endpoint()));
        }
        if (params.accountId() != null) {
            paramsMap.put(Identifier.of("AccountId"), Value.fromStr(params.accountId()));
        }
        if (params.accountIdEndpointMode() != null) {
            paramsMap.put(Identifier.of("AccountIdEndpointMode"), Value.fromStr(params.accountIdEndpointMode()));
        }
        return paramsMap;
    }

    Endpoint valueAsEndpointOrThrow(Value value) {
        if (value instanceof Value.Endpoint) {
            Value.Endpoint endpoint = value.expectEndpoint();
            Endpoint.Builder builder = Endpoint.builder();
            builder.url(URI.create(endpoint.getUrl()));
            Map<String, List<String>> headers = endpoint.getHeaders();
            if (headers != null) {
                headers.forEach((name, values) -> values.forEach(v -> builder.putHeader(name, v)));
            }
            addKnownProperties(builder, endpoint.getProperties());
            return builder.build();
        } else if (value instanceof Value.Str) {
            String errorMsg = value.expectString();
            if (errorMsg.contains("Invalid ARN") && errorMsg.contains(":s3:::")) {
                errorMsg += ". Use the bucket name instead of simple bucket ARNs in GetBucketLocationRequest.";
            }
            throw SdkClientException.create(errorMsg);
        } else {
            throw SdkClientException.create("Rule engine return neither an endpoint result or error value. Returned value was: "
                    + value);
        }
    }

    private static Rule endpointRule_1() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: FIPS and custom endpoint are not supported");
    }

    private static Rule endpointRule_2() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: Dualstack and custom endpoint are not supported");
    }

    private static Rule endpointRule_3() {
        return Rule.builder().endpoint(EndpointResult.builder().url(Expr.of("{Endpoint}")).build());
    }

    private static Rule endpointRule_0() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Endpoint")))).build()
                                        .validate()).build())
                .treeRule(Arrays.asList(endpointRule_1(), endpointRule_2(), endpointRule_3()));
    }

    private static Rule endpointRule_7() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: FIPS and local endpoint are not supported");
    }

    private static Rule endpointRule_8() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build())
                .error("Invalid Configuration: Dualstack and local endpoint are not supported");
    }

    private static Rule endpointRule_9() {
        return Rule.builder().endpoint(
                EndpointResult
                        .builder()
                        .url(Expr.of("http://localhost:8000"))
                        .addProperty(
                                Identifier.of("authSchemes"),
                                Literal.fromTuple(Arrays.asList(Literal.fromRecord(MapUtils.of(Identifier.of("name"),
                                        Literal.fromStr("sigv4"), Identifier.of("signingName"), Literal.fromStr("dynamodb"),
                                        Identifier.of("signingRegion"), Literal.fromStr("us-east-1")))))).build());
    }

    private static Rule endpointRule_6() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("Region")), Expr.of("local"))).build()
                                        .validate()).build())
                .treeRule(Arrays.asList(endpointRule_7(), endpointRule_8(), endpointRule_9()));
    }

    private static Rule endpointRule_10() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("required")))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isSet")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("AccountId")))).build().validate()))
                                        .build().validate()).build())
                .error("AccountIdEndpointMode is required but no AccountID was provided or able to be loaded.");
    }

    private static Rule endpointRule_11() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("AccountId"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws"))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("booleanEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                                .validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("booleanEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true)))
                                                .build().validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("isValidHostLabel")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("AccountId")), Expr.of(false)))
                                                .build().validate())).build().validate()).build())
                .error("Credentials-sourced account ID parameter is invalid");
    }

    private static Rule endpointRule_14() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("disabled")))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult.builder()
                                .url(Expr.of("https://dynamodb-fips.{Region}.{PartitionResult#dualStackDnsSuffix}")).build());
    }

    private static Rule endpointRule_15() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://dynamodb-fips.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                        .build());
    }

    private static Rule endpointRule_13() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_14(), endpointRule_15()));
    }

    private static Rule endpointRule_16() {
        return Rule.builder().error("FIPS and DualStack are enabled, but this partition does not support one or both");
    }

    private static Rule endpointRule_12() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_13(), endpointRule_16()));
    }

    private static Rule endpointRule_20() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("disabled")))
                                        .build().validate()).build())
                .endpoint(EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_21() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_19() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws-us-gov")))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_20(), endpointRule_21()));
    }

    private static Rule endpointRule_22() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("disabled")))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult.builder().url(Expr.of("https://dynamodb-fips.{Region}.{PartitionResult#dnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_23() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://dynamodb-fips.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_18() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsFIPS"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_19(), endpointRule_22(), endpointRule_23()));
    }

    private static Rule endpointRule_24() {
        return Rule.builder().error("FIPS is enabled but this partition does not support FIPS");
    }

    private static Rule endpointRule_17() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_18(), endpointRule_24()));
    }

    private static Rule endpointRule_27() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("disabled")))
                                        .build().validate()).build())
                .endpoint(
                        EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dualStackDnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_28() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dualStackDnsSuffix}")).build());
    }

    private static Rule endpointRule_26() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("booleanEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("supportsDualStack"))).build().validate(), Expr.of(true)))
                                        .build().validate()).build())
                .treeRule(Arrays.asList(endpointRule_27(), endpointRule_28()));
    }

    private static Rule endpointRule_29() {
        return Rule.builder().error("DualStack is enabled but this partition does not support DualStack");
    }

    private static Rule endpointRule_25() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("booleanEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_26(), endpointRule_29()));
    }

    private static Rule endpointRule_30() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")))).build().validate())
                                .build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(Expr.ref(Identifier.of("AccountIdEndpointMode")), Expr.of("disabled")))
                                        .build().validate()).build())
                .endpoint(EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_31() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("AccountId"))))
                                        .build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("stringEquals")
                                        .argv(Arrays.asList(
                                                FnNode.builder()
                                                        .fn("getAttr")
                                                        .argv(Arrays.asList(Expr.ref(Identifier.of("PartitionResult")),
                                                                Expr.of("name"))).build().validate(), Expr.of("aws"))).build()
                                        .validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("booleanEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("UseFIPS")), Expr.of(true))).build()
                                                .validate())).build().validate()).build())
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode
                                        .builder()
                                        .fn("not")
                                        .argv(Arrays.asList(FnNode.builder().fn("booleanEquals")
                                                .argv(Arrays.asList(Expr.ref(Identifier.of("UseDualStack")), Expr.of(true)))
                                                .build().validate())).build().validate()).build())
                .endpoint(
                        EndpointResult.builder().url(Expr.of("https://{AccountId}.ddb.{Region}.{PartitionResult#dnsSuffix}"))
                                .build());
    }

    private static Rule endpointRule_32() {
        return Rule.builder().endpoint(
                EndpointResult.builder().url(Expr.of("https://dynamodb.{Region}.{PartitionResult#dnsSuffix}")).build());
    }

    private static Rule endpointRule_5() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("aws.partition").argv(Arrays.asList(Expr.ref(Identifier.of("Region"))))
                                        .build().validate()).result("PartitionResult").build())
                .treeRule(
                        Arrays.asList(endpointRule_6(), endpointRule_10(), endpointRule_11(), endpointRule_12(),
                                endpointRule_17(), endpointRule_25(), endpointRule_30(), endpointRule_31(), endpointRule_32()));
    }

    private static Rule endpointRule_4() {
        return Rule
                .builder()
                .addCondition(
                        Condition
                                .builder()
                                .fn(FnNode.builder().fn("isSet").argv(Arrays.asList(Expr.ref(Identifier.of("Region")))).build()
                                        .validate()).build()).treeRule(Arrays.asList(endpointRule_5()));
    }

    private static Rule endpointRule_33() {
        return Rule.builder().error("Invalid Configuration: Missing Region");
    }

    private static EndpointRuleset ruleSet() {
        return EndpointRuleset
                .builder()
                .version("1.0")
                .serviceId(null)
                .parameters(
                        Parameters
                                .builder()
                                .addParameter(
                                        Parameter.builder().name("Region").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Region")
                                                .documentation("The AWS region used to dispatch the request.").build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseDualStack")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseDualStack")
                                                .documentation(
                                                        "When true, use the dual-stack endpoint. If the configured endpoint does not support dual-stack, dispatching the request MAY return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter
                                                .builder()
                                                .name("UseFIPS")
                                                .type(ParameterType.fromValue("Boolean"))
                                                .required(true)
                                                .builtIn("AWS::UseFIPS")
                                                .documentation(
                                                        "When true, send this request to the FIPS-compliant regional endpoint. If the configured endpoint does not have a FIPS compliant endpoint, dispatching the request will return an error.")
                                                .defaultValue(Value.fromBool(false)).build())
                                .addParameter(
                                        Parameter.builder().name("Endpoint").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("SDK::Endpoint")
                                                .documentation("Override the endpoint used to send this request").build())
                                .addParameter(
                                        Parameter.builder().name("AccountId").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Auth::AccountId")
                                                .documentation("The AWS AccountId used for the request.").build())
                                .addParameter(
                                        Parameter.builder().name("AccountIdEndpointMode").type(ParameterType.fromValue("String"))
                                                .required(false).builtIn("AWS::Auth::AccountIdEndpointMode")
                                                .documentation("The AccountId Endpoint Mode.").build()).build())
                .addRule(endpointRule_0()).addRule(endpointRule_4()).addRule(endpointRule_33()).build();
    }

    @Override
    public boolean equals(Object rhs) {
        return rhs != null && getClass().equals(rhs.getClass());
    }

    @Override
    public int hashCode() {
        return getClass().hashCode();
    }

    private void addKnownProperties(Endpoint.Builder builder, Map<String, Value> properties) {
        properties.forEach((n, v) -> {
            switch (n) {
            case "authSchemes":
                builder.putAttribute(AwsEndpointAttribute.AUTH_SCHEMES, endpointAuthSchemeStrategy.createAuthSchemes(v));
                break;
            default:
                LOG.debug(() -> "Ignoring unknown endpoint property: " + n);
                break;
            }
        });
    }
}
