/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the details of the replica.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ReplicaDescription implements SdkPojo, Serializable,
        ToCopyableBuilder<ReplicaDescription.Builder, ReplicaDescription> {
    private static final SdkField<String> REGION_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RegionName").getter(getter(ReplicaDescription::regionName)).setter(setter(Builder::regionName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RegionName").build()).build();

    private static final SdkField<String> REPLICA_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicaStatus").getter(getter(ReplicaDescription::replicaStatusAsString))
            .setter(setter(Builder::replicaStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaStatus").build()).build();

    private static final SdkField<String> REPLICA_STATUS_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ReplicaStatusDescription").getter(getter(ReplicaDescription::replicaStatusDescription))
            .setter(setter(Builder::replicaStatusDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaStatusDescription").build())
            .build();

    private static final SdkField<String> REPLICA_STATUS_PERCENT_PROGRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReplicaStatusPercentProgress")
            .getter(getter(ReplicaDescription::replicaStatusPercentProgress))
            .setter(setter(Builder::replicaStatusPercentProgress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaStatusPercentProgress")
                    .build()).build();

    private static final SdkField<String> KMS_MASTER_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("KMSMasterKeyId").getter(getter(ReplicaDescription::kmsMasterKeyId))
            .setter(setter(Builder::kmsMasterKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KMSMasterKeyId").build()).build();

    private static final SdkField<ProvisionedThroughputOverride> PROVISIONED_THROUGHPUT_OVERRIDE_FIELD = SdkField
            .<ProvisionedThroughputOverride> builder(MarshallingType.SDK_POJO)
            .memberName("ProvisionedThroughputOverride")
            .getter(getter(ReplicaDescription::provisionedThroughputOverride))
            .setter(setter(Builder::provisionedThroughputOverride))
            .constructor(ProvisionedThroughputOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProvisionedThroughputOverride")
                    .build()).build();

    private static final SdkField<OnDemandThroughputOverride> ON_DEMAND_THROUGHPUT_OVERRIDE_FIELD = SdkField
            .<OnDemandThroughputOverride> builder(MarshallingType.SDK_POJO)
            .memberName("OnDemandThroughputOverride")
            .getter(getter(ReplicaDescription::onDemandThroughputOverride))
            .setter(setter(Builder::onDemandThroughputOverride))
            .constructor(OnDemandThroughputOverride::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OnDemandThroughputOverride").build())
            .build();

    private static final SdkField<TableWarmThroughputDescription> WARM_THROUGHPUT_FIELD = SdkField
            .<TableWarmThroughputDescription> builder(MarshallingType.SDK_POJO).memberName("WarmThroughput")
            .getter(getter(ReplicaDescription::warmThroughput)).setter(setter(Builder::warmThroughput))
            .constructor(TableWarmThroughputDescription::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("WarmThroughput").build()).build();

    private static final SdkField<List<ReplicaGlobalSecondaryIndexDescription>> GLOBAL_SECONDARY_INDEXES_FIELD = SdkField
            .<List<ReplicaGlobalSecondaryIndexDescription>> builder(MarshallingType.LIST)
            .memberName("GlobalSecondaryIndexes")
            .getter(getter(ReplicaDescription::globalSecondaryIndexes))
            .setter(setter(Builder::globalSecondaryIndexes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GlobalSecondaryIndexes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ReplicaGlobalSecondaryIndexDescription> builder(MarshallingType.SDK_POJO)
                                            .constructor(ReplicaGlobalSecondaryIndexDescription::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> REPLICA_INACCESSIBLE_DATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReplicaInaccessibleDateTime")
            .getter(getter(ReplicaDescription::replicaInaccessibleDateTime))
            .setter(setter(Builder::replicaInaccessibleDateTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaInaccessibleDateTime")
                    .build()).build();

    private static final SdkField<TableClassSummary> REPLICA_TABLE_CLASS_SUMMARY_FIELD = SdkField
            .<TableClassSummary> builder(MarshallingType.SDK_POJO).memberName("ReplicaTableClassSummary")
            .getter(getter(ReplicaDescription::replicaTableClassSummary)).setter(setter(Builder::replicaTableClassSummary))
            .constructor(TableClassSummary::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReplicaTableClassSummary").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REGION_NAME_FIELD,
            REPLICA_STATUS_FIELD, REPLICA_STATUS_DESCRIPTION_FIELD, REPLICA_STATUS_PERCENT_PROGRESS_FIELD,
            KMS_MASTER_KEY_ID_FIELD, PROVISIONED_THROUGHPUT_OVERRIDE_FIELD, ON_DEMAND_THROUGHPUT_OVERRIDE_FIELD,
            WARM_THROUGHPUT_FIELD, GLOBAL_SECONDARY_INDEXES_FIELD, REPLICA_INACCESSIBLE_DATE_TIME_FIELD,
            REPLICA_TABLE_CLASS_SUMMARY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("RegionName", REGION_NAME_FIELD);
                    put("ReplicaStatus", REPLICA_STATUS_FIELD);
                    put("ReplicaStatusDescription", REPLICA_STATUS_DESCRIPTION_FIELD);
                    put("ReplicaStatusPercentProgress", REPLICA_STATUS_PERCENT_PROGRESS_FIELD);
                    put("KMSMasterKeyId", KMS_MASTER_KEY_ID_FIELD);
                    put("ProvisionedThroughputOverride", PROVISIONED_THROUGHPUT_OVERRIDE_FIELD);
                    put("OnDemandThroughputOverride", ON_DEMAND_THROUGHPUT_OVERRIDE_FIELD);
                    put("WarmThroughput", WARM_THROUGHPUT_FIELD);
                    put("GlobalSecondaryIndexes", GLOBAL_SECONDARY_INDEXES_FIELD);
                    put("ReplicaInaccessibleDateTime", REPLICA_INACCESSIBLE_DATE_TIME_FIELD);
                    put("ReplicaTableClassSummary", REPLICA_TABLE_CLASS_SUMMARY_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String regionName;

    private final String replicaStatus;

    private final String replicaStatusDescription;

    private final String replicaStatusPercentProgress;

    private final String kmsMasterKeyId;

    private final ProvisionedThroughputOverride provisionedThroughputOverride;

    private final OnDemandThroughputOverride onDemandThroughputOverride;

    private final TableWarmThroughputDescription warmThroughput;

    private final List<ReplicaGlobalSecondaryIndexDescription> globalSecondaryIndexes;

    private final Instant replicaInaccessibleDateTime;

    private final TableClassSummary replicaTableClassSummary;

    private ReplicaDescription(BuilderImpl builder) {
        this.regionName = builder.regionName;
        this.replicaStatus = builder.replicaStatus;
        this.replicaStatusDescription = builder.replicaStatusDescription;
        this.replicaStatusPercentProgress = builder.replicaStatusPercentProgress;
        this.kmsMasterKeyId = builder.kmsMasterKeyId;
        this.provisionedThroughputOverride = builder.provisionedThroughputOverride;
        this.onDemandThroughputOverride = builder.onDemandThroughputOverride;
        this.warmThroughput = builder.warmThroughput;
        this.globalSecondaryIndexes = builder.globalSecondaryIndexes;
        this.replicaInaccessibleDateTime = builder.replicaInaccessibleDateTime;
        this.replicaTableClassSummary = builder.replicaTableClassSummary;
    }

    /**
     * <p>
     * The name of the Region.
     * </p>
     * 
     * @return The name of the Region.
     */
    public final String regionName() {
        return regionName;
    }

    /**
     * <p>
     * The current state of the replica:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The replica is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The replica is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The replica is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The replica is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has been
     * disabled.
     * </p>
     * <note>
     * <p>
     * If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this replica
     * from the replication group. The replica will not be deleted and replication will stop from and to this region.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is inaccessible.
     * </p>
     * <note>
     * <p>
     * If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
     * replication group. The replica will not be deleted and replication will stop from and to this region.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicaStatus}
     * will return {@link ReplicaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #replicaStatusAsString}.
     * </p>
     * 
     * @return The current state of the replica:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The replica is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The replica is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The replica is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The replica is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has
     *         been disabled.
     *         </p>
     *         <note>
     *         <p>
     *         If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this
     *         replica from the replication group. The replica will not be deleted and replication will stop from and to
     *         this region.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is
     *         inaccessible.
     *         </p>
     *         <note>
     *         <p>
     *         If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
     *         replication group. The replica will not be deleted and replication will stop from and to this region.
     *         </p>
     *         </note></li>
     * @see ReplicaStatus
     */
    public final ReplicaStatus replicaStatus() {
        return ReplicaStatus.fromValue(replicaStatus);
    }

    /**
     * <p>
     * The current state of the replica:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - The replica is being created.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>UPDATING</code> - The replica is being updated.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETING</code> - The replica is being deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - The replica is ready for use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has been
     * disabled.
     * </p>
     * <note>
     * <p>
     * If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this replica
     * from the replication group. The replica will not be deleted and replication will stop from and to this region.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is inaccessible.
     * </p>
     * <note>
     * <p>
     * If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
     * replication group. The replica will not be deleted and replication will stop from and to this region.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #replicaStatus}
     * will return {@link ReplicaStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #replicaStatusAsString}.
     * </p>
     * 
     * @return The current state of the replica:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>CREATING</code> - The replica is being created.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>UPDATING</code> - The replica is being updated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>DELETING</code> - The replica is being deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ACTIVE</code> - The replica is ready for use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has
     *         been disabled.
     *         </p>
     *         <note>
     *         <p>
     *         If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this
     *         replica from the replication group. The replica will not be deleted and replication will stop from and to
     *         this region.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is
     *         inaccessible.
     *         </p>
     *         <note>
     *         <p>
     *         If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
     *         replication group. The replica will not be deleted and replication will stop from and to this region.
     *         </p>
     *         </note></li>
     * @see ReplicaStatus
     */
    public final String replicaStatusAsString() {
        return replicaStatus;
    }

    /**
     * <p>
     * Detailed information about the replica status.
     * </p>
     * 
     * @return Detailed information about the replica status.
     */
    public final String replicaStatusDescription() {
        return replicaStatusDescription;
    }

    /**
     * <p>
     * Specifies the progress of a Create, Update, or Delete action on the replica as a percentage.
     * </p>
     * 
     * @return Specifies the progress of a Create, Update, or Delete action on the replica as a percentage.
     */
    public final String replicaStatusPercentProgress() {
        return replicaStatusPercentProgress;
    }

    /**
     * <p>
     * The KMS key of the replica that will be used for KMS encryption.
     * </p>
     * 
     * @return The KMS key of the replica that will be used for KMS encryption.
     */
    public final String kmsMasterKeyId() {
        return kmsMasterKeyId;
    }

    /**
     * <p>
     * Replica-specific provisioned throughput. If not described, uses the source table's provisioned throughput
     * settings.
     * </p>
     * 
     * @return Replica-specific provisioned throughput. If not described, uses the source table's provisioned throughput
     *         settings.
     */
    public final ProvisionedThroughputOverride provisionedThroughputOverride() {
        return provisionedThroughputOverride;
    }

    /**
     * <p>
     * Overrides the maximum on-demand throughput settings for the specified replica table.
     * </p>
     * 
     * @return Overrides the maximum on-demand throughput settings for the specified replica table.
     */
    public final OnDemandThroughputOverride onDemandThroughputOverride() {
        return onDemandThroughputOverride;
    }

    /**
     * <p>
     * Represents the warm throughput value for this replica.
     * </p>
     * 
     * @return Represents the warm throughput value for this replica.
     */
    public final TableWarmThroughputDescription warmThroughput() {
        return warmThroughput;
    }

    /**
     * For responses, this returns true if the service returned a value for the GlobalSecondaryIndexes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasGlobalSecondaryIndexes() {
        return globalSecondaryIndexes != null && !(globalSecondaryIndexes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Replica-specific global secondary index settings.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGlobalSecondaryIndexes} method.
     * </p>
     * 
     * @return Replica-specific global secondary index settings.
     */
    public final List<ReplicaGlobalSecondaryIndexDescription> globalSecondaryIndexes() {
        return globalSecondaryIndexes;
    }

    /**
     * <p>
     * The time at which the replica was first detected as inaccessible. To determine cause of inaccessibility check the
     * <code>ReplicaStatus</code> property.
     * </p>
     * 
     * @return The time at which the replica was first detected as inaccessible. To determine cause of inaccessibility
     *         check the <code>ReplicaStatus</code> property.
     */
    public final Instant replicaInaccessibleDateTime() {
        return replicaInaccessibleDateTime;
    }

    /**
     * Returns the value of the ReplicaTableClassSummary property for this object.
     * 
     * @return The value of the ReplicaTableClassSummary property for this object.
     */
    public final TableClassSummary replicaTableClassSummary() {
        return replicaTableClassSummary;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(regionName());
        hashCode = 31 * hashCode + Objects.hashCode(replicaStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(replicaStatusDescription());
        hashCode = 31 * hashCode + Objects.hashCode(replicaStatusPercentProgress());
        hashCode = 31 * hashCode + Objects.hashCode(kmsMasterKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedThroughputOverride());
        hashCode = 31 * hashCode + Objects.hashCode(onDemandThroughputOverride());
        hashCode = 31 * hashCode + Objects.hashCode(warmThroughput());
        hashCode = 31 * hashCode + Objects.hashCode(hasGlobalSecondaryIndexes() ? globalSecondaryIndexes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(replicaInaccessibleDateTime());
        hashCode = 31 * hashCode + Objects.hashCode(replicaTableClassSummary());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ReplicaDescription)) {
            return false;
        }
        ReplicaDescription other = (ReplicaDescription) obj;
        return Objects.equals(regionName(), other.regionName())
                && Objects.equals(replicaStatusAsString(), other.replicaStatusAsString())
                && Objects.equals(replicaStatusDescription(), other.replicaStatusDescription())
                && Objects.equals(replicaStatusPercentProgress(), other.replicaStatusPercentProgress())
                && Objects.equals(kmsMasterKeyId(), other.kmsMasterKeyId())
                && Objects.equals(provisionedThroughputOverride(), other.provisionedThroughputOverride())
                && Objects.equals(onDemandThroughputOverride(), other.onDemandThroughputOverride())
                && Objects.equals(warmThroughput(), other.warmThroughput())
                && hasGlobalSecondaryIndexes() == other.hasGlobalSecondaryIndexes()
                && Objects.equals(globalSecondaryIndexes(), other.globalSecondaryIndexes())
                && Objects.equals(replicaInaccessibleDateTime(), other.replicaInaccessibleDateTime())
                && Objects.equals(replicaTableClassSummary(), other.replicaTableClassSummary());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ReplicaDescription").add("RegionName", regionName())
                .add("ReplicaStatus", replicaStatusAsString()).add("ReplicaStatusDescription", replicaStatusDescription())
                .add("ReplicaStatusPercentProgress", replicaStatusPercentProgress()).add("KMSMasterKeyId", kmsMasterKeyId())
                .add("ProvisionedThroughputOverride", provisionedThroughputOverride())
                .add("OnDemandThroughputOverride", onDemandThroughputOverride()).add("WarmThroughput", warmThroughput())
                .add("GlobalSecondaryIndexes", hasGlobalSecondaryIndexes() ? globalSecondaryIndexes() : null)
                .add("ReplicaInaccessibleDateTime", replicaInaccessibleDateTime())
                .add("ReplicaTableClassSummary", replicaTableClassSummary()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RegionName":
            return Optional.ofNullable(clazz.cast(regionName()));
        case "ReplicaStatus":
            return Optional.ofNullable(clazz.cast(replicaStatusAsString()));
        case "ReplicaStatusDescription":
            return Optional.ofNullable(clazz.cast(replicaStatusDescription()));
        case "ReplicaStatusPercentProgress":
            return Optional.ofNullable(clazz.cast(replicaStatusPercentProgress()));
        case "KMSMasterKeyId":
            return Optional.ofNullable(clazz.cast(kmsMasterKeyId()));
        case "ProvisionedThroughputOverride":
            return Optional.ofNullable(clazz.cast(provisionedThroughputOverride()));
        case "OnDemandThroughputOverride":
            return Optional.ofNullable(clazz.cast(onDemandThroughputOverride()));
        case "WarmThroughput":
            return Optional.ofNullable(clazz.cast(warmThroughput()));
        case "GlobalSecondaryIndexes":
            return Optional.ofNullable(clazz.cast(globalSecondaryIndexes()));
        case "ReplicaInaccessibleDateTime":
            return Optional.ofNullable(clazz.cast(replicaInaccessibleDateTime()));
        case "ReplicaTableClassSummary":
            return Optional.ofNullable(clazz.cast(replicaTableClassSummary()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ReplicaDescription, T> g) {
        return obj -> g.apply((ReplicaDescription) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ReplicaDescription> {
        /**
         * <p>
         * The name of the Region.
         * </p>
         * 
         * @param regionName
         *        The name of the Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder regionName(String regionName);

        /**
         * <p>
         * The current state of the replica:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The replica is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The replica is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The replica is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The replica is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has been
         * disabled.
         * </p>
         * <note>
         * <p>
         * If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this
         * replica from the replication group. The replica will not be deleted and replication will stop from and to
         * this region.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is inaccessible.
         * </p>
         * <note>
         * <p>
         * If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
         * replication group. The replica will not be deleted and replication will stop from and to this region.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param replicaStatus
         *        The current state of the replica:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The replica is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The replica is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The replica is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The replica is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has
         *        been disabled.
         *        </p>
         *        <note>
         *        <p>
         *        If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove
         *        this replica from the replication group. The replica will not be deleted and replication will stop
         *        from and to this region.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is
         *        inaccessible.
         *        </p>
         *        <note>
         *        <p>
         *        If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
         *        replication group. The replica will not be deleted and replication will stop from and to this region.
         *        </p>
         *        </note></li>
         * @see ReplicaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicaStatus
         */
        Builder replicaStatus(String replicaStatus);

        /**
         * <p>
         * The current state of the replica:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>CREATING</code> - The replica is being created.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>UPDATING</code> - The replica is being updated.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>DELETING</code> - The replica is being deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ACTIVE</code> - The replica is ready for use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has been
         * disabled.
         * </p>
         * <note>
         * <p>
         * If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove this
         * replica from the replication group. The replica will not be deleted and replication will stop from and to
         * this region.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is inaccessible.
         * </p>
         * <note>
         * <p>
         * If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
         * replication group. The replica will not be deleted and replication will stop from and to this region.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param replicaStatus
         *        The current state of the replica:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>CREATING</code> - The replica is being created.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>UPDATING</code> - The replica is being updated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>DELETING</code> - The replica is being deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ACTIVE</code> - The replica is ready for use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>REGION_DISABLED</code> - The replica is inaccessible because the Amazon Web Services Region has
         *        been disabled.
         *        </p>
         *        <note>
         *        <p>
         *        If the Amazon Web Services Region remains inaccessible for more than 20 hours, DynamoDB will remove
         *        this replica from the replication group. The replica will not be deleted and replication will stop
         *        from and to this region.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>INACCESSIBLE_ENCRYPTION_CREDENTIALS </code> - The KMS key used to encrypt the table is
         *        inaccessible.
         *        </p>
         *        <note>
         *        <p>
         *        If the KMS key remains inaccessible for more than 20 hours, DynamoDB will remove this replica from the
         *        replication group. The replica will not be deleted and replication will stop from and to this region.
         *        </p>
         *        </note></li>
         * @see ReplicaStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicaStatus
         */
        Builder replicaStatus(ReplicaStatus replicaStatus);

        /**
         * <p>
         * Detailed information about the replica status.
         * </p>
         * 
         * @param replicaStatusDescription
         *        Detailed information about the replica status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaStatusDescription(String replicaStatusDescription);

        /**
         * <p>
         * Specifies the progress of a Create, Update, or Delete action on the replica as a percentage.
         * </p>
         * 
         * @param replicaStatusPercentProgress
         *        Specifies the progress of a Create, Update, or Delete action on the replica as a percentage.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaStatusPercentProgress(String replicaStatusPercentProgress);

        /**
         * <p>
         * The KMS key of the replica that will be used for KMS encryption.
         * </p>
         * 
         * @param kmsMasterKeyId
         *        The KMS key of the replica that will be used for KMS encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsMasterKeyId(String kmsMasterKeyId);

        /**
         * <p>
         * Replica-specific provisioned throughput. If not described, uses the source table's provisioned throughput
         * settings.
         * </p>
         * 
         * @param provisionedThroughputOverride
         *        Replica-specific provisioned throughput. If not described, uses the source table's provisioned
         *        throughput settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedThroughputOverride(ProvisionedThroughputOverride provisionedThroughputOverride);

        /**
         * <p>
         * Replica-specific provisioned throughput. If not described, uses the source table's provisioned throughput
         * settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link ProvisionedThroughputOverride.Builder}
         * avoiding the need to create one manually via {@link ProvisionedThroughputOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ProvisionedThroughputOverride.Builder#build()} is called
         * immediately and its result is passed to {@link #provisionedThroughputOverride(ProvisionedThroughputOverride)}.
         * 
         * @param provisionedThroughputOverride
         *        a consumer that will call methods on {@link ProvisionedThroughputOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedThroughputOverride(ProvisionedThroughputOverride)
         */
        default Builder provisionedThroughputOverride(
                Consumer<ProvisionedThroughputOverride.Builder> provisionedThroughputOverride) {
            return provisionedThroughputOverride(ProvisionedThroughputOverride.builder()
                    .applyMutation(provisionedThroughputOverride).build());
        }

        /**
         * <p>
         * Overrides the maximum on-demand throughput settings for the specified replica table.
         * </p>
         * 
         * @param onDemandThroughputOverride
         *        Overrides the maximum on-demand throughput settings for the specified replica table.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandThroughputOverride(OnDemandThroughputOverride onDemandThroughputOverride);

        /**
         * <p>
         * Overrides the maximum on-demand throughput settings for the specified replica table.
         * </p>
         * This is a convenience method that creates an instance of the {@link OnDemandThroughputOverride.Builder}
         * avoiding the need to create one manually via {@link OnDemandThroughputOverride#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link OnDemandThroughputOverride.Builder#build()} is called immediately
         * and its result is passed to {@link #onDemandThroughputOverride(OnDemandThroughputOverride)}.
         * 
         * @param onDemandThroughputOverride
         *        a consumer that will call methods on {@link OnDemandThroughputOverride.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #onDemandThroughputOverride(OnDemandThroughputOverride)
         */
        default Builder onDemandThroughputOverride(Consumer<OnDemandThroughputOverride.Builder> onDemandThroughputOverride) {
            return onDemandThroughputOverride(OnDemandThroughputOverride.builder().applyMutation(onDemandThroughputOverride)
                    .build());
        }

        /**
         * <p>
         * Represents the warm throughput value for this replica.
         * </p>
         * 
         * @param warmThroughput
         *        Represents the warm throughput value for this replica.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder warmThroughput(TableWarmThroughputDescription warmThroughput);

        /**
         * <p>
         * Represents the warm throughput value for this replica.
         * </p>
         * This is a convenience method that creates an instance of the {@link TableWarmThroughputDescription.Builder}
         * avoiding the need to create one manually via {@link TableWarmThroughputDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableWarmThroughputDescription.Builder#build()} is called
         * immediately and its result is passed to {@link #warmThroughput(TableWarmThroughputDescription)}.
         * 
         * @param warmThroughput
         *        a consumer that will call methods on {@link TableWarmThroughputDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #warmThroughput(TableWarmThroughputDescription)
         */
        default Builder warmThroughput(Consumer<TableWarmThroughputDescription.Builder> warmThroughput) {
            return warmThroughput(TableWarmThroughputDescription.builder().applyMutation(warmThroughput).build());
        }

        /**
         * <p>
         * Replica-specific global secondary index settings.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        Replica-specific global secondary index settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(Collection<ReplicaGlobalSecondaryIndexDescription> globalSecondaryIndexes);

        /**
         * <p>
         * Replica-specific global secondary index settings.
         * </p>
         * 
         * @param globalSecondaryIndexes
         *        Replica-specific global secondary index settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder globalSecondaryIndexes(ReplicaGlobalSecondaryIndexDescription... globalSecondaryIndexes);

        /**
         * <p>
         * Replica-specific global secondary index settings.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.dynamodb.model.ReplicaGlobalSecondaryIndexDescription.Builder}
         * avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.dynamodb.model.ReplicaGlobalSecondaryIndexDescription#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.dynamodb.model.ReplicaGlobalSecondaryIndexDescription.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #globalSecondaryIndexes(List<ReplicaGlobalSecondaryIndexDescription>)}.
         * 
         * @param globalSecondaryIndexes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.dynamodb.model.ReplicaGlobalSecondaryIndexDescription.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #globalSecondaryIndexes(java.util.Collection<ReplicaGlobalSecondaryIndexDescription>)
         */
        Builder globalSecondaryIndexes(Consumer<ReplicaGlobalSecondaryIndexDescription.Builder>... globalSecondaryIndexes);

        /**
         * <p>
         * The time at which the replica was first detected as inaccessible. To determine cause of inaccessibility check
         * the <code>ReplicaStatus</code> property.
         * </p>
         * 
         * @param replicaInaccessibleDateTime
         *        The time at which the replica was first detected as inaccessible. To determine cause of
         *        inaccessibility check the <code>ReplicaStatus</code> property.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaInaccessibleDateTime(Instant replicaInaccessibleDateTime);

        /**
         * Sets the value of the ReplicaTableClassSummary property for this object.
         *
         * @param replicaTableClassSummary
         *        The new value for the ReplicaTableClassSummary property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder replicaTableClassSummary(TableClassSummary replicaTableClassSummary);

        /**
         * Sets the value of the ReplicaTableClassSummary property for this object.
         *
         * This is a convenience method that creates an instance of the {@link TableClassSummary.Builder} avoiding the
         * need to create one manually via {@link TableClassSummary#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TableClassSummary.Builder#build()} is called immediately and its
         * result is passed to {@link #replicaTableClassSummary(TableClassSummary)}.
         * 
         * @param replicaTableClassSummary
         *        a consumer that will call methods on {@link TableClassSummary.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #replicaTableClassSummary(TableClassSummary)
         */
        default Builder replicaTableClassSummary(Consumer<TableClassSummary.Builder> replicaTableClassSummary) {
            return replicaTableClassSummary(TableClassSummary.builder().applyMutation(replicaTableClassSummary).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String regionName;

        private String replicaStatus;

        private String replicaStatusDescription;

        private String replicaStatusPercentProgress;

        private String kmsMasterKeyId;

        private ProvisionedThroughputOverride provisionedThroughputOverride;

        private OnDemandThroughputOverride onDemandThroughputOverride;

        private TableWarmThroughputDescription warmThroughput;

        private List<ReplicaGlobalSecondaryIndexDescription> globalSecondaryIndexes = DefaultSdkAutoConstructList.getInstance();

        private Instant replicaInaccessibleDateTime;

        private TableClassSummary replicaTableClassSummary;

        private BuilderImpl() {
        }

        private BuilderImpl(ReplicaDescription model) {
            regionName(model.regionName);
            replicaStatus(model.replicaStatus);
            replicaStatusDescription(model.replicaStatusDescription);
            replicaStatusPercentProgress(model.replicaStatusPercentProgress);
            kmsMasterKeyId(model.kmsMasterKeyId);
            provisionedThroughputOverride(model.provisionedThroughputOverride);
            onDemandThroughputOverride(model.onDemandThroughputOverride);
            warmThroughput(model.warmThroughput);
            globalSecondaryIndexes(model.globalSecondaryIndexes);
            replicaInaccessibleDateTime(model.replicaInaccessibleDateTime);
            replicaTableClassSummary(model.replicaTableClassSummary);
        }

        public final String getRegionName() {
            return regionName;
        }

        public final void setRegionName(String regionName) {
            this.regionName = regionName;
        }

        @Override
        public final Builder regionName(String regionName) {
            this.regionName = regionName;
            return this;
        }

        public final String getReplicaStatus() {
            return replicaStatus;
        }

        public final void setReplicaStatus(String replicaStatus) {
            this.replicaStatus = replicaStatus;
        }

        @Override
        public final Builder replicaStatus(String replicaStatus) {
            this.replicaStatus = replicaStatus;
            return this;
        }

        @Override
        public final Builder replicaStatus(ReplicaStatus replicaStatus) {
            this.replicaStatus(replicaStatus == null ? null : replicaStatus.toString());
            return this;
        }

        public final String getReplicaStatusDescription() {
            return replicaStatusDescription;
        }

        public final void setReplicaStatusDescription(String replicaStatusDescription) {
            this.replicaStatusDescription = replicaStatusDescription;
        }

        @Override
        public final Builder replicaStatusDescription(String replicaStatusDescription) {
            this.replicaStatusDescription = replicaStatusDescription;
            return this;
        }

        public final String getReplicaStatusPercentProgress() {
            return replicaStatusPercentProgress;
        }

        public final void setReplicaStatusPercentProgress(String replicaStatusPercentProgress) {
            this.replicaStatusPercentProgress = replicaStatusPercentProgress;
        }

        @Override
        public final Builder replicaStatusPercentProgress(String replicaStatusPercentProgress) {
            this.replicaStatusPercentProgress = replicaStatusPercentProgress;
            return this;
        }

        public final String getKmsMasterKeyId() {
            return kmsMasterKeyId;
        }

        public final void setKmsMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
        }

        @Override
        public final Builder kmsMasterKeyId(String kmsMasterKeyId) {
            this.kmsMasterKeyId = kmsMasterKeyId;
            return this;
        }

        public final ProvisionedThroughputOverride.Builder getProvisionedThroughputOverride() {
            return provisionedThroughputOverride != null ? provisionedThroughputOverride.toBuilder() : null;
        }

        public final void setProvisionedThroughputOverride(ProvisionedThroughputOverride.BuilderImpl provisionedThroughputOverride) {
            this.provisionedThroughputOverride = provisionedThroughputOverride != null ? provisionedThroughputOverride.build()
                    : null;
        }

        @Override
        public final Builder provisionedThroughputOverride(ProvisionedThroughputOverride provisionedThroughputOverride) {
            this.provisionedThroughputOverride = provisionedThroughputOverride;
            return this;
        }

        public final OnDemandThroughputOverride.Builder getOnDemandThroughputOverride() {
            return onDemandThroughputOverride != null ? onDemandThroughputOverride.toBuilder() : null;
        }

        public final void setOnDemandThroughputOverride(OnDemandThroughputOverride.BuilderImpl onDemandThroughputOverride) {
            this.onDemandThroughputOverride = onDemandThroughputOverride != null ? onDemandThroughputOverride.build() : null;
        }

        @Override
        public final Builder onDemandThroughputOverride(OnDemandThroughputOverride onDemandThroughputOverride) {
            this.onDemandThroughputOverride = onDemandThroughputOverride;
            return this;
        }

        public final TableWarmThroughputDescription.Builder getWarmThroughput() {
            return warmThroughput != null ? warmThroughput.toBuilder() : null;
        }

        public final void setWarmThroughput(TableWarmThroughputDescription.BuilderImpl warmThroughput) {
            this.warmThroughput = warmThroughput != null ? warmThroughput.build() : null;
        }

        @Override
        public final Builder warmThroughput(TableWarmThroughputDescription warmThroughput) {
            this.warmThroughput = warmThroughput;
            return this;
        }

        public final List<ReplicaGlobalSecondaryIndexDescription.Builder> getGlobalSecondaryIndexes() {
            List<ReplicaGlobalSecondaryIndexDescription.Builder> result = ReplicaGlobalSecondaryIndexDescriptionListCopier
                    .copyToBuilder(this.globalSecondaryIndexes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setGlobalSecondaryIndexes(
                Collection<ReplicaGlobalSecondaryIndexDescription.BuilderImpl> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = ReplicaGlobalSecondaryIndexDescriptionListCopier
                    .copyFromBuilder(globalSecondaryIndexes);
        }

        @Override
        public final Builder globalSecondaryIndexes(Collection<ReplicaGlobalSecondaryIndexDescription> globalSecondaryIndexes) {
            this.globalSecondaryIndexes = ReplicaGlobalSecondaryIndexDescriptionListCopier.copy(globalSecondaryIndexes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder globalSecondaryIndexes(ReplicaGlobalSecondaryIndexDescription... globalSecondaryIndexes) {
            globalSecondaryIndexes(Arrays.asList(globalSecondaryIndexes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder globalSecondaryIndexes(
                Consumer<ReplicaGlobalSecondaryIndexDescription.Builder>... globalSecondaryIndexes) {
            globalSecondaryIndexes(Stream.of(globalSecondaryIndexes)
                    .map(c -> ReplicaGlobalSecondaryIndexDescription.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Instant getReplicaInaccessibleDateTime() {
            return replicaInaccessibleDateTime;
        }

        public final void setReplicaInaccessibleDateTime(Instant replicaInaccessibleDateTime) {
            this.replicaInaccessibleDateTime = replicaInaccessibleDateTime;
        }

        @Override
        public final Builder replicaInaccessibleDateTime(Instant replicaInaccessibleDateTime) {
            this.replicaInaccessibleDateTime = replicaInaccessibleDateTime;
            return this;
        }

        public final TableClassSummary.Builder getReplicaTableClassSummary() {
            return replicaTableClassSummary != null ? replicaTableClassSummary.toBuilder() : null;
        }

        public final void setReplicaTableClassSummary(TableClassSummary.BuilderImpl replicaTableClassSummary) {
            this.replicaTableClassSummary = replicaTableClassSummary != null ? replicaTableClassSummary.build() : null;
        }

        @Override
        public final Builder replicaTableClassSummary(TableClassSummary replicaTableClassSummary) {
            this.replicaTableClassSummary = replicaTableClassSummary;
            return this;
        }

        @Override
        public ReplicaDescription build() {
            return new ReplicaDescription(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
