/*
 * Copyright 2014-2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.utils.CollectionUtils;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An ordered list of errors for each item in the request which caused the transaction to get cancelled. The values of
 * the list are ordered according to the ordering of the <code>TransactWriteItems</code> request parameter. If no error
 * occurred for the associated item an error with a Null code and Null message will be present.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CancellationReason implements SdkPojo, Serializable,
        ToCopyableBuilder<CancellationReason.Builder, CancellationReason> {
    private static final SdkField<Map<String, AttributeValue>> ITEM_FIELD = SdkField
            .<Map<String, AttributeValue>> builder(MarshallingType.MAP)
            .getter(getter(CancellationReason::item))
            .setter(setter(Builder::item))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Item").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<AttributeValue> builder(MarshallingType.SDK_POJO)
                                            .constructor(AttributeValue::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CancellationReason::code)).setter(setter(Builder::code))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Code").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CancellationReason::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ITEM_FIELD, CODE_FIELD,
            MESSAGE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Map<String, AttributeValue> item;

    private final String code;

    private final String message;

    private CancellationReason(BuilderImpl builder) {
        this.item = builder.item;
        this.code = builder.code;
        this.message = builder.message;
    }

    /**
     * <p>
     * Item in the request which caused the transaction to get cancelled.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Item in the request which caused the transaction to get cancelled.
     */
    public Map<String, AttributeValue> item() {
        return item;
    }

    /**
     * <p>
     * Status code for the result of the cancelled transaction.
     * </p>
     * 
     * @return Status code for the result of the cancelled transaction.
     */
    public String code() {
        return code;
    }

    /**
     * <p>
     * Cancellation reason message description.
     * </p>
     * 
     * @return Cancellation reason message description.
     */
    public String message() {
        return message;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(item());
        hashCode = 31 * hashCode + Objects.hashCode(code());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CancellationReason)) {
            return false;
        }
        CancellationReason other = (CancellationReason) obj;
        return Objects.equals(item(), other.item()) && Objects.equals(code(), other.code())
                && Objects.equals(message(), other.message());
    }

    @Override
    public String toString() {
        return ToString.builder("CancellationReason").add("Item", item()).add("Code", code()).add("Message", message()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Item":
            return Optional.ofNullable(clazz.cast(item()));
        case "Code":
            return Optional.ofNullable(clazz.cast(code()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CancellationReason, T> g) {
        return obj -> g.apply((CancellationReason) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CancellationReason> {
        /**
         * <p>
         * Item in the request which caused the transaction to get cancelled.
         * </p>
         * 
         * @param item
         *        Item in the request which caused the transaction to get cancelled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder item(Map<String, AttributeValue> item);

        /**
         * <p>
         * Status code for the result of the cancelled transaction.
         * </p>
         * 
         * @param code
         *        Status code for the result of the cancelled transaction.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder code(String code);

        /**
         * <p>
         * Cancellation reason message description.
         * </p>
         * 
         * @param message
         *        Cancellation reason message description.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);
    }

    static final class BuilderImpl implements Builder {
        private Map<String, AttributeValue> item = DefaultSdkAutoConstructMap.getInstance();

        private String code;

        private String message;

        private BuilderImpl() {
        }

        private BuilderImpl(CancellationReason model) {
            item(model.item);
            code(model.code);
            message(model.message);
        }

        public final Map<String, AttributeValue.Builder> getItem() {
            return item != null ? CollectionUtils.mapValues(item, AttributeValue::toBuilder) : null;
        }

        @Override
        public final Builder item(Map<String, AttributeValue> item) {
            this.item = AttributeMapCopier.copy(item);
            return this;
        }

        public final void setItem(Map<String, AttributeValue.BuilderImpl> item) {
            this.item = AttributeMapCopier.copyFromBuilder(item);
        }

        public final String getCode() {
            return code;
        }

        @Override
        public final Builder code(String code) {
            this.code = code;
            return this;
        }

        public final void setCode(String code) {
            this.code = code;
        }

        public final String getMessage() {
            return message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public CancellationReason build() {
            return new CancellationReason(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
