/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb;

import java.net.URI;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.config.AwsClientOption;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRefreshCache;
import software.amazon.awssdk.core.endpointdiscovery.EndpointDiscoveryRequest;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.identity.spi.AwsCredentialsIdentity;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.dynamodb.internal.DynamoDbServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.dynamodb.model.BackupInUseException;
import software.amazon.awssdk.services.dynamodb.model.BackupNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.BatchExecuteStatementRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchExecuteStatementResponse;
import software.amazon.awssdk.services.dynamodb.model.BatchGetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchGetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.BatchWriteItemResponse;
import software.amazon.awssdk.services.dynamodb.model.ConditionalCheckFailedException;
import software.amazon.awssdk.services.dynamodb.model.ContinuousBackupsUnavailableException;
import software.amazon.awssdk.services.dynamodb.model.CreateBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.CreateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.CreateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.CreateTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteItemResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteResourcePolicyResponse;
import software.amazon.awssdk.services.dynamodb.model.DeleteTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DeleteTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContinuousBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeEndpointsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeEndpointsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeExportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeGlobalTableSettingsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeImportResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeLimitsRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeLimitsResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableReplicaAutoScalingResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTableResponse;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeTimeToLiveResponse;
import software.amazon.awssdk.services.dynamodb.model.DisableKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.DisableKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.DuplicateItemException;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbException;
import software.amazon.awssdk.services.dynamodb.model.EnableKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.EnableKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.ExecuteStatementRequest;
import software.amazon.awssdk.services.dynamodb.model.ExecuteStatementResponse;
import software.amazon.awssdk.services.dynamodb.model.ExecuteTransactionRequest;
import software.amazon.awssdk.services.dynamodb.model.ExecuteTransactionResponse;
import software.amazon.awssdk.services.dynamodb.model.ExportConflictException;
import software.amazon.awssdk.services.dynamodb.model.ExportNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ExportTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.ExportTableToPointInTimeResponse;
import software.amazon.awssdk.services.dynamodb.model.GetItemRequest;
import software.amazon.awssdk.services.dynamodb.model.GetItemResponse;
import software.amazon.awssdk.services.dynamodb.model.GetResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.GetResourcePolicyResponse;
import software.amazon.awssdk.services.dynamodb.model.GlobalTableAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.GlobalTableNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.IdempotentParameterMismatchException;
import software.amazon.awssdk.services.dynamodb.model.ImportConflictException;
import software.amazon.awssdk.services.dynamodb.model.ImportNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ImportTableRequest;
import software.amazon.awssdk.services.dynamodb.model.ImportTableResponse;
import software.amazon.awssdk.services.dynamodb.model.IndexNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.InternalServerErrorException;
import software.amazon.awssdk.services.dynamodb.model.InvalidExportTimeException;
import software.amazon.awssdk.services.dynamodb.model.InvalidRestoreTimeException;
import software.amazon.awssdk.services.dynamodb.model.ItemCollectionSizeLimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.LimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ListBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListExportsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListExportsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListGlobalTablesRequest;
import software.amazon.awssdk.services.dynamodb.model.ListGlobalTablesResponse;
import software.amazon.awssdk.services.dynamodb.model.ListImportsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListImportsResponse;
import software.amazon.awssdk.services.dynamodb.model.ListTablesRequest;
import software.amazon.awssdk.services.dynamodb.model.ListTablesResponse;
import software.amazon.awssdk.services.dynamodb.model.ListTagsOfResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.ListTagsOfResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.PointInTimeRecoveryUnavailableException;
import software.amazon.awssdk.services.dynamodb.model.PolicyNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ProvisionedThroughputExceededException;
import software.amazon.awssdk.services.dynamodb.model.PutItemRequest;
import software.amazon.awssdk.services.dynamodb.model.PutItemResponse;
import software.amazon.awssdk.services.dynamodb.model.PutResourcePolicyRequest;
import software.amazon.awssdk.services.dynamodb.model.PutResourcePolicyResponse;
import software.amazon.awssdk.services.dynamodb.model.QueryRequest;
import software.amazon.awssdk.services.dynamodb.model.QueryResponse;
import software.amazon.awssdk.services.dynamodb.model.ReplicaAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.ReplicaNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.ReplicatedWriteConflictException;
import software.amazon.awssdk.services.dynamodb.model.RequestLimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ResourceInUseException;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableFromBackupRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableFromBackupResponse;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableToPointInTimeRequest;
import software.amazon.awssdk.services.dynamodb.model.RestoreTableToPointInTimeResponse;
import software.amazon.awssdk.services.dynamodb.model.ScanRequest;
import software.amazon.awssdk.services.dynamodb.model.ScanResponse;
import software.amazon.awssdk.services.dynamodb.model.TableAlreadyExistsException;
import software.amazon.awssdk.services.dynamodb.model.TableInUseException;
import software.amazon.awssdk.services.dynamodb.model.TableNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.TagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.TagResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactGetItemsRequest;
import software.amazon.awssdk.services.dynamodb.model.TransactGetItemsResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactWriteItemsRequest;
import software.amazon.awssdk.services.dynamodb.model.TransactWriteItemsResponse;
import software.amazon.awssdk.services.dynamodb.model.TransactionCanceledException;
import software.amazon.awssdk.services.dynamodb.model.TransactionConflictException;
import software.amazon.awssdk.services.dynamodb.model.TransactionInProgressException;
import software.amazon.awssdk.services.dynamodb.model.UntagResourceRequest;
import software.amazon.awssdk.services.dynamodb.model.UntagResourceResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateContinuousBackupsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContinuousBackupsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateContributorInsightsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateContributorInsightsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableSettingsRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateGlobalTableSettingsResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateItemResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateKinesisStreamingDestinationRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateKinesisStreamingDestinationResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableReplicaAutoScalingRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableReplicaAutoScalingResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTableResponse;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveRequest;
import software.amazon.awssdk.services.dynamodb.model.UpdateTimeToLiveResponse;
import software.amazon.awssdk.services.dynamodb.transform.BatchExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.BatchGetItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.BatchWriteItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.CreateTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DeleteTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeContinuousBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeEndpointsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeExportRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeGlobalTableSettingsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeImportRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeKinesisStreamingDestinationRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeLimitsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTableReplicaAutoScalingRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DescribeTimeToLiveRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.DisableKinesisStreamingDestinationRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.EnableKinesisStreamingDestinationRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ExecuteStatementRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ExecuteTransactionRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ExportTableToPointInTimeRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.GetItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.GetResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ImportTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListExportsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListGlobalTablesRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListImportsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListTablesRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ListTagsOfResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.PutItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.PutResourcePolicyRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.QueryRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.RestoreTableFromBackupRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.RestoreTableToPointInTimeRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.ScanRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TransactGetItemsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.TransactWriteItemsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateContinuousBackupsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateContributorInsightsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateGlobalTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateGlobalTableSettingsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateItemRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateKinesisStreamingDestinationRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTableReplicaAutoScalingRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTableRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.transform.UpdateTimeToLiveRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.waiters.DynamoDbWaiter;
import software.amazon.awssdk.utils.CompletableFutureUtils;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DynamoDbClient}.
 *
 * @see DynamoDbClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDynamoDbClient implements DynamoDbClient {
    private static final Logger log = Logger.loggerFor(DefaultDynamoDbClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    private EndpointDiscoveryRefreshCache endpointDiscoveryCache;

    protected DefaultDynamoDbClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
        if (clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED)) {
            this.endpointDiscoveryCache = EndpointDiscoveryRefreshCache.create(DynamoDbEndpointDiscoveryCacheLoader.create(this));
        }
    }

    /**
     * <p>
     * This operation allows you to perform batch reads or writes on data stored in DynamoDB, using PartiQL. Each read
     * statement in a <code>BatchExecuteStatement</code> must specify an equality condition on all key attributes. This
     * enforces that each <code>SELECT</code> statement in a batch returns at most a single item. For more information,
     * see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ql-reference.multiplestatements.batching.html"
     * >Running batch operations with PartiQL for DynamoDB </a>.
     * </p>
     * <note>
     * <p>
     * The entire batch must consist of either read statements or write statements, you cannot mix both in one batch.
     * </p>
     * </note> <important>
     * <p>
     * A HTTP 200 response does not mean that all statements in the BatchExecuteStatement succeeded. Error details for
     * individual statements can be found under the <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_BatchStatementResponse.html#DDB-Type-BatchStatementResponse-Error"
     * >Error</a> field of the <code>BatchStatementResponse</code> for each statement.
     * </p>
     * </important>
     *
     * @param batchExecuteStatementRequest
     * @return Result of the BatchExecuteStatement operation returned by the service.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.BatchExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchExecuteStatement"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public BatchExecuteStatementResponse batchExecuteStatement(BatchExecuteStatementRequest batchExecuteStatementRequest)
            throws RequestLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, BatchExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchExecuteStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchExecuteStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<BatchExecuteStatementRequest, BatchExecuteStatementResponse>()
                    .withOperationName("BatchExecuteStatement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(batchExecuteStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>BatchGetItem</code> operation returns the attributes of one or more items from one or more tables. You
     * identify requested items by primary key.
     * </p>
     * <p>
     * A single operation can retrieve up to 16 MB of data, which can contain as many as 100 items.
     * <code>BatchGetItem</code> returns a partial result if the response size limit is exceeded, the table's
     * provisioned throughput is exceeded, more than 1MB per partition is requested, or an internal processing failure
     * occurs. If a partial result is returned, the operation returns a value for <code>UnprocessedKeys</code>. You can
     * use this value to retry the operation starting with the next item to get.
     * </p>
     * <important>
     * <p>
     * If you request more than 100 items, <code>BatchGetItem</code> returns a <code>ValidationException</code> with the
     * message "Too many items requested for the BatchGetItem call."
     * </p>
     * </important>
     * <p>
     * For example, if you ask to retrieve 100 items, but each individual item is 300 KB in size, the system returns 52
     * items (so as not to exceed the 16 MB limit). It also returns an appropriate <code>UnprocessedKeys</code> value so
     * you can get the next page of results. If desired, your application can include its own logic to assemble the
     * pages of results into one dataset.
     * </p>
     * <p>
     * If <i>none</i> of the items can be processed due to insufficient provisioned throughput on all of the tables in
     * the request, then <code>BatchGetItem</code> returns a <code>ProvisionedThroughputExceededException</code>. If
     * <i>at least one</i> of the items is successfully processed, then <code>BatchGetItem</code> completes
     * successfully, while returning the keys of the unread items in <code>UnprocessedKeys</code>.
     * </p>
     * <important>
     * <p>
     * If DynamoDB returns any unprocessed items, you should retry the batch operation on those items. However, <i>we
     * strongly recommend that you use an exponential backoff algorithm</i>. If you retry the batch operation
     * immediately, the underlying read or write requests can still fail due to throttling on the individual tables. If
     * you delay the batch operation using exponential backoff, the individual requests in the batch are much more
     * likely to succeed.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#BatchOperations">Batch
     * Operations and Error Handling</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * By default, <code>BatchGetItem</code> performs eventually consistent reads on every table in the request. If you
     * want strongly consistent reads instead, you can set <code>ConsistentRead</code> to <code>true</code> for any or
     * all tables.
     * </p>
     * <p>
     * In order to minimize response latency, <code>BatchGetItem</code> may retrieve items in parallel.
     * </p>
     * <p>
     * When designing your application, keep in mind that DynamoDB does not return items in any particular order. To
     * help parse the response by item, include the primary key values for the items in your request in the
     * <code>ProjectionExpression</code> parameter.
     * </p>
     * <p>
     * If a requested item does not exist, it is not returned in the result. Requests for nonexistent items consume the
     * minimum read capacity units according to the type of read. For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithTables.html#CapacityUnitCalculations"
     * >Working with Tables</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param batchGetItemRequest
     *        Represents the input of a <code>BatchGetItem</code> operation.
     * @return Result of the BatchGetItem operation returned by the service.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.BatchGetItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchGetItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchGetItemResponse batchGetItem(BatchGetItemRequest batchGetItemRequest)
            throws ProvisionedThroughputExceededException, ResourceNotFoundException, RequestLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchGetItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchGetItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = batchGetItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(batchGetItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchGetItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchGetItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchGetItem");

            return clientHandler.execute(new ClientExecutionParams<BatchGetItemRequest, BatchGetItemResponse>()
                    .withOperationName("BatchGetItem").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(batchGetItemRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchGetItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>BatchWriteItem</code> operation puts or deletes multiple items in one or more tables. A single call to
     * <code>BatchWriteItem</code> can transmit up to 16MB of data over the network, consisting of up to 25 item put or
     * delete operations. While individual items can be up to 400 KB once stored, it's important to note that an item's
     * representation might be greater than 400KB while being sent in DynamoDB's JSON format for the API call. For more
     * details on this distinction, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html"
     * >Naming Rules and Data Types</a>.
     * </p>
     * <note>
     * <p>
     * <code>BatchWriteItem</code> cannot update items. If you perform a <code>BatchWriteItem</code> operation on an
     * existing item, that item's values will be overwritten by the operation and it will appear like it was updated. To
     * update items, we recommend you use the <code>UpdateItem</code> action.
     * </p>
     * </note>
     * <p>
     * The individual <code>PutItem</code> and <code>DeleteItem</code> operations specified in
     * <code>BatchWriteItem</code> are atomic; however <code>BatchWriteItem</code> as a whole is not. If any requested
     * operations fail because the table's provisioned throughput is exceeded or an internal processing failure occurs,
     * the failed operations are returned in the <code>UnprocessedItems</code> response parameter. You can investigate
     * and optionally resend the requests. Typically, you would call <code>BatchWriteItem</code> in a loop. Each
     * iteration would check for unprocessed items and submit a new <code>BatchWriteItem</code> request with those
     * unprocessed items until all items have been processed.
     * </p>
     * <p>
     * For tables and indexes with provisioned capacity, if none of the items can be processed due to insufficient
     * provisioned throughput on all of the tables in the request, then <code>BatchWriteItem</code> returns a
     * <code>ProvisionedThroughputExceededException</code>. For all tables and indexes, if none of the items can be
     * processed due to other throttling scenarios (such as exceeding partition level limits), then
     * <code>BatchWriteItem</code> returns a <code>ThrottlingException</code>.
     * </p>
     * <important>
     * <p>
     * If DynamoDB returns any unprocessed items, you should retry the batch operation on those items. However, <i>we
     * strongly recommend that you use an exponential backoff algorithm</i>. If you retry the batch operation
     * immediately, the underlying read or write requests can still fail due to throttling on the individual tables. If
     * you delay the batch operation using exponential backoff, the individual requests in the batch are much more
     * likely to succeed.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/ErrorHandling.html#Programming.Errors.BatchOperations"
     * >Batch Operations and Error Handling</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * </important>
     * <p>
     * With <code>BatchWriteItem</code>, you can efficiently write or delete large amounts of data, such as from Amazon
     * EMR, or copy data from another database into DynamoDB. In order to improve performance with these large-scale
     * operations, <code>BatchWriteItem</code> does not behave in the same way as individual <code>PutItem</code> and
     * <code>DeleteItem</code> calls would. For example, you cannot specify conditions on individual put and delete
     * requests, and <code>BatchWriteItem</code> does not return deleted items in the response.
     * </p>
     * <p>
     * If you use a programming language that supports concurrency, you can use threads to write items in parallel. Your
     * application must include the necessary logic to manage the threads. With languages that don't support threading,
     * you must update or delete the specified items one at a time. In both situations, <code>BatchWriteItem</code>
     * performs the specified put and delete operations in parallel, giving you the power of the thread pool approach
     * without having to introduce complexity into your application.
     * </p>
     * <p>
     * Parallel processing reduces latency, but each specified put and delete request consumes the same number of write
     * capacity units whether it is processed in parallel or not. Delete operations on nonexistent items consume one
     * write capacity unit.
     * </p>
     * <p>
     * If one or more of the following is true, DynamoDB rejects the entire batch write operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * One or more tables specified in the <code>BatchWriteItem</code> request does not exist.
     * </p>
     * </li>
     * <li>
     * <p>
     * Primary key attributes specified on an item in the request do not match those in the corresponding table's
     * primary key schema.
     * </p>
     * </li>
     * <li>
     * <p>
     * You try to perform multiple operations on the same item in the same <code>BatchWriteItem</code> request. For
     * example, you cannot put and delete the same item in the same <code>BatchWriteItem</code> request.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your request contains at least two items with identical hash and range keys (which essentially is two put
     * operations).
     * </p>
     * </li>
     * <li>
     * <p>
     * There are more than 25 requests in the batch.
     * </p>
     * </li>
     * <li>
     * <p>
     * Any individual item in a batch exceeds 400 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The total request size exceeds 16 MB.
     * </p>
     * </li>
     * <li>
     * <p>
     * Any individual items with keys exceeding the key length limits. For a partition key, the limit is 2048 bytes and
     * for a sort key, the limit is 1024 bytes.
     * </p>
     * </li>
     * </ul>
     *
     * @param batchWriteItemRequest
     *        Represents the input of a <code>BatchWriteItem</code> operation.
     * @return Result of the BatchWriteItem operation returned by the service.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ItemCollectionSizeLimitExceededException
     *         An item collection is too large. This exception is only returned for tables that have one or more local
     *         secondary indexes.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.BatchWriteItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/BatchWriteItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public BatchWriteItemResponse batchWriteItem(BatchWriteItemRequest batchWriteItemRequest)
            throws ProvisionedThroughputExceededException, ResourceNotFoundException, ItemCollectionSizeLimitExceededException,
            RequestLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<BatchWriteItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                BatchWriteItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = batchWriteItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(batchWriteItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(batchWriteItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, batchWriteItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "BatchWriteItem");

            return clientHandler.execute(new ClientExecutionParams<BatchWriteItemRequest, BatchWriteItemResponse>()
                    .withOperationName("BatchWriteItem").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(batchWriteItemRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new BatchWriteItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a backup for an existing table.
     * </p>
     * <p>
     * Each time you create an on-demand backup, the entire table data is backed up. There is no limit to the number of
     * on-demand backups that can be taken.
     * </p>
     * <p>
     * When you create an on-demand backup, a time marker of the request is cataloged, and the backup is created
     * asynchronously, by applying all changes until the time of the request to the last full table snapshot. Backup
     * requests are processed instantaneously and become available for restore within minutes.
     * </p>
     * <p>
     * You can call <code>CreateBackup</code> at a maximum rate of 50 times per second.
     * </p>
     * <p>
     * All backups in DynamoDB work without consuming any provisioned throughput on the table.
     * </p>
     * <p>
     * If you submit a backup request on 2018-12-14 at 14:25:00, the backup is guaranteed to contain all data committed
     * to the table up to 14:24:00, and data committed after 14:26:00 will not be. The backup might contain data
     * modifications made between 14:24:00 and 14:26:00. On-demand backup does not support causal consistency.
     * </p>
     * <p>
     * Along with data, the following are also included on the backups:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Global secondary indexes (GSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Local secondary indexes (LSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Streams
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned read and write capacity
     * </p>
     * </li>
     * </ul>
     *
     * @param createBackupRequest
     * @return Result of the CreateBackup operation returned by the service.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws TableInUseException
     *         A target table with the specified name is either being created or deleted.
     * @throws ContinuousBackupsUnavailableException
     *         Backups have not yet been enabled for this table.
     * @throws BackupInUseException
     *         There is another ongoing conflicting backup control plane operation on the table. The backup is either
     *         being created, deleted or restored to a table.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.CreateBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupResponse createBackup(CreateBackupRequest createBackupRequest) throws TableNotFoundException,
            TableInUseException, ContinuousBackupsUnavailableException, BackupInUseException, LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = createBackupRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(createBackupRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackup");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                    .withOperationName("CreateBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(createBackupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a global table from an existing table. A global table creates a replication relationship between two or
     * more DynamoDB tables with the same table name in the provided Regions.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important>
     * <p>
     * If you want to add a new replica table to a global table, each of the following conditions must be true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The table must have the same primary key as all of the other replicas.
     * </p>
     * </li>
     * <li>
     * <p>
     * The table must have the same name as all of the other replicas.
     * </p>
     * </li>
     * <li>
     * <p>
     * The table must have DynamoDB Streams enabled, with the stream containing both the new and the old images of the
     * item.
     * </p>
     * </li>
     * <li>
     * <p>
     * None of the replica tables in the global table can contain any data.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If global secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The global secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * </ul>
     * <p>
     * If local secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The local secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The local secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * </ul>
     * <important>
     * <p>
     * Write capacity settings should be set consistently across your replica tables and secondary indexes. DynamoDB
     * strongly recommends enabling auto scaling to manage the write capacity settings for all of your global tables
     * replicas and indexes.
     * </p>
     * <p>
     * If you prefer to manage write capacity settings manually, you should provision equal replicated write capacity
     * units to your replica tables. You should also provision equal replicated write capacity units to matching
     * secondary indexes across your global table.
     * </p>
     * </important>
     *
     * @param createGlobalTableRequest
     * @return Result of the CreateGlobalTable operation returned by the service.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws GlobalTableAlreadyExistsException
     *         The specified global table already exists.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.CreateGlobalTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateGlobalTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateGlobalTableResponse createGlobalTable(CreateGlobalTableRequest createGlobalTableRequest)
            throws LimitExceededException, InternalServerErrorException, GlobalTableAlreadyExistsException,
            TableNotFoundException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateGlobalTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = createGlobalTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(createGlobalTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createGlobalTableRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateGlobalTable");

            return clientHandler.execute(new ClientExecutionParams<CreateGlobalTableRequest, CreateGlobalTableResponse>()
                    .withOperationName("CreateGlobalTable").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(createGlobalTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateGlobalTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>CreateTable</code> operation adds a new table to your account. In an Amazon Web Services account, table
     * names must be unique within each Region. That is, you can have two tables with same name if you create the tables
     * in different Regions.
     * </p>
     * <p>
     * <code>CreateTable</code> is an asynchronous operation. Upon receiving a <code>CreateTable</code> request,
     * DynamoDB immediately returns a response with a <code>TableStatus</code> of <code>CREATING</code>. After the table
     * is created, DynamoDB sets the <code>TableStatus</code> to <code>ACTIVE</code>. You can perform read and write
     * operations only on an <code>ACTIVE</code> table.
     * </p>
     * <p>
     * You can optionally define secondary indexes on the new table, as part of the <code>CreateTable</code> operation.
     * If you want to create multiple tables with secondary indexes on them, you must create the tables sequentially.
     * Only one table with secondary indexes can be in the <code>CREATING</code> state at any given time.
     * </p>
     * <p>
     * You can use the <code>DescribeTable</code> action to check the table status.
     * </p>
     *
     * @param createTableRequest
     *        Represents the input of a <code>CreateTable</code> operation.
     * @return Result of the CreateTable operation returned by the service.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.CreateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/CreateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateTableResponse createTable(CreateTableRequest createTableRequest) throws ResourceInUseException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = createTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(createTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateTable");

            return clientHandler.execute(new ClientExecutionParams<CreateTableRequest, CreateTableResponse>()
                    .withOperationName("CreateTable").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(createTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing backup of a table.
     * </p>
     * <p>
     * You can call <code>DeleteBackup</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param deleteBackupRequest
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws BackupNotFoundException
     *         Backup not found for the given BackupARN.
     * @throws BackupInUseException
     *         There is another ongoing conflicting backup control plane operation on the table. The backup is either
     *         being created, deleted or restored to a table.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws BackupNotFoundException,
            BackupInUseException, LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = deleteBackupRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(deleteBackupRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                    .withOperationName("DeleteBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteBackupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a single item in a table by primary key. You can perform a conditional delete operation that deletes the
     * item if it exists, or if it has an expected attribute value.
     * </p>
     * <p>
     * In addition to deleting an item, you can also return the item's attribute values in the same operation, using the
     * <code>ReturnValues</code> parameter.
     * </p>
     * <p>
     * Unless you specify conditions, the <code>DeleteItem</code> is an idempotent operation; running it multiple times
     * on the same item or attribute does <i>not</i> result in an error response.
     * </p>
     * <p>
     * Conditional deletes are useful for deleting items only if specific conditions are met. If those conditions are
     * met, DynamoDB performs the delete. Otherwise, the item is not deleted.
     * </p>
     *
     * @param deleteItemRequest
     *        Represents the input of a <code>DeleteItem</code> operation.
     * @return Result of the DeleteItem operation returned by the service.
     * @throws ConditionalCheckFailedException
     *         A condition specified in the operation failed to be evaluated.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ItemCollectionSizeLimitExceededException
     *         An item collection is too large. This exception is only returned for tables that have one or more local
     *         secondary indexes.
     * @throws TransactionConflictException
     *         Operation was rejected because there is an ongoing transaction for the item.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ReplicatedWriteConflictException
     *         The request was rejected because one or more items in the request are being modified by a request in
     *         another Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DeleteItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteItemResponse deleteItem(DeleteItemRequest deleteItemRequest) throws ConditionalCheckFailedException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ItemCollectionSizeLimitExceededException,
            TransactionConflictException, RequestLimitExceededException, InternalServerErrorException,
            ReplicatedWriteConflictException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = deleteItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(deleteItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteItem");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteItemRequest, DeleteItemResponse>().withOperationName("DeleteItem")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteItemRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the resource-based policy attached to the resource, which can be a table or stream.
     * </p>
     * <p>
     * <code>DeleteResourcePolicy</code> is an idempotent operation; running it multiple times on the same resource
     * <i>doesn't</i> result in an error response, unless you specify an <code>ExpectedRevisionId</code>, which will
     * then return a <code>PolicyNotFoundException</code>.
     * </p>
     * <important>
     * <p>
     * To make sure that you don't inadvertently lock yourself out of your own resources, the root principal in your
     * Amazon Web Services account can perform <code>DeleteResourcePolicy</code> requests, even if your resource-based
     * policy explicitly denies the root principal's access.
     * </p>
     * </important> <note>
     * <p>
     * <code>DeleteResourcePolicy</code> is an asynchronous operation. If you issue a <code>GetResourcePolicy</code>
     * request immediately after running the <code>DeleteResourcePolicy</code> request, DynamoDB might still return the
     * deleted policy. This is because the policy for your resource might not have been deleted yet. Wait for a few
     * seconds, and then try the <code>GetResourcePolicy</code> request again.
     * </p>
     * </note>
     *
     * @param deleteResourcePolicyRequest
     * @return Result of the DeleteResourcePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws PolicyNotFoundException
     *         The operation tried to access a nonexistent resource-based policy.</p>
     *         <p>
     *         If you specified an <code>ExpectedRevisionId</code>, it's possible that a policy is present for the
     *         resource but its revision ID didn't match the expected value.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DeleteResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteResourcePolicyResponse deleteResourcePolicy(DeleteResourcePolicyRequest deleteResourcePolicyRequest)
            throws ResourceNotFoundException, InternalServerErrorException, PolicyNotFoundException, ResourceInUseException,
            LimitExceededException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = deleteResourcePolicyRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(deleteResourcePolicyRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<DeleteResourcePolicyRequest, DeleteResourcePolicyResponse>()
                    .withOperationName("DeleteResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(deleteResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>DeleteTable</code> operation deletes a table and all of its items. After a <code>DeleteTable</code>
     * request, the specified table is in the <code>DELETING</code> state until DynamoDB completes the deletion. If the
     * table is in the <code>ACTIVE</code> state, you can delete it. If a table is in <code>CREATING</code> or
     * <code>UPDATING</code> states, then DynamoDB returns a <code>ResourceInUseException</code>. If the specified table
     * does not exist, DynamoDB returns a <code>ResourceNotFoundException</code>. If table is already in the
     * <code>DELETING</code> state, no error is returned.
     * </p>
     * <important>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version).
     * </p>
     * </important> <note>
     * <p>
     * DynamoDB might continue to accept data read and write operations, such as <code>GetItem</code> and
     * <code>PutItem</code>, on a table in the <code>DELETING</code> state until the table deletion is complete. For the
     * full list of table states, see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_TableDescription.html#DDB-Type-TableDescription-TableStatus"
     * >TableStatus</a>.
     * </p>
     * </note>
     * <p>
     * When you delete a table, any indexes on that table are also deleted.
     * </p>
     * <p>
     * If you have DynamoDB Streams enabled on the table, then the corresponding stream on that table goes into the
     * <code>DISABLED</code> state, and the stream is automatically deleted after 24 hours.
     * </p>
     * <p>
     * Use the <code>DescribeTable</code> action to check the status of the table.
     * </p>
     *
     * @param deleteTableRequest
     *        Represents the input of a <code>DeleteTable</code> operation.
     * @return Result of the DeleteTable operation returned by the service.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DeleteTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DeleteTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteTableResponse deleteTable(DeleteTableRequest deleteTableRequest) throws ResourceInUseException,
            ResourceNotFoundException, LimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = deleteTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(deleteTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteTable");

            return clientHandler.execute(new ClientExecutionParams<DeleteTableRequest, DeleteTableResponse>()
                    .withOperationName("DeleteTable").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing backup of a table.
     * </p>
     * <p>
     * You can call <code>DescribeBackup</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param describeBackupRequest
     * @return Result of the DescribeBackup operation returned by the service.
     * @throws BackupNotFoundException
     *         Backup not found for the given BackupARN.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupResponse describeBackup(DescribeBackupRequest describeBackupRequest) throws BackupNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeBackupRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeBackupRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackup");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupRequest, DescribeBackupResponse>()
                    .withOperationName("DescribeBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(describeBackupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Checks the status of continuous backups and point in time recovery on the specified table. Continuous backups are
     * <code>ENABLED</code> on all tables at table creation. If point in time recovery is enabled,
     * <code>PointInTimeRecoveryStatus</code> will be set to ENABLED.
     * </p>
     * <p>
     * After continuous backups and point in time recovery are enabled, you can restore to any point in time within
     * <code>EarliestRestorableDateTime</code> and <code>LatestRestorableDateTime</code>.
     * </p>
     * <p>
     * <code>LatestRestorableDateTime</code> is typically 5 minutes before the current time. You can restore your table
     * to any point in time in the last 35 days. You can set the recovery period to any value between 1 and 35 days.
     * </p>
     * <p>
     * You can call <code>DescribeContinuousBackups</code> at a maximum rate of 10 times per second.
     * </p>
     *
     * @param describeContinuousBackupsRequest
     * @return Result of the DescribeContinuousBackups operation returned by the service.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeContinuousBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeContinuousBackups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeContinuousBackupsResponse describeContinuousBackups(
            DescribeContinuousBackupsRequest describeContinuousBackupsRequest) throws TableNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContinuousBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeContinuousBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeContinuousBackupsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeContinuousBackupsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContinuousBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContinuousBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContinuousBackups");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeContinuousBackupsRequest, DescribeContinuousBackupsResponse>()
                            .withOperationName("DescribeContinuousBackups").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(describeContinuousBackupsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeContinuousBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about contributor insights for a given table or global secondary index.
     * </p>
     *
     * @param describeContributorInsightsRequest
     * @return Result of the DescribeContributorInsights operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeContributorInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeContributorInsightsResponse describeContributorInsights(
            DescribeContributorInsightsRequest describeContributorInsightsRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeContributorInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeContributorInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeContributorInsights");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeContributorInsightsRequest, DescribeContributorInsightsResponse>()
                            .withOperationName("DescribeContributorInsights").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeContributorInsightsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeContributorInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the regional endpoint information. For more information on policy permissions, please see <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/inter-network-traffic-privacy.html#inter-network-traffic-DescribeEndpoints"
     * >Internetwork traffic privacy</a>.
     * </p>
     *
     * @param describeEndpointsRequest
     * @return Result of the DescribeEndpoints operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeEndpoints
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeEndpoints" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeEndpointsResponse describeEndpoints(DescribeEndpointsRequest describeEndpointsRequest)
            throws AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeEndpointsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeEndpointsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeEndpointsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeEndpointsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeEndpoints");

            return clientHandler.execute(new ClientExecutionParams<DescribeEndpointsRequest, DescribeEndpointsResponse>()
                    .withOperationName("DescribeEndpoints").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeEndpointsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeEndpointsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes an existing table export.
     * </p>
     *
     * @param describeExportRequest
     * @return Result of the DescribeExport operation returned by the service.
     * @throws ExportNotFoundException
     *         The specified export was not found.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeExport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeExport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeExportResponse describeExport(DescribeExportRequest describeExportRequest) throws ExportNotFoundException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeExportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeExportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeExportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeExportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeExport");

            return clientHandler.execute(new ClientExecutionParams<DescribeExportRequest, DescribeExportResponse>()
                    .withOperationName("DescribeExport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeExportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeExportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the specified global table.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important>
     *
     * @param describeGlobalTableRequest
     * @return Result of the DescribeGlobalTable operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws GlobalTableNotFoundException
     *         The specified global table does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeGlobalTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeGlobalTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeGlobalTableResponse describeGlobalTable(DescribeGlobalTableRequest describeGlobalTableRequest)
            throws InternalServerErrorException, GlobalTableNotFoundException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGlobalTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeGlobalTableRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeGlobalTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGlobalTableRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGlobalTable");

            return clientHandler.execute(new ClientExecutionParams<DescribeGlobalTableRequest, DescribeGlobalTableResponse>()
                    .withOperationName("DescribeGlobalTable").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(describeGlobalTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeGlobalTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes Region-specific settings for a global table.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important>
     *
     * @param describeGlobalTableSettingsRequest
     * @return Result of the DescribeGlobalTableSettings operation returned by the service.
     * @throws GlobalTableNotFoundException
     *         The specified global table does not exist.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeGlobalTableSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeGlobalTableSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeGlobalTableSettingsResponse describeGlobalTableSettings(
            DescribeGlobalTableSettingsRequest describeGlobalTableSettingsRequest) throws GlobalTableNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeGlobalTableSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeGlobalTableSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeGlobalTableSettingsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeGlobalTableSettingsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeGlobalTableSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeGlobalTableSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeGlobalTableSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeGlobalTableSettingsRequest, DescribeGlobalTableSettingsResponse>()
                            .withOperationName("DescribeGlobalTableSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(describeGlobalTableSettingsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeGlobalTableSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Represents the properties of the import.
     * </p>
     *
     * @param describeImportRequest
     * @return Result of the DescribeImport operation returned by the service.
     * @throws ImportNotFoundException
     *         The specified import was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeImport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeImport" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeImportResponse describeImport(DescribeImportRequest describeImportRequest) throws ImportNotFoundException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeImportResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeImportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeImportRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeImportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeImport");

            return clientHandler.execute(new ClientExecutionParams<DescribeImportRequest, DescribeImportResponse>()
                    .withOperationName("DescribeImport").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeImportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeImportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the status of Kinesis streaming.
     * </p>
     *
     * @param describeKinesisStreamingDestinationRequest
     * @return Result of the DescribeKinesisStreamingDestination operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeKinesisStreamingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeKinesisStreamingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeKinesisStreamingDestinationResponse describeKinesisStreamingDestination(
            DescribeKinesisStreamingDestinationRequest describeKinesisStreamingDestinationRequest)
            throws ResourceNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeKinesisStreamingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeKinesisStreamingDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeKinesisStreamingDestinationRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null))
                    .build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeKinesisStreamingDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeKinesisStreamingDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeKinesisStreamingDestinationRequest, DescribeKinesisStreamingDestinationResponse>()
                            .withOperationName("DescribeKinesisStreamingDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(describeKinesisStreamingDestinationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeKinesisStreamingDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the current provisioned-capacity quotas for your Amazon Web Services account in a Region, both for the
     * Region as a whole and for any one DynamoDB table that you create there.
     * </p>
     * <p>
     * When you establish an Amazon Web Services account, the account has initial quotas on the maximum read capacity
     * units and write capacity units that you can provision across all of your DynamoDB tables in a given Region. Also,
     * there are per-table quotas that apply when you create a table there. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html">Service, Account, and Table
     * Quotas</a> page in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * Although you can increase these quotas by filing a case at <a
     * href="https://console.aws.amazon.com/support/home#/">Amazon Web Services Support Center</a>, obtaining the
     * increase is not instantaneous. The <code>DescribeLimits</code> action lets you write code to compare the capacity
     * you are currently using to those quotas imposed by your account so that you have enough time to apply for an
     * increase before you hit a quota.
     * </p>
     * <p>
     * For example, you could use one of the Amazon Web Services SDKs to do the following:
     * </p>
     * <ol>
     * <li>
     * <p>
     * Call <code>DescribeLimits</code> for a particular Region to obtain your current account quotas on provisioned
     * capacity there.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create a variable to hold the aggregate read capacity units provisioned for all your tables in that Region, and
     * one to hold the aggregate write capacity units. Zero them both.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call <code>ListTables</code> to obtain a list of all your DynamoDB tables.
     * </p>
     * </li>
     * <li>
     * <p>
     * For each table name listed by <code>ListTables</code>, do the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Call <code>DescribeTable</code> with the table name.
     * </p>
     * </li>
     * <li>
     * <p>
     * Use the data returned by <code>DescribeTable</code> to add the read capacity units and write capacity units
     * provisioned for the table itself to your variables.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the table has one or more global secondary indexes (GSIs), loop over these GSIs and add their provisioned
     * capacity values to your variables as well.
     * </p>
     * </li>
     * </ul>
     * </li>
     * <li>
     * <p>
     * Report the account quotas for that Region returned by <code>DescribeLimits</code>, along with the total current
     * provisioned capacity levels you have calculated.
     * </p>
     * </li>
     * </ol>
     * <p>
     * This will let you see whether you are getting close to your account-level quotas.
     * </p>
     * <p>
     * The per-table quotas apply only when you are creating a new table. They restrict the sum of the provisioned
     * capacity of the new table itself and all its global secondary indexes.
     * </p>
     * <p>
     * For existing tables and their GSIs, DynamoDB doesn't let you increase provisioned capacity extremely rapidly, but
     * the only quota that applies is that the aggregate provisioned capacity over all your tables and GSIs cannot
     * exceed either of the per-account quotas.
     * </p>
     * <note>
     * <p>
     * <code>DescribeLimits</code> should only be called periodically. You can expect throttling errors if you call it
     * more than once in a minute.
     * </p>
     * </note>
     * <p>
     * The <code>DescribeLimits</code> Request element has no content.
     * </p>
     *
     * @param describeLimitsRequest
     *        Represents the input of a <code>DescribeLimits</code> operation. Has no content.
     * @return Result of the DescribeLimits operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeLimits
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeLimits" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeLimitsResponse describeLimits(DescribeLimitsRequest describeLimitsRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeLimitsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeLimitsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeLimitsRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeLimitsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeLimitsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeLimitsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeLimits");

            return clientHandler.execute(new ClientExecutionParams<DescribeLimitsRequest, DescribeLimitsResponse>()
                    .withOperationName("DescribeLimits").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(describeLimitsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeLimitsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the table, including the current status of the table, when it was created, the primary
     * key schema, and any indexes on the table.
     * </p>
     * <important>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version).
     * </p>
     * </important> <note>
     * <p>
     * If you issue a <code>DescribeTable</code> request immediately after a <code>CreateTable</code> request, DynamoDB
     * might return a <code>ResourceNotFoundException</code>. This is because <code>DescribeTable</code> uses an
     * eventually consistent query, and the metadata for your table might not be available at that moment. Wait for a
     * few seconds, and then try the <code>DescribeTable</code> request again.
     * </p>
     * </note>
     *
     * @param describeTableRequest
     *        Represents the input of a <code>DescribeTable</code> operation.
     * @return Result of the DescribeTable operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeTableResponse describeTable(DescribeTableRequest describeTableRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTable");

            return clientHandler.execute(new ClientExecutionParams<DescribeTableRequest, DescribeTableResponse>()
                    .withOperationName("DescribeTable").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(describeTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes auto scaling settings across replicas of the global table at once.
     * </p>
     * <important>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version).
     * </p>
     * </important>
     *
     * @param describeTableReplicaAutoScalingRequest
     * @return Result of the DescribeTableReplicaAutoScaling operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeTableReplicaAutoScaling
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTableReplicaAutoScaling"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeTableReplicaAutoScalingResponse describeTableReplicaAutoScaling(
            DescribeTableReplicaAutoScalingRequest describeTableReplicaAutoScalingRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTableReplicaAutoScalingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTableReplicaAutoScalingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTableReplicaAutoScalingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeTableReplicaAutoScalingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTableReplicaAutoScaling");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeTableReplicaAutoScalingRequest, DescribeTableReplicaAutoScalingResponse>()
                            .withOperationName("DescribeTableReplicaAutoScaling").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeTableReplicaAutoScalingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeTableReplicaAutoScalingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gives a description of the Time to Live (TTL) status on the specified table.
     * </p>
     *
     * @param describeTimeToLiveRequest
     * @return Result of the DescribeTimeToLive operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DescribeTimeToLive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DescribeTimeToLive" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeTimeToLiveResponse describeTimeToLive(DescribeTimeToLiveRequest describeTimeToLiveRequest)
            throws ResourceNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeTimeToLiveResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeTimeToLiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = describeTimeToLiveRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(describeTimeToLiveRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeTimeToLiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeTimeToLiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeTimeToLive");

            return clientHandler.execute(new ClientExecutionParams<DescribeTimeToLiveRequest, DescribeTimeToLiveResponse>()
                    .withOperationName("DescribeTimeToLive").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(describeTimeToLiveRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeTimeToLiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops replication from the DynamoDB table to the Kinesis data stream. This is done without deleting either of the
     * resources.
     * </p>
     *
     * @param disableKinesisStreamingDestinationRequest
     * @return Result of the DisableKinesisStreamingDestination operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.DisableKinesisStreamingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/DisableKinesisStreamingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableKinesisStreamingDestinationResponse disableKinesisStreamingDestination(
            DisableKinesisStreamingDestinationRequest disableKinesisStreamingDestinationRequest)
            throws InternalServerErrorException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisableKinesisStreamingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisableKinesisStreamingDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = disableKinesisStreamingDestinationRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(disableKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null))
                    .build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disableKinesisStreamingDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disableKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableKinesisStreamingDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<DisableKinesisStreamingDestinationRequest, DisableKinesisStreamingDestinationResponse>()
                            .withOperationName("DisableKinesisStreamingDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(disableKinesisStreamingDestinationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisableKinesisStreamingDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts table data replication to the specified Kinesis data stream at a timestamp chosen during the enable
     * workflow. If this operation doesn't return results immediately, use DescribeKinesisStreamingDestination to check
     * if streaming to the Kinesis data stream is ACTIVE.
     * </p>
     *
     * @param enableKinesisStreamingDestinationRequest
     * @return Result of the EnableKinesisStreamingDestination operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.EnableKinesisStreamingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/EnableKinesisStreamingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public EnableKinesisStreamingDestinationResponse enableKinesisStreamingDestination(
            EnableKinesisStreamingDestinationRequest enableKinesisStreamingDestinationRequest)
            throws InternalServerErrorException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<EnableKinesisStreamingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, EnableKinesisStreamingDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = enableKinesisStreamingDestinationRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(enableKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(enableKinesisStreamingDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                enableKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableKinesisStreamingDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<EnableKinesisStreamingDestinationRequest, EnableKinesisStreamingDestinationResponse>()
                            .withOperationName("EnableKinesisStreamingDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(enableKinesisStreamingDestinationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new EnableKinesisStreamingDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation allows you to perform reads and singleton writes on data stored in DynamoDB, using PartiQL.
     * </p>
     * <p>
     * For PartiQL reads (<code>SELECT</code> statement), if the total number of processed items exceeds the maximum
     * dataset size limit of 1 MB, the read stops and results are returned to the user as a
     * <code>LastEvaluatedKey</code> value to continue the read in a subsequent operation. If the filter criteria in
     * <code>WHERE</code> clause does not match any data, the read will return an empty result set.
     * </p>
     * <p>
     * A single <code>SELECT</code> statement response can return up to the maximum number of items (if using the Limit
     * parameter) or a maximum of 1 MB of data (and then apply any filtering to the results using <code>WHERE</code>
     * clause). If <code>LastEvaluatedKey</code> is present in the response, you need to paginate the result set. If
     * <code>NextToken</code> is present, you need to paginate the result set and include <code>NextToken</code>.
     * </p>
     *
     * @param executeStatementRequest
     * @return Result of the ExecuteStatement operation returned by the service.
     * @throws ConditionalCheckFailedException
     *         A condition specified in the operation failed to be evaluated.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ItemCollectionSizeLimitExceededException
     *         An item collection is too large. This exception is only returned for tables that have one or more local
     *         secondary indexes.
     * @throws TransactionConflictException
     *         Operation was rejected because there is an ongoing transaction for the item.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws DuplicateItemException
     *         There was an attempt to insert an item with the same primary key as an item that already exists in the
     *         DynamoDB table.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ExecuteStatement
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ExecuteStatement" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ExecuteStatementResponse executeStatement(ExecuteStatementRequest executeStatementRequest)
            throws ConditionalCheckFailedException, ProvisionedThroughputExceededException, ResourceNotFoundException,
            ItemCollectionSizeLimitExceededException, TransactionConflictException, RequestLimitExceededException,
            InternalServerErrorException, DuplicateItemException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteStatementResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ExecuteStatementResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeStatementRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeStatementRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteStatement");

            return clientHandler.execute(new ClientExecutionParams<ExecuteStatementRequest, ExecuteStatementResponse>()
                    .withOperationName("ExecuteStatement").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(executeStatementRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteStatementRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation allows you to perform transactional reads or writes on data stored in DynamoDB, using PartiQL.
     * </p>
     * <note>
     * <p>
     * The entire transaction must consist of either read statements or write statements, you cannot mix both in one
     * transaction. The EXISTS function is an exception and can be used to check the condition of specific attributes of
     * the item in a similar manner to <code>ConditionCheck</code> in the <a href=
     * "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/transaction-apis.html#transaction-apis-txwriteitems"
     * >TransactWriteItems</a> API.
     * </p>
     * </note>
     *
     * @param executeTransactionRequest
     * @return Result of the ExecuteTransaction operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws TransactionCanceledException
     *         The entire transaction request was canceled.</p>
     *         <p>
     *         DynamoDB cancels a <code>TransactWriteItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A condition in one of the condition expressions is not met.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactWriteItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than one action in the <code>TransactWriteItems</code> operation targets the same item.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An item size becomes too large (larger than 400 KB), or a local secondary index (LSI) becomes too large,
     *         or a similar validation error occurs because of changes made by the transaction.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactWriteItems</code> operation that conflicts with a concurrent
     *         <code>TransactWriteItems</code> request. In this case the <code>TransactWriteItems</code> operation fails
     *         with a <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         DynamoDB cancels a <code>TransactGetItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactGetItems</code> operation that conflicts with a concurrent
     *         <code>PutItem</code>, <code>UpdateItem</code>, <code>DeleteItem</code> or <code>TransactWriteItems</code>
     *         request. In this case the <code>TransactGetItems</code> operation fails with a
     *         <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactGetItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If using Java, DynamoDB lists the cancellation reasons on the <code>CancellationReasons</code> property.
     *         This property is not set for other languages. Transaction cancellation reasons are ordered in the order
     *         of requested items, if an item has no error it will have <code>None</code> code and <code>Null</code>
     *         message.
     *         </p>
     *         </note>
     *         <p>
     *         Cancellation reason codes and possible error messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         No Errors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>None</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: <code>null</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Conditional Check Failed:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ConditionalCheckFailed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: The conditional request failed.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Item Collection Size Limit Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ItemCollectionSizeLimitExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Collection size exceeded.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Transaction Conflict:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>TransactionConflict</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Transaction is ongoing for the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned Throughput Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for the table was exceeded. Consider increasing your
     *         provisioning level with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This Message is received when provisioned throughput is exceeded is on a provisioned DynamoDB table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for one or more global secondary indexes of the table was
     *         exceeded. Consider increasing your provisioning level for the under-provisioned global secondary indexes
     *         with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when provisioned throughput is exceeded is on a provisioned GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Throttling Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ThrottlingError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity of your table or index. DynamoDB is automatically scaling your
     *         table or index so please try again shortly. If exceptions persist, check if you have a hot key:
     *         https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand table as DynamoDB is automatically
     *         scaling the table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity for one or more global secondary indexes. DynamoDB is
     *         automatically scaling your index so please try again shortly.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand GSI as DynamoDB is automatically
     *         scaling the GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Validation Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ValidationError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more parameter values were invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key beyond allowed size limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key to unsupported type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An operand in the update expression has an incorrect data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Item size to update has exceeded the maximum allowed size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Number overflow. Attempting to store a number with magnitude larger than supported range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Type mismatch for attribute to update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Nesting Levels have exceeded supported limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The document path provided in the update expression is invalid for update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The provided expression refers to an attribute that does not exist in the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @throws TransactionInProgressException
     *         The transaction with the given request token is already in progress.</p>
     *         <p>
     *         Recommended Settings
     *         </p>
     *         <note>
     *         <p>
     *         This is a general recommendation for handling the <code>TransactionInProgressException</code>. These
     *         settings help ensure that the client retries will trigger completion of the ongoing <code>
     *         TransactWriteItems</code> request.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         Set <code>clientExecutionTimeout</code> to a value that allows at least one retry to be processed after 5
     *         seconds have elapsed since the first attempt for the <code>TransactWriteItems</code> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set <code>socketTimeout</code> to a value a little lower than the <code>requestTimeout</code> setting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>requestTimeout</code> should be set based on the time taken for the individual retries of a single
     *         HTTP request for your use case, but setting it to 1 second or higher should work well to reduce chances
     *         of retries and <code>TransactionInProgressException</code> errors.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use exponential backoff when retrying and tune backoff if needed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Assuming <a href=
     *         "https://github.com/aws/aws-sdk-java/blob/fd409dee8ae23fb8953e0bb4dbde65536a7e0514/aws-java-sdk-core/src/main/java/com/amazonaws/retry/PredefinedRetryPolicies.java#L97"
     *         >default retry policy</a>, example timeout settings based on the guidelines above are as follows:
     *         </p>
     *         <p>
     *         Example timeline:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         0-1000 first attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1000-1500 first sleep/delay (default retry policy uses 500 ms as base delay for 4xx errors)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1500-2500 second attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         2500-3500 second sleep/delay (500 * 2, exponential backoff)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         3500-4500 third attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500-6500 third sleep/delay (500 * 2^2)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         6500-7500 fourth attempt (this can trigger inline recovery since 5 seconds have elapsed since the first
     *         attempt reached TC)
     *         </p>
     *         </li>
     * @throws IdempotentParameterMismatchException
     *         DynamoDB rejected the request because you retried a request with a different payload but with an
     *         idempotent token that was already used.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ExecuteTransaction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ExecuteTransaction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ExecuteTransactionResponse executeTransaction(ExecuteTransactionRequest executeTransactionRequest)
            throws ResourceNotFoundException, TransactionCanceledException, TransactionInProgressException,
            IdempotentParameterMismatchException, ProvisionedThroughputExceededException, RequestLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExecuteTransactionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExecuteTransactionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(executeTransactionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, executeTransactionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExecuteTransaction");

            return clientHandler.execute(new ClientExecutionParams<ExecuteTransactionRequest, ExecuteTransactionResponse>()
                    .withOperationName("ExecuteTransaction").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(executeTransactionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ExecuteTransactionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports table data to an S3 bucket. The table must have point in time recovery enabled, and you can export data
     * from any time within the point in time recovery window.
     * </p>
     *
     * @param exportTableToPointInTimeRequest
     * @return Result of the ExportTableToPointInTime operation returned by the service.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws PointInTimeRecoveryUnavailableException
     *         Point in time recovery has not yet been enabled for this source table.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InvalidExportTimeException
     *         The specified <code>ExportTime</code> is outside of the point in time recovery window.
     * @throws ExportConflictException
     *         There was a conflict when writing to the specified S3 bucket.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ExportTableToPointInTime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ExportTableToPointInTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportTableToPointInTimeResponse exportTableToPointInTime(
            ExportTableToPointInTimeRequest exportTableToPointInTimeRequest) throws TableNotFoundException,
            PointInTimeRecoveryUnavailableException, LimitExceededException, InvalidExportTimeException, ExportConflictException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportTableToPointInTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExportTableToPointInTimeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(exportTableToPointInTimeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, exportTableToPointInTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportTableToPointInTime");

            return clientHandler
                    .execute(new ClientExecutionParams<ExportTableToPointInTimeRequest, ExportTableToPointInTimeResponse>()
                            .withOperationName("ExportTableToPointInTime").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(exportTableToPointInTimeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExportTableToPointInTimeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>GetItem</code> operation returns a set of attributes for the item with the given primary key. If there
     * is no matching item, <code>GetItem</code> does not return any data and there will be no <code>Item</code> element
     * in the response.
     * </p>
     * <p>
     * <code>GetItem</code> provides an eventually consistent read by default. If your application requires a strongly
     * consistent read, set <code>ConsistentRead</code> to <code>true</code>. Although a strongly consistent read might
     * take more time than an eventually consistent read, it always returns the last updated value.
     * </p>
     *
     * @param getItemRequest
     *        Represents the input of a <code>GetItem</code> operation.
     * @return Result of the GetItem operation returned by the service.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.GetItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/GetItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetItemResponse getItem(GetItemRequest getItemRequest) throws ProvisionedThroughputExceededException,
            ResourceNotFoundException, RequestLimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = getItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(getItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetItem");

            return clientHandler.execute(new ClientExecutionParams<GetItemRequest, GetItemResponse>()
                    .withOperationName("GetItem").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(getItemRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new GetItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the resource-based policy document attached to the resource, which can be a table or stream, in JSON
     * format.
     * </p>
     * <p>
     * <code>GetResourcePolicy</code> follows an <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.ReadConsistency.html">
     * <i>eventually consistent</i> </a> model. The following list describes the outcomes when you issue the
     * <code>GetResourcePolicy</code> request immediately after issuing another request:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If you issue a <code>GetResourcePolicy</code> request immediately after a <code>PutResourcePolicy</code> request,
     * DynamoDB might return a <code>PolicyNotFoundException</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you issue a <code>GetResourcePolicy</code>request immediately after a <code>DeleteResourcePolicy</code>
     * request, DynamoDB might return the policy that was present before the deletion request.
     * </p>
     * </li>
     * <li>
     * <p>
     * If you issue a <code>GetResourcePolicy</code> request immediately after a <code>CreateTable</code> request, which
     * includes a resource-based policy, DynamoDB might return a <code>ResourceNotFoundException</code> or a
     * <code>PolicyNotFoundException</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Because <code>GetResourcePolicy</code> uses an <i>eventually consistent</i> query, the metadata for your policy
     * or table might not be available at that moment. Wait for a few seconds, and then retry the
     * <code>GetResourcePolicy</code> request.
     * </p>
     * <p>
     * After a <code>GetResourcePolicy</code> request returns a policy created using the <code>PutResourcePolicy</code>
     * request, the policy will be applied in the authorization of requests to the resource. Because this process is
     * eventually consistent, it will take some time to apply the policy to all requests to a resource. Policies that
     * you attach while creating a table using the <code>CreateTable</code> request will always be applied to all
     * requests for that table.
     * </p>
     *
     * @param getResourcePolicyRequest
     * @return Result of the GetResourcePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws PolicyNotFoundException
     *         The operation tried to access a nonexistent resource-based policy.</p>
     *         <p>
     *         If you specified an <code>ExpectedRevisionId</code>, it's possible that a policy is present for the
     *         resource but its revision ID didn't match the expected value.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.GetResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/GetResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetResourcePolicyResponse getResourcePolicy(GetResourcePolicyRequest getResourcePolicyRequest)
            throws ResourceNotFoundException, InternalServerErrorException, PolicyNotFoundException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = getResourcePolicyRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(getResourcePolicyRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<GetResourcePolicyRequest, GetResourcePolicyResponse>()
                    .withOperationName("GetResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(getResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Imports table data from an S3 bucket.
     * </p>
     *
     * @param importTableRequest
     * @return Result of the ImportTable operation returned by the service.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ImportConflictException
     *         There was a conflict when importing from the specified S3 source. This can occur when the current import
     *         conflicts with a previous import request that had the same client token.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ImportTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ImportTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ImportTableResponse importTable(ImportTableRequest importTableRequest) throws ResourceInUseException,
            LimitExceededException, ImportConflictException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ImportTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ImportTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(importTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, importTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportTable");

            return clientHandler.execute(new ClientExecutionParams<ImportTableRequest, ImportTableResponse>()
                    .withOperationName("ImportTable").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(importTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ImportTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List DynamoDB backups that are associated with an Amazon Web Services account and weren't made with Amazon Web
     * Services Backup. To list these backups for a given table, specify <code>TableName</code>.
     * <code>ListBackups</code> returns a paginated list of results with at most 1 MB worth of items in a page. You can
     * also specify a maximum number of entries to be returned in a page.
     * </p>
     * <p>
     * In the request, start time is inclusive, but end time is exclusive. Note that these boundaries are for the time
     * at which the original backup was requested.
     * </p>
     * <p>
     * You can call <code>ListBackups</code> a maximum of five times per second.
     * </p>
     * <p>
     * If you want to retrieve the complete list of backups made with Amazon Web Services Backup, use the <a
     * href="https://docs.aws.amazon.com/aws-backup/latest/devguide/API_ListBackupJobs.html">Amazon Web Services Backup
     * list API.</a>
     * </p>
     *
     * @param listBackupsRequest
     * @return Result of the ListBackups operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListBackupsResponse listBackups(ListBackupsRequest listBackupsRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listBackupsRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listBackupsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listBackupsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListBackups");

            return clientHandler.execute(new ClientExecutionParams<ListBackupsRequest, ListBackupsResponse>()
                    .withOperationName("ListBackups").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(listBackupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of ContributorInsightsSummary for a table and all its global secondary indexes.
     * </p>
     *
     * @param listContributorInsightsRequest
     * @return Result of the ListContributorInsights operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListContributorInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListContributorInsightsResponse listContributorInsights(ListContributorInsightsRequest listContributorInsightsRequest)
            throws ResourceNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListContributorInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listContributorInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListContributorInsights");

            return clientHandler
                    .execute(new ClientExecutionParams<ListContributorInsightsRequest, ListContributorInsightsResponse>()
                            .withOperationName("ListContributorInsights").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(listContributorInsightsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListContributorInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists completed exports within the past 90 days.
     * </p>
     *
     * @param listExportsRequest
     * @return Result of the ListExports operation returned by the service.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListExports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListExports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListExportsResponse listExports(ListExportsRequest listExportsRequest) throws LimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListExportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListExportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listExportsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listExportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListExports");

            return clientHandler.execute(new ClientExecutionParams<ListExportsRequest, ListExportsResponse>()
                    .withOperationName("ListExports").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listExportsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListExportsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists all global tables that have a replica in the specified Region.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important>
     *
     * @param listGlobalTablesRequest
     * @return Result of the ListGlobalTables operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListGlobalTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListGlobalTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListGlobalTablesResponse listGlobalTables(ListGlobalTablesRequest listGlobalTablesRequest)
            throws InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListGlobalTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListGlobalTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listGlobalTablesRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listGlobalTablesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listGlobalTablesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listGlobalTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListGlobalTables");

            return clientHandler.execute(new ClientExecutionParams<ListGlobalTablesRequest, ListGlobalTablesResponse>()
                    .withOperationName("ListGlobalTables").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(listGlobalTablesRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListGlobalTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists completed imports within the past 90 days.
     * </p>
     *
     * @param listImportsRequest
     * @return Result of the ListImports operation returned by the service.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListImports
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListImports" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListImportsResponse listImports(ListImportsRequest listImportsRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListImportsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListImportsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listImportsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listImportsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListImports");

            return clientHandler.execute(new ClientExecutionParams<ListImportsRequest, ListImportsResponse>()
                    .withOperationName("ListImports").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listImportsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListImportsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of table names associated with the current account and endpoint. The output from
     * <code>ListTables</code> is paginated, with each page returning a maximum of 100 table names.
     * </p>
     *
     * @param listTablesRequest
     *        Represents the input of a <code>ListTables</code> operation.
     * @return Result of the ListTables operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListTables
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListTables" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTablesResponse listTables(ListTablesRequest listTablesRequest) throws InternalServerErrorException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTablesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTablesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listTablesRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listTablesRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTablesRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTablesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTables");

            return clientHandler
                    .execute(new ClientExecutionParams<ListTablesRequest, ListTablesResponse>().withOperationName("ListTables")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                            .withRequestConfiguration(clientConfiguration).withInput(listTablesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ListTablesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List all tags on an Amazon DynamoDB resource. You can call ListTagsOfResource up to 10 times per second, per
     * account.
     * </p>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param listTagsOfResourceRequest
     * @return Result of the ListTagsOfResource operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.ListTagsOfResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/ListTagsOfResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListTagsOfResourceResponse listTagsOfResource(ListTagsOfResourceRequest listTagsOfResourceRequest)
            throws ResourceNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsOfResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsOfResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = listTagsOfResourceRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(listTagsOfResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsOfResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsOfResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsOfResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsOfResourceRequest, ListTagsOfResourceResponse>()
                    .withOperationName("ListTagsOfResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(listTagsOfResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsOfResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new item, or replaces an old item with a new item. If an item that has the same primary key as the new
     * item already exists in the specified table, the new item completely replaces the existing item. You can perform a
     * conditional put operation (add a new item if one with the specified primary key doesn't exist), or replace an
     * existing item if it has certain attribute values. You can return the item's attribute values in the same
     * operation, using the <code>ReturnValues</code> parameter.
     * </p>
     * <p>
     * When you add an item, the primary key attributes are the only required attributes.
     * </p>
     * <p>
     * Empty String and Binary attribute values are allowed. Attribute values of type String and Binary must have a
     * length greater than zero if the attribute is used as a key attribute for a table or index. Set type attributes
     * cannot be empty.
     * </p>
     * <p>
     * Invalid Requests with empty values will be rejected with a <code>ValidationException</code> exception.
     * </p>
     * <note>
     * <p>
     * To prevent a new item from replacing an existing item, use a conditional expression that contains the
     * <code>attribute_not_exists</code> function with the name of the attribute being used as the partition key for the
     * table. Since every record must contain that attribute, the <code>attribute_not_exists</code> function will only
     * succeed if no matching item exists.
     * </p>
     * </note>
     * <p>
     * For more information about <code>PutItem</code>, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/WorkingWithItems.html">Working with
     * Items</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param putItemRequest
     *        Represents the input of a <code>PutItem</code> operation.
     * @return Result of the PutItem operation returned by the service.
     * @throws ConditionalCheckFailedException
     *         A condition specified in the operation failed to be evaluated.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ItemCollectionSizeLimitExceededException
     *         An item collection is too large. This exception is only returned for tables that have one or more local
     *         secondary indexes.
     * @throws TransactionConflictException
     *         Operation was rejected because there is an ongoing transaction for the item.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ReplicatedWriteConflictException
     *         The request was rejected because one or more items in the request are being modified by a request in
     *         another Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.PutItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/PutItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutItemResponse putItem(PutItemRequest putItemRequest) throws ConditionalCheckFailedException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ItemCollectionSizeLimitExceededException,
            TransactionConflictException, RequestLimitExceededException, InternalServerErrorException,
            ReplicatedWriteConflictException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = putItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(putItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutItem");

            return clientHandler.execute(new ClientExecutionParams<PutItemRequest, PutItemResponse>()
                    .withOperationName("PutItem").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(putItemRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new PutItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Attaches a resource-based policy document to the resource, which can be a table or stream. When you attach a
     * resource-based policy using this API, the policy application is <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.ReadConsistency.html">
     * <i>eventually consistent</i> </a>.
     * </p>
     * <p>
     * <code>PutResourcePolicy</code> is an idempotent operation; running it multiple times on the same resource using
     * the same policy document will return the same revision ID. If you specify an <code>ExpectedRevisionId</code> that
     * doesn't match the current policy's <code>RevisionId</code>, the <code>PolicyNotFoundException</code> will be
     * returned.
     * </p>
     * <note>
     * <p>
     * <code>PutResourcePolicy</code> is an asynchronous operation. If you issue a <code>GetResourcePolicy</code>
     * request immediately after a <code>PutResourcePolicy</code> request, DynamoDB might return your previous policy,
     * if there was one, or return the <code>PolicyNotFoundException</code>. This is because
     * <code>GetResourcePolicy</code> uses an eventually consistent query, and the metadata for your policy or table
     * might not be available at that moment. Wait for a few seconds, and then try the <code>GetResourcePolicy</code>
     * request again.
     * </p>
     * </note>
     *
     * @param putResourcePolicyRequest
     * @return Result of the PutResourcePolicy operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws PolicyNotFoundException
     *         The operation tried to access a nonexistent resource-based policy.
     *         </p>
     *         <p>
     *         If you specified an <code>ExpectedRevisionId</code>, it's possible that a policy is present for the
     *         resource but its revision ID didn't match the expected value.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.PutResourcePolicy
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/PutResourcePolicy" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutResourcePolicyResponse putResourcePolicy(PutResourcePolicyRequest putResourcePolicyRequest)
            throws ResourceNotFoundException, InternalServerErrorException, LimitExceededException, PolicyNotFoundException,
            ResourceInUseException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<PutResourcePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                PutResourcePolicyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = putResourcePolicyRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(putResourcePolicyRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(putResourcePolicyRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putResourcePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutResourcePolicy");

            return clientHandler.execute(new ClientExecutionParams<PutResourcePolicyRequest, PutResourcePolicyResponse>()
                    .withOperationName("PutResourcePolicy").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(putResourcePolicyRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutResourcePolicyRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You must provide the name of the partition key attribute and a single value for that attribute.
     * <code>Query</code> returns all items with that partition key value. Optionally, you can provide a sort key
     * attribute and use a comparison operator to refine the search results.
     * </p>
     * <p>
     * Use the <code>KeyConditionExpression</code> parameter to provide a specific value for the partition key. The
     * <code>Query</code> operation will return all of the items from the table or index with that partition key value.
     * You can optionally narrow the scope of the <code>Query</code> operation by specifying a sort key value and a
     * comparison operator in <code>KeyConditionExpression</code>. To further refine the <code>Query</code> results, you
     * can optionally provide a <code>FilterExpression</code>. A <code>FilterExpression</code> determines which items
     * within the results should be returned to you. All of the other results are discarded.
     * </p>
     * <p>
     * A <code>Query</code> operation always returns a result set. If no matching items are found, the result set will
     * be empty. Queries that do not return results consume the minimum number of read capacity units for that type of
     * read operation.
     * </p>
     * <note>
     * <p>
     * DynamoDB calculates the number of read capacity units consumed based on item size, not on the amount of data that
     * is returned to an application. The number of capacity units consumed will be the same whether you request all of
     * the attributes (the default behavior) or just some of them (using a projection expression). The number will also
     * be the same whether or not you use a <code>FilterExpression</code>.
     * </p>
     * </note>
     * <p>
     * <code>Query</code> results are always sorted by the sort key value. If the data type of the sort key is Number,
     * the results are returned in numeric order; otherwise, the results are returned in order of UTF-8 bytes. By
     * default, the sort order is ascending. To reverse the order, set the <code>ScanIndexForward</code> parameter to
     * false.
     * </p>
     * <p>
     * A single <code>Query</code> operation will read up to the maximum number of items set (if using the
     * <code>Limit</code> parameter) or a maximum of 1 MB of data and then apply any filtering to the results using
     * <code>FilterExpression</code>. If <code>LastEvaluatedKey</code> is present in the response, you will need to
     * paginate the result set. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Query.html#Query.Pagination">Paginating
     * the Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>FilterExpression</code> is applied after a <code>Query</code> finishes, but before the results are
     * returned. A <code>FilterExpression</code> cannot contain partition key or sort key attributes. You need to
     * specify those attributes in the <code>KeyConditionExpression</code>.
     * </p>
     * <note>
     * <p>
     * A <code>Query</code> operation can return an empty result set and a <code>LastEvaluatedKey</code> if all the
     * items read for the page of results are filtered out.
     * </p>
     * </note>
     * <p>
     * You can query a table, a local secondary index, or a global secondary index. For a query on a table or on a local
     * secondary index, you can set the <code>ConsistentRead</code> parameter to <code>true</code> and obtain a strongly
     * consistent result. Global secondary indexes support eventually consistent reads only, so do not specify
     * <code>ConsistentRead</code> when querying a global secondary index.
     * </p>
     *
     * @param queryRequest
     *        Represents the input of a <code>Query</code> operation.
     * @return Result of the Query operation returned by the service.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.Query
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Query" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public QueryResponse query(QueryRequest queryRequest) throws ProvisionedThroughputExceededException,
            ResourceNotFoundException, RequestLimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<QueryResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                QueryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = queryRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(queryRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(queryRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, queryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Query");

            return clientHandler.execute(new ClientExecutionParams<QueryRequest, QueryResponse>().withOperationName("Query")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(queryRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new QueryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new table from an existing backup. Any number of users can execute up to 50 concurrent restores (any
     * type of restore) in a given account.
     * </p>
     * <p>
     * You can call <code>RestoreTableFromBackup</code> at a maximum rate of 10 times per second.
     * </p>
     * <p>
     * You must manually set up the following on the restored table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto scaling policies
     * </p>
     * </li>
     * <li>
     * <p>
     * IAM policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Time to Live (TTL) settings
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableFromBackupRequest
     * @return Result of the RestoreTableFromBackup operation returned by the service.
     * @throws TableAlreadyExistsException
     *         A target table with the specified name already exists.
     * @throws TableInUseException
     *         A target table with the specified name is either being created or deleted.
     * @throws BackupNotFoundException
     *         Backup not found for the given BackupARN.
     * @throws BackupInUseException
     *         There is another ongoing conflicting backup control plane operation on the table. The backup is either
     *         being created, deleted or restored to a table.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.RestoreTableFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/RestoreTableFromBackup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RestoreTableFromBackupResponse restoreTableFromBackup(RestoreTableFromBackupRequest restoreTableFromBackupRequest)
            throws TableAlreadyExistsException, TableInUseException, BackupNotFoundException, BackupInUseException,
            LimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreTableFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreTableFromBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = restoreTableFromBackupRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(restoreTableFromBackupRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreTableFromBackupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTableFromBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<RestoreTableFromBackupRequest, RestoreTableFromBackupResponse>()
                            .withOperationName("RestoreTableFromBackup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(restoreTableFromBackupRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RestoreTableFromBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Restores the specified table to the specified point in time within <code>EarliestRestorableDateTime</code> and
     * <code>LatestRestorableDateTime</code>. You can restore your table to any point in time in the last 35 days. You
     * can set the recovery period to any value between 1 and 35 days. Any number of users can execute up to 50
     * concurrent restores (any type of restore) in a given account.
     * </p>
     * <p>
     * When you restore using point in time recovery, DynamoDB restores your table data to the state based on the
     * selected date and time (day:hour:minute:second) to a new table.
     * </p>
     * <p>
     * Along with data, the following are also included on the new restored table using point in time recovery:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Global secondary indexes (GSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Local secondary indexes (LSIs)
     * </p>
     * </li>
     * <li>
     * <p>
     * Provisioned read and write capacity
     * </p>
     * </li>
     * <li>
     * <p>
     * Encryption settings
     * </p>
     * <important>
     * <p>
     * All these settings come from the current settings of the source table at the time of restore.
     * </p>
     * </important></li>
     * </ul>
     * <p>
     * You must manually set up the following on the restored table:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Auto scaling policies
     * </p>
     * </li>
     * <li>
     * <p>
     * IAM policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon CloudWatch metrics and alarms
     * </p>
     * </li>
     * <li>
     * <p>
     * Tags
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Time to Live (TTL) settings
     * </p>
     * </li>
     * <li>
     * <p>
     * Point in time recovery settings
     * </p>
     * </li>
     * </ul>
     *
     * @param restoreTableToPointInTimeRequest
     * @return Result of the RestoreTableToPointInTime operation returned by the service.
     * @throws TableAlreadyExistsException
     *         A target table with the specified name already exists.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws TableInUseException
     *         A target table with the specified name is either being created or deleted.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InvalidRestoreTimeException
     *         An invalid restore time was specified. RestoreDateTime must be between EarliestRestorableDateTime and
     *         LatestRestorableDateTime.
     * @throws PointInTimeRecoveryUnavailableException
     *         Point in time recovery has not yet been enabled for this source table.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.RestoreTableToPointInTime
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/RestoreTableToPointInTime"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public RestoreTableToPointInTimeResponse restoreTableToPointInTime(
            RestoreTableToPointInTimeRequest restoreTableToPointInTimeRequest) throws TableAlreadyExistsException,
            TableNotFoundException, TableInUseException, LimitExceededException, InvalidRestoreTimeException,
            PointInTimeRecoveryUnavailableException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreTableToPointInTimeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreTableToPointInTimeResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = restoreTableToPointInTimeRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(restoreTableToPointInTimeRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreTableToPointInTimeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreTableToPointInTimeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreTableToPointInTime");

            return clientHandler
                    .execute(new ClientExecutionParams<RestoreTableToPointInTimeRequest, RestoreTableToPointInTimeResponse>()
                            .withOperationName("RestoreTableToPointInTime").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(restoreTableToPointInTimeRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RestoreTableToPointInTimeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>Scan</code> operation returns one or more items and item attributes by accessing every item in a table
     * or a secondary index. To have DynamoDB return fewer items, you can provide a <code>FilterExpression</code>
     * operation.
     * </p>
     * <p>
     * If the total size of scanned items exceeds the maximum dataset size limit of 1 MB, the scan completes and results
     * are returned to the user. The <code>LastEvaluatedKey</code> value is also returned and the requestor can use the
     * <code>LastEvaluatedKey</code> to continue the scan in a subsequent operation. Each scan response also includes
     * number of items that were scanned (ScannedCount) as part of the request. If using a <code>FilterExpression</code>
     * , a scan result can result in no items meeting the criteria and the <code>Count</code> will result in zero. If
     * you did not use a <code>FilterExpression</code> in the scan request, then <code>Count</code> is the same as
     * <code>ScannedCount</code>.
     * </p>
     * <note>
     * <p>
     * <code>Count</code> and <code>ScannedCount</code> only return the count of items specific to a single scan request
     * and, unless the table is less than 1MB, do not represent the total number of items in the table.
     * </p>
     * </note>
     * <p>
     * A single <code>Scan</code> operation first reads up to the maximum number of items set (if using the
     * <code>Limit</code> parameter) or a maximum of 1 MB of data and then applies any filtering to the results if a
     * <code>FilterExpression</code> is provided. If <code>LastEvaluatedKey</code> is present in the response,
     * pagination is required to complete the full table scan. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.Pagination">Paginating the
     * Results</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * <code>Scan</code> operations proceed sequentially; however, for faster performance on a large table or secondary
     * index, applications can request a parallel <code>Scan</code> operation by providing the <code>Segment</code> and
     * <code>TotalSegments</code> parameters. For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Scan.html#Scan.ParallelScan">Parallel
     * Scan</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     * <p>
     * By default, a <code>Scan</code> uses eventually consistent reads when accessing the items in a table. Therefore,
     * the results from an eventually consistent <code>Scan</code> may not include the latest item changes at the time
     * the scan iterates through each item in the table. If you require a strongly consistent read of each item as the
     * scan iterates through the items in the table, you can set the <code>ConsistentRead</code> parameter to true.
     * Strong consistency only relates to the consistency of the read at the item level.
     * </p>
     * <note>
     * <p>
     * DynamoDB does not provide snapshot isolation for a scan operation when the <code>ConsistentRead</code> parameter
     * is set to true. Thus, a DynamoDB scan operation does not guarantee that all reads in a scan see a consistent
     * snapshot of the table when the scan operation was requested.
     * </p>
     * </note>
     *
     * @param scanRequest
     *        Represents the input of a <code>Scan</code> operation.
     * @return Result of the Scan operation returned by the service.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.Scan
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/Scan" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ScanResponse scan(ScanRequest scanRequest) throws ProvisionedThroughputExceededException, ResourceNotFoundException,
            RequestLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ScanResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ScanResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = scanRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(scanRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(scanRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, scanRequest.overrideConfiguration()
                .orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "Scan");

            return clientHandler.execute(new ClientExecutionParams<ScanRequest, ScanResponse>().withOperationName("Scan")
                    .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(scanRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ScanRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Associate a set of tags with an Amazon DynamoDB resource. You can then activate these user-defined tags so that
     * they appear on the Billing and Cost Management console for cost allocation tracking. You can call TagResource up
     * to five times per second, per account.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>TagResource</code> is an asynchronous operation. If you issue a <a>ListTagsOfResource</a> request
     * immediately after a <code>TagResource</code> request, DynamoDB might return your previous tag set, if there was
     * one, or an empty tag set. This is because <code>ListTagsOfResource</code> uses an eventually consistent query,
     * and the metadata for your tags or table might not be available at that moment. Wait for a few seconds, and then
     * try the <code>ListTagsOfResource</code> request again.
     * </p>
     * </li>
     * <li>
     * <p>
     * The application or removal of tags using <code>TagResource</code> and <code>UntagResource</code> APIs is
     * eventually consistent. <code>ListTagsOfResource</code> API will only reflect the changes after a few seconds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws LimitExceededException,
            ResourceNotFoundException, InternalServerErrorException, ResourceInUseException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = tagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(tagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>TransactGetItems</code> is a synchronous operation that atomically retrieves multiple items from one or
     * more tables (but not from indexes) in a single account and Region. A <code>TransactGetItems</code> call can
     * contain up to 100 <code>TransactGetItem</code> objects, each of which contains a <code>Get</code> structure that
     * specifies an item to retrieve from a table in the account and Region. A call to <code>TransactGetItems</code>
     * cannot retrieve items from tables in more than one Amazon Web Services account or Region. The aggregate size of
     * the items in the transaction cannot exceed 4 MB.
     * </p>
     * <p>
     * DynamoDB rejects the entire <code>TransactGetItems</code> request if any of the following is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A conflicting operation is in the process of updating an item to be read.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is insufficient provisioned capacity for the transaction to be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is a user error, such as an invalid data format.
     * </p>
     * </li>
     * <li>
     * <p>
     * The aggregate size of the items in the transaction exceeded 4 MB.
     * </p>
     * </li>
     * </ul>
     *
     * @param transactGetItemsRequest
     * @return Result of the TransactGetItems operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws TransactionCanceledException
     *         The entire transaction request was canceled.</p>
     *         <p>
     *         DynamoDB cancels a <code>TransactWriteItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A condition in one of the condition expressions is not met.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactWriteItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than one action in the <code>TransactWriteItems</code> operation targets the same item.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An item size becomes too large (larger than 400 KB), or a local secondary index (LSI) becomes too large,
     *         or a similar validation error occurs because of changes made by the transaction.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactWriteItems</code> operation that conflicts with a concurrent
     *         <code>TransactWriteItems</code> request. In this case the <code>TransactWriteItems</code> operation fails
     *         with a <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         DynamoDB cancels a <code>TransactGetItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactGetItems</code> operation that conflicts with a concurrent
     *         <code>PutItem</code>, <code>UpdateItem</code>, <code>DeleteItem</code> or <code>TransactWriteItems</code>
     *         request. In this case the <code>TransactGetItems</code> operation fails with a
     *         <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactGetItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If using Java, DynamoDB lists the cancellation reasons on the <code>CancellationReasons</code> property.
     *         This property is not set for other languages. Transaction cancellation reasons are ordered in the order
     *         of requested items, if an item has no error it will have <code>None</code> code and <code>Null</code>
     *         message.
     *         </p>
     *         </note>
     *         <p>
     *         Cancellation reason codes and possible error messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         No Errors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>None</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: <code>null</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Conditional Check Failed:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ConditionalCheckFailed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: The conditional request failed.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Item Collection Size Limit Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ItemCollectionSizeLimitExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Collection size exceeded.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Transaction Conflict:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>TransactionConflict</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Transaction is ongoing for the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned Throughput Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for the table was exceeded. Consider increasing your
     *         provisioning level with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This Message is received when provisioned throughput is exceeded is on a provisioned DynamoDB table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for one or more global secondary indexes of the table was
     *         exceeded. Consider increasing your provisioning level for the under-provisioned global secondary indexes
     *         with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when provisioned throughput is exceeded is on a provisioned GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Throttling Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ThrottlingError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity of your table or index. DynamoDB is automatically scaling your
     *         table or index so please try again shortly. If exceptions persist, check if you have a hot key:
     *         https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand table as DynamoDB is automatically
     *         scaling the table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity for one or more global secondary indexes. DynamoDB is
     *         automatically scaling your index so please try again shortly.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand GSI as DynamoDB is automatically
     *         scaling the GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Validation Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ValidationError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more parameter values were invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key beyond allowed size limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key to unsupported type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An operand in the update expression has an incorrect data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Item size to update has exceeded the maximum allowed size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Number overflow. Attempting to store a number with magnitude larger than supported range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Type mismatch for attribute to update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Nesting Levels have exceeded supported limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The document path provided in the update expression is invalid for update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The provided expression refers to an attribute that does not exist in the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.TransactGetItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TransactGetItems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TransactGetItemsResponse transactGetItems(TransactGetItemsRequest transactGetItemsRequest)
            throws ResourceNotFoundException, TransactionCanceledException, ProvisionedThroughputExceededException,
            RequestLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException,
            DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TransactGetItemsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TransactGetItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = transactGetItemsRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(transactGetItemsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(transactGetItemsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transactGetItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransactGetItems");

            return clientHandler.execute(new ClientExecutionParams<TransactGetItemsRequest, TransactGetItemsResponse>()
                    .withOperationName("TransactGetItems").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(transactGetItemsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TransactGetItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>TransactWriteItems</code> is a synchronous write operation that groups up to 100 action requests. These
     * actions can target items in different tables, but not in different Amazon Web Services accounts or Regions, and
     * no two actions can target the same item. For example, you cannot both <code>ConditionCheck</code> and
     * <code>Update</code> the same item. The aggregate size of the items in the transaction cannot exceed 4 MB.
     * </p>
     * <p>
     * The actions are completed atomically so that either all of them succeed, or all of them fail. They are defined by
     * the following objects:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Put</code>  —   Initiates a <code>PutItem</code> operation to write a new item. This structure specifies
     * the primary key of the item to be written, the name of the table to write it in, an optional condition expression
     * that must be satisfied for the write to succeed, a list of the item's attributes, and a field indicating whether
     * to retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Update</code>  —   Initiates an <code>UpdateItem</code> operation to update an existing item. This
     * structure specifies the primary key of the item to be updated, the name of the table where it resides, an
     * optional condition expression that must be satisfied for the update to succeed, an expression that defines one or
     * more attributes to be updated, and a field indicating whether to retrieve the item's attributes if the condition
     * is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Delete</code>  —   Initiates a <code>DeleteItem</code> operation to delete an existing item. This structure
     * specifies the primary key of the item to be deleted, the name of the table where it resides, an optional
     * condition expression that must be satisfied for the deletion to succeed, and a field indicating whether to
     * retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConditionCheck</code>  —   Applies a condition to an item that is not being modified by the transaction.
     * This structure specifies the primary key of the item to be checked, the name of the table where it resides, a
     * condition expression that must be satisfied for the transaction to succeed, and a field indicating whether to
     * retrieve the item's attributes if the condition is not met.
     * </p>
     * </li>
     * </ul>
     * <p>
     * DynamoDB rejects the entire <code>TransactWriteItems</code> request if any of the following is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A condition in one of the condition expressions is not met.
     * </p>
     * </li>
     * <li>
     * <p>
     * An ongoing operation is in the process of updating the same item.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is insufficient provisioned capacity for the transaction to be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * An item size becomes too large (bigger than 400 KB), a local secondary index (LSI) becomes too large, or a
     * similar validation error occurs because of changes made by the transaction.
     * </p>
     * </li>
     * <li>
     * <p>
     * The aggregate size of the items in the transaction exceeds 4 MB.
     * </p>
     * </li>
     * <li>
     * <p>
     * There is a user error, such as an invalid data format.
     * </p>
     * </li>
     * </ul>
     *
     * @param transactWriteItemsRequest
     * @return Result of the TransactWriteItems operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws TransactionCanceledException
     *         The entire transaction request was canceled.</p>
     *         <p>
     *         DynamoDB cancels a <code>TransactWriteItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A condition in one of the condition expressions is not met.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactWriteItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         More than one action in the <code>TransactWriteItems</code> operation targets the same item.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An item size becomes too large (larger than 400 KB), or a local secondary index (LSI) becomes too large,
     *         or a similar validation error occurs because of changes made by the transaction.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactWriteItems</code> operation that conflicts with a concurrent
     *         <code>TransactWriteItems</code> request. In this case the <code>TransactWriteItems</code> operation fails
     *         with a <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         DynamoDB cancels a <code>TransactGetItems</code> request under the following circumstances:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         There is an ongoing <code>TransactGetItems</code> operation that conflicts with a concurrent
     *         <code>PutItem</code>, <code>UpdateItem</code>, <code>DeleteItem</code> or <code>TransactWriteItems</code>
     *         request. In this case the <code>TransactGetItems</code> operation fails with a
     *         <code>TransactionCanceledException</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         A table in the <code>TransactGetItems</code> request is in a different account or region.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is insufficient provisioned capacity for the transaction to be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         There is a user error, such as an invalid data format.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         If using Java, DynamoDB lists the cancellation reasons on the <code>CancellationReasons</code> property.
     *         This property is not set for other languages. Transaction cancellation reasons are ordered in the order
     *         of requested items, if an item has no error it will have <code>None</code> code and <code>Null</code>
     *         message.
     *         </p>
     *         </note>
     *         <p>
     *         Cancellation reason codes and possible error messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         No Errors:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>None</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: <code>null</code>
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Conditional Check Failed:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ConditionalCheckFailed</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: The conditional request failed.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Item Collection Size Limit Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ItemCollectionSizeLimitExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Collection size exceeded.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Transaction Conflict:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>TransactionConflict</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Message: Transaction is ongoing for the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Provisioned Throughput Exceeded:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ProvisionedThroughputExceeded</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for the table was exceeded. Consider increasing your
     *         provisioning level with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This Message is received when provisioned throughput is exceeded is on a provisioned DynamoDB table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         The level of configured provisioned throughput for one or more global secondary indexes of the table was
     *         exceeded. Consider increasing your provisioning level for the under-provisioned global secondary indexes
     *         with the UpdateTable API.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when provisioned throughput is exceeded is on a provisioned GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Throttling Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ThrottlingError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity of your table or index. DynamoDB is automatically scaling your
     *         table or index so please try again shortly. If exceptions persist, check if you have a hot key:
     *         https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/bp-partition-key-design.html.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand table as DynamoDB is automatically
     *         scaling the table.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         Throughput exceeds the current capacity for one or more global secondary indexes. DynamoDB is
     *         automatically scaling your index so please try again shortly.
     *         </p>
     *         <note>
     *         <p>
     *         This message is returned when writes get throttled on an On-Demand GSI as DynamoDB is automatically
     *         scaling the GSI.
     *         </p>
     *         </note></li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     *         <li>
     *         <p>
     *         Validation Error:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Code: <code>ValidationError</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Messages:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         One or more parameter values were invalid.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key beyond allowed size limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The update expression attempted to update the secondary index key to unsupported type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         An operand in the update expression has an incorrect data type.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Item size to update has exceeded the maximum allowed size.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Number overflow. Attempting to store a number with magnitude larger than supported range.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Type mismatch for attribute to update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Nesting Levels have exceeded supported limits.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The document path provided in the update expression is invalid for update.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The provided expression refers to an attribute that does not exist in the item.
     *         </p>
     *         </li>
     *         </ul>
     *         </li>
     *         </ul>
     *         </li>
     * @throws TransactionInProgressException
     *         The transaction with the given request token is already in progress.</p>
     *         <p>
     *         Recommended Settings
     *         </p>
     *         <note>
     *         <p>
     *         This is a general recommendation for handling the <code>TransactionInProgressException</code>. These
     *         settings help ensure that the client retries will trigger completion of the ongoing <code>
     *         TransactWriteItems</code> request.
     *         </p>
     *         </note>
     *         <ul>
     *         <li>
     *         <p>
     *         Set <code>clientExecutionTimeout</code> to a value that allows at least one retry to be processed after 5
     *         seconds have elapsed since the first attempt for the <code>TransactWriteItems</code> operation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Set <code>socketTimeout</code> to a value a little lower than the <code>requestTimeout</code> setting.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>requestTimeout</code> should be set based on the time taken for the individual retries of a single
     *         HTTP request for your use case, but setting it to 1 second or higher should work well to reduce chances
     *         of retries and <code>TransactionInProgressException</code> errors.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Use exponential backoff when retrying and tune backoff if needed.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Assuming <a href=
     *         "https://github.com/aws/aws-sdk-java/blob/fd409dee8ae23fb8953e0bb4dbde65536a7e0514/aws-java-sdk-core/src/main/java/com/amazonaws/retry/PredefinedRetryPolicies.java#L97"
     *         >default retry policy</a>, example timeout settings based on the guidelines above are as follows:
     *         </p>
     *         <p>
     *         Example timeline:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         0-1000 first attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1000-1500 first sleep/delay (default retry policy uses 500 ms as base delay for 4xx errors)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         1500-2500 second attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         2500-3500 second sleep/delay (500 * 2, exponential backoff)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         3500-4500 third attempt
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         4500-6500 third sleep/delay (500 * 2^2)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         6500-7500 fourth attempt (this can trigger inline recovery since 5 seconds have elapsed since the first
     *         attempt reached TC)
     *         </p>
     *         </li>
     * @throws IdempotentParameterMismatchException
     *         DynamoDB rejected the request because you retried a request with a different payload but with an
     *         idempotent token that was already used.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.TransactWriteItems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/TransactWriteItems" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public TransactWriteItemsResponse transactWriteItems(TransactWriteItemsRequest transactWriteItemsRequest)
            throws ResourceNotFoundException, TransactionCanceledException, TransactionInProgressException,
            IdempotentParameterMismatchException, ProvisionedThroughputExceededException, RequestLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TransactWriteItemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, TransactWriteItemsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = transactWriteItemsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(transactWriteItemsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(transactWriteItemsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, transactWriteItemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TransactWriteItems");

            return clientHandler.execute(new ClientExecutionParams<TransactWriteItemsRequest, TransactWriteItemsResponse>()
                    .withOperationName("TransactWriteItems").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(transactWriteItemsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TransactWriteItemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the association of tags from an Amazon DynamoDB resource. You can call <code>UntagResource</code> up to
     * five times per second, per account.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>UntagResource</code> is an asynchronous operation. If you issue a <a>ListTagsOfResource</a> request
     * immediately after an <code>UntagResource</code> request, DynamoDB might return your previous tag set, if there
     * was one, or an empty tag set. This is because <code>ListTagsOfResource</code> uses an eventually consistent
     * query, and the metadata for your tags or table might not be available at that moment. Wait for a few seconds, and
     * then try the <code>ListTagsOfResource</code> request again.
     * </p>
     * </li>
     * <li>
     * <p>
     * The application or removal of tags using <code>TagResource</code> and <code>UntagResource</code> APIs is
     * eventually consistent. <code>ListTagsOfResource</code> API will only reflect the changes after a few seconds.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For an overview on tagging DynamoDB resources, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Tagging.html">Tagging for DynamoDB</a> in
     * the <i>Amazon DynamoDB Developer Guide</i>.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws LimitExceededException,
            ResourceNotFoundException, InternalServerErrorException, ResourceInUseException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = untagResourceRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(untagResourceRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(untagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * <code>UpdateContinuousBackups</code> enables or disables point in time recovery for the specified table. A
     * successful <code>UpdateContinuousBackups</code> call returns the current
     * <code>ContinuousBackupsDescription</code>. Continuous backups are <code>ENABLED</code> on all tables at table
     * creation. If point in time recovery is enabled, <code>PointInTimeRecoveryStatus</code> will be set to ENABLED.
     * </p>
     * <p>
     * Once continuous backups and point in time recovery are enabled, you can restore to any point in time within
     * <code>EarliestRestorableDateTime</code> and <code>LatestRestorableDateTime</code>.
     * </p>
     * <p>
     * <code>LatestRestorableDateTime</code> is typically 5 minutes before the current time. You can restore your table
     * to any point in time in the last 35 days. You can set the <code>RecoveryPeriodInDays</code> to any value between
     * 1 and 35 days.
     * </p>
     *
     * @param updateContinuousBackupsRequest
     * @return Result of the UpdateContinuousBackups operation returned by the service.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws ContinuousBackupsUnavailableException
     *         Backups have not yet been enabled for this table.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateContinuousBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateContinuousBackups"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContinuousBackupsResponse updateContinuousBackups(UpdateContinuousBackupsRequest updateContinuousBackupsRequest)
            throws TableNotFoundException, ContinuousBackupsUnavailableException, InternalServerErrorException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContinuousBackupsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContinuousBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateContinuousBackupsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateContinuousBackupsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContinuousBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContinuousBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContinuousBackups");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateContinuousBackupsRequest, UpdateContinuousBackupsResponse>()
                            .withOperationName("UpdateContinuousBackups").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(updateContinuousBackupsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateContinuousBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the status for contributor insights for a specific table or index. CloudWatch Contributor Insights for
     * DynamoDB graphs display the partition key and (if applicable) sort key of frequently accessed items and
     * frequently throttled items in plaintext. If you require the use of Amazon Web Services Key Management Service
     * (KMS) to encrypt this table’s partition key and sort key data with an Amazon Web Services managed key or customer
     * managed key, you should not enable CloudWatch Contributor Insights for DynamoDB for this table.
     * </p>
     *
     * @param updateContributorInsightsRequest
     * @return Result of the UpdateContributorInsights operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateContributorInsights
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateContributorInsights"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateContributorInsightsResponse updateContributorInsights(
            UpdateContributorInsightsRequest updateContributorInsightsRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateContributorInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateContributorInsightsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateContributorInsightsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateContributorInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateContributorInsights");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateContributorInsightsRequest, UpdateContributorInsightsResponse>()
                            .withOperationName("UpdateContributorInsights").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateContributorInsightsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateContributorInsightsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or removes replicas in the specified global table. The global table must already exist to be able to use
     * this operation. Any replica to be added must be empty, have the same name as the global table, have the same key
     * schema, have DynamoDB Streams enabled, and have the same provisioned and maximum write capacity units.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important> <note>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version). If
     * you are using global tables <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Version 2019.11.21</a>
     * you can use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_UpdateTable.html">UpdateTable</a>
     * instead.
     * </p>
     * <p>
     * Although you can use <code>UpdateGlobalTable</code> to add replicas and remove replicas in a single request, for
     * simplicity we recommend that you issue separate requests for adding or removing replicas.
     * </p>
     * </note>
     * <p>
     * If global secondary indexes are specified, then the following conditions must also be met:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The global secondary indexes must have the same name.
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same hash key and sort key (if present).
     * </p>
     * </li>
     * <li>
     * <p>
     * The global secondary indexes must have the same provisioned and maximum write capacity units.
     * </p>
     * </li>
     * </ul>
     *
     * @param updateGlobalTableRequest
     * @return Result of the UpdateGlobalTable operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws GlobalTableNotFoundException
     *         The specified global table does not exist.
     * @throws ReplicaAlreadyExistsException
     *         The specified replica is already part of the global table.
     * @throws ReplicaNotFoundException
     *         The specified replica is no longer part of the global table.
     * @throws TableNotFoundException
     *         A source table with the name <code>TableName</code> does not currently exist within the subscriber's
     *         account or the subscriber is operating in the wrong Amazon Web Services Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateGlobalTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateGlobalTable" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public UpdateGlobalTableResponse updateGlobalTable(UpdateGlobalTableRequest updateGlobalTableRequest)
            throws InternalServerErrorException, GlobalTableNotFoundException, ReplicaAlreadyExistsException,
            ReplicaNotFoundException, TableNotFoundException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGlobalTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateGlobalTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateGlobalTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateGlobalTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGlobalTableRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalTable");

            return clientHandler.execute(new ClientExecutionParams<UpdateGlobalTableRequest, UpdateGlobalTableResponse>()
                    .withOperationName("UpdateGlobalTable").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(updateGlobalTableRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateGlobalTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates settings for a global table.
     * </p>
     * <important>
     * <p>
     * This documentation is for version 2017.11.29 (Legacy) of global tables, which should be avoided for new global
     * tables. Customers should use <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/GlobalTables.html">Global Tables version
     * 2019.11.21 (Current)</a> when possible, because it provides greater flexibility, higher efficiency, and consumes
     * less write capacity than 2017.11.29 (Legacy).
     * </p>
     * <p>
     * To determine which version you're using, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/globaltables.DetermineVersion.html"
     * >Determining the global table version you are using</a>. To update existing global tables from version 2017.11.29
     * (Legacy) to version 2019.11.21 (Current), see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2globaltables_upgrade.html">Upgrading
     * global tables</a>.
     * </p>
     * </important>
     *
     * @param updateGlobalTableSettingsRequest
     * @return Result of the UpdateGlobalTableSettings operation returned by the service.
     * @throws GlobalTableNotFoundException
     *         The specified global table does not exist.
     * @throws ReplicaNotFoundException
     *         The specified replica is no longer part of the global table.
     * @throws IndexNotFoundException
     *         The operation tried to access a nonexistent index.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateGlobalTableSettings
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateGlobalTableSettings"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateGlobalTableSettingsResponse updateGlobalTableSettings(
            UpdateGlobalTableSettingsRequest updateGlobalTableSettingsRequest) throws GlobalTableNotFoundException,
            ReplicaNotFoundException, IndexNotFoundException, LimitExceededException, ResourceInUseException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateGlobalTableSettingsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateGlobalTableSettingsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateGlobalTableSettingsRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateGlobalTableSettingsRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateGlobalTableSettingsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateGlobalTableSettingsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateGlobalTableSettings");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateGlobalTableSettingsRequest, UpdateGlobalTableSettingsResponse>()
                            .withOperationName("UpdateGlobalTableSettings").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(updateGlobalTableSettingsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateGlobalTableSettingsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Edits an existing item's attributes, or adds a new item to the table if it does not already exist. You can put,
     * delete, or add attribute values. You can also perform a conditional update on an existing item (insert a new
     * attribute name-value pair if it doesn't exist, or replace an existing name-value pair if it has certain expected
     * attribute values).
     * </p>
     * <p>
     * You can also return the item's attribute values in the same <code>UpdateItem</code> operation using the
     * <code>ReturnValues</code> parameter.
     * </p>
     *
     * @param updateItemRequest
     *        Represents the input of an <code>UpdateItem</code> operation.
     * @return Result of the UpdateItem operation returned by the service.
     * @throws ConditionalCheckFailedException
     *         A condition specified in the operation failed to be evaluated.
     * @throws ProvisionedThroughputExceededException
     *         Your request rate is too high. The Amazon Web Services SDKs for DynamoDB automatically retry requests
     *         that receive this exception. Your request is eventually successful, unless your retry queue is too large
     *         to finish. Reduce the frequency of requests and use exponential backoff. For more information, go to <a
     *         href=
     *         "https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Programming.Errors.html#Programming.Errors.RetryAndBackoff"
     *         >Error Retries and Exponential Backoff</a> in the <i>Amazon DynamoDB Developer Guide</i>.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ItemCollectionSizeLimitExceededException
     *         An item collection is too large. This exception is only returned for tables that have one or more local
     *         secondary indexes.
     * @throws TransactionConflictException
     *         Operation was rejected because there is an ongoing transaction for the item.
     * @throws RequestLimitExceededException
     *         Throughput exceeds the current throughput quota for your account. Please contact <a
     *         href="https://aws.amazon.com/support">Amazon Web ServicesSupport</a> to request a quota increase.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ReplicatedWriteConflictException
     *         The request was rejected because one or more items in the request are being modified by a request in
     *         another Region.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateItem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateItem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateItemResponse updateItem(UpdateItemRequest updateItemRequest) throws ConditionalCheckFailedException,
            ProvisionedThroughputExceededException, ResourceNotFoundException, ItemCollectionSizeLimitExceededException,
            TransactionConflictException, RequestLimitExceededException, InternalServerErrorException,
            ReplicatedWriteConflictException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateItemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateItemResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateItemRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateItemRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateItemRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateItemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateItem");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateItemRequest, UpdateItemResponse>().withOperationName("UpdateItem")
                            .withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                            .withRequestConfiguration(clientConfiguration).withInput(updateItemRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateItemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The command to update the Kinesis stream destination.
     * </p>
     *
     * @param updateKinesisStreamingDestinationRequest
     * @return Result of the UpdateKinesisStreamingDestination operation returned by the service.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateKinesisStreamingDestination
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateKinesisStreamingDestination"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateKinesisStreamingDestinationResponse updateKinesisStreamingDestination(
            UpdateKinesisStreamingDestinationRequest updateKinesisStreamingDestinationRequest)
            throws InternalServerErrorException, LimitExceededException, ResourceInUseException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateKinesisStreamingDestinationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateKinesisStreamingDestinationResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateKinesisStreamingDestinationRequest
                    .overrideConfiguration().flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateKinesisStreamingDestinationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateKinesisStreamingDestinationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateKinesisStreamingDestination");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateKinesisStreamingDestinationRequest, UpdateKinesisStreamingDestinationResponse>()
                            .withOperationName("UpdateKinesisStreamingDestination").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                            .withInput(updateKinesisStreamingDestinationRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateKinesisStreamingDestinationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies the provisioned throughput settings, global secondary indexes, or DynamoDB Streams settings for a given
     * table.
     * </p>
     * <important>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version).
     * </p>
     * </important>
     * <p>
     * You can only perform one of the following operations at once:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Modify the provisioned throughput settings of the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Remove a global secondary index from the table.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create a new global secondary index on the table. After the index begins backfilling, you can use
     * <code>UpdateTable</code> to perform other operations.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <code>UpdateTable</code> is an asynchronous operation; while it's executing, the table status changes from
     * <code>ACTIVE</code> to <code>UPDATING</code>. While it's <code>UPDATING</code>, you can't issue another
     * <code>UpdateTable</code> request. When the table returns to the <code>ACTIVE</code> state, the
     * <code>UpdateTable</code> operation is complete.
     * </p>
     *
     * @param updateTableRequest
     *        Represents the input of an <code>UpdateTable</code> operation.
     * @return Result of the UpdateTable operation returned by the service.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateTable
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTable" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTableResponse updateTable(UpdateTableRequest updateTableRequest) throws ResourceInUseException,
            ResourceNotFoundException, LimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTableResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTableResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateTableRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateTableRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTableRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTableRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTable");

            return clientHandler.execute(new ClientExecutionParams<UpdateTableRequest, UpdateTableResponse>()
                    .withOperationName("UpdateTable").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).discoveredEndpoint(cachedEndpoint)
                    .withRequestConfiguration(clientConfiguration).withInput(updateTableRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTableRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates auto scaling settings on your global tables at once.
     * </p>
     * <important>
     * <p>
     * For global tables, this operation only applies to global tables using Version 2019.11.21 (Current version).
     * </p>
     * </important>
     *
     * @param updateTableReplicaAutoScalingRequest
     * @return Result of the UpdateTableReplicaAutoScaling operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateTableReplicaAutoScaling
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTableReplicaAutoScaling"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateTableReplicaAutoScalingResponse updateTableReplicaAutoScaling(
            UpdateTableReplicaAutoScalingRequest updateTableReplicaAutoScalingRequest) throws ResourceNotFoundException,
            ResourceInUseException, LimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTableReplicaAutoScalingResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateTableReplicaAutoScalingResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTableReplicaAutoScalingRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateTableReplicaAutoScalingRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTableReplicaAutoScaling");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateTableReplicaAutoScalingRequest, UpdateTableReplicaAutoScalingResponse>()
                            .withOperationName("UpdateTableReplicaAutoScaling").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateTableReplicaAutoScalingRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateTableReplicaAutoScalingRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * The <code>UpdateTimeToLive</code> method enables or disables Time to Live (TTL) for the specified table. A
     * successful <code>UpdateTimeToLive</code> call returns the current <code>TimeToLiveSpecification</code>. It can
     * take up to one hour for the change to fully process. Any additional <code>UpdateTimeToLive</code> calls for the
     * same table during this one hour duration result in a <code>ValidationException</code>.
     * </p>
     * <p>
     * TTL compares the current time in epoch time format to the time stored in the TTL attribute of an item. If the
     * epoch time value stored in the attribute is less than the current time, the item is marked as expired and
     * subsequently deleted.
     * </p>
     * <note>
     * <p>
     * The epoch time format is the number of seconds elapsed since 12:00:00 AM January 1, 1970 UTC.
     * </p>
     * </note>
     * <p>
     * DynamoDB deletes expired items on a best-effort basis to ensure availability of throughput for other data
     * operations.
     * </p>
     * <important>
     * <p>
     * DynamoDB typically deletes expired items within two days of expiration. The exact duration within which an item
     * gets deleted after expiration is specific to the nature of the workload. Items that have expired and not been
     * deleted will still show up in reads, queries, and scans.
     * </p>
     * </important>
     * <p>
     * As items are deleted, they are removed from any local secondary index and global secondary index immediately in
     * the same eventually consistent way as a standard delete operation.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/TTL.html">Time To Live</a> in the Amazon
     * DynamoDB Developer Guide.
     * </p>
     *
     * @param updateTimeToLiveRequest
     *        Represents the input of an <code>UpdateTimeToLive</code> operation.
     * @return Result of the UpdateTimeToLive operation returned by the service.
     * @throws ResourceInUseException
     *         The operation conflicts with the resource's availability. For example:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         You attempted to recreate an existing table.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to delete a table currently in the <code>CREATING</code> state.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You tried to update a resource that was already being updated.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         When appropriate, wait for the ongoing update to complete and attempt the request again.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken.
     *         </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbClient.UpdateTimeToLive
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/dynamodb-2012-08-10/UpdateTimeToLive" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateTimeToLiveResponse updateTimeToLive(UpdateTimeToLiveRequest updateTimeToLiveRequest)
            throws ResourceInUseException, ResourceNotFoundException, LimitExceededException, InternalServerErrorException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateTimeToLiveResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateTimeToLiveResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        boolean endpointDiscoveryEnabled = clientConfiguration.option(SdkClientOption.ENDPOINT_DISCOVERY_ENABLED);
        boolean endpointOverridden = clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).isEndpointOverridden();
        URI cachedEndpoint = null;
        if (endpointDiscoveryEnabled) {
            CompletableFuture<? extends AwsCredentialsIdentity> identityFuture = updateTimeToLiveRequest.overrideConfiguration()
                    .flatMap(AwsRequestOverrideConfiguration::credentialsIdentityProvider)
                    .orElseGet(() -> clientConfiguration.option(AwsClientOption.CREDENTIALS_IDENTITY_PROVIDER)).resolveIdentity();
            String key = CompletableFutureUtils.joinLikeSync(identityFuture).accessKeyId();
            EndpointDiscoveryRequest endpointDiscoveryRequest = EndpointDiscoveryRequest.builder().required(false)
                    .defaultEndpoint(clientConfiguration.option(SdkClientOption.CLIENT_ENDPOINT_PROVIDER).clientEndpoint())
                    .overrideConfiguration(updateTimeToLiveRequest.overrideConfiguration().orElse(null)).build();
            cachedEndpoint = endpointDiscoveryCache.get(key, endpointDiscoveryRequest);
        }
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateTimeToLiveRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateTimeToLiveRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateTimeToLive");

            return clientHandler.execute(new ClientExecutionParams<UpdateTimeToLiveRequest, UpdateTimeToLiveResponse>()
                    .withOperationName("UpdateTimeToLive").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .discoveredEndpoint(cachedEndpoint).withRequestConfiguration(clientConfiguration)
                    .withInput(updateTimeToLiveRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateTimeToLiveRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * Create an instance of {@link DynamoDbWaiter} using this client.
     * <p>
     * Waiters created via this method are managed by the SDK and resources will be released when the service client is
     * closed.
     *
     * @return an instance of {@link DynamoDbWaiter}
     */
    @Override
    public DynamoDbWaiter waiter() {
        return DynamoDbWaiter.builder().client(this).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        if (plugins.isEmpty()) {
            return configuration.build();
        }
        DynamoDbServiceClientConfigurationBuilder serviceConfigBuilder = new DynamoDbServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DynamoDbException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestLimitExceeded")
                                .exceptionBuilderSupplier(RequestLimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlobalTableAlreadyExistsException")
                                .exceptionBuilderSupplier(GlobalTableAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImportConflictException")
                                .exceptionBuilderSupplier(ImportConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConditionalCheckFailedException")
                                .exceptionBuilderSupplier(ConditionalCheckFailedException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PolicyNotFoundException")
                                .exceptionBuilderSupplier(PolicyNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("GlobalTableNotFoundException")
                                .exceptionBuilderSupplier(GlobalTableNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ItemCollectionSizeLimitExceededException")
                                .exceptionBuilderSupplier(ItemCollectionSizeLimitExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicaNotFoundException")
                                .exceptionBuilderSupplier(ReplicaNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupInUseException")
                                .exceptionBuilderSupplier(BackupInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ContinuousBackupsUnavailableException")
                                .exceptionBuilderSupplier(ContinuousBackupsUnavailableException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IdempotentParameterMismatchException")
                                .exceptionBuilderSupplier(IdempotentParameterMismatchException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExportNotFoundException")
                                .exceptionBuilderSupplier(ExportNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionInProgressException")
                                .exceptionBuilderSupplier(TransactionInProgressException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableInUseException")
                                .exceptionBuilderSupplier(TableInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ProvisionedThroughputExceededException")
                                .exceptionBuilderSupplier(ProvisionedThroughputExceededException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PointInTimeRecoveryUnavailableException")
                                .exceptionBuilderSupplier(PointInTimeRecoveryUnavailableException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceInUseException")
                                .exceptionBuilderSupplier(ResourceInUseException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableAlreadyExistsException")
                                .exceptionBuilderSupplier(TableAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ExportConflictException")
                                .exceptionBuilderSupplier(ExportConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionConflictException")
                                .exceptionBuilderSupplier(TransactionConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRestoreTimeException")
                                .exceptionBuilderSupplier(InvalidRestoreTimeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicaAlreadyExistsException")
                                .exceptionBuilderSupplier(ReplicaAlreadyExistsException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ReplicatedWriteConflictException")
                                .exceptionBuilderSupplier(ReplicatedWriteConflictException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("BackupNotFoundException")
                                .exceptionBuilderSupplier(BackupNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("IndexNotFoundException")
                                .exceptionBuilderSupplier(IndexNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TableNotFoundException")
                                .exceptionBuilderSupplier(TableNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("DuplicateItemException")
                                .exceptionBuilderSupplier(DuplicateItemException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ImportNotFoundException")
                                .exceptionBuilderSupplier(ImportNotFoundException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TransactionCanceledException")
                                .exceptionBuilderSupplier(TransactionCanceledException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidExportTimeException")
                                .exceptionBuilderSupplier(InvalidExportTimeException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerError")
                                .exceptionBuilderSupplier(InternalServerErrorException::builder).httpStatusCode(500).build());
    }

    @Override
    public final DynamoDbServiceClientConfiguration serviceClientConfiguration() {
        return new DynamoDbServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
