/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for DeleteTags.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeleteTagsRequest extends Ec2Request implements
        ToCopyableBuilder<DeleteTagsRequest.Builder, DeleteTagsRequest> {
    private static final SdkField<List<String>> RESOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DeleteTagsRequest::resources))
            .setter(setter(Builder::resources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("resourceId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(DeleteTagsRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tag").unmarshallLocationName("tag")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCES_FIELD, TAGS_FIELD));

    private final List<String> resources;

    private final List<Tag> tags;

    private DeleteTagsRequest(BuilderImpl builder) {
        super(builder);
        this.resources = builder.resources;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The IDs of one or more resources.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The IDs of one or more resources.
     */
    public List<String> resources() {
        return resources;
    }

    /**
     * <p>
     * One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If you specify a
     * tag key without a tag value, we delete any tag with this key regardless of its value. If you specify a tag key
     * with an empty string as the tag value, we delete the tag only if its value is an empty string.
     * </p>
     * <p>
     * If you omit this parameter, we delete all user-defined tags for the specified resources. We do not delete
     * AWS-generated tags (tags that have the <code>aws:</code> prefix).
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If you
     *         specify a tag key without a tag value, we delete any tag with this key regardless of its value. If you
     *         specify a tag key with an empty string as the tag value, we delete the tag only if its value is an empty
     *         string.</p>
     *         <p>
     *         If you omit this parameter, we delete all user-defined tags for the specified resources. We do not delete
     *         AWS-generated tags (tags that have the <code>aws:</code> prefix).
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resources());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeleteTagsRequest)) {
            return false;
        }
        DeleteTagsRequest other = (DeleteTagsRequest) obj;
        return Objects.equals(resources(), other.resources()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        return ToString.builder("DeleteTagsRequest").add("Resources", resources()).add("Tags", tags()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Resources":
            return Optional.ofNullable(clazz.cast(resources()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DeleteTagsRequest, T> g) {
        return obj -> g.apply((DeleteTagsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, DeleteTagsRequest> {
        /**
         * <p>
         * The IDs of one or more resources.
         * </p>
         * 
         * @param resources
         *        The IDs of one or more resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(Collection<String> resources);

        /**
         * <p>
         * The IDs of one or more resources.
         * </p>
         * 
         * @param resources
         *        The IDs of one or more resources.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resources(String... resources);

        /**
         * <p>
         * One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If you
         * specify a tag key without a tag value, we delete any tag with this key regardless of its value. If you
         * specify a tag key with an empty string as the tag value, we delete the tag only if its value is an empty
         * string.
         * </p>
         * <p>
         * If you omit this parameter, we delete all user-defined tags for the specified resources. We do not delete
         * AWS-generated tags (tags that have the <code>aws:</code> prefix).
         * </p>
         * 
         * @param tags
         *        One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If
         *        you specify a tag key without a tag value, we delete any tag with this key regardless of its value. If
         *        you specify a tag key with an empty string as the tag value, we delete the tag only if its value is an
         *        empty string.</p>
         *        <p>
         *        If you omit this parameter, we delete all user-defined tags for the specified resources. We do not
         *        delete AWS-generated tags (tags that have the <code>aws:</code> prefix).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If you
         * specify a tag key without a tag value, we delete any tag with this key regardless of its value. If you
         * specify a tag key with an empty string as the tag value, we delete the tag only if its value is an empty
         * string.
         * </p>
         * <p>
         * If you omit this parameter, we delete all user-defined tags for the specified resources. We do not delete
         * AWS-generated tags (tags that have the <code>aws:</code> prefix).
         * </p>
         * 
         * @param tags
         *        One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If
         *        you specify a tag key without a tag value, we delete any tag with this key regardless of its value. If
         *        you specify a tag key with an empty string as the tag value, we delete the tag only if its value is an
         *        empty string.</p>
         *        <p>
         *        If you omit this parameter, we delete all user-defined tags for the specified resources. We do not
         *        delete AWS-generated tags (tags that have the <code>aws:</code> prefix).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * One or more tags to delete. Specify a tag key and an optional tag value to delete specific tags. If you
         * specify a tag key without a tag value, we delete any tag with this key regardless of its value. If you
         * specify a tag key with an empty string as the tag value, we delete the tag only if its value is an empty
         * string.
         * </p>
         * <p>
         * If you omit this parameter, we delete all user-defined tags for the specified resources. We do not delete
         * AWS-generated tags (tags that have the <code>aws:</code> prefix).
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private List<String> resources = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(DeleteTagsRequest model) {
            super(model);
            resources(model.resources);
            tags(model.tags);
        }

        public final Collection<String> getResources() {
            return resources;
        }

        @Override
        public final Builder resources(Collection<String> resources) {
            this.resources = ResourceIdListCopier.copy(resources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resources(String... resources) {
            resources(Arrays.asList(resources));
            return this;
        }

        public final void setResources(Collection<String> resources) {
            this.resources = ResourceIdListCopier.copy(resources);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DeleteTagsRequest build() {
            return new DeleteTagsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
