/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a block device for an EBS volume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsBlockDevice implements SdkPojo, Serializable, ToCopyableBuilder<EbsBlockDevice.Builder, EbsBlockDevice> {
    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EbsBlockDevice::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<Boolean> DELETE_ON_TERMINATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(EbsBlockDevice::deleteOnTermination))
            .setter(setter(Builder::deleteOnTermination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteOnTermination")
                    .unmarshallLocationName("deleteOnTermination").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EbsBlockDevice::iops))
            .setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops")
                    .unmarshallLocationName("iops").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(EbsBlockDevice::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("KmsKeyId").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(EbsBlockDevice::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("snapshotId").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(EbsBlockDevice::volumeSize))
            .setter(setter(Builder::volumeSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSize")
                    .unmarshallLocationName("volumeSize").build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(EbsBlockDevice::volumeTypeAsString))
            .setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeType")
                    .unmarshallLocationName("volumeType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENCRYPTED_FIELD,
            DELETE_ON_TERMINATION_FIELD, IOPS_FIELD, KMS_KEY_ID_FIELD, SNAPSHOT_ID_FIELD, VOLUME_SIZE_FIELD, VOLUME_TYPE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean encrypted;

    private final Boolean deleteOnTermination;

    private final Integer iops;

    private final String kmsKeyId;

    private final String snapshotId;

    private final Integer volumeSize;

    private final String volumeType;

    private EbsBlockDevice(BuilderImpl builder) {
        this.encrypted = builder.encrypted;
        this.deleteOnTermination = builder.deleteOnTermination;
        this.iops = builder.iops;
        this.kmsKeyId = builder.kmsKeyId;
        this.snapshotId = builder.snapshotId;
        this.volumeSize = builder.volumeSize;
        this.volumeType = builder.volumeType;
    }

    /**
     * <p>
     * Indicates whether the EBS volume is encrypted. Encrypted volumes can only be attached to instances that support
     * Amazon EBS encryption. If you are creating a volume from a snapshot, you can't specify an encryption value. This
     * is because only blank volumes can be encrypted on creation.
     * </p>
     * 
     * @return Indicates whether the EBS volume is encrypted. Encrypted volumes can only be attached to instances that
     *         support Amazon EBS encryption. If you are creating a volume from a snapshot, you can't specify an
     *         encryption value. This is because only blank volumes can be encrypted on creation.
     */
    public Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * Indicates whether the EBS volume is deleted on instance termination.
     * </p>
     * 
     * @return Indicates whether the EBS volume is deleted on instance termination.
     */
    public Boolean deleteOnTermination() {
        return deleteOnTermination;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS) that the volume supports. For <code>io1</code>, this represents
     * the number of IOPS that are provisioned for the volume. For <code>gp2</code>, this represents the baseline
     * performance of the volume and the rate at which the volume accumulates I/O credits for bursting. For more
     * information about General Purpose SSD baseline performance, I/O credits, and bursting, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in the
     * <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * Constraint: Range is 100-20000 IOPS for <code>io1</code> volumes and 100-10000 IOPS for <code>gp2</code> volumes.
     * </p>
     * <p>
     * Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in requests
     * to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS) that the volume supports. For <code>io1</code>, this
     *         represents the number of IOPS that are provisioned for the volume. For <code>gp2</code>, this represents
     *         the baseline performance of the volume and the rate at which the volume accumulates I/O credits for
     *         bursting. For more information about General Purpose SSD baseline performance, I/O credits, and bursting,
     *         see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
     *         Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
     *         <p>
     *         Constraint: Range is 100-20000 IOPS for <code>io1</code> volumes and 100-10000 IOPS for <code>gp2</code>
     *         volumes.
     *         </p>
     *         <p>
     *         Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in
     *         requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
     *         volumes.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * Identifier (key ID, key alias, ID ARN, or alias ARN) for a user-managed CMK under which the EBS volume is
     * encrypted.
     * </p>
     * <p>
     * This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>, and
     * <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">RequestSpotInstances
     * </a>.
     * </p>
     * 
     * @return Identifier (key ID, key alias, ID ARN, or alias ARN) for a user-managed CMK under which the EBS volume is
     *         encrypted.</p>
     *         <p>
     *         This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
     *         href
     *         ="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>,
     *         and <a href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">
     *         RequestSpotInstances</a>.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The ID of the snapshot.
     * </p>
     * 
     * @return The ID of the snapshot.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The size of the volume, in GiB.
     * </p>
     * <p>
     * Constraints: 1-16384 for General Purpose SSD (<code>gp2</code>), 4-16384 for Provisioned IOPS SSD (
     * <code>io1</code>), 500-16384 for Throughput Optimized HDD (<code>st1</code>), 500-16384 for Cold HDD (
     * <code>sc1</code>), and 1-1024 for Magnetic (<code>standard</code>) volumes. If you specify a snapshot, the volume
     * size must be equal to or larger than the snapshot size.
     * </p>
     * <p>
     * Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is the
     * snapshot size.
     * </p>
     * 
     * @return The size of the volume, in GiB.</p>
     *         <p>
     *         Constraints: 1-16384 for General Purpose SSD (<code>gp2</code>), 4-16384 for Provisioned IOPS SSD (
     *         <code>io1</code>), 500-16384 for Throughput Optimized HDD (<code>st1</code>), 500-16384 for Cold HDD (
     *         <code>sc1</code>), and 1-1024 for Magnetic (<code>standard</code>) volumes. If you specify a snapshot,
     *         the volume size must be equal to or larger than the snapshot size.
     *         </p>
     *         <p>
     *         Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is
     *         the snapshot size.
     */
    public Integer volumeSize() {
        return volumeSize;
    }

    /**
     * <p>
     * The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
     * .
     * </p>
     * <p>
     * Default: <code>standard</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
     *         <code>standard</code>.</p>
     *         <p>
     *         Default: <code>standard</code>
     * @see VolumeType
     */
    public VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
     * .
     * </p>
     * <p>
     * Default: <code>standard</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
     *         <code>standard</code>.</p>
     *         <p>
     *         Default: <code>standard</code>
     * @see VolumeType
     */
    public String volumeTypeAsString() {
        return volumeType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(deleteOnTermination());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSize());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsBlockDevice)) {
            return false;
        }
        EbsBlockDevice other = (EbsBlockDevice) obj;
        return Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(deleteOnTermination(), other.deleteOnTermination()) && Objects.equals(iops(), other.iops())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(volumeSize(), other.volumeSize())
                && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString());
    }

    @Override
    public String toString() {
        return ToString.builder("EbsBlockDevice").add("Encrypted", encrypted()).add("DeleteOnTermination", deleteOnTermination())
                .add("Iops", iops()).add("KmsKeyId", kmsKeyId()).add("SnapshotId", snapshotId()).add("VolumeSize", volumeSize())
                .add("VolumeType", volumeTypeAsString()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "DeleteOnTermination":
            return Optional.ofNullable(clazz.cast(deleteOnTermination()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "VolumeSize":
            return Optional.ofNullable(clazz.cast(volumeSize()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EbsBlockDevice, T> g) {
        return obj -> g.apply((EbsBlockDevice) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EbsBlockDevice> {
        /**
         * <p>
         * Indicates whether the EBS volume is encrypted. Encrypted volumes can only be attached to instances that
         * support Amazon EBS encryption. If you are creating a volume from a snapshot, you can't specify an encryption
         * value. This is because only blank volumes can be encrypted on creation.
         * </p>
         * 
         * @param encrypted
         *        Indicates whether the EBS volume is encrypted. Encrypted volumes can only be attached to instances
         *        that support Amazon EBS encryption. If you are creating a volume from a snapshot, you can't specify an
         *        encryption value. This is because only blank volumes can be encrypted on creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * Indicates whether the EBS volume is deleted on instance termination.
         * </p>
         * 
         * @param deleteOnTermination
         *        Indicates whether the EBS volume is deleted on instance termination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteOnTermination(Boolean deleteOnTermination);

        /**
         * <p>
         * The number of I/O operations per second (IOPS) that the volume supports. For <code>io1</code>, this
         * represents the number of IOPS that are provisioned for the volume. For <code>gp2</code>, this represents the
         * baseline performance of the volume and the rate at which the volume accumulates I/O credits for bursting. For
         * more information about General Purpose SSD baseline performance, I/O credits, and bursting, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in
         * the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * <p>
         * Constraint: Range is 100-20000 IOPS for <code>io1</code> volumes and 100-10000 IOPS for <code>gp2</code>
         * volumes.
         * </p>
         * <p>
         * Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used in
         * requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS) that the volume supports. For <code>io1</code>, this
         *        represents the number of IOPS that are provisioned for the volume. For <code>gp2</code>, this
         *        represents the baseline performance of the volume and the rate at which the volume accumulates I/O
         *        credits for bursting. For more information about General Purpose SSD baseline performance, I/O
         *        credits, and bursting, see <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
         *        Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         *        <p>
         *        Constraint: Range is 100-20000 IOPS for <code>io1</code> volumes and 100-10000 IOPS for
         *        <code>gp2</code> volumes.
         *        </p>
         *        <p>
         *        Condition: This parameter is required for requests to create <code>io1</code> volumes; it is not used
         *        in requests to create <code>gp2</code>, <code>st1</code>, <code>sc1</code>, or <code>standard</code>
         *        volumes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * Identifier (key ID, key alias, ID ARN, or alias ARN) for a user-managed CMK under which the EBS volume is
         * encrypted.
         * </p>
         * <p>
         * This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>,
         * and <a href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">
         * RequestSpotInstances</a>.
         * </p>
         * 
         * @param kmsKeyId
         *        Identifier (key ID, key alias, ID ARN, or alias ARN) for a user-managed CMK under which the EBS volume
         *        is encrypted.</p>
         *        <p>
         *        This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>,
         *        <a href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">
         *        RequestSpotFleet</a>, and <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html"
         *        >RequestSpotInstances</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The ID of the snapshot.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The size of the volume, in GiB.
         * </p>
         * <p>
         * Constraints: 1-16384 for General Purpose SSD (<code>gp2</code>), 4-16384 for Provisioned IOPS SSD (
         * <code>io1</code>), 500-16384 for Throughput Optimized HDD (<code>st1</code>), 500-16384 for Cold HDD (
         * <code>sc1</code>), and 1-1024 for Magnetic (<code>standard</code>) volumes. If you specify a snapshot, the
         * volume size must be equal to or larger than the snapshot size.
         * </p>
         * <p>
         * Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is the
         * snapshot size.
         * </p>
         * 
         * @param volumeSize
         *        The size of the volume, in GiB.</p>
         *        <p>
         *        Constraints: 1-16384 for General Purpose SSD (<code>gp2</code>), 4-16384 for Provisioned IOPS SSD (
         *        <code>io1</code>), 500-16384 for Throughput Optimized HDD (<code>st1</code>), 500-16384 for Cold HDD (
         *        <code>sc1</code>), and 1-1024 for Magnetic (<code>standard</code>) volumes. If you specify a snapshot,
         *        the volume size must be equal to or larger than the snapshot size.
         *        </p>
         *        <p>
         *        Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is
         *        the snapshot size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSize(Integer volumeSize);

        /**
         * <p>
         * The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
         * <code>standard</code>.
         * </p>
         * <p>
         * Default: <code>standard</code>
         * </p>
         * 
         * @param volumeType
         *        The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
         *        <code>standard</code>.</p>
         *        <p>
         *        Default: <code>standard</code>
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
         * <code>standard</code>.
         * </p>
         * <p>
         * Default: <code>standard</code>
         * </p>
         * 
         * @param volumeType
         *        The volume type: <code>gp2</code>, <code>io1</code>, <code>st1</code>, <code>sc1</code>, or
         *        <code>standard</code>.</p>
         *        <p>
         *        Default: <code>standard</code>
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);
    }

    static final class BuilderImpl implements Builder {
        private Boolean encrypted;

        private Boolean deleteOnTermination;

        private Integer iops;

        private String kmsKeyId;

        private String snapshotId;

        private Integer volumeSize;

        private String volumeType;

        private BuilderImpl() {
        }

        private BuilderImpl(EbsBlockDevice model) {
            encrypted(model.encrypted);
            deleteOnTermination(model.deleteOnTermination);
            iops(model.iops);
            kmsKeyId(model.kmsKeyId);
            snapshotId(model.snapshotId);
            volumeSize(model.volumeSize);
            volumeType(model.volumeType);
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        public final Boolean getDeleteOnTermination() {
            return deleteOnTermination;
        }

        @Override
        public final Builder deleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
            return this;
        }

        public final void setDeleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final Integer getVolumeSize() {
            return volumeSize;
        }

        @Override
        public final Builder volumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
            return this;
        }

        public final void setVolumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
        }

        public final String getVolumeType() {
            return volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType.toString());
            return this;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        @Override
        public EbsBlockDevice build() {
            return new EbsBlockDevice(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
