/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.paginators;

import java.util.Iterator;
import java.util.function.Function;
import javax.annotation.Generated;
import software.amazon.awssdk.core.pagination.NextPageFetcher;
import software.amazon.awssdk.core.pagination.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.SdkIterable;
import software.amazon.awssdk.services.ec2.EC2Client;
import software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsResponse;
import software.amazon.awssdk.services.ec2.model.SpotFleetRequestConfig;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ec2.EC2Client#describeSpotFleetRequestsIterable(software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeSpotFleetRequestsPaginator responses = client.describeSpotFleetRequestsIterable(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.ec2.paginators.DescribeSpotFleetRequestsPaginator responses = client
 *             .describeSpotFleetRequestsIterable(request);
 *     for (software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeSpotFleetRequestsPaginator responses = client.describeSpotFleetRequestsIterable(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeSpotFleetRequests(software.amazon.awssdk.services.ec2.model.DescribeSpotFleetRequestsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeSpotFleetRequestsPaginator implements SdkIterable<DescribeSpotFleetRequestsResponse> {
    private final EC2Client client;

    private final DescribeSpotFleetRequestsRequest firstRequest;

    private final NextPageFetcher nextPageFetcher;

    public DescribeSpotFleetRequestsPaginator(final EC2Client client, final DescribeSpotFleetRequestsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new DescribeSpotFleetRequestsResponseFetcher();
    }

    @Override
    public Iterator<DescribeSpotFleetRequestsResponse> iterator() {
        return new PaginatedResponsesIterator(nextPageFetcher);
    }

    /**
     * Returns an iterable to iterate through the paginated
     * {@link DescribeSpotFleetRequestsResponse#spotFleetRequestConfigs()} member. The returned iterable is used to
     * iterate through the results across all response pages and not a single page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public SdkIterable<SpotFleetRequestConfig> spotFleetRequestConfigs() {
        Function<DescribeSpotFleetRequestsResponse, Iterator<SpotFleetRequestConfig>> getIterator = response -> response != null ? response
                .spotFleetRequestConfigs().iterator() : null;

        return new PaginatedItemsIterable(this, getIterator);
    }

    private class DescribeSpotFleetRequestsResponseFetcher implements NextPageFetcher<DescribeSpotFleetRequestsResponse> {
        @Override
        public boolean hasNextPage(DescribeSpotFleetRequestsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public DescribeSpotFleetRequestsResponse nextPage(DescribeSpotFleetRequestsResponse previousPage) {
            if (previousPage == null) {
                return client.describeSpotFleetRequests(firstRequest);
            }
            return client.describeSpotFleetRequests(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
