/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.core.AwsRequestOverrideConfig;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for ImportInstance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ImportInstanceRequest extends EC2Request implements
        ToCopyableBuilder<ImportInstanceRequest.Builder, ImportInstanceRequest> {
    private final String description;

    private final List<DiskImage> diskImages;

    private final ImportInstanceLaunchSpecification launchSpecification;

    private final String platform;

    private ImportInstanceRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.diskImages = builder.diskImages;
        this.launchSpecification = builder.launchSpecification;
        this.platform = builder.platform;
    }

    /**
     * <p>
     * A description for the instance being imported.
     * </p>
     * 
     * @return A description for the instance being imported.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The disk image.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The disk image.
     */
    public List<DiskImage> diskImages() {
        return diskImages;
    }

    /**
     * <p>
     * The launch specification.
     * </p>
     * 
     * @return The launch specification.
     */
    public ImportInstanceLaunchSpecification launchSpecification() {
        return launchSpecification;
    }

    /**
     * <p>
     * The instance operating system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformString}.
     * </p>
     * 
     * @return The instance operating system.
     * @see PlatformValues
     */
    public PlatformValues platform() {
        return PlatformValues.fromValue(platform);
    }

    /**
     * <p>
     * The instance operating system.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #platform} will
     * return {@link PlatformValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #platformString}.
     * </p>
     * 
     * @return The instance operating system.
     * @see PlatformValues
     */
    public String platformString() {
        return platform;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(diskImages());
        hashCode = 31 * hashCode + Objects.hashCode(launchSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(platformString());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportInstanceRequest)) {
            return false;
        }
        ImportInstanceRequest other = (ImportInstanceRequest) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(diskImages(), other.diskImages())
                && Objects.equals(launchSpecification(), other.launchSpecification())
                && Objects.equals(platformString(), other.platformString());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (description() != null) {
            sb.append("Description: ").append(description()).append(",");
        }
        if (diskImages() != null) {
            sb.append("DiskImages: ").append(diskImages()).append(",");
        }
        if (launchSpecification() != null) {
            sb.append("LaunchSpecification: ").append(launchSpecification()).append(",");
        }
        if (platformString() != null) {
            sb.append("Platform: ").append(platformString()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.of(clazz.cast(description()));
        case "DiskImages":
            return Optional.of(clazz.cast(diskImages()));
        case "LaunchSpecification":
            return Optional.of(clazz.cast(launchSpecification()));
        case "Platform":
            return Optional.of(clazz.cast(platformString()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends EC2Request.Builder, CopyableBuilder<Builder, ImportInstanceRequest> {
        /**
         * <p>
         * A description for the instance being imported.
         * </p>
         * 
         * @param description
         *        A description for the instance being imported.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The disk image.
         * </p>
         * 
         * @param diskImages
         *        The disk image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskImages(Collection<DiskImage> diskImages);

        /**
         * <p>
         * The disk image.
         * </p>
         * 
         * @param diskImages
         *        The disk image.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder diskImages(DiskImage... diskImages);

        /**
         * <p>
         * The launch specification.
         * </p>
         * 
         * @param launchSpecification
         *        The launch specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchSpecification(ImportInstanceLaunchSpecification launchSpecification);

        /**
         * <p>
         * The launch specification.
         * </p>
         * This is a convenience that creates an instance of the {@link ImportInstanceLaunchSpecification.Builder}
         * avoiding the need to create one manually via {@link ImportInstanceLaunchSpecification#builder()}.
         *
         * When the {@link Consumer} completes, {@link ImportInstanceLaunchSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #launchSpecification(ImportInstanceLaunchSpecification)}.
         * 
         * @param launchSpecification
         *        a consumer that will call methods on {@link ImportInstanceLaunchSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchSpecification(ImportInstanceLaunchSpecification)
         */
        default Builder launchSpecification(Consumer<ImportInstanceLaunchSpecification.Builder> launchSpecification) {
            return launchSpecification(ImportInstanceLaunchSpecification.builder().apply(launchSpecification).build());
        }

        /**
         * <p>
         * The instance operating system.
         * </p>
         * 
         * @param platform
         *        The instance operating system.
         * @see PlatformValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformValues
         */
        Builder platform(String platform);

        /**
         * <p>
         * The instance operating system.
         * </p>
         * 
         * @param platform
         *        The instance operating system.
         * @see PlatformValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PlatformValues
         */
        Builder platform(PlatformValues platform);

        @Override
        Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig);
    }

    static final class BuilderImpl extends EC2Request.BuilderImpl implements Builder {
        private String description;

        private List<DiskImage> diskImages;

        private ImportInstanceLaunchSpecification launchSpecification;

        private String platform;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportInstanceRequest model) {
            description(model.description);
            diskImages(model.diskImages);
            launchSpecification(model.launchSpecification);
            platform(model.platform);
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<DiskImage.Builder> getDiskImages() {
            return diskImages != null ? diskImages.stream().map(DiskImage::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder diskImages(Collection<DiskImage> diskImages) {
            this.diskImages = DiskImageListCopier.copy(diskImages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder diskImages(DiskImage... diskImages) {
            diskImages(Arrays.asList(diskImages));
            return this;
        }

        public final void setDiskImages(Collection<DiskImage.BuilderImpl> diskImages) {
            this.diskImages = DiskImageListCopier.copyFromBuilder(diskImages);
        }

        public final ImportInstanceLaunchSpecification.Builder getLaunchSpecification() {
            return launchSpecification != null ? launchSpecification.toBuilder() : null;
        }

        @Override
        public final Builder launchSpecification(ImportInstanceLaunchSpecification launchSpecification) {
            this.launchSpecification = launchSpecification;
            return this;
        }

        public final void setLaunchSpecification(ImportInstanceLaunchSpecification.BuilderImpl launchSpecification) {
            this.launchSpecification = launchSpecification != null ? launchSpecification.build() : null;
        }

        public final String getPlatform() {
            return platform;
        }

        @Override
        public final Builder platform(String platform) {
            this.platform = platform;
            return this;
        }

        @Override
        public final Builder platform(PlatformValues platform) {
            this.platform(platform.toString());
            return this;
        }

        public final void setPlatform(String platform) {
            this.platform = platform;
        }

        @Override
        public Builder requestOverrideConfig(AwsRequestOverrideConfig awsRequestOverrideConfig) {
            super.requestOverrideConfig(awsRequestOverrideConfig);
            return this;
        }

        @Override
        public Builder requestOverrideConfig(Consumer<AwsRequestOverrideConfig.Builder> builderConsumer) {
            super.requestOverrideConfig(builderConsumer);
            return this;
        }

        @Override
        public ImportInstanceRequest build() {
            return new ImportInstanceRequest(this);
        }
    }
}
