/*
 * Copyright 2012-2017 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.annotation.Generated;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a NAT gateway.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class NatGateway implements ToCopyableBuilder<NatGateway.Builder, NatGateway> {
    private final Instant createTime;

    private final Instant deleteTime;

    private final String failureCode;

    private final String failureMessage;

    private final List<NatGatewayAddress> natGatewayAddresses;

    private final String natGatewayId;

    private final ProvisionedBandwidth provisionedBandwidth;

    private final String state;

    private final String subnetId;

    private final String vpcId;

    private final List<Tag> tags;

    private NatGateway(BuilderImpl builder) {
        this.createTime = builder.createTime;
        this.deleteTime = builder.deleteTime;
        this.failureCode = builder.failureCode;
        this.failureMessage = builder.failureMessage;
        this.natGatewayAddresses = builder.natGatewayAddresses;
        this.natGatewayId = builder.natGatewayId;
        this.provisionedBandwidth = builder.provisionedBandwidth;
        this.state = builder.state;
        this.subnetId = builder.subnetId;
        this.vpcId = builder.vpcId;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The date and time the NAT gateway was created.
     * </p>
     * 
     * @return The date and time the NAT gateway was created.
     */
    public Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * The date and time the NAT gateway was deleted, if applicable.
     * </p>
     * 
     * @return The date and time the NAT gateway was deleted, if applicable.
     */
    public Instant deleteTime() {
        return deleteTime;
    }

    /**
     * <p>
     * If the NAT gateway could not be created, specifies the error code for the failure. (
     * <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
     * <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> | <code>InternalError</code>
     * | <code>InvalidSubnetID.NotFound</code>)
     * </p>
     * 
     * @return If the NAT gateway could not be created, specifies the error code for the failure. (
     *         <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
     *         <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
     *         <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
     */
    public String failureCode() {
        return failureCode;
    }

    /**
     * <p>
     * If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to the
     * error code.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For InsufficientFreeAddressesInSubnet: "Subnet has insufficient free addresses to create this NAT gateway"
     * </p>
     * </li>
     * <li>
     * <p>
     * For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
     * </p>
     * </li>
     * <li>
     * <p>
     * For InvalidAllocationID.NotFound:
     * "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
     * </p>
     * </li>
     * <li>
     * <p>
     * For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
     * </p>
     * </li>
     * <li>
     * <p>
     * For InternalError:
     * "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
     * </p>
     * </li>
     * <li>
     * <p>
     * For InvalidSubnetID.NotFound: "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to
     *         the error code.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For InsufficientFreeAddressesInSubnet:
     *         "Subnet has insufficient free addresses to create this NAT gateway"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InvalidAllocationID.NotFound:
     *         "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InternalError:
     *         "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For InvalidSubnetID.NotFound:
     *         "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
     *         </p>
     *         </li>
     */
    public String failureMessage() {
        return failureMessage;
    }

    /**
     * <p>
     * Information about the IP addresses and network interface associated with the NAT gateway.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return Information about the IP addresses and network interface associated with the NAT gateway.
     */
    public List<NatGatewayAddress> natGatewayAddresses() {
        return natGatewayAddresses;
    }

    /**
     * <p>
     * The ID of the NAT gateway.
     * </p>
     * 
     * @return The ID of the NAT gateway.
     */
    public String natGatewayId() {
        return natGatewayId;
    }

    /**
     * <p>
     * Reserved. If you need to sustain traffic greater than the <a
     * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>, contact
     * us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
     * </p>
     * 
     * @return Reserved. If you need to sustain traffic greater than the <a
     *         href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>,
     *         contact us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
     */
    public ProvisionedBandwidth provisionedBandwidth() {
        return provisionedBandwidth;
    }

    /**
     * <p>
     * The state of the NAT gateway.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     * <code>failureMessage</code> fields for the reason.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the NAT
     * gateway, and does not indicate the health of the NAT gateway.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link NatGatewayState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateString}.
     * </p>
     * 
     * @return The state of the NAT gateway.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     *         <code>failureMessage</code> fields for the reason.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete
     *         the NAT gateway, and does not indicate the health of the NAT gateway.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
     *         traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     *         </p>
     *         </li>
     * @see NatGatewayState
     */
    public NatGatewayState state() {
        return NatGatewayState.fromValue(state);
    }

    /**
     * <p>
     * The state of the NAT gateway.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     * <code>failureMessage</code> fields for the reason.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the NAT
     * gateway, and does not indicate the health of the NAT gateway.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link NatGatewayState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateString}.
     * </p>
     * 
     * @return The state of the NAT gateway.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
     *         <code>failureMessage</code> fields for the reason.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete
     *         the NAT gateway, and does not indicate the health of the NAT gateway.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
     *         traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
     *         </p>
     *         </li>
     * @see NatGatewayState
     */
    public String stateString() {
        return state;
    }

    /**
     * <p>
     * The ID of the subnet in which the NAT gateway is located.
     * </p>
     * 
     * @return The ID of the subnet in which the NAT gateway is located.
     */
    public String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the VPC in which the NAT gateway is located.
     * </p>
     * 
     * @return The ID of the VPC in which the NAT gateway is located.
     */
    public String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The tags for the NAT gateway.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags for the NAT gateway.
     */
    public List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(deleteTime());
        hashCode = 31 * hashCode + Objects.hashCode(failureCode());
        hashCode = 31 * hashCode + Objects.hashCode(failureMessage());
        hashCode = 31 * hashCode + Objects.hashCode(natGatewayAddresses());
        hashCode = 31 * hashCode + Objects.hashCode(natGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(provisionedBandwidth());
        hashCode = 31 * hashCode + Objects.hashCode(stateString());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NatGateway)) {
            return false;
        }
        NatGateway other = (NatGateway) obj;
        return Objects.equals(createTime(), other.createTime()) && Objects.equals(deleteTime(), other.deleteTime())
                && Objects.equals(failureCode(), other.failureCode()) && Objects.equals(failureMessage(), other.failureMessage())
                && Objects.equals(natGatewayAddresses(), other.natGatewayAddresses())
                && Objects.equals(natGatewayId(), other.natGatewayId())
                && Objects.equals(provisionedBandwidth(), other.provisionedBandwidth())
                && Objects.equals(stateString(), other.stateString()) && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(tags(), other.tags());
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("{");
        if (createTime() != null) {
            sb.append("CreateTime: ").append(createTime()).append(",");
        }
        if (deleteTime() != null) {
            sb.append("DeleteTime: ").append(deleteTime()).append(",");
        }
        if (failureCode() != null) {
            sb.append("FailureCode: ").append(failureCode()).append(",");
        }
        if (failureMessage() != null) {
            sb.append("FailureMessage: ").append(failureMessage()).append(",");
        }
        if (natGatewayAddresses() != null) {
            sb.append("NatGatewayAddresses: ").append(natGatewayAddresses()).append(",");
        }
        if (natGatewayId() != null) {
            sb.append("NatGatewayId: ").append(natGatewayId()).append(",");
        }
        if (provisionedBandwidth() != null) {
            sb.append("ProvisionedBandwidth: ").append(provisionedBandwidth()).append(",");
        }
        if (stateString() != null) {
            sb.append("State: ").append(stateString()).append(",");
        }
        if (subnetId() != null) {
            sb.append("SubnetId: ").append(subnetId()).append(",");
        }
        if (vpcId() != null) {
            sb.append("VpcId: ").append(vpcId()).append(",");
        }
        if (tags() != null) {
            sb.append("Tags: ").append(tags()).append(",");
        }
        if (sb.length() > 1) {
            sb.setLength(sb.length() - 1);
        }
        sb.append("}");
        return sb.toString();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CreateTime":
            return Optional.of(clazz.cast(createTime()));
        case "DeleteTime":
            return Optional.of(clazz.cast(deleteTime()));
        case "FailureCode":
            return Optional.of(clazz.cast(failureCode()));
        case "FailureMessage":
            return Optional.of(clazz.cast(failureMessage()));
        case "NatGatewayAddresses":
            return Optional.of(clazz.cast(natGatewayAddresses()));
        case "NatGatewayId":
            return Optional.of(clazz.cast(natGatewayId()));
        case "ProvisionedBandwidth":
            return Optional.of(clazz.cast(provisionedBandwidth()));
        case "State":
            return Optional.of(clazz.cast(stateString()));
        case "SubnetId":
            return Optional.of(clazz.cast(subnetId()));
        case "VpcId":
            return Optional.of(clazz.cast(vpcId()));
        case "Tags":
            return Optional.of(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    public interface Builder extends CopyableBuilder<Builder, NatGateway> {
        /**
         * <p>
         * The date and time the NAT gateway was created.
         * </p>
         * 
         * @param createTime
         *        The date and time the NAT gateway was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * The date and time the NAT gateway was deleted, if applicable.
         * </p>
         * 
         * @param deleteTime
         *        The date and time the NAT gateway was deleted, if applicable.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteTime(Instant deleteTime);

        /**
         * <p>
         * If the NAT gateway could not be created, specifies the error code for the failure. (
         * <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
         * <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
         * <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
         * </p>
         * 
         * @param failureCode
         *        If the NAT gateway could not be created, specifies the error code for the failure. (
         *        <code>InsufficientFreeAddressesInSubnet</code> | <code>Gateway.NotAttached</code> |
         *        <code>InvalidAllocationID.NotFound</code> | <code>Resource.AlreadyAssociated</code> |
         *        <code>InternalError</code> | <code>InvalidSubnetID.NotFound</code>)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureCode(String failureCode);

        /**
         * <p>
         * If the NAT gateway could not be created, specifies the error message for the failure, that corresponds to the
         * error code.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For InsufficientFreeAddressesInSubnet: "Subnet has insufficient free addresses to create this NAT gateway"
         * </p>
         * </li>
         * <li>
         * <p>
         * For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
         * </p>
         * </li>
         * <li>
         * <p>
         * For InvalidAllocationID.NotFound:
         * "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
         * </p>
         * </li>
         * <li>
         * <p>
         * For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
         * </p>
         * </li>
         * <li>
         * <p>
         * For InternalError:
         * "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
         * </p>
         * </li>
         * <li>
         * <p>
         * For InvalidSubnetID.NotFound: "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
         * </p>
         * </li>
         * </ul>
         * 
         * @param failureMessage
         *        If the NAT gateway could not be created, specifies the error message for the failure, that corresponds
         *        to the error code.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For InsufficientFreeAddressesInSubnet:
         *        "Subnet has insufficient free addresses to create this NAT gateway"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Gateway.NotAttached: "Network vpc-xxxxxxxx has no Internet gateway attached"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InvalidAllocationID.NotFound:
         *        "Elastic IP address eipalloc-xxxxxxxx could not be associated with this NAT gateway"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For Resource.AlreadyAssociated: "Elastic IP address eipalloc-xxxxxxxx is already associated"
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InternalError:
         *        "Network interface eni-xxxxxxxx, created and used internally by this NAT gateway is in an invalid state. Please try again."
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For InvalidSubnetID.NotFound:
         *        "The specified subnet subnet-xxxxxxxx does not exist or could not be found."
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureMessage(String failureMessage);

        /**
         * <p>
         * Information about the IP addresses and network interface associated with the NAT gateway.
         * </p>
         * 
         * @param natGatewayAddresses
         *        Information about the IP addresses and network interface associated with the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayAddresses(Collection<NatGatewayAddress> natGatewayAddresses);

        /**
         * <p>
         * Information about the IP addresses and network interface associated with the NAT gateway.
         * </p>
         * 
         * @param natGatewayAddresses
         *        Information about the IP addresses and network interface associated with the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayAddresses(NatGatewayAddress... natGatewayAddresses);

        /**
         * <p>
         * The ID of the NAT gateway.
         * </p>
         * 
         * @param natGatewayId
         *        The ID of the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder natGatewayId(String natGatewayId);

        /**
         * <p>
         * Reserved. If you need to sustain traffic greater than the <a
         * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>,
         * contact us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
         * </p>
         * 
         * @param provisionedBandwidth
         *        Reserved. If you need to sustain traffic greater than the <a
         *        href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented
         *        limits</a>, contact us through the <a href="https://console.aws.amazon.com/support/home?">Support
         *        Center</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder provisionedBandwidth(ProvisionedBandwidth provisionedBandwidth);

        /**
         * <p>
         * Reserved. If you need to sustain traffic greater than the <a
         * href="http://docs.aws.amazon.com/AmazonVPC/latest/UserGuide/vpc-nat-gateway.html">documented limits</a>,
         * contact us through the <a href="https://console.aws.amazon.com/support/home?">Support Center</a>.
         * </p>
         * This is a convenience that creates an instance of the {@link ProvisionedBandwidth.Builder} avoiding the need
         * to create one manually via {@link ProvisionedBandwidth#builder()}.
         *
         * When the {@link Consumer} completes, {@link ProvisionedBandwidth.Builder#build()} is called immediately and
         * its result is passed to {@link #provisionedBandwidth(ProvisionedBandwidth)}.
         * 
         * @param provisionedBandwidth
         *        a consumer that will call methods on {@link ProvisionedBandwidth.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #provisionedBandwidth(ProvisionedBandwidth)
         */
        default Builder provisionedBandwidth(Consumer<ProvisionedBandwidth.Builder> provisionedBandwidth) {
            return provisionedBandwidth(ProvisionedBandwidth.builder().apply(provisionedBandwidth).build());
        }

        /**
         * <p>
         * The state of the NAT gateway.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         * <code>failureMessage</code> fields for the reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the
         * NAT gateway, and does not indicate the health of the NAT gateway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
         * traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the NAT gateway.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         *        <code>failureMessage</code> fields for the reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The NAT gateway is able to process traffic. This status remains until you
         *        delete the NAT gateway, and does not indicate the health of the NAT gateway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be
         *        processing traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         *        </p>
         *        </li>
         * @see NatGatewayState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NatGatewayState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the NAT gateway.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         * <code>failureMessage</code> fields for the reason.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>available</code>: The NAT gateway is able to process traffic. This status remains until you delete the
         * NAT gateway, and does not indicate the health of the NAT gateway.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be processing
         * traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the NAT gateway.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>pending</code>: The NAT gateway is being created and is not ready to process traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code>: The NAT gateway could not be created. Check the <code>failureCode</code> and
         *        <code>failureMessage</code> fields for the reason.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>available</code>: The NAT gateway is able to process traffic. This status remains until you
         *        delete the NAT gateway, and does not indicate the health of the NAT gateway.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleting</code>: The NAT gateway is in the process of being terminated and may still be
         *        processing traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deleted</code>: The NAT gateway has been terminated and is no longer processing traffic.
         *        </p>
         *        </li>
         * @see NatGatewayState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NatGatewayState
         */
        Builder state(NatGatewayState state);

        /**
         * <p>
         * The ID of the subnet in which the NAT gateway is located.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet in which the NAT gateway is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the VPC in which the NAT gateway is located.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC in which the NAT gateway is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The tags for the NAT gateway.
         * </p>
         * 
         * @param tags
         *        The tags for the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags for the NAT gateway.
         * </p>
         * 
         * @param tags
         *        The tags for the NAT gateway.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);
    }

    static final class BuilderImpl implements Builder {
        private Instant createTime;

        private Instant deleteTime;

        private String failureCode;

        private String failureMessage;

        private List<NatGatewayAddress> natGatewayAddresses;

        private String natGatewayId;

        private ProvisionedBandwidth provisionedBandwidth;

        private String state;

        private String subnetId;

        private String vpcId;

        private List<Tag> tags;

        private BuilderImpl() {
        }

        private BuilderImpl(NatGateway model) {
            createTime(model.createTime);
            deleteTime(model.deleteTime);
            failureCode(model.failureCode);
            failureMessage(model.failureMessage);
            natGatewayAddresses(model.natGatewayAddresses);
            natGatewayId(model.natGatewayId);
            provisionedBandwidth(model.provisionedBandwidth);
            state(model.state);
            subnetId(model.subnetId);
            vpcId(model.vpcId);
            tags(model.tags);
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        @Override
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        public final Instant getDeleteTime() {
            return deleteTime;
        }

        @Override
        public final Builder deleteTime(Instant deleteTime) {
            this.deleteTime = deleteTime;
            return this;
        }

        public final void setDeleteTime(Instant deleteTime) {
            this.deleteTime = deleteTime;
        }

        public final String getFailureCode() {
            return failureCode;
        }

        @Override
        public final Builder failureCode(String failureCode) {
            this.failureCode = failureCode;
            return this;
        }

        public final void setFailureCode(String failureCode) {
            this.failureCode = failureCode;
        }

        public final String getFailureMessage() {
            return failureMessage;
        }

        @Override
        public final Builder failureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
            return this;
        }

        public final void setFailureMessage(String failureMessage) {
            this.failureMessage = failureMessage;
        }

        public final Collection<NatGatewayAddress.Builder> getNatGatewayAddresses() {
            return natGatewayAddresses != null ? natGatewayAddresses.stream().map(NatGatewayAddress::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder natGatewayAddresses(Collection<NatGatewayAddress> natGatewayAddresses) {
            this.natGatewayAddresses = NatGatewayAddressListCopier.copy(natGatewayAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder natGatewayAddresses(NatGatewayAddress... natGatewayAddresses) {
            natGatewayAddresses(Arrays.asList(natGatewayAddresses));
            return this;
        }

        public final void setNatGatewayAddresses(Collection<NatGatewayAddress.BuilderImpl> natGatewayAddresses) {
            this.natGatewayAddresses = NatGatewayAddressListCopier.copyFromBuilder(natGatewayAddresses);
        }

        public final String getNatGatewayId() {
            return natGatewayId;
        }

        @Override
        public final Builder natGatewayId(String natGatewayId) {
            this.natGatewayId = natGatewayId;
            return this;
        }

        public final void setNatGatewayId(String natGatewayId) {
            this.natGatewayId = natGatewayId;
        }

        public final ProvisionedBandwidth.Builder getProvisionedBandwidth() {
            return provisionedBandwidth != null ? provisionedBandwidth.toBuilder() : null;
        }

        @Override
        public final Builder provisionedBandwidth(ProvisionedBandwidth provisionedBandwidth) {
            this.provisionedBandwidth = provisionedBandwidth;
            return this;
        }

        public final void setProvisionedBandwidth(ProvisionedBandwidth.BuilderImpl provisionedBandwidth) {
            this.provisionedBandwidth = provisionedBandwidth != null ? provisionedBandwidth.build() : null;
        }

        public final String getState() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(NatGatewayState state) {
            this.state(state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public final String getVpcId() {
            return vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public NatGateway build() {
            return new NatGateway(this);
        }
    }
}
