/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import javax.annotation.Generated;
import org.reactivestreams.Subscriber;
import software.amazon.awssdk.core.pagination.async.AsyncPageFetcher;
import software.amazon.awssdk.core.pagination.async.EmptySubscription;
import software.amazon.awssdk.core.pagination.async.PaginatedItemsPublisher;
import software.amazon.awssdk.core.pagination.async.ResponsesSubscription;
import software.amazon.awssdk.core.pagination.async.SdkPublisher;
import software.amazon.awssdk.services.ec2.EC2AsyncClient;
import software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsResponse;
import software.amazon.awssdk.services.ec2.model.ReservedInstancesOffering;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ec2.EC2AsyncClient#describeReservedInstancesOfferingsPaginator(software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsRequest)}
 * operation which is a paginated operation. This class is a type of {@link org.reactivestreams.Publisher} which can be
 * used to provide a sequence of
 * {@link software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsResponse} response pages as per
 * demand from the subscriber.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. If there are errors in your request, you will see the failures
 * only after you start streaming the data. The subscribe method should be called as a request to start streaming data.
 * For more info, see {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the
 * subscribe method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data
 * from the starting request.
 * </p>
 *
 * <p>
 * The following are few ways to use the response class:
 * </p>
 * 1) Using the forEach helper method
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeReservedInstancesOfferingsPublisher publisher = client.describeReservedInstancesOfferingsPaginator(request);
 * CompletableFuture<Void> future = publisher.forEach(res -> { // Do something with the response });
 * future.get();
 * }
 * </pre>
 *
 * 2) Using a custom subscriber
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribeReservedInstancesOfferingsPublisher publisher = client.describeReservedInstancesOfferingsPaginator(request);
 * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsResponse>() {
 * 
 * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
 * 
 * 
 * public void onNext(software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsResponse response) { //... };
 * });}
 * </pre>
 * 
 * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describeReservedInstancesOfferings(software.amazon.awssdk.services.ec2.model.DescribeReservedInstancesOfferingsRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribeReservedInstancesOfferingsPublisher implements SdkPublisher<DescribeReservedInstancesOfferingsResponse> {
    private final EC2AsyncClient client;

    private final DescribeReservedInstancesOfferingsRequest firstRequest;

    private final AsyncPageFetcher nextPageFetcher;

    private boolean isLastPage;

    public DescribeReservedInstancesOfferingsPublisher(final EC2AsyncClient client,
            final DescribeReservedInstancesOfferingsRequest firstRequest) {
        this(client, firstRequest, false);
    }

    private DescribeReservedInstancesOfferingsPublisher(final EC2AsyncClient client,
            final DescribeReservedInstancesOfferingsRequest firstRequest, final boolean isLastPage) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.isLastPage = isLastPage;
        this.nextPageFetcher = new DescribeReservedInstancesOfferingsResponseFetcher();
    }

    @Override
    public void subscribe(Subscriber<? super DescribeReservedInstancesOfferingsResponse> subscriber) {
        subscriber.onSubscribe(new ResponsesSubscription(subscriber, nextPageFetcher));
    }

    /**
     * Returns a publisher that can be used to get a stream of data. You need to subscribe to the publisher to request
     * the stream of data. The publisher has a helper forEach method that takes in a {@link java.util.function.Consumer}
     * and then applies that consumer to each response returned by the service.
     */
    public final SdkPublisher<ReservedInstancesOffering> reservedInstancesOfferings() {
        Function<DescribeReservedInstancesOfferingsResponse, Iterator<ReservedInstancesOffering>> getIterator = response -> {
            if (response != null && response.reservedInstancesOfferings() != null) {
                return response.reservedInstancesOfferings().iterator();
            }
            return Collections.emptyIterator();
        };
        return new PaginatedItemsPublisher(new DescribeReservedInstancesOfferingsResponseFetcher(), getIterator, isLastPage);
    }

    /**
     * <p>
     * A helper method to resume the pages in case of unexpected failures. The method takes the last successful response
     * page as input and returns an instance of {@link DescribeReservedInstancesOfferingsPublisher} that can be used to
     * retrieve the consecutive pages that follows the input page.
     * </p>
     */
    public final DescribeReservedInstancesOfferingsPublisher resume(
            final DescribeReservedInstancesOfferingsResponse lastSuccessfulPage) {
        if (nextPageFetcher.hasNextPage(lastSuccessfulPage)) {
            return new DescribeReservedInstancesOfferingsPublisher(client, firstRequest.toBuilder()
                    .nextToken(lastSuccessfulPage.nextToken()).build());
        }
        return new DescribeReservedInstancesOfferingsPublisher(client, firstRequest, true) {
            @Override
            public void subscribe(Subscriber<? super DescribeReservedInstancesOfferingsResponse> subscriber) {
                subscriber.onSubscribe(new EmptySubscription(subscriber));
            }
        };
    }

    private class DescribeReservedInstancesOfferingsResponseFetcher implements
            AsyncPageFetcher<DescribeReservedInstancesOfferingsResponse> {
        @Override
        public boolean hasNextPage(final DescribeReservedInstancesOfferingsResponse previousPage) {
            return previousPage.nextToken() != null;
        }

        @Override
        public CompletableFuture<DescribeReservedInstancesOfferingsResponse> nextPage(
                final DescribeReservedInstancesOfferingsResponse previousPage) {
            if (previousPage == null) {
                return client.describeReservedInstancesOfferings(firstRequest);
            }
            return client
                    .describeReservedInstancesOfferings(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
