/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for CreateVolume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateVolumeRequest extends Ec2Request implements
        ToCopyableBuilder<CreateVolumeRequest.Builder, CreateVolumeRequest> {
    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVolumeRequest::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("AvailabilityZone").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .getter(getter(CreateVolumeRequest::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateVolumeRequest::iops))
            .setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops")
                    .unmarshallLocationName("Iops").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVolumeRequest::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("KmsKeyId").build()).build();

    private static final SdkField<Integer> SIZE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateVolumeRequest::size))
            .setter(setter(Builder::size))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Size")
                    .unmarshallLocationName("Size").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVolumeRequest::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("SnapshotId").build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateVolumeRequest::volumeTypeAsString))
            .setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeType")
                    .unmarshallLocationName("VolumeType").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .getter(getter(CreateVolumeRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVAILABILITY_ZONE_FIELD,
            ENCRYPTED_FIELD, IOPS_FIELD, KMS_KEY_ID_FIELD, SIZE_FIELD, SNAPSHOT_ID_FIELD, VOLUME_TYPE_FIELD,
            TAG_SPECIFICATIONS_FIELD));

    private final String availabilityZone;

    private final Boolean encrypted;

    private final Integer iops;

    private final String kmsKeyId;

    private final Integer size;

    private final String snapshotId;

    private final String volumeType;

    private final List<TagSpecification> tagSpecifications;

    private CreateVolumeRequest(BuilderImpl builder) {
        super(builder);
        this.availabilityZone = builder.availabilityZone;
        this.encrypted = builder.encrypted;
        this.iops = builder.iops;
        this.kmsKeyId = builder.kmsKeyId;
        this.size = builder.size;
        this.snapshotId = builder.snapshotId;
        this.volumeType = builder.volumeType;
        this.tagSpecifications = builder.tagSpecifications;
    }

    /**
     * <p>
     * The Availability Zone in which to create the volume. Use <a>DescribeAvailabilityZones</a> to list the
     * Availability Zones that are currently available to you.
     * </p>
     * 
     * @return The Availability Zone in which to create the volume. Use <a>DescribeAvailabilityZones</a> to list the
     *         Availability Zones that are currently available to you.
     */
    public String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * Specifies whether the volume should be encrypted. Encrypted Amazon EBS volumes may only be attached to instances
     * that support Amazon EBS encryption. Volumes that are created from encrypted snapshots are automatically
     * encrypted. There is no way to create an encrypted volume from an unencrypted snapshot or vice versa. If your AMI
     * uses encrypted volumes, you can only launch it on supported instance types. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html">Amazon EBS Encryption</a> in the
     * <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * 
     * @return Specifies whether the volume should be encrypted. Encrypted Amazon EBS volumes may only be attached to
     *         instances that support Amazon EBS encryption. Volumes that are created from encrypted snapshots are
     *         automatically encrypted. There is no way to create an encrypted volume from an unencrypted snapshot or
     *         vice versa. If your AMI uses encrypted volumes, you can only launch it on supported instance types. For
     *         more information, see <a
     *         href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html">Amazon EBS Encryption</a> in
     *         the <i>Amazon Elastic Compute Cloud User Guide</i>.
     */
    public Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS) to provision for the volume, with a maximum ratio of 50 IOPS/GiB.
     * Range is 100 to 64,000IOPS for volumes in most regions. Maximum IOPS of 64,000 is guaranteed only on <a
     * href="AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based instances</a>. Other instance
     * families guarantee performance up to 32,000 IOPS. For more information, see <a
     * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in the
     * <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * This parameter is valid only for Provisioned IOPS SSD (io1) volumes.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS) to provision for the volume, with a maximum ratio of 50
     *         IOPS/GiB. Range is 100 to 64,000IOPS for volumes in most regions. Maximum IOPS of 64,000 is guaranteed
     *         only on <a href="AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based
     *         instances</a>. Other instance families guarantee performance up to 32,000 IOPS. For more information, see
     *         <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
     *         Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
     *         <p>
     *         This parameter is valid only for Provisioned IOPS SSD (io1) volumes.
     */
    public Integer iops() {
        return iops;
    }

    /**
     * <p>
     * An identifier for the AWS Key Management Service (AWS KMS) customer master key (CMK) to use when creating the
     * encrypted volume. This parameter is only required if you want to use a non-default CMK; if this parameter is not
     * specified, the default CMK for EBS is used. If a <code>KmsKeyId</code> is specified, the <code>Encrypted</code>
     * flag must also be set.
     * </p>
     * <p>
     * The CMK identifier may be provided in any of the following formats:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID
     * </p>
     * </li>
     * <li>
     * <p>
     * Key alias
     * </p>
     * </li>
     * <li>
     * <p>
     * ARN using key ID. The ID ARN contains the <code>arn:aws:kms</code> namespace, followed by the region of the CMK,
     * the AWS account ID of the CMK owner, the <code>key</code> namespace, and then the CMK ID. For example,
     * arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:key/<i>abcd1234-a123-456a-a12b-a123b4cd56ef</i>.
     * </p>
     * </li>
     * <li>
     * <p>
     * ARN using key alias. The alias ARN contains the <code>arn:aws:kms</code> namespace, followed by the region of the
     * CMK, the AWS account ID of the CMK owner, the <code>alias</code> namespace, and then the CMK alias. For example,
     * arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:alias/<i>ExampleAlias</i>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * AWS parses <code>KmsKeyId</code> asynchronously, meaning that the action you call may appear to complete even
     * though you provided an invalid identifier. The action will eventually fail.
     * </p>
     * 
     * @return An identifier for the AWS Key Management Service (AWS KMS) customer master key (CMK) to use when creating
     *         the encrypted volume. This parameter is only required if you want to use a non-default CMK; if this
     *         parameter is not specified, the default CMK for EBS is used. If a <code>KmsKeyId</code> is specified, the
     *         <code>Encrypted</code> flag must also be set. </p>
     *         <p>
     *         The CMK identifier may be provided in any of the following formats:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key alias
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ARN using key ID. The ID ARN contains the <code>arn:aws:kms</code> namespace, followed by the region of
     *         the CMK, the AWS account ID of the CMK owner, the <code>key</code> namespace, and then the CMK ID. For
     *         example,
     *         arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:key/<i>abcd1234-a123-456a-a12b-a123b4cd56ef</i>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         ARN using key alias. The alias ARN contains the <code>arn:aws:kms</code> namespace, followed by the
     *         region of the CMK, the AWS account ID of the CMK owner, the <code>alias</code> namespace, and then the
     *         CMK alias. For example, arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:alias/<i>ExampleAlias</i>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         AWS parses <code>KmsKeyId</code> asynchronously, meaning that the action you call may appear to complete
     *         even though you provided an invalid identifier. The action will eventually fail.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The size of the volume, in GiBs.
     * </p>
     * <p>
     * Constraints: 1-16,384 for <code>gp2</code>, 4-16,384 for <code>io1</code>, 500-16,384 for <code>st1</code>,
     * 500-16,384 for <code>sc1</code>, and 1-1,024 for <code>standard</code>. If you specify a snapshot, the volume
     * size must be equal to or larger than the snapshot size.
     * </p>
     * <p>
     * Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is the
     * snapshot size.
     * </p>
     * 
     * @return The size of the volume, in GiBs.</p>
     *         <p>
     *         Constraints: 1-16,384 for <code>gp2</code>, 4-16,384 for <code>io1</code>, 500-16,384 for
     *         <code>st1</code>, 500-16,384 for <code>sc1</code>, and 1-1,024 for <code>standard</code>. If you specify
     *         a snapshot, the volume size must be equal to or larger than the snapshot size.
     *         </p>
     *         <p>
     *         Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is
     *         the snapshot size.
     */
    public Integer size() {
        return size;
    }

    /**
     * <p>
     * The snapshot from which to create the volume.
     * </p>
     * 
     * @return The snapshot from which to create the volume.
     */
    public String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS SSD,
     * <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code> for
     * Magnetic volumes.
     * </p>
     * <p>
     * Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
     * eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1, ap-southeast-2,
     * ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to <code>gp2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned
     *         IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
     *         <code>standard</code> for Magnetic volumes.</p>
     *         <p>
     *         Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
     *         eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1,
     *         ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to
     *         <code>gp2</code>.
     * @see VolumeType
     */
    public VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS SSD,
     * <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code> for
     * Magnetic volumes.
     * </p>
     * <p>
     * Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
     * eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1, ap-southeast-2,
     * ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to <code>gp2</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned
     *         IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
     *         <code>standard</code> for Magnetic volumes.</p>
     *         <p>
     *         Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
     *         eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1,
     *         ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to
     *         <code>gp2</code>.
     * @see VolumeType
     */
    public String volumeTypeAsString() {
        return volumeType;
    }

    /**
     * <p>
     * The tags to apply to the volume during creation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * 
     * @return The tags to apply to the volume during creation.
     */
    public List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(size());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tagSpecifications());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateVolumeRequest)) {
            return false;
        }
        CreateVolumeRequest other = (CreateVolumeRequest) obj;
        return Objects.equals(availabilityZone(), other.availabilityZone()) && Objects.equals(encrypted(), other.encrypted())
                && Objects.equals(iops(), other.iops()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(size(), other.size()) && Objects.equals(snapshotId(), other.snapshotId())
                && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString())
                && Objects.equals(tagSpecifications(), other.tagSpecifications());
    }

    @Override
    public String toString() {
        return ToString.builder("CreateVolumeRequest").add("AvailabilityZone", availabilityZone()).add("Encrypted", encrypted())
                .add("Iops", iops()).add("KmsKeyId", kmsKeyId()).add("Size", size()).add("SnapshotId", snapshotId())
                .add("VolumeType", volumeTypeAsString()).add("TagSpecifications", tagSpecifications()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Size":
            return Optional.ofNullable(clazz.cast(size()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateVolumeRequest, T> g) {
        return obj -> g.apply((CreateVolumeRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateVolumeRequest> {
        /**
         * <p>
         * The Availability Zone in which to create the volume. Use <a>DescribeAvailabilityZones</a> to list the
         * Availability Zones that are currently available to you.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone in which to create the volume. Use <a>DescribeAvailabilityZones</a> to list the
         *        Availability Zones that are currently available to you.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * Specifies whether the volume should be encrypted. Encrypted Amazon EBS volumes may only be attached to
         * instances that support Amazon EBS encryption. Volumes that are created from encrypted snapshots are
         * automatically encrypted. There is no way to create an encrypted volume from an unencrypted snapshot or vice
         * versa. If your AMI uses encrypted volumes, you can only launch it on supported instance types. For more
         * information, see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html">Amazon EBS
         * Encryption</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * 
         * @param encrypted
         *        Specifies whether the volume should be encrypted. Encrypted Amazon EBS volumes may only be attached to
         *        instances that support Amazon EBS encryption. Volumes that are created from encrypted snapshots are
         *        automatically encrypted. There is no way to create an encrypted volume from an unencrypted snapshot or
         *        vice versa. If your AMI uses encrypted volumes, you can only launch it on supported instance types.
         *        For more information, see <a
         *        href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html">Amazon EBS Encryption</a>
         *        in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The number of I/O operations per second (IOPS) to provision for the volume, with a maximum ratio of 50
         * IOPS/GiB. Range is 100 to 64,000IOPS for volumes in most regions. Maximum IOPS of 64,000 is guaranteed only
         * on <a href="AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based instances</a>. Other
         * instance families guarantee performance up to 32,000 IOPS. For more information, see <a
         * href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume Types</a> in
         * the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * <p>
         * This parameter is valid only for Provisioned IOPS SSD (io1) volumes.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS) to provision for the volume, with a maximum ratio of 50
         *        IOPS/GiB. Range is 100 to 64,000IOPS for volumes in most regions. Maximum IOPS of 64,000 is guaranteed
         *        only on <a href="AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Nitro-based
         *        instances</a>. Other instance families guarantee performance up to 32,000 IOPS. For more information,
         *        see <a href="http://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS Volume
         *        Types</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         *        <p>
         *        This parameter is valid only for Provisioned IOPS SSD (io1) volumes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * An identifier for the AWS Key Management Service (AWS KMS) customer master key (CMK) to use when creating the
         * encrypted volume. This parameter is only required if you want to use a non-default CMK; if this parameter is
         * not specified, the default CMK for EBS is used. If a <code>KmsKeyId</code> is specified, the
         * <code>Encrypted</code> flag must also be set.
         * </p>
         * <p>
         * The CMK identifier may be provided in any of the following formats:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID
         * </p>
         * </li>
         * <li>
         * <p>
         * Key alias
         * </p>
         * </li>
         * <li>
         * <p>
         * ARN using key ID. The ID ARN contains the <code>arn:aws:kms</code> namespace, followed by the region of the
         * CMK, the AWS account ID of the CMK owner, the <code>key</code> namespace, and then the CMK ID. For example,
         * arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:key/<i>abcd1234-a123-456a-a12b-a123b4cd56ef</i>.
         * </p>
         * </li>
         * <li>
         * <p>
         * ARN using key alias. The alias ARN contains the <code>arn:aws:kms</code> namespace, followed by the region of
         * the CMK, the AWS account ID of the CMK owner, the <code>alias</code> namespace, and then the CMK alias. For
         * example, arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:alias/<i>ExampleAlias</i>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * AWS parses <code>KmsKeyId</code> asynchronously, meaning that the action you call may appear to complete even
         * though you provided an invalid identifier. The action will eventually fail.
         * </p>
         * 
         * @param kmsKeyId
         *        An identifier for the AWS Key Management Service (AWS KMS) customer master key (CMK) to use when
         *        creating the encrypted volume. This parameter is only required if you want to use a non-default CMK;
         *        if this parameter is not specified, the default CMK for EBS is used. If a <code>KmsKeyId</code> is
         *        specified, the <code>Encrypted</code> flag must also be set. </p>
         *        <p>
         *        The CMK identifier may be provided in any of the following formats:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key alias
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ARN using key ID. The ID ARN contains the <code>arn:aws:kms</code> namespace, followed by the region
         *        of the CMK, the AWS account ID of the CMK owner, the <code>key</code> namespace, and then the CMK ID.
         *        For example,
         *        arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:key/<i>abcd1234-a123-456a-a12b-a123b4cd56ef</i>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        ARN using key alias. The alias ARN contains the <code>arn:aws:kms</code> namespace, followed by the
         *        region of the CMK, the AWS account ID of the CMK owner, the <code>alias</code> namespace, and then the
         *        CMK alias. For example, arn:aws:kms:<i>us-east-1</i>:<i>012345678910</i>:alias/<i>ExampleAlias</i>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        AWS parses <code>KmsKeyId</code> asynchronously, meaning that the action you call may appear to
         *        complete even though you provided an invalid identifier. The action will eventually fail.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The size of the volume, in GiBs.
         * </p>
         * <p>
         * Constraints: 1-16,384 for <code>gp2</code>, 4-16,384 for <code>io1</code>, 500-16,384 for <code>st1</code>,
         * 500-16,384 for <code>sc1</code>, and 1-1,024 for <code>standard</code>. If you specify a snapshot, the volume
         * size must be equal to or larger than the snapshot size.
         * </p>
         * <p>
         * Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is the
         * snapshot size.
         * </p>
         * 
         * @param size
         *        The size of the volume, in GiBs.</p>
         *        <p>
         *        Constraints: 1-16,384 for <code>gp2</code>, 4-16,384 for <code>io1</code>, 500-16,384 for
         *        <code>st1</code>, 500-16,384 for <code>sc1</code>, and 1-1,024 for <code>standard</code>. If you
         *        specify a snapshot, the volume size must be equal to or larger than the snapshot size.
         *        </p>
         *        <p>
         *        Default: If you're creating the volume from a snapshot and don't specify a volume size, the default is
         *        the snapshot size.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder size(Integer size);

        /**
         * <p>
         * The snapshot from which to create the volume.
         * </p>
         * 
         * @param snapshotId
         *        The snapshot from which to create the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS
         * SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code>
         * for Magnetic volumes.
         * </p>
         * <p>
         * Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
         * eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1,
         * ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to
         * <code>gp2</code>.
         * </p>
         * 
         * @param volumeType
         *        The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for
         *        Provisioned IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
         *        <code>standard</code> for Magnetic volumes.</p>
         *        <p>
         *        Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1,
         *        eu-west-1, eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2,
         *        ap-southeast-1, ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS
         *        defaults to <code>gp2</code>.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for Provisioned IOPS
         * SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or <code>standard</code>
         * for Magnetic volumes.
         * </p>
         * <p>
         * Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1, eu-west-1,
         * eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2, ap-southeast-1,
         * ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS defaults to
         * <code>gp2</code>.
         * </p>
         * 
         * @param volumeType
         *        The volume type. This can be <code>gp2</code> for General Purpose SSD, <code>io1</code> for
         *        Provisioned IOPS SSD, <code>st1</code> for Throughput Optimized HDD, <code>sc1</code> for Cold HDD, or
         *        <code>standard</code> for Magnetic volumes.</p>
         *        <p>
         *        Defaults: If no volume type is specified, the default is <code>standard</code> in us-east-1,
         *        eu-west-1, eu-central-1, us-west-2, us-west-1, sa-east-1, ap-northeast-1, ap-northeast-2,
         *        ap-southeast-1, ap-southeast-2, ap-south-1, us-gov-west-1, and cn-north-1. In all other regions, EBS
         *        defaults to <code>gp2</code>.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);

        /**
         * <p>
         * The tags to apply to the volume during creation.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the volume during creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the volume during creation.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the volume during creation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the volume during creation.
         * </p>
         * This is a convenience that creates an instance of the {@link List<TagSpecification>.Builder} avoiding the
         * need to create one manually via {@link List<TagSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<TagSpecification>.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on {@link List<TagSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(List<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String availabilityZone;

        private Boolean encrypted;

        private Integer iops;

        private String kmsKeyId;

        private Integer size;

        private String snapshotId;

        private String volumeType;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateVolumeRequest model) {
            super(model);
            availabilityZone(model.availabilityZone);
            encrypted(model.encrypted);
            iops(model.iops);
            kmsKeyId(model.kmsKeyId);
            size(model.size);
            snapshotId(model.snapshotId);
            volumeType(model.volumeType);
            tagSpecifications(model.tagSpecifications);
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        public final Integer getIops() {
            return iops;
        }

        @Override
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final Integer getSize() {
            return size;
        }

        @Override
        public final Builder size(Integer size) {
            this.size = size;
            return this;
        }

        public final void setSize(Integer size) {
            this.size = size;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        @Override
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        public final String getVolumeType() {
            return volumeType;
        }

        @Override
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType.toString());
            return this;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        public final Collection<TagSpecification.Builder> getTagSpecifications() {
            return tagSpecifications != null ? tagSpecifications.stream().map(TagSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateVolumeRequest build() {
            return new CreateVolumeRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
