/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a VPN connection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpnConnection implements SdkPojo, Serializable, ToCopyableBuilder<VpnConnection.Builder, VpnConnection> {
    private static final SdkField<String> CUSTOMER_GATEWAY_CONFIGURATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::customerGatewayConfiguration))
            .setter(setter(Builder::customerGatewayConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerGatewayConfiguration")
                    .unmarshallLocationName("customerGatewayConfiguration").build()).build();

    private static final SdkField<String> CUSTOMER_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::customerGatewayId))
            .setter(setter(Builder::customerGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerGatewayId")
                    .unmarshallLocationName("customerGatewayId").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::category))
            .setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category")
                    .unmarshallLocationName("category").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::typeAsString))
            .setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type")
                    .unmarshallLocationName("type").build()).build();

    private static final SdkField<String> VPN_CONNECTION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::vpnConnectionId))
            .setter(setter(Builder::vpnConnectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnConnectionId")
                    .unmarshallLocationName("vpnConnectionId").build()).build();

    private static final SdkField<String> VPN_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::vpnGatewayId))
            .setter(setter(Builder::vpnGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnGatewayId")
                    .unmarshallLocationName("vpnGatewayId").build()).build();

    private static final SdkField<String> TRANSIT_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(VpnConnection::transitGatewayId))
            .setter(setter(Builder::transitGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitGatewayId")
                    .unmarshallLocationName("transitGatewayId").build()).build();

    private static final SdkField<VpnConnectionOptions> OPTIONS_FIELD = SdkField
            .<VpnConnectionOptions> builder(MarshallingType.SDK_POJO)
            .getter(getter(VpnConnection::options))
            .setter(setter(Builder::options))
            .constructor(VpnConnectionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Options")
                    .unmarshallLocationName("options").build()).build();

    private static final SdkField<List<VpnStaticRoute>> ROUTES_FIELD = SdkField
            .<List<VpnStaticRoute>> builder(MarshallingType.LIST)
            .getter(getter(VpnConnection::routes))
            .setter(setter(Builder::routes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Routes")
                    .unmarshallLocationName("routes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<VpnStaticRoute> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpnStaticRoute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(VpnConnection::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<VgwTelemetry>> VGW_TELEMETRY_FIELD = SdkField
            .<List<VgwTelemetry>> builder(MarshallingType.LIST)
            .getter(getter(VpnConnection::vgwTelemetry))
            .setter(setter(Builder::vgwTelemetry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VgwTelemetry")
                    .unmarshallLocationName("vgwTelemetry").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<VgwTelemetry> builder(MarshallingType.SDK_POJO)
                                            .constructor(VgwTelemetry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CUSTOMER_GATEWAY_CONFIGURATION_FIELD, CUSTOMER_GATEWAY_ID_FIELD, CATEGORY_FIELD, STATE_FIELD, TYPE_FIELD,
            VPN_CONNECTION_ID_FIELD, VPN_GATEWAY_ID_FIELD, TRANSIT_GATEWAY_ID_FIELD, OPTIONS_FIELD, ROUTES_FIELD, TAGS_FIELD,
            VGW_TELEMETRY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String customerGatewayConfiguration;

    private final String customerGatewayId;

    private final String category;

    private final String state;

    private final String type;

    private final String vpnConnectionId;

    private final String vpnGatewayId;

    private final String transitGatewayId;

    private final VpnConnectionOptions options;

    private final List<VpnStaticRoute> routes;

    private final List<Tag> tags;

    private final List<VgwTelemetry> vgwTelemetry;

    private VpnConnection(BuilderImpl builder) {
        this.customerGatewayConfiguration = builder.customerGatewayConfiguration;
        this.customerGatewayId = builder.customerGatewayId;
        this.category = builder.category;
        this.state = builder.state;
        this.type = builder.type;
        this.vpnConnectionId = builder.vpnConnectionId;
        this.vpnGatewayId = builder.vpnGatewayId;
        this.transitGatewayId = builder.transitGatewayId;
        this.options = builder.options;
        this.routes = builder.routes;
        this.tags = builder.tags;
        this.vgwTelemetry = builder.vgwTelemetry;
    }

    /**
     * <p>
     * The configuration information for the VPN connection's customer gateway (in the native XML format). This element
     * is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
     * <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
     * <code>available</code> state.
     * </p>
     * 
     * @return The configuration information for the VPN connection's customer gateway (in the native XML format). This
     *         element is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
     *         <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
     *         <code>available</code> state.
     */
    public String customerGatewayConfiguration() {
        return customerGatewayConfiguration;
    }

    /**
     * <p>
     * The ID of the customer gateway at your end of the VPN connection.
     * </p>
     * 
     * @return The ID of the customer gateway at your end of the VPN connection.
     */
    public String customerGatewayId() {
        return customerGatewayId;
    }

    /**
     * <p>
     * The category of the VPN connection. A value of <code>VPN</code> indicates an AWS VPN connection. A value of
     * <code>VPN-Classic</code> indicates an AWS Classic VPN connection.
     * </p>
     * 
     * @return The category of the VPN connection. A value of <code>VPN</code> indicates an AWS VPN connection. A value
     *         of <code>VPN-Classic</code> indicates an AWS Classic VPN connection.
     */
    public String category() {
        return category;
    }

    /**
     * <p>
     * The current state of the VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpnState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the VPN connection.
     * @see VpnState
     */
    public VpnState state() {
        return VpnState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpnState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the VPN connection.
     * @see VpnState
     */
    public String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The type of VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GatewayType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of VPN connection.
     * @see GatewayType
     */
    public GatewayType type() {
        return GatewayType.fromValue(type);
    }

    /**
     * <p>
     * The type of VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GatewayType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of VPN connection.
     * @see GatewayType
     */
    public String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The ID of the VPN connection.
     * </p>
     * 
     * @return The ID of the VPN connection.
     */
    public String vpnConnectionId() {
        return vpnConnectionId;
    }

    /**
     * <p>
     * The ID of the virtual private gateway at the AWS side of the VPN connection.
     * </p>
     * 
     * @return The ID of the virtual private gateway at the AWS side of the VPN connection.
     */
    public String vpnGatewayId() {
        return vpnGatewayId;
    }

    /**
     * <p>
     * The ID of the transit gateway associated with the VPN connection.
     * </p>
     * 
     * @return The ID of the transit gateway associated with the VPN connection.
     */
    public String transitGatewayId() {
        return transitGatewayId;
    }

    /**
     * <p>
     * The VPN connection options.
     * </p>
     * 
     * @return The VPN connection options.
     */
    public VpnConnectionOptions options() {
        return options;
    }

    /**
     * Returns true if the Routes property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasRoutes() {
        return routes != null && !(routes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The static routes associated with the VPN connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRoutes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The static routes associated with the VPN connection.
     */
    public List<VpnStaticRoute> routes() {
        return routes;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the VPN connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Any tags assigned to the VPN connection.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * Returns true if the VgwTelemetry property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasVgwTelemetry() {
        return vgwTelemetry != null && !(vgwTelemetry instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the VPN tunnel.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVgwTelemetry()} to see if a value was sent in this field.
     * </p>
     * 
     * @return Information about the VPN tunnel.
     */
    public List<VgwTelemetry> vgwTelemetry() {
        return vgwTelemetry;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(customerGatewayConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(customerGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(category());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpnConnectionId());
        hashCode = 31 * hashCode + Objects.hashCode(vpnGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(transitGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(options());
        hashCode = 31 * hashCode + Objects.hashCode(routes());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(vgwTelemetry());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpnConnection)) {
            return false;
        }
        VpnConnection other = (VpnConnection) obj;
        return Objects.equals(customerGatewayConfiguration(), other.customerGatewayConfiguration())
                && Objects.equals(customerGatewayId(), other.customerGatewayId()) && Objects.equals(category(), other.category())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(vpnConnectionId(), other.vpnConnectionId())
                && Objects.equals(vpnGatewayId(), other.vpnGatewayId())
                && Objects.equals(transitGatewayId(), other.transitGatewayId()) && Objects.equals(options(), other.options())
                && Objects.equals(routes(), other.routes()) && Objects.equals(tags(), other.tags())
                && Objects.equals(vgwTelemetry(), other.vgwTelemetry());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("VpnConnection").add("CustomerGatewayConfiguration", customerGatewayConfiguration())
                .add("CustomerGatewayId", customerGatewayId()).add("Category", category()).add("State", stateAsString())
                .add("Type", typeAsString()).add("VpnConnectionId", vpnConnectionId()).add("VpnGatewayId", vpnGatewayId())
                .add("TransitGatewayId", transitGatewayId()).add("Options", options()).add("Routes", routes())
                .add("Tags", tags()).add("VgwTelemetry", vgwTelemetry()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CustomerGatewayConfiguration":
            return Optional.ofNullable(clazz.cast(customerGatewayConfiguration()));
        case "CustomerGatewayId":
            return Optional.ofNullable(clazz.cast(customerGatewayId()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "VpnConnectionId":
            return Optional.ofNullable(clazz.cast(vpnConnectionId()));
        case "VpnGatewayId":
            return Optional.ofNullable(clazz.cast(vpnGatewayId()));
        case "TransitGatewayId":
            return Optional.ofNullable(clazz.cast(transitGatewayId()));
        case "Options":
            return Optional.ofNullable(clazz.cast(options()));
        case "Routes":
            return Optional.ofNullable(clazz.cast(routes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "VgwTelemetry":
            return Optional.ofNullable(clazz.cast(vgwTelemetry()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VpnConnection, T> g) {
        return obj -> g.apply((VpnConnection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpnConnection> {
        /**
         * <p>
         * The configuration information for the VPN connection's customer gateway (in the native XML format). This
         * element is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
         * <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
         * <code>available</code> state.
         * </p>
         * 
         * @param customerGatewayConfiguration
         *        The configuration information for the VPN connection's customer gateway (in the native XML format).
         *        This element is always present in the <a>CreateVpnConnection</a> response; however, it's present in
         *        the <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code>
         *        or <code>available</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerGatewayConfiguration(String customerGatewayConfiguration);

        /**
         * <p>
         * The ID of the customer gateway at your end of the VPN connection.
         * </p>
         * 
         * @param customerGatewayId
         *        The ID of the customer gateway at your end of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerGatewayId(String customerGatewayId);

        /**
         * <p>
         * The category of the VPN connection. A value of <code>VPN</code> indicates an AWS VPN connection. A value of
         * <code>VPN-Classic</code> indicates an AWS Classic VPN connection.
         * </p>
         * 
         * @param category
         *        The category of the VPN connection. A value of <code>VPN</code> indicates an AWS VPN connection. A
         *        value of <code>VPN-Classic</code> indicates an AWS Classic VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String category);

        /**
         * <p>
         * The current state of the VPN connection.
         * </p>
         * 
         * @param state
         *        The current state of the VPN connection.
         * @see VpnState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the VPN connection.
         * </p>
         * 
         * @param state
         *        The current state of the VPN connection.
         * @see VpnState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnState
         */
        Builder state(VpnState state);

        /**
         * <p>
         * The type of VPN connection.
         * </p>
         * 
         * @param type
         *        The type of VPN connection.
         * @see GatewayType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of VPN connection.
         * </p>
         * 
         * @param type
         *        The type of VPN connection.
         * @see GatewayType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayType
         */
        Builder type(GatewayType type);

        /**
         * <p>
         * The ID of the VPN connection.
         * </p>
         * 
         * @param vpnConnectionId
         *        The ID of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpnConnectionId(String vpnConnectionId);

        /**
         * <p>
         * The ID of the virtual private gateway at the AWS side of the VPN connection.
         * </p>
         * 
         * @param vpnGatewayId
         *        The ID of the virtual private gateway at the AWS side of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpnGatewayId(String vpnGatewayId);

        /**
         * <p>
         * The ID of the transit gateway associated with the VPN connection.
         * </p>
         * 
         * @param transitGatewayId
         *        The ID of the transit gateway associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitGatewayId(String transitGatewayId);

        /**
         * <p>
         * The VPN connection options.
         * </p>
         * 
         * @param options
         *        The VPN connection options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(VpnConnectionOptions options);

        /**
         * <p>
         * The VPN connection options.
         * </p>
         * This is a convenience that creates an instance of the {@link VpnConnectionOptions.Builder} avoiding the need
         * to create one manually via {@link VpnConnectionOptions#builder()}.
         *
         * When the {@link Consumer} completes, {@link VpnConnectionOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #options(VpnConnectionOptions)}.
         * 
         * @param options
         *        a consumer that will call methods on {@link VpnConnectionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #options(VpnConnectionOptions)
         */
        default Builder options(Consumer<VpnConnectionOptions.Builder> options) {
            return options(VpnConnectionOptions.builder().applyMutation(options).build());
        }

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * 
         * @param routes
         *        The static routes associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routes(Collection<VpnStaticRoute> routes);

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * 
         * @param routes
         *        The static routes associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routes(VpnStaticRoute... routes);

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * This is a convenience that creates an instance of the {@link List<VpnStaticRoute>.Builder} avoiding the need
         * to create one manually via {@link List<VpnStaticRoute>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<VpnStaticRoute>.Builder#build()} is called immediately and
         * its result is passed to {@link #routes(List<VpnStaticRoute>)}.
         * 
         * @param routes
         *        a consumer that will call methods on {@link List<VpnStaticRoute>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routes(List<VpnStaticRoute>)
         */
        Builder routes(Consumer<VpnStaticRoute.Builder>... routes);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * 
         * @param vgwTelemetry
         *        Information about the VPN tunnel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vgwTelemetry(Collection<VgwTelemetry> vgwTelemetry);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * 
         * @param vgwTelemetry
         *        Information about the VPN tunnel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vgwTelemetry(VgwTelemetry... vgwTelemetry);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * This is a convenience that creates an instance of the {@link List<VgwTelemetry>.Builder} avoiding the need to
         * create one manually via {@link List<VgwTelemetry>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<VgwTelemetry>.Builder#build()} is called immediately and its
         * result is passed to {@link #vgwTelemetry(List<VgwTelemetry>)}.
         * 
         * @param vgwTelemetry
         *        a consumer that will call methods on {@link List<VgwTelemetry>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vgwTelemetry(List<VgwTelemetry>)
         */
        Builder vgwTelemetry(Consumer<VgwTelemetry.Builder>... vgwTelemetry);
    }

    static final class BuilderImpl implements Builder {
        private String customerGatewayConfiguration;

        private String customerGatewayId;

        private String category;

        private String state;

        private String type;

        private String vpnConnectionId;

        private String vpnGatewayId;

        private String transitGatewayId;

        private VpnConnectionOptions options;

        private List<VpnStaticRoute> routes = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<VgwTelemetry> vgwTelemetry = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VpnConnection model) {
            customerGatewayConfiguration(model.customerGatewayConfiguration);
            customerGatewayId(model.customerGatewayId);
            category(model.category);
            state(model.state);
            type(model.type);
            vpnConnectionId(model.vpnConnectionId);
            vpnGatewayId(model.vpnGatewayId);
            transitGatewayId(model.transitGatewayId);
            options(model.options);
            routes(model.routes);
            tags(model.tags);
            vgwTelemetry(model.vgwTelemetry);
        }

        public final String getCustomerGatewayConfiguration() {
            return customerGatewayConfiguration;
        }

        @Override
        public final Builder customerGatewayConfiguration(String customerGatewayConfiguration) {
            this.customerGatewayConfiguration = customerGatewayConfiguration;
            return this;
        }

        public final void setCustomerGatewayConfiguration(String customerGatewayConfiguration) {
            this.customerGatewayConfiguration = customerGatewayConfiguration;
        }

        public final String getCustomerGatewayId() {
            return customerGatewayId;
        }

        @Override
        public final Builder customerGatewayId(String customerGatewayId) {
            this.customerGatewayId = customerGatewayId;
            return this;
        }

        public final void setCustomerGatewayId(String customerGatewayId) {
            this.customerGatewayId = customerGatewayId;
        }

        public final String getCategory() {
            return category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        public final String getStateAsString() {
            return state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(VpnState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final void setState(String state) {
            this.state = state;
        }

        public final String getTypeAsString() {
            return type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(GatewayType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final void setType(String type) {
            this.type = type;
        }

        public final String getVpnConnectionId() {
            return vpnConnectionId;
        }

        @Override
        public final Builder vpnConnectionId(String vpnConnectionId) {
            this.vpnConnectionId = vpnConnectionId;
            return this;
        }

        public final void setVpnConnectionId(String vpnConnectionId) {
            this.vpnConnectionId = vpnConnectionId;
        }

        public final String getVpnGatewayId() {
            return vpnGatewayId;
        }

        @Override
        public final Builder vpnGatewayId(String vpnGatewayId) {
            this.vpnGatewayId = vpnGatewayId;
            return this;
        }

        public final void setVpnGatewayId(String vpnGatewayId) {
            this.vpnGatewayId = vpnGatewayId;
        }

        public final String getTransitGatewayId() {
            return transitGatewayId;
        }

        @Override
        public final Builder transitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
            return this;
        }

        public final void setTransitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
        }

        public final VpnConnectionOptions.Builder getOptions() {
            return options != null ? options.toBuilder() : null;
        }

        @Override
        public final Builder options(VpnConnectionOptions options) {
            this.options = options;
            return this;
        }

        public final void setOptions(VpnConnectionOptions.BuilderImpl options) {
            this.options = options != null ? options.build() : null;
        }

        public final Collection<VpnStaticRoute.Builder> getRoutes() {
            return routes != null ? routes.stream().map(VpnStaticRoute::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder routes(Collection<VpnStaticRoute> routes) {
            this.routes = VpnStaticRouteListCopier.copy(routes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routes(VpnStaticRoute... routes) {
            routes(Arrays.asList(routes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routes(Consumer<VpnStaticRoute.Builder>... routes) {
            routes(Stream.of(routes).map(c -> VpnStaticRoute.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRoutes(Collection<VpnStaticRoute.BuilderImpl> routes) {
            this.routes = VpnStaticRouteListCopier.copyFromBuilder(routes);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        public final Collection<VgwTelemetry.Builder> getVgwTelemetry() {
            return vgwTelemetry != null ? vgwTelemetry.stream().map(VgwTelemetry::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder vgwTelemetry(Collection<VgwTelemetry> vgwTelemetry) {
            this.vgwTelemetry = VgwTelemetryListCopier.copy(vgwTelemetry);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vgwTelemetry(VgwTelemetry... vgwTelemetry) {
            vgwTelemetry(Arrays.asList(vgwTelemetry));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vgwTelemetry(Consumer<VgwTelemetry.Builder>... vgwTelemetry) {
            vgwTelemetry(Stream.of(vgwTelemetry).map(c -> VgwTelemetry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setVgwTelemetry(Collection<VgwTelemetry.BuilderImpl> vgwTelemetry) {
            this.vgwTelemetry = VgwTelemetryListCopier.copyFromBuilder(vgwTelemetry);
        }

        @Override
        public VpnConnection build() {
            return new VpnConnection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
