/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.paginators;

import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedItemsIterable;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.ec2.Ec2Client;
import software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatRequest;
import software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatResponse;
import software.amazon.awssdk.services.ec2.model.PrincipalIdFormat;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.ec2.Ec2Client#describePrincipalIdFormatPaginator(software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatResponse} that can be used to iterate
 * through all the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribePrincipalIdFormatIterable responses = client.describePrincipalIdFormatPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.ec2.paginators.DescribePrincipalIdFormatIterable responses = client
 *             .describePrincipalIdFormatPaginator(request);
 *     for (software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.ec2.paginators.DescribePrincipalIdFormatIterable responses = client.describePrincipalIdFormatPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #describePrincipalIdFormat(software.amazon.awssdk.services.ec2.model.DescribePrincipalIdFormatRequest)}
 * operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class DescribePrincipalIdFormatIterable implements SdkIterable<DescribePrincipalIdFormatResponse> {
    private final Ec2Client client;

    private final DescribePrincipalIdFormatRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public DescribePrincipalIdFormatIterable(Ec2Client client, DescribePrincipalIdFormatRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new DescribePrincipalIdFormatResponseFetcher();
    }

    @Override
    public Iterator<DescribePrincipalIdFormatResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    /**
     * Returns an iterable to iterate through the paginated {@link DescribePrincipalIdFormatResponse#principals()}
     * member. The returned iterable is used to iterate through the results across all response pages and not a single
     * page.
     *
     * This method is useful if you are interested in iterating over the paginated member in the response pages instead
     * of the top level pages. Similar to iteration over pages, this method internally makes service calls to get the
     * next list of results until the iteration stops or there are no more results.
     */
    public final SdkIterable<PrincipalIdFormat> principals() {
        Function<DescribePrincipalIdFormatResponse, Iterator<PrincipalIdFormat>> getIterator = response -> {
            if (response != null && response.principals() != null) {
                return response.principals().iterator();
            }
            return Collections.emptyIterator();
        };
        return PaginatedItemsIterable.<DescribePrincipalIdFormatResponse, PrincipalIdFormat> builder().pagesIterable(this)
                .itemIteratorFunction(getIterator).build();
    }

    private class DescribePrincipalIdFormatResponseFetcher implements SyncPageFetcher<DescribePrincipalIdFormatResponse> {
        @Override
        public boolean hasNextPage(DescribePrincipalIdFormatResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public DescribePrincipalIdFormatResponse nextPage(DescribePrincipalIdFormatResponse previousPage) {
            if (previousPage == null) {
                return client.describePrincipalIdFormat(firstRequest);
            }
            return client.describePrincipalIdFormat(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
