/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The parameters for a network interface.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LaunchTemplateInstanceNetworkInterfaceSpecificationRequest
        implements
        SdkPojo,
        Serializable,
        ToCopyableBuilder<LaunchTemplateInstanceNetworkInterfaceSpecificationRequest.Builder, LaunchTemplateInstanceNetworkInterfaceSpecificationRequest> {
    private static final SdkField<Boolean> ASSOCIATE_CARRIER_IP_ADDRESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AssociateCarrierIpAddress")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::associateCarrierIpAddress))
            .setter(setter(Builder::associateCarrierIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociateCarrierIpAddress")
                    .unmarshallLocationName("AssociateCarrierIpAddress").build()).build();

    private static final SdkField<Boolean> ASSOCIATE_PUBLIC_IP_ADDRESS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AssociatePublicIpAddress")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::associatePublicIpAddress))
            .setter(setter(Builder::associatePublicIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AssociatePublicIpAddress")
                    .unmarshallLocationName("AssociatePublicIpAddress").build()).build();

    private static final SdkField<Boolean> DELETE_ON_TERMINATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteOnTermination")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::deleteOnTermination))
            .setter(setter(Builder::deleteOnTermination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteOnTermination")
                    .unmarshallLocationName("DeleteOnTermination").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<Integer> DEVICE_INDEX_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("DeviceIndex")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::deviceIndex))
            .setter(setter(Builder::deviceIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceIndex")
                    .unmarshallLocationName("DeviceIndex").build()).build();

    private static final SdkField<List<String>> GROUPS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Groups")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::groups))
            .setter(setter(Builder::groups))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupId")
                    .unmarshallLocationName("SecurityGroupId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("SecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("SecurityGroupId").unmarshallLocationName("SecurityGroupId")
                                                    .build()).build()).build()).build();

    private static final SdkField<String> INTERFACE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InterfaceType")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::interfaceType))
            .setter(setter(Builder::interfaceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InterfaceType")
                    .unmarshallLocationName("InterfaceType").build()).build();

    private static final SdkField<Integer> IPV6_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Ipv6AddressCount")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::ipv6AddressCount))
            .setter(setter(Builder::ipv6AddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6AddressCount")
                    .unmarshallLocationName("Ipv6AddressCount").build()).build();

    private static final SdkField<List<InstanceIpv6AddressRequest>> IPV6_ADDRESSES_FIELD = SdkField
            .<List<InstanceIpv6AddressRequest>> builder(MarshallingType.LIST)
            .memberName("Ipv6Addresses")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::ipv6Addresses))
            .setter(setter(Builder::ipv6Addresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6Addresses")
                    .unmarshallLocationName("Ipv6Addresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("InstanceIpv6Address")
                            .memberFieldInfo(
                                    SdkField.<InstanceIpv6AddressRequest> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceIpv6AddressRequest::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("InstanceIpv6Address")
                                                    .unmarshallLocationName("InstanceIpv6Address").build()).build()).build())
            .build();

    private static final SdkField<String> NETWORK_INTERFACE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkInterfaceId")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::networkInterfaceId))
            .setter(setter(Builder::networkInterfaceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceId")
                    .unmarshallLocationName("NetworkInterfaceId").build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PrivateIpAddress")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::privateIpAddress))
            .setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress")
                    .unmarshallLocationName("PrivateIpAddress").build()).build();

    private static final SdkField<List<PrivateIpAddressSpecification>> PRIVATE_IP_ADDRESSES_FIELD = SdkField
            .<List<PrivateIpAddressSpecification>> builder(MarshallingType.LIST)
            .memberName("PrivateIpAddresses")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::privateIpAddresses))
            .setter(setter(Builder::privateIpAddresses))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddresses")
                    .unmarshallLocationName("PrivateIpAddresses").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<PrivateIpAddressSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(PrivateIpAddressSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("SecondaryPrivateIpAddressCount")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::secondaryPrivateIpAddressCount))
            .setter(setter(Builder::secondaryPrivateIpAddressCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecondaryPrivateIpAddressCount")
                    .unmarshallLocationName("SecondaryPrivateIpAddressCount").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("SubnetId").build()).build();

    private static final SdkField<Integer> NETWORK_CARD_INDEX_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("NetworkCardIndex")
            .getter(getter(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest::networkCardIndex))
            .setter(setter(Builder::networkCardIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkCardIndex")
                    .unmarshallLocationName("NetworkCardIndex").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            ASSOCIATE_CARRIER_IP_ADDRESS_FIELD, ASSOCIATE_PUBLIC_IP_ADDRESS_FIELD, DELETE_ON_TERMINATION_FIELD,
            DESCRIPTION_FIELD, DEVICE_INDEX_FIELD, GROUPS_FIELD, INTERFACE_TYPE_FIELD, IPV6_ADDRESS_COUNT_FIELD,
            IPV6_ADDRESSES_FIELD, NETWORK_INTERFACE_ID_FIELD, PRIVATE_IP_ADDRESS_FIELD, PRIVATE_IP_ADDRESSES_FIELD,
            SECONDARY_PRIVATE_IP_ADDRESS_COUNT_FIELD, SUBNET_ID_FIELD, NETWORK_CARD_INDEX_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean associateCarrierIpAddress;

    private final Boolean associatePublicIpAddress;

    private final Boolean deleteOnTermination;

    private final String description;

    private final Integer deviceIndex;

    private final List<String> groups;

    private final String interfaceType;

    private final Integer ipv6AddressCount;

    private final List<InstanceIpv6AddressRequest> ipv6Addresses;

    private final String networkInterfaceId;

    private final String privateIpAddress;

    private final List<PrivateIpAddressSpecification> privateIpAddresses;

    private final Integer secondaryPrivateIpAddressCount;

    private final String subnetId;

    private final Integer networkCardIndex;

    private LaunchTemplateInstanceNetworkInterfaceSpecificationRequest(BuilderImpl builder) {
        this.associateCarrierIpAddress = builder.associateCarrierIpAddress;
        this.associatePublicIpAddress = builder.associatePublicIpAddress;
        this.deleteOnTermination = builder.deleteOnTermination;
        this.description = builder.description;
        this.deviceIndex = builder.deviceIndex;
        this.groups = builder.groups;
        this.interfaceType = builder.interfaceType;
        this.ipv6AddressCount = builder.ipv6AddressCount;
        this.ipv6Addresses = builder.ipv6Addresses;
        this.networkInterfaceId = builder.networkInterfaceId;
        this.privateIpAddress = builder.privateIpAddress;
        this.privateIpAddresses = builder.privateIpAddresses;
        this.secondaryPrivateIpAddressCount = builder.secondaryPrivateIpAddressCount;
        this.subnetId = builder.subnetId;
        this.networkCardIndex = builder.networkCardIndex;
    }

    /**
     * <p>
     * Associates a Carrier IP address with eth0 for a new network interface.
     * </p>
     * <p>
     * Use this option when you launch an instance in a Wavelength Zone and want to associate a Carrier IP address with
     * the network interface. For more information about Carrier IP addresses, see <a
     * href="https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#provider-owned-ip"
     * >Carrier IP addresses</a> in the <i>AWS Wavelength Developer Guide</i>.
     * </p>
     * 
     * @return Associates a Carrier IP address with eth0 for a new network interface.</p>
     *         <p>
     *         Use this option when you launch an instance in a Wavelength Zone and want to associate a Carrier IP
     *         address with the network interface. For more information about Carrier IP addresses, see <a href=
     *         "https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#provider-owned-ip"
     *         >Carrier IP addresses</a> in the <i>AWS Wavelength Developer Guide</i>.
     */
    public Boolean associateCarrierIpAddress() {
        return associateCarrierIpAddress;
    }

    /**
     * <p>
     * Associates a public IPv4 address with eth0 for a new network interface.
     * </p>
     * 
     * @return Associates a public IPv4 address with eth0 for a new network interface.
     */
    public Boolean associatePublicIpAddress() {
        return associatePublicIpAddress;
    }

    /**
     * <p>
     * Indicates whether the network interface is deleted when the instance is terminated.
     * </p>
     * 
     * @return Indicates whether the network interface is deleted when the instance is terminated.
     */
    public Boolean deleteOnTermination() {
        return deleteOnTermination;
    }

    /**
     * <p>
     * A description for the network interface.
     * </p>
     * 
     * @return A description for the network interface.
     */
    public String description() {
        return description;
    }

    /**
     * <p>
     * The device index for the network interface attachment.
     * </p>
     * 
     * @return The device index for the network interface attachment.
     */
    public Integer deviceIndex() {
        return deviceIndex;
    }

    /**
     * Returns true if the Groups property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasGroups() {
        return groups != null && !(groups instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of one or more security groups.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasGroups()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IDs of one or more security groups.
     */
    public List<String> groups() {
        return groups;
    }

    /**
     * <p>
     * The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For more
     * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric
     * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
     * </p>
     * <p>
     * If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
     * </p>
     * <p>
     * Valid values: <code>interface</code> | <code>efa</code>
     * </p>
     * 
     * @return The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For
     *         more information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic
     *         Fabric Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
     *         <p>
     *         If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
     *         </p>
     *         <p>
     *         Valid values: <code>interface</code> | <code>efa</code>
     */
    public String interfaceType() {
        return interfaceType;
    }

    /**
     * <p>
     * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
     * </p>
     * 
     * @return The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
     *         addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
     */
    public Integer ipv6AddressCount() {
        return ipv6AddressCount;
    }

    /**
     * Returns true if the Ipv6Addresses property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIpv6Addresses() {
        return ipv6Addresses != null && !(ipv6Addresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option if
     * you're specifying a number of IPv6 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIpv6Addresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
     *         option if you're specifying a number of IPv6 addresses.
     */
    public List<InstanceIpv6AddressRequest> ipv6Addresses() {
        return ipv6Addresses;
    }

    /**
     * <p>
     * The ID of the network interface.
     * </p>
     * 
     * @return The ID of the network interface.
     */
    public String networkInterfaceId() {
        return networkInterfaceId;
    }

    /**
     * <p>
     * The primary private IPv4 address of the network interface.
     * </p>
     * 
     * @return The primary private IPv4 address of the network interface.
     */
    public String privateIpAddress() {
        return privateIpAddress;
    }

    /**
     * Returns true if the PrivateIpAddresses property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasPrivateIpAddresses() {
        return privateIpAddresses != null && !(privateIpAddresses instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more private IPv4 addresses.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasPrivateIpAddresses()} to see if a value was sent in this field.
     * </p>
     * 
     * @return One or more private IPv4 addresses.
     */
    public List<PrivateIpAddressSpecification> privateIpAddresses() {
        return privateIpAddresses;
    }

    /**
     * <p>
     * The number of secondary private IPv4 addresses to assign to a network interface.
     * </p>
     * 
     * @return The number of secondary private IPv4 addresses to assign to a network interface.
     */
    public Integer secondaryPrivateIpAddressCount() {
        return secondaryPrivateIpAddressCount;
    }

    /**
     * <p>
     * The ID of the subnet for the network interface.
     * </p>
     * 
     * @return The ID of the subnet for the network interface.
     */
    public String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The index of the network card. Some instance types support multiple network cards. The primary network interface
     * must be assigned to network card index 0. The default is network card index 0.
     * </p>
     * 
     * @return The index of the network card. Some instance types support multiple network cards. The primary network
     *         interface must be assigned to network card index 0. The default is network card index 0.
     */
    public Integer networkCardIndex() {
        return networkCardIndex;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(associateCarrierIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(associatePublicIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(deleteOnTermination());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(deviceIndex());
        hashCode = 31 * hashCode + Objects.hashCode(hasGroups() ? groups() : null);
        hashCode = 31 * hashCode + Objects.hashCode(interfaceType());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6AddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpv6Addresses() ? ipv6Addresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceId());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        hashCode = 31 * hashCode + Objects.hashCode(hasPrivateIpAddresses() ? privateIpAddresses() : null);
        hashCode = 31 * hashCode + Objects.hashCode(secondaryPrivateIpAddressCount());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(networkCardIndex());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LaunchTemplateInstanceNetworkInterfaceSpecificationRequest)) {
            return false;
        }
        LaunchTemplateInstanceNetworkInterfaceSpecificationRequest other = (LaunchTemplateInstanceNetworkInterfaceSpecificationRequest) obj;
        return Objects.equals(associateCarrierIpAddress(), other.associateCarrierIpAddress())
                && Objects.equals(associatePublicIpAddress(), other.associatePublicIpAddress())
                && Objects.equals(deleteOnTermination(), other.deleteOnTermination())
                && Objects.equals(description(), other.description()) && Objects.equals(deviceIndex(), other.deviceIndex())
                && hasGroups() == other.hasGroups() && Objects.equals(groups(), other.groups())
                && Objects.equals(interfaceType(), other.interfaceType())
                && Objects.equals(ipv6AddressCount(), other.ipv6AddressCount()) && hasIpv6Addresses() == other.hasIpv6Addresses()
                && Objects.equals(ipv6Addresses(), other.ipv6Addresses())
                && Objects.equals(networkInterfaceId(), other.networkInterfaceId())
                && Objects.equals(privateIpAddress(), other.privateIpAddress())
                && hasPrivateIpAddresses() == other.hasPrivateIpAddresses()
                && Objects.equals(privateIpAddresses(), other.privateIpAddresses())
                && Objects.equals(secondaryPrivateIpAddressCount(), other.secondaryPrivateIpAddressCount())
                && Objects.equals(subnetId(), other.subnetId()) && Objects.equals(networkCardIndex(), other.networkCardIndex());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("LaunchTemplateInstanceNetworkInterfaceSpecificationRequest")
                .add("AssociateCarrierIpAddress", associateCarrierIpAddress())
                .add("AssociatePublicIpAddress", associatePublicIpAddress()).add("DeleteOnTermination", deleteOnTermination())
                .add("Description", description()).add("DeviceIndex", deviceIndex()).add("Groups", hasGroups() ? groups() : null)
                .add("InterfaceType", interfaceType()).add("Ipv6AddressCount", ipv6AddressCount())
                .add("Ipv6Addresses", hasIpv6Addresses() ? ipv6Addresses() : null)
                .add("NetworkInterfaceId", networkInterfaceId()).add("PrivateIpAddress", privateIpAddress())
                .add("PrivateIpAddresses", hasPrivateIpAddresses() ? privateIpAddresses() : null)
                .add("SecondaryPrivateIpAddressCount", secondaryPrivateIpAddressCount()).add("SubnetId", subnetId())
                .add("NetworkCardIndex", networkCardIndex()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AssociateCarrierIpAddress":
            return Optional.ofNullable(clazz.cast(associateCarrierIpAddress()));
        case "AssociatePublicIpAddress":
            return Optional.ofNullable(clazz.cast(associatePublicIpAddress()));
        case "DeleteOnTermination":
            return Optional.ofNullable(clazz.cast(deleteOnTermination()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "DeviceIndex":
            return Optional.ofNullable(clazz.cast(deviceIndex()));
        case "Groups":
            return Optional.ofNullable(clazz.cast(groups()));
        case "InterfaceType":
            return Optional.ofNullable(clazz.cast(interfaceType()));
        case "Ipv6AddressCount":
            return Optional.ofNullable(clazz.cast(ipv6AddressCount()));
        case "Ipv6Addresses":
            return Optional.ofNullable(clazz.cast(ipv6Addresses()));
        case "NetworkInterfaceId":
            return Optional.ofNullable(clazz.cast(networkInterfaceId()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        case "PrivateIpAddresses":
            return Optional.ofNullable(clazz.cast(privateIpAddresses()));
        case "SecondaryPrivateIpAddressCount":
            return Optional.ofNullable(clazz.cast(secondaryPrivateIpAddressCount()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "NetworkCardIndex":
            return Optional.ofNullable(clazz.cast(networkCardIndex()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LaunchTemplateInstanceNetworkInterfaceSpecificationRequest, T> g) {
        return obj -> g.apply((LaunchTemplateInstanceNetworkInterfaceSpecificationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo,
            CopyableBuilder<Builder, LaunchTemplateInstanceNetworkInterfaceSpecificationRequest> {
        /**
         * <p>
         * Associates a Carrier IP address with eth0 for a new network interface.
         * </p>
         * <p>
         * Use this option when you launch an instance in a Wavelength Zone and want to associate a Carrier IP address
         * with the network interface. For more information about Carrier IP addresses, see <a href=
         * "https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#provider-owned-ip"
         * >Carrier IP addresses</a> in the <i>AWS Wavelength Developer Guide</i>.
         * </p>
         * 
         * @param associateCarrierIpAddress
         *        Associates a Carrier IP address with eth0 for a new network interface.</p>
         *        <p>
         *        Use this option when you launch an instance in a Wavelength Zone and want to associate a Carrier IP
         *        address with the network interface. For more information about Carrier IP addresses, see <a href=
         *        "https://docs.aws.amazon.com/wavelength/latest/developerguide/how-wavelengths-work.html#provider-owned-ip"
         *        >Carrier IP addresses</a> in the <i>AWS Wavelength Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associateCarrierIpAddress(Boolean associateCarrierIpAddress);

        /**
         * <p>
         * Associates a public IPv4 address with eth0 for a new network interface.
         * </p>
         * 
         * @param associatePublicIpAddress
         *        Associates a public IPv4 address with eth0 for a new network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder associatePublicIpAddress(Boolean associatePublicIpAddress);

        /**
         * <p>
         * Indicates whether the network interface is deleted when the instance is terminated.
         * </p>
         * 
         * @param deleteOnTermination
         *        Indicates whether the network interface is deleted when the instance is terminated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteOnTermination(Boolean deleteOnTermination);

        /**
         * <p>
         * A description for the network interface.
         * </p>
         * 
         * @param description
         *        A description for the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The device index for the network interface attachment.
         * </p>
         * 
         * @param deviceIndex
         *        The device index for the network interface attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceIndex(Integer deviceIndex);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(Collection<String> groups);

        /**
         * <p>
         * The IDs of one or more security groups.
         * </p>
         * 
         * @param groups
         *        The IDs of one or more security groups.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groups(String... groups);

        /**
         * <p>
         * The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>. For more
         * information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric
         * Adapter</a> in the <i>Amazon Elastic Compute Cloud User Guide</i>.
         * </p>
         * <p>
         * If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
         * </p>
         * <p>
         * Valid values: <code>interface</code> | <code>efa</code>
         * </p>
         * 
         * @param interfaceType
         *        The type of network interface. To create an Elastic Fabric Adapter (EFA), specify <code>efa</code>.
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/efa.html">Elastic Fabric Adapter</a> in the
         *        <i>Amazon Elastic Compute Cloud User Guide</i>.</p>
         *        <p>
         *        If you are not creating an EFA, specify <code>interface</code> or omit this parameter.
         *        </p>
         *        <p>
         *        Valid values: <code>interface</code> | <code>efa</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder interfaceType(String interfaceType);

        /**
         * <p>
         * The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the IPv6
         * addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
         * </p>
         * 
         * @param ipv6AddressCount
         *        The number of IPv6 addresses to assign to a network interface. Amazon EC2 automatically selects the
         *        IPv6 addresses from the subnet range. You can't use this option if specifying specific IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6AddressCount(Integer ipv6AddressCount);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(Collection<InstanceIpv6AddressRequest> ipv6Addresses);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * 
         * @param ipv6Addresses
         *        One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this
         *        option if you're specifying a number of IPv6 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6Addresses(InstanceIpv6AddressRequest... ipv6Addresses);

        /**
         * <p>
         * One or more specific IPv6 addresses from the IPv6 CIDR block range of your subnet. You can't use this option
         * if you're specifying a number of IPv6 addresses.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceIpv6AddressRequest>.Builder}
         * avoiding the need to create one manually via {@link List<InstanceIpv6AddressRequest>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceIpv6AddressRequest>.Builder#build()} is called
         * immediately and its result is passed to {@link #ipv6Addresses(List<InstanceIpv6AddressRequest>)}.
         * 
         * @param ipv6Addresses
         *        a consumer that will call methods on {@link List<InstanceIpv6AddressRequest>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipv6Addresses(List<InstanceIpv6AddressRequest>)
         */
        Builder ipv6Addresses(Consumer<InstanceIpv6AddressRequest.Builder>... ipv6Addresses);

        /**
         * <p>
         * The ID of the network interface.
         * </p>
         * 
         * @param networkInterfaceId
         *        The ID of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceId(String networkInterfaceId);

        /**
         * <p>
         * The primary private IPv4 address of the network interface.
         * </p>
         * 
         * @param privateIpAddress
         *        The primary private IPv4 address of the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * 
         * @param privateIpAddresses
         *        One or more private IPv4 addresses.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses);

        /**
         * <p>
         * One or more private IPv4 addresses.
         * </p>
         * This is a convenience that creates an instance of the {@link List<PrivateIpAddressSpecification>.Builder}
         * avoiding the need to create one manually via {@link List<PrivateIpAddressSpecification>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<PrivateIpAddressSpecification>.Builder#build()} is called
         * immediately and its result is passed to {@link #privateIpAddresses(List<PrivateIpAddressSpecification>)}.
         * 
         * @param privateIpAddresses
         *        a consumer that will call methods on {@link List<PrivateIpAddressSpecification>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #privateIpAddresses(List<PrivateIpAddressSpecification>)
         */
        Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses);

        /**
         * <p>
         * The number of secondary private IPv4 addresses to assign to a network interface.
         * </p>
         * 
         * @param secondaryPrivateIpAddressCount
         *        The number of secondary private IPv4 addresses to assign to a network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount);

        /**
         * <p>
         * The ID of the subnet for the network interface.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet for the network interface.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The index of the network card. Some instance types support multiple network cards. The primary network
         * interface must be assigned to network card index 0. The default is network card index 0.
         * </p>
         * 
         * @param networkCardIndex
         *        The index of the network card. Some instance types support multiple network cards. The primary network
         *        interface must be assigned to network card index 0. The default is network card index 0.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkCardIndex(Integer networkCardIndex);
    }

    static final class BuilderImpl implements Builder {
        private Boolean associateCarrierIpAddress;

        private Boolean associatePublicIpAddress;

        private Boolean deleteOnTermination;

        private String description;

        private Integer deviceIndex;

        private List<String> groups = DefaultSdkAutoConstructList.getInstance();

        private String interfaceType;

        private Integer ipv6AddressCount;

        private List<InstanceIpv6AddressRequest> ipv6Addresses = DefaultSdkAutoConstructList.getInstance();

        private String networkInterfaceId;

        private String privateIpAddress;

        private List<PrivateIpAddressSpecification> privateIpAddresses = DefaultSdkAutoConstructList.getInstance();

        private Integer secondaryPrivateIpAddressCount;

        private String subnetId;

        private Integer networkCardIndex;

        private BuilderImpl() {
        }

        private BuilderImpl(LaunchTemplateInstanceNetworkInterfaceSpecificationRequest model) {
            associateCarrierIpAddress(model.associateCarrierIpAddress);
            associatePublicIpAddress(model.associatePublicIpAddress);
            deleteOnTermination(model.deleteOnTermination);
            description(model.description);
            deviceIndex(model.deviceIndex);
            groups(model.groups);
            interfaceType(model.interfaceType);
            ipv6AddressCount(model.ipv6AddressCount);
            ipv6Addresses(model.ipv6Addresses);
            networkInterfaceId(model.networkInterfaceId);
            privateIpAddress(model.privateIpAddress);
            privateIpAddresses(model.privateIpAddresses);
            secondaryPrivateIpAddressCount(model.secondaryPrivateIpAddressCount);
            subnetId(model.subnetId);
            networkCardIndex(model.networkCardIndex);
        }

        public final Boolean getAssociateCarrierIpAddress() {
            return associateCarrierIpAddress;
        }

        @Override
        public final Builder associateCarrierIpAddress(Boolean associateCarrierIpAddress) {
            this.associateCarrierIpAddress = associateCarrierIpAddress;
            return this;
        }

        public final void setAssociateCarrierIpAddress(Boolean associateCarrierIpAddress) {
            this.associateCarrierIpAddress = associateCarrierIpAddress;
        }

        public final Boolean getAssociatePublicIpAddress() {
            return associatePublicIpAddress;
        }

        @Override
        public final Builder associatePublicIpAddress(Boolean associatePublicIpAddress) {
            this.associatePublicIpAddress = associatePublicIpAddress;
            return this;
        }

        public final void setAssociatePublicIpAddress(Boolean associatePublicIpAddress) {
            this.associatePublicIpAddress = associatePublicIpAddress;
        }

        public final Boolean getDeleteOnTermination() {
            return deleteOnTermination;
        }

        @Override
        public final Builder deleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
            return this;
        }

        public final void setDeleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Integer getDeviceIndex() {
            return deviceIndex;
        }

        @Override
        public final Builder deviceIndex(Integer deviceIndex) {
            this.deviceIndex = deviceIndex;
            return this;
        }

        public final void setDeviceIndex(Integer deviceIndex) {
            this.deviceIndex = deviceIndex;
        }

        public final Collection<String> getGroups() {
            if (groups instanceof SdkAutoConstructList) {
                return null;
            }
            return groups;
        }

        @Override
        public final Builder groups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groups(String... groups) {
            groups(Arrays.asList(groups));
            return this;
        }

        public final void setGroups(Collection<String> groups) {
            this.groups = SecurityGroupIdStringListCopier.copy(groups);
        }

        public final String getInterfaceType() {
            return interfaceType;
        }

        @Override
        public final Builder interfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
            return this;
        }

        public final void setInterfaceType(String interfaceType) {
            this.interfaceType = interfaceType;
        }

        public final Integer getIpv6AddressCount() {
            return ipv6AddressCount;
        }

        @Override
        public final Builder ipv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
            return this;
        }

        public final void setIpv6AddressCount(Integer ipv6AddressCount) {
            this.ipv6AddressCount = ipv6AddressCount;
        }

        public final Collection<InstanceIpv6AddressRequest.Builder> getIpv6Addresses() {
            if (ipv6Addresses instanceof SdkAutoConstructList) {
                return null;
            }
            return ipv6Addresses != null ? ipv6Addresses.stream().map(InstanceIpv6AddressRequest::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder ipv6Addresses(Collection<InstanceIpv6AddressRequest> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListRequestCopier.copy(ipv6Addresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(InstanceIpv6AddressRequest... ipv6Addresses) {
            ipv6Addresses(Arrays.asList(ipv6Addresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipv6Addresses(Consumer<InstanceIpv6AddressRequest.Builder>... ipv6Addresses) {
            ipv6Addresses(Stream.of(ipv6Addresses).map(c -> InstanceIpv6AddressRequest.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setIpv6Addresses(Collection<InstanceIpv6AddressRequest.BuilderImpl> ipv6Addresses) {
            this.ipv6Addresses = InstanceIpv6AddressListRequestCopier.copyFromBuilder(ipv6Addresses);
        }

        public final String getNetworkInterfaceId() {
            return networkInterfaceId;
        }

        @Override
        public final Builder networkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
            return this;
        }

        public final void setNetworkInterfaceId(String networkInterfaceId) {
            this.networkInterfaceId = networkInterfaceId;
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        public final Collection<PrivateIpAddressSpecification.Builder> getPrivateIpAddresses() {
            if (privateIpAddresses instanceof SdkAutoConstructList) {
                return null;
            }
            return privateIpAddresses != null ? privateIpAddresses.stream().map(PrivateIpAddressSpecification::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder privateIpAddresses(Collection<PrivateIpAddressSpecification> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copy(privateIpAddresses);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(PrivateIpAddressSpecification... privateIpAddresses) {
            privateIpAddresses(Arrays.asList(privateIpAddresses));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder privateIpAddresses(Consumer<PrivateIpAddressSpecification.Builder>... privateIpAddresses) {
            privateIpAddresses(Stream.of(privateIpAddresses)
                    .map(c -> PrivateIpAddressSpecification.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setPrivateIpAddresses(Collection<PrivateIpAddressSpecification.BuilderImpl> privateIpAddresses) {
            this.privateIpAddresses = PrivateIpAddressSpecificationListCopier.copyFromBuilder(privateIpAddresses);
        }

        public final Integer getSecondaryPrivateIpAddressCount() {
            return secondaryPrivateIpAddressCount;
        }

        @Override
        public final Builder secondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
            return this;
        }

        public final void setSecondaryPrivateIpAddressCount(Integer secondaryPrivateIpAddressCount) {
            this.secondaryPrivateIpAddressCount = secondaryPrivateIpAddressCount;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        public final Integer getNetworkCardIndex() {
            return networkCardIndex;
        }

        @Override
        public final Builder networkCardIndex(Integer networkCardIndex) {
            this.networkCardIndex = networkCardIndex;
            return this;
        }

        public final void setNetworkCardIndex(Integer networkCardIndex) {
            this.networkCardIndex = networkCardIndex;
        }

        @Override
        public LaunchTemplateInstanceNetworkInterfaceSpecificationRequest build() {
            return new LaunchTemplateInstanceNetworkInterfaceSpecificationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
