/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateStoreImageTaskRequest extends Ec2Request implements
        ToCopyableBuilder<CreateStoreImageTaskRequest.Builder, CreateStoreImageTaskRequest> {
    private static final SdkField<String> IMAGE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ImageId")
            .getter(getter(CreateStoreImageTaskRequest::imageId))
            .setter(setter(Builder::imageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ImageId")
                    .unmarshallLocationName("ImageId").build()).build();

    private static final SdkField<String> BUCKET_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Bucket")
            .getter(getter(CreateStoreImageTaskRequest::bucket))
            .setter(setter(Builder::bucket))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Bucket")
                    .unmarshallLocationName("Bucket").build()).build();

    private static final SdkField<List<S3ObjectTag>> S3_OBJECT_TAGS_FIELD = SdkField
            .<List<S3ObjectTag>> builder(MarshallingType.LIST)
            .memberName("S3ObjectTags")
            .getter(getter(CreateStoreImageTaskRequest::s3ObjectTags))
            .setter(setter(Builder::s3ObjectTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3ObjectTag")
                    .unmarshallLocationName("S3ObjectTag").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<S3ObjectTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(S3ObjectTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(IMAGE_ID_FIELD, BUCKET_FIELD,
            S3_OBJECT_TAGS_FIELD));

    private final String imageId;

    private final String bucket;

    private final List<S3ObjectTag> s3ObjectTags;

    private CreateStoreImageTaskRequest(BuilderImpl builder) {
        super(builder);
        this.imageId = builder.imageId;
        this.bucket = builder.bucket;
        this.s3ObjectTags = builder.s3ObjectTags;
    }

    /**
     * <p>
     * The ID of the AMI.
     * </p>
     * 
     * @return The ID of the AMI.
     */
    public final String imageId() {
        return imageId;
    }

    /**
     * <p>
     * The name of the Amazon S3 bucket in which the AMI object will be stored. The bucket must be in the Region in
     * which the request is being made. The AMI object appears in the bucket only after the upload task has completed.
     * </p>
     * 
     * @return The name of the Amazon S3 bucket in which the AMI object will be stored. The bucket must be in the Region
     *         in which the request is being made. The AMI object appears in the bucket only after the upload task has
     *         completed.
     */
    public final String bucket() {
        return bucket;
    }

    /**
     * For responses, this returns true if the service returned a value for the S3ObjectTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasS3ObjectTags() {
        return s3ObjectTags != null && !(s3ObjectTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasS3ObjectTags} method.
     * </p>
     * 
     * @return The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
     */
    public final List<S3ObjectTag> s3ObjectTags() {
        return s3ObjectTags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(imageId());
        hashCode = 31 * hashCode + Objects.hashCode(bucket());
        hashCode = 31 * hashCode + Objects.hashCode(hasS3ObjectTags() ? s3ObjectTags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateStoreImageTaskRequest)) {
            return false;
        }
        CreateStoreImageTaskRequest other = (CreateStoreImageTaskRequest) obj;
        return Objects.equals(imageId(), other.imageId()) && Objects.equals(bucket(), other.bucket())
                && hasS3ObjectTags() == other.hasS3ObjectTags() && Objects.equals(s3ObjectTags(), other.s3ObjectTags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateStoreImageTaskRequest").add("ImageId", imageId()).add("Bucket", bucket())
                .add("S3ObjectTags", hasS3ObjectTags() ? s3ObjectTags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ImageId":
            return Optional.ofNullable(clazz.cast(imageId()));
        case "Bucket":
            return Optional.ofNullable(clazz.cast(bucket()));
        case "S3ObjectTags":
            return Optional.ofNullable(clazz.cast(s3ObjectTags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateStoreImageTaskRequest, T> g) {
        return obj -> g.apply((CreateStoreImageTaskRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateStoreImageTaskRequest> {
        /**
         * <p>
         * The ID of the AMI.
         * </p>
         * 
         * @param imageId
         *        The ID of the AMI.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder imageId(String imageId);

        /**
         * <p>
         * The name of the Amazon S3 bucket in which the AMI object will be stored. The bucket must be in the Region in
         * which the request is being made. The AMI object appears in the bucket only after the upload task has
         * completed.
         * </p>
         * 
         * @param bucket
         *        The name of the Amazon S3 bucket in which the AMI object will be stored. The bucket must be in the
         *        Region in which the request is being made. The AMI object appears in the bucket only after the upload
         *        task has completed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucket(String bucket);

        /**
         * <p>
         * The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
         * </p>
         * 
         * @param s3ObjectTags
         *        The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ObjectTags(Collection<S3ObjectTag> s3ObjectTags);

        /**
         * <p>
         * The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
         * </p>
         * 
         * @param s3ObjectTags
         *        The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ObjectTags(S3ObjectTag... s3ObjectTags);

        /**
         * <p>
         * The tags to apply to the AMI object that will be stored in the Amazon S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link List<S3ObjectTag>.Builder} avoiding the
         * need to create one manually via {@link List<S3ObjectTag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<S3ObjectTag>.Builder#build()} is called immediately and its
         * result is passed to {@link #s3ObjectTags(List<S3ObjectTag>)}.
         * 
         * @param s3ObjectTags
         *        a consumer that will call methods on {@link List<S3ObjectTag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3ObjectTags(List<S3ObjectTag>)
         */
        Builder s3ObjectTags(Consumer<S3ObjectTag.Builder>... s3ObjectTags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String imageId;

        private String bucket;

        private List<S3ObjectTag> s3ObjectTags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateStoreImageTaskRequest model) {
            super(model);
            imageId(model.imageId);
            bucket(model.bucket);
            s3ObjectTags(model.s3ObjectTags);
        }

        public final String getImageId() {
            return imageId;
        }

        public final void setImageId(String imageId) {
            this.imageId = imageId;
        }

        @Override
        public final Builder imageId(String imageId) {
            this.imageId = imageId;
            return this;
        }

        public final String getBucket() {
            return bucket;
        }

        public final void setBucket(String bucket) {
            this.bucket = bucket;
        }

        @Override
        public final Builder bucket(String bucket) {
            this.bucket = bucket;
            return this;
        }

        public final List<S3ObjectTag.Builder> getS3ObjectTags() {
            List<S3ObjectTag.Builder> result = S3ObjectTagListCopier.copyToBuilder(this.s3ObjectTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setS3ObjectTags(Collection<S3ObjectTag.BuilderImpl> s3ObjectTags) {
            this.s3ObjectTags = S3ObjectTagListCopier.copyFromBuilder(s3ObjectTags);
        }

        @Override
        public final Builder s3ObjectTags(Collection<S3ObjectTag> s3ObjectTags) {
            this.s3ObjectTags = S3ObjectTagListCopier.copy(s3ObjectTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3ObjectTags(S3ObjectTag... s3ObjectTags) {
            s3ObjectTags(Arrays.asList(s3ObjectTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3ObjectTags(Consumer<S3ObjectTag.Builder>... s3ObjectTags) {
            s3ObjectTags(Stream.of(s3ObjectTags).map(c -> S3ObjectTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateStoreImageTaskRequest build() {
            return new CreateStoreImageTaskRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
