/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the configuration of Spot Instances in an EC2 Fleet request.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SpotOptionsRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<SpotOptionsRequest.Builder, SpotOptionsRequest> {
    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy")
            .getter(getter(SpotOptionsRequest::allocationStrategyAsString))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy")
                    .unmarshallLocationName("AllocationStrategy").build()).build();

    private static final SdkField<FleetSpotMaintenanceStrategiesRequest> MAINTENANCE_STRATEGIES_FIELD = SdkField
            .<FleetSpotMaintenanceStrategiesRequest> builder(MarshallingType.SDK_POJO)
            .memberName("MaintenanceStrategies")
            .getter(getter(SpotOptionsRequest::maintenanceStrategies))
            .setter(setter(Builder::maintenanceStrategies))
            .constructor(FleetSpotMaintenanceStrategiesRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaintenanceStrategies")
                    .unmarshallLocationName("MaintenanceStrategies").build()).build();

    private static final SdkField<String> INSTANCE_INTERRUPTION_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceInterruptionBehavior")
            .getter(getter(SpotOptionsRequest::instanceInterruptionBehaviorAsString))
            .setter(setter(Builder::instanceInterruptionBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceInterruptionBehavior")
                    .unmarshallLocationName("InstanceInterruptionBehavior").build()).build();

    private static final SdkField<Integer> INSTANCE_POOLS_TO_USE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstancePoolsToUseCount")
            .getter(getter(SpotOptionsRequest::instancePoolsToUseCount))
            .setter(setter(Builder::instancePoolsToUseCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstancePoolsToUseCount")
                    .unmarshallLocationName("InstancePoolsToUseCount").build()).build();

    private static final SdkField<Boolean> SINGLE_INSTANCE_TYPE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SingleInstanceType")
            .getter(getter(SpotOptionsRequest::singleInstanceType))
            .setter(setter(Builder::singleInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SingleInstanceType")
                    .unmarshallLocationName("SingleInstanceType").build()).build();

    private static final SdkField<Boolean> SINGLE_AVAILABILITY_ZONE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("SingleAvailabilityZone")
            .getter(getter(SpotOptionsRequest::singleAvailabilityZone))
            .setter(setter(Builder::singleAvailabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SingleAvailabilityZone")
                    .unmarshallLocationName("SingleAvailabilityZone").build()).build();

    private static final SdkField<Integer> MIN_TARGET_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MinTargetCapacity")
            .getter(getter(SpotOptionsRequest::minTargetCapacity))
            .setter(setter(Builder::minTargetCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinTargetCapacity")
                    .unmarshallLocationName("MinTargetCapacity").build()).build();

    private static final SdkField<String> MAX_TOTAL_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MaxTotalPrice")
            .getter(getter(SpotOptionsRequest::maxTotalPrice))
            .setter(setter(Builder::maxTotalPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxTotalPrice")
                    .unmarshallLocationName("MaxTotalPrice").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ALLOCATION_STRATEGY_FIELD,
            MAINTENANCE_STRATEGIES_FIELD, INSTANCE_INTERRUPTION_BEHAVIOR_FIELD, INSTANCE_POOLS_TO_USE_COUNT_FIELD,
            SINGLE_INSTANCE_TYPE_FIELD, SINGLE_AVAILABILITY_ZONE_FIELD, MIN_TARGET_CAPACITY_FIELD, MAX_TOTAL_PRICE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String allocationStrategy;

    private final FleetSpotMaintenanceStrategiesRequest maintenanceStrategies;

    private final String instanceInterruptionBehavior;

    private final Integer instancePoolsToUseCount;

    private final Boolean singleInstanceType;

    private final Boolean singleAvailabilityZone;

    private final Integer minTargetCapacity;

    private final String maxTotalPrice;

    private SpotOptionsRequest(BuilderImpl builder) {
        this.allocationStrategy = builder.allocationStrategy;
        this.maintenanceStrategies = builder.maintenanceStrategies;
        this.instanceInterruptionBehavior = builder.instanceInterruptionBehavior;
        this.instancePoolsToUseCount = builder.instancePoolsToUseCount;
        this.singleInstanceType = builder.singleInstanceType;
        this.singleAvailabilityZone = builder.singleAvailabilityZone;
        this.minTargetCapacity = builder.minTargetCapacity;
        this.maxTotalPrice = builder.maxTotalPrice;
    }

    /**
     * <p>
     * The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance pools
     * specified by the EC2 Fleet launch configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
     * strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that has
     * available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come from the next
     * cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling your desired
     * capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that
     * your desired capacity is met, you might receive Spot Instances from several pools.
     * </p>
     * <p>
     * <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you specify.
     * </p>
     * <p>
     * <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools with
     * optimal capacity for the number of instances that are launching. To give certain instance types a higher chance
     * of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for each instance type by
     * using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>. You can assign the same
     * priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a best-effort basis,
     * but optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported only if your fleet
     * uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set to
     * <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
     * </p>
     * <p>
     * Default: <code>lowest-price</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link SpotAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance
     *         pools specified by the EC2 Fleet launch configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
     *         strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that
     *         has available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come
     *         from the next cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling
     *         your desired capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest
     *         pool. To ensure that your desired capacity is met, you might receive Spot Instances from several pools.
     *         </p>
     *         <p>
     *         <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you
     *         specify.
     *         </p>
     *         <p>
     *         <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools
     *         with optimal capacity for the number of instances that are launching. To give certain instance types a
     *         higher chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for
     *         each instance type by using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>.
     *         You can assign the same priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the
     *         priorities on a best-effort basis, but optimizes for capacity first.
     *         <code>capacity-optimized-prioritized</code> is supported only if your fleet uses a launch template. Note
     *         that if the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, the same
     *         priority is applied when fulfilling On-Demand capacity.
     *         </p>
     *         <p>
     *         Default: <code>lowest-price</code>
     * @see SpotAllocationStrategy
     */
    public final SpotAllocationStrategy allocationStrategy() {
        return SpotAllocationStrategy.fromValue(allocationStrategy);
    }

    /**
     * <p>
     * The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance pools
     * specified by the EC2 Fleet launch configuration. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
     * strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that has
     * available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come from the next
     * cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling your desired
     * capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that
     * your desired capacity is met, you might receive Spot Instances from several pools.
     * </p>
     * <p>
     * <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you specify.
     * </p>
     * <p>
     * <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools with
     * optimal capacity for the number of instances that are launching. To give certain instance types a higher chance
     * of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for each instance type by
     * using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>. You can assign the same
     * priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a best-effort basis,
     * but optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported only if your fleet
     * uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set to
     * <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
     * </p>
     * <p>
     * Default: <code>lowest-price</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allocationStrategy} will return {@link SpotAllocationStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #allocationStrategyAsString}.
     * </p>
     * 
     * @return The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance
     *         pools specified by the EC2 Fleet launch configuration. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
     *         strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that
     *         has available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come
     *         from the next cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling
     *         your desired capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest
     *         pool. To ensure that your desired capacity is met, you might receive Spot Instances from several pools.
     *         </p>
     *         <p>
     *         <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you
     *         specify.
     *         </p>
     *         <p>
     *         <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools
     *         with optimal capacity for the number of instances that are launching. To give certain instance types a
     *         higher chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for
     *         each instance type by using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>.
     *         You can assign the same priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the
     *         priorities on a best-effort basis, but optimizes for capacity first.
     *         <code>capacity-optimized-prioritized</code> is supported only if your fleet uses a launch template. Note
     *         that if the On-Demand <code>AllocationStrategy</code> is set to <code>prioritized</code>, the same
     *         priority is applied when fulfilling On-Demand capacity.
     *         </p>
     *         <p>
     *         Default: <code>lowest-price</code>
     * @see SpotAllocationStrategy
     */
    public final String allocationStrategyAsString() {
        return allocationStrategy;
    }

    /**
     * <p>
     * The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
     * </p>
     * 
     * @return The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
     */
    public final FleetSpotMaintenanceStrategiesRequest maintenanceStrategies() {
        return maintenanceStrategies;
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted.
     * </p>
     * <p>
     * Default: <code>terminate</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link SpotInstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted.</p>
     *         <p>
     *         Default: <code>terminate</code>
     * @see SpotInstanceInterruptionBehavior
     */
    public final SpotInstanceInterruptionBehavior instanceInterruptionBehavior() {
        return SpotInstanceInterruptionBehavior.fromValue(instanceInterruptionBehavior);
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted.
     * </p>
     * <p>
     * Default: <code>terminate</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link SpotInstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}
     * . The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted.</p>
     *         <p>
     *         Default: <code>terminate</code>
     * @see SpotInstanceInterruptionBehavior
     */
    public final String instanceInterruptionBehaviorAsString() {
        return instanceInterruptionBehavior;
    }

    /**
     * <p>
     * The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot
     * <code>AllocationStrategy</code> is set to <code>lowest-price</code>. EC2 Fleet selects the cheapest Spot pools
     * and evenly allocates your target Spot capacity across the number of Spot pools that you specify.
     * </p>
     * <p>
     * Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best effort
     * basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will continue to
     * fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity is met, you
     * might receive Spot Instances from more than the number of pools that you specified. Similarly, if most of the
     * pools have no Spot capacity, you might receive your full target capacity from fewer than the number of pools that
     * you specified.
     * </p>
     * 
     * @return The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot
     *         <code>AllocationStrategy</code> is set to <code>lowest-price</code>. EC2 Fleet selects the cheapest Spot
     *         pools and evenly allocates your target Spot capacity across the number of Spot pools that you
     *         specify.</p>
     *         <p>
     *         Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best
     *         effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will
     *         continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target
     *         capacity is met, you might receive Spot Instances from more than the number of pools that you specified.
     *         Similarly, if most of the pools have no Spot capacity, you might receive your full target capacity from
     *         fewer than the number of pools that you specified.
     */
    public final Integer instancePoolsToUseCount() {
        return instancePoolsToUseCount;
    }

    /**
     * <p>
     * Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * 
     * @return Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     */
    public final Boolean singleInstanceType() {
        return singleInstanceType;
    }

    /**
     * <p>
     * Indicates that the fleet launches all Spot Instances into a single Availability Zone.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * 
     * @return Indicates that the fleet launches all Spot Instances into a single Availability Zone.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     */
    public final Boolean singleAvailabilityZone() {
        return singleAvailabilityZone;
    }

    /**
     * <p>
     * The minimum target capacity for Spot Instances in the fleet. If the minimum target capacity is not reached, the
     * fleet launches no instances.
     * </p>
     * <p>
     * Supported only for fleets of type <code>instant</code>.
     * </p>
     * <p>
     * At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
     * <code>SingleInstanceType</code>
     * </p>
     * 
     * @return The minimum target capacity for Spot Instances in the fleet. If the minimum target capacity is not
     *         reached, the fleet launches no instances.</p>
     *         <p>
     *         Supported only for fleets of type <code>instant</code>.
     *         </p>
     *         <p>
     *         At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
     *         <code>SingleInstanceType</code>
     */
    public final Integer minTargetCapacity() {
        return minTargetCapacity;
    }

    /**
     * <p>
     * The maximum amount per hour for Spot Instances that you're willing to pay. We do not recommend using this
     * parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the
     * current Spot price.
     * </p>
     * <important>
     * <p>
     * If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do not
     * specify this parameter.
     * </p>
     * </important>
     * 
     * @return The maximum amount per hour for Spot Instances that you're willing to pay. We do not recommend using this
     *         parameter because it can lead to increased interruptions. If you do not specify this parameter, you will
     *         pay the current Spot price.</p> <important>
     *         <p>
     *         If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do
     *         not specify this parameter.
     *         </p>
     */
    public final String maxTotalPrice() {
        return maxTotalPrice;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maintenanceStrategies());
        hashCode = 31 * hashCode + Objects.hashCode(instanceInterruptionBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(instancePoolsToUseCount());
        hashCode = 31 * hashCode + Objects.hashCode(singleInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(singleAvailabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(minTargetCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(maxTotalPrice());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SpotOptionsRequest)) {
            return false;
        }
        SpotOptionsRequest other = (SpotOptionsRequest) obj;
        return Objects.equals(allocationStrategyAsString(), other.allocationStrategyAsString())
                && Objects.equals(maintenanceStrategies(), other.maintenanceStrategies())
                && Objects.equals(instanceInterruptionBehaviorAsString(), other.instanceInterruptionBehaviorAsString())
                && Objects.equals(instancePoolsToUseCount(), other.instancePoolsToUseCount())
                && Objects.equals(singleInstanceType(), other.singleInstanceType())
                && Objects.equals(singleAvailabilityZone(), other.singleAvailabilityZone())
                && Objects.equals(minTargetCapacity(), other.minTargetCapacity())
                && Objects.equals(maxTotalPrice(), other.maxTotalPrice());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SpotOptionsRequest").add("AllocationStrategy", allocationStrategyAsString())
                .add("MaintenanceStrategies", maintenanceStrategies())
                .add("InstanceInterruptionBehavior", instanceInterruptionBehaviorAsString())
                .add("InstancePoolsToUseCount", instancePoolsToUseCount()).add("SingleInstanceType", singleInstanceType())
                .add("SingleAvailabilityZone", singleAvailabilityZone()).add("MinTargetCapacity", minTargetCapacity())
                .add("MaxTotalPrice", maxTotalPrice()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategyAsString()));
        case "MaintenanceStrategies":
            return Optional.ofNullable(clazz.cast(maintenanceStrategies()));
        case "InstanceInterruptionBehavior":
            return Optional.ofNullable(clazz.cast(instanceInterruptionBehaviorAsString()));
        case "InstancePoolsToUseCount":
            return Optional.ofNullable(clazz.cast(instancePoolsToUseCount()));
        case "SingleInstanceType":
            return Optional.ofNullable(clazz.cast(singleInstanceType()));
        case "SingleAvailabilityZone":
            return Optional.ofNullable(clazz.cast(singleAvailabilityZone()));
        case "MinTargetCapacity":
            return Optional.ofNullable(clazz.cast(minTargetCapacity()));
        case "MaxTotalPrice":
            return Optional.ofNullable(clazz.cast(maxTotalPrice()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<SpotOptionsRequest, T> g) {
        return obj -> g.apply((SpotOptionsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SpotOptionsRequest> {
        /**
         * <p>
         * The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance pools
         * specified by the EC2 Fleet launch configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
         * strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that has
         * available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come from the
         * next cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling your desired
         * capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure
         * that your desired capacity is met, you might receive Spot Instances from several pools.
         * </p>
         * <p>
         * <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you specify.
         * </p>
         * <p>
         * <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools with
         * optimal capacity for the number of instances that are launching. To give certain instance types a higher
         * chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for each instance
         * type by using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>. You can assign
         * the same priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a
         * best-effort basis, but optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported
         * only if your fleet uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set
         * to <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
         * </p>
         * <p>
         * Default: <code>lowest-price</code>
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines how to allocate the target Spot Instance capacity across the Spot
         *        Instance pools specified by the EC2 Fleet launch configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html"
         *        >Allocation strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that
         *        has available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come
         *        from the next cheapest pool that has available capacity. If a pool runs out of capacity before
         *        fulfilling your desired capacity, EC2 Fleet will continue to fulfill your request by drawing from the
         *        next cheapest pool. To ensure that your desired capacity is met, you might receive Spot Instances from
         *        several pools.
         *        </p>
         *        <p>
         *        <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you
         *        specify.
         *        </p>
         *        <p>
         *        <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools
         *        with optimal capacity for the number of instances that are launching. To give certain instance types a
         *        higher chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for
         *        each instance type by using the <code>Priority</code> parameter for
         *        <code>LaunchTemplateOverrides</code>. You can assign the same priority to different
         *        <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a best-effort basis, but
         *        optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported only if your
         *        fleet uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set to
         *        <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
         *        </p>
         *        <p>
         *        Default: <code>lowest-price</code>
         * @see SpotAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotAllocationStrategy
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * The strategy that determines how to allocate the target Spot Instance capacity across the Spot Instance pools
         * specified by the EC2 Fleet launch configuration. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html">Allocation
         * strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that has
         * available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come from the
         * next cheapest pool that has available capacity. If a pool runs out of capacity before fulfilling your desired
         * capacity, EC2 Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure
         * that your desired capacity is met, you might receive Spot Instances from several pools.
         * </p>
         * <p>
         * <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you specify.
         * </p>
         * <p>
         * <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools with
         * optimal capacity for the number of instances that are launching. To give certain instance types a higher
         * chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for each instance
         * type by using the <code>Priority</code> parameter for <code>LaunchTemplateOverrides</code>. You can assign
         * the same priority to different <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a
         * best-effort basis, but optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported
         * only if your fleet uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set
         * to <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
         * </p>
         * <p>
         * Default: <code>lowest-price</code>
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy that determines how to allocate the target Spot Instance capacity across the Spot
         *        Instance pools specified by the EC2 Fleet launch configuration. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-allocation-strategy.html"
         *        >Allocation strategies for Spot Instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        <code>lowest-price</code> - EC2 Fleet launches instances from the lowest-price Spot Instance pool that
         *        has available capacity. If the cheapest pool doesn't have available capacity, the Spot Instances come
         *        from the next cheapest pool that has available capacity. If a pool runs out of capacity before
         *        fulfilling your desired capacity, EC2 Fleet will continue to fulfill your request by drawing from the
         *        next cheapest pool. To ensure that your desired capacity is met, you might receive Spot Instances from
         *        several pools.
         *        </p>
         *        <p>
         *        <code>diversified</code> - EC2 Fleet launches instances from all of the Spot Instance pools that you
         *        specify.
         *        </p>
         *        <p>
         *        <code>capacity-optimized</code> (recommended) - EC2 Fleet launches instances from Spot Instance pools
         *        with optimal capacity for the number of instances that are launching. To give certain instance types a
         *        higher chance of launching first, use <code>capacity-optimized-prioritized</code>. Set a priority for
         *        each instance type by using the <code>Priority</code> parameter for
         *        <code>LaunchTemplateOverrides</code>. You can assign the same priority to different
         *        <code>LaunchTemplateOverrides</code>. EC2 implements the priorities on a best-effort basis, but
         *        optimizes for capacity first. <code>capacity-optimized-prioritized</code> is supported only if your
         *        fleet uses a launch template. Note that if the On-Demand <code>AllocationStrategy</code> is set to
         *        <code>prioritized</code>, the same priority is applied when fulfilling On-Demand capacity.
         *        </p>
         *        <p>
         *        Default: <code>lowest-price</code>
         * @see SpotAllocationStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotAllocationStrategy
         */
        Builder allocationStrategy(SpotAllocationStrategy allocationStrategy);

        /**
         * <p>
         * The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
         * </p>
         * 
         * @param maintenanceStrategies
         *        The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maintenanceStrategies(FleetSpotMaintenanceStrategiesRequest maintenanceStrategies);

        /**
         * <p>
         * The strategies for managing your Spot Instances that are at an elevated risk of being interrupted.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link FleetSpotMaintenanceStrategiesRequest.Builder} avoiding the need to create one manually via
         * {@link FleetSpotMaintenanceStrategiesRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link FleetSpotMaintenanceStrategiesRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #maintenanceStrategies(FleetSpotMaintenanceStrategiesRequest)}.
         * 
         * @param maintenanceStrategies
         *        a consumer that will call methods on {@link FleetSpotMaintenanceStrategiesRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #maintenanceStrategies(FleetSpotMaintenanceStrategiesRequest)
         */
        default Builder maintenanceStrategies(Consumer<FleetSpotMaintenanceStrategiesRequest.Builder> maintenanceStrategies) {
            return maintenanceStrategies(FleetSpotMaintenanceStrategiesRequest.builder().applyMutation(maintenanceStrategies)
                    .build());
        }

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted.
         * </p>
         * <p>
         * Default: <code>terminate</code>
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted.</p>
         *        <p>
         *        Default: <code>terminate</code>
         * @see SpotInstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(String instanceInterruptionBehavior);

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted.
         * </p>
         * <p>
         * Default: <code>terminate</code>
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted.</p>
         *        <p>
         *        Default: <code>terminate</code>
         * @see SpotInstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(SpotInstanceInterruptionBehavior instanceInterruptionBehavior);

        /**
         * <p>
         * The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot
         * <code>AllocationStrategy</code> is set to <code>lowest-price</code>. EC2 Fleet selects the cheapest Spot
         * pools and evenly allocates your target Spot capacity across the number of Spot pools that you specify.
         * </p>
         * <p>
         * Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a best
         * effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2 Fleet will
         * continue to fulfill your request by drawing from the next cheapest pool. To ensure that your target capacity
         * is met, you might receive Spot Instances from more than the number of pools that you specified. Similarly, if
         * most of the pools have no Spot capacity, you might receive your full target capacity from fewer than the
         * number of pools that you specified.
         * </p>
         * 
         * @param instancePoolsToUseCount
         *        The number of Spot pools across which to allocate your target Spot capacity. Supported only when Spot
         *        <code>AllocationStrategy</code> is set to <code>lowest-price</code>. EC2 Fleet selects the cheapest
         *        Spot pools and evenly allocates your target Spot capacity across the number of Spot pools that you
         *        specify.</p>
         *        <p>
         *        Note that EC2 Fleet attempts to draw Spot Instances from the number of pools that you specify on a
         *        best effort basis. If a pool runs out of Spot capacity before fulfilling your target capacity, EC2
         *        Fleet will continue to fulfill your request by drawing from the next cheapest pool. To ensure that
         *        your target capacity is met, you might receive Spot Instances from more than the number of pools that
         *        you specified. Similarly, if most of the pools have no Spot capacity, you might receive your full
         *        target capacity from fewer than the number of pools that you specified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instancePoolsToUseCount(Integer instancePoolsToUseCount);

        /**
         * <p>
         * Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * 
         * @param singleInstanceType
         *        Indicates that the fleet uses a single instance type to launch all Spot Instances in the fleet.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder singleInstanceType(Boolean singleInstanceType);

        /**
         * <p>
         * Indicates that the fleet launches all Spot Instances into a single Availability Zone.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * 
         * @param singleAvailabilityZone
         *        Indicates that the fleet launches all Spot Instances into a single Availability Zone.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder singleAvailabilityZone(Boolean singleAvailabilityZone);

        /**
         * <p>
         * The minimum target capacity for Spot Instances in the fleet. If the minimum target capacity is not reached,
         * the fleet launches no instances.
         * </p>
         * <p>
         * Supported only for fleets of type <code>instant</code>.
         * </p>
         * <p>
         * At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
         * <code>SingleInstanceType</code>
         * </p>
         * 
         * @param minTargetCapacity
         *        The minimum target capacity for Spot Instances in the fleet. If the minimum target capacity is not
         *        reached, the fleet launches no instances.</p>
         *        <p>
         *        Supported only for fleets of type <code>instant</code>.
         *        </p>
         *        <p>
         *        At least one of the following must be specified: <code>SingleAvailabilityZone</code> |
         *        <code>SingleInstanceType</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minTargetCapacity(Integer minTargetCapacity);

        /**
         * <p>
         * The maximum amount per hour for Spot Instances that you're willing to pay. We do not recommend using this
         * parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay
         * the current Spot price.
         * </p>
         * <important>
         * <p>
         * If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do not
         * specify this parameter.
         * </p>
         * </important>
         * 
         * @param maxTotalPrice
         *        The maximum amount per hour for Spot Instances that you're willing to pay. We do not recommend using
         *        this parameter because it can lead to increased interruptions. If you do not specify this parameter,
         *        you will pay the current Spot price.</p> <important>
         *        <p>
         *        If you specify a maximum price, your Spot Instances will be interrupted more frequently than if you do
         *        not specify this parameter.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxTotalPrice(String maxTotalPrice);
    }

    static final class BuilderImpl implements Builder {
        private String allocationStrategy;

        private FleetSpotMaintenanceStrategiesRequest maintenanceStrategies;

        private String instanceInterruptionBehavior;

        private Integer instancePoolsToUseCount;

        private Boolean singleInstanceType;

        private Boolean singleAvailabilityZone;

        private Integer minTargetCapacity;

        private String maxTotalPrice;

        private BuilderImpl() {
        }

        private BuilderImpl(SpotOptionsRequest model) {
            allocationStrategy(model.allocationStrategy);
            maintenanceStrategies(model.maintenanceStrategies);
            instanceInterruptionBehavior(model.instanceInterruptionBehavior);
            instancePoolsToUseCount(model.instancePoolsToUseCount);
            singleInstanceType(model.singleInstanceType);
            singleAvailabilityZone(model.singleAvailabilityZone);
            minTargetCapacity(model.minTargetCapacity);
            maxTotalPrice(model.maxTotalPrice);
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        @Override
        public final Builder allocationStrategy(SpotAllocationStrategy allocationStrategy) {
            this.allocationStrategy(allocationStrategy == null ? null : allocationStrategy.toString());
            return this;
        }

        public final FleetSpotMaintenanceStrategiesRequest.Builder getMaintenanceStrategies() {
            return maintenanceStrategies != null ? maintenanceStrategies.toBuilder() : null;
        }

        public final void setMaintenanceStrategies(FleetSpotMaintenanceStrategiesRequest.BuilderImpl maintenanceStrategies) {
            this.maintenanceStrategies = maintenanceStrategies != null ? maintenanceStrategies.build() : null;
        }

        @Override
        public final Builder maintenanceStrategies(FleetSpotMaintenanceStrategiesRequest maintenanceStrategies) {
            this.maintenanceStrategies = maintenanceStrategies;
            return this;
        }

        public final String getInstanceInterruptionBehavior() {
            return instanceInterruptionBehavior;
        }

        public final void setInstanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
        }

        @Override
        public final Builder instanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
            return this;
        }

        @Override
        public final Builder instanceInterruptionBehavior(SpotInstanceInterruptionBehavior instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior(instanceInterruptionBehavior == null ? null : instanceInterruptionBehavior
                    .toString());
            return this;
        }

        public final Integer getInstancePoolsToUseCount() {
            return instancePoolsToUseCount;
        }

        public final void setInstancePoolsToUseCount(Integer instancePoolsToUseCount) {
            this.instancePoolsToUseCount = instancePoolsToUseCount;
        }

        @Override
        public final Builder instancePoolsToUseCount(Integer instancePoolsToUseCount) {
            this.instancePoolsToUseCount = instancePoolsToUseCount;
            return this;
        }

        public final Boolean getSingleInstanceType() {
            return singleInstanceType;
        }

        public final void setSingleInstanceType(Boolean singleInstanceType) {
            this.singleInstanceType = singleInstanceType;
        }

        @Override
        public final Builder singleInstanceType(Boolean singleInstanceType) {
            this.singleInstanceType = singleInstanceType;
            return this;
        }

        public final Boolean getSingleAvailabilityZone() {
            return singleAvailabilityZone;
        }

        public final void setSingleAvailabilityZone(Boolean singleAvailabilityZone) {
            this.singleAvailabilityZone = singleAvailabilityZone;
        }

        @Override
        public final Builder singleAvailabilityZone(Boolean singleAvailabilityZone) {
            this.singleAvailabilityZone = singleAvailabilityZone;
            return this;
        }

        public final Integer getMinTargetCapacity() {
            return minTargetCapacity;
        }

        public final void setMinTargetCapacity(Integer minTargetCapacity) {
            this.minTargetCapacity = minTargetCapacity;
        }

        @Override
        public final Builder minTargetCapacity(Integer minTargetCapacity) {
            this.minTargetCapacity = minTargetCapacity;
            return this;
        }

        public final String getMaxTotalPrice() {
            return maxTotalPrice;
        }

        public final void setMaxTotalPrice(String maxTotalPrice) {
            this.maxTotalPrice = maxTotalPrice;
        }

        @Override
        public final Builder maxTotalPrice(String maxTotalPrice) {
            this.maxTotalPrice = maxTotalPrice;
            return this;
        }

        @Override
        public SpotOptionsRequest build() {
            return new SpotOptionsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
