/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.retry.backoff.BackoffStrategy;
import software.amazon.awssdk.core.retry.backoff.FixedDelayBackoffStrategy;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.core.waiters.WaiterState;
import software.amazon.awssdk.services.ec2.Ec2AsyncClient;
import software.amazon.awssdk.services.ec2.model.DescribeBundleTasksRequest;
import software.amazon.awssdk.services.ec2.model.DescribeBundleTasksResponse;
import software.amazon.awssdk.services.ec2.model.DescribeConversionTasksRequest;
import software.amazon.awssdk.services.ec2.model.DescribeConversionTasksResponse;
import software.amazon.awssdk.services.ec2.model.DescribeCustomerGatewaysRequest;
import software.amazon.awssdk.services.ec2.model.DescribeCustomerGatewaysResponse;
import software.amazon.awssdk.services.ec2.model.DescribeExportTasksRequest;
import software.amazon.awssdk.services.ec2.model.DescribeExportTasksResponse;
import software.amazon.awssdk.services.ec2.model.DescribeImagesRequest;
import software.amazon.awssdk.services.ec2.model.DescribeImagesResponse;
import software.amazon.awssdk.services.ec2.model.DescribeInstanceStatusRequest;
import software.amazon.awssdk.services.ec2.model.DescribeInstanceStatusResponse;
import software.amazon.awssdk.services.ec2.model.DescribeInstancesRequest;
import software.amazon.awssdk.services.ec2.model.DescribeInstancesResponse;
import software.amazon.awssdk.services.ec2.model.DescribeKeyPairsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeKeyPairsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeNatGatewaysRequest;
import software.amazon.awssdk.services.ec2.model.DescribeNatGatewaysResponse;
import software.amazon.awssdk.services.ec2.model.DescribeNetworkInterfacesRequest;
import software.amazon.awssdk.services.ec2.model.DescribeNetworkInterfacesResponse;
import software.amazon.awssdk.services.ec2.model.DescribeSecurityGroupsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeSecurityGroupsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeSpotInstanceRequestsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeSpotInstanceRequestsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeSubnetsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeSubnetsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeVolumesRequest;
import software.amazon.awssdk.services.ec2.model.DescribeVolumesResponse;
import software.amazon.awssdk.services.ec2.model.DescribeVpcPeeringConnectionsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeVpcPeeringConnectionsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeVpcsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeVpcsResponse;
import software.amazon.awssdk.services.ec2.model.DescribeVpnConnectionsRequest;
import software.amazon.awssdk.services.ec2.model.DescribeVpnConnectionsResponse;
import software.amazon.awssdk.services.ec2.model.Ec2Request;
import software.amazon.awssdk.services.ec2.model.GetPasswordDataRequest;
import software.amazon.awssdk.services.ec2.model.GetPasswordDataResponse;
import software.amazon.awssdk.services.ec2.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultEc2AsyncWaiter implements Ec2AsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final Ec2AsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeInstancesResponse> instanceExistsWaiter;

    private final AsyncWaiter<DescribeBundleTasksResponse> bundleTaskCompleteWaiter;

    private final AsyncWaiter<DescribeConversionTasksResponse> conversionTaskCancelledWaiter;

    private final AsyncWaiter<DescribeConversionTasksResponse> conversionTaskCompletedWaiter;

    private final AsyncWaiter<DescribeConversionTasksResponse> conversionTaskDeletedWaiter;

    private final AsyncWaiter<DescribeCustomerGatewaysResponse> customerGatewayAvailableWaiter;

    private final AsyncWaiter<DescribeExportTasksResponse> exportTaskCancelledWaiter;

    private final AsyncWaiter<DescribeExportTasksResponse> exportTaskCompletedWaiter;

    private final AsyncWaiter<DescribeImagesResponse> imageExistsWaiter;

    private final AsyncWaiter<DescribeImagesResponse> imageAvailableWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceRunningWaiter;

    private final AsyncWaiter<DescribeInstanceStatusResponse> instanceStatusOkWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceStoppedWaiter;

    private final AsyncWaiter<DescribeInstancesResponse> instanceTerminatedWaiter;

    private final AsyncWaiter<DescribeKeyPairsResponse> keyPairExistsWaiter;

    private final AsyncWaiter<DescribeNatGatewaysResponse> natGatewayAvailableWaiter;

    private final AsyncWaiter<DescribeNetworkInterfacesResponse> networkInterfaceAvailableWaiter;

    private final AsyncWaiter<GetPasswordDataResponse> passwordDataAvailableWaiter;

    private final AsyncWaiter<DescribeSnapshotsResponse> snapshotCompletedWaiter;

    private final AsyncWaiter<DescribeSecurityGroupsResponse> securityGroupExistsWaiter;

    private final AsyncWaiter<DescribeSpotInstanceRequestsResponse> spotInstanceRequestFulfilledWaiter;

    private final AsyncWaiter<DescribeSubnetsResponse> subnetAvailableWaiter;

    private final AsyncWaiter<DescribeInstanceStatusResponse> systemStatusOkWaiter;

    private final AsyncWaiter<DescribeVolumesResponse> volumeAvailableWaiter;

    private final AsyncWaiter<DescribeVolumesResponse> volumeDeletedWaiter;

    private final AsyncWaiter<DescribeVolumesResponse> volumeInUseWaiter;

    private final AsyncWaiter<DescribeVpcsResponse> vpcAvailableWaiter;

    private final AsyncWaiter<DescribeVpcsResponse> vpcExistsWaiter;

    private final AsyncWaiter<DescribeVpnConnectionsResponse> vpnConnectionAvailableWaiter;

    private final AsyncWaiter<DescribeVpnConnectionsResponse> vpnConnectionDeletedWaiter;

    private final AsyncWaiter<DescribeVpcPeeringConnectionsResponse> vpcPeeringConnectionExistsWaiter;

    private final AsyncWaiter<DescribeVpcPeeringConnectionsResponse> vpcPeeringConnectionDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultEc2AsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = Ec2AsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.instanceExistsWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceExistsWaiterAcceptors())
                .overrideConfiguration(instanceExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.bundleTaskCompleteWaiter = AsyncWaiter.builder(DescribeBundleTasksResponse.class)
                .acceptors(bundleTaskCompleteWaiterAcceptors())
                .overrideConfiguration(bundleTaskCompleteWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.conversionTaskCancelledWaiter = AsyncWaiter.builder(DescribeConversionTasksResponse.class)
                .acceptors(conversionTaskCancelledWaiterAcceptors())
                .overrideConfiguration(conversionTaskCancelledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.conversionTaskCompletedWaiter = AsyncWaiter.builder(DescribeConversionTasksResponse.class)
                .acceptors(conversionTaskCompletedWaiterAcceptors())
                .overrideConfiguration(conversionTaskCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.conversionTaskDeletedWaiter = AsyncWaiter.builder(DescribeConversionTasksResponse.class)
                .acceptors(conversionTaskDeletedWaiterAcceptors())
                .overrideConfiguration(conversionTaskDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.customerGatewayAvailableWaiter = AsyncWaiter.builder(DescribeCustomerGatewaysResponse.class)
                .acceptors(customerGatewayAvailableWaiterAcceptors())
                .overrideConfiguration(customerGatewayAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.exportTaskCancelledWaiter = AsyncWaiter.builder(DescribeExportTasksResponse.class)
                .acceptors(exportTaskCancelledWaiterAcceptors())
                .overrideConfiguration(exportTaskCancelledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.exportTaskCompletedWaiter = AsyncWaiter.builder(DescribeExportTasksResponse.class)
                .acceptors(exportTaskCompletedWaiterAcceptors())
                .overrideConfiguration(exportTaskCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageExistsWaiter = AsyncWaiter.builder(DescribeImagesResponse.class).acceptors(imageExistsWaiterAcceptors())
                .overrideConfiguration(imageExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.imageAvailableWaiter = AsyncWaiter.builder(DescribeImagesResponse.class).acceptors(imageAvailableWaiterAcceptors())
                .overrideConfiguration(imageAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceRunningWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceRunningWaiterAcceptors())
                .overrideConfiguration(instanceRunningWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceStatusOkWaiter = AsyncWaiter.builder(DescribeInstanceStatusResponse.class)
                .acceptors(instanceStatusOkWaiterAcceptors())
                .overrideConfiguration(instanceStatusOkWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceStoppedWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceStoppedWaiterAcceptors())
                .overrideConfiguration(instanceStoppedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.instanceTerminatedWaiter = AsyncWaiter.builder(DescribeInstancesResponse.class)
                .acceptors(instanceTerminatedWaiterAcceptors())
                .overrideConfiguration(instanceTerminatedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.keyPairExistsWaiter = AsyncWaiter.builder(DescribeKeyPairsResponse.class).acceptors(keyPairExistsWaiterAcceptors())
                .overrideConfiguration(keyPairExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.natGatewayAvailableWaiter = AsyncWaiter.builder(DescribeNatGatewaysResponse.class)
                .acceptors(natGatewayAvailableWaiterAcceptors())
                .overrideConfiguration(natGatewayAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.networkInterfaceAvailableWaiter = AsyncWaiter.builder(DescribeNetworkInterfacesResponse.class)
                .acceptors(networkInterfaceAvailableWaiterAcceptors())
                .overrideConfiguration(networkInterfaceAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.passwordDataAvailableWaiter = AsyncWaiter.builder(GetPasswordDataResponse.class)
                .acceptors(passwordDataAvailableWaiterAcceptors())
                .overrideConfiguration(passwordDataAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.snapshotCompletedWaiter = AsyncWaiter.builder(DescribeSnapshotsResponse.class)
                .acceptors(snapshotCompletedWaiterAcceptors())
                .overrideConfiguration(snapshotCompletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.securityGroupExistsWaiter = AsyncWaiter.builder(DescribeSecurityGroupsResponse.class)
                .acceptors(securityGroupExistsWaiterAcceptors())
                .overrideConfiguration(securityGroupExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.spotInstanceRequestFulfilledWaiter = AsyncWaiter.builder(DescribeSpotInstanceRequestsResponse.class)
                .acceptors(spotInstanceRequestFulfilledWaiterAcceptors())
                .overrideConfiguration(spotInstanceRequestFulfilledWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.subnetAvailableWaiter = AsyncWaiter.builder(DescribeSubnetsResponse.class)
                .acceptors(subnetAvailableWaiterAcceptors())
                .overrideConfiguration(subnetAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.systemStatusOkWaiter = AsyncWaiter.builder(DescribeInstanceStatusResponse.class)
                .acceptors(systemStatusOkWaiterAcceptors())
                .overrideConfiguration(systemStatusOkWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.volumeAvailableWaiter = AsyncWaiter.builder(DescribeVolumesResponse.class)
                .acceptors(volumeAvailableWaiterAcceptors())
                .overrideConfiguration(volumeAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.volumeDeletedWaiter = AsyncWaiter.builder(DescribeVolumesResponse.class).acceptors(volumeDeletedWaiterAcceptors())
                .overrideConfiguration(volumeDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.volumeInUseWaiter = AsyncWaiter.builder(DescribeVolumesResponse.class).acceptors(volumeInUseWaiterAcceptors())
                .overrideConfiguration(volumeInUseWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpcAvailableWaiter = AsyncWaiter.builder(DescribeVpcsResponse.class).acceptors(vpcAvailableWaiterAcceptors())
                .overrideConfiguration(vpcAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpcExistsWaiter = AsyncWaiter.builder(DescribeVpcsResponse.class).acceptors(vpcExistsWaiterAcceptors())
                .overrideConfiguration(vpcExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpnConnectionAvailableWaiter = AsyncWaiter.builder(DescribeVpnConnectionsResponse.class)
                .acceptors(vpnConnectionAvailableWaiterAcceptors())
                .overrideConfiguration(vpnConnectionAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpnConnectionDeletedWaiter = AsyncWaiter.builder(DescribeVpnConnectionsResponse.class)
                .acceptors(vpnConnectionDeletedWaiterAcceptors())
                .overrideConfiguration(vpnConnectionDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpcPeeringConnectionExistsWaiter = AsyncWaiter.builder(DescribeVpcPeeringConnectionsResponse.class)
                .acceptors(vpcPeeringConnectionExistsWaiterAcceptors())
                .overrideConfiguration(vpcPeeringConnectionExistsWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.vpcPeeringConnectionDeletedWaiter = AsyncWaiter.builder(DescribeVpcPeeringConnectionsResponse.class)
                .acceptors(vpcPeeringConnectionDeletedWaiterAcceptors())
                .overrideConfiguration(vpcPeeringConnectionDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBundleTasksResponse>> waitUntilBundleTaskComplete(
            DescribeBundleTasksRequest describeBundleTasksRequest) {
        return bundleTaskCompleteWaiter.runAsync(() -> client
                .describeBundleTasks(applyWaitersUserAgent(describeBundleTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeBundleTasksResponse>> waitUntilBundleTaskComplete(
            DescribeBundleTasksRequest describeBundleTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return bundleTaskCompleteWaiter.runAsync(
                () -> client.describeBundleTasks(applyWaitersUserAgent(describeBundleTasksRequest)),
                bundleTaskCompleteWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskCancelled(
            DescribeConversionTasksRequest describeConversionTasksRequest) {
        return conversionTaskCancelledWaiter.runAsync(() -> client
                .describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskCancelled(
            DescribeConversionTasksRequest describeConversionTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return conversionTaskCancelledWaiter.runAsync(
                () -> client.describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)),
                conversionTaskCancelledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskCompleted(
            DescribeConversionTasksRequest describeConversionTasksRequest) {
        return conversionTaskCompletedWaiter.runAsync(() -> client
                .describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskCompleted(
            DescribeConversionTasksRequest describeConversionTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return conversionTaskCompletedWaiter.runAsync(
                () -> client.describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)),
                conversionTaskCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskDeleted(
            DescribeConversionTasksRequest describeConversionTasksRequest) {
        return conversionTaskDeletedWaiter.runAsync(() -> client
                .describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeConversionTasksResponse>> waitUntilConversionTaskDeleted(
            DescribeConversionTasksRequest describeConversionTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return conversionTaskDeletedWaiter.runAsync(
                () -> client.describeConversionTasks(applyWaitersUserAgent(describeConversionTasksRequest)),
                conversionTaskDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCustomerGatewaysResponse>> waitUntilCustomerGatewayAvailable(
            DescribeCustomerGatewaysRequest describeCustomerGatewaysRequest) {
        return customerGatewayAvailableWaiter.runAsync(() -> client
                .describeCustomerGateways(applyWaitersUserAgent(describeCustomerGatewaysRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCustomerGatewaysResponse>> waitUntilCustomerGatewayAvailable(
            DescribeCustomerGatewaysRequest describeCustomerGatewaysRequest, WaiterOverrideConfiguration overrideConfig) {
        return customerGatewayAvailableWaiter.runAsync(
                () -> client.describeCustomerGateways(applyWaitersUserAgent(describeCustomerGatewaysRequest)),
                customerGatewayAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportTasksResponse>> waitUntilExportTaskCancelled(
            DescribeExportTasksRequest describeExportTasksRequest) {
        return exportTaskCancelledWaiter.runAsync(() -> client
                .describeExportTasks(applyWaitersUserAgent(describeExportTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportTasksResponse>> waitUntilExportTaskCancelled(
            DescribeExportTasksRequest describeExportTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return exportTaskCancelledWaiter.runAsync(
                () -> client.describeExportTasks(applyWaitersUserAgent(describeExportTasksRequest)),
                exportTaskCancelledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportTasksResponse>> waitUntilExportTaskCompleted(
            DescribeExportTasksRequest describeExportTasksRequest) {
        return exportTaskCompletedWaiter.runAsync(() -> client
                .describeExportTasks(applyWaitersUserAgent(describeExportTasksRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeExportTasksResponse>> waitUntilExportTaskCompleted(
            DescribeExportTasksRequest describeExportTasksRequest, WaiterOverrideConfiguration overrideConfig) {
        return exportTaskCompletedWaiter.runAsync(
                () -> client.describeExportTasks(applyWaitersUserAgent(describeExportTasksRequest)),
                exportTaskCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImagesResponse>> waitUntilImageAvailable(
            DescribeImagesRequest describeImagesRequest) {
        return imageAvailableWaiter.runAsync(() -> client.describeImages(applyWaitersUserAgent(describeImagesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImagesResponse>> waitUntilImageAvailable(
            DescribeImagesRequest describeImagesRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageAvailableWaiter.runAsync(() -> client.describeImages(applyWaitersUserAgent(describeImagesRequest)),
                imageAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImagesResponse>> waitUntilImageExists(
            DescribeImagesRequest describeImagesRequest) {
        return imageExistsWaiter.runAsync(() -> client.describeImages(applyWaitersUserAgent(describeImagesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeImagesResponse>> waitUntilImageExists(
            DescribeImagesRequest describeImagesRequest, WaiterOverrideConfiguration overrideConfig) {
        return imageExistsWaiter.runAsync(() -> client.describeImages(applyWaitersUserAgent(describeImagesRequest)),
                imageExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceExists(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceExistsWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceExists(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceExistsWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceRunning(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceRunningWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceRunning(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceRunningWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceRunningWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstanceStatusResponse>> waitUntilInstanceStatusOk(
            DescribeInstanceStatusRequest describeInstanceStatusRequest) {
        return instanceStatusOkWaiter.runAsync(() -> client
                .describeInstanceStatus(applyWaitersUserAgent(describeInstanceStatusRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstanceStatusResponse>> waitUntilInstanceStatusOk(
            DescribeInstanceStatusRequest describeInstanceStatusRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceStatusOkWaiter.runAsync(
                () -> client.describeInstanceStatus(applyWaitersUserAgent(describeInstanceStatusRequest)),
                instanceStatusOkWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceStopped(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceStoppedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceStopped(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceStoppedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceStoppedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceTerminated(
            DescribeInstancesRequest describeInstancesRequest) {
        return instanceTerminatedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstancesResponse>> waitUntilInstanceTerminated(
            DescribeInstancesRequest describeInstancesRequest, WaiterOverrideConfiguration overrideConfig) {
        return instanceTerminatedWaiter.runAsync(() -> client.describeInstances(applyWaitersUserAgent(describeInstancesRequest)),
                instanceTerminatedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeKeyPairsResponse>> waitUntilKeyPairExists(
            DescribeKeyPairsRequest describeKeyPairsRequest) {
        return keyPairExistsWaiter.runAsync(() -> client.describeKeyPairs(applyWaitersUserAgent(describeKeyPairsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeKeyPairsResponse>> waitUntilKeyPairExists(
            DescribeKeyPairsRequest describeKeyPairsRequest, WaiterOverrideConfiguration overrideConfig) {
        return keyPairExistsWaiter.runAsync(() -> client.describeKeyPairs(applyWaitersUserAgent(describeKeyPairsRequest)),
                keyPairExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNatGatewaysResponse>> waitUntilNatGatewayAvailable(
            DescribeNatGatewaysRequest describeNatGatewaysRequest) {
        return natGatewayAvailableWaiter.runAsync(() -> client
                .describeNatGateways(applyWaitersUserAgent(describeNatGatewaysRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNatGatewaysResponse>> waitUntilNatGatewayAvailable(
            DescribeNatGatewaysRequest describeNatGatewaysRequest, WaiterOverrideConfiguration overrideConfig) {
        return natGatewayAvailableWaiter.runAsync(
                () -> client.describeNatGateways(applyWaitersUserAgent(describeNatGatewaysRequest)),
                natGatewayAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNetworkInterfacesResponse>> waitUntilNetworkInterfaceAvailable(
            DescribeNetworkInterfacesRequest describeNetworkInterfacesRequest) {
        return networkInterfaceAvailableWaiter.runAsync(() -> client
                .describeNetworkInterfaces(applyWaitersUserAgent(describeNetworkInterfacesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeNetworkInterfacesResponse>> waitUntilNetworkInterfaceAvailable(
            DescribeNetworkInterfacesRequest describeNetworkInterfacesRequest, WaiterOverrideConfiguration overrideConfig) {
        return networkInterfaceAvailableWaiter.runAsync(
                () -> client.describeNetworkInterfaces(applyWaitersUserAgent(describeNetworkInterfacesRequest)),
                networkInterfaceAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPasswordDataResponse>> waitUntilPasswordDataAvailable(
            GetPasswordDataRequest getPasswordDataRequest) {
        return passwordDataAvailableWaiter.runAsync(() -> client.getPasswordData(applyWaitersUserAgent(getPasswordDataRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<GetPasswordDataResponse>> waitUntilPasswordDataAvailable(
            GetPasswordDataRequest getPasswordDataRequest, WaiterOverrideConfiguration overrideConfig) {
        return passwordDataAvailableWaiter.runAsync(() -> client.getPasswordData(applyWaitersUserAgent(getPasswordDataRequest)),
                passwordDataAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSecurityGroupsResponse>> waitUntilSecurityGroupExists(
            DescribeSecurityGroupsRequest describeSecurityGroupsRequest) {
        return securityGroupExistsWaiter.runAsync(() -> client
                .describeSecurityGroups(applyWaitersUserAgent(describeSecurityGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSecurityGroupsResponse>> waitUntilSecurityGroupExists(
            DescribeSecurityGroupsRequest describeSecurityGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return securityGroupExistsWaiter.runAsync(
                () -> client.describeSecurityGroups(applyWaitersUserAgent(describeSecurityGroupsRequest)),
                securityGroupExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSnapshotsResponse>> waitUntilSnapshotCompleted(
            DescribeSnapshotsRequest describeSnapshotsRequest) {
        return snapshotCompletedWaiter.runAsync(() -> client.describeSnapshots(applyWaitersUserAgent(describeSnapshotsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSnapshotsResponse>> waitUntilSnapshotCompleted(
            DescribeSnapshotsRequest describeSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        return snapshotCompletedWaiter.runAsync(() -> client.describeSnapshots(applyWaitersUserAgent(describeSnapshotsRequest)),
                snapshotCompletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSpotInstanceRequestsResponse>> waitUntilSpotInstanceRequestFulfilled(
            DescribeSpotInstanceRequestsRequest describeSpotInstanceRequestsRequest) {
        return spotInstanceRequestFulfilledWaiter.runAsync(() -> client
                .describeSpotInstanceRequests(applyWaitersUserAgent(describeSpotInstanceRequestsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSpotInstanceRequestsResponse>> waitUntilSpotInstanceRequestFulfilled(
            DescribeSpotInstanceRequestsRequest describeSpotInstanceRequestsRequest, WaiterOverrideConfiguration overrideConfig) {
        return spotInstanceRequestFulfilledWaiter.runAsync(
                () -> client.describeSpotInstanceRequests(applyWaitersUserAgent(describeSpotInstanceRequestsRequest)),
                spotInstanceRequestFulfilledWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSubnetsResponse>> waitUntilSubnetAvailable(
            DescribeSubnetsRequest describeSubnetsRequest) {
        return subnetAvailableWaiter.runAsync(() -> client.describeSubnets(applyWaitersUserAgent(describeSubnetsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeSubnetsResponse>> waitUntilSubnetAvailable(
            DescribeSubnetsRequest describeSubnetsRequest, WaiterOverrideConfiguration overrideConfig) {
        return subnetAvailableWaiter.runAsync(() -> client.describeSubnets(applyWaitersUserAgent(describeSubnetsRequest)),
                subnetAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstanceStatusResponse>> waitUntilSystemStatusOk(
            DescribeInstanceStatusRequest describeInstanceStatusRequest) {
        return systemStatusOkWaiter.runAsync(() -> client
                .describeInstanceStatus(applyWaitersUserAgent(describeInstanceStatusRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeInstanceStatusResponse>> waitUntilSystemStatusOk(
            DescribeInstanceStatusRequest describeInstanceStatusRequest, WaiterOverrideConfiguration overrideConfig) {
        return systemStatusOkWaiter.runAsync(
                () -> client.describeInstanceStatus(applyWaitersUserAgent(describeInstanceStatusRequest)),
                systemStatusOkWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeAvailable(
            DescribeVolumesRequest describeVolumesRequest) {
        return volumeAvailableWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeAvailable(
            DescribeVolumesRequest describeVolumesRequest, WaiterOverrideConfiguration overrideConfig) {
        return volumeAvailableWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)),
                volumeAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeDeleted(
            DescribeVolumesRequest describeVolumesRequest) {
        return volumeDeletedWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeDeleted(
            DescribeVolumesRequest describeVolumesRequest, WaiterOverrideConfiguration overrideConfig) {
        return volumeDeletedWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)),
                volumeDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeInUse(
            DescribeVolumesRequest describeVolumesRequest) {
        return volumeInUseWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVolumesResponse>> waitUntilVolumeInUse(
            DescribeVolumesRequest describeVolumesRequest, WaiterOverrideConfiguration overrideConfig) {
        return volumeInUseWaiter.runAsync(() -> client.describeVolumes(applyWaitersUserAgent(describeVolumesRequest)),
                volumeInUseWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcsResponse>> waitUntilVpcAvailable(DescribeVpcsRequest describeVpcsRequest) {
        return vpcAvailableWaiter.runAsync(() -> client.describeVpcs(applyWaitersUserAgent(describeVpcsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcsResponse>> waitUntilVpcAvailable(DescribeVpcsRequest describeVpcsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return vpcAvailableWaiter.runAsync(() -> client.describeVpcs(applyWaitersUserAgent(describeVpcsRequest)),
                vpcAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcsResponse>> waitUntilVpcExists(DescribeVpcsRequest describeVpcsRequest) {
        return vpcExistsWaiter.runAsync(() -> client.describeVpcs(applyWaitersUserAgent(describeVpcsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcsResponse>> waitUntilVpcExists(DescribeVpcsRequest describeVpcsRequest,
            WaiterOverrideConfiguration overrideConfig) {
        return vpcExistsWaiter.runAsync(() -> client.describeVpcs(applyWaitersUserAgent(describeVpcsRequest)),
                vpcExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcPeeringConnectionsResponse>> waitUntilVpcPeeringConnectionDeleted(
            DescribeVpcPeeringConnectionsRequest describeVpcPeeringConnectionsRequest) {
        return vpcPeeringConnectionDeletedWaiter.runAsync(() -> client
                .describeVpcPeeringConnections(applyWaitersUserAgent(describeVpcPeeringConnectionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcPeeringConnectionsResponse>> waitUntilVpcPeeringConnectionDeleted(
            DescribeVpcPeeringConnectionsRequest describeVpcPeeringConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return vpcPeeringConnectionDeletedWaiter.runAsync(
                () -> client.describeVpcPeeringConnections(applyWaitersUserAgent(describeVpcPeeringConnectionsRequest)),
                vpcPeeringConnectionDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcPeeringConnectionsResponse>> waitUntilVpcPeeringConnectionExists(
            DescribeVpcPeeringConnectionsRequest describeVpcPeeringConnectionsRequest) {
        return vpcPeeringConnectionExistsWaiter.runAsync(() -> client
                .describeVpcPeeringConnections(applyWaitersUserAgent(describeVpcPeeringConnectionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpcPeeringConnectionsResponse>> waitUntilVpcPeeringConnectionExists(
            DescribeVpcPeeringConnectionsRequest describeVpcPeeringConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return vpcPeeringConnectionExistsWaiter.runAsync(
                () -> client.describeVpcPeeringConnections(applyWaitersUserAgent(describeVpcPeeringConnectionsRequest)),
                vpcPeeringConnectionExistsWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpnConnectionsResponse>> waitUntilVpnConnectionAvailable(
            DescribeVpnConnectionsRequest describeVpnConnectionsRequest) {
        return vpnConnectionAvailableWaiter.runAsync(() -> client
                .describeVpnConnections(applyWaitersUserAgent(describeVpnConnectionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpnConnectionsResponse>> waitUntilVpnConnectionAvailable(
            DescribeVpnConnectionsRequest describeVpnConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return vpnConnectionAvailableWaiter.runAsync(
                () -> client.describeVpnConnections(applyWaitersUserAgent(describeVpnConnectionsRequest)),
                vpnConnectionAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpnConnectionsResponse>> waitUntilVpnConnectionDeleted(
            DescribeVpnConnectionsRequest describeVpnConnectionsRequest) {
        return vpnConnectionDeletedWaiter.runAsync(() -> client
                .describeVpnConnections(applyWaitersUserAgent(describeVpnConnectionsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeVpnConnectionsResponse>> waitUntilVpnConnectionDeleted(
            DescribeVpnConnectionsRequest describeVpnConnectionsRequest, WaiterOverrideConfiguration overrideConfig) {
        return vpnConnectionDeletedWaiter.runAsync(
                () -> client.describeVpnConnections(applyWaitersUserAgent(describeVpnConnectionsRequest)),
                vpnConnectionDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Reservations").flatten().length().compare(">", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "InvalidInstanceID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeBundleTasksResponse>> bundleTaskCompleteWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeBundleTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("BundleTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "complete"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("BundleTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeConversionTasksResponse>> conversionTaskCancelledWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeConversionTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ConversionTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "cancelled"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeConversionTasksResponse>> conversionTaskCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeConversionTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ConversionTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "completed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ConversionTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "cancelled"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ConversionTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "cancelling"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeConversionTasksResponse>> conversionTaskDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeConversionTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ConversionTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCustomerGatewaysResponse>> customerGatewayAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCustomerGatewaysResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CustomerGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CustomerGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("CustomerGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeExportTasksResponse>> exportTaskCancelledWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeExportTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ExportTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "cancelled"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeExportTasksResponse>> exportTaskCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeExportTasksResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("ExportTasks").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "completed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImagesResponse>> imageExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImagesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("Images").flatten().length().compare(">", input.constant(0)).value(), true);
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidAMIID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeImagesResponse>> imageAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeImagesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Images").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Images").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceRunningWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "running"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "shutting-down"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "InvalidInstanceID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstanceStatusResponse>> instanceStatusOkWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstanceStatusResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("InstanceStatuses").flatten().field("InstanceStatus").field("Status")
                    .values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "ok"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects
                .equals(errorCode(error), "InvalidInstanceID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceStoppedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "stopped"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstancesResponse>> instanceTerminatedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstancesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "terminated"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Reservations").flatten().field("Instances").flatten().field("State")
                    .field("Name").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "stopping"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeKeyPairsResponse>> keyPairExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeKeyPairsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("KeyPairs").flatten().field("KeyName").length().compare(">", input.constant(0))
                    .value(), true);
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidKeyPair.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNatGatewaysResponse>> natGatewayAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNatGatewaysResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("NatGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("NatGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "failed"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("NatGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("NatGateways").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "NatGatewayNotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeNetworkInterfacesResponse>> networkInterfaceAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeNetworkInterfacesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("NetworkInterfaces").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InvalidNetworkInterfaceID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super GetPasswordDataResponse>> passwordDataAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super GetPasswordDataResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(input.field("PasswordData").length().compare(">", input.constant(0)).value(), true);
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeSnapshotsResponse>> snapshotCompletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeSnapshotsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Snapshots").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "completed"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeSecurityGroupsResponse>> securityGroupExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeSecurityGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            return Objects.equals(
                    input.field("SecurityGroups").flatten().field("GroupId").length().compare(">", input.constant(0)).value(),
                    true);
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidGroup.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeSpotInstanceRequestsResponse>> spotInstanceRequestFulfilledWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeSpotInstanceRequestsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "fulfilled"));
        }));
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().allMatch(v -> Objects.equals(v, "request-canceled-and-instance-running"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "schedule-expired"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty()
                    && resultValues.stream().anyMatch(v -> Objects.equals(v, "canceled-before-fulfillment"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "bad-parameters"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("SpotInstanceRequests").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "system-error"));
        }));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InvalidSpotInstanceRequestID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeSubnetsResponse>> subnetAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeSubnetsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Subnets").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeInstanceStatusResponse>> systemStatusOkWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeInstanceStatusResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("InstanceStatuses").flatten().field("SystemStatus").field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "ok"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVolumesResponse>> volumeAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVolumesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Volumes").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Volumes").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVolumesResponse>> volumeDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVolumesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Volumes").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidVolume.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVolumesResponse>> volumeInUseWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVolumesResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Volumes").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "in-use"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Volumes").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpcsResponse>> vpcAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpcsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("Vpcs").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpcsResponse>> vpcExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpcsResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "InvalidVpcID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpnConnectionsResponse>> vpnConnectionAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpnConnectionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpnConnections").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpnConnections").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpnConnections").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpnConnectionsResponse>> vpnConnectionDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpnConnectionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpnConnections").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpnConnections").flatten().field("State").values();
            return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "pending"));
        }));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpcPeeringConnectionsResponse>> vpcPeeringConnectionExistsWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpcPeeringConnectionsResponse>> result = new ArrayList<>();
        result.add(new WaitersRuntime.ResponseStatusAcceptor(200, WaiterState.SUCCESS));
        result.add(WaiterAcceptor.retryOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InvalidVpcPeeringConnectionID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeVpcPeeringConnectionsResponse>> vpcPeeringConnectionDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeVpcPeeringConnectionsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            WaitersRuntime.Value input = new WaitersRuntime.Value(response);
            List<Object> resultValues = input.field("VpcPeeringConnections").flatten().field("Status").field("Code").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "InvalidVpcPeeringConnectionID.NotFound")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration instanceExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration bundleTaskCompleteWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration conversionTaskCancelledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration conversionTaskCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration conversionTaskDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration customerGatewayAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration exportTaskCancelledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration exportTaskCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration imageAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceRunningWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceStatusOkWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceStoppedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration instanceTerminatedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration keyPairExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(6);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration natGatewayAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration networkInterfaceAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(10);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(20)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration passwordDataAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration snapshotCompletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration securityGroupExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(6);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(5)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration spotInstanceRequestFulfilledWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration subnetAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration systemStatusOkWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration volumeAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration volumeDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration volumeInUseWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpcAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpcExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(5);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(1)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpnConnectionAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpnConnectionDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpcPeeringConnectionExistsWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration vpcPeeringConnectionDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategy).orElse(
                FixedDelayBackoffStrategy.create(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategy(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static Ec2AsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends Ec2Request> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version("waiter").name("hll").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements Ec2AsyncWaiter.Builder {
        private Ec2AsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public Ec2AsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public Ec2AsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public Ec2AsyncWaiter.Builder client(Ec2AsyncClient client) {
            this.client = client;
            return this;
        }

        public Ec2AsyncWaiter build() {
            return new DefaultEc2AsyncWaiter(this);
        }
    }
}
