/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about a Capacity Reservation Fleet.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityReservationFleet implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityReservationFleet.Builder, CapacityReservationFleet> {
    private static final SdkField<String> CAPACITY_RESERVATION_FLEET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityReservationFleetId")
            .getter(getter(CapacityReservationFleet::capacityReservationFleetId))
            .setter(setter(Builder::capacityReservationFleetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationFleetId")
                    .unmarshallLocationName("capacityReservationFleetId").build()).build();

    private static final SdkField<String> CAPACITY_RESERVATION_FLEET_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CapacityReservationFleetArn")
            .getter(getter(CapacityReservationFleet::capacityReservationFleetArn))
            .setter(setter(Builder::capacityReservationFleetArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CapacityReservationFleetArn")
                    .unmarshallLocationName("capacityReservationFleetArn").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(CapacityReservationFleet::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<Integer> TOTAL_TARGET_CAPACITY_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("TotalTargetCapacity")
            .getter(getter(CapacityReservationFleet::totalTargetCapacity))
            .setter(setter(Builder::totalTargetCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalTargetCapacity")
                    .unmarshallLocationName("totalTargetCapacity").build()).build();

    private static final SdkField<Double> TOTAL_FULFILLED_CAPACITY_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("TotalFulfilledCapacity")
            .getter(getter(CapacityReservationFleet::totalFulfilledCapacity))
            .setter(setter(Builder::totalFulfilledCapacity))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalFulfilledCapacity")
                    .unmarshallLocationName("totalFulfilledCapacity").build()).build();

    private static final SdkField<String> TENANCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Tenancy")
            .getter(getter(CapacityReservationFleet::tenancyAsString))
            .setter(setter(Builder::tenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tenancy")
                    .unmarshallLocationName("tenancy").build()).build();

    private static final SdkField<Instant> END_DATE_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndDate")
            .getter(getter(CapacityReservationFleet::endDate))
            .setter(setter(Builder::endDate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndDate")
                    .unmarshallLocationName("endDate").build()).build();

    private static final SdkField<Instant> CREATE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreateTime")
            .getter(getter(CapacityReservationFleet::createTime))
            .setter(setter(Builder::createTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreateTime")
                    .unmarshallLocationName("createTime").build()).build();

    private static final SdkField<String> INSTANCE_MATCH_CRITERIA_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceMatchCriteria")
            .getter(getter(CapacityReservationFleet::instanceMatchCriteriaAsString))
            .setter(setter(Builder::instanceMatchCriteria))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceMatchCriteria")
                    .unmarshallLocationName("instanceMatchCriteria").build()).build();

    private static final SdkField<String> ALLOCATION_STRATEGY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllocationStrategy")
            .getter(getter(CapacityReservationFleet::allocationStrategy))
            .setter(setter(Builder::allocationStrategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationStrategy")
                    .unmarshallLocationName("allocationStrategy").build()).build();

    private static final SdkField<List<FleetCapacityReservation>> INSTANCE_TYPE_SPECIFICATIONS_FIELD = SdkField
            .<List<FleetCapacityReservation>> builder(MarshallingType.LIST)
            .memberName("InstanceTypeSpecifications")
            .getter(getter(CapacityReservationFleet::instanceTypeSpecifications))
            .setter(setter(Builder::instanceTypeSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTypeSpecificationSet")
                    .unmarshallLocationName("instanceTypeSpecificationSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<FleetCapacityReservation> builder(MarshallingType.SDK_POJO)
                                            .constructor(FleetCapacityReservation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(CapacityReservationFleet::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CAPACITY_RESERVATION_FLEET_ID_FIELD, CAPACITY_RESERVATION_FLEET_ARN_FIELD, STATE_FIELD, TOTAL_TARGET_CAPACITY_FIELD,
            TOTAL_FULFILLED_CAPACITY_FIELD, TENANCY_FIELD, END_DATE_FIELD, CREATE_TIME_FIELD, INSTANCE_MATCH_CRITERIA_FIELD,
            ALLOCATION_STRATEGY_FIELD, INSTANCE_TYPE_SPECIFICATIONS_FIELD, TAGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String capacityReservationFleetId;

    private final String capacityReservationFleetArn;

    private final String state;

    private final Integer totalTargetCapacity;

    private final Double totalFulfilledCapacity;

    private final String tenancy;

    private final Instant endDate;

    private final Instant createTime;

    private final String instanceMatchCriteria;

    private final String allocationStrategy;

    private final List<FleetCapacityReservation> instanceTypeSpecifications;

    private final List<Tag> tags;

    private CapacityReservationFleet(BuilderImpl builder) {
        this.capacityReservationFleetId = builder.capacityReservationFleetId;
        this.capacityReservationFleetArn = builder.capacityReservationFleetArn;
        this.state = builder.state;
        this.totalTargetCapacity = builder.totalTargetCapacity;
        this.totalFulfilledCapacity = builder.totalFulfilledCapacity;
        this.tenancy = builder.tenancy;
        this.endDate = builder.endDate;
        this.createTime = builder.createTime;
        this.instanceMatchCriteria = builder.instanceMatchCriteria;
        this.allocationStrategy = builder.allocationStrategy;
        this.instanceTypeSpecifications = builder.instanceTypeSpecifications;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the Capacity Reservation Fleet.
     * </p>
     * 
     * @return The ID of the Capacity Reservation Fleet.
     */
    public final String capacityReservationFleetId() {
        return capacityReservationFleetId;
    }

    /**
     * <p>
     * The ARN of the Capacity Reservation Fleet.
     * </p>
     * 
     * @return The ARN of the Capacity Reservation Fleet.
     */
    public final String capacityReservationFleetArn() {
        return capacityReservationFleetArn;
    }

    /**
     * <p>
     * The state of the Capacity Reservation Fleet. Possible states include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic Compute
     * Cloud is preparing to create the Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this state until
     * the modification is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is attempting
     * to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total target
     * capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is attempting to
     * asynchronously fulfill its total target capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
     * expiring. One or more of its Capacity reservations might still be active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
     * Reservations are expired. The Fleet can't create new Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or more of its
     * Capacity reservations might still be active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its Capacity
     * Reservations are cancelled and the Fleet can't create new Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified instance types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CapacityReservationFleetState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Capacity Reservation Fleet. Possible states include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic
     *         Compute Cloud is preparing to create the Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this
     *         state until the modification is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
     *         attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total
     *         target capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is
     *         attempting to asynchronously fulfill its total target capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
     *         expiring. One or more of its Capacity reservations might still be active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
     *         Reservations are expired. The Fleet can't create new Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or
     *         more of its Capacity reservations might still be active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
     *         Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified
     *         instance types.
     *         </p>
     *         </li>
     * @see CapacityReservationFleetState
     */
    public final CapacityReservationFleetState state() {
        return CapacityReservationFleetState.fromValue(state);
    }

    /**
     * <p>
     * The state of the Capacity Reservation Fleet. Possible states include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic Compute
     * Cloud is preparing to create the Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this state until
     * the modification is complete.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is attempting
     * to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total target
     * capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is attempting to
     * asynchronously fulfill its total target capacity.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
     * expiring. One or more of its Capacity reservations might still be active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
     * Reservations are expired. The Fleet can't create new Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or more of its
     * Capacity reservations might still be active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its Capacity
     * Reservations are cancelled and the Fleet can't create new Capacity Reservations.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified instance types.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link CapacityReservationFleetState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the Capacity Reservation Fleet. Possible states include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic
     *         Compute Cloud is preparing to create the Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this
     *         state until the modification is complete.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
     *         attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total
     *         target capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is
     *         attempting to asynchronously fulfill its total target capacity.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
     *         expiring. One or more of its Capacity reservations might still be active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
     *         Reservations are expired. The Fleet can't create new Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or
     *         more of its Capacity reservations might still be active.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
     *         Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified
     *         instance types.
     *         </p>
     *         </li>
     * @see CapacityReservationFleetState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The total number of capacity units for which the Capacity Reservation Fleet reserves capacity. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total target
     * capacity</a> in the Amazon EC2 User Guide.
     * </p>
     * 
     * @return The total number of capacity units for which the Capacity Reservation Fleet reserves capacity. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total
     *         target capacity</a> in the Amazon EC2 User Guide.
     */
    public final Integer totalTargetCapacity() {
        return totalTargetCapacity;
    }

    /**
     * <p>
     * The capacity units that have been fulfilled.
     * </p>
     * 
     * @return The capacity units that have been fulfilled.
     */
    public final Double totalFulfilledCapacity() {
        return totalFulfilledCapacity;
    }

    /**
     * <p>
     * The tenancy of the Capacity Reservation Fleet. Tenancies include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon Web
     * Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is dedicated to
     * a single Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link FleetCapacityReservationTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #tenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the Capacity Reservation Fleet. Tenancies include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
     *         Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
     *         dedicated to a single Amazon Web Services account.
     *         </p>
     *         </li>
     * @see FleetCapacityReservationTenancy
     */
    public final FleetCapacityReservationTenancy tenancy() {
        return FleetCapacityReservationTenancy.fromValue(tenancy);
    }

    /**
     * <p>
     * The tenancy of the Capacity Reservation Fleet. Tenancies include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon Web
     * Services accounts.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is dedicated to
     * a single Amazon Web Services account.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link FleetCapacityReservationTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #tenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the Capacity Reservation Fleet. Tenancies include:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
     *         Amazon Web Services accounts.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
     *         dedicated to a single Amazon Web Services account.
     *         </p>
     *         </li>
     * @see FleetCapacityReservationTenancy
     */
    public final String tenancyAsString() {
        return tenancy;
    }

    /**
     * <p>
     * The date and time at which the Capacity Reservation Fleet expires.
     * </p>
     * 
     * @return The date and time at which the Capacity Reservation Fleet expires.
     */
    public final Instant endDate() {
        return endDate;
    }

    /**
     * <p>
     * The date and time at which the Capacity Reservation Fleet was created.
     * </p>
     * 
     * @return The date and time at which the Capacity Reservation Fleet was created.
     */
    public final Instant createTime() {
        return createTime;
    }

    /**
     * <p>
     * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in
     * the Fleet inherit this instance matching criteria.
     * </p>
     * <p>
     * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means that
     * instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity
     * Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its
     * reserved capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceMatchCriteria} will return {@link FleetInstanceMatchCriteria#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceMatchCriteriaAsString}.
     * </p>
     * 
     * @return Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
     *         Reservations in the Fleet inherit this instance matching criteria.</p>
     *         <p>
     *         Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
     *         means that instances that have matching attributes (instance type, platform, and Availability Zone) run
     *         in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
     *         Reservation Fleet to use its reserved capacity.
     * @see FleetInstanceMatchCriteria
     */
    public final FleetInstanceMatchCriteria instanceMatchCriteria() {
        return FleetInstanceMatchCriteria.fromValue(instanceMatchCriteria);
    }

    /**
     * <p>
     * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity Reservations in
     * the Fleet inherit this instance matching criteria.
     * </p>
     * <p>
     * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means that
     * instances that have matching attributes (instance type, platform, and Availability Zone) run in the Capacity
     * Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet to use its
     * reserved capacity.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceMatchCriteria} will return {@link FleetInstanceMatchCriteria#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceMatchCriteriaAsString}.
     * </p>
     * 
     * @return Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
     *         Reservations in the Fleet inherit this instance matching criteria.</p>
     *         <p>
     *         Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
     *         means that instances that have matching attributes (instance type, platform, and Availability Zone) run
     *         in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
     *         Reservation Fleet to use its reserved capacity.
     * @see FleetInstanceMatchCriteria
     */
    public final String instanceMatchCriteriaAsString() {
        return instanceMatchCriteria;
    }

    /**
     * <p>
     * The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to use.
     * For more information, see For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy"> Allocation
     * strategy</a> in the Amazon EC2 User Guide.
     * </p>
     * 
     * @return The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to
     *         use. For more information, see For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
     *         Allocation strategy</a> in the Amazon EC2 User Guide.
     */
    public final String allocationStrategy() {
        return allocationStrategy;
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceTypeSpecifications property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasInstanceTypeSpecifications() {
        return instanceTypeSpecifications != null && !(instanceTypeSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the instance types for which to reserve the capacity.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceTypeSpecifications} method.
     * </p>
     * 
     * @return Information about the instance types for which to reserve the capacity.
     */
    public final List<FleetCapacityReservation> instanceTypeSpecifications() {
        return instanceTypeSpecifications;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the Capacity Reservation Fleet.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the Capacity Reservation Fleet.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservationFleetId());
        hashCode = 31 * hashCode + Objects.hashCode(capacityReservationFleetArn());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(totalTargetCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(totalFulfilledCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(tenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(endDate());
        hashCode = 31 * hashCode + Objects.hashCode(createTime());
        hashCode = 31 * hashCode + Objects.hashCode(instanceMatchCriteriaAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allocationStrategy());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityReservationFleet)) {
            return false;
        }
        CapacityReservationFleet other = (CapacityReservationFleet) obj;
        return Objects.equals(capacityReservationFleetId(), other.capacityReservationFleetId())
                && Objects.equals(capacityReservationFleetArn(), other.capacityReservationFleetArn())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(totalTargetCapacity(), other.totalTargetCapacity())
                && Objects.equals(totalFulfilledCapacity(), other.totalFulfilledCapacity())
                && Objects.equals(tenancyAsString(), other.tenancyAsString()) && Objects.equals(endDate(), other.endDate())
                && Objects.equals(createTime(), other.createTime())
                && Objects.equals(instanceMatchCriteriaAsString(), other.instanceMatchCriteriaAsString())
                && Objects.equals(allocationStrategy(), other.allocationStrategy())
                && hasInstanceTypeSpecifications() == other.hasInstanceTypeSpecifications()
                && Objects.equals(instanceTypeSpecifications(), other.instanceTypeSpecifications())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityReservationFleet").add("CapacityReservationFleetId", capacityReservationFleetId())
                .add("CapacityReservationFleetArn", capacityReservationFleetArn()).add("State", stateAsString())
                .add("TotalTargetCapacity", totalTargetCapacity()).add("TotalFulfilledCapacity", totalFulfilledCapacity())
                .add("Tenancy", tenancyAsString()).add("EndDate", endDate()).add("CreateTime", createTime())
                .add("InstanceMatchCriteria", instanceMatchCriteriaAsString()).add("AllocationStrategy", allocationStrategy())
                .add("InstanceTypeSpecifications", hasInstanceTypeSpecifications() ? instanceTypeSpecifications() : null)
                .add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CapacityReservationFleetId":
            return Optional.ofNullable(clazz.cast(capacityReservationFleetId()));
        case "CapacityReservationFleetArn":
            return Optional.ofNullable(clazz.cast(capacityReservationFleetArn()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "TotalTargetCapacity":
            return Optional.ofNullable(clazz.cast(totalTargetCapacity()));
        case "TotalFulfilledCapacity":
            return Optional.ofNullable(clazz.cast(totalFulfilledCapacity()));
        case "Tenancy":
            return Optional.ofNullable(clazz.cast(tenancyAsString()));
        case "EndDate":
            return Optional.ofNullable(clazz.cast(endDate()));
        case "CreateTime":
            return Optional.ofNullable(clazz.cast(createTime()));
        case "InstanceMatchCriteria":
            return Optional.ofNullable(clazz.cast(instanceMatchCriteriaAsString()));
        case "AllocationStrategy":
            return Optional.ofNullable(clazz.cast(allocationStrategy()));
        case "InstanceTypeSpecifications":
            return Optional.ofNullable(clazz.cast(instanceTypeSpecifications()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CapacityReservationFleet, T> g) {
        return obj -> g.apply((CapacityReservationFleet) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityReservationFleet> {
        /**
         * <p>
         * The ID of the Capacity Reservation Fleet.
         * </p>
         * 
         * @param capacityReservationFleetId
         *        The ID of the Capacity Reservation Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationFleetId(String capacityReservationFleetId);

        /**
         * <p>
         * The ARN of the Capacity Reservation Fleet.
         * </p>
         * 
         * @param capacityReservationFleetArn
         *        The ARN of the Capacity Reservation Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capacityReservationFleetArn(String capacityReservationFleetArn);

        /**
         * <p>
         * The state of the Capacity Reservation Fleet. Possible states include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic Compute
         * Cloud is preparing to create the Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this state
         * until the modification is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
         * attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total target
         * capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is attempting to
         * asynchronously fulfill its total target capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
         * expiring. One or more of its Capacity reservations might still be active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
         * Reservations are expired. The Fleet can't create new Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or more of
         * its Capacity reservations might still be active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
         * Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified instance
         * types.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the Capacity Reservation Fleet. Possible states include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic
         *        Compute Cloud is preparing to create the Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this
         *        state until the modification is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
         *        attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total
         *        target capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is
         *        attempting to asynchronously fulfill its total target capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process
         *        of expiring. One or more of its Capacity reservations might still be active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its
         *        Capacity Reservations are expired. The Fleet can't create new Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or
         *        more of its Capacity reservations might still be active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
         *        Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified
         *        instance types.
         *        </p>
         *        </li>
         * @see CapacityReservationFleetState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityReservationFleetState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the Capacity Reservation Fleet. Possible states include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic Compute
         * Cloud is preparing to create the Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this state
         * until the modification is complete.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
         * attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total target
         * capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is attempting to
         * asynchronously fulfill its total target capacity.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process of
         * expiring. One or more of its Capacity reservations might still be active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its Capacity
         * Reservations are expired. The Fleet can't create new Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or more of
         * its Capacity reservations might still be active.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
         * Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified instance
         * types.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the Capacity Reservation Fleet. Possible states include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>submitted</code> - The Capacity Reservation Fleet request has been submitted and Amazon Elastic
         *        Compute Cloud is preparing to create the Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>modifying</code> - The Capacity Reservation Fleet is being modified. The Fleet remains in this
         *        state until the modification is complete.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>active</code> - The Capacity Reservation Fleet has fulfilled its total target capacity and it is
         *        attempting to maintain this capacity. The Fleet remains in this state until it is modified or deleted.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>partially_fulfilled</code> - The Capacity Reservation Fleet has partially fulfilled its total
         *        target capacity. There is insufficient Amazon EC2 to fulfill the total target capacity. The Fleet is
         *        attempting to asynchronously fulfill its total target capacity.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>expiring</code> - The Capacity Reservation Fleet has reach its end date and it is in the process
         *        of expiring. One or more of its Capacity reservations might still be active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>expired</code> - The Capacity Reservation Fleet has reach its end date. The Fleet and its
         *        Capacity Reservations are expired. The Fleet can't create new Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cancelling</code> - The Capacity Reservation Fleet is in the process of being cancelled. One or
         *        more of its Capacity reservations might still be active.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>cancelled</code> - The Capacity Reservation Fleet has been manually cancelled. The Fleet and its
         *        Capacity Reservations are cancelled and the Fleet can't create new Capacity Reservations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed</code> - The Capacity Reservation Fleet failed to reserve capacity for the specified
         *        instance types.
         *        </p>
         *        </li>
         * @see CapacityReservationFleetState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityReservationFleetState
         */
        Builder state(CapacityReservationFleetState state);

        /**
         * <p>
         * The total number of capacity units for which the Capacity Reservation Fleet reserves capacity. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total target
         * capacity</a> in the Amazon EC2 User Guide.
         * </p>
         * 
         * @param totalTargetCapacity
         *        The total number of capacity units for which the Capacity Reservation Fleet reserves capacity. For
         *        more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#target-capacity">Total
         *        target capacity</a> in the Amazon EC2 User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalTargetCapacity(Integer totalTargetCapacity);

        /**
         * <p>
         * The capacity units that have been fulfilled.
         * </p>
         * 
         * @param totalFulfilledCapacity
         *        The capacity units that have been fulfilled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalFulfilledCapacity(Double totalFulfilledCapacity);

        /**
         * <p>
         * The tenancy of the Capacity Reservation Fleet. Tenancies include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon
         * Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
         * dedicated to a single Amazon Web Services account.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tenancy
         *        The tenancy of the Capacity Reservation Fleet. Tenancies include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
         *        Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
         *        dedicated to a single Amazon Web Services account.
         *        </p>
         *        </li>
         * @see FleetCapacityReservationTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationTenancy
         */
        Builder tenancy(String tenancy);

        /**
         * <p>
         * The tenancy of the Capacity Reservation Fleet. Tenancies include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other Amazon
         * Web Services accounts.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
         * dedicated to a single Amazon Web Services account.
         * </p>
         * </li>
         * </ul>
         * 
         * @param tenancy
         *        The tenancy of the Capacity Reservation Fleet. Tenancies include:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>default</code> - The Capacity Reservation Fleet is created on hardware that is shared with other
         *        Amazon Web Services accounts.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>dedicated</code> - The Capacity Reservation Fleet is created on single-tenant hardware that is
         *        dedicated to a single Amazon Web Services account.
         *        </p>
         *        </li>
         * @see FleetCapacityReservationTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetCapacityReservationTenancy
         */
        Builder tenancy(FleetCapacityReservationTenancy tenancy);

        /**
         * <p>
         * The date and time at which the Capacity Reservation Fleet expires.
         * </p>
         * 
         * @param endDate
         *        The date and time at which the Capacity Reservation Fleet expires.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endDate(Instant endDate);

        /**
         * <p>
         * The date and time at which the Capacity Reservation Fleet was created.
         * </p>
         * 
         * @param createTime
         *        The date and time at which the Capacity Reservation Fleet was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createTime(Instant createTime);

        /**
         * <p>
         * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         * Reservations in the Fleet inherit this instance matching criteria.
         * </p>
         * <p>
         * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means
         * that instances that have matching attributes (instance type, platform, and Availability Zone) run in the
         * Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet
         * to use its reserved capacity.
         * </p>
         * 
         * @param instanceMatchCriteria
         *        Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         *        Reservations in the Fleet inherit this instance matching criteria.</p>
         *        <p>
         *        Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
         *        means that instances that have matching attributes (instance type, platform, and Availability Zone)
         *        run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
         *        Reservation Fleet to use its reserved capacity.
         * @see FleetInstanceMatchCriteria
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetInstanceMatchCriteria
         */
        Builder instanceMatchCriteria(String instanceMatchCriteria);

        /**
         * <p>
         * Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         * Reservations in the Fleet inherit this instance matching criteria.
         * </p>
         * <p>
         * Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This means
         * that instances that have matching attributes (instance type, platform, and Availability Zone) run in the
         * Capacity Reservations automatically. Instances do not need to explicitly target a Capacity Reservation Fleet
         * to use its reserved capacity.
         * </p>
         * 
         * @param instanceMatchCriteria
         *        Indicates the type of instance launches that the Capacity Reservation Fleet accepts. All Capacity
         *        Reservations in the Fleet inherit this instance matching criteria.</p>
         *        <p>
         *        Currently, Capacity Reservation Fleets support <code>open</code> instance matching criteria only. This
         *        means that instances that have matching attributes (instance type, platform, and Availability Zone)
         *        run in the Capacity Reservations automatically. Instances do not need to explicitly target a Capacity
         *        Reservation Fleet to use its reserved capacity.
         * @see FleetInstanceMatchCriteria
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FleetInstanceMatchCriteria
         */
        Builder instanceMatchCriteria(FleetInstanceMatchCriteria instanceMatchCriteria);

        /**
         * <p>
         * The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types to
         * use. For more information, see For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
         * Allocation strategy</a> in the Amazon EC2 User Guide.
         * </p>
         * 
         * @param allocationStrategy
         *        The strategy used by the Capacity Reservation Fleet to determine which of the specified instance types
         *        to use. For more information, see For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/crfleet-concepts.html#allocation-strategy">
         *        Allocation strategy</a> in the Amazon EC2 User Guide.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationStrategy(String allocationStrategy);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        Information about the instance types for which to reserve the capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(Collection<FleetCapacityReservation> instanceTypeSpecifications);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * 
         * @param instanceTypeSpecifications
         *        Information about the instance types for which to reserve the capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceTypeSpecifications(FleetCapacityReservation... instanceTypeSpecifications);

        /**
         * <p>
         * Information about the instance types for which to reserve the capacity.
         * </p>
         * This is a convenience that creates an instance of the {@link List<FleetCapacityReservation>.Builder} avoiding
         * the need to create one manually via {@link List<FleetCapacityReservation>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<FleetCapacityReservation>.Builder#build()} is called
         * immediately and its result is passed to {@link #instanceTypeSpecifications(List<FleetCapacityReservation>)}.
         * 
         * @param instanceTypeSpecifications
         *        a consumer that will call methods on {@link List<FleetCapacityReservation>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceTypeSpecifications(List<FleetCapacityReservation>)
         */
        Builder instanceTypeSpecifications(Consumer<FleetCapacityReservation.Builder>... instanceTypeSpecifications);

        /**
         * <p>
         * The tags assigned to the Capacity Reservation Fleet.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the Capacity Reservation Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the Capacity Reservation Fleet.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the Capacity Reservation Fleet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the Capacity Reservation Fleet.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String capacityReservationFleetId;

        private String capacityReservationFleetArn;

        private String state;

        private Integer totalTargetCapacity;

        private Double totalFulfilledCapacity;

        private String tenancy;

        private Instant endDate;

        private Instant createTime;

        private String instanceMatchCriteria;

        private String allocationStrategy;

        private List<FleetCapacityReservation> instanceTypeSpecifications = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityReservationFleet model) {
            capacityReservationFleetId(model.capacityReservationFleetId);
            capacityReservationFleetArn(model.capacityReservationFleetArn);
            state(model.state);
            totalTargetCapacity(model.totalTargetCapacity);
            totalFulfilledCapacity(model.totalFulfilledCapacity);
            tenancy(model.tenancy);
            endDate(model.endDate);
            createTime(model.createTime);
            instanceMatchCriteria(model.instanceMatchCriteria);
            allocationStrategy(model.allocationStrategy);
            instanceTypeSpecifications(model.instanceTypeSpecifications);
            tags(model.tags);
        }

        public final String getCapacityReservationFleetId() {
            return capacityReservationFleetId;
        }

        public final void setCapacityReservationFleetId(String capacityReservationFleetId) {
            this.capacityReservationFleetId = capacityReservationFleetId;
        }

        @Override
        @Transient
        public final Builder capacityReservationFleetId(String capacityReservationFleetId) {
            this.capacityReservationFleetId = capacityReservationFleetId;
            return this;
        }

        public final String getCapacityReservationFleetArn() {
            return capacityReservationFleetArn;
        }

        public final void setCapacityReservationFleetArn(String capacityReservationFleetArn) {
            this.capacityReservationFleetArn = capacityReservationFleetArn;
        }

        @Override
        @Transient
        public final Builder capacityReservationFleetArn(String capacityReservationFleetArn) {
            this.capacityReservationFleetArn = capacityReservationFleetArn;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(CapacityReservationFleetState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Integer getTotalTargetCapacity() {
            return totalTargetCapacity;
        }

        public final void setTotalTargetCapacity(Integer totalTargetCapacity) {
            this.totalTargetCapacity = totalTargetCapacity;
        }

        @Override
        @Transient
        public final Builder totalTargetCapacity(Integer totalTargetCapacity) {
            this.totalTargetCapacity = totalTargetCapacity;
            return this;
        }

        public final Double getTotalFulfilledCapacity() {
            return totalFulfilledCapacity;
        }

        public final void setTotalFulfilledCapacity(Double totalFulfilledCapacity) {
            this.totalFulfilledCapacity = totalFulfilledCapacity;
        }

        @Override
        @Transient
        public final Builder totalFulfilledCapacity(Double totalFulfilledCapacity) {
            this.totalFulfilledCapacity = totalFulfilledCapacity;
            return this;
        }

        public final String getTenancy() {
            return tenancy;
        }

        public final void setTenancy(String tenancy) {
            this.tenancy = tenancy;
        }

        @Override
        @Transient
        public final Builder tenancy(String tenancy) {
            this.tenancy = tenancy;
            return this;
        }

        @Override
        @Transient
        public final Builder tenancy(FleetCapacityReservationTenancy tenancy) {
            this.tenancy(tenancy == null ? null : tenancy.toString());
            return this;
        }

        public final Instant getEndDate() {
            return endDate;
        }

        public final void setEndDate(Instant endDate) {
            this.endDate = endDate;
        }

        @Override
        @Transient
        public final Builder endDate(Instant endDate) {
            this.endDate = endDate;
            return this;
        }

        public final Instant getCreateTime() {
            return createTime;
        }

        public final void setCreateTime(Instant createTime) {
            this.createTime = createTime;
        }

        @Override
        @Transient
        public final Builder createTime(Instant createTime) {
            this.createTime = createTime;
            return this;
        }

        public final String getInstanceMatchCriteria() {
            return instanceMatchCriteria;
        }

        public final void setInstanceMatchCriteria(String instanceMatchCriteria) {
            this.instanceMatchCriteria = instanceMatchCriteria;
        }

        @Override
        @Transient
        public final Builder instanceMatchCriteria(String instanceMatchCriteria) {
            this.instanceMatchCriteria = instanceMatchCriteria;
            return this;
        }

        @Override
        @Transient
        public final Builder instanceMatchCriteria(FleetInstanceMatchCriteria instanceMatchCriteria) {
            this.instanceMatchCriteria(instanceMatchCriteria == null ? null : instanceMatchCriteria.toString());
            return this;
        }

        public final String getAllocationStrategy() {
            return allocationStrategy;
        }

        public final void setAllocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
        }

        @Override
        @Transient
        public final Builder allocationStrategy(String allocationStrategy) {
            this.allocationStrategy = allocationStrategy;
            return this;
        }

        public final List<FleetCapacityReservation.Builder> getInstanceTypeSpecifications() {
            List<FleetCapacityReservation.Builder> result = FleetCapacityReservationSetCopier
                    .copyToBuilder(this.instanceTypeSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstanceTypeSpecifications(
                Collection<FleetCapacityReservation.BuilderImpl> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = FleetCapacityReservationSetCopier.copyFromBuilder(instanceTypeSpecifications);
        }

        @Override
        @Transient
        public final Builder instanceTypeSpecifications(Collection<FleetCapacityReservation> instanceTypeSpecifications) {
            this.instanceTypeSpecifications = FleetCapacityReservationSetCopier.copy(instanceTypeSpecifications);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder instanceTypeSpecifications(FleetCapacityReservation... instanceTypeSpecifications) {
            instanceTypeSpecifications(Arrays.asList(instanceTypeSpecifications));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder instanceTypeSpecifications(Consumer<FleetCapacityReservation.Builder>... instanceTypeSpecifications) {
            instanceTypeSpecifications(Stream.of(instanceTypeSpecifications)
                    .map(c -> FleetCapacityReservation.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public CapacityReservationFleet build() {
            return new CapacityReservationFleet(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
