/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a block device for an EBS volume.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsBlockDevice implements SdkPojo, Serializable, ToCopyableBuilder<EbsBlockDevice.Builder, EbsBlockDevice> {
    private static final SdkField<Boolean> DELETE_ON_TERMINATION_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DeleteOnTermination")
            .getter(getter(EbsBlockDevice::deleteOnTermination))
            .setter(setter(Builder::deleteOnTermination))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeleteOnTermination")
                    .unmarshallLocationName("deleteOnTermination").build()).build();

    private static final SdkField<Integer> IOPS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Iops")
            .getter(getter(EbsBlockDevice::iops))
            .setter(setter(Builder::iops))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Iops")
                    .unmarshallLocationName("iops").build()).build();

    private static final SdkField<String> SNAPSHOT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SnapshotId")
            .getter(getter(EbsBlockDevice::snapshotId))
            .setter(setter(Builder::snapshotId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotId")
                    .unmarshallLocationName("snapshotId").build()).build();

    private static final SdkField<Integer> VOLUME_SIZE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("VolumeSize")
            .getter(getter(EbsBlockDevice::volumeSize))
            .setter(setter(Builder::volumeSize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeSize")
                    .unmarshallLocationName("volumeSize").build()).build();

    private static final SdkField<String> VOLUME_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VolumeType")
            .getter(getter(EbsBlockDevice::volumeTypeAsString))
            .setter(setter(Builder::volumeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VolumeType")
                    .unmarshallLocationName("volumeType").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId")
            .getter(getter(EbsBlockDevice::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("KmsKeyId").build()).build();

    private static final SdkField<Integer> THROUGHPUT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("Throughput")
            .getter(getter(EbsBlockDevice::throughput))
            .setter(setter(Builder::throughput))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Throughput")
                    .unmarshallLocationName("throughput").build()).build();

    private static final SdkField<String> OUTPOST_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OutpostArn")
            .getter(getter(EbsBlockDevice::outpostArn))
            .setter(setter(Builder::outpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostArn")
                    .unmarshallLocationName("outpostArn").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Encrypted")
            .getter(getter(EbsBlockDevice::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DELETE_ON_TERMINATION_FIELD,
            IOPS_FIELD, SNAPSHOT_ID_FIELD, VOLUME_SIZE_FIELD, VOLUME_TYPE_FIELD, KMS_KEY_ID_FIELD, THROUGHPUT_FIELD,
            OUTPOST_ARN_FIELD, ENCRYPTED_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean deleteOnTermination;

    private final Integer iops;

    private final String snapshotId;

    private final Integer volumeSize;

    private final String volumeType;

    private final String kmsKeyId;

    private final Integer throughput;

    private final String outpostArn;

    private final Boolean encrypted;

    private EbsBlockDevice(BuilderImpl builder) {
        this.deleteOnTermination = builder.deleteOnTermination;
        this.iops = builder.iops;
        this.snapshotId = builder.snapshotId;
        this.volumeSize = builder.volumeSize;
        this.volumeType = builder.volumeType;
        this.kmsKeyId = builder.kmsKeyId;
        this.throughput = builder.throughput;
        this.outpostArn = builder.outpostArn;
        this.encrypted = builder.encrypted;
    }

    /**
     * <p>
     * Indicates whether the EBS volume is deleted on instance termination. For more information, see <a href=
     * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination"
     * >Preserving Amazon EBS volumes on instance termination</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return Indicates whether the EBS volume is deleted on instance termination. For more information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination"
     *         >Preserving Amazon EBS volumes on instance termination</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final Boolean deleteOnTermination() {
        return deleteOnTermination;
    }

    /**
     * <p>
     * The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and <code>io2</code>
     * volumes, this represents the number of IOPS that are provisioned for the volume. For <code>gp2</code> volumes,
     * this represents the baseline performance of the volume and the rate at which the volume accumulates I/O credits
     * for bursting.
     * </p>
     * <p>
     * The following are the supported values for each volume type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>gp3</code>: 3,000-16,000 IOPS
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io1</code>: 100-64,000 IOPS
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io2</code>: 100-64,000 IOPS
     * </p>
     * </li>
     * </ul>
     * <p>
     * For <code>io1</code> and <code>io2</code> volumes, we guarantee 64,000 IOPS only for <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
     * built on the Nitro System</a>. Other instance families guarantee performance up to 32,000 IOPS.
     * </p>
     * <p>
     * This parameter is required for <code>io1</code> and <code>io2</code> volumes. The default for <code>gp3</code>
     * volumes is 3,000 IOPS. This parameter is not supported for <code>gp2</code>, <code>st1</code>, <code>sc1</code>,
     * or <code>standard</code> volumes.
     * </p>
     * 
     * @return The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and
     *         <code>io2</code> volumes, this represents the number of IOPS that are provisioned for the volume. For
     *         <code>gp2</code> volumes, this represents the baseline performance of the volume and the rate at which
     *         the volume accumulates I/O credits for bursting.</p>
     *         <p>
     *         The following are the supported values for each volume type:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>gp3</code>: 3,000-16,000 IOPS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io1</code>: 100-64,000 IOPS
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io2</code>: 100-64,000 IOPS
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For <code>io1</code> and <code>io2</code> volumes, we guarantee 64,000 IOPS only for <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
     *         >Instances built on the Nitro System</a>. Other instance families guarantee performance up to 32,000
     *         IOPS.
     *         </p>
     *         <p>
     *         This parameter is required for <code>io1</code> and <code>io2</code> volumes. The default for
     *         <code>gp3</code> volumes is 3,000 IOPS. This parameter is not supported for <code>gp2</code>,
     *         <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
     */
    public final Integer iops() {
        return iops;
    }

    /**
     * <p>
     * The ID of the snapshot.
     * </p>
     * 
     * @return The ID of the snapshot.
     */
    public final String snapshotId() {
        return snapshotId;
    }

    /**
     * <p>
     * The size of the volume, in GiBs. You must specify either a snapshot ID or a volume size. If you specify a
     * snapshot, the default is the snapshot size. You can specify a volume size that is equal to or larger than the
     * snapshot size.
     * </p>
     * <p>
     * The following are the supported volumes sizes for each volume type:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>gp2</code> and <code>gp3</code>:1-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>io1</code> and <code>io2</code>: 4-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>st1</code> and <code>sc1</code>: 125-16,384
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>standard</code>: 1-1,024
     * </p>
     * </li>
     * </ul>
     * 
     * @return The size of the volume, in GiBs. You must specify either a snapshot ID or a volume size. If you specify a
     *         snapshot, the default is the snapshot size. You can specify a volume size that is equal to or larger than
     *         the snapshot size.</p>
     *         <p>
     *         The following are the supported volumes sizes for each volume type:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>gp2</code> and <code>gp3</code>:1-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>io1</code> and <code>io2</code>: 4-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>st1</code> and <code>sc1</code>: 125-16,384
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>standard</code>: 1-1,024
     *         </p>
     *         </li>
     */
    public final Integer volumeSize() {
        return volumeSize;
    }

    /**
     * <p>
     * The volume type. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume types</a> in the
     * <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>, you must specify the
     * IOPS that the volume supports.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume
     *         types</a> in the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>
     *         , you must specify the IOPS that the volume supports.
     * @see VolumeType
     */
    public final VolumeType volumeType() {
        return VolumeType.fromValue(volumeType);
    }

    /**
     * <p>
     * The volume type. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume types</a> in the
     * <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>, you must specify the
     * IOPS that the volume supports.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #volumeType} will
     * return {@link VolumeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #volumeTypeAsString}.
     * </p>
     * 
     * @return The volume type. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume
     *         types</a> in the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>
     *         , you must specify the IOPS that the volume supports.
     * @see VolumeType
     */
    public final String volumeTypeAsString() {
        return volumeType;
    }

    /**
     * <p>
     * Identifier (key ID, key alias, ID ARN, or alias ARN) for a customer managed CMK under which the EBS volume is
     * encrypted.
     * </p>
     * <p>
     * This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>, and
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">
     * RequestSpotInstances</a>.
     * </p>
     * 
     * @return Identifier (key ID, key alias, ID ARN, or alias ARN) for a customer managed CMK under which the EBS
     *         volume is encrypted.</p>
     *         <p>
     *         This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
     *         href
     *         ="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>,
     *         and <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">
     *         RequestSpotInstances</a>.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The throughput that the volume supports, in MiB/s.
     * </p>
     * <p>
     * This parameter is valid only for <code>gp3</code> volumes.
     * </p>
     * <p>
     * Valid Range: Minimum value of 125. Maximum value of 1000.
     * </p>
     * 
     * @return The throughput that the volume supports, in MiB/s.</p>
     *         <p>
     *         This parameter is valid only for <code>gp3</code> volumes.
     *         </p>
     *         <p>
     *         Valid Range: Minimum value of 125. Maximum value of 1000.
     */
    public final Integer throughput() {
        return throughput;
    }

    /**
     * <p>
     * The ARN of the Outpost on which the snapshot is stored.
     * </p>
     * 
     * @return The ARN of the Outpost on which the snapshot is stored.
     */
    public final String outpostArn() {
        return outpostArn;
    }

    /**
     * <p>
     * Indicates whether the encryption state of an EBS volume is changed while being restored from a backing snapshot.
     * The effect of setting the encryption state to <code>true</code> depends on the volume origin (new or from a
     * snapshot), starting encryption state, ownership, and whether encryption by default is enabled. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-parameters">Amazon EBS
     * encryption</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * In no case can you remove encryption from an encrypted volume.
     * </p>
     * <p>
     * Encrypted volumes can only be attached to instances that support Amazon EBS encryption. For more information, see
     * <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances"
     * >Supported instance types</a>.
     * </p>
     * <p>
     * This parameter is not returned by .
     * </p>
     * 
     * @return Indicates whether the encryption state of an EBS volume is changed while being restored from a backing
     *         snapshot. The effect of setting the encryption state to <code>true</code> depends on the volume origin
     *         (new or from a snapshot), starting encryption state, ownership, and whether encryption by default is
     *         enabled. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-parameters"
     *         >Amazon EBS encryption</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         In no case can you remove encryption from an encrypted volume.
     *         </p>
     *         <p>
     *         Encrypted volumes can only be attached to instances that support Amazon EBS encryption. For more
     *         information, see <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances"
     *         >Supported instance types</a>.
     *         </p>
     *         <p>
     *         This parameter is not returned by .
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deleteOnTermination());
        hashCode = 31 * hashCode + Objects.hashCode(iops());
        hashCode = 31 * hashCode + Objects.hashCode(snapshotId());
        hashCode = 31 * hashCode + Objects.hashCode(volumeSize());
        hashCode = 31 * hashCode + Objects.hashCode(volumeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(throughput());
        hashCode = 31 * hashCode + Objects.hashCode(outpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsBlockDevice)) {
            return false;
        }
        EbsBlockDevice other = (EbsBlockDevice) obj;
        return Objects.equals(deleteOnTermination(), other.deleteOnTermination()) && Objects.equals(iops(), other.iops())
                && Objects.equals(snapshotId(), other.snapshotId()) && Objects.equals(volumeSize(), other.volumeSize())
                && Objects.equals(volumeTypeAsString(), other.volumeTypeAsString())
                && Objects.equals(kmsKeyId(), other.kmsKeyId()) && Objects.equals(throughput(), other.throughput())
                && Objects.equals(outpostArn(), other.outpostArn()) && Objects.equals(encrypted(), other.encrypted());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EbsBlockDevice").add("DeleteOnTermination", deleteOnTermination()).add("Iops", iops())
                .add("SnapshotId", snapshotId()).add("VolumeSize", volumeSize()).add("VolumeType", volumeTypeAsString())
                .add("KmsKeyId", kmsKeyId()).add("Throughput", throughput()).add("OutpostArn", outpostArn())
                .add("Encrypted", encrypted()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DeleteOnTermination":
            return Optional.ofNullable(clazz.cast(deleteOnTermination()));
        case "Iops":
            return Optional.ofNullable(clazz.cast(iops()));
        case "SnapshotId":
            return Optional.ofNullable(clazz.cast(snapshotId()));
        case "VolumeSize":
            return Optional.ofNullable(clazz.cast(volumeSize()));
        case "VolumeType":
            return Optional.ofNullable(clazz.cast(volumeTypeAsString()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Throughput":
            return Optional.ofNullable(clazz.cast(throughput()));
        case "OutpostArn":
            return Optional.ofNullable(clazz.cast(outpostArn()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EbsBlockDevice, T> g) {
        return obj -> g.apply((EbsBlockDevice) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EbsBlockDevice> {
        /**
         * <p>
         * Indicates whether the EBS volume is deleted on instance termination. For more information, see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination"
         * >Preserving Amazon EBS volumes on instance termination</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param deleteOnTermination
         *        Indicates whether the EBS volume is deleted on instance termination. For more information, see <a
         *        href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#preserving-volumes-on-termination"
         *        >Preserving Amazon EBS volumes on instance termination</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deleteOnTermination(Boolean deleteOnTermination);

        /**
         * <p>
         * The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and <code>io2</code>
         * volumes, this represents the number of IOPS that are provisioned for the volume. For <code>gp2</code>
         * volumes, this represents the baseline performance of the volume and the rate at which the volume accumulates
         * I/O credits for bursting.
         * </p>
         * <p>
         * The following are the supported values for each volume type:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>gp3</code>: 3,000-16,000 IOPS
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io1</code>: 100-64,000 IOPS
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io2</code>: 100-64,000 IOPS
         * </p>
         * </li>
         * </ul>
         * <p>
         * For <code>io1</code> and <code>io2</code> volumes, we guarantee 64,000 IOPS only for <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances">Instances
         * built on the Nitro System</a>. Other instance families guarantee performance up to 32,000 IOPS.
         * </p>
         * <p>
         * This parameter is required for <code>io1</code> and <code>io2</code> volumes. The default for
         * <code>gp3</code> volumes is 3,000 IOPS. This parameter is not supported for <code>gp2</code>,
         * <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
         * </p>
         * 
         * @param iops
         *        The number of I/O operations per second (IOPS). For <code>gp3</code>, <code>io1</code>, and
         *        <code>io2</code> volumes, this represents the number of IOPS that are provisioned for the volume. For
         *        <code>gp2</code> volumes, this represents the baseline performance of the volume and the rate at which
         *        the volume accumulates I/O credits for bursting.</p>
         *        <p>
         *        The following are the supported values for each volume type:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>gp3</code>: 3,000-16,000 IOPS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io1</code>: 100-64,000 IOPS
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io2</code>: 100-64,000 IOPS
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For <code>io1</code> and <code>io2</code> volumes, we guarantee 64,000 IOPS only for <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html#ec2-nitro-instances"
         *        >Instances built on the Nitro System</a>. Other instance families guarantee performance up to 32,000
         *        IOPS.
         *        </p>
         *        <p>
         *        This parameter is required for <code>io1</code> and <code>io2</code> volumes. The default for
         *        <code>gp3</code> volumes is 3,000 IOPS. This parameter is not supported for <code>gp2</code>,
         *        <code>st1</code>, <code>sc1</code>, or <code>standard</code> volumes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iops(Integer iops);

        /**
         * <p>
         * The ID of the snapshot.
         * </p>
         * 
         * @param snapshotId
         *        The ID of the snapshot.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotId(String snapshotId);

        /**
         * <p>
         * The size of the volume, in GiBs. You must specify either a snapshot ID or a volume size. If you specify a
         * snapshot, the default is the snapshot size. You can specify a volume size that is equal to or larger than the
         * snapshot size.
         * </p>
         * <p>
         * The following are the supported volumes sizes for each volume type:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>gp2</code> and <code>gp3</code>:1-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>io1</code> and <code>io2</code>: 4-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>st1</code> and <code>sc1</code>: 125-16,384
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>standard</code>: 1-1,024
         * </p>
         * </li>
         * </ul>
         * 
         * @param volumeSize
         *        The size of the volume, in GiBs. You must specify either a snapshot ID or a volume size. If you
         *        specify a snapshot, the default is the snapshot size. You can specify a volume size that is equal to
         *        or larger than the snapshot size.</p>
         *        <p>
         *        The following are the supported volumes sizes for each volume type:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>gp2</code> and <code>gp3</code>:1-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>io1</code> and <code>io2</code>: 4-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>st1</code> and <code>sc1</code>: 125-16,384
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>standard</code>: 1-1,024
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeSize(Integer volumeSize);

        /**
         * <p>
         * The volume type. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume types</a> in
         * the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>, you must
         * specify the IOPS that the volume supports.
         * </p>
         * 
         * @param volumeType
         *        The volume type. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume
         *        types</a> in the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or
         *        <code>io2</code>, you must specify the IOPS that the volume supports.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(String volumeType);

        /**
         * <p>
         * The volume type. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume types</a> in
         * the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or <code>io2</code>, you must
         * specify the IOPS that the volume supports.
         * </p>
         * 
         * @param volumeType
         *        The volume type. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSVolumeTypes.html">Amazon EBS volume
         *        types</a> in the <i>Amazon EC2 User Guide</i>. If the volume type is <code>io1</code> or
         *        <code>io2</code>, you must specify the IOPS that the volume supports.
         * @see VolumeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VolumeType
         */
        Builder volumeType(VolumeType volumeType);

        /**
         * <p>
         * Identifier (key ID, key alias, ID ARN, or alias ARN) for a customer managed CMK under which the EBS volume is
         * encrypted.
         * </p>
         * <p>
         * This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>, <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">RequestSpotFleet</a>,
         * and <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html">
         * RequestSpotInstances</a>.
         * </p>
         * 
         * @param kmsKeyId
         *        Identifier (key ID, key alias, ID ARN, or alias ARN) for a customer managed CMK under which the EBS
         *        volume is encrypted.</p>
         *        <p>
         *        This parameter is only supported on <code>BlockDeviceMapping</code> objects called by <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a>,
         *        <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotFleet.html">
         *        RequestSpotFleet</a>, and <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RequestSpotInstances.html"
         *        >RequestSpotInstances</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The throughput that the volume supports, in MiB/s.
         * </p>
         * <p>
         * This parameter is valid only for <code>gp3</code> volumes.
         * </p>
         * <p>
         * Valid Range: Minimum value of 125. Maximum value of 1000.
         * </p>
         * 
         * @param throughput
         *        The throughput that the volume supports, in MiB/s.</p>
         *        <p>
         *        This parameter is valid only for <code>gp3</code> volumes.
         *        </p>
         *        <p>
         *        Valid Range: Minimum value of 125. Maximum value of 1000.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder throughput(Integer throughput);

        /**
         * <p>
         * The ARN of the Outpost on which the snapshot is stored.
         * </p>
         * 
         * @param outpostArn
         *        The ARN of the Outpost on which the snapshot is stored.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostArn(String outpostArn);

        /**
         * <p>
         * Indicates whether the encryption state of an EBS volume is changed while being restored from a backing
         * snapshot. The effect of setting the encryption state to <code>true</code> depends on the volume origin (new
         * or from a snapshot), starting encryption state, ownership, and whether encryption by default is enabled. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-parameters">Amazon
         * EBS encryption</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * In no case can you remove encryption from an encrypted volume.
         * </p>
         * <p>
         * Encrypted volumes can only be attached to instances that support Amazon EBS encryption. For more information,
         * see <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances"
         * >Supported instance types</a>.
         * </p>
         * <p>
         * This parameter is not returned by .
         * </p>
         * 
         * @param encrypted
         *        Indicates whether the encryption state of an EBS volume is changed while being restored from a backing
         *        snapshot. The effect of setting the encryption state to <code>true</code> depends on the volume origin
         *        (new or from a snapshot), starting encryption state, ownership, and whether encryption by default is
         *        enabled. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#encryption-parameters"
         *        >Amazon EBS encryption</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        In no case can you remove encryption from an encrypted volume.
         *        </p>
         *        <p>
         *        Encrypted volumes can only be attached to instances that support Amazon EBS encryption. For more
         *        information, see <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/EBSEncryption.html#EBSEncryption_supported_instances"
         *        >Supported instance types</a>.
         *        </p>
         *        <p>
         *        This parameter is not returned by .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);
    }

    static final class BuilderImpl implements Builder {
        private Boolean deleteOnTermination;

        private Integer iops;

        private String snapshotId;

        private Integer volumeSize;

        private String volumeType;

        private String kmsKeyId;

        private Integer throughput;

        private String outpostArn;

        private Boolean encrypted;

        private BuilderImpl() {
        }

        private BuilderImpl(EbsBlockDevice model) {
            deleteOnTermination(model.deleteOnTermination);
            iops(model.iops);
            snapshotId(model.snapshotId);
            volumeSize(model.volumeSize);
            volumeType(model.volumeType);
            kmsKeyId(model.kmsKeyId);
            throughput(model.throughput);
            outpostArn(model.outpostArn);
            encrypted(model.encrypted);
        }

        public final Boolean getDeleteOnTermination() {
            return deleteOnTermination;
        }

        public final void setDeleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
        }

        @Override
        @Transient
        public final Builder deleteOnTermination(Boolean deleteOnTermination) {
            this.deleteOnTermination = deleteOnTermination;
            return this;
        }

        public final Integer getIops() {
            return iops;
        }

        public final void setIops(Integer iops) {
            this.iops = iops;
        }

        @Override
        @Transient
        public final Builder iops(Integer iops) {
            this.iops = iops;
            return this;
        }

        public final String getSnapshotId() {
            return snapshotId;
        }

        public final void setSnapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
        }

        @Override
        @Transient
        public final Builder snapshotId(String snapshotId) {
            this.snapshotId = snapshotId;
            return this;
        }

        public final Integer getVolumeSize() {
            return volumeSize;
        }

        public final void setVolumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
        }

        @Override
        @Transient
        public final Builder volumeSize(Integer volumeSize) {
            this.volumeSize = volumeSize;
            return this;
        }

        public final String getVolumeType() {
            return volumeType;
        }

        public final void setVolumeType(String volumeType) {
            this.volumeType = volumeType;
        }

        @Override
        @Transient
        public final Builder volumeType(String volumeType) {
            this.volumeType = volumeType;
            return this;
        }

        @Override
        @Transient
        public final Builder volumeType(VolumeType volumeType) {
            this.volumeType(volumeType == null ? null : volumeType.toString());
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        @Transient
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final Integer getThroughput() {
            return throughput;
        }

        public final void setThroughput(Integer throughput) {
            this.throughput = throughput;
        }

        @Override
        @Transient
        public final Builder throughput(Integer throughput) {
            this.throughput = throughput;
            return this;
        }

        public final String getOutpostArn() {
            return outpostArn;
        }

        public final void setOutpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
        }

        @Override
        @Transient
        public final Builder outpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
            return this;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        @Transient
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        @Override
        public EbsBlockDevice build() {
            return new EbsBlockDevice(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
