/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the properties of the Dedicated Host.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Host implements SdkPojo, Serializable, ToCopyableBuilder<Host.Builder, Host> {
    private static final SdkField<String> AUTO_PLACEMENT_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AutoPlacement")
            .getter(getter(Host::autoPlacementAsString))
            .setter(setter(Builder::autoPlacement))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoPlacement")
                    .unmarshallLocationName("autoPlacement").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(Host::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final SdkField<AvailableCapacity> AVAILABLE_CAPACITY_FIELD = SdkField
            .<AvailableCapacity> builder(MarshallingType.SDK_POJO)
            .memberName("AvailableCapacity")
            .getter(getter(Host::availableCapacity))
            .setter(setter(Builder::availableCapacity))
            .constructor(AvailableCapacity::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailableCapacity")
                    .unmarshallLocationName("availableCapacity").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(Host::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("clientToken").build()).build();

    private static final SdkField<String> HOST_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HostId")
            .getter(getter(Host::hostId))
            .setter(setter(Builder::hostId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostId")
                    .unmarshallLocationName("hostId").build()).build();

    private static final SdkField<HostProperties> HOST_PROPERTIES_FIELD = SdkField
            .<HostProperties> builder(MarshallingType.SDK_POJO)
            .memberName("HostProperties")
            .getter(getter(Host::hostProperties))
            .setter(setter(Builder::hostProperties))
            .constructor(HostProperties::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostProperties")
                    .unmarshallLocationName("hostProperties").build()).build();

    private static final SdkField<String> HOST_RESERVATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HostReservationId")
            .getter(getter(Host::hostReservationId))
            .setter(setter(Builder::hostReservationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostReservationId")
                    .unmarshallLocationName("hostReservationId").build()).build();

    private static final SdkField<List<HostInstance>> INSTANCES_FIELD = SdkField
            .<List<HostInstance>> builder(MarshallingType.LIST)
            .memberName("Instances")
            .getter(getter(Host::instances))
            .setter(setter(Builder::instances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Instances")
                    .unmarshallLocationName("instances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<HostInstance> builder(MarshallingType.SDK_POJO)
                                            .constructor(HostInstance::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(Host::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<Instant> ALLOCATION_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("AllocationTime")
            .getter(getter(Host::allocationTime))
            .setter(setter(Builder::allocationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationTime")
                    .unmarshallLocationName("allocationTime").build()).build();

    private static final SdkField<Instant> RELEASE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReleaseTime")
            .getter(getter(Host::releaseTime))
            .setter(setter(Builder::releaseTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReleaseTime")
                    .unmarshallLocationName("releaseTime").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(Host::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> HOST_RECOVERY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("HostRecovery")
            .getter(getter(Host::hostRecoveryAsString))
            .setter(setter(Builder::hostRecovery))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("HostRecovery")
                    .unmarshallLocationName("hostRecovery").build()).build();

    private static final SdkField<String> ALLOWS_MULTIPLE_INSTANCE_TYPES_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AllowsMultipleInstanceTypes")
            .getter(getter(Host::allowsMultipleInstanceTypesAsString))
            .setter(setter(Builder::allowsMultipleInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowsMultipleInstanceTypes")
                    .unmarshallLocationName("allowsMultipleInstanceTypes").build()).build();

    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(Host::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(Host::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("availabilityZoneId").build()).build();

    private static final SdkField<Boolean> MEMBER_OF_SERVICE_LINKED_RESOURCE_GROUP_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("MemberOfServiceLinkedResourceGroup")
            .getter(getter(Host::memberOfServiceLinkedResourceGroup))
            .setter(setter(Builder::memberOfServiceLinkedResourceGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemberOfServiceLinkedResourceGroup")
                    .unmarshallLocationName("memberOfServiceLinkedResourceGroup").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUTO_PLACEMENT_FIELD,
            AVAILABILITY_ZONE_FIELD, AVAILABLE_CAPACITY_FIELD, CLIENT_TOKEN_FIELD, HOST_ID_FIELD, HOST_PROPERTIES_FIELD,
            HOST_RESERVATION_ID_FIELD, INSTANCES_FIELD, STATE_FIELD, ALLOCATION_TIME_FIELD, RELEASE_TIME_FIELD, TAGS_FIELD,
            HOST_RECOVERY_FIELD, ALLOWS_MULTIPLE_INSTANCE_TYPES_FIELD, OWNER_ID_FIELD, AVAILABILITY_ZONE_ID_FIELD,
            MEMBER_OF_SERVICE_LINKED_RESOURCE_GROUP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String autoPlacement;

    private final String availabilityZone;

    private final AvailableCapacity availableCapacity;

    private final String clientToken;

    private final String hostId;

    private final HostProperties hostProperties;

    private final String hostReservationId;

    private final List<HostInstance> instances;

    private final String state;

    private final Instant allocationTime;

    private final Instant releaseTime;

    private final List<Tag> tags;

    private final String hostRecovery;

    private final String allowsMultipleInstanceTypes;

    private final String ownerId;

    private final String availabilityZoneId;

    private final Boolean memberOfServiceLinkedResourceGroup;

    private Host(BuilderImpl builder) {
        this.autoPlacement = builder.autoPlacement;
        this.availabilityZone = builder.availabilityZone;
        this.availableCapacity = builder.availableCapacity;
        this.clientToken = builder.clientToken;
        this.hostId = builder.hostId;
        this.hostProperties = builder.hostProperties;
        this.hostReservationId = builder.hostReservationId;
        this.instances = builder.instances;
        this.state = builder.state;
        this.allocationTime = builder.allocationTime;
        this.releaseTime = builder.releaseTime;
        this.tags = builder.tags;
        this.hostRecovery = builder.hostRecovery;
        this.allowsMultipleInstanceTypes = builder.allowsMultipleInstanceTypes;
        this.ownerId = builder.ownerId;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.memberOfServiceLinkedResourceGroup = builder.memberOfServiceLinkedResourceGroup;
    }

    /**
     * <p>
     * Whether auto-placement is on or off.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoPlacement}
     * will return {@link AutoPlacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #autoPlacementAsString}.
     * </p>
     * 
     * @return Whether auto-placement is on or off.
     * @see AutoPlacement
     */
    public final AutoPlacement autoPlacement() {
        return AutoPlacement.fromValue(autoPlacement);
    }

    /**
     * <p>
     * Whether auto-placement is on or off.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #autoPlacement}
     * will return {@link AutoPlacement#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #autoPlacementAsString}.
     * </p>
     * 
     * @return Whether auto-placement is on or off.
     * @see AutoPlacement
     */
    public final String autoPlacementAsString() {
        return autoPlacement;
    }

    /**
     * <p>
     * The Availability Zone of the Dedicated Host.
     * </p>
     * 
     * @return The Availability Zone of the Dedicated Host.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * Information about the instances running on the Dedicated Host.
     * </p>
     * 
     * @return Information about the instances running on the Dedicated Host.
     */
    public final AvailableCapacity availableCapacity() {
        return availableCapacity;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     * Idempotency</a>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     *         Idempotency</a>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The ID of the Dedicated Host.
     * </p>
     * 
     * @return The ID of the Dedicated Host.
     */
    public final String hostId() {
        return hostId;
    }

    /**
     * <p>
     * The hardware specifications of the Dedicated Host.
     * </p>
     * 
     * @return The hardware specifications of the Dedicated Host.
     */
    public final HostProperties hostProperties() {
        return hostProperties;
    }

    /**
     * <p>
     * The reservation ID of the Dedicated Host. This returns a <code>null</code> response if the Dedicated Host doesn't
     * have an associated reservation.
     * </p>
     * 
     * @return The reservation ID of the Dedicated Host. This returns a <code>null</code> response if the Dedicated Host
     *         doesn't have an associated reservation.
     */
    public final String hostReservationId() {
        return hostReservationId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Instances property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasInstances() {
        return instances != null && !(instances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs and instance type that are currently running on the Dedicated Host.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstances} method.
     * </p>
     * 
     * @return The IDs and instance type that are currently running on the Dedicated Host.
     */
    public final List<HostInstance> instances() {
        return instances;
    }

    /**
     * <p>
     * The Dedicated Host's state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AllocationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The Dedicated Host's state.
     * @see AllocationState
     */
    public final AllocationState state() {
        return AllocationState.fromValue(state);
    }

    /**
     * <p>
     * The Dedicated Host's state.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link AllocationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The Dedicated Host's state.
     * @see AllocationState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The time that the Dedicated Host was allocated.
     * </p>
     * 
     * @return The time that the Dedicated Host was allocated.
     */
    public final Instant allocationTime() {
        return allocationTime;
    }

    /**
     * <p>
     * The time that the Dedicated Host was released.
     * </p>
     * 
     * @return The time that the Dedicated Host was released.
     */
    public final Instant releaseTime() {
        return releaseTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the Dedicated Host.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the Dedicated Host.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Indicates whether host recovery is enabled or disabled for the Dedicated Host.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hostRecovery} will
     * return {@link HostRecovery#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hostRecoveryAsString}.
     * </p>
     * 
     * @return Indicates whether host recovery is enabled or disabled for the Dedicated Host.
     * @see HostRecovery
     */
    public final HostRecovery hostRecovery() {
        return HostRecovery.fromValue(hostRecovery);
    }

    /**
     * <p>
     * Indicates whether host recovery is enabled or disabled for the Dedicated Host.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #hostRecovery} will
     * return {@link HostRecovery#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #hostRecoveryAsString}.
     * </p>
     * 
     * @return Indicates whether host recovery is enabled or disabled for the Dedicated Host.
     * @see HostRecovery
     */
    public final String hostRecoveryAsString() {
        return hostRecovery;
    }

    /**
     * <p>
     * Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the value
     * is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If the value is
     * <code>off</code>, the Dedicated Host supports a single instance type only.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsMultipleInstanceTypes} will return {@link AllowsMultipleInstanceTypes#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #allowsMultipleInstanceTypesAsString}.
     * </p>
     * 
     * @return Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the
     *         value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If
     *         the value is <code>off</code>, the Dedicated Host supports a single instance type only.
     * @see AllowsMultipleInstanceTypes
     */
    public final AllowsMultipleInstanceTypes allowsMultipleInstanceTypes() {
        return AllowsMultipleInstanceTypes.fromValue(allowsMultipleInstanceTypes);
    }

    /**
     * <p>
     * Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the value
     * is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If the value is
     * <code>off</code>, the Dedicated Host supports a single instance type only.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #allowsMultipleInstanceTypes} will return {@link AllowsMultipleInstanceTypes#UNKNOWN_TO_SDK_VERSION}. The
     * raw value returned by the service is available from {@link #allowsMultipleInstanceTypesAsString}.
     * </p>
     * 
     * @return Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the
     *         value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If
     *         the value is <code>off</code>, the Dedicated Host supports a single instance type only.
     * @see AllowsMultipleInstanceTypes
     */
    public final String allowsMultipleInstanceTypesAsString() {
        return allowsMultipleInstanceTypes;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the Dedicated Host.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the Dedicated Host.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the Availability Zone in which the Dedicated Host is allocated.
     * </p>
     * 
     * @return The ID of the Availability Zone in which the Dedicated Host is allocated.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * <p>
     * Indicates whether the Dedicated Host is in a host resource group. If <b>memberOfServiceLinkedResourceGroup</b> is
     * <code>true</code>, the host is in a host resource group; otherwise, it is not.
     * </p>
     * 
     * @return Indicates whether the Dedicated Host is in a host resource group. If
     *         <b>memberOfServiceLinkedResourceGroup</b> is <code>true</code>, the host is in a host resource group;
     *         otherwise, it is not.
     */
    public final Boolean memberOfServiceLinkedResourceGroup() {
        return memberOfServiceLinkedResourceGroup;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(autoPlacementAsString());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(availableCapacity());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(hostId());
        hashCode = 31 * hashCode + Objects.hashCode(hostProperties());
        hashCode = 31 * hashCode + Objects.hashCode(hostReservationId());
        hashCode = 31 * hashCode + Objects.hashCode(hasInstances() ? instances() : null);
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allocationTime());
        hashCode = 31 * hashCode + Objects.hashCode(releaseTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hostRecoveryAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allowsMultipleInstanceTypesAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(memberOfServiceLinkedResourceGroup());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Host)) {
            return false;
        }
        Host other = (Host) obj;
        return Objects.equals(autoPlacementAsString(), other.autoPlacementAsString())
                && Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(availableCapacity(), other.availableCapacity())
                && Objects.equals(clientToken(), other.clientToken()) && Objects.equals(hostId(), other.hostId())
                && Objects.equals(hostProperties(), other.hostProperties())
                && Objects.equals(hostReservationId(), other.hostReservationId()) && hasInstances() == other.hasInstances()
                && Objects.equals(instances(), other.instances()) && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(allocationTime(), other.allocationTime()) && Objects.equals(releaseTime(), other.releaseTime())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags())
                && Objects.equals(hostRecoveryAsString(), other.hostRecoveryAsString())
                && Objects.equals(allowsMultipleInstanceTypesAsString(), other.allowsMultipleInstanceTypesAsString())
                && Objects.equals(ownerId(), other.ownerId()) && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && Objects.equals(memberOfServiceLinkedResourceGroup(), other.memberOfServiceLinkedResourceGroup());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Host").add("AutoPlacement", autoPlacementAsString()).add("AvailabilityZone", availabilityZone())
                .add("AvailableCapacity", availableCapacity()).add("ClientToken", clientToken()).add("HostId", hostId())
                .add("HostProperties", hostProperties()).add("HostReservationId", hostReservationId())
                .add("Instances", hasInstances() ? instances() : null).add("State", stateAsString())
                .add("AllocationTime", allocationTime()).add("ReleaseTime", releaseTime()).add("Tags", hasTags() ? tags() : null)
                .add("HostRecovery", hostRecoveryAsString())
                .add("AllowsMultipleInstanceTypes", allowsMultipleInstanceTypesAsString()).add("OwnerId", ownerId())
                .add("AvailabilityZoneId", availabilityZoneId())
                .add("MemberOfServiceLinkedResourceGroup", memberOfServiceLinkedResourceGroup()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AutoPlacement":
            return Optional.ofNullable(clazz.cast(autoPlacementAsString()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "AvailableCapacity":
            return Optional.ofNullable(clazz.cast(availableCapacity()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "HostId":
            return Optional.ofNullable(clazz.cast(hostId()));
        case "HostProperties":
            return Optional.ofNullable(clazz.cast(hostProperties()));
        case "HostReservationId":
            return Optional.ofNullable(clazz.cast(hostReservationId()));
        case "Instances":
            return Optional.ofNullable(clazz.cast(instances()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "AllocationTime":
            return Optional.ofNullable(clazz.cast(allocationTime()));
        case "ReleaseTime":
            return Optional.ofNullable(clazz.cast(releaseTime()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "HostRecovery":
            return Optional.ofNullable(clazz.cast(hostRecoveryAsString()));
        case "AllowsMultipleInstanceTypes":
            return Optional.ofNullable(clazz.cast(allowsMultipleInstanceTypesAsString()));
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "MemberOfServiceLinkedResourceGroup":
            return Optional.ofNullable(clazz.cast(memberOfServiceLinkedResourceGroup()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Host, T> g) {
        return obj -> g.apply((Host) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Host> {
        /**
         * <p>
         * Whether auto-placement is on or off.
         * </p>
         * 
         * @param autoPlacement
         *        Whether auto-placement is on or off.
         * @see AutoPlacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoPlacement
         */
        Builder autoPlacement(String autoPlacement);

        /**
         * <p>
         * Whether auto-placement is on or off.
         * </p>
         * 
         * @param autoPlacement
         *        Whether auto-placement is on or off.
         * @see AutoPlacement
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AutoPlacement
         */
        Builder autoPlacement(AutoPlacement autoPlacement);

        /**
         * <p>
         * The Availability Zone of the Dedicated Host.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone of the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * Information about the instances running on the Dedicated Host.
         * </p>
         * 
         * @param availableCapacity
         *        Information about the instances running on the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availableCapacity(AvailableCapacity availableCapacity);

        /**
         * <p>
         * Information about the instances running on the Dedicated Host.
         * </p>
         * This is a convenience that creates an instance of the {@link AvailableCapacity.Builder} avoiding the need to
         * create one manually via {@link AvailableCapacity#builder()}.
         *
         * When the {@link Consumer} completes, {@link AvailableCapacity.Builder#build()} is called immediately and its
         * result is passed to {@link #availableCapacity(AvailableCapacity)}.
         * 
         * @param availableCapacity
         *        a consumer that will call methods on {@link AvailableCapacity.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #availableCapacity(AvailableCapacity)
         */
        default Builder availableCapacity(Consumer<AvailableCapacity.Builder> availableCapacity) {
            return availableCapacity(AvailableCapacity.builder().applyMutation(availableCapacity).build());
        }

        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         * Idempotency</a>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         *        Idempotency</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The ID of the Dedicated Host.
         * </p>
         * 
         * @param hostId
         *        The ID of the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostId(String hostId);

        /**
         * <p>
         * The hardware specifications of the Dedicated Host.
         * </p>
         * 
         * @param hostProperties
         *        The hardware specifications of the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostProperties(HostProperties hostProperties);

        /**
         * <p>
         * The hardware specifications of the Dedicated Host.
         * </p>
         * This is a convenience that creates an instance of the {@link HostProperties.Builder} avoiding the need to
         * create one manually via {@link HostProperties#builder()}.
         *
         * When the {@link Consumer} completes, {@link HostProperties.Builder#build()} is called immediately and its
         * result is passed to {@link #hostProperties(HostProperties)}.
         * 
         * @param hostProperties
         *        a consumer that will call methods on {@link HostProperties.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #hostProperties(HostProperties)
         */
        default Builder hostProperties(Consumer<HostProperties.Builder> hostProperties) {
            return hostProperties(HostProperties.builder().applyMutation(hostProperties).build());
        }

        /**
         * <p>
         * The reservation ID of the Dedicated Host. This returns a <code>null</code> response if the Dedicated Host
         * doesn't have an associated reservation.
         * </p>
         * 
         * @param hostReservationId
         *        The reservation ID of the Dedicated Host. This returns a <code>null</code> response if the Dedicated
         *        Host doesn't have an associated reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder hostReservationId(String hostReservationId);

        /**
         * <p>
         * The IDs and instance type that are currently running on the Dedicated Host.
         * </p>
         * 
         * @param instances
         *        The IDs and instance type that are currently running on the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Collection<HostInstance> instances);

        /**
         * <p>
         * The IDs and instance type that are currently running on the Dedicated Host.
         * </p>
         * 
         * @param instances
         *        The IDs and instance type that are currently running on the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(HostInstance... instances);

        /**
         * <p>
         * The IDs and instance type that are currently running on the Dedicated Host.
         * </p>
         * This is a convenience that creates an instance of the {@link List<HostInstance>.Builder} avoiding the need to
         * create one manually via {@link List<HostInstance>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<HostInstance>.Builder#build()} is called immediately and its
         * result is passed to {@link #instances(List<HostInstance>)}.
         * 
         * @param instances
         *        a consumer that will call methods on {@link List<HostInstance>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instances(List<HostInstance>)
         */
        Builder instances(Consumer<HostInstance.Builder>... instances);

        /**
         * <p>
         * The Dedicated Host's state.
         * </p>
         * 
         * @param state
         *        The Dedicated Host's state.
         * @see AllocationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllocationState
         */
        Builder state(String state);

        /**
         * <p>
         * The Dedicated Host's state.
         * </p>
         * 
         * @param state
         *        The Dedicated Host's state.
         * @see AllocationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllocationState
         */
        Builder state(AllocationState state);

        /**
         * <p>
         * The time that the Dedicated Host was allocated.
         * </p>
         * 
         * @param allocationTime
         *        The time that the Dedicated Host was allocated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationTime(Instant allocationTime);

        /**
         * <p>
         * The time that the Dedicated Host was released.
         * </p>
         * 
         * @param releaseTime
         *        The time that the Dedicated Host was released.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder releaseTime(Instant releaseTime);

        /**
         * <p>
         * Any tags assigned to the Dedicated Host.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the Dedicated Host.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the Dedicated Host.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Indicates whether host recovery is enabled or disabled for the Dedicated Host.
         * </p>
         * 
         * @param hostRecovery
         *        Indicates whether host recovery is enabled or disabled for the Dedicated Host.
         * @see HostRecovery
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HostRecovery
         */
        Builder hostRecovery(String hostRecovery);

        /**
         * <p>
         * Indicates whether host recovery is enabled or disabled for the Dedicated Host.
         * </p>
         * 
         * @param hostRecovery
         *        Indicates whether host recovery is enabled or disabled for the Dedicated Host.
         * @see HostRecovery
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see HostRecovery
         */
        Builder hostRecovery(HostRecovery hostRecovery);

        /**
         * <p>
         * Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the
         * value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If the
         * value is <code>off</code>, the Dedicated Host supports a single instance type only.
         * </p>
         * 
         * @param allowsMultipleInstanceTypes
         *        Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If
         *        the value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance
         *        family. If the value is <code>off</code>, the Dedicated Host supports a single instance type only.
         * @see AllowsMultipleInstanceTypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsMultipleInstanceTypes
         */
        Builder allowsMultipleInstanceTypes(String allowsMultipleInstanceTypes);

        /**
         * <p>
         * Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If the
         * value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance family. If the
         * value is <code>off</code>, the Dedicated Host supports a single instance type only.
         * </p>
         * 
         * @param allowsMultipleInstanceTypes
         *        Indicates whether the Dedicated Host supports multiple instance types of the same instance family. If
         *        the value is <code>on</code>, the Dedicated Host supports multiple instance types in the instance
         *        family. If the value is <code>off</code>, the Dedicated Host supports a single instance type only.
         * @see AllowsMultipleInstanceTypes
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AllowsMultipleInstanceTypes
         */
        Builder allowsMultipleInstanceTypes(AllowsMultipleInstanceTypes allowsMultipleInstanceTypes);

        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the Dedicated Host.
         * </p>
         * 
         * @param ownerId
         *        The ID of the Amazon Web Services account that owns the Dedicated Host.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the Availability Zone in which the Dedicated Host is allocated.
         * </p>
         * 
         * @param availabilityZoneId
         *        The ID of the Availability Zone in which the Dedicated Host is allocated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * Indicates whether the Dedicated Host is in a host resource group. If
         * <b>memberOfServiceLinkedResourceGroup</b> is <code>true</code>, the host is in a host resource group;
         * otherwise, it is not.
         * </p>
         * 
         * @param memberOfServiceLinkedResourceGroup
         *        Indicates whether the Dedicated Host is in a host resource group. If
         *        <b>memberOfServiceLinkedResourceGroup</b> is <code>true</code>, the host is in a host resource group;
         *        otherwise, it is not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memberOfServiceLinkedResourceGroup(Boolean memberOfServiceLinkedResourceGroup);
    }

    static final class BuilderImpl implements Builder {
        private String autoPlacement;

        private String availabilityZone;

        private AvailableCapacity availableCapacity;

        private String clientToken;

        private String hostId;

        private HostProperties hostProperties;

        private String hostReservationId;

        private List<HostInstance> instances = DefaultSdkAutoConstructList.getInstance();

        private String state;

        private Instant allocationTime;

        private Instant releaseTime;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String hostRecovery;

        private String allowsMultipleInstanceTypes;

        private String ownerId;

        private String availabilityZoneId;

        private Boolean memberOfServiceLinkedResourceGroup;

        private BuilderImpl() {
        }

        private BuilderImpl(Host model) {
            autoPlacement(model.autoPlacement);
            availabilityZone(model.availabilityZone);
            availableCapacity(model.availableCapacity);
            clientToken(model.clientToken);
            hostId(model.hostId);
            hostProperties(model.hostProperties);
            hostReservationId(model.hostReservationId);
            instances(model.instances);
            state(model.state);
            allocationTime(model.allocationTime);
            releaseTime(model.releaseTime);
            tags(model.tags);
            hostRecovery(model.hostRecovery);
            allowsMultipleInstanceTypes(model.allowsMultipleInstanceTypes);
            ownerId(model.ownerId);
            availabilityZoneId(model.availabilityZoneId);
            memberOfServiceLinkedResourceGroup(model.memberOfServiceLinkedResourceGroup);
        }

        public final String getAutoPlacement() {
            return autoPlacement;
        }

        public final void setAutoPlacement(String autoPlacement) {
            this.autoPlacement = autoPlacement;
        }

        @Override
        @Transient
        public final Builder autoPlacement(String autoPlacement) {
            this.autoPlacement = autoPlacement;
            return this;
        }

        @Override
        @Transient
        public final Builder autoPlacement(AutoPlacement autoPlacement) {
            this.autoPlacement(autoPlacement == null ? null : autoPlacement.toString());
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        @Transient
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final AvailableCapacity.Builder getAvailableCapacity() {
            return availableCapacity != null ? availableCapacity.toBuilder() : null;
        }

        public final void setAvailableCapacity(AvailableCapacity.BuilderImpl availableCapacity) {
            this.availableCapacity = availableCapacity != null ? availableCapacity.build() : null;
        }

        @Override
        @Transient
        public final Builder availableCapacity(AvailableCapacity availableCapacity) {
            this.availableCapacity = availableCapacity;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        @Transient
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getHostId() {
            return hostId;
        }

        public final void setHostId(String hostId) {
            this.hostId = hostId;
        }

        @Override
        @Transient
        public final Builder hostId(String hostId) {
            this.hostId = hostId;
            return this;
        }

        public final HostProperties.Builder getHostProperties() {
            return hostProperties != null ? hostProperties.toBuilder() : null;
        }

        public final void setHostProperties(HostProperties.BuilderImpl hostProperties) {
            this.hostProperties = hostProperties != null ? hostProperties.build() : null;
        }

        @Override
        @Transient
        public final Builder hostProperties(HostProperties hostProperties) {
            this.hostProperties = hostProperties;
            return this;
        }

        public final String getHostReservationId() {
            return hostReservationId;
        }

        public final void setHostReservationId(String hostReservationId) {
            this.hostReservationId = hostReservationId;
        }

        @Override
        @Transient
        public final Builder hostReservationId(String hostReservationId) {
            this.hostReservationId = hostReservationId;
            return this;
        }

        public final List<HostInstance.Builder> getInstances() {
            List<HostInstance.Builder> result = HostInstanceListCopier.copyToBuilder(this.instances);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstances(Collection<HostInstance.BuilderImpl> instances) {
            this.instances = HostInstanceListCopier.copyFromBuilder(instances);
        }

        @Override
        @Transient
        public final Builder instances(Collection<HostInstance> instances) {
            this.instances = HostInstanceListCopier.copy(instances);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder instances(HostInstance... instances) {
            instances(Arrays.asList(instances));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder instances(Consumer<HostInstance.Builder>... instances) {
            instances(Stream.of(instances).map(c -> HostInstance.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        @Transient
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        @Transient
        public final Builder state(AllocationState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final Instant getAllocationTime() {
            return allocationTime;
        }

        public final void setAllocationTime(Instant allocationTime) {
            this.allocationTime = allocationTime;
        }

        @Override
        @Transient
        public final Builder allocationTime(Instant allocationTime) {
            this.allocationTime = allocationTime;
            return this;
        }

        public final Instant getReleaseTime() {
            return releaseTime;
        }

        public final void setReleaseTime(Instant releaseTime) {
            this.releaseTime = releaseTime;
        }

        @Override
        @Transient
        public final Builder releaseTime(Instant releaseTime) {
            this.releaseTime = releaseTime;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        @Transient
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @Transient
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getHostRecovery() {
            return hostRecovery;
        }

        public final void setHostRecovery(String hostRecovery) {
            this.hostRecovery = hostRecovery;
        }

        @Override
        @Transient
        public final Builder hostRecovery(String hostRecovery) {
            this.hostRecovery = hostRecovery;
            return this;
        }

        @Override
        @Transient
        public final Builder hostRecovery(HostRecovery hostRecovery) {
            this.hostRecovery(hostRecovery == null ? null : hostRecovery.toString());
            return this;
        }

        public final String getAllowsMultipleInstanceTypes() {
            return allowsMultipleInstanceTypes;
        }

        public final void setAllowsMultipleInstanceTypes(String allowsMultipleInstanceTypes) {
            this.allowsMultipleInstanceTypes = allowsMultipleInstanceTypes;
        }

        @Override
        @Transient
        public final Builder allowsMultipleInstanceTypes(String allowsMultipleInstanceTypes) {
            this.allowsMultipleInstanceTypes = allowsMultipleInstanceTypes;
            return this;
        }

        @Override
        @Transient
        public final Builder allowsMultipleInstanceTypes(AllowsMultipleInstanceTypes allowsMultipleInstanceTypes) {
            this.allowsMultipleInstanceTypes(allowsMultipleInstanceTypes == null ? null : allowsMultipleInstanceTypes.toString());
            return this;
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        @Transient
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        @Transient
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final Boolean getMemberOfServiceLinkedResourceGroup() {
            return memberOfServiceLinkedResourceGroup;
        }

        public final void setMemberOfServiceLinkedResourceGroup(Boolean memberOfServiceLinkedResourceGroup) {
            this.memberOfServiceLinkedResourceGroup = memberOfServiceLinkedResourceGroup;
        }

        @Override
        @Transient
        public final Builder memberOfServiceLinkedResourceGroup(Boolean memberOfServiceLinkedResourceGroup) {
            this.memberOfServiceLinkedResourceGroup = memberOfServiceLinkedResourceGroup;
            return this;
        }

        @Override
        public Host build() {
            return new Host(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
