/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a VPN connection.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VpnConnection implements SdkPojo, Serializable, ToCopyableBuilder<VpnConnection.Builder, VpnConnection> {
    private static final SdkField<String> CUSTOMER_GATEWAY_CONFIGURATION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CustomerGatewayConfiguration")
            .getter(getter(VpnConnection::customerGatewayConfiguration))
            .setter(setter(Builder::customerGatewayConfiguration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerGatewayConfiguration")
                    .unmarshallLocationName("customerGatewayConfiguration").build()).build();

    private static final SdkField<String> CUSTOMER_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CustomerGatewayId")
            .getter(getter(VpnConnection::customerGatewayId))
            .setter(setter(Builder::customerGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CustomerGatewayId")
                    .unmarshallLocationName("customerGatewayId").build()).build();

    private static final SdkField<String> CATEGORY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Category")
            .getter(getter(VpnConnection::category))
            .setter(setter(Builder::category))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Category")
                    .unmarshallLocationName("category").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(VpnConnection::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Type")
            .getter(getter(VpnConnection::typeAsString))
            .setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type")
                    .unmarshallLocationName("type").build()).build();

    private static final SdkField<String> VPN_CONNECTION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpnConnectionId")
            .getter(getter(VpnConnection::vpnConnectionId))
            .setter(setter(Builder::vpnConnectionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnConnectionId")
                    .unmarshallLocationName("vpnConnectionId").build()).build();

    private static final SdkField<String> VPN_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpnGatewayId")
            .getter(getter(VpnConnection::vpnGatewayId))
            .setter(setter(Builder::vpnGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpnGatewayId")
                    .unmarshallLocationName("vpnGatewayId").build()).build();

    private static final SdkField<String> TRANSIT_GATEWAY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("TransitGatewayId")
            .getter(getter(VpnConnection::transitGatewayId))
            .setter(setter(Builder::transitGatewayId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TransitGatewayId")
                    .unmarshallLocationName("transitGatewayId").build()).build();

    private static final SdkField<String> CORE_NETWORK_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CoreNetworkArn")
            .getter(getter(VpnConnection::coreNetworkArn))
            .setter(setter(Builder::coreNetworkArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CoreNetworkArn")
                    .unmarshallLocationName("coreNetworkArn").build()).build();

    private static final SdkField<String> CORE_NETWORK_ATTACHMENT_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CoreNetworkAttachmentArn")
            .getter(getter(VpnConnection::coreNetworkAttachmentArn))
            .setter(setter(Builder::coreNetworkAttachmentArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CoreNetworkAttachmentArn")
                    .unmarshallLocationName("coreNetworkAttachmentArn").build()).build();

    private static final SdkField<String> GATEWAY_ASSOCIATION_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GatewayAssociationState")
            .getter(getter(VpnConnection::gatewayAssociationStateAsString))
            .setter(setter(Builder::gatewayAssociationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GatewayAssociationState")
                    .unmarshallLocationName("gatewayAssociationState").build()).build();

    private static final SdkField<VpnConnectionOptions> OPTIONS_FIELD = SdkField
            .<VpnConnectionOptions> builder(MarshallingType.SDK_POJO)
            .memberName("Options")
            .getter(getter(VpnConnection::options))
            .setter(setter(Builder::options))
            .constructor(VpnConnectionOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Options")
                    .unmarshallLocationName("options").build()).build();

    private static final SdkField<List<VpnStaticRoute>> ROUTES_FIELD = SdkField
            .<List<VpnStaticRoute>> builder(MarshallingType.LIST)
            .memberName("Routes")
            .getter(getter(VpnConnection::routes))
            .setter(setter(Builder::routes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Routes")
                    .unmarshallLocationName("routes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<VpnStaticRoute> builder(MarshallingType.SDK_POJO)
                                            .constructor(VpnStaticRoute::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VpnConnection::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<VgwTelemetry>> VGW_TELEMETRY_FIELD = SdkField
            .<List<VgwTelemetry>> builder(MarshallingType.LIST)
            .memberName("VgwTelemetry")
            .getter(getter(VpnConnection::vgwTelemetry))
            .setter(setter(Builder::vgwTelemetry))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VgwTelemetry")
                    .unmarshallLocationName("vgwTelemetry").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<VgwTelemetry> builder(MarshallingType.SDK_POJO)
                                            .constructor(VgwTelemetry::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CUSTOMER_GATEWAY_CONFIGURATION_FIELD, CUSTOMER_GATEWAY_ID_FIELD, CATEGORY_FIELD, STATE_FIELD, TYPE_FIELD,
            VPN_CONNECTION_ID_FIELD, VPN_GATEWAY_ID_FIELD, TRANSIT_GATEWAY_ID_FIELD, CORE_NETWORK_ARN_FIELD,
            CORE_NETWORK_ATTACHMENT_ARN_FIELD, GATEWAY_ASSOCIATION_STATE_FIELD, OPTIONS_FIELD, ROUTES_FIELD, TAGS_FIELD,
            VGW_TELEMETRY_FIELD));

    private static final long serialVersionUID = 1L;

    private final String customerGatewayConfiguration;

    private final String customerGatewayId;

    private final String category;

    private final String state;

    private final String type;

    private final String vpnConnectionId;

    private final String vpnGatewayId;

    private final String transitGatewayId;

    private final String coreNetworkArn;

    private final String coreNetworkAttachmentArn;

    private final String gatewayAssociationState;

    private final VpnConnectionOptions options;

    private final List<VpnStaticRoute> routes;

    private final List<Tag> tags;

    private final List<VgwTelemetry> vgwTelemetry;

    private VpnConnection(BuilderImpl builder) {
        this.customerGatewayConfiguration = builder.customerGatewayConfiguration;
        this.customerGatewayId = builder.customerGatewayId;
        this.category = builder.category;
        this.state = builder.state;
        this.type = builder.type;
        this.vpnConnectionId = builder.vpnConnectionId;
        this.vpnGatewayId = builder.vpnGatewayId;
        this.transitGatewayId = builder.transitGatewayId;
        this.coreNetworkArn = builder.coreNetworkArn;
        this.coreNetworkAttachmentArn = builder.coreNetworkAttachmentArn;
        this.gatewayAssociationState = builder.gatewayAssociationState;
        this.options = builder.options;
        this.routes = builder.routes;
        this.tags = builder.tags;
        this.vgwTelemetry = builder.vgwTelemetry;
    }

    /**
     * <p>
     * The configuration information for the VPN connection's customer gateway (in the native XML format). This element
     * is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
     * <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
     * <code>available</code> state.
     * </p>
     * 
     * @return The configuration information for the VPN connection's customer gateway (in the native XML format). This
     *         element is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
     *         <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
     *         <code>available</code> state.
     */
    public final String customerGatewayConfiguration() {
        return customerGatewayConfiguration;
    }

    /**
     * <p>
     * The ID of the customer gateway at your end of the VPN connection.
     * </p>
     * 
     * @return The ID of the customer gateway at your end of the VPN connection.
     */
    public final String customerGatewayId() {
        return customerGatewayId;
    }

    /**
     * <p>
     * The category of the VPN connection. A value of <code>VPN</code> indicates an Amazon Web Services VPN connection.
     * A value of <code>VPN-Classic</code> indicates an Amazon Web Services Classic VPN connection.
     * </p>
     * 
     * @return The category of the VPN connection. A value of <code>VPN</code> indicates an Amazon Web Services VPN
     *         connection. A value of <code>VPN-Classic</code> indicates an Amazon Web Services Classic VPN connection.
     */
    public final String category() {
        return category;
    }

    /**
     * <p>
     * The current state of the VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpnState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the VPN connection.
     * @see VpnState
     */
    public final VpnState state() {
        return VpnState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link VpnState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the VPN connection.
     * @see VpnState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The type of VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GatewayType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of VPN connection.
     * @see GatewayType
     */
    public final GatewayType type() {
        return GatewayType.fromValue(type);
    }

    /**
     * <p>
     * The type of VPN connection.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link GatewayType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of VPN connection.
     * @see GatewayType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The ID of the VPN connection.
     * </p>
     * 
     * @return The ID of the VPN connection.
     */
    public final String vpnConnectionId() {
        return vpnConnectionId;
    }

    /**
     * <p>
     * The ID of the virtual private gateway at the Amazon Web Services side of the VPN connection.
     * </p>
     * 
     * @return The ID of the virtual private gateway at the Amazon Web Services side of the VPN connection.
     */
    public final String vpnGatewayId() {
        return vpnGatewayId;
    }

    /**
     * <p>
     * The ID of the transit gateway associated with the VPN connection.
     * </p>
     * 
     * @return The ID of the transit gateway associated with the VPN connection.
     */
    public final String transitGatewayId() {
        return transitGatewayId;
    }

    /**
     * <p>
     * The ARN of the core network.
     * </p>
     * 
     * @return The ARN of the core network.
     */
    public final String coreNetworkArn() {
        return coreNetworkArn;
    }

    /**
     * <p>
     * The ARN of the core network attachment.
     * </p>
     * 
     * @return The ARN of the core network attachment.
     */
    public final String coreNetworkAttachmentArn() {
        return coreNetworkAttachmentArn;
    }

    /**
     * <p>
     * The current state of the gateway association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gatewayAssociationState} will return {@link GatewayAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #gatewayAssociationStateAsString}.
     * </p>
     * 
     * @return The current state of the gateway association.
     * @see GatewayAssociationState
     */
    public final GatewayAssociationState gatewayAssociationState() {
        return GatewayAssociationState.fromValue(gatewayAssociationState);
    }

    /**
     * <p>
     * The current state of the gateway association.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #gatewayAssociationState} will return {@link GatewayAssociationState#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #gatewayAssociationStateAsString}.
     * </p>
     * 
     * @return The current state of the gateway association.
     * @see GatewayAssociationState
     */
    public final String gatewayAssociationStateAsString() {
        return gatewayAssociationState;
    }

    /**
     * <p>
     * The VPN connection options.
     * </p>
     * 
     * @return The VPN connection options.
     */
    public final VpnConnectionOptions options() {
        return options;
    }

    /**
     * For responses, this returns true if the service returned a value for the Routes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasRoutes() {
        return routes != null && !(routes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The static routes associated with the VPN connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRoutes} method.
     * </p>
     * 
     * @return The static routes associated with the VPN connection.
     */
    public final List<VpnStaticRoute> routes() {
        return routes;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the VPN connection.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the VPN connection.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * For responses, this returns true if the service returned a value for the VgwTelemetry property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasVgwTelemetry() {
        return vgwTelemetry != null && !(vgwTelemetry instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Information about the VPN tunnel.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasVgwTelemetry} method.
     * </p>
     * 
     * @return Information about the VPN tunnel.
     */
    public final List<VgwTelemetry> vgwTelemetry() {
        return vgwTelemetry;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(customerGatewayConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(customerGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(category());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpnConnectionId());
        hashCode = 31 * hashCode + Objects.hashCode(vpnGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(transitGatewayId());
        hashCode = 31 * hashCode + Objects.hashCode(coreNetworkArn());
        hashCode = 31 * hashCode + Objects.hashCode(coreNetworkAttachmentArn());
        hashCode = 31 * hashCode + Objects.hashCode(gatewayAssociationStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(options());
        hashCode = 31 * hashCode + Objects.hashCode(hasRoutes() ? routes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasVgwTelemetry() ? vgwTelemetry() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VpnConnection)) {
            return false;
        }
        VpnConnection other = (VpnConnection) obj;
        return Objects.equals(customerGatewayConfiguration(), other.customerGatewayConfiguration())
                && Objects.equals(customerGatewayId(), other.customerGatewayId()) && Objects.equals(category(), other.category())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(typeAsString(), other.typeAsString())
                && Objects.equals(vpnConnectionId(), other.vpnConnectionId())
                && Objects.equals(vpnGatewayId(), other.vpnGatewayId())
                && Objects.equals(transitGatewayId(), other.transitGatewayId())
                && Objects.equals(coreNetworkArn(), other.coreNetworkArn())
                && Objects.equals(coreNetworkAttachmentArn(), other.coreNetworkAttachmentArn())
                && Objects.equals(gatewayAssociationStateAsString(), other.gatewayAssociationStateAsString())
                && Objects.equals(options(), other.options()) && hasRoutes() == other.hasRoutes()
                && Objects.equals(routes(), other.routes()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && hasVgwTelemetry() == other.hasVgwTelemetry()
                && Objects.equals(vgwTelemetry(), other.vgwTelemetry());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VpnConnection").add("CustomerGatewayConfiguration", customerGatewayConfiguration())
                .add("CustomerGatewayId", customerGatewayId()).add("Category", category()).add("State", stateAsString())
                .add("Type", typeAsString()).add("VpnConnectionId", vpnConnectionId()).add("VpnGatewayId", vpnGatewayId())
                .add("TransitGatewayId", transitGatewayId()).add("CoreNetworkArn", coreNetworkArn())
                .add("CoreNetworkAttachmentArn", coreNetworkAttachmentArn())
                .add("GatewayAssociationState", gatewayAssociationStateAsString()).add("Options", options())
                .add("Routes", hasRoutes() ? routes() : null).add("Tags", hasTags() ? tags() : null)
                .add("VgwTelemetry", hasVgwTelemetry() ? vgwTelemetry() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CustomerGatewayConfiguration":
            return Optional.ofNullable(clazz.cast(customerGatewayConfiguration()));
        case "CustomerGatewayId":
            return Optional.ofNullable(clazz.cast(customerGatewayId()));
        case "Category":
            return Optional.ofNullable(clazz.cast(category()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "VpnConnectionId":
            return Optional.ofNullable(clazz.cast(vpnConnectionId()));
        case "VpnGatewayId":
            return Optional.ofNullable(clazz.cast(vpnGatewayId()));
        case "TransitGatewayId":
            return Optional.ofNullable(clazz.cast(transitGatewayId()));
        case "CoreNetworkArn":
            return Optional.ofNullable(clazz.cast(coreNetworkArn()));
        case "CoreNetworkAttachmentArn":
            return Optional.ofNullable(clazz.cast(coreNetworkAttachmentArn()));
        case "GatewayAssociationState":
            return Optional.ofNullable(clazz.cast(gatewayAssociationStateAsString()));
        case "Options":
            return Optional.ofNullable(clazz.cast(options()));
        case "Routes":
            return Optional.ofNullable(clazz.cast(routes()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "VgwTelemetry":
            return Optional.ofNullable(clazz.cast(vgwTelemetry()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VpnConnection, T> g) {
        return obj -> g.apply((VpnConnection) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VpnConnection> {
        /**
         * <p>
         * The configuration information for the VPN connection's customer gateway (in the native XML format). This
         * element is always present in the <a>CreateVpnConnection</a> response; however, it's present in the
         * <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code> or
         * <code>available</code> state.
         * </p>
         * 
         * @param customerGatewayConfiguration
         *        The configuration information for the VPN connection's customer gateway (in the native XML format).
         *        This element is always present in the <a>CreateVpnConnection</a> response; however, it's present in
         *        the <a>DescribeVpnConnections</a> response only if the VPN connection is in the <code>pending</code>
         *        or <code>available</code> state.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerGatewayConfiguration(String customerGatewayConfiguration);

        /**
         * <p>
         * The ID of the customer gateway at your end of the VPN connection.
         * </p>
         * 
         * @param customerGatewayId
         *        The ID of the customer gateway at your end of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder customerGatewayId(String customerGatewayId);

        /**
         * <p>
         * The category of the VPN connection. A value of <code>VPN</code> indicates an Amazon Web Services VPN
         * connection. A value of <code>VPN-Classic</code> indicates an Amazon Web Services Classic VPN connection.
         * </p>
         * 
         * @param category
         *        The category of the VPN connection. A value of <code>VPN</code> indicates an Amazon Web Services VPN
         *        connection. A value of <code>VPN-Classic</code> indicates an Amazon Web Services Classic VPN
         *        connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder category(String category);

        /**
         * <p>
         * The current state of the VPN connection.
         * </p>
         * 
         * @param state
         *        The current state of the VPN connection.
         * @see VpnState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the VPN connection.
         * </p>
         * 
         * @param state
         *        The current state of the VPN connection.
         * @see VpnState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VpnState
         */
        Builder state(VpnState state);

        /**
         * <p>
         * The type of VPN connection.
         * </p>
         * 
         * @param type
         *        The type of VPN connection.
         * @see GatewayType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of VPN connection.
         * </p>
         * 
         * @param type
         *        The type of VPN connection.
         * @see GatewayType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayType
         */
        Builder type(GatewayType type);

        /**
         * <p>
         * The ID of the VPN connection.
         * </p>
         * 
         * @param vpnConnectionId
         *        The ID of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpnConnectionId(String vpnConnectionId);

        /**
         * <p>
         * The ID of the virtual private gateway at the Amazon Web Services side of the VPN connection.
         * </p>
         * 
         * @param vpnGatewayId
         *        The ID of the virtual private gateway at the Amazon Web Services side of the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpnGatewayId(String vpnGatewayId);

        /**
         * <p>
         * The ID of the transit gateway associated with the VPN connection.
         * </p>
         * 
         * @param transitGatewayId
         *        The ID of the transit gateway associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder transitGatewayId(String transitGatewayId);

        /**
         * <p>
         * The ARN of the core network.
         * </p>
         * 
         * @param coreNetworkArn
         *        The ARN of the core network.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder coreNetworkArn(String coreNetworkArn);

        /**
         * <p>
         * The ARN of the core network attachment.
         * </p>
         * 
         * @param coreNetworkAttachmentArn
         *        The ARN of the core network attachment.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder coreNetworkAttachmentArn(String coreNetworkAttachmentArn);

        /**
         * <p>
         * The current state of the gateway association.
         * </p>
         * 
         * @param gatewayAssociationState
         *        The current state of the gateway association.
         * @see GatewayAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayAssociationState
         */
        Builder gatewayAssociationState(String gatewayAssociationState);

        /**
         * <p>
         * The current state of the gateway association.
         * </p>
         * 
         * @param gatewayAssociationState
         *        The current state of the gateway association.
         * @see GatewayAssociationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see GatewayAssociationState
         */
        Builder gatewayAssociationState(GatewayAssociationState gatewayAssociationState);

        /**
         * <p>
         * The VPN connection options.
         * </p>
         * 
         * @param options
         *        The VPN connection options.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder options(VpnConnectionOptions options);

        /**
         * <p>
         * The VPN connection options.
         * </p>
         * This is a convenience method that creates an instance of the {@link VpnConnectionOptions.Builder} avoiding
         * the need to create one manually via {@link VpnConnectionOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VpnConnectionOptions.Builder#build()} is called immediately and
         * its result is passed to {@link #options(VpnConnectionOptions)}.
         * 
         * @param options
         *        a consumer that will call methods on {@link VpnConnectionOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #options(VpnConnectionOptions)
         */
        default Builder options(Consumer<VpnConnectionOptions.Builder> options) {
            return options(VpnConnectionOptions.builder().applyMutation(options).build());
        }

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * 
         * @param routes
         *        The static routes associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routes(Collection<VpnStaticRoute> routes);

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * 
         * @param routes
         *        The static routes associated with the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routes(VpnStaticRoute... routes);

        /**
         * <p>
         * The static routes associated with the VPN connection.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.VpnStaticRoute.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.VpnStaticRoute#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.VpnStaticRoute.Builder#build()} is called immediately and
         * its result is passed to {@link #routes(List<VpnStaticRoute>)}.
         * 
         * @param routes
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.VpnStaticRoute.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #routes(java.util.Collection<VpnStaticRoute>)
         */
        Builder routes(Consumer<VpnStaticRoute.Builder>... routes);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the VPN connection.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the VPN connection.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * 
         * @param vgwTelemetry
         *        Information about the VPN tunnel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vgwTelemetry(Collection<VgwTelemetry> vgwTelemetry);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * 
         * @param vgwTelemetry
         *        Information about the VPN tunnel.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vgwTelemetry(VgwTelemetry... vgwTelemetry);

        /**
         * <p>
         * Information about the VPN tunnel.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.VgwTelemetry.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.VgwTelemetry#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.VgwTelemetry.Builder#build()} is called immediately and its
         * result is passed to {@link #vgwTelemetry(List<VgwTelemetry>)}.
         * 
         * @param vgwTelemetry
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.VgwTelemetry.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vgwTelemetry(java.util.Collection<VgwTelemetry>)
         */
        Builder vgwTelemetry(Consumer<VgwTelemetry.Builder>... vgwTelemetry);
    }

    static final class BuilderImpl implements Builder {
        private String customerGatewayConfiguration;

        private String customerGatewayId;

        private String category;

        private String state;

        private String type;

        private String vpnConnectionId;

        private String vpnGatewayId;

        private String transitGatewayId;

        private String coreNetworkArn;

        private String coreNetworkAttachmentArn;

        private String gatewayAssociationState;

        private VpnConnectionOptions options;

        private List<VpnStaticRoute> routes = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<VgwTelemetry> vgwTelemetry = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(VpnConnection model) {
            customerGatewayConfiguration(model.customerGatewayConfiguration);
            customerGatewayId(model.customerGatewayId);
            category(model.category);
            state(model.state);
            type(model.type);
            vpnConnectionId(model.vpnConnectionId);
            vpnGatewayId(model.vpnGatewayId);
            transitGatewayId(model.transitGatewayId);
            coreNetworkArn(model.coreNetworkArn);
            coreNetworkAttachmentArn(model.coreNetworkAttachmentArn);
            gatewayAssociationState(model.gatewayAssociationState);
            options(model.options);
            routes(model.routes);
            tags(model.tags);
            vgwTelemetry(model.vgwTelemetry);
        }

        public final String getCustomerGatewayConfiguration() {
            return customerGatewayConfiguration;
        }

        public final void setCustomerGatewayConfiguration(String customerGatewayConfiguration) {
            this.customerGatewayConfiguration = customerGatewayConfiguration;
        }

        @Override
        public final Builder customerGatewayConfiguration(String customerGatewayConfiguration) {
            this.customerGatewayConfiguration = customerGatewayConfiguration;
            return this;
        }

        public final String getCustomerGatewayId() {
            return customerGatewayId;
        }

        public final void setCustomerGatewayId(String customerGatewayId) {
            this.customerGatewayId = customerGatewayId;
        }

        @Override
        public final Builder customerGatewayId(String customerGatewayId) {
            this.customerGatewayId = customerGatewayId;
            return this;
        }

        public final String getCategory() {
            return category;
        }

        public final void setCategory(String category) {
            this.category = category;
        }

        @Override
        public final Builder category(String category) {
            this.category = category;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(VpnState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(GatewayType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getVpnConnectionId() {
            return vpnConnectionId;
        }

        public final void setVpnConnectionId(String vpnConnectionId) {
            this.vpnConnectionId = vpnConnectionId;
        }

        @Override
        public final Builder vpnConnectionId(String vpnConnectionId) {
            this.vpnConnectionId = vpnConnectionId;
            return this;
        }

        public final String getVpnGatewayId() {
            return vpnGatewayId;
        }

        public final void setVpnGatewayId(String vpnGatewayId) {
            this.vpnGatewayId = vpnGatewayId;
        }

        @Override
        public final Builder vpnGatewayId(String vpnGatewayId) {
            this.vpnGatewayId = vpnGatewayId;
            return this;
        }

        public final String getTransitGatewayId() {
            return transitGatewayId;
        }

        public final void setTransitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
        }

        @Override
        public final Builder transitGatewayId(String transitGatewayId) {
            this.transitGatewayId = transitGatewayId;
            return this;
        }

        public final String getCoreNetworkArn() {
            return coreNetworkArn;
        }

        public final void setCoreNetworkArn(String coreNetworkArn) {
            this.coreNetworkArn = coreNetworkArn;
        }

        @Override
        public final Builder coreNetworkArn(String coreNetworkArn) {
            this.coreNetworkArn = coreNetworkArn;
            return this;
        }

        public final String getCoreNetworkAttachmentArn() {
            return coreNetworkAttachmentArn;
        }

        public final void setCoreNetworkAttachmentArn(String coreNetworkAttachmentArn) {
            this.coreNetworkAttachmentArn = coreNetworkAttachmentArn;
        }

        @Override
        public final Builder coreNetworkAttachmentArn(String coreNetworkAttachmentArn) {
            this.coreNetworkAttachmentArn = coreNetworkAttachmentArn;
            return this;
        }

        public final String getGatewayAssociationState() {
            return gatewayAssociationState;
        }

        public final void setGatewayAssociationState(String gatewayAssociationState) {
            this.gatewayAssociationState = gatewayAssociationState;
        }

        @Override
        public final Builder gatewayAssociationState(String gatewayAssociationState) {
            this.gatewayAssociationState = gatewayAssociationState;
            return this;
        }

        @Override
        public final Builder gatewayAssociationState(GatewayAssociationState gatewayAssociationState) {
            this.gatewayAssociationState(gatewayAssociationState == null ? null : gatewayAssociationState.toString());
            return this;
        }

        public final VpnConnectionOptions.Builder getOptions() {
            return options != null ? options.toBuilder() : null;
        }

        public final void setOptions(VpnConnectionOptions.BuilderImpl options) {
            this.options = options != null ? options.build() : null;
        }

        @Override
        public final Builder options(VpnConnectionOptions options) {
            this.options = options;
            return this;
        }

        public final List<VpnStaticRoute.Builder> getRoutes() {
            List<VpnStaticRoute.Builder> result = VpnStaticRouteListCopier.copyToBuilder(this.routes);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRoutes(Collection<VpnStaticRoute.BuilderImpl> routes) {
            this.routes = VpnStaticRouteListCopier.copyFromBuilder(routes);
        }

        @Override
        public final Builder routes(Collection<VpnStaticRoute> routes) {
            this.routes = VpnStaticRouteListCopier.copy(routes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routes(VpnStaticRoute... routes) {
            routes(Arrays.asList(routes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder routes(Consumer<VpnStaticRoute.Builder>... routes) {
            routes(Stream.of(routes).map(c -> VpnStaticRoute.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final List<VgwTelemetry.Builder> getVgwTelemetry() {
            List<VgwTelemetry.Builder> result = VgwTelemetryListCopier.copyToBuilder(this.vgwTelemetry);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setVgwTelemetry(Collection<VgwTelemetry.BuilderImpl> vgwTelemetry) {
            this.vgwTelemetry = VgwTelemetryListCopier.copyFromBuilder(vgwTelemetry);
        }

        @Override
        public final Builder vgwTelemetry(Collection<VgwTelemetry> vgwTelemetry) {
            this.vgwTelemetry = VgwTelemetryListCopier.copy(vgwTelemetry);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vgwTelemetry(VgwTelemetry... vgwTelemetry) {
            vgwTelemetry(Arrays.asList(vgwTelemetry));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder vgwTelemetry(Consumer<VgwTelemetry.Builder>... vgwTelemetry) {
            vgwTelemetry(Stream.of(vgwTelemetry).map(c -> VgwTelemetry.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public VpnConnection build() {
            return new VpnConnection(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
