/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An Amazon Web Services Verified Access endpoint specifies the application that Amazon Web Services Verified Access
 * provides access to. It must be attached to an Amazon Web Services Verified Access group. An Amazon Web Services
 * Verified Access endpoint must also have an attached access policy before you attached it to a group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VerifiedAccessEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<VerifiedAccessEndpoint.Builder, VerifiedAccessEndpoint> {
    private static final SdkField<String> VERIFIED_ACCESS_INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VerifiedAccessInstanceId")
            .getter(getter(VerifiedAccessEndpoint::verifiedAccessInstanceId))
            .setter(setter(Builder::verifiedAccessInstanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifiedAccessInstanceId")
                    .unmarshallLocationName("verifiedAccessInstanceId").build()).build();

    private static final SdkField<String> VERIFIED_ACCESS_GROUP_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VerifiedAccessGroupId")
            .getter(getter(VerifiedAccessEndpoint::verifiedAccessGroupId))
            .setter(setter(Builder::verifiedAccessGroupId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifiedAccessGroupId")
                    .unmarshallLocationName("verifiedAccessGroupId").build()).build();

    private static final SdkField<String> VERIFIED_ACCESS_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VerifiedAccessEndpointId")
            .getter(getter(VerifiedAccessEndpoint::verifiedAccessEndpointId))
            .setter(setter(Builder::verifiedAccessEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VerifiedAccessEndpointId")
                    .unmarshallLocationName("verifiedAccessEndpointId").build()).build();

    private static final SdkField<String> APPLICATION_DOMAIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ApplicationDomain")
            .getter(getter(VerifiedAccessEndpoint::applicationDomain))
            .setter(setter(Builder::applicationDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ApplicationDomain")
                    .unmarshallLocationName("applicationDomain").build()).build();

    private static final SdkField<String> ENDPOINT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointType")
            .getter(getter(VerifiedAccessEndpoint::endpointTypeAsString))
            .setter(setter(Builder::endpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointType")
                    .unmarshallLocationName("endpointType").build()).build();

    private static final SdkField<String> ATTACHMENT_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AttachmentType")
            .getter(getter(VerifiedAccessEndpoint::attachmentTypeAsString))
            .setter(setter(Builder::attachmentType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AttachmentType")
                    .unmarshallLocationName("attachmentType").build()).build();

    private static final SdkField<String> DOMAIN_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DomainCertificateArn")
            .getter(getter(VerifiedAccessEndpoint::domainCertificateArn))
            .setter(setter(Builder::domainCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DomainCertificateArn")
                    .unmarshallLocationName("domainCertificateArn").build()).build();

    private static final SdkField<String> ENDPOINT_DOMAIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EndpointDomain")
            .getter(getter(VerifiedAccessEndpoint::endpointDomain))
            .setter(setter(Builder::endpointDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointDomain")
                    .unmarshallLocationName("endpointDomain").build()).build();

    private static final SdkField<String> DEVICE_VALIDATION_DOMAIN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeviceValidationDomain")
            .getter(getter(VerifiedAccessEndpoint::deviceValidationDomain))
            .setter(setter(Builder::deviceValidationDomain))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeviceValidationDomain")
                    .unmarshallLocationName("deviceValidationDomain").build()).build();

    private static final SdkField<List<String>> SECURITY_GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SecurityGroupIds")
            .getter(getter(VerifiedAccessEndpoint::securityGroupIds))
            .setter(setter(Builder::securityGroupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SecurityGroupIdSet")
                    .unmarshallLocationName("securityGroupIdSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<VerifiedAccessEndpointLoadBalancerOptions> LOAD_BALANCER_OPTIONS_FIELD = SdkField
            .<VerifiedAccessEndpointLoadBalancerOptions> builder(MarshallingType.SDK_POJO)
            .memberName("LoadBalancerOptions")
            .getter(getter(VerifiedAccessEndpoint::loadBalancerOptions))
            .setter(setter(Builder::loadBalancerOptions))
            .constructor(VerifiedAccessEndpointLoadBalancerOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LoadBalancerOptions")
                    .unmarshallLocationName("loadBalancerOptions").build()).build();

    private static final SdkField<VerifiedAccessEndpointEniOptions> NETWORK_INTERFACE_OPTIONS_FIELD = SdkField
            .<VerifiedAccessEndpointEniOptions> builder(MarshallingType.SDK_POJO)
            .memberName("NetworkInterfaceOptions")
            .getter(getter(VerifiedAccessEndpoint::networkInterfaceOptions))
            .setter(setter(Builder::networkInterfaceOptions))
            .constructor(VerifiedAccessEndpointEniOptions::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceOptions")
                    .unmarshallLocationName("networkInterfaceOptions").build()).build();

    private static final SdkField<VerifiedAccessEndpointStatus> STATUS_FIELD = SdkField
            .<VerifiedAccessEndpointStatus> builder(MarshallingType.SDK_POJO)
            .memberName("Status")
            .getter(getter(VerifiedAccessEndpoint::status))
            .setter(setter(Builder::status))
            .constructor(VerifiedAccessEndpointStatus::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status")
                    .unmarshallLocationName("status").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(VerifiedAccessEndpoint::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<String> CREATION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CreationTime")
            .getter(getter(VerifiedAccessEndpoint::creationTime))
            .setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime")
                    .unmarshallLocationName("creationTime").build()).build();

    private static final SdkField<String> LAST_UPDATED_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LastUpdatedTime")
            .getter(getter(VerifiedAccessEndpoint::lastUpdatedTime))
            .setter(setter(Builder::lastUpdatedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastUpdatedTime")
                    .unmarshallLocationName("lastUpdatedTime").build()).build();

    private static final SdkField<String> DELETION_TIME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DeletionTime")
            .getter(getter(VerifiedAccessEndpoint::deletionTime))
            .setter(setter(Builder::deletionTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DeletionTime")
                    .unmarshallLocationName("deletionTime").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(VerifiedAccessEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<VerifiedAccessSseSpecificationResponse> SSE_SPECIFICATION_FIELD = SdkField
            .<VerifiedAccessSseSpecificationResponse> builder(MarshallingType.SDK_POJO)
            .memberName("SseSpecification")
            .getter(getter(VerifiedAccessEndpoint::sseSpecification))
            .setter(setter(Builder::sseSpecification))
            .constructor(VerifiedAccessSseSpecificationResponse::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SseSpecification")
                    .unmarshallLocationName("sseSpecification").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            VERIFIED_ACCESS_INSTANCE_ID_FIELD, VERIFIED_ACCESS_GROUP_ID_FIELD, VERIFIED_ACCESS_ENDPOINT_ID_FIELD,
            APPLICATION_DOMAIN_FIELD, ENDPOINT_TYPE_FIELD, ATTACHMENT_TYPE_FIELD, DOMAIN_CERTIFICATE_ARN_FIELD,
            ENDPOINT_DOMAIN_FIELD, DEVICE_VALIDATION_DOMAIN_FIELD, SECURITY_GROUP_IDS_FIELD, LOAD_BALANCER_OPTIONS_FIELD,
            NETWORK_INTERFACE_OPTIONS_FIELD, STATUS_FIELD, DESCRIPTION_FIELD, CREATION_TIME_FIELD, LAST_UPDATED_TIME_FIELD,
            DELETION_TIME_FIELD, TAGS_FIELD, SSE_SPECIFICATION_FIELD));

    private static final long serialVersionUID = 1L;

    private final String verifiedAccessInstanceId;

    private final String verifiedAccessGroupId;

    private final String verifiedAccessEndpointId;

    private final String applicationDomain;

    private final String endpointType;

    private final String attachmentType;

    private final String domainCertificateArn;

    private final String endpointDomain;

    private final String deviceValidationDomain;

    private final List<String> securityGroupIds;

    private final VerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions;

    private final VerifiedAccessEndpointEniOptions networkInterfaceOptions;

    private final VerifiedAccessEndpointStatus status;

    private final String description;

    private final String creationTime;

    private final String lastUpdatedTime;

    private final String deletionTime;

    private final List<Tag> tags;

    private final VerifiedAccessSseSpecificationResponse sseSpecification;

    private VerifiedAccessEndpoint(BuilderImpl builder) {
        this.verifiedAccessInstanceId = builder.verifiedAccessInstanceId;
        this.verifiedAccessGroupId = builder.verifiedAccessGroupId;
        this.verifiedAccessEndpointId = builder.verifiedAccessEndpointId;
        this.applicationDomain = builder.applicationDomain;
        this.endpointType = builder.endpointType;
        this.attachmentType = builder.attachmentType;
        this.domainCertificateArn = builder.domainCertificateArn;
        this.endpointDomain = builder.endpointDomain;
        this.deviceValidationDomain = builder.deviceValidationDomain;
        this.securityGroupIds = builder.securityGroupIds;
        this.loadBalancerOptions = builder.loadBalancerOptions;
        this.networkInterfaceOptions = builder.networkInterfaceOptions;
        this.status = builder.status;
        this.description = builder.description;
        this.creationTime = builder.creationTime;
        this.lastUpdatedTime = builder.lastUpdatedTime;
        this.deletionTime = builder.deletionTime;
        this.tags = builder.tags;
        this.sseSpecification = builder.sseSpecification;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services Verified Access instance.
     * </p>
     * 
     * @return The ID of the Amazon Web Services Verified Access instance.
     */
    public final String verifiedAccessInstanceId() {
        return verifiedAccessInstanceId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services Verified Access group.
     * </p>
     * 
     * @return The ID of the Amazon Web Services Verified Access group.
     */
    public final String verifiedAccessGroupId() {
        return verifiedAccessGroupId;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services Verified Access endpoint.
     * </p>
     * 
     * @return The ID of the Amazon Web Services Verified Access endpoint.
     */
    public final String verifiedAccessEndpointId() {
        return verifiedAccessEndpointId;
    }

    /**
     * <p>
     * The DNS name for users to reach your application.
     * </p>
     * 
     * @return The DNS name for users to reach your application.
     */
    public final String applicationDomain() {
        return applicationDomain;
    }

    /**
     * <p>
     * The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to an IP
     * address, load balancer or a network interface depending on the endpoint type specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link VerifiedAccessEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to
     *         an IP address, load balancer or a network interface depending on the endpoint type specified.
     * @see VerifiedAccessEndpointType
     */
    public final VerifiedAccessEndpointType endpointType() {
        return VerifiedAccessEndpointType.fromValue(endpointType);
    }

    /**
     * <p>
     * The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to an IP
     * address, load balancer or a network interface depending on the endpoint type specified.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link VerifiedAccessEndpointType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to
     *         an IP address, load balancer or a network interface depending on the endpoint type specified.
     * @see VerifiedAccessEndpointType
     */
    public final String endpointTypeAsString() {
        return endpointType;
    }

    /**
     * <p>
     * The type of attachment used to provide connectivity between the Amazon Web Services Verified Access endpoint and
     * the application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attachmentType}
     * will return {@link VerifiedAccessEndpointAttachmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #attachmentTypeAsString}.
     * </p>
     * 
     * @return The type of attachment used to provide connectivity between the Amazon Web Services Verified Access
     *         endpoint and the application.
     * @see VerifiedAccessEndpointAttachmentType
     */
    public final VerifiedAccessEndpointAttachmentType attachmentType() {
        return VerifiedAccessEndpointAttachmentType.fromValue(attachmentType);
    }

    /**
     * <p>
     * The type of attachment used to provide connectivity between the Amazon Web Services Verified Access endpoint and
     * the application.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #attachmentType}
     * will return {@link VerifiedAccessEndpointAttachmentType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the
     * service is available from {@link #attachmentTypeAsString}.
     * </p>
     * 
     * @return The type of attachment used to provide connectivity between the Amazon Web Services Verified Access
     *         endpoint and the application.
     * @see VerifiedAccessEndpointAttachmentType
     */
    public final String attachmentTypeAsString() {
        return attachmentType;
    }

    /**
     * <p>
     * The ARN of a public TLS/SSL certificate imported into or created with ACM.
     * </p>
     * 
     * @return The ARN of a public TLS/SSL certificate imported into or created with ACM.
     */
    public final String domainCertificateArn() {
        return domainCertificateArn;
    }

    /**
     * <p>
     * A DNS name that is generated for the endpoint.
     * </p>
     * 
     * @return A DNS name that is generated for the endpoint.
     */
    public final String endpointDomain() {
        return endpointDomain;
    }

    /**
     * <p>
     * Returned if endpoint has a device trust provider attached.
     * </p>
     * 
     * @return Returned if endpoint has a device trust provider attached.
     */
    public final String deviceValidationDomain() {
        return deviceValidationDomain;
    }

    /**
     * For responses, this returns true if the service returned a value for the SecurityGroupIds property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasSecurityGroupIds() {
        return securityGroupIds != null && !(securityGroupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the security groups for the endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSecurityGroupIds} method.
     * </p>
     * 
     * @return The IDs of the security groups for the endpoint.
     */
    public final List<String> securityGroupIds() {
        return securityGroupIds;
    }

    /**
     * <p>
     * The load balancer details if creating the Amazon Web Services Verified Access endpoint as
     * <code>load-balancer</code>type.
     * </p>
     * 
     * @return The load balancer details if creating the Amazon Web Services Verified Access endpoint as
     *         <code>load-balancer</code>type.
     */
    public final VerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions() {
        return loadBalancerOptions;
    }

    /**
     * <p>
     * The options for network-interface type endpoint.
     * </p>
     * 
     * @return The options for network-interface type endpoint.
     */
    public final VerifiedAccessEndpointEniOptions networkInterfaceOptions() {
        return networkInterfaceOptions;
    }

    /**
     * <p>
     * The endpoint status.
     * </p>
     * 
     * @return The endpoint status.
     */
    public final VerifiedAccessEndpointStatus status() {
        return status;
    }

    /**
     * <p>
     * A description for the Amazon Web Services Verified Access endpoint.
     * </p>
     * 
     * @return A description for the Amazon Web Services Verified Access endpoint.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The creation time.
     * </p>
     * 
     * @return The creation time.
     */
    public final String creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * The last updated time.
     * </p>
     * 
     * @return The last updated time.
     */
    public final String lastUpdatedTime() {
        return lastUpdatedTime;
    }

    /**
     * <p>
     * The deletion time.
     * </p>
     * 
     * @return The deletion time.
     */
    public final String deletionTime() {
        return deletionTime;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Describes the options in use for server side encryption.
     * </p>
     * 
     * @return Describes the options in use for server side encryption.
     */
    public final VerifiedAccessSseSpecificationResponse sseSpecification() {
        return sseSpecification;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(verifiedAccessInstanceId());
        hashCode = 31 * hashCode + Objects.hashCode(verifiedAccessGroupId());
        hashCode = 31 * hashCode + Objects.hashCode(verifiedAccessEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(applicationDomain());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(attachmentTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(domainCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(endpointDomain());
        hashCode = 31 * hashCode + Objects.hashCode(deviceValidationDomain());
        hashCode = 31 * hashCode + Objects.hashCode(hasSecurityGroupIds() ? securityGroupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(loadBalancerOptions());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceOptions());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastUpdatedTime());
        hashCode = 31 * hashCode + Objects.hashCode(deletionTime());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(sseSpecification());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VerifiedAccessEndpoint)) {
            return false;
        }
        VerifiedAccessEndpoint other = (VerifiedAccessEndpoint) obj;
        return Objects.equals(verifiedAccessInstanceId(), other.verifiedAccessInstanceId())
                && Objects.equals(verifiedAccessGroupId(), other.verifiedAccessGroupId())
                && Objects.equals(verifiedAccessEndpointId(), other.verifiedAccessEndpointId())
                && Objects.equals(applicationDomain(), other.applicationDomain())
                && Objects.equals(endpointTypeAsString(), other.endpointTypeAsString())
                && Objects.equals(attachmentTypeAsString(), other.attachmentTypeAsString())
                && Objects.equals(domainCertificateArn(), other.domainCertificateArn())
                && Objects.equals(endpointDomain(), other.endpointDomain())
                && Objects.equals(deviceValidationDomain(), other.deviceValidationDomain())
                && hasSecurityGroupIds() == other.hasSecurityGroupIds()
                && Objects.equals(securityGroupIds(), other.securityGroupIds())
                && Objects.equals(loadBalancerOptions(), other.loadBalancerOptions())
                && Objects.equals(networkInterfaceOptions(), other.networkInterfaceOptions())
                && Objects.equals(status(), other.status()) && Objects.equals(description(), other.description())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(lastUpdatedTime(), other.lastUpdatedTime())
                && Objects.equals(deletionTime(), other.deletionTime()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(sseSpecification(), other.sseSpecification());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VerifiedAccessEndpoint").add("VerifiedAccessInstanceId", verifiedAccessInstanceId())
                .add("VerifiedAccessGroupId", verifiedAccessGroupId())
                .add("VerifiedAccessEndpointId", verifiedAccessEndpointId()).add("ApplicationDomain", applicationDomain())
                .add("EndpointType", endpointTypeAsString()).add("AttachmentType", attachmentTypeAsString())
                .add("DomainCertificateArn", domainCertificateArn()).add("EndpointDomain", endpointDomain())
                .add("DeviceValidationDomain", deviceValidationDomain())
                .add("SecurityGroupIds", hasSecurityGroupIds() ? securityGroupIds() : null)
                .add("LoadBalancerOptions", loadBalancerOptions()).add("NetworkInterfaceOptions", networkInterfaceOptions())
                .add("Status", status()).add("Description", description()).add("CreationTime", creationTime())
                .add("LastUpdatedTime", lastUpdatedTime()).add("DeletionTime", deletionTime())
                .add("Tags", hasTags() ? tags() : null).add("SseSpecification", sseSpecification()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VerifiedAccessInstanceId":
            return Optional.ofNullable(clazz.cast(verifiedAccessInstanceId()));
        case "VerifiedAccessGroupId":
            return Optional.ofNullable(clazz.cast(verifiedAccessGroupId()));
        case "VerifiedAccessEndpointId":
            return Optional.ofNullable(clazz.cast(verifiedAccessEndpointId()));
        case "ApplicationDomain":
            return Optional.ofNullable(clazz.cast(applicationDomain()));
        case "EndpointType":
            return Optional.ofNullable(clazz.cast(endpointTypeAsString()));
        case "AttachmentType":
            return Optional.ofNullable(clazz.cast(attachmentTypeAsString()));
        case "DomainCertificateArn":
            return Optional.ofNullable(clazz.cast(domainCertificateArn()));
        case "EndpointDomain":
            return Optional.ofNullable(clazz.cast(endpointDomain()));
        case "DeviceValidationDomain":
            return Optional.ofNullable(clazz.cast(deviceValidationDomain()));
        case "SecurityGroupIds":
            return Optional.ofNullable(clazz.cast(securityGroupIds()));
        case "LoadBalancerOptions":
            return Optional.ofNullable(clazz.cast(loadBalancerOptions()));
        case "NetworkInterfaceOptions":
            return Optional.ofNullable(clazz.cast(networkInterfaceOptions()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "LastUpdatedTime":
            return Optional.ofNullable(clazz.cast(lastUpdatedTime()));
        case "DeletionTime":
            return Optional.ofNullable(clazz.cast(deletionTime()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "SseSpecification":
            return Optional.ofNullable(clazz.cast(sseSpecification()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VerifiedAccessEndpoint, T> g) {
        return obj -> g.apply((VerifiedAccessEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VerifiedAccessEndpoint> {
        /**
         * <p>
         * The ID of the Amazon Web Services Verified Access instance.
         * </p>
         * 
         * @param verifiedAccessInstanceId
         *        The ID of the Amazon Web Services Verified Access instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verifiedAccessInstanceId(String verifiedAccessInstanceId);

        /**
         * <p>
         * The ID of the Amazon Web Services Verified Access group.
         * </p>
         * 
         * @param verifiedAccessGroupId
         *        The ID of the Amazon Web Services Verified Access group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verifiedAccessGroupId(String verifiedAccessGroupId);

        /**
         * <p>
         * The ID of the Amazon Web Services Verified Access endpoint.
         * </p>
         * 
         * @param verifiedAccessEndpointId
         *        The ID of the Amazon Web Services Verified Access endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder verifiedAccessEndpointId(String verifiedAccessEndpointId);

        /**
         * <p>
         * The DNS name for users to reach your application.
         * </p>
         * 
         * @param applicationDomain
         *        The DNS name for users to reach your application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder applicationDomain(String applicationDomain);

        /**
         * <p>
         * The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to an IP
         * address, load balancer or a network interface depending on the endpoint type specified.
         * </p>
         * 
         * @param endpointType
         *        The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent
         *        to an IP address, load balancer or a network interface depending on the endpoint type specified.
         * @see VerifiedAccessEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointType
         */
        Builder endpointType(String endpointType);

        /**
         * <p>
         * The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent to an IP
         * address, load balancer or a network interface depending on the endpoint type specified.
         * </p>
         * 
         * @param endpointType
         *        The type of Amazon Web Services Verified Access endpoint. Incoming application requests will be sent
         *        to an IP address, load balancer or a network interface depending on the endpoint type specified.
         * @see VerifiedAccessEndpointType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointType
         */
        Builder endpointType(VerifiedAccessEndpointType endpointType);

        /**
         * <p>
         * The type of attachment used to provide connectivity between the Amazon Web Services Verified Access endpoint
         * and the application.
         * </p>
         * 
         * @param attachmentType
         *        The type of attachment used to provide connectivity between the Amazon Web Services Verified Access
         *        endpoint and the application.
         * @see VerifiedAccessEndpointAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointAttachmentType
         */
        Builder attachmentType(String attachmentType);

        /**
         * <p>
         * The type of attachment used to provide connectivity between the Amazon Web Services Verified Access endpoint
         * and the application.
         * </p>
         * 
         * @param attachmentType
         *        The type of attachment used to provide connectivity between the Amazon Web Services Verified Access
         *        endpoint and the application.
         * @see VerifiedAccessEndpointAttachmentType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see VerifiedAccessEndpointAttachmentType
         */
        Builder attachmentType(VerifiedAccessEndpointAttachmentType attachmentType);

        /**
         * <p>
         * The ARN of a public TLS/SSL certificate imported into or created with ACM.
         * </p>
         * 
         * @param domainCertificateArn
         *        The ARN of a public TLS/SSL certificate imported into or created with ACM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder domainCertificateArn(String domainCertificateArn);

        /**
         * <p>
         * A DNS name that is generated for the endpoint.
         * </p>
         * 
         * @param endpointDomain
         *        A DNS name that is generated for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointDomain(String endpointDomain);

        /**
         * <p>
         * Returned if endpoint has a device trust provider attached.
         * </p>
         * 
         * @param deviceValidationDomain
         *        Returned if endpoint has a device trust provider attached.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deviceValidationDomain(String deviceValidationDomain);

        /**
         * <p>
         * The IDs of the security groups for the endpoint.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(Collection<String> securityGroupIds);

        /**
         * <p>
         * The IDs of the security groups for the endpoint.
         * </p>
         * 
         * @param securityGroupIds
         *        The IDs of the security groups for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder securityGroupIds(String... securityGroupIds);

        /**
         * <p>
         * The load balancer details if creating the Amazon Web Services Verified Access endpoint as
         * <code>load-balancer</code>type.
         * </p>
         * 
         * @param loadBalancerOptions
         *        The load balancer details if creating the Amazon Web Services Verified Access endpoint as
         *        <code>load-balancer</code>type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder loadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions);

        /**
         * <p>
         * The load balancer details if creating the Amazon Web Services Verified Access endpoint as
         * <code>load-balancer</code>type.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link VerifiedAccessEndpointLoadBalancerOptions.Builder} avoiding the need to create one manually via
         * {@link VerifiedAccessEndpointLoadBalancerOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VerifiedAccessEndpointLoadBalancerOptions.Builder#build()} is
         * called immediately and its result is passed to
         * {@link #loadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions)}.
         * 
         * @param loadBalancerOptions
         *        a consumer that will call methods on {@link VerifiedAccessEndpointLoadBalancerOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #loadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions)
         */
        default Builder loadBalancerOptions(Consumer<VerifiedAccessEndpointLoadBalancerOptions.Builder> loadBalancerOptions) {
            return loadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions.builder().applyMutation(loadBalancerOptions)
                    .build());
        }

        /**
         * <p>
         * The options for network-interface type endpoint.
         * </p>
         * 
         * @param networkInterfaceOptions
         *        The options for network-interface type endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceOptions(VerifiedAccessEndpointEniOptions networkInterfaceOptions);

        /**
         * <p>
         * The options for network-interface type endpoint.
         * </p>
         * This is a convenience method that creates an instance of the {@link VerifiedAccessEndpointEniOptions.Builder}
         * avoiding the need to create one manually via {@link VerifiedAccessEndpointEniOptions#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VerifiedAccessEndpointEniOptions.Builder#build()} is called
         * immediately and its result is passed to {@link #networkInterfaceOptions(VerifiedAccessEndpointEniOptions)}.
         * 
         * @param networkInterfaceOptions
         *        a consumer that will call methods on {@link VerifiedAccessEndpointEniOptions.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaceOptions(VerifiedAccessEndpointEniOptions)
         */
        default Builder networkInterfaceOptions(Consumer<VerifiedAccessEndpointEniOptions.Builder> networkInterfaceOptions) {
            return networkInterfaceOptions(VerifiedAccessEndpointEniOptions.builder().applyMutation(networkInterfaceOptions)
                    .build());
        }

        /**
         * <p>
         * The endpoint status.
         * </p>
         * 
         * @param status
         *        The endpoint status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(VerifiedAccessEndpointStatus status);

        /**
         * <p>
         * The endpoint status.
         * </p>
         * This is a convenience method that creates an instance of the {@link VerifiedAccessEndpointStatus.Builder}
         * avoiding the need to create one manually via {@link VerifiedAccessEndpointStatus#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VerifiedAccessEndpointStatus.Builder#build()} is called
         * immediately and its result is passed to {@link #status(VerifiedAccessEndpointStatus)}.
         * 
         * @param status
         *        a consumer that will call methods on {@link VerifiedAccessEndpointStatus.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #status(VerifiedAccessEndpointStatus)
         */
        default Builder status(Consumer<VerifiedAccessEndpointStatus.Builder> status) {
            return status(VerifiedAccessEndpointStatus.builder().applyMutation(status).build());
        }

        /**
         * <p>
         * A description for the Amazon Web Services Verified Access endpoint.
         * </p>
         * 
         * @param description
         *        A description for the Amazon Web Services Verified Access endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The creation time.
         * </p>
         * 
         * @param creationTime
         *        The creation time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(String creationTime);

        /**
         * <p>
         * The last updated time.
         * </p>
         * 
         * @param lastUpdatedTime
         *        The last updated time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastUpdatedTime(String lastUpdatedTime);

        /**
         * <p>
         * The deletion time.
         * </p>
         * 
         * @param deletionTime
         *        The deletion time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deletionTime(String deletionTime);

        /**
         * <p>
         * The tags.
         * </p>
         * 
         * @param tags
         *        The tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags.
         * </p>
         * 
         * @param tags
         *        The tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Describes the options in use for server side encryption.
         * </p>
         * 
         * @param sseSpecification
         *        Describes the options in use for server side encryption.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sseSpecification(VerifiedAccessSseSpecificationResponse sseSpecification);

        /**
         * <p>
         * Describes the options in use for server side encryption.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link VerifiedAccessSseSpecificationResponse.Builder} avoiding the need to create one manually via
         * {@link VerifiedAccessSseSpecificationResponse#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VerifiedAccessSseSpecificationResponse.Builder#build()} is called
         * immediately and its result is passed to {@link #sseSpecification(VerifiedAccessSseSpecificationResponse)}.
         * 
         * @param sseSpecification
         *        a consumer that will call methods on {@link VerifiedAccessSseSpecificationResponse.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sseSpecification(VerifiedAccessSseSpecificationResponse)
         */
        default Builder sseSpecification(Consumer<VerifiedAccessSseSpecificationResponse.Builder> sseSpecification) {
            return sseSpecification(VerifiedAccessSseSpecificationResponse.builder().applyMutation(sseSpecification).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String verifiedAccessInstanceId;

        private String verifiedAccessGroupId;

        private String verifiedAccessEndpointId;

        private String applicationDomain;

        private String endpointType;

        private String attachmentType;

        private String domainCertificateArn;

        private String endpointDomain;

        private String deviceValidationDomain;

        private List<String> securityGroupIds = DefaultSdkAutoConstructList.getInstance();

        private VerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions;

        private VerifiedAccessEndpointEniOptions networkInterfaceOptions;

        private VerifiedAccessEndpointStatus status;

        private String description;

        private String creationTime;

        private String lastUpdatedTime;

        private String deletionTime;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private VerifiedAccessSseSpecificationResponse sseSpecification;

        private BuilderImpl() {
        }

        private BuilderImpl(VerifiedAccessEndpoint model) {
            verifiedAccessInstanceId(model.verifiedAccessInstanceId);
            verifiedAccessGroupId(model.verifiedAccessGroupId);
            verifiedAccessEndpointId(model.verifiedAccessEndpointId);
            applicationDomain(model.applicationDomain);
            endpointType(model.endpointType);
            attachmentType(model.attachmentType);
            domainCertificateArn(model.domainCertificateArn);
            endpointDomain(model.endpointDomain);
            deviceValidationDomain(model.deviceValidationDomain);
            securityGroupIds(model.securityGroupIds);
            loadBalancerOptions(model.loadBalancerOptions);
            networkInterfaceOptions(model.networkInterfaceOptions);
            status(model.status);
            description(model.description);
            creationTime(model.creationTime);
            lastUpdatedTime(model.lastUpdatedTime);
            deletionTime(model.deletionTime);
            tags(model.tags);
            sseSpecification(model.sseSpecification);
        }

        public final String getVerifiedAccessInstanceId() {
            return verifiedAccessInstanceId;
        }

        public final void setVerifiedAccessInstanceId(String verifiedAccessInstanceId) {
            this.verifiedAccessInstanceId = verifiedAccessInstanceId;
        }

        @Override
        public final Builder verifiedAccessInstanceId(String verifiedAccessInstanceId) {
            this.verifiedAccessInstanceId = verifiedAccessInstanceId;
            return this;
        }

        public final String getVerifiedAccessGroupId() {
            return verifiedAccessGroupId;
        }

        public final void setVerifiedAccessGroupId(String verifiedAccessGroupId) {
            this.verifiedAccessGroupId = verifiedAccessGroupId;
        }

        @Override
        public final Builder verifiedAccessGroupId(String verifiedAccessGroupId) {
            this.verifiedAccessGroupId = verifiedAccessGroupId;
            return this;
        }

        public final String getVerifiedAccessEndpointId() {
            return verifiedAccessEndpointId;
        }

        public final void setVerifiedAccessEndpointId(String verifiedAccessEndpointId) {
            this.verifiedAccessEndpointId = verifiedAccessEndpointId;
        }

        @Override
        public final Builder verifiedAccessEndpointId(String verifiedAccessEndpointId) {
            this.verifiedAccessEndpointId = verifiedAccessEndpointId;
            return this;
        }

        public final String getApplicationDomain() {
            return applicationDomain;
        }

        public final void setApplicationDomain(String applicationDomain) {
            this.applicationDomain = applicationDomain;
        }

        @Override
        public final Builder applicationDomain(String applicationDomain) {
            this.applicationDomain = applicationDomain;
            return this;
        }

        public final String getEndpointType() {
            return endpointType;
        }

        public final void setEndpointType(String endpointType) {
            this.endpointType = endpointType;
        }

        @Override
        public final Builder endpointType(String endpointType) {
            this.endpointType = endpointType;
            return this;
        }

        @Override
        public final Builder endpointType(VerifiedAccessEndpointType endpointType) {
            this.endpointType(endpointType == null ? null : endpointType.toString());
            return this;
        }

        public final String getAttachmentType() {
            return attachmentType;
        }

        public final void setAttachmentType(String attachmentType) {
            this.attachmentType = attachmentType;
        }

        @Override
        public final Builder attachmentType(String attachmentType) {
            this.attachmentType = attachmentType;
            return this;
        }

        @Override
        public final Builder attachmentType(VerifiedAccessEndpointAttachmentType attachmentType) {
            this.attachmentType(attachmentType == null ? null : attachmentType.toString());
            return this;
        }

        public final String getDomainCertificateArn() {
            return domainCertificateArn;
        }

        public final void setDomainCertificateArn(String domainCertificateArn) {
            this.domainCertificateArn = domainCertificateArn;
        }

        @Override
        public final Builder domainCertificateArn(String domainCertificateArn) {
            this.domainCertificateArn = domainCertificateArn;
            return this;
        }

        public final String getEndpointDomain() {
            return endpointDomain;
        }

        public final void setEndpointDomain(String endpointDomain) {
            this.endpointDomain = endpointDomain;
        }

        @Override
        public final Builder endpointDomain(String endpointDomain) {
            this.endpointDomain = endpointDomain;
            return this;
        }

        public final String getDeviceValidationDomain() {
            return deviceValidationDomain;
        }

        public final void setDeviceValidationDomain(String deviceValidationDomain) {
            this.deviceValidationDomain = deviceValidationDomain;
        }

        @Override
        public final Builder deviceValidationDomain(String deviceValidationDomain) {
            this.deviceValidationDomain = deviceValidationDomain;
            return this;
        }

        public final Collection<String> getSecurityGroupIds() {
            if (securityGroupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return securityGroupIds;
        }

        public final void setSecurityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
        }

        @Override
        public final Builder securityGroupIds(Collection<String> securityGroupIds) {
            this.securityGroupIds = SecurityGroupIdListCopier.copy(securityGroupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder securityGroupIds(String... securityGroupIds) {
            securityGroupIds(Arrays.asList(securityGroupIds));
            return this;
        }

        public final VerifiedAccessEndpointLoadBalancerOptions.Builder getLoadBalancerOptions() {
            return loadBalancerOptions != null ? loadBalancerOptions.toBuilder() : null;
        }

        public final void setLoadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions.BuilderImpl loadBalancerOptions) {
            this.loadBalancerOptions = loadBalancerOptions != null ? loadBalancerOptions.build() : null;
        }

        @Override
        public final Builder loadBalancerOptions(VerifiedAccessEndpointLoadBalancerOptions loadBalancerOptions) {
            this.loadBalancerOptions = loadBalancerOptions;
            return this;
        }

        public final VerifiedAccessEndpointEniOptions.Builder getNetworkInterfaceOptions() {
            return networkInterfaceOptions != null ? networkInterfaceOptions.toBuilder() : null;
        }

        public final void setNetworkInterfaceOptions(VerifiedAccessEndpointEniOptions.BuilderImpl networkInterfaceOptions) {
            this.networkInterfaceOptions = networkInterfaceOptions != null ? networkInterfaceOptions.build() : null;
        }

        @Override
        public final Builder networkInterfaceOptions(VerifiedAccessEndpointEniOptions networkInterfaceOptions) {
            this.networkInterfaceOptions = networkInterfaceOptions;
            return this;
        }

        public final VerifiedAccessEndpointStatus.Builder getStatus() {
            return status != null ? status.toBuilder() : null;
        }

        public final void setStatus(VerifiedAccessEndpointStatus.BuilderImpl status) {
            this.status = status != null ? status.build() : null;
        }

        @Override
        public final Builder status(VerifiedAccessEndpointStatus status) {
            this.status = status;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getCreationTime() {
            return creationTime;
        }

        public final void setCreationTime(String creationTime) {
            this.creationTime = creationTime;
        }

        @Override
        public final Builder creationTime(String creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final String getLastUpdatedTime() {
            return lastUpdatedTime;
        }

        public final void setLastUpdatedTime(String lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
        }

        @Override
        public final Builder lastUpdatedTime(String lastUpdatedTime) {
            this.lastUpdatedTime = lastUpdatedTime;
            return this;
        }

        public final String getDeletionTime() {
            return deletionTime;
        }

        public final void setDeletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
        }

        @Override
        public final Builder deletionTime(String deletionTime) {
            this.deletionTime = deletionTime;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final VerifiedAccessSseSpecificationResponse.Builder getSseSpecification() {
            return sseSpecification != null ? sseSpecification.toBuilder() : null;
        }

        public final void setSseSpecification(VerifiedAccessSseSpecificationResponse.BuilderImpl sseSpecification) {
            this.sseSpecification = sseSpecification != null ? sseSpecification.build() : null;
        }

        @Override
        public final Builder sseSpecification(VerifiedAccessSseSpecificationResponse sseSpecification) {
            this.sseSpecification = sseSpecification;
            return this;
        }

        @Override
        public VerifiedAccessEndpoint build() {
            return new VerifiedAccessEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
