/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about the instance topology.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceTopology implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceTopology.Builder, InstanceTopology> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(InstanceTopology::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("instanceId").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(InstanceTopology::instanceType))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("instanceType").build()).build();

    private static final SdkField<String> GROUP_NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("GroupName")
            .getter(getter(InstanceTopology::groupName))
            .setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName")
                    .unmarshallLocationName("groupName").build()).build();

    private static final SdkField<List<String>> NETWORK_NODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("NetworkNodes")
            .getter(getter(InstanceTopology::networkNodes))
            .setter(setter(Builder::networkNodes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkNodeSet")
                    .unmarshallLocationName("networkNodeSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(InstanceTopology::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("availabilityZone").build()).build();

    private static final SdkField<String> ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ZoneId")
            .getter(getter(InstanceTopology::zoneId))
            .setter(setter(Builder::zoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ZoneId")
                    .unmarshallLocationName("zoneId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            INSTANCE_TYPE_FIELD, GROUP_NAME_FIELD, NETWORK_NODES_FIELD, AVAILABILITY_ZONE_FIELD, ZONE_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final String instanceType;

    private final String groupName;

    private final List<String> networkNodes;

    private final String availabilityZone;

    private final String zoneId;

    private InstanceTopology(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.instanceType = builder.instanceType;
        this.groupName = builder.groupName;
        this.networkNodes = builder.networkNodes;
        this.availabilityZone = builder.availabilityZone;
        this.zoneId = builder.zoneId;
    }

    /**
     * <p>
     * The instance ID.
     * </p>
     * 
     * @return The instance ID.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * The instance type.
     * </p>
     * 
     * @return The instance type.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The name of the placement group that the instance is in.
     * </p>
     * 
     * @return The name of the placement group that the instance is in.
     */
    public final String groupName() {
        return groupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the NetworkNodes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasNetworkNodes() {
        return networkNodes != null && !(networkNodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The network nodes. The nodes are hashed based on your account. Instances from different accounts running under
     * the same server will return a different hashed list of strings.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasNetworkNodes} method.
     * </p>
     * 
     * @return The network nodes. The nodes are hashed based on your account. Instances from different accounts running
     *         under the same server will return a different hashed list of strings.
     */
    public final List<String> networkNodes() {
        return networkNodes;
    }

    /**
     * <p>
     * The name of the Availability Zone or Local Zone that the instance is in.
     * </p>
     * 
     * @return The name of the Availability Zone or Local Zone that the instance is in.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * The ID of the Availability Zone or Local Zone that the instance is in.
     * </p>
     * 
     * @return The ID of the Availability Zone or Local Zone that the instance is in.
     */
    public final String zoneId() {
        return zoneId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasNetworkNodes() ? networkNodes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(zoneId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceTopology)) {
            return false;
        }
        InstanceTopology other = (InstanceTopology) obj;
        return Objects.equals(instanceId(), other.instanceId()) && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(groupName(), other.groupName()) && hasNetworkNodes() == other.hasNetworkNodes()
                && Objects.equals(networkNodes(), other.networkNodes())
                && Objects.equals(availabilityZone(), other.availabilityZone()) && Objects.equals(zoneId(), other.zoneId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceTopology").add("InstanceId", instanceId()).add("InstanceType", instanceType())
                .add("GroupName", groupName()).add("NetworkNodes", hasNetworkNodes() ? networkNodes() : null)
                .add("AvailabilityZone", availabilityZone()).add("ZoneId", zoneId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "NetworkNodes":
            return Optional.ofNullable(clazz.cast(networkNodes()));
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "ZoneId":
            return Optional.ofNullable(clazz.cast(zoneId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceTopology, T> g) {
        return obj -> g.apply((InstanceTopology) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceTopology> {
        /**
         * <p>
         * The instance ID.
         * </p>
         * 
         * @param instanceId
         *        The instance ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * The instance type.
         * </p>
         * 
         * @param instanceType
         *        The instance type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The name of the placement group that the instance is in.
         * </p>
         * 
         * @param groupName
         *        The name of the placement group that the instance is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * <p>
         * The network nodes. The nodes are hashed based on your account. Instances from different accounts running
         * under the same server will return a different hashed list of strings.
         * </p>
         * 
         * @param networkNodes
         *        The network nodes. The nodes are hashed based on your account. Instances from different accounts
         *        running under the same server will return a different hashed list of strings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkNodes(Collection<String> networkNodes);

        /**
         * <p>
         * The network nodes. The nodes are hashed based on your account. Instances from different accounts running
         * under the same server will return a different hashed list of strings.
         * </p>
         * 
         * @param networkNodes
         *        The network nodes. The nodes are hashed based on your account. Instances from different accounts
         *        running under the same server will return a different hashed list of strings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkNodes(String... networkNodes);

        /**
         * <p>
         * The name of the Availability Zone or Local Zone that the instance is in.
         * </p>
         * 
         * @param availabilityZone
         *        The name of the Availability Zone or Local Zone that the instance is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * The ID of the Availability Zone or Local Zone that the instance is in.
         * </p>
         * 
         * @param zoneId
         *        The ID of the Availability Zone or Local Zone that the instance is in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder zoneId(String zoneId);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private String instanceType;

        private String groupName;

        private List<String> networkNodes = DefaultSdkAutoConstructList.getInstance();

        private String availabilityZone;

        private String zoneId;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceTopology model) {
            instanceId(model.instanceId);
            instanceType(model.instanceType);
            groupName(model.groupName);
            networkNodes(model.networkNodes);
            availabilityZone(model.availabilityZone);
            zoneId(model.zoneId);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getGroupName() {
            return groupName;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final Collection<String> getNetworkNodes() {
            if (networkNodes instanceof SdkAutoConstructList) {
                return null;
            }
            return networkNodes;
        }

        public final void setNetworkNodes(Collection<String> networkNodes) {
            this.networkNodes = NetworkNodesListCopier.copy(networkNodes);
        }

        @Override
        public final Builder networkNodes(Collection<String> networkNodes) {
            this.networkNodes = NetworkNodesListCopier.copy(networkNodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder networkNodes(String... networkNodes) {
            networkNodes(Arrays.asList(networkNodes));
            return this;
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final String getZoneId() {
            return zoneId;
        }

        public final void setZoneId(String zoneId) {
            this.zoneId = zoneId;
        }

        @Override
        public final Builder zoneId(String zoneId) {
            this.zoneId = zoneId;
            return this;
        }

        @Override
        public InstanceTopology build() {
            return new InstanceTopology(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
