/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An IPAM discovered resource CIDR. A discovered resource is a resource CIDR monitored under a resource discovery. The
 * following resources can be discovered: VPCs, Public IPv4 pools, VPC subnets, and Elastic IP addresses. The discovered
 * resource CIDR is the IP address range in CIDR notation that is associated with the resource.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamDiscoveredResourceCidr implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamDiscoveredResourceCidr.Builder, IpamDiscoveredResourceCidr> {
    private static final SdkField<String> IPAM_RESOURCE_DISCOVERY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamResourceDiscoveryId")
            .getter(getter(IpamDiscoveredResourceCidr::ipamResourceDiscoveryId))
            .setter(setter(Builder::ipamResourceDiscoveryId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamResourceDiscoveryId")
                    .unmarshallLocationName("ipamResourceDiscoveryId").build()).build();

    private static final SdkField<String> RESOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceRegion")
            .getter(getter(IpamDiscoveredResourceCidr::resourceRegion))
            .setter(setter(Builder::resourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceRegion")
                    .unmarshallLocationName("resourceRegion").build()).build();

    private static final SdkField<String> RESOURCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceId")
            .getter(getter(IpamDiscoveredResourceCidr::resourceId))
            .setter(setter(Builder::resourceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceId")
                    .unmarshallLocationName("resourceId").build()).build();

    private static final SdkField<String> RESOURCE_OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceOwnerId")
            .getter(getter(IpamDiscoveredResourceCidr::resourceOwnerId))
            .setter(setter(Builder::resourceOwnerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceOwnerId")
                    .unmarshallLocationName("resourceOwnerId").build()).build();

    private static final SdkField<String> RESOURCE_CIDR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceCidr")
            .getter(getter(IpamDiscoveredResourceCidr::resourceCidr))
            .setter(setter(Builder::resourceCidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceCidr")
                    .unmarshallLocationName("resourceCidr").build()).build();

    private static final SdkField<String> RESOURCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceType")
            .getter(getter(IpamDiscoveredResourceCidr::resourceTypeAsString))
            .setter(setter(Builder::resourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceType")
                    .unmarshallLocationName("resourceType").build()).build();

    private static final SdkField<List<IpamResourceTag>> RESOURCE_TAGS_FIELD = SdkField
            .<List<IpamResourceTag>> builder(MarshallingType.LIST)
            .memberName("ResourceTags")
            .getter(getter(IpamDiscoveredResourceCidr::resourceTags))
            .setter(setter(Builder::resourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceTagSet")
                    .unmarshallLocationName("resourceTagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Double> IP_USAGE_FIELD = SdkField
            .<Double> builder(MarshallingType.DOUBLE)
            .memberName("IpUsage")
            .getter(getter(IpamDiscoveredResourceCidr::ipUsage))
            .setter(setter(Builder::ipUsage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpUsage")
                    .unmarshallLocationName("ipUsage").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(IpamDiscoveredResourceCidr::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> NETWORK_INTERFACE_ATTACHMENT_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkInterfaceAttachmentStatus")
            .getter(getter(IpamDiscoveredResourceCidr::networkInterfaceAttachmentStatusAsString))
            .setter(setter(Builder::networkInterfaceAttachmentStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceAttachmentStatus")
                    .unmarshallLocationName("networkInterfaceAttachmentStatus").build()).build();

    private static final SdkField<Instant> SAMPLE_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("SampleTime")
            .getter(getter(IpamDiscoveredResourceCidr::sampleTime))
            .setter(setter(Builder::sampleTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SampleTime")
                    .unmarshallLocationName("sampleTime").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(IpamDiscoveredResourceCidr::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("availabilityZoneId").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            IPAM_RESOURCE_DISCOVERY_ID_FIELD, RESOURCE_REGION_FIELD, RESOURCE_ID_FIELD, RESOURCE_OWNER_ID_FIELD,
            RESOURCE_CIDR_FIELD, RESOURCE_TYPE_FIELD, RESOURCE_TAGS_FIELD, IP_USAGE_FIELD, VPC_ID_FIELD,
            NETWORK_INTERFACE_ATTACHMENT_STATUS_FIELD, SAMPLE_TIME_FIELD, AVAILABILITY_ZONE_ID_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ipamResourceDiscoveryId;

    private final String resourceRegion;

    private final String resourceId;

    private final String resourceOwnerId;

    private final String resourceCidr;

    private final String resourceType;

    private final List<IpamResourceTag> resourceTags;

    private final Double ipUsage;

    private final String vpcId;

    private final String networkInterfaceAttachmentStatus;

    private final Instant sampleTime;

    private final String availabilityZoneId;

    private IpamDiscoveredResourceCidr(BuilderImpl builder) {
        this.ipamResourceDiscoveryId = builder.ipamResourceDiscoveryId;
        this.resourceRegion = builder.resourceRegion;
        this.resourceId = builder.resourceId;
        this.resourceOwnerId = builder.resourceOwnerId;
        this.resourceCidr = builder.resourceCidr;
        this.resourceType = builder.resourceType;
        this.resourceTags = builder.resourceTags;
        this.ipUsage = builder.ipUsage;
        this.vpcId = builder.vpcId;
        this.networkInterfaceAttachmentStatus = builder.networkInterfaceAttachmentStatus;
        this.sampleTime = builder.sampleTime;
        this.availabilityZoneId = builder.availabilityZoneId;
    }

    /**
     * <p>
     * The resource discovery ID.
     * </p>
     * 
     * @return The resource discovery ID.
     */
    public final String ipamResourceDiscoveryId() {
        return ipamResourceDiscoveryId;
    }

    /**
     * <p>
     * The resource Region.
     * </p>
     * 
     * @return The resource Region.
     */
    public final String resourceRegion() {
        return resourceRegion;
    }

    /**
     * <p>
     * The resource ID.
     * </p>
     * 
     * @return The resource ID.
     */
    public final String resourceId() {
        return resourceId;
    }

    /**
     * <p>
     * The resource owner ID.
     * </p>
     * 
     * @return The resource owner ID.
     */
    public final String resourceOwnerId() {
        return resourceOwnerId;
    }

    /**
     * <p>
     * The resource CIDR.
     * </p>
     * 
     * @return The resource CIDR.
     */
    public final String resourceCidr() {
        return resourceCidr;
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type.
     * @see IpamResourceType
     */
    public final IpamResourceType resourceType() {
        return IpamResourceType.fromValue(resourceType);
    }

    /**
     * <p>
     * The resource type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #resourceType} will
     * return {@link IpamResourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #resourceTypeAsString}.
     * </p>
     * 
     * @return The resource type.
     * @see IpamResourceType
     */
    public final String resourceTypeAsString() {
        return resourceType;
    }

    /**
     * For responses, this returns true if the service returned a value for the ResourceTags property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasResourceTags() {
        return resourceTags != null && !(resourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The resource tags.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasResourceTags} method.
     * </p>
     * 
     * @return The resource tags.
     */
    public final List<IpamResourceTag> resourceTags() {
        return resourceTags;
    }

    /**
     * <p>
     * The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal by 100.
     * Note the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by subnet
     * CIDRs.
     * </p>
     * </li>
     * <li>
     * <p>
     * For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage of IPv4
     * address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the percentage of
     * IPv6 address space in use is not represented. The percentage of IPv6 address space in use cannot currently be
     * calculated.
     * </p>
     * </li>
     * <li>
     * <p>
     * For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's been
     * allocated to Elastic IP addresses (EIPs).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal
     *         by 100. Note the following:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by
     *         subnet CIDRs.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage
     *         of IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the
     *         percentage of IPv6 address space in use is not represented. The percentage of IPv6 address space in use
     *         cannot currently be calculated.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's
     *         been allocated to Elastic IP addresses (EIPs).
     *         </p>
     *         </li>
     */
    public final Double ipUsage() {
        return ipUsage;
    }

    /**
     * <p>
     * The VPC ID.
     * </p>
     * 
     * @return The VPC ID.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * For elastic IP addresses, this is the status of an attached network interface.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #networkInterfaceAttachmentStatus} will return
     * {@link IpamNetworkInterfaceAttachmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #networkInterfaceAttachmentStatusAsString}.
     * </p>
     * 
     * @return For elastic IP addresses, this is the status of an attached network interface.
     * @see IpamNetworkInterfaceAttachmentStatus
     */
    public final IpamNetworkInterfaceAttachmentStatus networkInterfaceAttachmentStatus() {
        return IpamNetworkInterfaceAttachmentStatus.fromValue(networkInterfaceAttachmentStatus);
    }

    /**
     * <p>
     * For elastic IP addresses, this is the status of an attached network interface.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #networkInterfaceAttachmentStatus} will return
     * {@link IpamNetworkInterfaceAttachmentStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #networkInterfaceAttachmentStatusAsString}.
     * </p>
     * 
     * @return For elastic IP addresses, this is the status of an attached network interface.
     * @see IpamNetworkInterfaceAttachmentStatus
     */
    public final String networkInterfaceAttachmentStatusAsString() {
        return networkInterfaceAttachmentStatus;
    }

    /**
     * <p>
     * The last successful resource discovery time.
     * </p>
     * 
     * @return The last successful resource discovery time.
     */
    public final Instant sampleTime() {
        return sampleTime;
    }

    /**
     * <p>
     * The Availability Zone ID.
     * </p>
     * 
     * @return The Availability Zone ID.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ipamResourceDiscoveryId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(resourceId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceOwnerId());
        hashCode = 31 * hashCode + Objects.hashCode(resourceCidr());
        hashCode = 31 * hashCode + Objects.hashCode(resourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasResourceTags() ? resourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(ipUsage());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceAttachmentStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sampleTime());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamDiscoveredResourceCidr)) {
            return false;
        }
        IpamDiscoveredResourceCidr other = (IpamDiscoveredResourceCidr) obj;
        return Objects.equals(ipamResourceDiscoveryId(), other.ipamResourceDiscoveryId())
                && Objects.equals(resourceRegion(), other.resourceRegion()) && Objects.equals(resourceId(), other.resourceId())
                && Objects.equals(resourceOwnerId(), other.resourceOwnerId())
                && Objects.equals(resourceCidr(), other.resourceCidr())
                && Objects.equals(resourceTypeAsString(), other.resourceTypeAsString())
                && hasResourceTags() == other.hasResourceTags() && Objects.equals(resourceTags(), other.resourceTags())
                && Objects.equals(ipUsage(), other.ipUsage()) && Objects.equals(vpcId(), other.vpcId())
                && Objects.equals(networkInterfaceAttachmentStatusAsString(), other.networkInterfaceAttachmentStatusAsString())
                && Objects.equals(sampleTime(), other.sampleTime())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamDiscoveredResourceCidr").add("IpamResourceDiscoveryId", ipamResourceDiscoveryId())
                .add("ResourceRegion", resourceRegion()).add("ResourceId", resourceId())
                .add("ResourceOwnerId", resourceOwnerId()).add("ResourceCidr", resourceCidr())
                .add("ResourceType", resourceTypeAsString()).add("ResourceTags", hasResourceTags() ? resourceTags() : null)
                .add("IpUsage", ipUsage()).add("VpcId", vpcId())
                .add("NetworkInterfaceAttachmentStatus", networkInterfaceAttachmentStatusAsString())
                .add("SampleTime", sampleTime()).add("AvailabilityZoneId", availabilityZoneId()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IpamResourceDiscoveryId":
            return Optional.ofNullable(clazz.cast(ipamResourceDiscoveryId()));
        case "ResourceRegion":
            return Optional.ofNullable(clazz.cast(resourceRegion()));
        case "ResourceId":
            return Optional.ofNullable(clazz.cast(resourceId()));
        case "ResourceOwnerId":
            return Optional.ofNullable(clazz.cast(resourceOwnerId()));
        case "ResourceCidr":
            return Optional.ofNullable(clazz.cast(resourceCidr()));
        case "ResourceType":
            return Optional.ofNullable(clazz.cast(resourceTypeAsString()));
        case "ResourceTags":
            return Optional.ofNullable(clazz.cast(resourceTags()));
        case "IpUsage":
            return Optional.ofNullable(clazz.cast(ipUsage()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "NetworkInterfaceAttachmentStatus":
            return Optional.ofNullable(clazz.cast(networkInterfaceAttachmentStatusAsString()));
        case "SampleTime":
            return Optional.ofNullable(clazz.cast(sampleTime()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IpamDiscoveredResourceCidr, T> g) {
        return obj -> g.apply((IpamDiscoveredResourceCidr) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamDiscoveredResourceCidr> {
        /**
         * <p>
         * The resource discovery ID.
         * </p>
         * 
         * @param ipamResourceDiscoveryId
         *        The resource discovery ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId);

        /**
         * <p>
         * The resource Region.
         * </p>
         * 
         * @param resourceRegion
         *        The resource Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRegion(String resourceRegion);

        /**
         * <p>
         * The resource ID.
         * </p>
         * 
         * @param resourceId
         *        The resource ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceId(String resourceId);

        /**
         * <p>
         * The resource owner ID.
         * </p>
         * 
         * @param resourceOwnerId
         *        The resource owner ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceOwnerId(String resourceOwnerId);

        /**
         * <p>
         * The resource CIDR.
         * </p>
         * 
         * @param resourceCidr
         *        The resource CIDR.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceCidr(String resourceCidr);

        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(String resourceType);

        /**
         * <p>
         * The resource type.
         * </p>
         * 
         * @param resourceType
         *        The resource type.
         * @see IpamResourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamResourceType
         */
        Builder resourceType(IpamResourceType resourceType);

        /**
         * <p>
         * The resource tags.
         * </p>
         * 
         * @param resourceTags
         *        The resource tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(Collection<IpamResourceTag> resourceTags);

        /**
         * <p>
         * The resource tags.
         * </p>
         * 
         * @param resourceTags
         *        The resource tags.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceTags(IpamResourceTag... resourceTags);

        /**
         * <p>
         * The resource tags.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder#build()} is called immediately and
         * its result is passed to {@link #resourceTags(List<IpamResourceTag>)}.
         * 
         * @param resourceTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #resourceTags(java.util.Collection<IpamResourceTag>)
         */
        Builder resourceTags(Consumer<IpamResourceTag.Builder>... resourceTags);

        /**
         * <p>
         * The percentage of IP address space in use. To convert the decimal to a percentage, multiply the decimal by
         * 100. Note the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * For resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by subnet
         * CIDRs.
         * </p>
         * </li>
         * <li>
         * <p>
         * For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the percentage of
         * IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR provisioned to it, the
         * percentage of IPv6 address space in use is not represented. The percentage of IPv6 address space in use
         * cannot currently be calculated.
         * </p>
         * </li>
         * <li>
         * <p>
         * For resources that are public IPv4 pools, this is the percentage of IP address space in the pool that's been
         * allocated to Elastic IP addresses (EIPs).
         * </p>
         * </li>
         * </ul>
         * 
         * @param ipUsage
         *        The percentage of IP address space in use. To convert the decimal to a percentage, multiply the
         *        decimal by 100. Note the following:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For resources that are VPCs, this is the percentage of IP address space in the VPC that's taken up by
         *        subnet CIDRs.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For resources that are subnets, if the subnet has an IPv4 CIDR provisioned to it, this is the
         *        percentage of IPv4 address space in the subnet that's in use. If the subnet has an IPv6 CIDR
         *        provisioned to it, the percentage of IPv6 address space in use is not represented. The percentage of
         *        IPv6 address space in use cannot currently be calculated.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For resources that are public IPv4 pools, this is the percentage of IP address space in the pool
         *        that's been allocated to Elastic IP addresses (EIPs).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipUsage(Double ipUsage);

        /**
         * <p>
         * The VPC ID.
         * </p>
         * 
         * @param vpcId
         *        The VPC ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * For elastic IP addresses, this is the status of an attached network interface.
         * </p>
         * 
         * @param networkInterfaceAttachmentStatus
         *        For elastic IP addresses, this is the status of an attached network interface.
         * @see IpamNetworkInterfaceAttachmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamNetworkInterfaceAttachmentStatus
         */
        Builder networkInterfaceAttachmentStatus(String networkInterfaceAttachmentStatus);

        /**
         * <p>
         * For elastic IP addresses, this is the status of an attached network interface.
         * </p>
         * 
         * @param networkInterfaceAttachmentStatus
         *        For elastic IP addresses, this is the status of an attached network interface.
         * @see IpamNetworkInterfaceAttachmentStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamNetworkInterfaceAttachmentStatus
         */
        Builder networkInterfaceAttachmentStatus(IpamNetworkInterfaceAttachmentStatus networkInterfaceAttachmentStatus);

        /**
         * <p>
         * The last successful resource discovery time.
         * </p>
         * 
         * @param sampleTime
         *        The last successful resource discovery time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sampleTime(Instant sampleTime);

        /**
         * <p>
         * The Availability Zone ID.
         * </p>
         * 
         * @param availabilityZoneId
         *        The Availability Zone ID.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);
    }

    static final class BuilderImpl implements Builder {
        private String ipamResourceDiscoveryId;

        private String resourceRegion;

        private String resourceId;

        private String resourceOwnerId;

        private String resourceCidr;

        private String resourceType;

        private List<IpamResourceTag> resourceTags = DefaultSdkAutoConstructList.getInstance();

        private Double ipUsage;

        private String vpcId;

        private String networkInterfaceAttachmentStatus;

        private Instant sampleTime;

        private String availabilityZoneId;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamDiscoveredResourceCidr model) {
            ipamResourceDiscoveryId(model.ipamResourceDiscoveryId);
            resourceRegion(model.resourceRegion);
            resourceId(model.resourceId);
            resourceOwnerId(model.resourceOwnerId);
            resourceCidr(model.resourceCidr);
            resourceType(model.resourceType);
            resourceTags(model.resourceTags);
            ipUsage(model.ipUsage);
            vpcId(model.vpcId);
            networkInterfaceAttachmentStatus(model.networkInterfaceAttachmentStatus);
            sampleTime(model.sampleTime);
            availabilityZoneId(model.availabilityZoneId);
        }

        public final String getIpamResourceDiscoveryId() {
            return ipamResourceDiscoveryId;
        }

        public final void setIpamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
        }

        @Override
        public final Builder ipamResourceDiscoveryId(String ipamResourceDiscoveryId) {
            this.ipamResourceDiscoveryId = ipamResourceDiscoveryId;
            return this;
        }

        public final String getResourceRegion() {
            return resourceRegion;
        }

        public final void setResourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
        }

        @Override
        public final Builder resourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
            return this;
        }

        public final String getResourceId() {
            return resourceId;
        }

        public final void setResourceId(String resourceId) {
            this.resourceId = resourceId;
        }

        @Override
        public final Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public final String getResourceOwnerId() {
            return resourceOwnerId;
        }

        public final void setResourceOwnerId(String resourceOwnerId) {
            this.resourceOwnerId = resourceOwnerId;
        }

        @Override
        public final Builder resourceOwnerId(String resourceOwnerId) {
            this.resourceOwnerId = resourceOwnerId;
            return this;
        }

        public final String getResourceCidr() {
            return resourceCidr;
        }

        public final void setResourceCidr(String resourceCidr) {
            this.resourceCidr = resourceCidr;
        }

        @Override
        public final Builder resourceCidr(String resourceCidr) {
            this.resourceCidr = resourceCidr;
            return this;
        }

        public final String getResourceType() {
            return resourceType;
        }

        public final void setResourceType(String resourceType) {
            this.resourceType = resourceType;
        }

        @Override
        public final Builder resourceType(String resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        @Override
        public final Builder resourceType(IpamResourceType resourceType) {
            this.resourceType(resourceType == null ? null : resourceType.toString());
            return this;
        }

        public final List<IpamResourceTag.Builder> getResourceTags() {
            List<IpamResourceTag.Builder> result = IpamResourceTagListCopier.copyToBuilder(this.resourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setResourceTags(Collection<IpamResourceTag.BuilderImpl> resourceTags) {
            this.resourceTags = IpamResourceTagListCopier.copyFromBuilder(resourceTags);
        }

        @Override
        public final Builder resourceTags(Collection<IpamResourceTag> resourceTags) {
            this.resourceTags = IpamResourceTagListCopier.copy(resourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(IpamResourceTag... resourceTags) {
            resourceTags(Arrays.asList(resourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder resourceTags(Consumer<IpamResourceTag.Builder>... resourceTags) {
            resourceTags(Stream.of(resourceTags).map(c -> IpamResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Double getIpUsage() {
            return ipUsage;
        }

        public final void setIpUsage(Double ipUsage) {
            this.ipUsage = ipUsage;
        }

        @Override
        public final Builder ipUsage(Double ipUsage) {
            this.ipUsage = ipUsage;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getNetworkInterfaceAttachmentStatus() {
            return networkInterfaceAttachmentStatus;
        }

        public final void setNetworkInterfaceAttachmentStatus(String networkInterfaceAttachmentStatus) {
            this.networkInterfaceAttachmentStatus = networkInterfaceAttachmentStatus;
        }

        @Override
        public final Builder networkInterfaceAttachmentStatus(String networkInterfaceAttachmentStatus) {
            this.networkInterfaceAttachmentStatus = networkInterfaceAttachmentStatus;
            return this;
        }

        @Override
        public final Builder networkInterfaceAttachmentStatus(
                IpamNetworkInterfaceAttachmentStatus networkInterfaceAttachmentStatus) {
            this.networkInterfaceAttachmentStatus(networkInterfaceAttachmentStatus == null ? null
                    : networkInterfaceAttachmentStatus.toString());
            return this;
        }

        public final Instant getSampleTime() {
            return sampleTime;
        }

        public final void setSampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
        }

        @Override
        public final Builder sampleTime(Instant sampleTime) {
            this.sampleTime = sampleTime;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        @Override
        public IpamDiscoveredResourceCidr build() {
            return new IpamDiscoveredResourceCidr(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
