/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateSnapshotsRequest extends Ec2Request implements
        ToCopyableBuilder<CreateSnapshotsRequest.Builder, CreateSnapshotsRequest> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CreateSnapshotsRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<InstanceSpecification> INSTANCE_SPECIFICATION_FIELD = SdkField
            .<InstanceSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("InstanceSpecification")
            .getter(getter(CreateSnapshotsRequest::instanceSpecification))
            .setter(setter(Builder::instanceSpecification))
            .constructor(InstanceSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceSpecification")
                    .unmarshallLocationName("InstanceSpecification").build()).build();

    private static final SdkField<String> OUTPOST_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OutpostArn")
            .getter(getter(CreateSnapshotsRequest::outpostArn))
            .setter(setter(Builder::outpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OutpostArn")
                    .unmarshallLocationName("OutpostArn").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CreateSnapshotsRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CreateSnapshotsRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("DryRun").build()).build();

    private static final SdkField<String> COPY_TAGS_FROM_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CopyTagsFromSource")
            .getter(getter(CreateSnapshotsRequest::copyTagsFromSourceAsString))
            .setter(setter(Builder::copyTagsFromSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyTagsFromSource")
                    .unmarshallLocationName("CopyTagsFromSource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays
            .asList(DESCRIPTION_FIELD, INSTANCE_SPECIFICATION_FIELD, OUTPOST_ARN_FIELD, TAG_SPECIFICATIONS_FIELD, DRY_RUN_FIELD,
                    COPY_TAGS_FROM_SOURCE_FIELD));

    private final String description;

    private final InstanceSpecification instanceSpecification;

    private final String outpostArn;

    private final List<TagSpecification> tagSpecifications;

    private final Boolean dryRun;

    private final String copyTagsFromSource;

    private CreateSnapshotsRequest(BuilderImpl builder) {
        super(builder);
        this.description = builder.description;
        this.instanceSpecification = builder.instanceSpecification;
        this.outpostArn = builder.outpostArn;
        this.tagSpecifications = builder.tagSpecifications;
        this.dryRun = builder.dryRun;
        this.copyTagsFromSource = builder.copyTagsFromSource;
    }

    /**
     * <p>
     * A description propagated to every snapshot specified by the instance.
     * </p>
     * 
     * @return A description propagated to every snapshot specified by the instance.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The instance to specify which volumes should be included in the snapshots.
     * </p>
     * 
     * @return The instance to specify which volumes should be included in the snapshots.
     */
    public final InstanceSpecification instanceSpecification() {
        return instanceSpecification;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Outpost on which to create the local snapshots.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To create snapshots from an instance in a Region, omit this parameter. The snapshots are created in the same
     * Region as the instance.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots from an instance on an Outpost and store the snapshots in the Region, omit this parameter.
     * The snapshots are created in the Region for the Outpost.
     * </p>
     * </li>
     * <li>
     * <p>
     * To create snapshots from an instance on an Outpost and store the snapshots on an Outpost, specify the ARN of the
     * destination Outpost. The snapshots must be created on the same Outpost as the instance.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#create-multivol-snapshot"> Create
     * multi-volume local snapshots from instances on an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Outpost on which to create the local snapshots.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To create snapshots from an instance in a Region, omit this parameter. The snapshots are created in the
     *         same Region as the instance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots from an instance on an Outpost and store the snapshots in the Region, omit this
     *         parameter. The snapshots are created in the Region for the Outpost.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To create snapshots from an instance on an Outpost and store the snapshots on an Outpost, specify the ARN
     *         of the destination Outpost. The snapshots must be created on the same Outpost as the instance.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#create-multivol-snapshot">
     *         Create multi-volume local snapshots from instances on an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final String outpostArn() {
        return outpostArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags to apply to every snapshot specified by the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return Tags to apply to every snapshot specified by the instance.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * Copies the tags from the specified volume to corresponding snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #copyTagsFromSource} will return {@link CopyTagsFromSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #copyTagsFromSourceAsString}.
     * </p>
     * 
     * @return Copies the tags from the specified volume to corresponding snapshot.
     * @see CopyTagsFromSource
     */
    public final CopyTagsFromSource copyTagsFromSource() {
        return CopyTagsFromSource.fromValue(copyTagsFromSource);
    }

    /**
     * <p>
     * Copies the tags from the specified volume to corresponding snapshot.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #copyTagsFromSource} will return {@link CopyTagsFromSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned
     * by the service is available from {@link #copyTagsFromSourceAsString}.
     * </p>
     * 
     * @return Copies the tags from the specified volume to corresponding snapshot.
     * @see CopyTagsFromSource
     */
    public final String copyTagsFromSourceAsString() {
        return copyTagsFromSource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(instanceSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(outpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(copyTagsFromSourceAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateSnapshotsRequest)) {
            return false;
        }
        CreateSnapshotsRequest other = (CreateSnapshotsRequest) obj;
        return Objects.equals(description(), other.description())
                && Objects.equals(instanceSpecification(), other.instanceSpecification())
                && Objects.equals(outpostArn(), other.outpostArn()) && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(dryRun(), other.dryRun())
                && Objects.equals(copyTagsFromSourceAsString(), other.copyTagsFromSourceAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateSnapshotsRequest").add("Description", description())
                .add("InstanceSpecification", instanceSpecification()).add("OutpostArn", outpostArn())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("DryRun", dryRun())
                .add("CopyTagsFromSource", copyTagsFromSourceAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "InstanceSpecification":
            return Optional.ofNullable(clazz.cast(instanceSpecification()));
        case "OutpostArn":
            return Optional.ofNullable(clazz.cast(outpostArn()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "CopyTagsFromSource":
            return Optional.ofNullable(clazz.cast(copyTagsFromSourceAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateSnapshotsRequest, T> g) {
        return obj -> g.apply((CreateSnapshotsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateSnapshotsRequest> {
        /**
         * <p>
         * A description propagated to every snapshot specified by the instance.
         * </p>
         * 
         * @param description
         *        A description propagated to every snapshot specified by the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The instance to specify which volumes should be included in the snapshots.
         * </p>
         * 
         * @param instanceSpecification
         *        The instance to specify which volumes should be included in the snapshots.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceSpecification(InstanceSpecification instanceSpecification);

        /**
         * <p>
         * The instance to specify which volumes should be included in the snapshots.
         * </p>
         * This is a convenience method that creates an instance of the {@link InstanceSpecification.Builder} avoiding
         * the need to create one manually via {@link InstanceSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link InstanceSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #instanceSpecification(InstanceSpecification)}.
         * 
         * @param instanceSpecification
         *        a consumer that will call methods on {@link InstanceSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instanceSpecification(InstanceSpecification)
         */
        default Builder instanceSpecification(Consumer<InstanceSpecification.Builder> instanceSpecification) {
            return instanceSpecification(InstanceSpecification.builder().applyMutation(instanceSpecification).build());
        }

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Outpost on which to create the local snapshots.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To create snapshots from an instance in a Region, omit this parameter. The snapshots are created in the same
         * Region as the instance.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots from an instance on an Outpost and store the snapshots in the Region, omit this
         * parameter. The snapshots are created in the Region for the Outpost.
         * </p>
         * </li>
         * <li>
         * <p>
         * To create snapshots from an instance on an Outpost and store the snapshots on an Outpost, specify the ARN of
         * the destination Outpost. The snapshots must be created on the same Outpost as the instance.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#create-multivol-snapshot">
         * Create multi-volume local snapshots from instances on an Outpost</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param outpostArn
         *        The Amazon Resource Name (ARN) of the Outpost on which to create the local snapshots.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To create snapshots from an instance in a Region, omit this parameter. The snapshots are created in
         *        the same Region as the instance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots from an instance on an Outpost and store the snapshots in the Region, omit this
         *        parameter. The snapshots are created in the Region for the Outpost.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To create snapshots from an instance on an Outpost and store the snapshots on an Outpost, specify the
         *        ARN of the destination Outpost. The snapshots must be created on the same Outpost as the instance.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#create-multivol-snapshot">
         *        Create multi-volume local snapshots from instances on an Outpost</a> in the <i>Amazon EBS User
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder outpostArn(String outpostArn);

        /**
         * <p>
         * Tags to apply to every snapshot specified by the instance.
         * </p>
         * 
         * @param tagSpecifications
         *        Tags to apply to every snapshot specified by the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * Tags to apply to every snapshot specified by the instance.
         * </p>
         * 
         * @param tagSpecifications
         *        Tags to apply to every snapshot specified by the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * Tags to apply to every snapshot specified by the instance.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * Copies the tags from the specified volume to corresponding snapshot.
         * </p>
         * 
         * @param copyTagsFromSource
         *        Copies the tags from the specified volume to corresponding snapshot.
         * @see CopyTagsFromSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CopyTagsFromSource
         */
        Builder copyTagsFromSource(String copyTagsFromSource);

        /**
         * <p>
         * Copies the tags from the specified volume to corresponding snapshot.
         * </p>
         * 
         * @param copyTagsFromSource
         *        Copies the tags from the specified volume to corresponding snapshot.
         * @see CopyTagsFromSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CopyTagsFromSource
         */
        Builder copyTagsFromSource(CopyTagsFromSource copyTagsFromSource);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String description;

        private InstanceSpecification instanceSpecification;

        private String outpostArn;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private String copyTagsFromSource;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateSnapshotsRequest model) {
            super(model);
            description(model.description);
            instanceSpecification(model.instanceSpecification);
            outpostArn(model.outpostArn);
            tagSpecifications(model.tagSpecifications);
            dryRun(model.dryRun);
            copyTagsFromSource(model.copyTagsFromSource);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final InstanceSpecification.Builder getInstanceSpecification() {
            return instanceSpecification != null ? instanceSpecification.toBuilder() : null;
        }

        public final void setInstanceSpecification(InstanceSpecification.BuilderImpl instanceSpecification) {
            this.instanceSpecification = instanceSpecification != null ? instanceSpecification.build() : null;
        }

        @Override
        public final Builder instanceSpecification(InstanceSpecification instanceSpecification) {
            this.instanceSpecification = instanceSpecification;
            return this;
        }

        public final String getOutpostArn() {
            return outpostArn;
        }

        public final void setOutpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
        }

        @Override
        public final Builder outpostArn(String outpostArn) {
            this.outpostArn = outpostArn;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final String getCopyTagsFromSource() {
            return copyTagsFromSource;
        }

        public final void setCopyTagsFromSource(String copyTagsFromSource) {
            this.copyTagsFromSource = copyTagsFromSource;
        }

        @Override
        public final Builder copyTagsFromSource(String copyTagsFromSource) {
            this.copyTagsFromSource = copyTagsFromSource;
            return this;
        }

        @Override
        public final Builder copyTagsFromSource(CopyTagsFromSource copyTagsFromSource) {
            this.copyTagsFromSource(copyTagsFromSource == null ? null : copyTagsFromSource.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateSnapshotsRequest build() {
            return new CreateSnapshotsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
