/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The instance details to specify which volumes should be snapshotted.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceSpecification.Builder, InstanceSpecification> {
    private static final SdkField<String> INSTANCE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceId")
            .getter(getter(InstanceSpecification::instanceId))
            .setter(setter(Builder::instanceId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceId")
                    .unmarshallLocationName("InstanceId").build()).build();

    private static final SdkField<Boolean> EXCLUDE_BOOT_VOLUME_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("ExcludeBootVolume")
            .getter(getter(InstanceSpecification::excludeBootVolume))
            .setter(setter(Builder::excludeBootVolume))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeBootVolume")
                    .unmarshallLocationName("ExcludeBootVolume").build()).build();

    private static final SdkField<List<String>> EXCLUDE_DATA_VOLUME_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludeDataVolumeIds")
            .getter(getter(InstanceSpecification::excludeDataVolumeIds))
            .setter(setter(Builder::excludeDataVolumeIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludeDataVolumeId")
                    .unmarshallLocationName("ExcludeDataVolumeId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("VolumeId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("VolumeId").unmarshallLocationName("VolumeId").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ID_FIELD,
            EXCLUDE_BOOT_VOLUME_FIELD, EXCLUDE_DATA_VOLUME_IDS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceId;

    private final Boolean excludeBootVolume;

    private final List<String> excludeDataVolumeIds;

    private InstanceSpecification(BuilderImpl builder) {
        this.instanceId = builder.instanceId;
        this.excludeBootVolume = builder.excludeBootVolume;
        this.excludeDataVolumeIds = builder.excludeDataVolumeIds;
    }

    /**
     * <p>
     * The instance to specify which volumes should be snapshotted.
     * </p>
     * 
     * @return The instance to specify which volumes should be snapshotted.
     */
    public final String instanceId() {
        return instanceId;
    }

    /**
     * <p>
     * Excludes the root volume from being snapshotted.
     * </p>
     * 
     * @return Excludes the root volume from being snapshotted.
     */
    public final Boolean excludeBootVolume() {
        return excludeBootVolume;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludeDataVolumeIds property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludeDataVolumeIds() {
        return excludeDataVolumeIds != null && !(excludeDataVolumeIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify the ID of
     * the root volume, the request fails. To exclude the root volume, use <b>ExcludeBootVolume</b>.
     * </p>
     * <p>
     * You can specify up to 40 volume IDs per request.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludeDataVolumeIds} method.
     * </p>
     * 
     * @return The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify the
     *         ID of the root volume, the request fails. To exclude the root volume, use <b>ExcludeBootVolume</b>.</p>
     *         <p>
     *         You can specify up to 40 volume IDs per request.
     */
    public final List<String> excludeDataVolumeIds() {
        return excludeDataVolumeIds;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceId());
        hashCode = 31 * hashCode + Objects.hashCode(excludeBootVolume());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludeDataVolumeIds() ? excludeDataVolumeIds() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceSpecification)) {
            return false;
        }
        InstanceSpecification other = (InstanceSpecification) obj;
        return Objects.equals(instanceId(), other.instanceId()) && Objects.equals(excludeBootVolume(), other.excludeBootVolume())
                && hasExcludeDataVolumeIds() == other.hasExcludeDataVolumeIds()
                && Objects.equals(excludeDataVolumeIds(), other.excludeDataVolumeIds());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceSpecification").add("InstanceId", instanceId())
                .add("ExcludeBootVolume", excludeBootVolume())
                .add("ExcludeDataVolumeIds", hasExcludeDataVolumeIds() ? excludeDataVolumeIds() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceId":
            return Optional.ofNullable(clazz.cast(instanceId()));
        case "ExcludeBootVolume":
            return Optional.ofNullable(clazz.cast(excludeBootVolume()));
        case "ExcludeDataVolumeIds":
            return Optional.ofNullable(clazz.cast(excludeDataVolumeIds()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceSpecification, T> g) {
        return obj -> g.apply((InstanceSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceSpecification> {
        /**
         * <p>
         * The instance to specify which volumes should be snapshotted.
         * </p>
         * 
         * @param instanceId
         *        The instance to specify which volumes should be snapshotted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceId(String instanceId);

        /**
         * <p>
         * Excludes the root volume from being snapshotted.
         * </p>
         * 
         * @param excludeBootVolume
         *        Excludes the root volume from being snapshotted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeBootVolume(Boolean excludeBootVolume);

        /**
         * <p>
         * The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify the ID
         * of the root volume, the request fails. To exclude the root volume, use <b>ExcludeBootVolume</b>.
         * </p>
         * <p>
         * You can specify up to 40 volume IDs per request.
         * </p>
         * 
         * @param excludeDataVolumeIds
         *        The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify
         *        the ID of the root volume, the request fails. To exclude the root volume, use
         *        <b>ExcludeBootVolume</b>.</p>
         *        <p>
         *        You can specify up to 40 volume IDs per request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeDataVolumeIds(Collection<String> excludeDataVolumeIds);

        /**
         * <p>
         * The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify the ID
         * of the root volume, the request fails. To exclude the root volume, use <b>ExcludeBootVolume</b>.
         * </p>
         * <p>
         * You can specify up to 40 volume IDs per request.
         * </p>
         * 
         * @param excludeDataVolumeIds
         *        The IDs of the data (non-root) volumes to exclude from the multi-volume snapshot set. If you specify
         *        the ID of the root volume, the request fails. To exclude the root volume, use
         *        <b>ExcludeBootVolume</b>.</p>
         *        <p>
         *        You can specify up to 40 volume IDs per request.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludeDataVolumeIds(String... excludeDataVolumeIds);
    }

    static final class BuilderImpl implements Builder {
        private String instanceId;

        private Boolean excludeBootVolume;

        private List<String> excludeDataVolumeIds = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceSpecification model) {
            instanceId(model.instanceId);
            excludeBootVolume(model.excludeBootVolume);
            excludeDataVolumeIds(model.excludeDataVolumeIds);
        }

        public final String getInstanceId() {
            return instanceId;
        }

        public final void setInstanceId(String instanceId) {
            this.instanceId = instanceId;
        }

        @Override
        public final Builder instanceId(String instanceId) {
            this.instanceId = instanceId;
            return this;
        }

        public final Boolean getExcludeBootVolume() {
            return excludeBootVolume;
        }

        public final void setExcludeBootVolume(Boolean excludeBootVolume) {
            this.excludeBootVolume = excludeBootVolume;
        }

        @Override
        public final Builder excludeBootVolume(Boolean excludeBootVolume) {
            this.excludeBootVolume = excludeBootVolume;
            return this;
        }

        public final Collection<String> getExcludeDataVolumeIds() {
            if (excludeDataVolumeIds instanceof SdkAutoConstructList) {
                return null;
            }
            return excludeDataVolumeIds;
        }

        public final void setExcludeDataVolumeIds(Collection<String> excludeDataVolumeIds) {
            this.excludeDataVolumeIds = VolumeIdStringListCopier.copy(excludeDataVolumeIds);
        }

        @Override
        public final Builder excludeDataVolumeIds(Collection<String> excludeDataVolumeIds) {
            this.excludeDataVolumeIds = VolumeIdStringListCopier.copy(excludeDataVolumeIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludeDataVolumeIds(String... excludeDataVolumeIds) {
            excludeDataVolumeIds(Arrays.asList(excludeDataVolumeIds));
            return this;
        }

        @Override
        public InstanceSpecification build() {
            return new InstanceSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
