/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In IPAM, a pool is a collection of contiguous IP addresses CIDRs. Pools enable you to organize your IP addresses
 * according to your routing and security needs. For example, if you have separate routing and security needs for
 * development and production applications, you can create a pool for each.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPool implements SdkPojo, Serializable, ToCopyableBuilder<IpamPool.Builder, IpamPool> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamPool::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPoolId")
            .getter(getter(IpamPool::ipamPoolId))
            .setter(setter(Builder::ipamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPoolId")
                    .unmarshallLocationName("ipamPoolId").build()).build();

    private static final SdkField<String> SOURCE_IPAM_POOL_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceIpamPoolId")
            .getter(getter(IpamPool::sourceIpamPoolId))
            .setter(setter(Builder::sourceIpamPoolId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIpamPoolId")
                    .unmarshallLocationName("sourceIpamPoolId").build()).build();

    private static final SdkField<String> IPAM_POOL_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPoolArn")
            .getter(getter(IpamPool::ipamPoolArn))
            .setter(setter(Builder::ipamPoolArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPoolArn")
                    .unmarshallLocationName("ipamPoolArn").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeArn")
            .getter(getter(IpamPool::ipamScopeArn))
            .setter(setter(Builder::ipamScopeArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeArn")
                    .unmarshallLocationName("ipamScopeArn").build()).build();

    private static final SdkField<String> IPAM_SCOPE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeType")
            .getter(getter(IpamPool::ipamScopeTypeAsString))
            .setter(setter(Builder::ipamScopeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeType")
                    .unmarshallLocationName("ipamScopeType").build()).build();

    private static final SdkField<String> IPAM_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamArn")
            .getter(getter(IpamPool::ipamArn))
            .setter(setter(Builder::ipamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamArn")
                    .unmarshallLocationName("ipamArn").build()).build();

    private static final SdkField<String> IPAM_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamRegion")
            .getter(getter(IpamPool::ipamRegion))
            .setter(setter(Builder::ipamRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamRegion")
                    .unmarshallLocationName("ipamRegion").build()).build();

    private static final SdkField<String> LOCALE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Locale")
            .getter(getter(IpamPool::locale))
            .setter(setter(Builder::locale))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Locale")
                    .unmarshallLocationName("locale").build()).build();

    private static final SdkField<Integer> POOL_DEPTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("PoolDepth")
            .getter(getter(IpamPool::poolDepth))
            .setter(setter(Builder::poolDepth))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PoolDepth")
                    .unmarshallLocationName("poolDepth").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamPool::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> STATE_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StateMessage")
            .getter(getter(IpamPool::stateMessage))
            .setter(setter(Builder::stateMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StateMessage")
                    .unmarshallLocationName("stateMessage").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(IpamPool::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<Boolean> AUTO_IMPORT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("AutoImport")
            .getter(getter(IpamPool::autoImport))
            .setter(setter(Builder::autoImport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AutoImport")
                    .unmarshallLocationName("autoImport").build()).build();

    private static final SdkField<Boolean> PUBLICLY_ADVERTISABLE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("PubliclyAdvertisable")
            .getter(getter(IpamPool::publiclyAdvertisable))
            .setter(setter(Builder::publiclyAdvertisable))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PubliclyAdvertisable")
                    .unmarshallLocationName("publiclyAdvertisable").build()).build();

    private static final SdkField<String> ADDRESS_FAMILY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressFamily")
            .getter(getter(IpamPool::addressFamilyAsString))
            .setter(setter(Builder::addressFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressFamily")
                    .unmarshallLocationName("addressFamily").build()).build();

    private static final SdkField<Integer> ALLOCATION_MIN_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationMinNetmaskLength")
            .getter(getter(IpamPool::allocationMinNetmaskLength))
            .setter(setter(Builder::allocationMinNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationMinNetmaskLength")
                    .unmarshallLocationName("allocationMinNetmaskLength").build()).build();

    private static final SdkField<Integer> ALLOCATION_MAX_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationMaxNetmaskLength")
            .getter(getter(IpamPool::allocationMaxNetmaskLength))
            .setter(setter(Builder::allocationMaxNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationMaxNetmaskLength")
                    .unmarshallLocationName("allocationMaxNetmaskLength").build()).build();

    private static final SdkField<Integer> ALLOCATION_DEFAULT_NETMASK_LENGTH_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("AllocationDefaultNetmaskLength")
            .getter(getter(IpamPool::allocationDefaultNetmaskLength))
            .setter(setter(Builder::allocationDefaultNetmaskLength))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationDefaultNetmaskLength")
                    .unmarshallLocationName("allocationDefaultNetmaskLength").build()).build();

    private static final SdkField<List<IpamResourceTag>> ALLOCATION_RESOURCE_TAGS_FIELD = SdkField
            .<List<IpamResourceTag>> builder(MarshallingType.LIST)
            .memberName("AllocationResourceTags")
            .getter(getter(IpamPool::allocationResourceTags))
            .setter(setter(Builder::allocationResourceTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllocationResourceTagSet")
                    .unmarshallLocationName("allocationResourceTagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<IpamResourceTag> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpamResourceTag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamPool::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> AWS_SERVICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AwsService")
            .getter(getter(IpamPool::awsServiceAsString))
            .setter(setter(Builder::awsService))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AwsService")
                    .unmarshallLocationName("awsService").build()).build();

    private static final SdkField<String> PUBLIC_IP_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PublicIpSource")
            .getter(getter(IpamPool::publicIpSourceAsString))
            .setter(setter(Builder::publicIpSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PublicIpSource")
                    .unmarshallLocationName("publicIpSource").build()).build();

    private static final SdkField<IpamPoolSourceResource> SOURCE_RESOURCE_FIELD = SdkField
            .<IpamPoolSourceResource> builder(MarshallingType.SDK_POJO)
            .memberName("SourceResource")
            .getter(getter(IpamPool::sourceResource))
            .setter(setter(Builder::sourceResource))
            .constructor(IpamPoolSourceResource::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceResource")
                    .unmarshallLocationName("sourceResource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_POOL_ID_FIELD, SOURCE_IPAM_POOL_ID_FIELD, IPAM_POOL_ARN_FIELD, IPAM_SCOPE_ARN_FIELD, IPAM_SCOPE_TYPE_FIELD,
            IPAM_ARN_FIELD, IPAM_REGION_FIELD, LOCALE_FIELD, POOL_DEPTH_FIELD, STATE_FIELD, STATE_MESSAGE_FIELD,
            DESCRIPTION_FIELD, AUTO_IMPORT_FIELD, PUBLICLY_ADVERTISABLE_FIELD, ADDRESS_FAMILY_FIELD,
            ALLOCATION_MIN_NETMASK_LENGTH_FIELD, ALLOCATION_MAX_NETMASK_LENGTH_FIELD, ALLOCATION_DEFAULT_NETMASK_LENGTH_FIELD,
            ALLOCATION_RESOURCE_TAGS_FIELD, TAGS_FIELD, AWS_SERVICE_FIELD, PUBLIC_IP_SOURCE_FIELD, SOURCE_RESOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamPoolId;

    private final String sourceIpamPoolId;

    private final String ipamPoolArn;

    private final String ipamScopeArn;

    private final String ipamScopeType;

    private final String ipamArn;

    private final String ipamRegion;

    private final String locale;

    private final Integer poolDepth;

    private final String state;

    private final String stateMessage;

    private final String description;

    private final Boolean autoImport;

    private final Boolean publiclyAdvertisable;

    private final String addressFamily;

    private final Integer allocationMinNetmaskLength;

    private final Integer allocationMaxNetmaskLength;

    private final Integer allocationDefaultNetmaskLength;

    private final List<IpamResourceTag> allocationResourceTags;

    private final List<Tag> tags;

    private final String awsService;

    private final String publicIpSource;

    private final IpamPoolSourceResource sourceResource;

    private IpamPool(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamPoolId = builder.ipamPoolId;
        this.sourceIpamPoolId = builder.sourceIpamPoolId;
        this.ipamPoolArn = builder.ipamPoolArn;
        this.ipamScopeArn = builder.ipamScopeArn;
        this.ipamScopeType = builder.ipamScopeType;
        this.ipamArn = builder.ipamArn;
        this.ipamRegion = builder.ipamRegion;
        this.locale = builder.locale;
        this.poolDepth = builder.poolDepth;
        this.state = builder.state;
        this.stateMessage = builder.stateMessage;
        this.description = builder.description;
        this.autoImport = builder.autoImport;
        this.publiclyAdvertisable = builder.publiclyAdvertisable;
        this.addressFamily = builder.addressFamily;
        this.allocationMinNetmaskLength = builder.allocationMinNetmaskLength;
        this.allocationMaxNetmaskLength = builder.allocationMaxNetmaskLength;
        this.allocationDefaultNetmaskLength = builder.allocationDefaultNetmaskLength;
        this.allocationResourceTags = builder.allocationResourceTags;
        this.tags = builder.tags;
        this.awsService = builder.awsService;
        this.publicIpSource = builder.publicIpSource;
        this.sourceResource = builder.sourceResource;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the owner of the IPAM pool.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the owner of the IPAM pool.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the IPAM pool.
     * </p>
     * 
     * @return The ID of the IPAM pool.
     */
    public final String ipamPoolId() {
        return ipamPoolId;
    }

    /**
     * <p>
     * The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source pool.
     * </p>
     * 
     * @return The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source
     *         pool.
     */
    public final String sourceIpamPoolId() {
        return sourceIpamPoolId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IPAM pool.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IPAM pool.
     */
    public final String ipamPoolArn() {
        return ipamPoolArn;
    }

    /**
     * <p>
     * The ARN of the scope of the IPAM pool.
     * </p>
     * 
     * @return The ARN of the scope of the IPAM pool.
     */
    public final String ipamScopeArn() {
        return ipamScopeArn;
    }

    /**
     * <p>
     * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope
     * represents the IP space for a single network. The private scope is intended for all private IP address space. The
     * public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across multiple
     * unconnected networks without causing IP address overlap or conflict.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipamScopeType}
     * will return {@link IpamScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipamScopeTypeAsString}.
     * </p>
     * 
     * @return In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each
     *         scope represents the IP space for a single network. The private scope is intended for all private IP
     *         address space. The public scope is intended for all public IP address space. Scopes enable you to reuse
     *         IP addresses across multiple unconnected networks without causing IP address overlap or conflict.
     * @see IpamScopeType
     */
    public final IpamScopeType ipamScopeType() {
        return IpamScopeType.fromValue(ipamScopeType);
    }

    /**
     * <p>
     * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope
     * represents the IP space for a single network. The private scope is intended for all private IP address space. The
     * public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across multiple
     * unconnected networks without causing IP address overlap or conflict.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipamScopeType}
     * will return {@link IpamScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipamScopeTypeAsString}.
     * </p>
     * 
     * @return In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each
     *         scope represents the IP space for a single network. The private scope is intended for all private IP
     *         address space. The public scope is intended for all public IP address space. Scopes enable you to reuse
     *         IP addresses across multiple unconnected networks without causing IP address overlap or conflict.
     * @see IpamScopeType
     */
    public final String ipamScopeTypeAsString() {
        return ipamScopeType;
    }

    /**
     * <p>
     * The ARN of the IPAM.
     * </p>
     * 
     * @return The ARN of the IPAM.
     */
    public final String ipamArn() {
        return ipamArn;
    }

    /**
     * <p>
     * The Amazon Web Services Region of the IPAM pool.
     * </p>
     * 
     * @return The Amazon Web Services Region of the IPAM pool.
     */
    public final String ipamRegion() {
        return ipamRegion;
    }

    /**
     * <p>
     * The locale of the IPAM pool.
     * </p>
     * <p>
     * The locale for the pool should be one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
     * </p>
     * </li>
     * <li>
     * <p>
     * The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be available for
     * allocations (<a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported Local
     * Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you choose an Amazon Web Services Region for locale that has not been configured as an operating Region for
     * the IPAM, you'll get an error.
     * </p>
     * 
     * @return The locale of the IPAM pool.</p>
     *         <p>
     *         The locale for the pool should be one of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be
     *         available for allocations (<a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported
     *         Local Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you choose an Amazon Web Services Region for locale that has not been configured as an operating
     *         Region for the IPAM, you'll get an error.
     */
    public final String locale() {
        return locale;
    }

    /**
     * <p>
     * The depth of pools in your IPAM pool. The pool depth quota is 10. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon VPC IPAM
     * User Guide</i>.
     * </p>
     * 
     * @return The depth of pools in your IPAM pool. The pool depth quota is 10. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon
     *         VPC IPAM User Guide</i>.
     */
    public final Integer poolDepth() {
        return poolDepth;
    }

    /**
     * <p>
     * The state of the IPAM pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPoolState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM pool.
     * @see IpamPoolState
     */
    public final IpamPoolState state() {
        return IpamPoolState.fromValue(state);
    }

    /**
     * <p>
     * The state of the IPAM pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPoolState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM pool.
     * @see IpamPoolState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The state message.
     * </p>
     * 
     * @return The state message.
     */
    public final String stateMessage() {
        return stateMessage;
    }

    /**
     * <p>
     * The description of the IPAM pool.
     * </p>
     * 
     * @return The description of the IPAM pool.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically
     * import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not already
     * be allocated to other resources in order for the import to succeed. IPAM will import a CIDR regardless of its
     * compliance with the pool's allocation rules, so a resource might be imported and subsequently marked as
     * noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the largest CIDR only. If IPAM
     * discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of them only.
     * </p>
     * <p>
     * A locale must be set on the pool for this feature to work.
     * </p>
     * 
     * @return If selected, IPAM will continuously look for resources within the CIDR range of this pool and
     *         automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these
     *         resources must not already be allocated to other resources in order for the import to succeed. IPAM will
     *         import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might be
     *         imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM
     *         will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will
     *         randomly import one of them only. </p>
     *         <p>
     *         A locale must be set on the pool for this feature to work.
     */
    public final Boolean autoImport() {
        return autoImport;
    }

    /**
     * <p>
     * Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily set to
     * <code>ipv4</code>.
     * </p>
     * 
     * @return Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily
     *         set to <code>ipv4</code>.
     */
    public final Boolean publiclyAdvertisable() {
        return publiclyAdvertisable;
    }

    /**
     * <p>
     * The address family of the pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The address family of the pool.
     * @see AddressFamily
     */
    public final AddressFamily addressFamily() {
        return AddressFamily.fromValue(addressFamily);
    }

    /**
     * <p>
     * The address family of the pool.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The address family of the pool.
     * @see AddressFamily
     */
    public final String addressFamilyAsString() {
        return addressFamily;
    }

    /**
     * <p>
     * The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum netmask
     * length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32.
     * Possible netmask lengths for IPv6 addresses are 0 - 128.
     * </p>
     * 
     * @return The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum
     *         netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses
     *         are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
     */
    public final Integer allocationMinNetmaskLength() {
        return allocationMinNetmaskLength;
    }

    /**
     * <p>
     * The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum netmask
     * length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses are 0 - 32.
     * Possible netmask lengths for IPv6 addresses are 0 - 128.
     * </p>
     * 
     * @return The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum
     *         netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4
     *         addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
     */
    public final Integer allocationMaxNetmaskLength() {
        return allocationMaxNetmaskLength;
    }

    /**
     * <p>
     * The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this pool is
     * 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
     * </p>
     * 
     * @return The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this
     *         pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
     */
    public final Integer allocationDefaultNetmaskLength() {
        return allocationDefaultNetmaskLength;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllocationResourceTags property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAllocationResourceTags() {
        return allocationResourceTags != null && !(allocationResourceTags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these tags
     * will not be allowed to allocate space from the pool. If the resources have their tags changed after they have
     * allocated space or if the allocation tagging requirements are changed on the pool, the resource may be marked as
     * noncompliant.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllocationResourceTags} method.
     * </p>
     * 
     * @return Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
     *         tags will not be allowed to allocate space from the pool. If the resources have their tags changed after
     *         they have allocated space or if the allocation tagging requirements are changed on the pool, the resource
     *         may be marked as noncompliant.
     */
    public final List<IpamResourceTag> allocationResourceTags() {
        return allocationResourceTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value
     * as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code> and the
     * value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code> for the
     * filter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the
     *         tag value as the filter value. For example, to find all resources that have a tag with the key
     *         <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name
     *         and <code>TeamA</code> for the filter value.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to use
     * space for Elastic IP addresses and VPCs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #awsService} will
     * return {@link IpamPoolAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #awsServiceAsString}.
     * </p>
     * 
     * @return Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
     *         users to use space for Elastic IP addresses and VPCs.
     * @see IpamPoolAwsService
     */
    public final IpamPoolAwsService awsService() {
        return IpamPoolAwsService.fromValue(awsService);
    }

    /**
     * <p>
     * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to use
     * space for Elastic IP addresses and VPCs.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #awsService} will
     * return {@link IpamPoolAwsService#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #awsServiceAsString}.
     * </p>
     * 
     * @return Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
     *         users to use space for Elastic IP addresses and VPCs.
     * @see IpamPoolAwsService
     */
    public final String awsServiceAsString() {
        return awsService;
    }

    /**
     * <p>
     * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the
     * public scope. Default is <code>BYOIP</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
     * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
     * top-level IPv6 pool. For information on increasing the default limit, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicIpSource}
     * will return {@link IpamPoolPublicIpSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #publicIpSourceAsString}.
     * </p>
     * 
     * @return The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools
     *         in the public scope. Default is <code>BYOIP</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in
     *         the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block
     *         to a top-level IPv6 pool. For information on increasing the default limit, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamPoolPublicIpSource
     */
    public final IpamPoolPublicIpSource publicIpSource() {
        return IpamPoolPublicIpSource.fromValue(publicIpSource);
    }

    /**
     * <p>
     * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in the
     * public scope. Default is <code>BYOIP</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
     * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
     * top-level IPv6 pool. For information on increasing the default limit, see <a
     * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the <i>Amazon VPC
     * IPAM User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicIpSource}
     * will return {@link IpamPoolPublicIpSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #publicIpSourceAsString}.
     * </p>
     * 
     * @return The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools
     *         in the public scope. Default is <code>BYOIP</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in
     *         the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block
     *         to a top-level IPv6 pool. For information on increasing the default limit, see <a
     *         href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the
     *         <i>Amazon VPC IPAM User Guide</i>.
     * @see IpamPoolPublicIpSource
     */
    public final String publicIpSourceAsString() {
        return publicIpSource;
    }

    /**
     * <p>
     * The resource used to provision CIDRs to a resource planning pool.
     * </p>
     * 
     * @return The resource used to provision CIDRs to a resource planning pool.
     */
    public final IpamPoolSourceResource sourceResource() {
        return sourceResource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceIpamPoolId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPoolArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(ipamArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamRegion());
        hashCode = 31 * hashCode + Objects.hashCode(locale());
        hashCode = 31 * hashCode + Objects.hashCode(poolDepth());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateMessage());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(autoImport());
        hashCode = 31 * hashCode + Objects.hashCode(publiclyAdvertisable());
        hashCode = 31 * hashCode + Objects.hashCode(addressFamilyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(allocationMinNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(allocationMaxNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(allocationDefaultNetmaskLength());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllocationResourceTags() ? allocationResourceTags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(awsServiceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(publicIpSourceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(sourceResource());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPool)) {
            return false;
        }
        IpamPool other = (IpamPool) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(ipamPoolId(), other.ipamPoolId())
                && Objects.equals(sourceIpamPoolId(), other.sourceIpamPoolId())
                && Objects.equals(ipamPoolArn(), other.ipamPoolArn()) && Objects.equals(ipamScopeArn(), other.ipamScopeArn())
                && Objects.equals(ipamScopeTypeAsString(), other.ipamScopeTypeAsString())
                && Objects.equals(ipamArn(), other.ipamArn()) && Objects.equals(ipamRegion(), other.ipamRegion())
                && Objects.equals(locale(), other.locale()) && Objects.equals(poolDepth(), other.poolDepth())
                && Objects.equals(stateAsString(), other.stateAsString()) && Objects.equals(stateMessage(), other.stateMessage())
                && Objects.equals(description(), other.description()) && Objects.equals(autoImport(), other.autoImport())
                && Objects.equals(publiclyAdvertisable(), other.publiclyAdvertisable())
                && Objects.equals(addressFamilyAsString(), other.addressFamilyAsString())
                && Objects.equals(allocationMinNetmaskLength(), other.allocationMinNetmaskLength())
                && Objects.equals(allocationMaxNetmaskLength(), other.allocationMaxNetmaskLength())
                && Objects.equals(allocationDefaultNetmaskLength(), other.allocationDefaultNetmaskLength())
                && hasAllocationResourceTags() == other.hasAllocationResourceTags()
                && Objects.equals(allocationResourceTags(), other.allocationResourceTags()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags()) && Objects.equals(awsServiceAsString(), other.awsServiceAsString())
                && Objects.equals(publicIpSourceAsString(), other.publicIpSourceAsString())
                && Objects.equals(sourceResource(), other.sourceResource());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPool").add("OwnerId", ownerId()).add("IpamPoolId", ipamPoolId())
                .add("SourceIpamPoolId", sourceIpamPoolId()).add("IpamPoolArn", ipamPoolArn())
                .add("IpamScopeArn", ipamScopeArn()).add("IpamScopeType", ipamScopeTypeAsString()).add("IpamArn", ipamArn())
                .add("IpamRegion", ipamRegion()).add("Locale", locale()).add("PoolDepth", poolDepth())
                .add("State", stateAsString()).add("StateMessage", stateMessage()).add("Description", description())
                .add("AutoImport", autoImport()).add("PubliclyAdvertisable", publiclyAdvertisable())
                .add("AddressFamily", addressFamilyAsString()).add("AllocationMinNetmaskLength", allocationMinNetmaskLength())
                .add("AllocationMaxNetmaskLength", allocationMaxNetmaskLength())
                .add("AllocationDefaultNetmaskLength", allocationDefaultNetmaskLength())
                .add("AllocationResourceTags", hasAllocationResourceTags() ? allocationResourceTags() : null)
                .add("Tags", hasTags() ? tags() : null).add("AwsService", awsServiceAsString())
                .add("PublicIpSource", publicIpSourceAsString()).add("SourceResource", sourceResource()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamPoolId":
            return Optional.ofNullable(clazz.cast(ipamPoolId()));
        case "SourceIpamPoolId":
            return Optional.ofNullable(clazz.cast(sourceIpamPoolId()));
        case "IpamPoolArn":
            return Optional.ofNullable(clazz.cast(ipamPoolArn()));
        case "IpamScopeArn":
            return Optional.ofNullable(clazz.cast(ipamScopeArn()));
        case "IpamScopeType":
            return Optional.ofNullable(clazz.cast(ipamScopeTypeAsString()));
        case "IpamArn":
            return Optional.ofNullable(clazz.cast(ipamArn()));
        case "IpamRegion":
            return Optional.ofNullable(clazz.cast(ipamRegion()));
        case "Locale":
            return Optional.ofNullable(clazz.cast(locale()));
        case "PoolDepth":
            return Optional.ofNullable(clazz.cast(poolDepth()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "StateMessage":
            return Optional.ofNullable(clazz.cast(stateMessage()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AutoImport":
            return Optional.ofNullable(clazz.cast(autoImport()));
        case "PubliclyAdvertisable":
            return Optional.ofNullable(clazz.cast(publiclyAdvertisable()));
        case "AddressFamily":
            return Optional.ofNullable(clazz.cast(addressFamilyAsString()));
        case "AllocationMinNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationMinNetmaskLength()));
        case "AllocationMaxNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationMaxNetmaskLength()));
        case "AllocationDefaultNetmaskLength":
            return Optional.ofNullable(clazz.cast(allocationDefaultNetmaskLength()));
        case "AllocationResourceTags":
            return Optional.ofNullable(clazz.cast(allocationResourceTags()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "AwsService":
            return Optional.ofNullable(clazz.cast(awsServiceAsString()));
        case "PublicIpSource":
            return Optional.ofNullable(clazz.cast(publicIpSourceAsString()));
        case "SourceResource":
            return Optional.ofNullable(clazz.cast(sourceResource()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<IpamPool, T> g) {
        return obj -> g.apply((IpamPool) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPool> {
        /**
         * <p>
         * The Amazon Web Services account ID of the owner of the IPAM pool.
         * </p>
         * 
         * @param ownerId
         *        The Amazon Web Services account ID of the owner of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the IPAM pool.
         * </p>
         * 
         * @param ipamPoolId
         *        The ID of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPoolId(String ipamPoolId);

        /**
         * <p>
         * The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing source
         * pool.
         * </p>
         * 
         * @param sourceIpamPoolId
         *        The ID of the source IPAM pool. You can use this option to create an IPAM pool within an existing
         *        source pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIpamPoolId(String sourceIpamPoolId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IPAM pool.
         * </p>
         * 
         * @param ipamPoolArn
         *        The Amazon Resource Name (ARN) of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPoolArn(String ipamPoolArn);

        /**
         * <p>
         * The ARN of the scope of the IPAM pool.
         * </p>
         * 
         * @param ipamScopeArn
         *        The ARN of the scope of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeArn(String ipamScopeArn);

        /**
         * <p>
         * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope
         * represents the IP space for a single network. The private scope is intended for all private IP address space.
         * The public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across
         * multiple unconnected networks without causing IP address overlap or conflict.
         * </p>
         * 
         * @param ipamScopeType
         *        In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each
         *        scope represents the IP space for a single network. The private scope is intended for all private IP
         *        address space. The public scope is intended for all public IP address space. Scopes enable you to
         *        reuse IP addresses across multiple unconnected networks without causing IP address overlap or
         *        conflict.
         * @see IpamScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeType
         */
        Builder ipamScopeType(String ipamScopeType);

        /**
         * <p>
         * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope
         * represents the IP space for a single network. The private scope is intended for all private IP address space.
         * The public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across
         * multiple unconnected networks without causing IP address overlap or conflict.
         * </p>
         * 
         * @param ipamScopeType
         *        In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each
         *        scope represents the IP space for a single network. The private scope is intended for all private IP
         *        address space. The public scope is intended for all public IP address space. Scopes enable you to
         *        reuse IP addresses across multiple unconnected networks without causing IP address overlap or
         *        conflict.
         * @see IpamScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeType
         */
        Builder ipamScopeType(IpamScopeType ipamScopeType);

        /**
         * <p>
         * The ARN of the IPAM.
         * </p>
         * 
         * @param ipamArn
         *        The ARN of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamArn(String ipamArn);

        /**
         * <p>
         * The Amazon Web Services Region of the IPAM pool.
         * </p>
         * 
         * @param ipamRegion
         *        The Amazon Web Services Region of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamRegion(String ipamRegion);

        /**
         * <p>
         * The locale of the IPAM pool.
         * </p>
         * <p>
         * The locale for the pool should be one of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
         * </p>
         * </li>
         * <li>
         * <p>
         * The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be available
         * for allocations (<a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported Local
         * Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you choose an Amazon Web Services Region for locale that has not been configured as an operating Region
         * for the IPAM, you'll get an error.
         * </p>
         * 
         * @param locale
         *        The locale of the IPAM pool.</p>
         *        <p>
         *        The locale for the pool should be one of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        An Amazon Web Services Region where you want this IPAM pool to be available for allocations.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The network border group for an Amazon Web Services Local Zone where you want this IPAM pool to be
         *        available for allocations (<a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-byoip.html#byoip-zone-avail">supported
         *        Local Zones</a>). This option is only available for IPAM IPv4 pools in the public scope.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you choose an Amazon Web Services Region for locale that has not been configured as an operating
         *        Region for the IPAM, you'll get an error.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder locale(String locale);

        /**
         * <p>
         * The depth of pools in your IPAM pool. The pool depth quota is 10. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the <i>Amazon VPC
         * IPAM User Guide</i>.
         * </p>
         * 
         * @param poolDepth
         *        The depth of pools in your IPAM pool. The pool depth quota is 10. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas in IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder poolDepth(Integer poolDepth);

        /**
         * <p>
         * The state of the IPAM pool.
         * </p>
         * 
         * @param state
         *        The state of the IPAM pool.
         * @see IpamPoolState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the IPAM pool.
         * </p>
         * 
         * @param state
         *        The state of the IPAM pool.
         * @see IpamPoolState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolState
         */
        Builder state(IpamPoolState state);

        /**
         * <p>
         * The state message.
         * </p>
         * 
         * @param stateMessage
         *        The state message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stateMessage(String stateMessage);

        /**
         * <p>
         * The description of the IPAM pool.
         * </p>
         * 
         * @param description
         *        The description of the IPAM pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * If selected, IPAM will continuously look for resources within the CIDR range of this pool and automatically
         * import them as allocations into your IPAM. The CIDRs that will be allocated for these resources must not
         * already be allocated to other resources in order for the import to succeed. IPAM will import a CIDR
         * regardless of its compliance with the pool's allocation rules, so a resource might be imported and
         * subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap, IPAM will import the
         * largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM will randomly import one of
         * them only.
         * </p>
         * <p>
         * A locale must be set on the pool for this feature to work.
         * </p>
         * 
         * @param autoImport
         *        If selected, IPAM will continuously look for resources within the CIDR range of this pool and
         *        automatically import them as allocations into your IPAM. The CIDRs that will be allocated for these
         *        resources must not already be allocated to other resources in order for the import to succeed. IPAM
         *        will import a CIDR regardless of its compliance with the pool's allocation rules, so a resource might
         *        be imported and subsequently marked as noncompliant. If IPAM discovers multiple CIDRs that overlap,
         *        IPAM will import the largest CIDR only. If IPAM discovers multiple CIDRs with matching CIDRs, IPAM
         *        will randomly import one of them only. </p>
         *        <p>
         *        A locale must be set on the pool for this feature to work.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder autoImport(Boolean autoImport);

        /**
         * <p>
         * Determines if a pool is publicly advertisable. This option is not available for pools with AddressFamily set
         * to <code>ipv4</code>.
         * </p>
         * 
         * @param publiclyAdvertisable
         *        Determines if a pool is publicly advertisable. This option is not available for pools with
         *        AddressFamily set to <code>ipv4</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder publiclyAdvertisable(Boolean publiclyAdvertisable);

        /**
         * <p>
         * The address family of the pool.
         * </p>
         * 
         * @param addressFamily
         *        The address family of the pool.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(String addressFamily);

        /**
         * <p>
         * The address family of the pool.
         * </p>
         * 
         * @param addressFamily
         *        The address family of the pool.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(AddressFamily addressFamily);

        /**
         * <p>
         * The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The minimum
         * netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4 addresses are
         * 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * </p>
         * 
         * @param allocationMinNetmaskLength
         *        The minimum netmask length required for CIDR allocations in this IPAM pool to be compliant. The
         *        minimum netmask length must be less than the maximum netmask length. Possible netmask lengths for IPv4
         *        addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationMinNetmaskLength(Integer allocationMinNetmaskLength);

        /**
         * <p>
         * The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The maximum
         * netmask length must be greater than the minimum netmask length. Possible netmask lengths for IPv4 addresses
         * are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * </p>
         * 
         * @param allocationMaxNetmaskLength
         *        The maximum netmask length possible for CIDR allocations in this IPAM pool to be compliant. The
         *        maximum netmask length must be greater than the minimum netmask length. Possible netmask lengths for
         *        IPv4 addresses are 0 - 32. Possible netmask lengths for IPv6 addresses are 0 - 128.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationMaxNetmaskLength(Integer allocationMaxNetmaskLength);

        /**
         * <p>
         * The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to this
         * pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
         * </p>
         * 
         * @param allocationDefaultNetmaskLength
         *        The default netmask length for allocations added to this pool. If, for example, the CIDR assigned to
         *        this pool is 10.0.0.0/8 and you enter 16 here, new allocations will default to 10.0.0.0/16.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * 
         * @param allocationResourceTags
         *        Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have
         *        these tags will not be allowed to allocate space from the pool. If the resources have their tags
         *        changed after they have allocated space or if the allocation tagging requirements are changed on the
         *        pool, the resource may be marked as noncompliant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationResourceTags(Collection<IpamResourceTag> allocationResourceTags);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * 
         * @param allocationResourceTags
         *        Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have
         *        these tags will not be allowed to allocate space from the pool. If the resources have their tags
         *        changed after they have allocated space or if the allocation tagging requirements are changed on the
         *        pool, the resource may be marked as noncompliant.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allocationResourceTags(IpamResourceTag... allocationResourceTags);

        /**
         * <p>
         * Tags that are required for resources that use CIDRs from this IPAM pool. Resources that do not have these
         * tags will not be allowed to allocate space from the pool. If the resources have their tags changed after they
         * have allocated space or if the allocation tagging requirements are changed on the pool, the resource may be
         * marked as noncompliant.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder#build()} is called immediately and
         * its result is passed to {@link #allocationResourceTags(List<IpamResourceTag>)}.
         * 
         * @param allocationResourceTags
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.IpamResourceTag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #allocationResourceTags(java.util.Collection<IpamResourceTag>)
         */
        Builder allocationResourceTags(Consumer<IpamResourceTag.Builder>... allocationResourceTags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to
         * use space for Elastic IP addresses and VPCs.
         * </p>
         * 
         * @param awsService
         *        Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
         *        users to use space for Elastic IP addresses and VPCs.
         * @see IpamPoolAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolAwsService
         */
        Builder awsService(String awsService);

        /**
         * <p>
         * Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows users to
         * use space for Elastic IP addresses and VPCs.
         * </p>
         * 
         * @param awsService
         *        Limits which service in Amazon Web Services that the pool can be used in. "ec2", for example, allows
         *        users to use space for Elastic IP addresses and VPCs.
         * @see IpamPoolAwsService
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolAwsService
         */
        Builder awsService(IpamPoolAwsService awsService);

        /**
         * <p>
         * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in
         * the public scope. Default is <code>BYOIP</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
         * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
         * top-level IPv6 pool. For information on increasing the default limit, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param publicIpSource
         *        The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to
         *        pools in the public scope. Default is <code>BYOIP</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a>
         *        in the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR
         *        block to a top-level IPv6 pool. For information on increasing the default limit, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamPoolPublicIpSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolPublicIpSource
         */
        Builder publicIpSource(String publicIpSource);

        /**
         * <p>
         * The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to pools in
         * the public scope. Default is <code>BYOIP</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a> in the
         * <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR block to a
         * top-level IPv6 pool. For information on increasing the default limit, see <a
         * href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the <i>Amazon
         * VPC IPAM User Guide</i>.
         * </p>
         * 
         * @param publicIpSource
         *        The IP address source for pools in the public scope. Only used for provisioning IP address CIDRs to
         *        pools in the public scope. Default is <code>BYOIP</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/intro-create-ipv6-pools.html">Create IPv6 pools</a>
         *        in the <i>Amazon VPC IPAM User Guide</i>. By default, you can add only one Amazon-provided IPv6 CIDR
         *        block to a top-level IPv6 pool. For information on increasing the default limit, see <a
         *        href="https://docs.aws.amazon.com/vpc/latest/ipam/quotas-ipam.html">Quotas for your IPAM</a> in the
         *        <i>Amazon VPC IPAM User Guide</i>.
         * @see IpamPoolPublicIpSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPoolPublicIpSource
         */
        Builder publicIpSource(IpamPoolPublicIpSource publicIpSource);

        /**
         * <p>
         * The resource used to provision CIDRs to a resource planning pool.
         * </p>
         * 
         * @param sourceResource
         *        The resource used to provision CIDRs to a resource planning pool.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceResource(IpamPoolSourceResource sourceResource);

        /**
         * <p>
         * The resource used to provision CIDRs to a resource planning pool.
         * </p>
         * This is a convenience method that creates an instance of the {@link IpamPoolSourceResource.Builder} avoiding
         * the need to create one manually via {@link IpamPoolSourceResource#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link IpamPoolSourceResource.Builder#build()} is called immediately and
         * its result is passed to {@link #sourceResource(IpamPoolSourceResource)}.
         * 
         * @param sourceResource
         *        a consumer that will call methods on {@link IpamPoolSourceResource.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #sourceResource(IpamPoolSourceResource)
         */
        default Builder sourceResource(Consumer<IpamPoolSourceResource.Builder> sourceResource) {
            return sourceResource(IpamPoolSourceResource.builder().applyMutation(sourceResource).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamPoolId;

        private String sourceIpamPoolId;

        private String ipamPoolArn;

        private String ipamScopeArn;

        private String ipamScopeType;

        private String ipamArn;

        private String ipamRegion;

        private String locale;

        private Integer poolDepth;

        private String state;

        private String stateMessage;

        private String description;

        private Boolean autoImport;

        private Boolean publiclyAdvertisable;

        private String addressFamily;

        private Integer allocationMinNetmaskLength;

        private Integer allocationMaxNetmaskLength;

        private Integer allocationDefaultNetmaskLength;

        private List<IpamResourceTag> allocationResourceTags = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String awsService;

        private String publicIpSource;

        private IpamPoolSourceResource sourceResource;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPool model) {
            ownerId(model.ownerId);
            ipamPoolId(model.ipamPoolId);
            sourceIpamPoolId(model.sourceIpamPoolId);
            ipamPoolArn(model.ipamPoolArn);
            ipamScopeArn(model.ipamScopeArn);
            ipamScopeType(model.ipamScopeType);
            ipamArn(model.ipamArn);
            ipamRegion(model.ipamRegion);
            locale(model.locale);
            poolDepth(model.poolDepth);
            state(model.state);
            stateMessage(model.stateMessage);
            description(model.description);
            autoImport(model.autoImport);
            publiclyAdvertisable(model.publiclyAdvertisable);
            addressFamily(model.addressFamily);
            allocationMinNetmaskLength(model.allocationMinNetmaskLength);
            allocationMaxNetmaskLength(model.allocationMaxNetmaskLength);
            allocationDefaultNetmaskLength(model.allocationDefaultNetmaskLength);
            allocationResourceTags(model.allocationResourceTags);
            tags(model.tags);
            awsService(model.awsService);
            publicIpSource(model.publicIpSource);
            sourceResource(model.sourceResource);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamPoolId() {
            return ipamPoolId;
        }

        public final void setIpamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
        }

        @Override
        public final Builder ipamPoolId(String ipamPoolId) {
            this.ipamPoolId = ipamPoolId;
            return this;
        }

        public final String getSourceIpamPoolId() {
            return sourceIpamPoolId;
        }

        public final void setSourceIpamPoolId(String sourceIpamPoolId) {
            this.sourceIpamPoolId = sourceIpamPoolId;
        }

        @Override
        public final Builder sourceIpamPoolId(String sourceIpamPoolId) {
            this.sourceIpamPoolId = sourceIpamPoolId;
            return this;
        }

        public final String getIpamPoolArn() {
            return ipamPoolArn;
        }

        public final void setIpamPoolArn(String ipamPoolArn) {
            this.ipamPoolArn = ipamPoolArn;
        }

        @Override
        public final Builder ipamPoolArn(String ipamPoolArn) {
            this.ipamPoolArn = ipamPoolArn;
            return this;
        }

        public final String getIpamScopeArn() {
            return ipamScopeArn;
        }

        public final void setIpamScopeArn(String ipamScopeArn) {
            this.ipamScopeArn = ipamScopeArn;
        }

        @Override
        public final Builder ipamScopeArn(String ipamScopeArn) {
            this.ipamScopeArn = ipamScopeArn;
            return this;
        }

        public final String getIpamScopeType() {
            return ipamScopeType;
        }

        public final void setIpamScopeType(String ipamScopeType) {
            this.ipamScopeType = ipamScopeType;
        }

        @Override
        public final Builder ipamScopeType(String ipamScopeType) {
            this.ipamScopeType = ipamScopeType;
            return this;
        }

        @Override
        public final Builder ipamScopeType(IpamScopeType ipamScopeType) {
            this.ipamScopeType(ipamScopeType == null ? null : ipamScopeType.toString());
            return this;
        }

        public final String getIpamArn() {
            return ipamArn;
        }

        public final void setIpamArn(String ipamArn) {
            this.ipamArn = ipamArn;
        }

        @Override
        public final Builder ipamArn(String ipamArn) {
            this.ipamArn = ipamArn;
            return this;
        }

        public final String getIpamRegion() {
            return ipamRegion;
        }

        public final void setIpamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
        }

        @Override
        public final Builder ipamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
            return this;
        }

        public final String getLocale() {
            return locale;
        }

        public final void setLocale(String locale) {
            this.locale = locale;
        }

        @Override
        public final Builder locale(String locale) {
            this.locale = locale;
            return this;
        }

        public final Integer getPoolDepth() {
            return poolDepth;
        }

        public final void setPoolDepth(Integer poolDepth) {
            this.poolDepth = poolDepth;
        }

        @Override
        public final Builder poolDepth(Integer poolDepth) {
            this.poolDepth = poolDepth;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamPoolState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getStateMessage() {
            return stateMessage;
        }

        public final void setStateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
        }

        @Override
        public final Builder stateMessage(String stateMessage) {
            this.stateMessage = stateMessage;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Boolean getAutoImport() {
            return autoImport;
        }

        public final void setAutoImport(Boolean autoImport) {
            this.autoImport = autoImport;
        }

        @Override
        public final Builder autoImport(Boolean autoImport) {
            this.autoImport = autoImport;
            return this;
        }

        public final Boolean getPubliclyAdvertisable() {
            return publiclyAdvertisable;
        }

        public final void setPubliclyAdvertisable(Boolean publiclyAdvertisable) {
            this.publiclyAdvertisable = publiclyAdvertisable;
        }

        @Override
        public final Builder publiclyAdvertisable(Boolean publiclyAdvertisable) {
            this.publiclyAdvertisable = publiclyAdvertisable;
            return this;
        }

        public final String getAddressFamily() {
            return addressFamily;
        }

        public final void setAddressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
        }

        @Override
        public final Builder addressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
            return this;
        }

        @Override
        public final Builder addressFamily(AddressFamily addressFamily) {
            this.addressFamily(addressFamily == null ? null : addressFamily.toString());
            return this;
        }

        public final Integer getAllocationMinNetmaskLength() {
            return allocationMinNetmaskLength;
        }

        public final void setAllocationMinNetmaskLength(Integer allocationMinNetmaskLength) {
            this.allocationMinNetmaskLength = allocationMinNetmaskLength;
        }

        @Override
        public final Builder allocationMinNetmaskLength(Integer allocationMinNetmaskLength) {
            this.allocationMinNetmaskLength = allocationMinNetmaskLength;
            return this;
        }

        public final Integer getAllocationMaxNetmaskLength() {
            return allocationMaxNetmaskLength;
        }

        public final void setAllocationMaxNetmaskLength(Integer allocationMaxNetmaskLength) {
            this.allocationMaxNetmaskLength = allocationMaxNetmaskLength;
        }

        @Override
        public final Builder allocationMaxNetmaskLength(Integer allocationMaxNetmaskLength) {
            this.allocationMaxNetmaskLength = allocationMaxNetmaskLength;
            return this;
        }

        public final Integer getAllocationDefaultNetmaskLength() {
            return allocationDefaultNetmaskLength;
        }

        public final void setAllocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength) {
            this.allocationDefaultNetmaskLength = allocationDefaultNetmaskLength;
        }

        @Override
        public final Builder allocationDefaultNetmaskLength(Integer allocationDefaultNetmaskLength) {
            this.allocationDefaultNetmaskLength = allocationDefaultNetmaskLength;
            return this;
        }

        public final List<IpamResourceTag.Builder> getAllocationResourceTags() {
            List<IpamResourceTag.Builder> result = IpamResourceTagListCopier.copyToBuilder(this.allocationResourceTags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAllocationResourceTags(Collection<IpamResourceTag.BuilderImpl> allocationResourceTags) {
            this.allocationResourceTags = IpamResourceTagListCopier.copyFromBuilder(allocationResourceTags);
        }

        @Override
        public final Builder allocationResourceTags(Collection<IpamResourceTag> allocationResourceTags) {
            this.allocationResourceTags = IpamResourceTagListCopier.copy(allocationResourceTags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allocationResourceTags(IpamResourceTag... allocationResourceTags) {
            allocationResourceTags(Arrays.asList(allocationResourceTags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allocationResourceTags(Consumer<IpamResourceTag.Builder>... allocationResourceTags) {
            allocationResourceTags(Stream.of(allocationResourceTags).map(c -> IpamResourceTag.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getAwsService() {
            return awsService;
        }

        public final void setAwsService(String awsService) {
            this.awsService = awsService;
        }

        @Override
        public final Builder awsService(String awsService) {
            this.awsService = awsService;
            return this;
        }

        @Override
        public final Builder awsService(IpamPoolAwsService awsService) {
            this.awsService(awsService == null ? null : awsService.toString());
            return this;
        }

        public final String getPublicIpSource() {
            return publicIpSource;
        }

        public final void setPublicIpSource(String publicIpSource) {
            this.publicIpSource = publicIpSource;
        }

        @Override
        public final Builder publicIpSource(String publicIpSource) {
            this.publicIpSource = publicIpSource;
            return this;
        }

        @Override
        public final Builder publicIpSource(IpamPoolPublicIpSource publicIpSource) {
            this.publicIpSource(publicIpSource == null ? null : publicIpSource.toString());
            return this;
        }

        public final IpamPoolSourceResource.Builder getSourceResource() {
            return sourceResource != null ? sourceResource.toBuilder() : null;
        }

        public final void setSourceResource(IpamPoolSourceResource.BuilderImpl sourceResource) {
            this.sourceResource = sourceResource != null ? sourceResource.build() : null;
        }

        @Override
        public final Builder sourceResource(IpamPoolSourceResource sourceResource) {
            this.sourceResource = sourceResource;
            return this;
        }

        @Override
        public IpamPool build() {
            return new IpamPool(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
