/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about an address range that is provisioned for use with your Amazon Web Services resources through bring
 * your own IP addresses (BYOIP).
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ByoipCidr implements SdkPojo, Serializable, ToCopyableBuilder<ByoipCidr.Builder, ByoipCidr> {
    private static final SdkField<String> CIDR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Cidr")
            .getter(getter(ByoipCidr::cidr))
            .setter(setter(Builder::cidr))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Cidr")
                    .unmarshallLocationName("cidr").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(ByoipCidr::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<List<AsnAssociation>> ASN_ASSOCIATIONS_FIELD = SdkField
            .<List<AsnAssociation>> builder(MarshallingType.LIST)
            .memberName("AsnAssociations")
            .getter(getter(ByoipCidr::asnAssociations))
            .setter(setter(Builder::asnAssociations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AsnAssociationSet")
                    .unmarshallLocationName("asnAssociationSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<AsnAssociation> builder(MarshallingType.SDK_POJO)
                                            .constructor(AsnAssociation::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> STATUS_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("StatusMessage")
            .getter(getter(ByoipCidr::statusMessage))
            .setter(setter(Builder::statusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusMessage")
                    .unmarshallLocationName("statusMessage").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(ByoipCidr::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> NETWORK_BORDER_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NetworkBorderGroup")
            .getter(getter(ByoipCidr::networkBorderGroup))
            .setter(setter(Builder::networkBorderGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkBorderGroup")
                    .unmarshallLocationName("networkBorderGroup").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CIDR_FIELD, DESCRIPTION_FIELD,
            ASN_ASSOCIATIONS_FIELD, STATUS_MESSAGE_FIELD, STATE_FIELD, NETWORK_BORDER_GROUP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String cidr;

    private final String description;

    private final List<AsnAssociation> asnAssociations;

    private final String statusMessage;

    private final String state;

    private final String networkBorderGroup;

    private ByoipCidr(BuilderImpl builder) {
        this.cidr = builder.cidr;
        this.description = builder.description;
        this.asnAssociations = builder.asnAssociations;
        this.statusMessage = builder.statusMessage;
        this.state = builder.state;
        this.networkBorderGroup = builder.networkBorderGroup;
    }

    /**
     * <p>
     * The address range, in CIDR notation.
     * </p>
     * 
     * @return The address range, in CIDR notation.
     */
    public final String cidr() {
        return cidr;
    }

    /**
     * <p>
     * The description of the address range.
     * </p>
     * 
     * @return The description of the address range.
     */
    public final String description() {
        return description;
    }

    /**
     * For responses, this returns true if the service returned a value for the AsnAssociations property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAsnAssociations() {
        return asnAssociations != null && !(asnAssociations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The BYOIP CIDR associations with ASNs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAsnAssociations} method.
     * </p>
     * 
     * @return The BYOIP CIDR associations with ASNs.
     */
    public final List<AsnAssociation> asnAssociations() {
        return asnAssociations;
    }

    /**
     * <p>
     * Upon success, contains the ID of the address pool. Otherwise, contains an error message.
     * </p>
     * 
     * @return Upon success, contains the ID of the address pool. Otherwise, contains an error message.
     */
    public final String statusMessage() {
        return statusMessage;
    }

    /**
     * <p>
     * The state of the address range.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deprovisioned</code>: The address range is deprovisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure that all
     * EIPs from the range have been deallocated and try again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed-provision</code>: The request to provision the address range was unsuccessful.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not currently
     * advertised.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be advertised.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ByoipCidrState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the address range.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deprovisioned</code>: The address range is deprovisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure
     *         that all EIPs from the range have been deallocated and try again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed-provision</code>: The request to provision the address range was unsuccessful.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
     *         pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's
     *         pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not
     *         currently advertised.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
     *         advertised.
     *         </p>
     *         </li>
     * @see ByoipCidrState
     */
    public final ByoipCidrState state() {
        return ByoipCidrState.fromValue(state);
    }

    /**
     * <p>
     * The state of the address range.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>deprovisioned</code>: The address range is deprovisioned.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure that all
     * EIPs from the range have been deallocated and try again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>failed-provision</code>: The request to provision the address range was unsuccessful.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's pending.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not currently
     * advertised.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be advertised.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link ByoipCidrState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the address range.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>deprovisioned</code>: The address range is deprovisioned.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure
     *         that all EIPs from the range have been deallocated and try again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>failed-provision</code>: The request to provision the address range was unsuccessful.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
     *         pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's
     *         pending.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not
     *         currently advertised.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
     *         advertised.
     *         </p>
     *         </li>
     * @see ByoipCidrState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * If you have <a href="https://docs.aws.amazon.com/local-zones/latest/ug/how-local-zones-work.html">Local Zones</a>
     * enabled, you can choose a network border group for Local Zones when you provision and advertise a BYOIPv4 CIDR.
     * Choose the network border group carefully as the EIP and the Amazon Web Services resource it is associated with
     * must reside in the same network border group.
     * </p>
     * <p>
     * You can provision BYOIP address ranges to and advertise them in the following Local Zone network border groups:
     * </p>
     * <ul>
     * <li>
     * <p>
     * us-east-1-dfw-2
     * </p>
     * </li>
     * <li>
     * <p>
     * us-west-2-lax-1
     * </p>
     * </li>
     * <li>
     * <p>
     * us-west-2-phx-2
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * You cannot provision or advertise BYOIPv6 address ranges in Local Zones at this time.
     * </p>
     * </note>
     * 
     * @return If you have <a href="https://docs.aws.amazon.com/local-zones/latest/ug/how-local-zones-work.html">Local
     *         Zones</a> enabled, you can choose a network border group for Local Zones when you provision and advertise
     *         a BYOIPv4 CIDR. Choose the network border group carefully as the EIP and the Amazon Web Services resource
     *         it is associated with must reside in the same network border group.</p>
     *         <p>
     *         You can provision BYOIP address ranges to and advertise them in the following Local Zone network border
     *         groups:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         us-east-1-dfw-2
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         us-west-2-lax-1
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         us-west-2-phx-2
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         You cannot provision or advertise BYOIPv6 address ranges in Local Zones at this time.
     *         </p>
     */
    public final String networkBorderGroup() {
        return networkBorderGroup;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cidr());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(hasAsnAssociations() ? asnAssociations() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusMessage());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(networkBorderGroup());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ByoipCidr)) {
            return false;
        }
        ByoipCidr other = (ByoipCidr) obj;
        return Objects.equals(cidr(), other.cidr()) && Objects.equals(description(), other.description())
                && hasAsnAssociations() == other.hasAsnAssociations()
                && Objects.equals(asnAssociations(), other.asnAssociations())
                && Objects.equals(statusMessage(), other.statusMessage())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(networkBorderGroup(), other.networkBorderGroup());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ByoipCidr").add("Cidr", cidr()).add("Description", description())
                .add("AsnAssociations", hasAsnAssociations() ? asnAssociations() : null).add("StatusMessage", statusMessage())
                .add("State", stateAsString()).add("NetworkBorderGroup", networkBorderGroup()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Cidr":
            return Optional.ofNullable(clazz.cast(cidr()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AsnAssociations":
            return Optional.ofNullable(clazz.cast(asnAssociations()));
        case "StatusMessage":
            return Optional.ofNullable(clazz.cast(statusMessage()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "NetworkBorderGroup":
            return Optional.ofNullable(clazz.cast(networkBorderGroup()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ByoipCidr, T> g) {
        return obj -> g.apply((ByoipCidr) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ByoipCidr> {
        /**
         * <p>
         * The address range, in CIDR notation.
         * </p>
         * 
         * @param cidr
         *        The address range, in CIDR notation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidr(String cidr);

        /**
         * <p>
         * The description of the address range.
         * </p>
         * 
         * @param description
         *        The description of the address range.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The BYOIP CIDR associations with ASNs.
         * </p>
         * 
         * @param asnAssociations
         *        The BYOIP CIDR associations with ASNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asnAssociations(Collection<AsnAssociation> asnAssociations);

        /**
         * <p>
         * The BYOIP CIDR associations with ASNs.
         * </p>
         * 
         * @param asnAssociations
         *        The BYOIP CIDR associations with ASNs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder asnAssociations(AsnAssociation... asnAssociations);

        /**
         * <p>
         * The BYOIP CIDR associations with ASNs.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.AsnAssociation.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.AsnAssociation#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.AsnAssociation.Builder#build()} is called immediately and
         * its result is passed to {@link #asnAssociations(List<AsnAssociation>)}.
         * 
         * @param asnAssociations
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.AsnAssociation.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #asnAssociations(java.util.Collection<AsnAssociation>)
         */
        Builder asnAssociations(Consumer<AsnAssociation.Builder>... asnAssociations);

        /**
         * <p>
         * Upon success, contains the ID of the address pool. Otherwise, contains an error message.
         * </p>
         * 
         * @param statusMessage
         *        Upon success, contains the ID of the address pool. Otherwise, contains an error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusMessage(String statusMessage);

        /**
         * <p>
         * The state of the address range.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deprovisioned</code>: The address range is deprovisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure that
         * all EIPs from the range have been deallocated and try again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed-provision</code>: The request to provision the address range was unsuccessful.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
         * pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not currently
         * advertised.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
         * advertised.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the address range.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deprovisioned</code>: The address range is deprovisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure
         *        that all EIPs from the range have been deallocated and try again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed-provision</code>: The request to provision the address range was unsuccessful.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
         *        pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's
         *        pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not
         *        currently advertised.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
         *        advertised.
         *        </p>
         *        </li>
         * @see ByoipCidrState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ByoipCidrState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the address range.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>deprovisioned</code>: The address range is deprovisioned.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure that
         * all EIPs from the range have been deallocated and try again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>failed-provision</code>: The request to provision the address range was unsuccessful.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
         * pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's pending.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not currently
         * advertised.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
         * advertised.
         * </p>
         * </li>
         * </ul>
         * 
         * @param state
         *        The state of the address range.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>advertised</code>: The address range is being advertised to the internet by Amazon Web Services.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>deprovisioned</code>: The address range is deprovisioned.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed-deprovision</code>: The request to deprovision the address range was unsuccessful. Ensure
         *        that all EIPs from the range have been deallocated and try again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>failed-provision</code>: The request to provision the address range was unsuccessful.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending-deprovision</code>: You’ve submitted a request to deprovision an address range and it's
         *        pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>pending-provision</code>: You’ve submitted a request to provision an address range and it's
         *        pending.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned</code>: The address range is provisioned and can be advertised. The range is not
         *        currently advertised.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>provisioned-not-publicly-advertisable</code>: The address range is provisioned and cannot be
         *        advertised.
         *        </p>
         *        </li>
         * @see ByoipCidrState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ByoipCidrState
         */
        Builder state(ByoipCidrState state);

        /**
         * <p>
         * If you have <a href="https://docs.aws.amazon.com/local-zones/latest/ug/how-local-zones-work.html">Local
         * Zones</a> enabled, you can choose a network border group for Local Zones when you provision and advertise a
         * BYOIPv4 CIDR. Choose the network border group carefully as the EIP and the Amazon Web Services resource it is
         * associated with must reside in the same network border group.
         * </p>
         * <p>
         * You can provision BYOIP address ranges to and advertise them in the following Local Zone network border
         * groups:
         * </p>
         * <ul>
         * <li>
         * <p>
         * us-east-1-dfw-2
         * </p>
         * </li>
         * <li>
         * <p>
         * us-west-2-lax-1
         * </p>
         * </li>
         * <li>
         * <p>
         * us-west-2-phx-2
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * You cannot provision or advertise BYOIPv6 address ranges in Local Zones at this time.
         * </p>
         * </note>
         * 
         * @param networkBorderGroup
         *        If you have <a
         *        href="https://docs.aws.amazon.com/local-zones/latest/ug/how-local-zones-work.html">Local Zones</a>
         *        enabled, you can choose a network border group for Local Zones when you provision and advertise a
         *        BYOIPv4 CIDR. Choose the network border group carefully as the EIP and the Amazon Web Services
         *        resource it is associated with must reside in the same network border group.</p>
         *        <p>
         *        You can provision BYOIP address ranges to and advertise them in the following Local Zone network
         *        border groups:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        us-east-1-dfw-2
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        us-west-2-lax-1
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        us-west-2-phx-2
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        You cannot provision or advertise BYOIPv6 address ranges in Local Zones at this time.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBorderGroup(String networkBorderGroup);
    }

    static final class BuilderImpl implements Builder {
        private String cidr;

        private String description;

        private List<AsnAssociation> asnAssociations = DefaultSdkAutoConstructList.getInstance();

        private String statusMessage;

        private String state;

        private String networkBorderGroup;

        private BuilderImpl() {
        }

        private BuilderImpl(ByoipCidr model) {
            cidr(model.cidr);
            description(model.description);
            asnAssociations(model.asnAssociations);
            statusMessage(model.statusMessage);
            state(model.state);
            networkBorderGroup(model.networkBorderGroup);
        }

        public final String getCidr() {
            return cidr;
        }

        public final void setCidr(String cidr) {
            this.cidr = cidr;
        }

        @Override
        public final Builder cidr(String cidr) {
            this.cidr = cidr;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final List<AsnAssociation.Builder> getAsnAssociations() {
            List<AsnAssociation.Builder> result = AsnAssociationSetCopier.copyToBuilder(this.asnAssociations);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setAsnAssociations(Collection<AsnAssociation.BuilderImpl> asnAssociations) {
            this.asnAssociations = AsnAssociationSetCopier.copyFromBuilder(asnAssociations);
        }

        @Override
        public final Builder asnAssociations(Collection<AsnAssociation> asnAssociations) {
            this.asnAssociations = AsnAssociationSetCopier.copy(asnAssociations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder asnAssociations(AsnAssociation... asnAssociations) {
            asnAssociations(Arrays.asList(asnAssociations));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder asnAssociations(Consumer<AsnAssociation.Builder>... asnAssociations) {
            asnAssociations(Stream.of(asnAssociations).map(c -> AsnAssociation.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getStatusMessage() {
            return statusMessage;
        }

        public final void setStatusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
        }

        @Override
        public final Builder statusMessage(String statusMessage) {
            this.statusMessage = statusMessage;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(ByoipCidrState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getNetworkBorderGroup() {
            return networkBorderGroup;
        }

        public final void setNetworkBorderGroup(String networkBorderGroup) {
            this.networkBorderGroup = networkBorderGroup;
        }

        @Override
        public final Builder networkBorderGroup(String networkBorderGroup) {
            this.networkBorderGroup = networkBorderGroup;
            return this;
        }

        @Override
        public ByoipCidr build() {
            return new ByoipCidr(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
