/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The attributes for the instance types. When you specify instance attributes, Amazon EC2 will identify instance types
 * with these attributes.
 * </p>
 * <p>
 * You must specify <code>VCpuCount</code> and <code>MemoryMiB</code>. All other attributes are optional. Any
 * unspecified optional attribute is set to its default.
 * </p>
 * <p>
 * When you specify multiple attributes, you get instance types that satisfy all of the specified attributes. If you
 * specify multiple values for an attribute, you get instance types that satisfy any of the specified values.
 * </p>
 * <p>
 * To limit the list of instance types from which Amazon EC2 can identify matching instance types, you can use one of
 * the following parameters, but not both in the same request:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>AllowedInstanceTypes</code> - The instance types to include in the list. All other instance types are ignored,
 * even if they match your specified attributes.
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ExcludedInstanceTypes</code> - The instance types to exclude from the list, even if they match your specified
 * attributes.
 * </p>
 * </li>
 * </ul>
 * <note>
 * <p>
 * If you specify <code>InstanceRequirements</code>, you can't specify <code>InstanceType</code>.
 * </p>
 * <p>
 * Attribute-based instance type selection is only supported when using Auto Scaling groups, EC2 Fleet, and Spot Fleet
 * to launch instances. If you plan to use the launch template in the <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-instance-wizard.html">launch instance
 * wizard</a>, or with the <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_RunInstances.html">RunInstances</a> API or <a
 * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-ec2-instance.html"
 * >AWS::EC2::Instance</a> Amazon Web Services CloudFormation resource, you can't specify
 * <code>InstanceRequirements</code>.
 * </p>
 * </note>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-fleet-attribute-based-instance-type-selection.html"
 * >Attribute-based instance type selection for EC2 Fleet</a>, <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-fleet-attribute-based-instance-type-selection.html"
 * >Attribute-based instance type selection for Spot Fleet</a>, and <a
 * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-placement-score.html">Spot placement score</a> in the
 * <i>Amazon EC2 User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRequirementsRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRequirementsRequest.Builder, InstanceRequirementsRequest> {
    private static final SdkField<VCpuCountRangeRequest> V_CPU_COUNT_FIELD = SdkField
            .<VCpuCountRangeRequest> builder(MarshallingType.SDK_POJO)
            .memberName("VCpuCount")
            .getter(getter(InstanceRequirementsRequest::vCpuCount))
            .setter(setter(Builder::vCpuCount))
            .constructor(VCpuCountRangeRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VCpuCount")
                    .unmarshallLocationName("VCpuCount").build()).build();

    private static final SdkField<MemoryMiBRequest> MEMORY_MIB_FIELD = SdkField
            .<MemoryMiBRequest> builder(MarshallingType.SDK_POJO)
            .memberName("MemoryMiB")
            .getter(getter(InstanceRequirementsRequest::memoryMiB))
            .setter(setter(Builder::memoryMiB))
            .constructor(MemoryMiBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryMiB")
                    .unmarshallLocationName("MemoryMiB").build()).build();

    private static final SdkField<List<String>> CPU_MANUFACTURERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("CpuManufacturers")
            .getter(getter(InstanceRequirementsRequest::cpuManufacturersAsStrings))
            .setter(setter(Builder::cpuManufacturersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CpuManufacturer")
                    .unmarshallLocationName("CpuManufacturer").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<MemoryGiBPerVCpuRequest> MEMORY_GI_B_PER_V_CPU_FIELD = SdkField
            .<MemoryGiBPerVCpuRequest> builder(MarshallingType.SDK_POJO)
            .memberName("MemoryGiBPerVCpu")
            .getter(getter(InstanceRequirementsRequest::memoryGiBPerVCpu))
            .setter(setter(Builder::memoryGiBPerVCpu))
            .constructor(MemoryGiBPerVCpuRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MemoryGiBPerVCpu")
                    .unmarshallLocationName("MemoryGiBPerVCpu").build()).build();

    private static final SdkField<List<String>> EXCLUDED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExcludedInstanceTypes")
            .getter(getter(InstanceRequirementsRequest::excludedInstanceTypes))
            .setter(setter(Builder::excludedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedInstanceType")
                    .unmarshallLocationName("ExcludedInstanceType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> INSTANCE_GENERATIONS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("InstanceGenerations")
            .getter(getter(InstanceRequirementsRequest::instanceGenerationsAsStrings))
            .setter(setter(Builder::instanceGenerationsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceGeneration")
                    .unmarshallLocationName("InstanceGeneration").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> SPOT_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("SpotMaxPricePercentageOverLowestPrice")
            .getter(getter(InstanceRequirementsRequest::spotMaxPricePercentageOverLowestPrice))
            .setter(setter(Builder::spotMaxPricePercentageOverLowestPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("SpotMaxPricePercentageOverLowestPrice")
                    .unmarshallLocationName("SpotMaxPricePercentageOverLowestPrice").build()).build();

    private static final SdkField<Integer> ON_DEMAND_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("OnDemandMaxPricePercentageOverLowestPrice")
            .getter(getter(InstanceRequirementsRequest::onDemandMaxPricePercentageOverLowestPrice))
            .setter(setter(Builder::onDemandMaxPricePercentageOverLowestPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("OnDemandMaxPricePercentageOverLowestPrice")
                    .unmarshallLocationName("OnDemandMaxPricePercentageOverLowestPrice").build()).build();

    private static final SdkField<String> BARE_METAL_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("BareMetal")
            .getter(getter(InstanceRequirementsRequest::bareMetalAsString))
            .setter(setter(Builder::bareMetal))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BareMetal")
                    .unmarshallLocationName("BareMetal").build()).build();

    private static final SdkField<String> BURSTABLE_PERFORMANCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("BurstablePerformance")
            .getter(getter(InstanceRequirementsRequest::burstablePerformanceAsString))
            .setter(setter(Builder::burstablePerformance))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BurstablePerformance")
                    .unmarshallLocationName("BurstablePerformance").build()).build();

    private static final SdkField<Boolean> REQUIRE_HIBERNATE_SUPPORT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("RequireHibernateSupport")
            .getter(getter(InstanceRequirementsRequest::requireHibernateSupport))
            .setter(setter(Builder::requireHibernateSupport))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RequireHibernateSupport")
                    .unmarshallLocationName("RequireHibernateSupport").build()).build();

    private static final SdkField<NetworkInterfaceCountRequest> NETWORK_INTERFACE_COUNT_FIELD = SdkField
            .<NetworkInterfaceCountRequest> builder(MarshallingType.SDK_POJO)
            .memberName("NetworkInterfaceCount")
            .getter(getter(InstanceRequirementsRequest::networkInterfaceCount))
            .setter(setter(Builder::networkInterfaceCount))
            .constructor(NetworkInterfaceCountRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkInterfaceCount")
                    .unmarshallLocationName("NetworkInterfaceCount").build()).build();

    private static final SdkField<String> LOCAL_STORAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LocalStorage")
            .getter(getter(InstanceRequirementsRequest::localStorageAsString))
            .setter(setter(Builder::localStorage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalStorage")
                    .unmarshallLocationName("LocalStorage").build()).build();

    private static final SdkField<List<String>> LOCAL_STORAGE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("LocalStorageTypes")
            .getter(getter(InstanceRequirementsRequest::localStorageTypesAsStrings))
            .setter(setter(Builder::localStorageTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LocalStorageType")
                    .unmarshallLocationName("LocalStorageType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<TotalLocalStorageGBRequest> TOTAL_LOCAL_STORAGE_GB_FIELD = SdkField
            .<TotalLocalStorageGBRequest> builder(MarshallingType.SDK_POJO)
            .memberName("TotalLocalStorageGB")
            .getter(getter(InstanceRequirementsRequest::totalLocalStorageGB))
            .setter(setter(Builder::totalLocalStorageGB))
            .constructor(TotalLocalStorageGBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TotalLocalStorageGB")
                    .unmarshallLocationName("TotalLocalStorageGB").build()).build();

    private static final SdkField<BaselineEbsBandwidthMbpsRequest> BASELINE_EBS_BANDWIDTH_MBPS_FIELD = SdkField
            .<BaselineEbsBandwidthMbpsRequest> builder(MarshallingType.SDK_POJO)
            .memberName("BaselineEbsBandwidthMbps")
            .getter(getter(InstanceRequirementsRequest::baselineEbsBandwidthMbps))
            .setter(setter(Builder::baselineEbsBandwidthMbps))
            .constructor(BaselineEbsBandwidthMbpsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BaselineEbsBandwidthMbps")
                    .unmarshallLocationName("BaselineEbsBandwidthMbps").build()).build();

    private static final SdkField<List<String>> ACCELERATOR_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AcceleratorTypes")
            .getter(getter(InstanceRequirementsRequest::acceleratorTypesAsStrings))
            .setter(setter(Builder::acceleratorTypesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorType")
                    .unmarshallLocationName("AcceleratorType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<AcceleratorCountRequest> ACCELERATOR_COUNT_FIELD = SdkField
            .<AcceleratorCountRequest> builder(MarshallingType.SDK_POJO)
            .memberName("AcceleratorCount")
            .getter(getter(InstanceRequirementsRequest::acceleratorCount))
            .setter(setter(Builder::acceleratorCount))
            .constructor(AcceleratorCountRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorCount")
                    .unmarshallLocationName("AcceleratorCount").build()).build();

    private static final SdkField<List<String>> ACCELERATOR_MANUFACTURERS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AcceleratorManufacturers")
            .getter(getter(InstanceRequirementsRequest::acceleratorManufacturersAsStrings))
            .setter(setter(Builder::acceleratorManufacturersWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorManufacturer")
                    .unmarshallLocationName("AcceleratorManufacturer").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<List<String>> ACCELERATOR_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AcceleratorNames")
            .getter(getter(InstanceRequirementsRequest::acceleratorNamesAsStrings))
            .setter(setter(Builder::acceleratorNamesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorName")
                    .unmarshallLocationName("AcceleratorName").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<AcceleratorTotalMemoryMiBRequest> ACCELERATOR_TOTAL_MEMORY_MIB_FIELD = SdkField
            .<AcceleratorTotalMemoryMiBRequest> builder(MarshallingType.SDK_POJO)
            .memberName("AcceleratorTotalMemoryMiB")
            .getter(getter(InstanceRequirementsRequest::acceleratorTotalMemoryMiB))
            .setter(setter(Builder::acceleratorTotalMemoryMiB))
            .constructor(AcceleratorTotalMemoryMiBRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorTotalMemoryMiB")
                    .unmarshallLocationName("AcceleratorTotalMemoryMiB").build()).build();

    private static final SdkField<NetworkBandwidthGbpsRequest> NETWORK_BANDWIDTH_GBPS_FIELD = SdkField
            .<NetworkBandwidthGbpsRequest> builder(MarshallingType.SDK_POJO)
            .memberName("NetworkBandwidthGbps")
            .getter(getter(InstanceRequirementsRequest::networkBandwidthGbps))
            .setter(setter(Builder::networkBandwidthGbps))
            .constructor(NetworkBandwidthGbpsRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NetworkBandwidthGbps")
                    .unmarshallLocationName("NetworkBandwidthGbps").build()).build();

    private static final SdkField<List<String>> ALLOWED_INSTANCE_TYPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("AllowedInstanceTypes")
            .getter(getter(InstanceRequirementsRequest::allowedInstanceTypes))
            .setter(setter(Builder::allowedInstanceTypes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AllowedInstanceType")
                    .unmarshallLocationName("AllowedInstanceType").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Integer> MAX_SPOT_PRICE_AS_PERCENTAGE_OF_OPTIMAL_ON_DEMAND_PRICE_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxSpotPriceAsPercentageOfOptimalOnDemandPrice")
            .getter(getter(InstanceRequirementsRequest::maxSpotPriceAsPercentageOfOptimalOnDemandPrice))
            .setter(setter(Builder::maxSpotPriceAsPercentageOfOptimalOnDemandPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                    .locationName("MaxSpotPriceAsPercentageOfOptimalOnDemandPrice")
                    .unmarshallLocationName("MaxSpotPriceAsPercentageOfOptimalOnDemandPrice").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(V_CPU_COUNT_FIELD,
            MEMORY_MIB_FIELD, CPU_MANUFACTURERS_FIELD, MEMORY_GI_B_PER_V_CPU_FIELD, EXCLUDED_INSTANCE_TYPES_FIELD,
            INSTANCE_GENERATIONS_FIELD, SPOT_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD,
            ON_DEMAND_MAX_PRICE_PERCENTAGE_OVER_LOWEST_PRICE_FIELD, BARE_METAL_FIELD, BURSTABLE_PERFORMANCE_FIELD,
            REQUIRE_HIBERNATE_SUPPORT_FIELD, NETWORK_INTERFACE_COUNT_FIELD, LOCAL_STORAGE_FIELD, LOCAL_STORAGE_TYPES_FIELD,
            TOTAL_LOCAL_STORAGE_GB_FIELD, BASELINE_EBS_BANDWIDTH_MBPS_FIELD, ACCELERATOR_TYPES_FIELD, ACCELERATOR_COUNT_FIELD,
            ACCELERATOR_MANUFACTURERS_FIELD, ACCELERATOR_NAMES_FIELD, ACCELERATOR_TOTAL_MEMORY_MIB_FIELD,
            NETWORK_BANDWIDTH_GBPS_FIELD, ALLOWED_INSTANCE_TYPES_FIELD,
            MAX_SPOT_PRICE_AS_PERCENTAGE_OF_OPTIMAL_ON_DEMAND_PRICE_FIELD));

    private static final long serialVersionUID = 1L;

    private final VCpuCountRangeRequest vCpuCount;

    private final MemoryMiBRequest memoryMiB;

    private final List<String> cpuManufacturers;

    private final MemoryGiBPerVCpuRequest memoryGiBPerVCpu;

    private final List<String> excludedInstanceTypes;

    private final List<String> instanceGenerations;

    private final Integer spotMaxPricePercentageOverLowestPrice;

    private final Integer onDemandMaxPricePercentageOverLowestPrice;

    private final String bareMetal;

    private final String burstablePerformance;

    private final Boolean requireHibernateSupport;

    private final NetworkInterfaceCountRequest networkInterfaceCount;

    private final String localStorage;

    private final List<String> localStorageTypes;

    private final TotalLocalStorageGBRequest totalLocalStorageGB;

    private final BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps;

    private final List<String> acceleratorTypes;

    private final AcceleratorCountRequest acceleratorCount;

    private final List<String> acceleratorManufacturers;

    private final List<String> acceleratorNames;

    private final AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB;

    private final NetworkBandwidthGbpsRequest networkBandwidthGbps;

    private final List<String> allowedInstanceTypes;

    private final Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice;

    private InstanceRequirementsRequest(BuilderImpl builder) {
        this.vCpuCount = builder.vCpuCount;
        this.memoryMiB = builder.memoryMiB;
        this.cpuManufacturers = builder.cpuManufacturers;
        this.memoryGiBPerVCpu = builder.memoryGiBPerVCpu;
        this.excludedInstanceTypes = builder.excludedInstanceTypes;
        this.instanceGenerations = builder.instanceGenerations;
        this.spotMaxPricePercentageOverLowestPrice = builder.spotMaxPricePercentageOverLowestPrice;
        this.onDemandMaxPricePercentageOverLowestPrice = builder.onDemandMaxPricePercentageOverLowestPrice;
        this.bareMetal = builder.bareMetal;
        this.burstablePerformance = builder.burstablePerformance;
        this.requireHibernateSupport = builder.requireHibernateSupport;
        this.networkInterfaceCount = builder.networkInterfaceCount;
        this.localStorage = builder.localStorage;
        this.localStorageTypes = builder.localStorageTypes;
        this.totalLocalStorageGB = builder.totalLocalStorageGB;
        this.baselineEbsBandwidthMbps = builder.baselineEbsBandwidthMbps;
        this.acceleratorTypes = builder.acceleratorTypes;
        this.acceleratorCount = builder.acceleratorCount;
        this.acceleratorManufacturers = builder.acceleratorManufacturers;
        this.acceleratorNames = builder.acceleratorNames;
        this.acceleratorTotalMemoryMiB = builder.acceleratorTotalMemoryMiB;
        this.networkBandwidthGbps = builder.networkBandwidthGbps;
        this.allowedInstanceTypes = builder.allowedInstanceTypes;
        this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = builder.maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
    }

    /**
     * <p>
     * The minimum and maximum number of vCPUs.
     * </p>
     * 
     * @return The minimum and maximum number of vCPUs.
     */
    public final VCpuCountRangeRequest vCpuCount() {
        return vCpuCount;
    }

    /**
     * <p>
     * The minimum and maximum amount of memory, in MiB.
     * </p>
     * 
     * @return The minimum and maximum amount of memory, in MiB.
     */
    public final MemoryMiBRequest memoryMiB() {
        return memoryMiB;
    }

    /**
     * <p>
     * The CPU manufacturers to include.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with Intel CPUs, specify <code>intel</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD CPUs, specify <code>amd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU
     * architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
     * </p>
     * </note>
     * <p>
     * Default: Any manufacturer
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuManufacturers} method.
     * </p>
     * 
     * @return The CPU manufacturers to include.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with Intel CPUs, specify <code>intel</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD CPUs, specify <code>amd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
     *         compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
     *         template.
     *         </p>
     *         </note>
     *         <p>
     *         Default: Any manufacturer
     */
    public final List<CpuManufacturer> cpuManufacturers() {
        return CpuManufacturerSetCopier.copyStringToEnum(cpuManufacturers);
    }

    /**
     * For responses, this returns true if the service returned a value for the CpuManufacturers property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasCpuManufacturers() {
        return cpuManufacturers != null && !(cpuManufacturers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The CPU manufacturers to include.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with Intel CPUs, specify <code>intel</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD CPUs, specify <code>amd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible CPU
     * architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
     * </p>
     * </note>
     * <p>
     * Default: Any manufacturer
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasCpuManufacturers} method.
     * </p>
     * 
     * @return The CPU manufacturers to include.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with Intel CPUs, specify <code>intel</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD CPUs, specify <code>amd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
     *         compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
     *         template.
     *         </p>
     *         </note>
     *         <p>
     *         Default: Any manufacturer
     */
    public final List<String> cpuManufacturersAsStrings() {
        return cpuManufacturers;
    }

    /**
     * <p>
     * The minimum and maximum amount of memory per vCPU, in GiB.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum amount of memory per vCPU, in GiB.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final MemoryGiBPerVCpuRequest memoryGiBPerVCpu() {
        return memoryGiBPerVCpu;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedInstanceTypes property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasExcludedInstanceTypes() {
        return excludedInstanceTypes != null && !(excludedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types to exclude.
     * </p>
     * <p>
     * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to exclude an
     * instance family, type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
     * <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
     * </p>
     * <p>
     * For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family, which
     * includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will exclude all the M5a
     * instance types, but not the M5n instance types.
     * </p>
     * <note>
     * <p>
     * If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>.
     * </p>
     * </note>
     * <p>
     * Default: No excluded instance types
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedInstanceTypes} method.
     * </p>
     * 
     * @return The instance types to exclude.</p>
     *         <p>
     *         You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to exclude
     *         an instance family, type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
     *         <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
     *         </p>
     *         <p>
     *         For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family, which
     *         includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will exclude all
     *         the M5a instance types, but not the M5n instance types.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Default: No excluded instance types
     */
    public final List<String> excludedInstanceTypes() {
        return excludedInstanceTypes;
    }

    /**
     * <p>
     * Indicates whether current or previous generation instance types are included. The current generation instance
     * types are recommended for use. Current generation instance types are typically the latest two to three
     * generations in each instance family. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
     * <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * For current generation instance types, specify <code>current</code>.
     * </p>
     * <p>
     * For previous generation instance types, specify <code>previous</code>.
     * </p>
     * <p>
     * Default: Current and previous generation instance types
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGenerations} method.
     * </p>
     * 
     * @return Indicates whether current or previous generation instance types are included. The current generation
     *         instance types are recommended for use. Current generation instance types are typically the latest two to
     *         three generations in each instance family. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
     *         <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         For current generation instance types, specify <code>current</code>.
     *         </p>
     *         <p>
     *         For previous generation instance types, specify <code>previous</code>.
     *         </p>
     *         <p>
     *         Default: Current and previous generation instance types
     */
    public final List<InstanceGeneration> instanceGenerations() {
        return InstanceGenerationSetCopier.copyStringToEnum(instanceGenerations);
    }

    /**
     * For responses, this returns true if the service returned a value for the InstanceGenerations property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasInstanceGenerations() {
        return instanceGenerations != null && !(instanceGenerations instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates whether current or previous generation instance types are included. The current generation instance
     * types are recommended for use. Current generation instance types are typically the latest two to three
     * generations in each instance family. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
     * <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * For current generation instance types, specify <code>current</code>.
     * </p>
     * <p>
     * For previous generation instance types, specify <code>previous</code>.
     * </p>
     * <p>
     * Default: Current and previous generation instance types
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstanceGenerations} method.
     * </p>
     * 
     * @return Indicates whether current or previous generation instance types are included. The current generation
     *         instance types are recommended for use. Current generation instance types are typically the latest two to
     *         three generations in each instance family. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
     *         <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         For current generation instance types, specify <code>current</code>.
     *         </p>
     *         <p>
     *         For previous generation instance types, specify <code>previous</code>.
     *         </p>
     *         <p>
     *         Default: Current and previous generation instance types
     */
    public final List<String> instanceGenerationsAsStrings() {
        return instanceGenerations;
    }

    /**
     * <p>
     * [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an identified
     * Spot price. The identified Spot price is the Spot price of the lowest priced current generation C, M, or R
     * instance type with your specified attributes. If no current generation C, M, or R instance type matches your
     * attributes, then the identified Spot price is from the lowest priced current generation instance types, and
     * failing that, from the lowest priced previous generation instance types that match your attributes. When Amazon
     * EC2 selects instance types with your attributes, it will exclude instance types whose Spot price exceeds your
     * specified threshold.
     * </p>
     * <p>
     * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     * </p>
     * <p>
     * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     * protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.
     * </p>
     * <p>
     * This parameter is not supported for <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     * >GetSpotPlacementScores</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
     * >GetInstanceTypesFromInstanceRequirements</a>.
     * </p>
     * <note>
     * <p>
     * Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
     * <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify either, Amazon
     * EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types.
     * To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types
     * that match your attributes, include one of these parameters and specify a high value, such as <code>999999</code>
     * .
     * </p>
     * </note>
     * <p>
     * Default: <code>100</code>
     * </p>
     * 
     * @return [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an
     *         identified Spot price. The identified Spot price is the Spot price of the lowest priced current
     *         generation C, M, or R instance type with your specified attributes. If no current generation C, M, or R
     *         instance type matches your attributes, then the identified Spot price is from the lowest priced current
     *         generation instance types, and failing that, from the lowest priced previous generation instance types
     *         that match your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude
     *         instance types whose Spot price exceeds your specified threshold.</p>
     *         <p>
     *         The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     *         </p>
     *         <p>
     *         If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     *         protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance
     *         price.
     *         </p>
     *         <p>
     *         This parameter is not supported for <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     *         >GetSpotPlacementScores</a> and <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
     *         >GetInstanceTypesFromInstanceRequirements</a>.
     *         </p>
     *         <note>
     *         <p>
     *         Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
     *         <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify
     *         either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide
     *         range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want
     *         to consider all instance types that match your attributes, include one of these parameters and specify a
     *         high value, such as <code>999999</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Default: <code>100</code>
     */
    public final Integer spotMaxPricePercentageOverLowestPrice() {
        return spotMaxPricePercentageOverLowestPrice;
    }

    /**
     * <p>
     * [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an
     * identified On-Demand price. The identified On-Demand price is the price of the lowest priced current generation
     * C, M, or R instance type with your specified attributes. When Amazon EC2 selects instance types with your
     * attributes, it will exclude instance types whose price exceeds your specified threshold.
     * </p>
     * <p>
     * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     * </p>
     * <p>
     * To indicate no price protection threshold, specify a high value, such as <code>999999</code>.
     * </p>
     * <p>
     * This parameter is not supported for <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     * >GetSpotPlacementScores</a> and <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
     * >GetInstanceTypesFromInstanceRequirements</a>.
     * </p>
     * <note>
     * <p>
     * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     * protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.
     * </p>
     * </note>
     * <p>
     * Default: <code>20</code>
     * </p>
     * 
     * @return [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an
     *         identified On-Demand price. The identified On-Demand price is the price of the lowest priced current
     *         generation C, M, or R instance type with your specified attributes. When Amazon EC2 selects instance
     *         types with your attributes, it will exclude instance types whose price exceeds your specified
     *         threshold.</p>
     *         <p>
     *         The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     *         </p>
     *         <p>
     *         To indicate no price protection threshold, specify a high value, such as <code>999999</code>.
     *         </p>
     *         <p>
     *         This parameter is not supported for <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     *         >GetSpotPlacementScores</a> and <a href=
     *         "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
     *         >GetInstanceTypesFromInstanceRequirements</a>.
     *         </p>
     *         <note>
     *         <p>
     *         If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     *         protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance
     *         price.
     *         </p>
     *         </note>
     *         <p>
     *         Default: <code>20</code>
     */
    public final Integer onDemandMaxPricePercentageOverLowestPrice() {
        return onDemandMaxPricePercentageOverLowestPrice;
    }

    /**
     * <p>
     * Indicates whether bare metal instance types must be included, excluded, or required.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include bare metal instance types, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only bare metal instance types, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude bare metal instance types, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>excluded</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bareMetal} will
     * return {@link BareMetal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bareMetalAsString}.
     * </p>
     * 
     * @return Indicates whether bare metal instance types must be included, excluded, or required.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include bare metal instance types, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only bare metal instance types, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude bare metal instance types, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>excluded</code>
     * @see BareMetal
     */
    public final BareMetal bareMetal() {
        return BareMetal.fromValue(bareMetal);
    }

    /**
     * <p>
     * Indicates whether bare metal instance types must be included, excluded, or required.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include bare metal instance types, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only bare metal instance types, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude bare metal instance types, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>excluded</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #bareMetal} will
     * return {@link BareMetal#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #bareMetalAsString}.
     * </p>
     * 
     * @return Indicates whether bare metal instance types must be included, excluded, or required.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include bare metal instance types, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only bare metal instance types, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude bare metal instance types, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>excluded</code>
     * @see BareMetal
     */
    public final String bareMetalAsString() {
        return bareMetal;
    }

    /**
     * <p>
     * Indicates whether burstable performance T instance types are included, excluded, or required. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
     * performance instances</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include burstable performance instance types, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only burstable performance instance types, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude burstable performance instance types, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>excluded</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #burstablePerformance} will return {@link BurstablePerformance#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #burstablePerformanceAsString}.
     * </p>
     * 
     * @return Indicates whether burstable performance T instance types are included, excluded, or required. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
     *         performance instances</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include burstable performance instance types, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only burstable performance instance types, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude burstable performance instance types, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>excluded</code>
     * @see BurstablePerformance
     */
    public final BurstablePerformance burstablePerformance() {
        return BurstablePerformance.fromValue(burstablePerformance);
    }

    /**
     * <p>
     * Indicates whether burstable performance T instance types are included, excluded, or required. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
     * performance instances</a>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include burstable performance instance types, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only burstable performance instance types, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude burstable performance instance types, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>excluded</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #burstablePerformance} will return {@link BurstablePerformance#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #burstablePerformanceAsString}.
     * </p>
     * 
     * @return Indicates whether burstable performance T instance types are included, excluded, or required. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
     *         performance instances</a>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include burstable performance instance types, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only burstable performance instance types, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude burstable performance instance types, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>excluded</code>
     * @see BurstablePerformance
     */
    public final String burstablePerformanceAsString() {
        return burstablePerformance;
    }

    /**
     * <p>
     * Indicates whether instance types must support hibernation for On-Demand Instances.
     * </p>
     * <p>
     * This parameter is not supported for <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     * >GetSpotPlacementScores</a>.
     * </p>
     * <p>
     * Default: <code>false</code>
     * </p>
     * 
     * @return Indicates whether instance types must support hibernation for On-Demand Instances.</p>
     *         <p>
     *         This parameter is not supported for <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
     *         >GetSpotPlacementScores</a>.
     *         </p>
     *         <p>
     *         Default: <code>false</code>
     */
    public final Boolean requireHibernateSupport() {
        return requireHibernateSupport;
    }

    /**
     * <p>
     * The minimum and maximum number of network interfaces.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum number of network interfaces.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final NetworkInterfaceCountRequest networkInterfaceCount() {
        return networkInterfaceCount;
    }

    /**
     * <p>
     * Indicates whether instance types with instance store volumes are included, excluded, or required. For more
     * information, <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2
     * instance store</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include instance types with instance store volumes, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only instance types with instance store volumes, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude instance types with instance store volumes, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>included</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #localStorage} will
     * return {@link LocalStorage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localStorageAsString}.
     * </p>
     * 
     * @return Indicates whether instance types with instance store volumes are included, excluded, or required. For
     *         more information, <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include instance types with instance store volumes, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only instance types with instance store volumes, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude instance types with instance store volumes, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>included</code>
     * @see LocalStorage
     */
    public final LocalStorage localStorage() {
        return LocalStorage.fromValue(localStorage);
    }

    /**
     * <p>
     * Indicates whether instance types with instance store volumes are included, excluded, or required. For more
     * information, <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2
     * instance store</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include instance types with instance store volumes, specify <code>included</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To require only instance types with instance store volumes, specify <code>required</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To exclude instance types with instance store volumes, specify <code>excluded</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>included</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #localStorage} will
     * return {@link LocalStorage#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #localStorageAsString}.
     * </p>
     * 
     * @return Indicates whether instance types with instance store volumes are included, excluded, or required. For
     *         more information, <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
     *         store</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include instance types with instance store volumes, specify <code>included</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To require only instance types with instance store volumes, specify <code>required</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To exclude instance types with instance store volumes, specify <code>excluded</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>included</code>
     * @see LocalStorage
     */
    public final String localStorageAsString() {
        return localStorage;
    }

    /**
     * <p>
     * The type of local storage that is required.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>hdd</code> and <code>ssd</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocalStorageTypes} method.
     * </p>
     * 
     * @return The type of local storage that is required.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>hdd</code> and <code>ssd</code>
     */
    public final List<LocalStorageType> localStorageTypes() {
        return LocalStorageTypeSetCopier.copyStringToEnum(localStorageTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the LocalStorageTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLocalStorageTypes() {
        return localStorageTypes != null && !(localStorageTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The type of local storage that is required.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: <code>hdd</code> and <code>ssd</code>
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLocalStorageTypes} method.
     * </p>
     * 
     * @return The type of local storage that is required.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: <code>hdd</code> and <code>ssd</code>
     */
    public final List<String> localStorageTypesAsStrings() {
        return localStorageTypes;
    }

    /**
     * <p>
     * The minimum and maximum amount of total local storage, in GB.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum amount of total local storage, in GB.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final TotalLocalStorageGBRequest totalLocalStorageGB() {
        return totalLocalStorageGB;
    }

    /**
     * <p>
     * The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html">Amazon EBS–optimized instances</a>
     * in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html">Amazon EBS–optimized
     *         instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps() {
        return baselineEbsBandwidthMbps;
    }

    /**
     * <p>
     * The accelerator types that must be on the instance type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include instance types with GPU hardware, specify <code>gpu</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To include instance types with FPGA hardware, specify <code>fpga</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To include instance types with inference hardware, specify <code>inference</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any accelerator type
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorTypes} method.
     * </p>
     * 
     * @return The accelerator types that must be on the instance type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include instance types with GPU hardware, specify <code>gpu</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To include instance types with FPGA hardware, specify <code>fpga</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To include instance types with inference hardware, specify <code>inference</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any accelerator type
     */
    public final List<AcceleratorType> acceleratorTypes() {
        return AcceleratorTypeSetCopier.copyStringToEnum(acceleratorTypes);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorTypes property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAcceleratorTypes() {
        return acceleratorTypes != null && !(acceleratorTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The accelerator types that must be on the instance type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To include instance types with GPU hardware, specify <code>gpu</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To include instance types with FPGA hardware, specify <code>fpga</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To include instance types with inference hardware, specify <code>inference</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any accelerator type
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorTypes} method.
     * </p>
     * 
     * @return The accelerator types that must be on the instance type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To include instance types with GPU hardware, specify <code>gpu</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To include instance types with FPGA hardware, specify <code>fpga</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To include instance types with inference hardware, specify <code>inference</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any accelerator type
     */
    public final List<String> acceleratorTypesAsStrings() {
        return acceleratorTypes;
    }

    /**
     * <p>
     * The minimum and maximum number of accelerators (GPUs, FPGAs, or Amazon Web Services Inferentia chips) on an
     * instance.
     * </p>
     * <p>
     * To exclude accelerator-enabled instance types, set <code>Max</code> to <code>0</code>.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum number of accelerators (GPUs, FPGAs, or Amazon Web Services Inferentia chips) on
     *         an instance.</p>
     *         <p>
     *         To exclude accelerator-enabled instance types, set <code>Max</code> to <code>0</code>.
     *         </p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final AcceleratorCountRequest acceleratorCount() {
        return acceleratorCount;
    }

    /**
     * <p>
     * Indicates whether instance types must have accelerators by specific manufacturers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD devices, specify <code>amd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Habana devices, specify <code>habana</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA devices, specify <code>nvidia</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Xilinx devices, specify <code>xilinx</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any manufacturer
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorManufacturers} method.
     * </p>
     * 
     * @return Indicates whether instance types must have accelerators by specific manufacturers.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD devices, specify <code>amd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Habana devices, specify <code>habana</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA devices, specify <code>nvidia</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Xilinx devices, specify <code>xilinx</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any manufacturer
     */
    public final List<AcceleratorManufacturer> acceleratorManufacturers() {
        return AcceleratorManufacturerSetCopier.copyStringToEnum(acceleratorManufacturers);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorManufacturers property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasAcceleratorManufacturers() {
        return acceleratorManufacturers != null && !(acceleratorManufacturers instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Indicates whether instance types must have accelerators by specific manufacturers.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD devices, specify <code>amd</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Habana devices, specify <code>habana</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA devices, specify <code>nvidia</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Xilinx devices, specify <code>xilinx</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any manufacturer
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorManufacturers} method.
     * </p>
     * 
     * @return Indicates whether instance types must have accelerators by specific manufacturers.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD devices, specify <code>amd</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Habana devices, specify <code>habana</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA devices, specify <code>nvidia</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Xilinx devices, specify <code>xilinx</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any manufacturer
     */
    public final List<String> acceleratorManufacturersAsStrings() {
        return acceleratorManufacturers;
    }

    /**
     * <p>
     * The accelerators that must be on the instance type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any accelerator
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorNames} method.
     * </p>
     * 
     * @return The accelerators that must be on the instance type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any accelerator
     */
    public final List<AcceleratorName> acceleratorNames() {
        return AcceleratorNameSetCopier.copyStringToEnum(acceleratorNames);
    }

    /**
     * For responses, this returns true if the service returned a value for the AcceleratorNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAcceleratorNames() {
        return acceleratorNames != null && !(acceleratorNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The accelerators that must be on the instance type.
     * </p>
     * <ul>
     * <li>
     * <p>
     * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Any accelerator
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAcceleratorNames} method.
     * </p>
     * 
     * @return The accelerators that must be on the instance type.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Any accelerator
     */
    public final List<String> acceleratorNamesAsStrings() {
        return acceleratorNames;
    }

    /**
     * <p>
     * The minimum and maximum amount of total accelerator memory, in MiB.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum amount of total accelerator memory, in MiB.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB() {
        return acceleratorTotalMemoryMiB;
    }

    /**
     * <p>
     * The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html">Amazon EC2
     * instance network bandwidth</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * Default: No minimum or maximum limits
     * </p>
     * 
     * @return The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html">Amazon EC2
     *         instance network bandwidth</a> in the <i>Amazon EC2 User Guide</i>.</p>
     *         <p>
     *         Default: No minimum or maximum limits
     */
    public final NetworkBandwidthGbpsRequest networkBandwidthGbps() {
        return networkBandwidthGbps;
    }

    /**
     * For responses, this returns true if the service returned a value for the AllowedInstanceTypes property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasAllowedInstanceTypes() {
        return allowedInstanceTypes != null && !(allowedInstanceTypes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The instance types to apply your specified attributes against. All other instance types are ignored, even if they
     * match your specified attributes.
     * </p>
     * <p>
     * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow an
     * instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>, <code>c5*.*</code>,
     * <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
     * </p>
     * <p>
     * For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family, which includes
     * all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will allow all the M5a instance
     * types, but not the M5n instance types.
     * </p>
     * <note>
     * <p>
     * If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>.
     * </p>
     * </note>
     * <p>
     * Default: All instance types
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasAllowedInstanceTypes} method.
     * </p>
     * 
     * @return The instance types to apply your specified attributes against. All other instance types are ignored, even
     *         if they match your specified attributes.</p>
     *         <p>
     *         You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow an
     *         instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
     *         <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
     *         </p>
     *         <p>
     *         For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family, which
     *         includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will allow all the
     *         M5a instance types, but not the M5n instance types.
     *         </p>
     *         <note>
     *         <p>
     *         If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>.
     *         </p>
     *         </note>
     *         <p>
     *         Default: All instance types
     */
    public final List<String> allowedInstanceTypes() {
        return allowedInstanceTypes;
    }

    /**
     * <p>
     * [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified On-Demand
     * price. The identified On-Demand price is the price of the lowest priced current generation C, M, or R instance
     * type with your specified attributes. If no current generation C, M, or R instance type matches your attributes,
     * then the identified price is from the lowest priced current generation instance types, and failing that, from the
     * lowest priced previous generation instance types that match your attributes. When Amazon EC2 selects instance
     * types with your attributes, it will exclude instance types whose price exceeds your specified threshold.
     * </p>
     * <p>
     * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     * </p>
     * <p>
     * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     * protection threshold is based on the per vCPU or per memory price instead of the per instance price.
     * </p>
     * <note>
     * <p>
     * Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
     * <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify either, Amazon
     * EC2 will automatically apply optimal price protection to consistently select from a wide range of instance types.
     * To indicate no price protection threshold for Spot Instances, meaning you want to consider all instance types
     * that match your attributes, include one of these parameters and specify a high value, such as <code>999999</code>
     * .
     * </p>
     * </note>
     * 
     * @return [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified
     *         On-Demand price. The identified On-Demand price is the price of the lowest priced current generation C,
     *         M, or R instance type with your specified attributes. If no current generation C, M, or R instance type
     *         matches your attributes, then the identified price is from the lowest priced current generation instance
     *         types, and failing that, from the lowest priced previous generation instance types that match your
     *         attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types
     *         whose price exceeds your specified threshold.</p>
     *         <p>
     *         The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
     *         </p>
     *         <p>
     *         If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
     *         protection threshold is based on the per vCPU or per memory price instead of the per instance price.
     *         </p>
     *         <note>
     *         <p>
     *         Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
     *         <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify
     *         either, Amazon EC2 will automatically apply optimal price protection to consistently select from a wide
     *         range of instance types. To indicate no price protection threshold for Spot Instances, meaning you want
     *         to consider all instance types that match your attributes, include one of these parameters and specify a
     *         high value, such as <code>999999</code>.
     *         </p>
     */
    public final Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
        return maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(vCpuCount());
        hashCode = 31 * hashCode + Objects.hashCode(memoryMiB());
        hashCode = 31 * hashCode + Objects.hashCode(hasCpuManufacturers() ? cpuManufacturersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(memoryGiBPerVCpu());
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedInstanceTypes() ? excludedInstanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasInstanceGenerations() ? instanceGenerationsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(spotMaxPricePercentageOverLowestPrice());
        hashCode = 31 * hashCode + Objects.hashCode(onDemandMaxPricePercentageOverLowestPrice());
        hashCode = 31 * hashCode + Objects.hashCode(bareMetalAsString());
        hashCode = 31 * hashCode + Objects.hashCode(burstablePerformanceAsString());
        hashCode = 31 * hashCode + Objects.hashCode(requireHibernateSupport());
        hashCode = 31 * hashCode + Objects.hashCode(networkInterfaceCount());
        hashCode = 31 * hashCode + Objects.hashCode(localStorageAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasLocalStorageTypes() ? localStorageTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(totalLocalStorageGB());
        hashCode = 31 * hashCode + Objects.hashCode(baselineEbsBandwidthMbps());
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorTypes() ? acceleratorTypesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorCount());
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorManufacturers() ? acceleratorManufacturersAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasAcceleratorNames() ? acceleratorNamesAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorTotalMemoryMiB());
        hashCode = 31 * hashCode + Objects.hashCode(networkBandwidthGbps());
        hashCode = 31 * hashCode + Objects.hashCode(hasAllowedInstanceTypes() ? allowedInstanceTypes() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxSpotPriceAsPercentageOfOptimalOnDemandPrice());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRequirementsRequest)) {
            return false;
        }
        InstanceRequirementsRequest other = (InstanceRequirementsRequest) obj;
        return Objects.equals(vCpuCount(), other.vCpuCount())
                && Objects.equals(memoryMiB(), other.memoryMiB())
                && hasCpuManufacturers() == other.hasCpuManufacturers()
                && Objects.equals(cpuManufacturersAsStrings(), other.cpuManufacturersAsStrings())
                && Objects.equals(memoryGiBPerVCpu(), other.memoryGiBPerVCpu())
                && hasExcludedInstanceTypes() == other.hasExcludedInstanceTypes()
                && Objects.equals(excludedInstanceTypes(), other.excludedInstanceTypes())
                && hasInstanceGenerations() == other.hasInstanceGenerations()
                && Objects.equals(instanceGenerationsAsStrings(), other.instanceGenerationsAsStrings())
                && Objects.equals(spotMaxPricePercentageOverLowestPrice(), other.spotMaxPricePercentageOverLowestPrice())
                && Objects.equals(onDemandMaxPricePercentageOverLowestPrice(), other.onDemandMaxPricePercentageOverLowestPrice())
                && Objects.equals(bareMetalAsString(), other.bareMetalAsString())
                && Objects.equals(burstablePerformanceAsString(), other.burstablePerformanceAsString())
                && Objects.equals(requireHibernateSupport(), other.requireHibernateSupport())
                && Objects.equals(networkInterfaceCount(), other.networkInterfaceCount())
                && Objects.equals(localStorageAsString(), other.localStorageAsString())
                && hasLocalStorageTypes() == other.hasLocalStorageTypes()
                && Objects.equals(localStorageTypesAsStrings(), other.localStorageTypesAsStrings())
                && Objects.equals(totalLocalStorageGB(), other.totalLocalStorageGB())
                && Objects.equals(baselineEbsBandwidthMbps(), other.baselineEbsBandwidthMbps())
                && hasAcceleratorTypes() == other.hasAcceleratorTypes()
                && Objects.equals(acceleratorTypesAsStrings(), other.acceleratorTypesAsStrings())
                && Objects.equals(acceleratorCount(), other.acceleratorCount())
                && hasAcceleratorManufacturers() == other.hasAcceleratorManufacturers()
                && Objects.equals(acceleratorManufacturersAsStrings(), other.acceleratorManufacturersAsStrings())
                && hasAcceleratorNames() == other.hasAcceleratorNames()
                && Objects.equals(acceleratorNamesAsStrings(), other.acceleratorNamesAsStrings())
                && Objects.equals(acceleratorTotalMemoryMiB(), other.acceleratorTotalMemoryMiB())
                && Objects.equals(networkBandwidthGbps(), other.networkBandwidthGbps())
                && hasAllowedInstanceTypes() == other.hasAllowedInstanceTypes()
                && Objects.equals(allowedInstanceTypes(), other.allowedInstanceTypes())
                && Objects.equals(maxSpotPriceAsPercentageOfOptimalOnDemandPrice(),
                        other.maxSpotPriceAsPercentageOfOptimalOnDemandPrice());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("InstanceRequirementsRequest").add("VCpuCount", vCpuCount()).add("MemoryMiB", memoryMiB())
                .add("CpuManufacturers", hasCpuManufacturers() ? cpuManufacturersAsStrings() : null)
                .add("MemoryGiBPerVCpu", memoryGiBPerVCpu())
                .add("ExcludedInstanceTypes", hasExcludedInstanceTypes() ? excludedInstanceTypes() : null)
                .add("InstanceGenerations", hasInstanceGenerations() ? instanceGenerationsAsStrings() : null)
                .add("SpotMaxPricePercentageOverLowestPrice", spotMaxPricePercentageOverLowestPrice())
                .add("OnDemandMaxPricePercentageOverLowestPrice", onDemandMaxPricePercentageOverLowestPrice())
                .add("BareMetal", bareMetalAsString()).add("BurstablePerformance", burstablePerformanceAsString())
                .add("RequireHibernateSupport", requireHibernateSupport()).add("NetworkInterfaceCount", networkInterfaceCount())
                .add("LocalStorage", localStorageAsString())
                .add("LocalStorageTypes", hasLocalStorageTypes() ? localStorageTypesAsStrings() : null)
                .add("TotalLocalStorageGB", totalLocalStorageGB()).add("BaselineEbsBandwidthMbps", baselineEbsBandwidthMbps())
                .add("AcceleratorTypes", hasAcceleratorTypes() ? acceleratorTypesAsStrings() : null)
                .add("AcceleratorCount", acceleratorCount())
                .add("AcceleratorManufacturers", hasAcceleratorManufacturers() ? acceleratorManufacturersAsStrings() : null)
                .add("AcceleratorNames", hasAcceleratorNames() ? acceleratorNamesAsStrings() : null)
                .add("AcceleratorTotalMemoryMiB", acceleratorTotalMemoryMiB())
                .add("NetworkBandwidthGbps", networkBandwidthGbps())
                .add("AllowedInstanceTypes", hasAllowedInstanceTypes() ? allowedInstanceTypes() : null)
                .add("MaxSpotPriceAsPercentageOfOptimalOnDemandPrice", maxSpotPriceAsPercentageOfOptimalOnDemandPrice()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "VCpuCount":
            return Optional.ofNullable(clazz.cast(vCpuCount()));
        case "MemoryMiB":
            return Optional.ofNullable(clazz.cast(memoryMiB()));
        case "CpuManufacturers":
            return Optional.ofNullable(clazz.cast(cpuManufacturersAsStrings()));
        case "MemoryGiBPerVCpu":
            return Optional.ofNullable(clazz.cast(memoryGiBPerVCpu()));
        case "ExcludedInstanceTypes":
            return Optional.ofNullable(clazz.cast(excludedInstanceTypes()));
        case "InstanceGenerations":
            return Optional.ofNullable(clazz.cast(instanceGenerationsAsStrings()));
        case "SpotMaxPricePercentageOverLowestPrice":
            return Optional.ofNullable(clazz.cast(spotMaxPricePercentageOverLowestPrice()));
        case "OnDemandMaxPricePercentageOverLowestPrice":
            return Optional.ofNullable(clazz.cast(onDemandMaxPricePercentageOverLowestPrice()));
        case "BareMetal":
            return Optional.ofNullable(clazz.cast(bareMetalAsString()));
        case "BurstablePerformance":
            return Optional.ofNullable(clazz.cast(burstablePerformanceAsString()));
        case "RequireHibernateSupport":
            return Optional.ofNullable(clazz.cast(requireHibernateSupport()));
        case "NetworkInterfaceCount":
            return Optional.ofNullable(clazz.cast(networkInterfaceCount()));
        case "LocalStorage":
            return Optional.ofNullable(clazz.cast(localStorageAsString()));
        case "LocalStorageTypes":
            return Optional.ofNullable(clazz.cast(localStorageTypesAsStrings()));
        case "TotalLocalStorageGB":
            return Optional.ofNullable(clazz.cast(totalLocalStorageGB()));
        case "BaselineEbsBandwidthMbps":
            return Optional.ofNullable(clazz.cast(baselineEbsBandwidthMbps()));
        case "AcceleratorTypes":
            return Optional.ofNullable(clazz.cast(acceleratorTypesAsStrings()));
        case "AcceleratorCount":
            return Optional.ofNullable(clazz.cast(acceleratorCount()));
        case "AcceleratorManufacturers":
            return Optional.ofNullable(clazz.cast(acceleratorManufacturersAsStrings()));
        case "AcceleratorNames":
            return Optional.ofNullable(clazz.cast(acceleratorNamesAsStrings()));
        case "AcceleratorTotalMemoryMiB":
            return Optional.ofNullable(clazz.cast(acceleratorTotalMemoryMiB()));
        case "NetworkBandwidthGbps":
            return Optional.ofNullable(clazz.cast(networkBandwidthGbps()));
        case "AllowedInstanceTypes":
            return Optional.ofNullable(clazz.cast(allowedInstanceTypes()));
        case "MaxSpotPriceAsPercentageOfOptimalOnDemandPrice":
            return Optional.ofNullable(clazz.cast(maxSpotPriceAsPercentageOfOptimalOnDemandPrice()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceRequirementsRequest, T> g) {
        return obj -> g.apply((InstanceRequirementsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRequirementsRequest> {
        /**
         * <p>
         * The minimum and maximum number of vCPUs.
         * </p>
         * 
         * @param vCpuCount
         *        The minimum and maximum number of vCPUs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vCpuCount(VCpuCountRangeRequest vCpuCount);

        /**
         * <p>
         * The minimum and maximum number of vCPUs.
         * </p>
         * This is a convenience method that creates an instance of the {@link VCpuCountRangeRequest.Builder} avoiding
         * the need to create one manually via {@link VCpuCountRangeRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link VCpuCountRangeRequest.Builder#build()} is called immediately and
         * its result is passed to {@link #vCpuCount(VCpuCountRangeRequest)}.
         * 
         * @param vCpuCount
         *        a consumer that will call methods on {@link VCpuCountRangeRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #vCpuCount(VCpuCountRangeRequest)
         */
        default Builder vCpuCount(Consumer<VCpuCountRangeRequest.Builder> vCpuCount) {
            return vCpuCount(VCpuCountRangeRequest.builder().applyMutation(vCpuCount).build());
        }

        /**
         * <p>
         * The minimum and maximum amount of memory, in MiB.
         * </p>
         * 
         * @param memoryMiB
         *        The minimum and maximum amount of memory, in MiB.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryMiB(MemoryMiBRequest memoryMiB);

        /**
         * <p>
         * The minimum and maximum amount of memory, in MiB.
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryMiBRequest.Builder} avoiding the
         * need to create one manually via {@link MemoryMiBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryMiBRequest.Builder#build()} is called immediately and its
         * result is passed to {@link #memoryMiB(MemoryMiBRequest)}.
         * 
         * @param memoryMiB
         *        a consumer that will call methods on {@link MemoryMiBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryMiB(MemoryMiBRequest)
         */
        default Builder memoryMiB(Consumer<MemoryMiBRequest.Builder> memoryMiB) {
            return memoryMiB(MemoryMiBRequest.builder().applyMutation(memoryMiB).build());
        }

        /**
         * <p>
         * The CPU manufacturers to include.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Intel CPUs, specify <code>intel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD CPUs, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible
         * CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
         * </p>
         * </note>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Intel CPUs, specify <code>intel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD CPUs, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
         *        compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
         *        template.
         *        </p>
         *        </note>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturersWithStrings(Collection<String> cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Intel CPUs, specify <code>intel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD CPUs, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible
         * CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
         * </p>
         * </note>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Intel CPUs, specify <code>intel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD CPUs, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
         *        compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
         *        template.
         *        </p>
         *        </note>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturersWithStrings(String... cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Intel CPUs, specify <code>intel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD CPUs, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible
         * CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
         * </p>
         * </note>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Intel CPUs, specify <code>intel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD CPUs, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
         *        compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
         *        template.
         *        </p>
         *        </note>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturers(Collection<CpuManufacturer> cpuManufacturers);

        /**
         * <p>
         * The CPU manufacturers to include.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Intel CPUs, specify <code>intel</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD CPUs, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a compatible
         * CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch template.
         * </p>
         * </note>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param cpuManufacturers
         *        The CPU manufacturers to include.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Intel CPUs, specify <code>intel</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD CPUs, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services CPUs, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        Don't confuse the CPU manufacturer with the CPU architecture. Instances will be launched with a
         *        compatible CPU architecture based on the Amazon Machine Image (AMI) that you specify in your launch
         *        template.
         *        </p>
         *        </note>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cpuManufacturers(CpuManufacturer... cpuManufacturers);

        /**
         * <p>
         * The minimum and maximum amount of memory per vCPU, in GiB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param memoryGiBPerVCpu
         *        The minimum and maximum amount of memory per vCPU, in GiB.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memoryGiBPerVCpu(MemoryGiBPerVCpuRequest memoryGiBPerVCpu);

        /**
         * <p>
         * The minimum and maximum amount of memory per vCPU, in GiB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link MemoryGiBPerVCpuRequest.Builder} avoiding
         * the need to create one manually via {@link MemoryGiBPerVCpuRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link MemoryGiBPerVCpuRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #memoryGiBPerVCpu(MemoryGiBPerVCpuRequest)}.
         * 
         * @param memoryGiBPerVCpu
         *        a consumer that will call methods on {@link MemoryGiBPerVCpuRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memoryGiBPerVCpu(MemoryGiBPerVCpuRequest)
         */
        default Builder memoryGiBPerVCpu(Consumer<MemoryGiBPerVCpuRequest.Builder> memoryGiBPerVCpu) {
            return memoryGiBPerVCpu(MemoryGiBPerVCpuRequest.builder().applyMutation(memoryGiBPerVCpu).build());
        }

        /**
         * <p>
         * The instance types to exclude.
         * </p>
         * <p>
         * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to exclude an
         * instance family, type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
         * <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         * </p>
         * <p>
         * For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family, which
         * includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will exclude all the
         * M5a instance types, but not the M5n instance types.
         * </p>
         * <note>
         * <p>
         * If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>.
         * </p>
         * </note>
         * <p>
         * Default: No excluded instance types
         * </p>
         * 
         * @param excludedInstanceTypes
         *        The instance types to exclude.</p>
         *        <p>
         *        You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to
         *        exclude an instance family, type, size, or generation. The following are examples:
         *        <code>m5.8xlarge</code>, <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         *        </p>
         *        <p>
         *        For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family,
         *        which includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will
         *        exclude all the M5a instance types, but not the M5n instance types.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>
         *        .
         *        </p>
         *        </note>
         *        <p>
         *        Default: No excluded instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes);

        /**
         * <p>
         * The instance types to exclude.
         * </p>
         * <p>
         * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to exclude an
         * instance family, type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
         * <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         * </p>
         * <p>
         * For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family, which
         * includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will exclude all the
         * M5a instance types, but not the M5n instance types.
         * </p>
         * <note>
         * <p>
         * If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>.
         * </p>
         * </note>
         * <p>
         * Default: No excluded instance types
         * </p>
         * 
         * @param excludedInstanceTypes
         *        The instance types to exclude.</p>
         *        <p>
         *        You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to
         *        exclude an instance family, type, size, or generation. The following are examples:
         *        <code>m5.8xlarge</code>, <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         *        </p>
         *        <p>
         *        For example, if you specify <code>c5*</code>,Amazon EC2 will exclude the entire C5 instance family,
         *        which includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will
         *        exclude all the M5a instance types, but not the M5n instance types.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>ExcludedInstanceTypes</code>, you can't specify <code>AllowedInstanceTypes</code>
         *        .
         *        </p>
         *        </note>
         *        <p>
         *        Default: No excluded instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedInstanceTypes(String... excludedInstanceTypes);

        /**
         * <p>
         * Indicates whether current or previous generation instance types are included. The current generation instance
         * types are recommended for use. Current generation instance types are typically the latest two to three
         * generations in each instance family. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
         * <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * For current generation instance types, specify <code>current</code>.
         * </p>
         * <p>
         * For previous generation instance types, specify <code>previous</code>.
         * </p>
         * <p>
         * Default: Current and previous generation instance types
         * </p>
         * 
         * @param instanceGenerations
         *        Indicates whether current or previous generation instance types are included. The current generation
         *        instance types are recommended for use. Current generation instance types are typically the latest two
         *        to three generations in each instance family. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in
         *        the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        For current generation instance types, specify <code>current</code>.
         *        </p>
         *        <p>
         *        For previous generation instance types, specify <code>previous</code>.
         *        </p>
         *        <p>
         *        Default: Current and previous generation instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerationsWithStrings(Collection<String> instanceGenerations);

        /**
         * <p>
         * Indicates whether current or previous generation instance types are included. The current generation instance
         * types are recommended for use. Current generation instance types are typically the latest two to three
         * generations in each instance family. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
         * <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * For current generation instance types, specify <code>current</code>.
         * </p>
         * <p>
         * For previous generation instance types, specify <code>previous</code>.
         * </p>
         * <p>
         * Default: Current and previous generation instance types
         * </p>
         * 
         * @param instanceGenerations
         *        Indicates whether current or previous generation instance types are included. The current generation
         *        instance types are recommended for use. Current generation instance types are typically the latest two
         *        to three generations in each instance family. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in
         *        the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        For current generation instance types, specify <code>current</code>.
         *        </p>
         *        <p>
         *        For previous generation instance types, specify <code>previous</code>.
         *        </p>
         *        <p>
         *        Default: Current and previous generation instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerationsWithStrings(String... instanceGenerations);

        /**
         * <p>
         * Indicates whether current or previous generation instance types are included. The current generation instance
         * types are recommended for use. Current generation instance types are typically the latest two to three
         * generations in each instance family. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
         * <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * For current generation instance types, specify <code>current</code>.
         * </p>
         * <p>
         * For previous generation instance types, specify <code>previous</code>.
         * </p>
         * <p>
         * Default: Current and previous generation instance types
         * </p>
         * 
         * @param instanceGenerations
         *        Indicates whether current or previous generation instance types are included. The current generation
         *        instance types are recommended for use. Current generation instance types are typically the latest two
         *        to three generations in each instance family. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in
         *        the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        For current generation instance types, specify <code>current</code>.
         *        </p>
         *        <p>
         *        For previous generation instance types, specify <code>previous</code>.
         *        </p>
         *        <p>
         *        Default: Current and previous generation instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerations(Collection<InstanceGeneration> instanceGenerations);

        /**
         * <p>
         * Indicates whether current or previous generation instance types are included. The current generation instance
         * types are recommended for use. Current generation instance types are typically the latest two to three
         * generations in each instance family. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in the
         * <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * For current generation instance types, specify <code>current</code>.
         * </p>
         * <p>
         * For previous generation instance types, specify <code>previous</code>.
         * </p>
         * <p>
         * Default: Current and previous generation instance types
         * </p>
         * 
         * @param instanceGenerations
         *        Indicates whether current or previous generation instance types are included. The current generation
         *        instance types are recommended for use. Current generation instance types are typically the latest two
         *        to three generations in each instance family. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Instance types</a> in
         *        the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        For current generation instance types, specify <code>current</code>.
         *        </p>
         *        <p>
         *        For previous generation instance types, specify <code>previous</code>.
         *        </p>
         *        <p>
         *        Default: Current and previous generation instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceGenerations(InstanceGeneration... instanceGenerations);

        /**
         * <p>
         * [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an
         * identified Spot price. The identified Spot price is the Spot price of the lowest priced current generation C,
         * M, or R instance type with your specified attributes. If no current generation C, M, or R instance type
         * matches your attributes, then the identified Spot price is from the lowest priced current generation instance
         * types, and failing that, from the lowest priced previous generation instance types that match your
         * attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance types whose
         * Spot price exceeds your specified threshold.
         * </p>
         * <p>
         * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         * </p>
         * <p>
         * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
         * protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.
         * </p>
         * <p>
         * This parameter is not supported for <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         * >GetSpotPlacementScores</a> and <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
         * >GetInstanceTypesFromInstanceRequirements</a>.
         * </p>
         * <note>
         * <p>
         * Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
         * <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify either,
         * Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of
         * instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider
         * all instance types that match your attributes, include one of these parameters and specify a high value, such
         * as <code>999999</code>.
         * </p>
         * </note>
         * <p>
         * Default: <code>100</code>
         * </p>
         * 
         * @param spotMaxPricePercentageOverLowestPrice
         *        [Price protection] The price protection threshold for Spot Instances, as a percentage higher than an
         *        identified Spot price. The identified Spot price is the Spot price of the lowest priced current
         *        generation C, M, or R instance type with your specified attributes. If no current generation C, M, or
         *        R instance type matches your attributes, then the identified Spot price is from the lowest priced
         *        current generation instance types, and failing that, from the lowest priced previous generation
         *        instance types that match your attributes. When Amazon EC2 selects instance types with your
         *        attributes, it will exclude instance types whose Spot price exceeds your specified threshold.</p>
         *        <p>
         *        The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         *        </p>
         *        <p>
         *        If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the
         *        price protection threshold is applied based on the per-vCPU or per-memory price instead of the
         *        per-instance price.
         *        </p>
         *        <p>
         *        This parameter is not supported for <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         *        >GetSpotPlacementScores</a> and <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
         *        >GetInstanceTypesFromInstanceRequirements</a>.
         *        </p>
         *        <note>
         *        <p>
         *        Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
         *        <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify
         *        either, Amazon EC2 will automatically apply optimal price protection to consistently select from a
         *        wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning
         *        you want to consider all instance types that match your attributes, include one of these parameters
         *        and specify a high value, such as <code>999999</code>.
         *        </p>
         *        </note>
         *        <p>
         *        Default: <code>100</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice);

        /**
         * <p>
         * [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than an
         * identified On-Demand price. The identified On-Demand price is the price of the lowest priced current
         * generation C, M, or R instance type with your specified attributes. When Amazon EC2 selects instance types
         * with your attributes, it will exclude instance types whose price exceeds your specified threshold.
         * </p>
         * <p>
         * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         * </p>
         * <p>
         * To indicate no price protection threshold, specify a high value, such as <code>999999</code>.
         * </p>
         * <p>
         * This parameter is not supported for <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         * >GetSpotPlacementScores</a> and <a href=
         * "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
         * >GetInstanceTypesFromInstanceRequirements</a>.
         * </p>
         * <note>
         * <p>
         * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
         * protection threshold is applied based on the per-vCPU or per-memory price instead of the per-instance price.
         * </p>
         * </note>
         * <p>
         * Default: <code>20</code>
         * </p>
         * 
         * @param onDemandMaxPricePercentageOverLowestPrice
         *        [Price protection] The price protection threshold for On-Demand Instances, as a percentage higher than
         *        an identified On-Demand price. The identified On-Demand price is the price of the lowest priced
         *        current generation C, M, or R instance type with your specified attributes. When Amazon EC2 selects
         *        instance types with your attributes, it will exclude instance types whose price exceeds your specified
         *        threshold.</p>
         *        <p>
         *        The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         *        </p>
         *        <p>
         *        To indicate no price protection threshold, specify a high value, such as <code>999999</code>.
         *        </p>
         *        <p>
         *        This parameter is not supported for <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         *        >GetSpotPlacementScores</a> and <a href=
         *        "https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetInstanceTypesFromInstanceRequirements.html"
         *        >GetInstanceTypesFromInstanceRequirements</a>.
         *        </p>
         *        <note>
         *        <p>
         *        If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the
         *        price protection threshold is applied based on the per-vCPU or per-memory price instead of the
         *        per-instance price.
         *        </p>
         *        </note>
         *        <p>
         *        Default: <code>20</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder onDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice);

        /**
         * <p>
         * Indicates whether bare metal instance types must be included, excluded, or required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include bare metal instance types, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only bare metal instance types, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude bare metal instance types, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>excluded</code>
         * </p>
         * 
         * @param bareMetal
         *        Indicates whether bare metal instance types must be included, excluded, or required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include bare metal instance types, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only bare metal instance types, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude bare metal instance types, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>excluded</code>
         * @see BareMetal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BareMetal
         */
        Builder bareMetal(String bareMetal);

        /**
         * <p>
         * Indicates whether bare metal instance types must be included, excluded, or required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include bare metal instance types, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only bare metal instance types, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude bare metal instance types, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>excluded</code>
         * </p>
         * 
         * @param bareMetal
         *        Indicates whether bare metal instance types must be included, excluded, or required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include bare metal instance types, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only bare metal instance types, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude bare metal instance types, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>excluded</code>
         * @see BareMetal
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BareMetal
         */
        Builder bareMetal(BareMetal bareMetal);

        /**
         * <p>
         * Indicates whether burstable performance T instance types are included, excluded, or required. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
         * performance instances</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include burstable performance instance types, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only burstable performance instance types, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude burstable performance instance types, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>excluded</code>
         * </p>
         * 
         * @param burstablePerformance
         *        Indicates whether burstable performance T instance types are included, excluded, or required. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html"
         *        >Burstable performance instances</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include burstable performance instance types, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only burstable performance instance types, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude burstable performance instance types, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>excluded</code>
         * @see BurstablePerformance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BurstablePerformance
         */
        Builder burstablePerformance(String burstablePerformance);

        /**
         * <p>
         * Indicates whether burstable performance T instance types are included, excluded, or required. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html">Burstable
         * performance instances</a>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include burstable performance instance types, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only burstable performance instance types, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude burstable performance instance types, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>excluded</code>
         * </p>
         * 
         * @param burstablePerformance
         *        Indicates whether burstable performance T instance types are included, excluded, or required. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances.html"
         *        >Burstable performance instances</a>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include burstable performance instance types, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only burstable performance instance types, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude burstable performance instance types, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>excluded</code>
         * @see BurstablePerformance
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see BurstablePerformance
         */
        Builder burstablePerformance(BurstablePerformance burstablePerformance);

        /**
         * <p>
         * Indicates whether instance types must support hibernation for On-Demand Instances.
         * </p>
         * <p>
         * This parameter is not supported for <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         * >GetSpotPlacementScores</a>.
         * </p>
         * <p>
         * Default: <code>false</code>
         * </p>
         * 
         * @param requireHibernateSupport
         *        Indicates whether instance types must support hibernation for On-Demand Instances.</p>
         *        <p>
         *        This parameter is not supported for <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_GetSpotPlacementScores.html"
         *        >GetSpotPlacementScores</a>.
         *        </p>
         *        <p>
         *        Default: <code>false</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder requireHibernateSupport(Boolean requireHibernateSupport);

        /**
         * <p>
         * The minimum and maximum number of network interfaces.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param networkInterfaceCount
         *        The minimum and maximum number of network interfaces.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkInterfaceCount(NetworkInterfaceCountRequest networkInterfaceCount);

        /**
         * <p>
         * The minimum and maximum number of network interfaces.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkInterfaceCountRequest.Builder}
         * avoiding the need to create one manually via {@link NetworkInterfaceCountRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkInterfaceCountRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #networkInterfaceCount(NetworkInterfaceCountRequest)}.
         * 
         * @param networkInterfaceCount
         *        a consumer that will call methods on {@link NetworkInterfaceCountRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkInterfaceCount(NetworkInterfaceCountRequest)
         */
        default Builder networkInterfaceCount(Consumer<NetworkInterfaceCountRequest.Builder> networkInterfaceCount) {
            return networkInterfaceCount(NetworkInterfaceCountRequest.builder().applyMutation(networkInterfaceCount).build());
        }

        /**
         * <p>
         * Indicates whether instance types with instance store volumes are included, excluded, or required. For more
         * information, <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2
         * instance store</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with instance store volumes, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only instance types with instance store volumes, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude instance types with instance store volumes, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>included</code>
         * </p>
         * 
         * @param localStorage
         *        Indicates whether instance types with instance store volumes are included, excluded, or required. For
         *        more information, <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with instance store volumes, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only instance types with instance store volumes, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude instance types with instance store volumes, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>included</code>
         * @see LocalStorage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalStorage
         */
        Builder localStorage(String localStorage);

        /**
         * <p>
         * Indicates whether instance types with instance store volumes are included, excluded, or required. For more
         * information, <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2
         * instance store</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with instance store volumes, specify <code>included</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To require only instance types with instance store volumes, specify <code>required</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To exclude instance types with instance store volumes, specify <code>excluded</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>included</code>
         * </p>
         * 
         * @param localStorage
         *        Indicates whether instance types with instance store volumes are included, excluded, or required. For
         *        more information, <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/InstanceStorage.html">Amazon EC2 instance
         *        store</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with instance store volumes, specify <code>included</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To require only instance types with instance store volumes, specify <code>required</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To exclude instance types with instance store volumes, specify <code>excluded</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>included</code>
         * @see LocalStorage
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LocalStorage
         */
        Builder localStorage(LocalStorage localStorage);

        /**
         * <p>
         * The type of local storage that is required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>hdd</code> and <code>ssd</code>
         * </p>
         * 
         * @param localStorageTypes
         *        The type of local storage that is required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>hdd</code> and <code>ssd</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypesWithStrings(Collection<String> localStorageTypes);

        /**
         * <p>
         * The type of local storage that is required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>hdd</code> and <code>ssd</code>
         * </p>
         * 
         * @param localStorageTypes
         *        The type of local storage that is required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>hdd</code> and <code>ssd</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypesWithStrings(String... localStorageTypes);

        /**
         * <p>
         * The type of local storage that is required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>hdd</code> and <code>ssd</code>
         * </p>
         * 
         * @param localStorageTypes
         *        The type of local storage that is required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>hdd</code> and <code>ssd</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypes(Collection<LocalStorageType> localStorageTypes);

        /**
         * <p>
         * The type of local storage that is required.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: <code>hdd</code> and <code>ssd</code>
         * </p>
         * 
         * @param localStorageTypes
         *        The type of local storage that is required.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with hard disk drive (HDD) storage, specify <code>hdd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with solid state drive (SSD) storage, specify <code>ssd</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: <code>hdd</code> and <code>ssd</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder localStorageTypes(LocalStorageType... localStorageTypes);

        /**
         * <p>
         * The minimum and maximum amount of total local storage, in GB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param totalLocalStorageGB
         *        The minimum and maximum amount of total local storage, in GB.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder totalLocalStorageGB(TotalLocalStorageGBRequest totalLocalStorageGB);

        /**
         * <p>
         * The minimum and maximum amount of total local storage, in GB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link TotalLocalStorageGBRequest.Builder}
         * avoiding the need to create one manually via {@link TotalLocalStorageGBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link TotalLocalStorageGBRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #totalLocalStorageGB(TotalLocalStorageGBRequest)}.
         * 
         * @param totalLocalStorageGB
         *        a consumer that will call methods on {@link TotalLocalStorageGBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #totalLocalStorageGB(TotalLocalStorageGBRequest)
         */
        default Builder totalLocalStorageGB(Consumer<TotalLocalStorageGBRequest.Builder> totalLocalStorageGB) {
            return totalLocalStorageGB(TotalLocalStorageGBRequest.builder().applyMutation(totalLocalStorageGB).build());
        }

        /**
         * <p>
         * The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html">Amazon EBS–optimized
         * instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param baselineEbsBandwidthMbps
         *        The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html">Amazon EBS–optimized
         *        instances</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps);

        /**
         * <p>
         * The minimum and maximum baseline bandwidth to Amazon EBS, in Mbps. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ebs-optimized.html">Amazon EBS–optimized
         * instances</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link BaselineEbsBandwidthMbpsRequest.Builder}
         * avoiding the need to create one manually via {@link BaselineEbsBandwidthMbpsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link BaselineEbsBandwidthMbpsRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest)}.
         * 
         * @param baselineEbsBandwidthMbps
         *        a consumer that will call methods on {@link BaselineEbsBandwidthMbpsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest)
         */
        default Builder baselineEbsBandwidthMbps(Consumer<BaselineEbsBandwidthMbpsRequest.Builder> baselineEbsBandwidthMbps) {
            return baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest.builder().applyMutation(baselineEbsBandwidthMbps)
                    .build());
        }

        /**
         * <p>
         * The accelerator types that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with GPU hardware, specify <code>gpu</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with FPGA hardware, specify <code>fpga</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with inference hardware, specify <code>inference</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator type
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with GPU hardware, specify <code>gpu</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with FPGA hardware, specify <code>fpga</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with inference hardware, specify <code>inference</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator type
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypesWithStrings(Collection<String> acceleratorTypes);

        /**
         * <p>
         * The accelerator types that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with GPU hardware, specify <code>gpu</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with FPGA hardware, specify <code>fpga</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with inference hardware, specify <code>inference</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator type
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with GPU hardware, specify <code>gpu</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with FPGA hardware, specify <code>fpga</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with inference hardware, specify <code>inference</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator type
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypesWithStrings(String... acceleratorTypes);

        /**
         * <p>
         * The accelerator types that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with GPU hardware, specify <code>gpu</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with FPGA hardware, specify <code>fpga</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with inference hardware, specify <code>inference</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator type
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with GPU hardware, specify <code>gpu</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with FPGA hardware, specify <code>fpga</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with inference hardware, specify <code>inference</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator type
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypes(Collection<AcceleratorType> acceleratorTypes);

        /**
         * <p>
         * The accelerator types that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To include instance types with GPU hardware, specify <code>gpu</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with FPGA hardware, specify <code>fpga</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To include instance types with inference hardware, specify <code>inference</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator type
         * </p>
         * 
         * @param acceleratorTypes
         *        The accelerator types that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To include instance types with GPU hardware, specify <code>gpu</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with FPGA hardware, specify <code>fpga</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To include instance types with inference hardware, specify <code>inference</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator type
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTypes(AcceleratorType... acceleratorTypes);

        /**
         * <p>
         * The minimum and maximum number of accelerators (GPUs, FPGAs, or Amazon Web Services Inferentia chips) on an
         * instance.
         * </p>
         * <p>
         * To exclude accelerator-enabled instance types, set <code>Max</code> to <code>0</code>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param acceleratorCount
         *        The minimum and maximum number of accelerators (GPUs, FPGAs, or Amazon Web Services Inferentia chips)
         *        on an instance.</p>
         *        <p>
         *        To exclude accelerator-enabled instance types, set <code>Max</code> to <code>0</code>.
         *        </p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorCount(AcceleratorCountRequest acceleratorCount);

        /**
         * <p>
         * The minimum and maximum number of accelerators (GPUs, FPGAs, or Amazon Web Services Inferentia chips) on an
         * instance.
         * </p>
         * <p>
         * To exclude accelerator-enabled instance types, set <code>Max</code> to <code>0</code>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link AcceleratorCountRequest.Builder} avoiding
         * the need to create one manually via {@link AcceleratorCountRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AcceleratorCountRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #acceleratorCount(AcceleratorCountRequest)}.
         * 
         * @param acceleratorCount
         *        a consumer that will call methods on {@link AcceleratorCountRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #acceleratorCount(AcceleratorCountRequest)
         */
        default Builder acceleratorCount(Consumer<AcceleratorCountRequest.Builder> acceleratorCount) {
            return acceleratorCount(AcceleratorCountRequest.builder().applyMutation(acceleratorCount).build());
        }

        /**
         * <p>
         * Indicates whether instance types must have accelerators by specific manufacturers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD devices, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Habana devices, specify <code>habana</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA devices, specify <code>nvidia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx devices, specify <code>xilinx</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param acceleratorManufacturers
         *        Indicates whether instance types must have accelerators by specific manufacturers.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD devices, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Habana devices, specify <code>habana</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA devices, specify <code>nvidia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx devices, specify <code>xilinx</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturersWithStrings(Collection<String> acceleratorManufacturers);

        /**
         * <p>
         * Indicates whether instance types must have accelerators by specific manufacturers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD devices, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Habana devices, specify <code>habana</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA devices, specify <code>nvidia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx devices, specify <code>xilinx</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param acceleratorManufacturers
         *        Indicates whether instance types must have accelerators by specific manufacturers.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD devices, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Habana devices, specify <code>habana</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA devices, specify <code>nvidia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx devices, specify <code>xilinx</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturersWithStrings(String... acceleratorManufacturers);

        /**
         * <p>
         * Indicates whether instance types must have accelerators by specific manufacturers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD devices, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Habana devices, specify <code>habana</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA devices, specify <code>nvidia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx devices, specify <code>xilinx</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param acceleratorManufacturers
         *        Indicates whether instance types must have accelerators by specific manufacturers.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD devices, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Habana devices, specify <code>habana</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA devices, specify <code>nvidia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx devices, specify <code>xilinx</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturers(Collection<AcceleratorManufacturer> acceleratorManufacturers);

        /**
         * <p>
         * Indicates whether instance types must have accelerators by specific manufacturers.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD devices, specify <code>amd</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Habana devices, specify <code>habana</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA devices, specify <code>nvidia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx devices, specify <code>xilinx</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any manufacturer
         * </p>
         * 
         * @param acceleratorManufacturers
         *        Indicates whether instance types must have accelerators by specific manufacturers.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services devices, specify <code>amazon-web-services</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD devices, specify <code>amd</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Habana devices, specify <code>habana</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA devices, specify <code>nvidia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx devices, specify <code>xilinx</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any manufacturer
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorManufacturers(AcceleratorManufacturer... acceleratorManufacturers);

        /**
         * <p>
         * The accelerators that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator
         * </p>
         * 
         * @param acceleratorNames
         *        The accelerators that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNamesWithStrings(Collection<String> acceleratorNames);

        /**
         * <p>
         * The accelerators that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator
         * </p>
         * 
         * @param acceleratorNames
         *        The accelerators that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNamesWithStrings(String... acceleratorNames);

        /**
         * <p>
         * The accelerators that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator
         * </p>
         * 
         * @param acceleratorNames
         *        The accelerators that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNames(Collection<AcceleratorName> acceleratorNames);

        /**
         * <p>
         * The accelerators that must be on the instance type.
         * </p>
         * <ul>
         * <li>
         * <p>
         * For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Any accelerator
         * </p>
         * 
         * @param acceleratorNames
         *        The accelerators that must be on the instance type.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A10G GPUs, specify <code>a10g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA A100 GPUs, specify <code>a100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA H100 GPUs, specify <code>h100</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Amazon Web Services Inferentia chips, specify <code>inferentia</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA GRID K520 GPUs, specify <code>k520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA K80 GPUs, specify <code>k80</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA M60 GPUs, specify <code>m60</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with AMD Radeon Pro V520 GPUs, specify <code>radeon-pro-v520</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4 GPUs, specify <code>t4</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA T4G GPUs, specify <code>t4g</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with Xilinx VU9P FPGAs, specify <code>vu9p</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For instance types with NVIDIA V100 GPUs, specify <code>v100</code>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Any accelerator
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorNames(AcceleratorName... acceleratorNames);

        /**
         * <p>
         * The minimum and maximum amount of total accelerator memory, in MiB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param acceleratorTotalMemoryMiB
         *        The minimum and maximum amount of total accelerator memory, in MiB.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB);

        /**
         * <p>
         * The minimum and maximum amount of total accelerator memory, in MiB.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link AcceleratorTotalMemoryMiBRequest.Builder}
         * avoiding the need to create one manually via {@link AcceleratorTotalMemoryMiBRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link AcceleratorTotalMemoryMiBRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest)}.
         * 
         * @param acceleratorTotalMemoryMiB
         *        a consumer that will call methods on {@link AcceleratorTotalMemoryMiBRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest)
         */
        default Builder acceleratorTotalMemoryMiB(Consumer<AcceleratorTotalMemoryMiBRequest.Builder> acceleratorTotalMemoryMiB) {
            return acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest.builder().applyMutation(acceleratorTotalMemoryMiB)
                    .build());
        }

        /**
         * <p>
         * The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html">Amazon EC2
         * instance network bandwidth</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * 
         * @param networkBandwidthGbps
         *        The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html">Amazon
         *        EC2 instance network bandwidth</a> in the <i>Amazon EC2 User Guide</i>.</p>
         *        <p>
         *        Default: No minimum or maximum limits
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkBandwidthGbps(NetworkBandwidthGbpsRequest networkBandwidthGbps);

        /**
         * <p>
         * The minimum and maximum amount of baseline network bandwidth, in gigabits per second (Gbps). For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-instance-network-bandwidth.html">Amazon EC2
         * instance network bandwidth</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * <p>
         * Default: No minimum or maximum limits
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkBandwidthGbpsRequest.Builder}
         * avoiding the need to create one manually via {@link NetworkBandwidthGbpsRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkBandwidthGbpsRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #networkBandwidthGbps(NetworkBandwidthGbpsRequest)}.
         * 
         * @param networkBandwidthGbps
         *        a consumer that will call methods on {@link NetworkBandwidthGbpsRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkBandwidthGbps(NetworkBandwidthGbpsRequest)
         */
        default Builder networkBandwidthGbps(Consumer<NetworkBandwidthGbpsRequest.Builder> networkBandwidthGbps) {
            return networkBandwidthGbps(NetworkBandwidthGbpsRequest.builder().applyMutation(networkBandwidthGbps).build());
        }

        /**
         * <p>
         * The instance types to apply your specified attributes against. All other instance types are ignored, even if
         * they match your specified attributes.
         * </p>
         * <p>
         * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow an
         * instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>, <code>c5*.*</code>,
         * <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         * </p>
         * <p>
         * For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family, which
         * includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will allow all the M5a
         * instance types, but not the M5n instance types.
         * </p>
         * <note>
         * <p>
         * If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>.
         * </p>
         * </note>
         * <p>
         * Default: All instance types
         * </p>
         * 
         * @param allowedInstanceTypes
         *        The instance types to apply your specified attributes against. All other instance types are ignored,
         *        even if they match your specified attributes.</p>
         *        <p>
         *        You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow
         *        an instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
         *        <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         *        </p>
         *        <p>
         *        For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family,
         *        which includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will
         *        allow all the M5a instance types, but not the M5n instance types.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>
         *        .
         *        </p>
         *        </note>
         *        <p>
         *        Default: All instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes);

        /**
         * <p>
         * The instance types to apply your specified attributes against. All other instance types are ignored, even if
         * they match your specified attributes.
         * </p>
         * <p>
         * You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow an
         * instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>, <code>c5*.*</code>,
         * <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         * </p>
         * <p>
         * For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family, which
         * includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will allow all the M5a
         * instance types, but not the M5n instance types.
         * </p>
         * <note>
         * <p>
         * If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>.
         * </p>
         * </note>
         * <p>
         * Default: All instance types
         * </p>
         * 
         * @param allowedInstanceTypes
         *        The instance types to apply your specified attributes against. All other instance types are ignored,
         *        even if they match your specified attributes.</p>
         *        <p>
         *        You can use strings with one or more wild cards, represented by an asterisk (<code>*</code>), to allow
         *        an instance type, size, or generation. The following are examples: <code>m5.8xlarge</code>,
         *        <code>c5*.*</code>, <code>m5a.*</code>, <code>r*</code>, <code>*3*</code>.
         *        </p>
         *        <p>
         *        For example, if you specify <code>c5*</code>,Amazon EC2 will allow the entire C5 instance family,
         *        which includes all C5a and C5n instance types. If you specify <code>m5a.*</code>, Amazon EC2 will
         *        allow all the M5a instance types, but not the M5n instance types.
         *        </p>
         *        <note>
         *        <p>
         *        If you specify <code>AllowedInstanceTypes</code>, you can't specify <code>ExcludedInstanceTypes</code>
         *        .
         *        </p>
         *        </note>
         *        <p>
         *        Default: All instance types
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder allowedInstanceTypes(String... allowedInstanceTypes);

        /**
         * <p>
         * [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified
         * On-Demand price. The identified On-Demand price is the price of the lowest priced current generation C, M, or
         * R instance type with your specified attributes. If no current generation C, M, or R instance type matches
         * your attributes, then the identified price is from the lowest priced current generation instance types, and
         * failing that, from the lowest priced previous generation instance types that match your attributes. When
         * Amazon EC2 selects instance types with your attributes, it will exclude instance types whose price exceeds
         * your specified threshold.
         * </p>
         * <p>
         * The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         * </p>
         * <p>
         * If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the price
         * protection threshold is based on the per vCPU or per memory price instead of the per instance price.
         * </p>
         * <note>
         * <p>
         * Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
         * <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify either,
         * Amazon EC2 will automatically apply optimal price protection to consistently select from a wide range of
         * instance types. To indicate no price protection threshold for Spot Instances, meaning you want to consider
         * all instance types that match your attributes, include one of these parameters and specify a high value, such
         * as <code>999999</code>.
         * </p>
         * </note>
         * 
         * @param maxSpotPriceAsPercentageOfOptimalOnDemandPrice
         *        [Price protection] The price protection threshold for Spot Instances, as a percentage of an identified
         *        On-Demand price. The identified On-Demand price is the price of the lowest priced current generation
         *        C, M, or R instance type with your specified attributes. If no current generation C, M, or R instance
         *        type matches your attributes, then the identified price is from the lowest priced current generation
         *        instance types, and failing that, from the lowest priced previous generation instance types that match
         *        your attributes. When Amazon EC2 selects instance types with your attributes, it will exclude instance
         *        types whose price exceeds your specified threshold.</p>
         *        <p>
         *        The parameter accepts an integer, which Amazon EC2 interprets as a percentage.
         *        </p>
         *        <p>
         *        If you set <code>TargetCapacityUnitType</code> to <code>vcpu</code> or <code>memory-mib</code>, the
         *        price protection threshold is based on the per vCPU or per memory price instead of the per instance
         *        price.
         *        </p>
         *        <note>
         *        <p>
         *        Only one of <code>SpotMaxPricePercentageOverLowestPrice</code> or
         *        <code>MaxSpotPriceAsPercentageOfOptimalOnDemandPrice</code> can be specified. If you don't specify
         *        either, Amazon EC2 will automatically apply optimal price protection to consistently select from a
         *        wide range of instance types. To indicate no price protection threshold for Spot Instances, meaning
         *        you want to consider all instance types that match your attributes, include one of these parameters
         *        and specify a high value, such as <code>999999</code>.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice);
    }

    static final class BuilderImpl implements Builder {
        private VCpuCountRangeRequest vCpuCount;

        private MemoryMiBRequest memoryMiB;

        private List<String> cpuManufacturers = DefaultSdkAutoConstructList.getInstance();

        private MemoryGiBPerVCpuRequest memoryGiBPerVCpu;

        private List<String> excludedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private List<String> instanceGenerations = DefaultSdkAutoConstructList.getInstance();

        private Integer spotMaxPricePercentageOverLowestPrice;

        private Integer onDemandMaxPricePercentageOverLowestPrice;

        private String bareMetal;

        private String burstablePerformance;

        private Boolean requireHibernateSupport;

        private NetworkInterfaceCountRequest networkInterfaceCount;

        private String localStorage;

        private List<String> localStorageTypes = DefaultSdkAutoConstructList.getInstance();

        private TotalLocalStorageGBRequest totalLocalStorageGB;

        private BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps;

        private List<String> acceleratorTypes = DefaultSdkAutoConstructList.getInstance();

        private AcceleratorCountRequest acceleratorCount;

        private List<String> acceleratorManufacturers = DefaultSdkAutoConstructList.getInstance();

        private List<String> acceleratorNames = DefaultSdkAutoConstructList.getInstance();

        private AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB;

        private NetworkBandwidthGbpsRequest networkBandwidthGbps;

        private List<String> allowedInstanceTypes = DefaultSdkAutoConstructList.getInstance();

        private Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRequirementsRequest model) {
            vCpuCount(model.vCpuCount);
            memoryMiB(model.memoryMiB);
            cpuManufacturersWithStrings(model.cpuManufacturers);
            memoryGiBPerVCpu(model.memoryGiBPerVCpu);
            excludedInstanceTypes(model.excludedInstanceTypes);
            instanceGenerationsWithStrings(model.instanceGenerations);
            spotMaxPricePercentageOverLowestPrice(model.spotMaxPricePercentageOverLowestPrice);
            onDemandMaxPricePercentageOverLowestPrice(model.onDemandMaxPricePercentageOverLowestPrice);
            bareMetal(model.bareMetal);
            burstablePerformance(model.burstablePerformance);
            requireHibernateSupport(model.requireHibernateSupport);
            networkInterfaceCount(model.networkInterfaceCount);
            localStorage(model.localStorage);
            localStorageTypesWithStrings(model.localStorageTypes);
            totalLocalStorageGB(model.totalLocalStorageGB);
            baselineEbsBandwidthMbps(model.baselineEbsBandwidthMbps);
            acceleratorTypesWithStrings(model.acceleratorTypes);
            acceleratorCount(model.acceleratorCount);
            acceleratorManufacturersWithStrings(model.acceleratorManufacturers);
            acceleratorNamesWithStrings(model.acceleratorNames);
            acceleratorTotalMemoryMiB(model.acceleratorTotalMemoryMiB);
            networkBandwidthGbps(model.networkBandwidthGbps);
            allowedInstanceTypes(model.allowedInstanceTypes);
            maxSpotPriceAsPercentageOfOptimalOnDemandPrice(model.maxSpotPriceAsPercentageOfOptimalOnDemandPrice);
        }

        public final VCpuCountRangeRequest.Builder getVCpuCount() {
            return vCpuCount != null ? vCpuCount.toBuilder() : null;
        }

        public final void setVCpuCount(VCpuCountRangeRequest.BuilderImpl vCpuCount) {
            this.vCpuCount = vCpuCount != null ? vCpuCount.build() : null;
        }

        @Override
        public final Builder vCpuCount(VCpuCountRangeRequest vCpuCount) {
            this.vCpuCount = vCpuCount;
            return this;
        }

        public final MemoryMiBRequest.Builder getMemoryMiB() {
            return memoryMiB != null ? memoryMiB.toBuilder() : null;
        }

        public final void setMemoryMiB(MemoryMiBRequest.BuilderImpl memoryMiB) {
            this.memoryMiB = memoryMiB != null ? memoryMiB.build() : null;
        }

        @Override
        public final Builder memoryMiB(MemoryMiBRequest memoryMiB) {
            this.memoryMiB = memoryMiB;
            return this;
        }

        public final Collection<String> getCpuManufacturers() {
            if (cpuManufacturers instanceof SdkAutoConstructList) {
                return null;
            }
            return cpuManufacturers;
        }

        public final void setCpuManufacturers(Collection<String> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copy(cpuManufacturers);
        }

        @Override
        public final Builder cpuManufacturersWithStrings(Collection<String> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copy(cpuManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuManufacturersWithStrings(String... cpuManufacturers) {
            cpuManufacturersWithStrings(Arrays.asList(cpuManufacturers));
            return this;
        }

        @Override
        public final Builder cpuManufacturers(Collection<CpuManufacturer> cpuManufacturers) {
            this.cpuManufacturers = CpuManufacturerSetCopier.copyEnumToString(cpuManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder cpuManufacturers(CpuManufacturer... cpuManufacturers) {
            cpuManufacturers(Arrays.asList(cpuManufacturers));
            return this;
        }

        public final MemoryGiBPerVCpuRequest.Builder getMemoryGiBPerVCpu() {
            return memoryGiBPerVCpu != null ? memoryGiBPerVCpu.toBuilder() : null;
        }

        public final void setMemoryGiBPerVCpu(MemoryGiBPerVCpuRequest.BuilderImpl memoryGiBPerVCpu) {
            this.memoryGiBPerVCpu = memoryGiBPerVCpu != null ? memoryGiBPerVCpu.build() : null;
        }

        @Override
        public final Builder memoryGiBPerVCpu(MemoryGiBPerVCpuRequest memoryGiBPerVCpu) {
            this.memoryGiBPerVCpu = memoryGiBPerVCpu;
            return this;
        }

        public final Collection<String> getExcludedInstanceTypes() {
            if (excludedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return excludedInstanceTypes;
        }

        public final void setExcludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = ExcludedInstanceTypeSetCopier.copy(excludedInstanceTypes);
        }

        @Override
        public final Builder excludedInstanceTypes(Collection<String> excludedInstanceTypes) {
            this.excludedInstanceTypes = ExcludedInstanceTypeSetCopier.copy(excludedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedInstanceTypes(String... excludedInstanceTypes) {
            excludedInstanceTypes(Arrays.asList(excludedInstanceTypes));
            return this;
        }

        public final Collection<String> getInstanceGenerations() {
            if (instanceGenerations instanceof SdkAutoConstructList) {
                return null;
            }
            return instanceGenerations;
        }

        public final void setInstanceGenerations(Collection<String> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copy(instanceGenerations);
        }

        @Override
        public final Builder instanceGenerationsWithStrings(Collection<String> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copy(instanceGenerations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGenerationsWithStrings(String... instanceGenerations) {
            instanceGenerationsWithStrings(Arrays.asList(instanceGenerations));
            return this;
        }

        @Override
        public final Builder instanceGenerations(Collection<InstanceGeneration> instanceGenerations) {
            this.instanceGenerations = InstanceGenerationSetCopier.copyEnumToString(instanceGenerations);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instanceGenerations(InstanceGeneration... instanceGenerations) {
            instanceGenerations(Arrays.asList(instanceGenerations));
            return this;
        }

        public final Integer getSpotMaxPricePercentageOverLowestPrice() {
            return spotMaxPricePercentageOverLowestPrice;
        }

        public final void setSpotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice) {
            this.spotMaxPricePercentageOverLowestPrice = spotMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final Builder spotMaxPricePercentageOverLowestPrice(Integer spotMaxPricePercentageOverLowestPrice) {
            this.spotMaxPricePercentageOverLowestPrice = spotMaxPricePercentageOverLowestPrice;
            return this;
        }

        public final Integer getOnDemandMaxPricePercentageOverLowestPrice() {
            return onDemandMaxPricePercentageOverLowestPrice;
        }

        public final void setOnDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice) {
            this.onDemandMaxPricePercentageOverLowestPrice = onDemandMaxPricePercentageOverLowestPrice;
        }

        @Override
        public final Builder onDemandMaxPricePercentageOverLowestPrice(Integer onDemandMaxPricePercentageOverLowestPrice) {
            this.onDemandMaxPricePercentageOverLowestPrice = onDemandMaxPricePercentageOverLowestPrice;
            return this;
        }

        public final String getBareMetal() {
            return bareMetal;
        }

        public final void setBareMetal(String bareMetal) {
            this.bareMetal = bareMetal;
        }

        @Override
        public final Builder bareMetal(String bareMetal) {
            this.bareMetal = bareMetal;
            return this;
        }

        @Override
        public final Builder bareMetal(BareMetal bareMetal) {
            this.bareMetal(bareMetal == null ? null : bareMetal.toString());
            return this;
        }

        public final String getBurstablePerformance() {
            return burstablePerformance;
        }

        public final void setBurstablePerformance(String burstablePerformance) {
            this.burstablePerformance = burstablePerformance;
        }

        @Override
        public final Builder burstablePerformance(String burstablePerformance) {
            this.burstablePerformance = burstablePerformance;
            return this;
        }

        @Override
        public final Builder burstablePerformance(BurstablePerformance burstablePerformance) {
            this.burstablePerformance(burstablePerformance == null ? null : burstablePerformance.toString());
            return this;
        }

        public final Boolean getRequireHibernateSupport() {
            return requireHibernateSupport;
        }

        public final void setRequireHibernateSupport(Boolean requireHibernateSupport) {
            this.requireHibernateSupport = requireHibernateSupport;
        }

        @Override
        public final Builder requireHibernateSupport(Boolean requireHibernateSupport) {
            this.requireHibernateSupport = requireHibernateSupport;
            return this;
        }

        public final NetworkInterfaceCountRequest.Builder getNetworkInterfaceCount() {
            return networkInterfaceCount != null ? networkInterfaceCount.toBuilder() : null;
        }

        public final void setNetworkInterfaceCount(NetworkInterfaceCountRequest.BuilderImpl networkInterfaceCount) {
            this.networkInterfaceCount = networkInterfaceCount != null ? networkInterfaceCount.build() : null;
        }

        @Override
        public final Builder networkInterfaceCount(NetworkInterfaceCountRequest networkInterfaceCount) {
            this.networkInterfaceCount = networkInterfaceCount;
            return this;
        }

        public final String getLocalStorage() {
            return localStorage;
        }

        public final void setLocalStorage(String localStorage) {
            this.localStorage = localStorage;
        }

        @Override
        public final Builder localStorage(String localStorage) {
            this.localStorage = localStorage;
            return this;
        }

        @Override
        public final Builder localStorage(LocalStorage localStorage) {
            this.localStorage(localStorage == null ? null : localStorage.toString());
            return this;
        }

        public final Collection<String> getLocalStorageTypes() {
            if (localStorageTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return localStorageTypes;
        }

        public final void setLocalStorageTypes(Collection<String> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copy(localStorageTypes);
        }

        @Override
        public final Builder localStorageTypesWithStrings(Collection<String> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copy(localStorageTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localStorageTypesWithStrings(String... localStorageTypes) {
            localStorageTypesWithStrings(Arrays.asList(localStorageTypes));
            return this;
        }

        @Override
        public final Builder localStorageTypes(Collection<LocalStorageType> localStorageTypes) {
            this.localStorageTypes = LocalStorageTypeSetCopier.copyEnumToString(localStorageTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder localStorageTypes(LocalStorageType... localStorageTypes) {
            localStorageTypes(Arrays.asList(localStorageTypes));
            return this;
        }

        public final TotalLocalStorageGBRequest.Builder getTotalLocalStorageGB() {
            return totalLocalStorageGB != null ? totalLocalStorageGB.toBuilder() : null;
        }

        public final void setTotalLocalStorageGB(TotalLocalStorageGBRequest.BuilderImpl totalLocalStorageGB) {
            this.totalLocalStorageGB = totalLocalStorageGB != null ? totalLocalStorageGB.build() : null;
        }

        @Override
        public final Builder totalLocalStorageGB(TotalLocalStorageGBRequest totalLocalStorageGB) {
            this.totalLocalStorageGB = totalLocalStorageGB;
            return this;
        }

        public final BaselineEbsBandwidthMbpsRequest.Builder getBaselineEbsBandwidthMbps() {
            return baselineEbsBandwidthMbps != null ? baselineEbsBandwidthMbps.toBuilder() : null;
        }

        public final void setBaselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest.BuilderImpl baselineEbsBandwidthMbps) {
            this.baselineEbsBandwidthMbps = baselineEbsBandwidthMbps != null ? baselineEbsBandwidthMbps.build() : null;
        }

        @Override
        public final Builder baselineEbsBandwidthMbps(BaselineEbsBandwidthMbpsRequest baselineEbsBandwidthMbps) {
            this.baselineEbsBandwidthMbps = baselineEbsBandwidthMbps;
            return this;
        }

        public final Collection<String> getAcceleratorTypes() {
            if (acceleratorTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorTypes;
        }

        public final void setAcceleratorTypes(Collection<String> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copy(acceleratorTypes);
        }

        @Override
        public final Builder acceleratorTypesWithStrings(Collection<String> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copy(acceleratorTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorTypesWithStrings(String... acceleratorTypes) {
            acceleratorTypesWithStrings(Arrays.asList(acceleratorTypes));
            return this;
        }

        @Override
        public final Builder acceleratorTypes(Collection<AcceleratorType> acceleratorTypes) {
            this.acceleratorTypes = AcceleratorTypeSetCopier.copyEnumToString(acceleratorTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorTypes(AcceleratorType... acceleratorTypes) {
            acceleratorTypes(Arrays.asList(acceleratorTypes));
            return this;
        }

        public final AcceleratorCountRequest.Builder getAcceleratorCount() {
            return acceleratorCount != null ? acceleratorCount.toBuilder() : null;
        }

        public final void setAcceleratorCount(AcceleratorCountRequest.BuilderImpl acceleratorCount) {
            this.acceleratorCount = acceleratorCount != null ? acceleratorCount.build() : null;
        }

        @Override
        public final Builder acceleratorCount(AcceleratorCountRequest acceleratorCount) {
            this.acceleratorCount = acceleratorCount;
            return this;
        }

        public final Collection<String> getAcceleratorManufacturers() {
            if (acceleratorManufacturers instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorManufacturers;
        }

        public final void setAcceleratorManufacturers(Collection<String> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copy(acceleratorManufacturers);
        }

        @Override
        public final Builder acceleratorManufacturersWithStrings(Collection<String> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copy(acceleratorManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorManufacturersWithStrings(String... acceleratorManufacturers) {
            acceleratorManufacturersWithStrings(Arrays.asList(acceleratorManufacturers));
            return this;
        }

        @Override
        public final Builder acceleratorManufacturers(Collection<AcceleratorManufacturer> acceleratorManufacturers) {
            this.acceleratorManufacturers = AcceleratorManufacturerSetCopier.copyEnumToString(acceleratorManufacturers);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorManufacturers(AcceleratorManufacturer... acceleratorManufacturers) {
            acceleratorManufacturers(Arrays.asList(acceleratorManufacturers));
            return this;
        }

        public final Collection<String> getAcceleratorNames() {
            if (acceleratorNames instanceof SdkAutoConstructList) {
                return null;
            }
            return acceleratorNames;
        }

        public final void setAcceleratorNames(Collection<String> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copy(acceleratorNames);
        }

        @Override
        public final Builder acceleratorNamesWithStrings(Collection<String> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copy(acceleratorNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorNamesWithStrings(String... acceleratorNames) {
            acceleratorNamesWithStrings(Arrays.asList(acceleratorNames));
            return this;
        }

        @Override
        public final Builder acceleratorNames(Collection<AcceleratorName> acceleratorNames) {
            this.acceleratorNames = AcceleratorNameSetCopier.copyEnumToString(acceleratorNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder acceleratorNames(AcceleratorName... acceleratorNames) {
            acceleratorNames(Arrays.asList(acceleratorNames));
            return this;
        }

        public final AcceleratorTotalMemoryMiBRequest.Builder getAcceleratorTotalMemoryMiB() {
            return acceleratorTotalMemoryMiB != null ? acceleratorTotalMemoryMiB.toBuilder() : null;
        }

        public final void setAcceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest.BuilderImpl acceleratorTotalMemoryMiB) {
            this.acceleratorTotalMemoryMiB = acceleratorTotalMemoryMiB != null ? acceleratorTotalMemoryMiB.build() : null;
        }

        @Override
        public final Builder acceleratorTotalMemoryMiB(AcceleratorTotalMemoryMiBRequest acceleratorTotalMemoryMiB) {
            this.acceleratorTotalMemoryMiB = acceleratorTotalMemoryMiB;
            return this;
        }

        public final NetworkBandwidthGbpsRequest.Builder getNetworkBandwidthGbps() {
            return networkBandwidthGbps != null ? networkBandwidthGbps.toBuilder() : null;
        }

        public final void setNetworkBandwidthGbps(NetworkBandwidthGbpsRequest.BuilderImpl networkBandwidthGbps) {
            this.networkBandwidthGbps = networkBandwidthGbps != null ? networkBandwidthGbps.build() : null;
        }

        @Override
        public final Builder networkBandwidthGbps(NetworkBandwidthGbpsRequest networkBandwidthGbps) {
            this.networkBandwidthGbps = networkBandwidthGbps;
            return this;
        }

        public final Collection<String> getAllowedInstanceTypes() {
            if (allowedInstanceTypes instanceof SdkAutoConstructList) {
                return null;
            }
            return allowedInstanceTypes;
        }

        public final void setAllowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = AllowedInstanceTypeSetCopier.copy(allowedInstanceTypes);
        }

        @Override
        public final Builder allowedInstanceTypes(Collection<String> allowedInstanceTypes) {
            this.allowedInstanceTypes = AllowedInstanceTypeSetCopier.copy(allowedInstanceTypes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder allowedInstanceTypes(String... allowedInstanceTypes) {
            allowedInstanceTypes(Arrays.asList(allowedInstanceTypes));
            return this;
        }

        public final Integer getMaxSpotPriceAsPercentageOfOptimalOnDemandPrice() {
            return maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        }

        public final void setMaxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice) {
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
        }

        @Override
        public final Builder maxSpotPriceAsPercentageOfOptimalOnDemandPrice(Integer maxSpotPriceAsPercentageOfOptimalOnDemandPrice) {
            this.maxSpotPriceAsPercentageOfOptimalOnDemandPrice = maxSpotPriceAsPercentageOfOptimalOnDemandPrice;
            return this;
        }

        @Override
        public InstanceRequirementsRequest build() {
            return new InstanceRequirementsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
