/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for CopyImage.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CopyImageRequest extends Ec2Request implements ToCopyableBuilder<CopyImageRequest.Builder, CopyImageRequest> {
    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(CopyImageRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("ClientToken").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(CopyImageRequest::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("Description").build()).build();

    private static final SdkField<Boolean> ENCRYPTED_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Encrypted")
            .getter(getter(CopyImageRequest::encrypted))
            .setter(setter(Builder::encrypted))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Encrypted")
                    .unmarshallLocationName("encrypted").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("KmsKeyId")
            .getter(getter(CopyImageRequest::kmsKeyId))
            .setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId")
                    .unmarshallLocationName("kmsKeyId").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Name")
            .getter(getter(CopyImageRequest::name))
            .setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name")
                    .unmarshallLocationName("Name").build()).build();

    private static final SdkField<String> SOURCE_IMAGE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceImageId")
            .getter(getter(CopyImageRequest::sourceImageId))
            .setter(setter(Builder::sourceImageId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceImageId")
                    .unmarshallLocationName("SourceImageId").build()).build();

    private static final SdkField<String> SOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SourceRegion")
            .getter(getter(CopyImageRequest::sourceRegion))
            .setter(setter(Builder::sourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceRegion")
                    .unmarshallLocationName("SourceRegion").build()).build();

    private static final SdkField<String> DESTINATION_OUTPOST_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("DestinationOutpostArn")
            .getter(getter(CopyImageRequest::destinationOutpostArn))
            .setter(setter(Builder::destinationOutpostArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DestinationOutpostArn")
                    .unmarshallLocationName("DestinationOutpostArn").build()).build();

    private static final SdkField<Boolean> COPY_IMAGE_TAGS_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CopyImageTags")
            .getter(getter(CopyImageRequest::copyImageTags))
            .setter(setter(Builder::copyImageTags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CopyImageTags")
                    .unmarshallLocationName("CopyImageTags").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(CopyImageRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(CopyImageRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("dryRun").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLIENT_TOKEN_FIELD,
            DESCRIPTION_FIELD, ENCRYPTED_FIELD, KMS_KEY_ID_FIELD, NAME_FIELD, SOURCE_IMAGE_ID_FIELD, SOURCE_REGION_FIELD,
            DESTINATION_OUTPOST_ARN_FIELD, COPY_IMAGE_TAGS_FIELD, TAG_SPECIFICATIONS_FIELD, DRY_RUN_FIELD));

    private final String clientToken;

    private final String description;

    private final Boolean encrypted;

    private final String kmsKeyId;

    private final String name;

    private final String sourceImageId;

    private final String sourceRegion;

    private final String destinationOutpostArn;

    private final Boolean copyImageTags;

    private final List<TagSpecification> tagSpecifications;

    private final Boolean dryRun;

    private CopyImageRequest(BuilderImpl builder) {
        super(builder);
        this.clientToken = builder.clientToken;
        this.description = builder.description;
        this.encrypted = builder.encrypted;
        this.kmsKeyId = builder.kmsKeyId;
        this.name = builder.name;
        this.sourceImageId = builder.sourceImageId;
        this.sourceRegion = builder.sourceRegion;
        this.destinationOutpostArn = builder.destinationOutpostArn;
        this.copyImageTags = builder.copyImageTags;
        this.tagSpecifications = builder.tagSpecifications;
        this.dryRun = builder.dryRun;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier you provide to ensure idempotency of the request. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     * idempotency</a> in the <i>Amazon EC2 API Reference</i>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier you provide to ensure idempotency of the request. For more information,
     *         see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
     *         idempotency</a> in the <i>Amazon EC2 API Reference</i>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * A description for the new AMI in the destination Region.
     * </p>
     * 
     * @return A description for the new AMI in the destination Region.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * Specifies whether the destination snapshots of the copied image should be encrypted. You can encrypt a copy of an
     * unencrypted snapshot, but you cannot create an unencrypted copy of an encrypted snapshot. The default KMS key for
     * Amazon EBS is used unless you specify a non-default Key Management Service (KMS) KMS key using
     * <code>KmsKeyId</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIEncryption.html">Use encryption with EBS-backed
     * AMIs</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return Specifies whether the destination snapshots of the copied image should be encrypted. You can encrypt a
     *         copy of an unencrypted snapshot, but you cannot create an unencrypted copy of an encrypted snapshot. The
     *         default KMS key for Amazon EBS is used unless you specify a non-default Key Management Service (KMS) KMS
     *         key using <code>KmsKeyId</code>. For more information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIEncryption.html">Use encryption with
     *         EBS-backed AMIs</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final Boolean encrypted() {
        return encrypted;
    }

    /**
     * <p>
     * The identifier of the symmetric Key Management Service (KMS) KMS key to use when creating encrypted volumes. If
     * this parameter is not specified, your Amazon Web Services managed KMS key for Amazon EBS is used. If you specify
     * a KMS key, you must also set the encrypted state to <code>true</code>.
     * </p>
     * <p>
     * You can specify a KMS key using any of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
     * </p>
     * </li>
     * <li>
     * <p>
     * Key alias. For example, alias/ExampleAlias.
     * </p>
     * </li>
     * <li>
     * <p>
     * Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
     * </p>
     * </li>
     * <li>
     * <p>
     * Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an identifier that is not
     * valid, the action can appear to complete, but eventually fails.
     * </p>
     * <p>
     * The specified KMS key must exist in the destination Region.
     * </p>
     * <p>
     * Amazon EBS does not support asymmetric KMS keys.
     * </p>
     * 
     * @return The identifier of the symmetric Key Management Service (KMS) KMS key to use when creating encrypted
     *         volumes. If this parameter is not specified, your Amazon Web Services managed KMS key for Amazon EBS is
     *         used. If you specify a KMS key, you must also set the encrypted state to <code>true</code>.</p>
     *         <p>
     *         You can specify a KMS key using any of the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key alias. For example, alias/ExampleAlias.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an identifier
     *         that is not valid, the action can appear to complete, but eventually fails.
     *         </p>
     *         <p>
     *         The specified KMS key must exist in the destination Region.
     *         </p>
     *         <p>
     *         Amazon EBS does not support asymmetric KMS keys.
     */
    public final String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The name of the new AMI in the destination Region.
     * </p>
     * 
     * @return The name of the new AMI in the destination Region.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The ID of the AMI to copy.
     * </p>
     * 
     * @return The ID of the AMI to copy.
     */
    public final String sourceImageId() {
        return sourceImageId;
    }

    /**
     * <p>
     * The name of the Region that contains the AMI to copy.
     * </p>
     * 
     * @return The name of the Region that contains the AMI to copy.
     */
    public final String sourceRegion() {
        return sourceRegion;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI. Only specify this parameter when copying
     * an AMI from an Amazon Web Services Region to an Outpost. The AMI must be in the Region of the destination
     * Outpost. You cannot copy an AMI from an Outpost to a Region, from one Outpost to another, or within the same
     * Outpost.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-amis">Copy AMIs from an
     * Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI. Only specify this parameter when
     *         copying an AMI from an Amazon Web Services Region to an Outpost. The AMI must be in the Region of the
     *         destination Outpost. You cannot copy an AMI from an Outpost to a Region, from one Outpost to another, or
     *         within the same Outpost.</p>
     *         <p>
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-amis">Copy AMIs from
     *         an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
     */
    public final String destinationOutpostArn() {
        return destinationOutpostArn;
    }

    /**
     * <p>
     * Indicates whether to include your user-defined AMI tags when copying the AMI.
     * </p>
     * <p>
     * The following tags will not be copied:
     * </p>
     * <ul>
     * <li>
     * <p>
     * System tags (prefixed with <code>aws:</code>)
     * </p>
     * </li>
     * <li>
     * <p>
     * For public and shared AMIs, user-defined tags that are attached by other Amazon Web Services accounts
     * </p>
     * </li>
     * </ul>
     * <p>
     * Default: Your user-defined AMI tags are not copied.
     * </p>
     * 
     * @return Indicates whether to include your user-defined AMI tags when copying the AMI.</p>
     *         <p>
     *         The following tags will not be copied:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         System tags (prefixed with <code>aws:</code>)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For public and shared AMIs, user-defined tags that are attached by other Amazon Web Services accounts
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Default: Your user-defined AMI tags are not copied.
     */
    public final Boolean copyImageTags() {
        return copyImageTags;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.
     * </p>
     * <ul>
     * <li>
     * <p>
     * To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag is
     * applied to all the new snapshots.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you specify other values for <code>ResourceType</code>, the request fails.
     * </p>
     * <p>
     * To tag an AMI or snapshot after it has been created, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same
     *         tag is applied to all the new snapshots.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you specify other values for <code>ResourceType</code>, the request fails.
     *         </p>
     *         <p>
     *         To tag an AMI or snapshot after it has been created, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(encrypted());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(sourceImageId());
        hashCode = 31 * hashCode + Objects.hashCode(sourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(destinationOutpostArn());
        hashCode = 31 * hashCode + Objects.hashCode(copyImageTags());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CopyImageRequest)) {
            return false;
        }
        CopyImageRequest other = (CopyImageRequest) obj;
        return Objects.equals(clientToken(), other.clientToken()) && Objects.equals(description(), other.description())
                && Objects.equals(encrypted(), other.encrypted()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(name(), other.name()) && Objects.equals(sourceImageId(), other.sourceImageId())
                && Objects.equals(sourceRegion(), other.sourceRegion())
                && Objects.equals(destinationOutpostArn(), other.destinationOutpostArn())
                && Objects.equals(copyImageTags(), other.copyImageTags())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications()) && Objects.equals(dryRun(), other.dryRun());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CopyImageRequest").add("ClientToken", clientToken()).add("Description", description())
                .add("Encrypted", encrypted()).add("KmsKeyId", kmsKeyId()).add("Name", name())
                .add("SourceImageId", sourceImageId()).add("SourceRegion", sourceRegion())
                .add("DestinationOutpostArn", destinationOutpostArn()).add("CopyImageTags", copyImageTags())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null).add("DryRun", dryRun()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Encrypted":
            return Optional.ofNullable(clazz.cast(encrypted()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "SourceImageId":
            return Optional.ofNullable(clazz.cast(sourceImageId()));
        case "SourceRegion":
            return Optional.ofNullable(clazz.cast(sourceRegion()));
        case "DestinationOutpostArn":
            return Optional.ofNullable(clazz.cast(destinationOutpostArn()));
        case "CopyImageTags":
            return Optional.ofNullable(clazz.cast(copyImageTags()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CopyImageRequest, T> g) {
        return obj -> g.apply((CopyImageRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, CopyImageRequest> {
        /**
         * <p>
         * Unique, case-sensitive identifier you provide to ensure idempotency of the request. For more information, see
         * <a href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         * idempotency</a> in the <i>Amazon EC2 API Reference</i>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier you provide to ensure idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/Run_Instance_Idempotency.html">Ensuring
         *        idempotency</a> in the <i>Amazon EC2 API Reference</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * A description for the new AMI in the destination Region.
         * </p>
         * 
         * @param description
         *        A description for the new AMI in the destination Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * Specifies whether the destination snapshots of the copied image should be encrypted. You can encrypt a copy
         * of an unencrypted snapshot, but you cannot create an unencrypted copy of an encrypted snapshot. The default
         * KMS key for Amazon EBS is used unless you specify a non-default Key Management Service (KMS) KMS key using
         * <code>KmsKeyId</code>. For more information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIEncryption.html">Use encryption with EBS-backed
         * AMIs</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param encrypted
         *        Specifies whether the destination snapshots of the copied image should be encrypted. You can encrypt a
         *        copy of an unencrypted snapshot, but you cannot create an unencrypted copy of an encrypted snapshot.
         *        The default KMS key for Amazon EBS is used unless you specify a non-default Key Management Service
         *        (KMS) KMS key using <code>KmsKeyId</code>. For more information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/AMIEncryption.html">Use encryption with
         *        EBS-backed AMIs</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encrypted(Boolean encrypted);

        /**
         * <p>
         * The identifier of the symmetric Key Management Service (KMS) KMS key to use when creating encrypted volumes.
         * If this parameter is not specified, your Amazon Web Services managed KMS key for Amazon EBS is used. If you
         * specify a KMS key, you must also set the encrypted state to <code>true</code>.
         * </p>
         * <p>
         * You can specify a KMS key using any of the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
         * </p>
         * </li>
         * <li>
         * <p>
         * Key alias. For example, alias/ExampleAlias.
         * </p>
         * </li>
         * <li>
         * <p>
         * Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
         * </p>
         * </li>
         * <li>
         * <p>
         * Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an identifier that is
         * not valid, the action can appear to complete, but eventually fails.
         * </p>
         * <p>
         * The specified KMS key must exist in the destination Region.
         * </p>
         * <p>
         * Amazon EBS does not support asymmetric KMS keys.
         * </p>
         * 
         * @param kmsKeyId
         *        The identifier of the symmetric Key Management Service (KMS) KMS key to use when creating encrypted
         *        volumes. If this parameter is not specified, your Amazon Web Services managed KMS key for Amazon EBS
         *        is used. If you specify a KMS key, you must also set the encrypted state to <code>true</code>.</p>
         *        <p>
         *        You can specify a KMS key using any of the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Key ID. For example, 1234abcd-12ab-34cd-56ef-1234567890ab.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key alias. For example, alias/ExampleAlias.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Key ARN. For example, arn:aws:kms:us-east-1:012345678910:key/1234abcd-12ab-34cd-56ef-1234567890ab.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Alias ARN. For example, arn:aws:kms:us-east-1:012345678910:alias/ExampleAlias.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Amazon Web Services authenticates the KMS key asynchronously. Therefore, if you specify an identifier
         *        that is not valid, the action can appear to complete, but eventually fails.
         *        </p>
         *        <p>
         *        The specified KMS key must exist in the destination Region.
         *        </p>
         *        <p>
         *        Amazon EBS does not support asymmetric KMS keys.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The name of the new AMI in the destination Region.
         * </p>
         * 
         * @param name
         *        The name of the new AMI in the destination Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The ID of the AMI to copy.
         * </p>
         * 
         * @param sourceImageId
         *        The ID of the AMI to copy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceImageId(String sourceImageId);

        /**
         * <p>
         * The name of the Region that contains the AMI to copy.
         * </p>
         * 
         * @param sourceRegion
         *        The name of the Region that contains the AMI to copy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceRegion(String sourceRegion);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI. Only specify this parameter when
         * copying an AMI from an Amazon Web Services Region to an Outpost. The AMI must be in the Region of the
         * destination Outpost. You cannot copy an AMI from an Outpost to a Region, from one Outpost to another, or
         * within the same Outpost.
         * </p>
         * <p>
         * For more information, see <a
         * href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-amis">Copy AMIs from an
         * Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
         * </p>
         * 
         * @param destinationOutpostArn
         *        The Amazon Resource Name (ARN) of the Outpost to which to copy the AMI. Only specify this parameter
         *        when copying an AMI from an Amazon Web Services Region to an Outpost. The AMI must be in the Region of
         *        the destination Outpost. You cannot copy an AMI from an Outpost to a Region, from one Outpost to
         *        another, or within the same Outpost.</p>
         *        <p>
         *        For more information, see <a
         *        href="https://docs.aws.amazon.com/ebs/latest/userguide/snapshots-outposts.html#copy-amis">Copy AMIs
         *        from an Amazon Web Services Region to an Outpost</a> in the <i>Amazon EBS User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder destinationOutpostArn(String destinationOutpostArn);

        /**
         * <p>
         * Indicates whether to include your user-defined AMI tags when copying the AMI.
         * </p>
         * <p>
         * The following tags will not be copied:
         * </p>
         * <ul>
         * <li>
         * <p>
         * System tags (prefixed with <code>aws:</code>)
         * </p>
         * </li>
         * <li>
         * <p>
         * For public and shared AMIs, user-defined tags that are attached by other Amazon Web Services accounts
         * </p>
         * </li>
         * </ul>
         * <p>
         * Default: Your user-defined AMI tags are not copied.
         * </p>
         * 
         * @param copyImageTags
         *        Indicates whether to include your user-defined AMI tags when copying the AMI.</p>
         *        <p>
         *        The following tags will not be copied:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        System tags (prefixed with <code>aws:</code>)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For public and shared AMIs, user-defined tags that are attached by other Amazon Web Services accounts
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Default: Your user-defined AMI tags are not copied.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder copyImageTags(Boolean copyImageTags);

        /**
         * <p>
         * The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag
         * is applied to all the new snapshots.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify other values for <code>ResourceType</code>, the request fails.
         * </p>
         * <p>
         * To tag an AMI or snapshot after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The
         *        same tag is applied to all the new snapshots.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify other values for <code>ResourceType</code>, the request fails.
         *        </p>
         *        <p>
         *        To tag an AMI or snapshot after it has been created, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag
         * is applied to all the new snapshots.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify other values for <code>ResourceType</code>, the request fails.
         * </p>
         * <p>
         * To tag an AMI or snapshot after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * 
         * @param tagSpecifications
         *        The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The
         *        same tag is applied to all the new snapshots.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you specify other values for <code>ResourceType</code>, the request fails.
         *        </p>
         *        <p>
         *        To tag an AMI or snapshot after it has been created, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The tags to apply to the new AMI and new snapshots. You can tag the AMI, the snapshots, or both.
         * </p>
         * <ul>
         * <li>
         * <p>
         * To tag the new AMI, the value for <code>ResourceType</code> must be <code>image</code>.
         * </p>
         * </li>
         * <li>
         * <p>
         * To tag the new snapshots, the value for <code>ResourceType</code> must be <code>snapshot</code>. The same tag
         * is applied to all the new snapshots.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you specify other values for <code>ResourceType</code>, the request fails.
         * </p>
         * <p>
         * To tag an AMI or snapshot after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String clientToken;

        private String description;

        private Boolean encrypted;

        private String kmsKeyId;

        private String name;

        private String sourceImageId;

        private String sourceRegion;

        private String destinationOutpostArn;

        private Boolean copyImageTags;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private BuilderImpl() {
        }

        private BuilderImpl(CopyImageRequest model) {
            super(model);
            clientToken(model.clientToken);
            description(model.description);
            encrypted(model.encrypted);
            kmsKeyId(model.kmsKeyId);
            name(model.name);
            sourceImageId(model.sourceImageId);
            sourceRegion(model.sourceRegion);
            destinationOutpostArn(model.destinationOutpostArn);
            copyImageTags(model.copyImageTags);
            tagSpecifications(model.tagSpecifications);
            dryRun(model.dryRun);
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Boolean getEncrypted() {
            return encrypted;
        }

        public final void setEncrypted(Boolean encrypted) {
            this.encrypted = encrypted;
        }

        @Override
        public final Builder encrypted(Boolean encrypted) {
            this.encrypted = encrypted;
            return this;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getSourceImageId() {
            return sourceImageId;
        }

        public final void setSourceImageId(String sourceImageId) {
            this.sourceImageId = sourceImageId;
        }

        @Override
        public final Builder sourceImageId(String sourceImageId) {
            this.sourceImageId = sourceImageId;
            return this;
        }

        public final String getSourceRegion() {
            return sourceRegion;
        }

        public final void setSourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
        }

        @Override
        public final Builder sourceRegion(String sourceRegion) {
            this.sourceRegion = sourceRegion;
            return this;
        }

        public final String getDestinationOutpostArn() {
            return destinationOutpostArn;
        }

        public final void setDestinationOutpostArn(String destinationOutpostArn) {
            this.destinationOutpostArn = destinationOutpostArn;
        }

        @Override
        public final Builder destinationOutpostArn(String destinationOutpostArn) {
            this.destinationOutpostArn = destinationOutpostArn;
            return this;
        }

        public final Boolean getCopyImageTags() {
            return copyImageTags;
        }

        public final void setCopyImageTags(Boolean copyImageTags) {
            this.copyImageTags = copyImageTags;
        }

        @Override
        public final Builder copyImageTags(Boolean copyImageTags) {
            this.copyImageTags = copyImageTags;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CopyImageRequest build() {
            return new CopyImageRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
