/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for DescribeReservedInstancesOfferings.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeReservedInstancesOfferingsRequest extends Ec2Request implements
        ToCopyableBuilder<DescribeReservedInstancesOfferingsRequest.Builder, DescribeReservedInstancesOfferingsRequest> {
    private static final SdkField<String> AVAILABILITY_ZONE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZone")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::availabilityZone))
            .setter(setter(Builder::availabilityZone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZone")
                    .unmarshallLocationName("AvailabilityZone").build()).build();

    private static final SdkField<Boolean> INCLUDE_MARKETPLACE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IncludeMarketplace")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::includeMarketplace))
            .setter(setter(Builder::includeMarketplace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IncludeMarketplace")
                    .unmarshallLocationName("IncludeMarketplace").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("InstanceType").build()).build();

    private static final SdkField<Long> MAX_DURATION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MaxDuration")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::maxDuration))
            .setter(setter(Builder::maxDuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxDuration")
                    .unmarshallLocationName("MaxDuration").build()).build();

    private static final SdkField<Integer> MAX_INSTANCE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxInstanceCount")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::maxInstanceCount))
            .setter(setter(Builder::maxInstanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxInstanceCount")
                    .unmarshallLocationName("MaxInstanceCount").build()).build();

    private static final SdkField<Long> MIN_DURATION_FIELD = SdkField
            .<Long> builder(MarshallingType.LONG)
            .memberName("MinDuration")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::minDuration))
            .setter(setter(Builder::minDuration))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinDuration")
                    .unmarshallLocationName("MinDuration").build()).build();

    private static final SdkField<String> OFFERING_CLASS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OfferingClass")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::offeringClassAsString))
            .setter(setter(Builder::offeringClass))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OfferingClass")
                    .unmarshallLocationName("OfferingClass").build()).build();

    private static final SdkField<String> PRODUCT_DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ProductDescription")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::productDescriptionAsString))
            .setter(setter(Builder::productDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ProductDescription")
                    .unmarshallLocationName("ProductDescription").build()).build();

    private static final SdkField<List<String>> RESERVED_INSTANCES_OFFERING_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ReservedInstancesOfferingIds")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::reservedInstancesOfferingIds))
            .setter(setter(Builder::reservedInstancesOfferingIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservedInstancesOfferingId")
                    .unmarshallLocationName("ReservedInstancesOfferingId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").unmarshallLocationName("member").build()).build())
                            .build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("dryRun").build()).build();

    private static final SdkField<List<Filter>> FILTERS_FIELD = SdkField
            .<List<Filter>> builder(MarshallingType.LIST)
            .memberName("Filters")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filter")
                    .unmarshallLocationName("Filter").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("Filter")
                            .memberFieldInfo(
                                    SdkField.<Filter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Filter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Filter").unmarshallLocationName("Filter").build()).build())
                            .build()).build();

    private static final SdkField<String> INSTANCE_TENANCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceTenancy")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::instanceTenancyAsString))
            .setter(setter(Builder::instanceTenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceTenancy")
                    .unmarshallLocationName("instanceTenancy").build()).build();

    private static final SdkField<String> OFFERING_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OfferingType")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::offeringTypeAsString))
            .setter(setter(Builder::offeringType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OfferingType")
                    .unmarshallLocationName("offeringType").build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("NextToken")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::nextToken))
            .setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken")
                    .unmarshallLocationName("nextToken").build()).build();

    private static final SdkField<Integer> MAX_RESULTS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("MaxResults")
            .getter(getter(DescribeReservedInstancesOfferingsRequest::maxResults))
            .setter(setter(Builder::maxResults))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MaxResults")
                    .unmarshallLocationName("maxResults").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AVAILABILITY_ZONE_FIELD,
            INCLUDE_MARKETPLACE_FIELD, INSTANCE_TYPE_FIELD, MAX_DURATION_FIELD, MAX_INSTANCE_COUNT_FIELD, MIN_DURATION_FIELD,
            OFFERING_CLASS_FIELD, PRODUCT_DESCRIPTION_FIELD, RESERVED_INSTANCES_OFFERING_IDS_FIELD, DRY_RUN_FIELD, FILTERS_FIELD,
            INSTANCE_TENANCY_FIELD, OFFERING_TYPE_FIELD, NEXT_TOKEN_FIELD, MAX_RESULTS_FIELD));

    private final String availabilityZone;

    private final Boolean includeMarketplace;

    private final String instanceType;

    private final Long maxDuration;

    private final Integer maxInstanceCount;

    private final Long minDuration;

    private final String offeringClass;

    private final String productDescription;

    private final List<String> reservedInstancesOfferingIds;

    private final Boolean dryRun;

    private final List<Filter> filters;

    private final String instanceTenancy;

    private final String offeringType;

    private final String nextToken;

    private final Integer maxResults;

    private DescribeReservedInstancesOfferingsRequest(BuilderImpl builder) {
        super(builder);
        this.availabilityZone = builder.availabilityZone;
        this.includeMarketplace = builder.includeMarketplace;
        this.instanceType = builder.instanceType;
        this.maxDuration = builder.maxDuration;
        this.maxInstanceCount = builder.maxInstanceCount;
        this.minDuration = builder.minDuration;
        this.offeringClass = builder.offeringClass;
        this.productDescription = builder.productDescription;
        this.reservedInstancesOfferingIds = builder.reservedInstancesOfferingIds;
        this.dryRun = builder.dryRun;
        this.filters = builder.filters;
        this.instanceTenancy = builder.instanceTenancy;
        this.offeringType = builder.offeringType;
        this.nextToken = builder.nextToken;
        this.maxResults = builder.maxResults;
    }

    /**
     * <p>
     * The Availability Zone in which the Reserved Instance can be used.
     * </p>
     * 
     * @return The Availability Zone in which the Reserved Instance can be used.
     */
    public final String availabilityZone() {
        return availabilityZone;
    }

    /**
     * <p>
     * Include Reserved Instance Marketplace offerings in the response.
     * </p>
     * 
     * @return Include Reserved Instance Marketplace offerings in the response.
     */
    public final Boolean includeMarketplace() {
        return includeMarketplace;
    }

    /**
     * <p>
     * The instance type that the reservation will cover (for example, <code>m1.small</code>). For more information, see
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance types</a>
     * in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type that the reservation will cover (for example, <code>m1.small</code>). For more
     *         information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon
     *         EC2 instance types</a> in the <i>Amazon EC2 User Guide</i>.
     * @see InstanceType
     */
    public final InstanceType instanceType() {
        return InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type that the reservation will cover (for example, <code>m1.small</code>). For more information, see
     * <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance types</a>
     * in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type that the reservation will cover (for example, <code>m1.small</code>). For more
     *         information, see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon
     *         EC2 instance types</a> in the <i>Amazon EC2 User Guide</i>.
     * @see InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The maximum duration (in seconds) to filter when searching for offerings.
     * </p>
     * <p>
     * Default: 94608000 (3 years)
     * </p>
     * 
     * @return The maximum duration (in seconds) to filter when searching for offerings.</p>
     *         <p>
     *         Default: 94608000 (3 years)
     */
    public final Long maxDuration() {
        return maxDuration;
    }

    /**
     * <p>
     * The maximum number of instances to filter when searching for offerings.
     * </p>
     * <p>
     * Default: 20
     * </p>
     * 
     * @return The maximum number of instances to filter when searching for offerings.</p>
     *         <p>
     *         Default: 20
     */
    public final Integer maxInstanceCount() {
        return maxInstanceCount;
    }

    /**
     * <p>
     * The minimum duration (in seconds) to filter when searching for offerings.
     * </p>
     * <p>
     * Default: 2592000 (1 month)
     * </p>
     * 
     * @return The minimum duration (in seconds) to filter when searching for offerings.</p>
     *         <p>
     *         Default: 2592000 (1 month)
     */
    public final Long minDuration() {
        return minDuration;
    }

    /**
     * <p>
     * The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #offeringClass}
     * will return {@link OfferingClassType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #offeringClassAsString}.
     * </p>
     * 
     * @return The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
     * @see OfferingClassType
     */
    public final OfferingClassType offeringClass() {
        return OfferingClassType.fromValue(offeringClass);
    }

    /**
     * <p>
     * The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #offeringClass}
     * will return {@link OfferingClassType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #offeringClassAsString}.
     * </p>
     * 
     * @return The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
     * @see OfferingClassType
     */
    public final String offeringClassAsString() {
        return offeringClass;
    }

    /**
     * <p>
     * The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in the
     * description are for use with Amazon VPC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #productDescription} will return {@link RIProductDescription#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #productDescriptionAsString}.
     * </p>
     * 
     * @return The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in
     *         the description are for use with Amazon VPC.
     * @see RIProductDescription
     */
    public final RIProductDescription productDescription() {
        return RIProductDescription.fromValue(productDescription);
    }

    /**
     * <p>
     * The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in the
     * description are for use with Amazon VPC.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #productDescription} will return {@link RIProductDescription#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #productDescriptionAsString}.
     * </p>
     * 
     * @return The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in
     *         the description are for use with Amazon VPC.
     * @see RIProductDescription
     */
    public final String productDescriptionAsString() {
        return productDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReservedInstancesOfferingIds property.
     * This DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasReservedInstancesOfferingIds() {
        return reservedInstancesOfferingIds != null && !(reservedInstancesOfferingIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more Reserved Instances offering IDs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReservedInstancesOfferingIds} method.
     * </p>
     * 
     * @return One or more Reserved Instances offering IDs.
     */
    public final List<String> reservedInstancesOfferingIds() {
        return reservedInstancesOfferingIds;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more filters.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years), in seconds
     * (<code>31536000</code> | <code>94608000</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>instance-type</code> - The instance type that is covered by the reservation.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace offerings. When
     * this filter is not used, which is the default behavior, all offerings from both Amazon Web Services and the
     * Reserved Instance Marketplace are listed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>product-description</code> - The Reserved Instance product platform description (<code>Linux/UNIX</code> |
     * <code>Linux with SQL Server Standard</code> | <code>Linux with SQL Server Web</code> |
     * <code>Linux with SQL Server Enterprise</code> | <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
     * <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
     * <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
     * <code>Windows with SQL Server Enterprise</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or <code>Region</code>).
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return One or more filters.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years), in
     *         seconds (<code>31536000</code> | <code>94608000</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>instance-type</code> - The instance type that is covered by the reservation.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace offerings.
     *         When this filter is not used, which is the default behavior, all offerings from both Amazon Web Services
     *         and the Reserved Instance Marketplace are listed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>product-description</code> - The Reserved Instance product platform description (
     *         <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
     *         <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
     *         <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
     *         <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
     *         <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
     *         <code>Windows with SQL Server Enterprise</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
     *         <code>Region</code>).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
     *         </p>
     *         </li>
     */
    public final List<Filter> filters() {
        return filters;
    }

    /**
     * <p>
     * The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
     * <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e., Dedicated
     * Instances).
     * </p>
     * <p>
     * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the <code>default</code> or
     * <code>dedicated</code> values only.
     * </p>
     * <p>
     * Default: <code>default</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceTenancy}
     * will return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
     *         <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e.,
     *         Dedicated Instances).</p>
     *         <p>
     *         <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
     *         <code>default</code> or <code>dedicated</code> values only.
     *         </p>
     *         <p>
     *         Default: <code>default</code>
     * @see Tenancy
     */
    public final Tenancy instanceTenancy() {
        return Tenancy.fromValue(instanceTenancy);
    }

    /**
     * <p>
     * The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
     * <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e., Dedicated
     * Instances).
     * </p>
     * <p>
     * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the <code>default</code> or
     * <code>dedicated</code> values only.
     * </p>
     * <p>
     * Default: <code>default</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceTenancy}
     * will return {@link Tenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
     *         <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e.,
     *         Dedicated Instances).</p>
     *         <p>
     *         <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
     *         <code>default</code> or <code>dedicated</code> values only.
     *         </p>
     *         <p>
     *         Default: <code>default</code>
     * @see Tenancy
     */
    public final String instanceTenancyAsString() {
        return instanceTenancy;
    }

    /**
     * <p>
     * The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you only
     * have access to the <code>Medium Utilization</code> Reserved Instance offering type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #offeringType} will
     * return {@link OfferingTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #offeringTypeAsString}.
     * </p>
     * 
     * @return The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you
     *         only have access to the <code>Medium Utilization</code> Reserved Instance offering type.
     * @see OfferingTypeValues
     */
    public final OfferingTypeValues offeringType() {
        return OfferingTypeValues.fromValue(offeringType);
    }

    /**
     * <p>
     * The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you only
     * have access to the <code>Medium Utilization</code> Reserved Instance offering type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #offeringType} will
     * return {@link OfferingTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #offeringTypeAsString}.
     * </p>
     * 
     * @return The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you
     *         only have access to the <code>Medium Utilization</code> Reserved Instance offering type.
     * @see OfferingTypeValues
     */
    public final String offeringTypeAsString() {
        return offeringType;
    }

    /**
     * <p>
     * The token to retrieve the next page of results.
     * </p>
     * 
     * @return The token to retrieve the next page of results.
     */
    public final String nextToken() {
        return nextToken;
    }

    /**
     * <p>
     * The maximum number of results to return for the request in a single page. The remaining results of the initial
     * request can be seen by sending another request with the returned <code>NextToken</code> value. The maximum is
     * 100.
     * </p>
     * <p>
     * Default: 100
     * </p>
     * 
     * @return The maximum number of results to return for the request in a single page. The remaining results of the
     *         initial request can be seen by sending another request with the returned <code>NextToken</code> value.
     *         The maximum is 100.</p>
     *         <p>
     *         Default: 100
     */
    public final Integer maxResults() {
        return maxResults;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZone());
        hashCode = 31 * hashCode + Objects.hashCode(includeMarketplace());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(maxDuration());
        hashCode = 31 * hashCode + Objects.hashCode(maxInstanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(minDuration());
        hashCode = 31 * hashCode + Objects.hashCode(offeringClassAsString());
        hashCode = 31 * hashCode + Objects.hashCode(productDescriptionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasReservedInstancesOfferingIds() ? reservedInstancesOfferingIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceTenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(offeringTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        hashCode = 31 * hashCode + Objects.hashCode(maxResults());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeReservedInstancesOfferingsRequest)) {
            return false;
        }
        DescribeReservedInstancesOfferingsRequest other = (DescribeReservedInstancesOfferingsRequest) obj;
        return Objects.equals(availabilityZone(), other.availabilityZone())
                && Objects.equals(includeMarketplace(), other.includeMarketplace())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(maxDuration(), other.maxDuration())
                && Objects.equals(maxInstanceCount(), other.maxInstanceCount())
                && Objects.equals(minDuration(), other.minDuration())
                && Objects.equals(offeringClassAsString(), other.offeringClassAsString())
                && Objects.equals(productDescriptionAsString(), other.productDescriptionAsString())
                && hasReservedInstancesOfferingIds() == other.hasReservedInstancesOfferingIds()
                && Objects.equals(reservedInstancesOfferingIds(), other.reservedInstancesOfferingIds())
                && Objects.equals(dryRun(), other.dryRun()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters())
                && Objects.equals(instanceTenancyAsString(), other.instanceTenancyAsString())
                && Objects.equals(offeringTypeAsString(), other.offeringTypeAsString())
                && Objects.equals(nextToken(), other.nextToken()) && Objects.equals(maxResults(), other.maxResults());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DescribeReservedInstancesOfferingsRequest").add("AvailabilityZone", availabilityZone())
                .add("IncludeMarketplace", includeMarketplace()).add("InstanceType", instanceTypeAsString())
                .add("MaxDuration", maxDuration()).add("MaxInstanceCount", maxInstanceCount()).add("MinDuration", minDuration())
                .add("OfferingClass", offeringClassAsString()).add("ProductDescription", productDescriptionAsString())
                .add("ReservedInstancesOfferingIds", hasReservedInstancesOfferingIds() ? reservedInstancesOfferingIds() : null)
                .add("DryRun", dryRun()).add("Filters", hasFilters() ? filters() : null)
                .add("InstanceTenancy", instanceTenancyAsString()).add("OfferingType", offeringTypeAsString())
                .add("NextToken", nextToken()).add("MaxResults", maxResults()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AvailabilityZone":
            return Optional.ofNullable(clazz.cast(availabilityZone()));
        case "IncludeMarketplace":
            return Optional.ofNullable(clazz.cast(includeMarketplace()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "MaxDuration":
            return Optional.ofNullable(clazz.cast(maxDuration()));
        case "MaxInstanceCount":
            return Optional.ofNullable(clazz.cast(maxInstanceCount()));
        case "MinDuration":
            return Optional.ofNullable(clazz.cast(minDuration()));
        case "OfferingClass":
            return Optional.ofNullable(clazz.cast(offeringClassAsString()));
        case "ProductDescription":
            return Optional.ofNullable(clazz.cast(productDescriptionAsString()));
        case "ReservedInstancesOfferingIds":
            return Optional.ofNullable(clazz.cast(reservedInstancesOfferingIds()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "InstanceTenancy":
            return Optional.ofNullable(clazz.cast(instanceTenancyAsString()));
        case "OfferingType":
            return Optional.ofNullable(clazz.cast(offeringTypeAsString()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        case "MaxResults":
            return Optional.ofNullable(clazz.cast(maxResults()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeReservedInstancesOfferingsRequest, T> g) {
        return obj -> g.apply((DescribeReservedInstancesOfferingsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo,
            CopyableBuilder<Builder, DescribeReservedInstancesOfferingsRequest> {
        /**
         * <p>
         * The Availability Zone in which the Reserved Instance can be used.
         * </p>
         * 
         * @param availabilityZone
         *        The Availability Zone in which the Reserved Instance can be used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZone(String availabilityZone);

        /**
         * <p>
         * Include Reserved Instance Marketplace offerings in the response.
         * </p>
         * 
         * @param includeMarketplace
         *        Include Reserved Instance Marketplace offerings in the response.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder includeMarketplace(Boolean includeMarketplace);

        /**
         * <p>
         * The instance type that the reservation will cover (for example, <code>m1.small</code>). For more information,
         * see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance
         * types</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param instanceType
         *        The instance type that the reservation will cover (for example, <code>m1.small</code>). For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance
         *        types</a> in the <i>Amazon EC2 User Guide</i>.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type that the reservation will cover (for example, <code>m1.small</code>). For more information,
         * see <a href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance
         * types</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param instanceType
         *        The instance type that the reservation will cover (for example, <code>m1.small</code>). For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/instance-types.html">Amazon EC2 instance
         *        types</a> in the <i>Amazon EC2 User Guide</i>.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(InstanceType instanceType);

        /**
         * <p>
         * The maximum duration (in seconds) to filter when searching for offerings.
         * </p>
         * <p>
         * Default: 94608000 (3 years)
         * </p>
         * 
         * @param maxDuration
         *        The maximum duration (in seconds) to filter when searching for offerings.</p>
         *        <p>
         *        Default: 94608000 (3 years)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxDuration(Long maxDuration);

        /**
         * <p>
         * The maximum number of instances to filter when searching for offerings.
         * </p>
         * <p>
         * Default: 20
         * </p>
         * 
         * @param maxInstanceCount
         *        The maximum number of instances to filter when searching for offerings.</p>
         *        <p>
         *        Default: 20
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxInstanceCount(Integer maxInstanceCount);

        /**
         * <p>
         * The minimum duration (in seconds) to filter when searching for offerings.
         * </p>
         * <p>
         * Default: 2592000 (1 month)
         * </p>
         * 
         * @param minDuration
         *        The minimum duration (in seconds) to filter when searching for offerings.</p>
         *        <p>
         *        Default: 2592000 (1 month)
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minDuration(Long minDuration);

        /**
         * <p>
         * The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
         * </p>
         * 
         * @param offeringClass
         *        The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
         * @see OfferingClassType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OfferingClassType
         */
        Builder offeringClass(String offeringClass);

        /**
         * <p>
         * The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
         * </p>
         * 
         * @param offeringClass
         *        The offering class of the Reserved Instance. Can be <code>standard</code> or <code>convertible</code>.
         * @see OfferingClassType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OfferingClassType
         */
        Builder offeringClass(OfferingClassType offeringClass);

        /**
         * <p>
         * The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in the
         * description are for use with Amazon VPC.
         * </p>
         * 
         * @param productDescription
         *        The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code>
         *        in the description are for use with Amazon VPC.
         * @see RIProductDescription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RIProductDescription
         */
        Builder productDescription(String productDescription);

        /**
         * <p>
         * The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code> in the
         * description are for use with Amazon VPC.
         * </p>
         * 
         * @param productDescription
         *        The Reserved Instance product platform description. Instances that include <code>(Amazon VPC)</code>
         *        in the description are for use with Amazon VPC.
         * @see RIProductDescription
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RIProductDescription
         */
        Builder productDescription(RIProductDescription productDescription);

        /**
         * <p>
         * One or more Reserved Instances offering IDs.
         * </p>
         * 
         * @param reservedInstancesOfferingIds
         *        One or more Reserved Instances offering IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedInstancesOfferingIds(Collection<String> reservedInstancesOfferingIds);

        /**
         * <p>
         * One or more Reserved Instances offering IDs.
         * </p>
         * 
         * @param reservedInstancesOfferingIds
         *        One or more Reserved Instances offering IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservedInstancesOfferingIds(String... reservedInstancesOfferingIds);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * One or more filters.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years), in
         * seconds (<code>31536000</code> | <code>94608000</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type that is covered by the reservation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace offerings.
         * When this filter is not used, which is the default behavior, all offerings from both Amazon Web Services and
         * the Reserved Instance Marketplace are listed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>product-description</code> - The Reserved Instance product platform description (
         * <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
         * <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
         * <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
         * <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
         * <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
         * <code>Windows with SQL Server Enterprise</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
         * <code>Region</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
         * </p>
         * </li>
         * </ul>
         * 
         * @param filters
         *        One or more filters.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years),
         *        in seconds (<code>31536000</code> | <code>94608000</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-type</code> - The instance type that is covered by the reservation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace
         *        offerings. When this filter is not used, which is the default behavior, all offerings from both Amazon
         *        Web Services and the Reserved Instance Marketplace are listed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>product-description</code> - The Reserved Instance product platform description (
         *        <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
         *        <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
         *        <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
         *        <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
         *        <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
         *        <code>Windows with SQL Server Enterprise</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
         *        <code>Region</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<Filter> filters);

        /**
         * <p>
         * One or more filters.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years), in
         * seconds (<code>31536000</code> | <code>94608000</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type that is covered by the reservation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace offerings.
         * When this filter is not used, which is the default behavior, all offerings from both Amazon Web Services and
         * the Reserved Instance Marketplace are listed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>product-description</code> - The Reserved Instance product platform description (
         * <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
         * <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
         * <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
         * <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
         * <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
         * <code>Windows with SQL Server Enterprise</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
         * <code>Region</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
         * </p>
         * </li>
         * </ul>
         * 
         * @param filters
         *        One or more filters.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years),
         *        in seconds (<code>31536000</code> | <code>94608000</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>instance-type</code> - The instance type that is covered by the reservation.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace
         *        offerings. When this filter is not used, which is the default behavior, all offerings from both Amazon
         *        Web Services and the Reserved Instance Marketplace are listed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>product-description</code> - The Reserved Instance product platform description (
         *        <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
         *        <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
         *        <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
         *        <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
         *        <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
         *        <code>Windows with SQL Server Enterprise</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
         *        <code>Region</code>).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Filter... filters);

        /**
         * <p>
         * One or more filters.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>availability-zone</code> - The Availability Zone where the Reserved Instance can be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>duration</code> - The duration of the Reserved Instance (for example, one year or three years), in
         * seconds (<code>31536000</code> | <code>94608000</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>fixed-price</code> - The purchase price of the Reserved Instance (for example, 9800.0).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>instance-type</code> - The instance type that is covered by the reservation.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>marketplace</code> - Set to <code>true</code> to show only Reserved Instance Marketplace offerings.
         * When this filter is not used, which is the default behavior, all offerings from both Amazon Web Services and
         * the Reserved Instance Marketplace are listed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>product-description</code> - The Reserved Instance product platform description (
         * <code>Linux/UNIX</code> | <code>Linux with SQL Server Standard</code> |
         * <code>Linux with SQL Server Web</code> | <code>Linux with SQL Server Enterprise</code> |
         * <code>SUSE Linux</code> | <code>Red Hat Enterprise Linux</code> |
         * <code>Red Hat Enterprise Linux with HA</code> | <code>Windows</code> |
         * <code>Windows with SQL Server Standard</code> | <code>Windows with SQL Server Web</code> |
         * <code>Windows with SQL Server Enterprise</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>reserved-instances-offering-id</code> - The Reserved Instances offering ID.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>scope</code> - The scope of the Reserved Instance (<code>Availability Zone</code> or
         * <code>Region</code>).
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>usage-price</code> - The usage price of the Reserved Instance, per hour (for example, 0.84).
         * </p>
         * </li>
         * </ul>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Filter.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Filter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Filter.Builder#build()}
         * is called immediately and its result is passed to {@link #filters(List<Filter>)}.
         * 
         * @param filters
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Filter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(java.util.Collection<Filter>)
         */
        Builder filters(Consumer<Filter.Builder>... filters);

        /**
         * <p>
         * The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
         * <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e., Dedicated
         * Instances).
         * </p>
         * <p>
         * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         * <code>default</code> or <code>dedicated</code> values only.
         * </p>
         * <p>
         * Default: <code>default</code>
         * </p>
         * 
         * @param instanceTenancy
         *        The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
         *        <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e.,
         *        Dedicated Instances).</p>
         *        <p>
         *        <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         *        <code>default</code> or <code>dedicated</code> values only.
         *        </p>
         *        <p>
         *        Default: <code>default</code>
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder instanceTenancy(String instanceTenancy);

        /**
         * <p>
         * The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
         * <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e., Dedicated
         * Instances).
         * </p>
         * <p>
         * <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         * <code>default</code> or <code>dedicated</code> values only.
         * </p>
         * <p>
         * Default: <code>default</code>
         * </p>
         * 
         * @param instanceTenancy
         *        The tenancy of the instances covered by the reservation. A Reserved Instance with a tenancy of
         *        <code>dedicated</code> is applied to instances that run in a VPC on single-tenant hardware (i.e.,
         *        Dedicated Instances).</p>
         *        <p>
         *        <b>Important:</b> The <code>host</code> value cannot be used with this parameter. Use the
         *        <code>default</code> or <code>dedicated</code> values only.
         *        </p>
         *        <p>
         *        Default: <code>default</code>
         * @see Tenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Tenancy
         */
        Builder instanceTenancy(Tenancy instanceTenancy);

        /**
         * <p>
         * The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you only
         * have access to the <code>Medium Utilization</code> Reserved Instance offering type.
         * </p>
         * 
         * @param offeringType
         *        The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version,
         *        you only have access to the <code>Medium Utilization</code> Reserved Instance offering type.
         * @see OfferingTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OfferingTypeValues
         */
        Builder offeringType(String offeringType);

        /**
         * <p>
         * The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version, you only
         * have access to the <code>Medium Utilization</code> Reserved Instance offering type.
         * </p>
         * 
         * @param offeringType
         *        The Reserved Instance offering type. If you are using tools that predate the 2011-11-01 API version,
         *        you only have access to the <code>Medium Utilization</code> Reserved Instance offering type.
         * @see OfferingTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see OfferingTypeValues
         */
        Builder offeringType(OfferingTypeValues offeringType);

        /**
         * <p>
         * The token to retrieve the next page of results.
         * </p>
         * 
         * @param nextToken
         *        The token to retrieve the next page of results.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);

        /**
         * <p>
         * The maximum number of results to return for the request in a single page. The remaining results of the
         * initial request can be seen by sending another request with the returned <code>NextToken</code> value. The
         * maximum is 100.
         * </p>
         * <p>
         * Default: 100
         * </p>
         * 
         * @param maxResults
         *        The maximum number of results to return for the request in a single page. The remaining results of the
         *        initial request can be seen by sending another request with the returned <code>NextToken</code> value.
         *        The maximum is 100.</p>
         *        <p>
         *        Default: 100
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxResults(Integer maxResults);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private String availabilityZone;

        private Boolean includeMarketplace;

        private String instanceType;

        private Long maxDuration;

        private Integer maxInstanceCount;

        private Long minDuration;

        private String offeringClass;

        private String productDescription;

        private List<String> reservedInstancesOfferingIds = DefaultSdkAutoConstructList.getInstance();

        private Boolean dryRun;

        private List<Filter> filters = DefaultSdkAutoConstructList.getInstance();

        private String instanceTenancy;

        private String offeringType;

        private String nextToken;

        private Integer maxResults;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeReservedInstancesOfferingsRequest model) {
            super(model);
            availabilityZone(model.availabilityZone);
            includeMarketplace(model.includeMarketplace);
            instanceType(model.instanceType);
            maxDuration(model.maxDuration);
            maxInstanceCount(model.maxInstanceCount);
            minDuration(model.minDuration);
            offeringClass(model.offeringClass);
            productDescription(model.productDescription);
            reservedInstancesOfferingIds(model.reservedInstancesOfferingIds);
            dryRun(model.dryRun);
            filters(model.filters);
            instanceTenancy(model.instanceTenancy);
            offeringType(model.offeringType);
            nextToken(model.nextToken);
            maxResults(model.maxResults);
        }

        public final String getAvailabilityZone() {
            return availabilityZone;
        }

        public final void setAvailabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
        }

        @Override
        public final Builder availabilityZone(String availabilityZone) {
            this.availabilityZone = availabilityZone;
            return this;
        }

        public final Boolean getIncludeMarketplace() {
            return includeMarketplace;
        }

        public final void setIncludeMarketplace(Boolean includeMarketplace) {
            this.includeMarketplace = includeMarketplace;
        }

        @Override
        public final Builder includeMarketplace(Boolean includeMarketplace) {
            this.includeMarketplace = includeMarketplace;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Long getMaxDuration() {
            return maxDuration;
        }

        public final void setMaxDuration(Long maxDuration) {
            this.maxDuration = maxDuration;
        }

        @Override
        public final Builder maxDuration(Long maxDuration) {
            this.maxDuration = maxDuration;
            return this;
        }

        public final Integer getMaxInstanceCount() {
            return maxInstanceCount;
        }

        public final void setMaxInstanceCount(Integer maxInstanceCount) {
            this.maxInstanceCount = maxInstanceCount;
        }

        @Override
        public final Builder maxInstanceCount(Integer maxInstanceCount) {
            this.maxInstanceCount = maxInstanceCount;
            return this;
        }

        public final Long getMinDuration() {
            return minDuration;
        }

        public final void setMinDuration(Long minDuration) {
            this.minDuration = minDuration;
        }

        @Override
        public final Builder minDuration(Long minDuration) {
            this.minDuration = minDuration;
            return this;
        }

        public final String getOfferingClass() {
            return offeringClass;
        }

        public final void setOfferingClass(String offeringClass) {
            this.offeringClass = offeringClass;
        }

        @Override
        public final Builder offeringClass(String offeringClass) {
            this.offeringClass = offeringClass;
            return this;
        }

        @Override
        public final Builder offeringClass(OfferingClassType offeringClass) {
            this.offeringClass(offeringClass == null ? null : offeringClass.toString());
            return this;
        }

        public final String getProductDescription() {
            return productDescription;
        }

        public final void setProductDescription(String productDescription) {
            this.productDescription = productDescription;
        }

        @Override
        public final Builder productDescription(String productDescription) {
            this.productDescription = productDescription;
            return this;
        }

        @Override
        public final Builder productDescription(RIProductDescription productDescription) {
            this.productDescription(productDescription == null ? null : productDescription.toString());
            return this;
        }

        public final Collection<String> getReservedInstancesOfferingIds() {
            if (reservedInstancesOfferingIds instanceof SdkAutoConstructList) {
                return null;
            }
            return reservedInstancesOfferingIds;
        }

        public final void setReservedInstancesOfferingIds(Collection<String> reservedInstancesOfferingIds) {
            this.reservedInstancesOfferingIds = ReservedInstancesOfferingIdStringListCopier.copy(reservedInstancesOfferingIds);
        }

        @Override
        public final Builder reservedInstancesOfferingIds(Collection<String> reservedInstancesOfferingIds) {
            this.reservedInstancesOfferingIds = ReservedInstancesOfferingIdStringListCopier.copy(reservedInstancesOfferingIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder reservedInstancesOfferingIds(String... reservedInstancesOfferingIds) {
            reservedInstancesOfferingIds(Arrays.asList(reservedInstancesOfferingIds));
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final List<Filter.Builder> getFilters() {
            List<Filter.Builder> result = FilterListCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<Filter.BuilderImpl> filters) {
            this.filters = FilterListCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<Filter> filters) {
            this.filters = FilterListCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Filter... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<Filter.Builder>... filters) {
            filters(Stream.of(filters).map(c -> Filter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getInstanceTenancy() {
            return instanceTenancy;
        }

        public final void setInstanceTenancy(String instanceTenancy) {
            this.instanceTenancy = instanceTenancy;
        }

        @Override
        public final Builder instanceTenancy(String instanceTenancy) {
            this.instanceTenancy = instanceTenancy;
            return this;
        }

        @Override
        public final Builder instanceTenancy(Tenancy instanceTenancy) {
            this.instanceTenancy(instanceTenancy == null ? null : instanceTenancy.toString());
            return this;
        }

        public final String getOfferingType() {
            return offeringType;
        }

        public final void setOfferingType(String offeringType) {
            this.offeringType = offeringType;
        }

        @Override
        public final Builder offeringType(String offeringType) {
            this.offeringType = offeringType;
            return this;
        }

        @Override
        public final Builder offeringType(OfferingTypeValues offeringType) {
            this.offeringType(offeringType == null ? null : offeringType.toString());
            return this;
        }

        public final String getNextToken() {
            return nextToken;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final Integer getMaxResults() {
            return maxResults;
        }

        public final void setMaxResults(Integer maxResults) {
            this.maxResults = maxResults;
        }

        @Override
        public final Builder maxResults(Integer maxResults) {
            this.maxResults = maxResults;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public DescribeReservedInstancesOfferingsRequest build() {
            return new DescribeReservedInstancesOfferingsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
