/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains the parameters for RequestSpotInstances.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RequestSpotInstancesRequest extends Ec2Request implements
        ToCopyableBuilder<RequestSpotInstancesRequest.Builder, RequestSpotInstancesRequest> {
    private static final SdkField<RequestSpotLaunchSpecification> LAUNCH_SPECIFICATION_FIELD = SdkField
            .<RequestSpotLaunchSpecification> builder(MarshallingType.SDK_POJO)
            .memberName("LaunchSpecification")
            .getter(getter(RequestSpotInstancesRequest::launchSpecification))
            .setter(setter(Builder::launchSpecification))
            .constructor(RequestSpotLaunchSpecification::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchSpecification")
                    .unmarshallLocationName("LaunchSpecification").build()).build();

    private static final SdkField<List<TagSpecification>> TAG_SPECIFICATIONS_FIELD = SdkField
            .<List<TagSpecification>> builder(MarshallingType.LIST)
            .memberName("TagSpecifications")
            .getter(getter(RequestSpotInstancesRequest::tagSpecifications))
            .setter(setter(Builder::tagSpecifications))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSpecification")
                    .unmarshallLocationName("TagSpecification").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<TagSpecification> builder(MarshallingType.SDK_POJO)
                                            .constructor(TagSpecification::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> INSTANCE_INTERRUPTION_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceInterruptionBehavior")
            .getter(getter(RequestSpotInstancesRequest::instanceInterruptionBehaviorAsString))
            .setter(setter(Builder::instanceInterruptionBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceInterruptionBehavior")
                    .unmarshallLocationName("InstanceInterruptionBehavior").build()).build();

    private static final SdkField<Boolean> DRY_RUN_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("DryRun")
            .getter(getter(RequestSpotInstancesRequest::dryRun))
            .setter(setter(Builder::dryRun))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DryRun")
                    .unmarshallLocationName("dryRun").build()).build();

    private static final SdkField<String> SPOT_PRICE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SpotPrice")
            .getter(getter(RequestSpotInstancesRequest::spotPrice))
            .setter(setter(Builder::spotPrice))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SpotPrice")
                    .unmarshallLocationName("spotPrice").build()).build();

    private static final SdkField<String> CLIENT_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ClientToken")
            .getter(getter(RequestSpotInstancesRequest::clientToken))
            .setter(setter(Builder::clientToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ClientToken")
                    .unmarshallLocationName("clientToken").build()).build();

    private static final SdkField<Integer> INSTANCE_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("InstanceCount")
            .getter(getter(RequestSpotInstancesRequest::instanceCount))
            .setter(setter(Builder::instanceCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceCount")
                    .unmarshallLocationName("instanceCount").build()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Type")
            .getter(getter(RequestSpotInstancesRequest::typeAsString))
            .setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Type")
                    .unmarshallLocationName("type").build()).build();

    private static final SdkField<Instant> VALID_FROM_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ValidFrom")
            .getter(getter(RequestSpotInstancesRequest::validFrom))
            .setter(setter(Builder::validFrom))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidFrom")
                    .unmarshallLocationName("validFrom").build()).build();

    private static final SdkField<Instant> VALID_UNTIL_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ValidUntil")
            .getter(getter(RequestSpotInstancesRequest::validUntil))
            .setter(setter(Builder::validUntil))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ValidUntil")
                    .unmarshallLocationName("validUntil").build()).build();

    private static final SdkField<String> LAUNCH_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LaunchGroup")
            .getter(getter(RequestSpotInstancesRequest::launchGroup))
            .setter(setter(Builder::launchGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LaunchGroup")
                    .unmarshallLocationName("launchGroup").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_GROUP_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneGroup")
            .getter(getter(RequestSpotInstancesRequest::availabilityZoneGroup))
            .setter(setter(Builder::availabilityZoneGroup))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneGroup")
                    .unmarshallLocationName("availabilityZoneGroup").build()).build();

    private static final SdkField<Integer> BLOCK_DURATION_MINUTES_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("BlockDurationMinutes")
            .getter(getter(RequestSpotInstancesRequest::blockDurationMinutes))
            .setter(setter(Builder::blockDurationMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("BlockDurationMinutes")
                    .unmarshallLocationName("blockDurationMinutes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(LAUNCH_SPECIFICATION_FIELD,
            TAG_SPECIFICATIONS_FIELD, INSTANCE_INTERRUPTION_BEHAVIOR_FIELD, DRY_RUN_FIELD, SPOT_PRICE_FIELD, CLIENT_TOKEN_FIELD,
            INSTANCE_COUNT_FIELD, TYPE_FIELD, VALID_FROM_FIELD, VALID_UNTIL_FIELD, LAUNCH_GROUP_FIELD,
            AVAILABILITY_ZONE_GROUP_FIELD, BLOCK_DURATION_MINUTES_FIELD));

    private final RequestSpotLaunchSpecification launchSpecification;

    private final List<TagSpecification> tagSpecifications;

    private final String instanceInterruptionBehavior;

    private final Boolean dryRun;

    private final String spotPrice;

    private final String clientToken;

    private final Integer instanceCount;

    private final String type;

    private final Instant validFrom;

    private final Instant validUntil;

    private final String launchGroup;

    private final String availabilityZoneGroup;

    private final Integer blockDurationMinutes;

    private RequestSpotInstancesRequest(BuilderImpl builder) {
        super(builder);
        this.launchSpecification = builder.launchSpecification;
        this.tagSpecifications = builder.tagSpecifications;
        this.instanceInterruptionBehavior = builder.instanceInterruptionBehavior;
        this.dryRun = builder.dryRun;
        this.spotPrice = builder.spotPrice;
        this.clientToken = builder.clientToken;
        this.instanceCount = builder.instanceCount;
        this.type = builder.type;
        this.validFrom = builder.validFrom;
        this.validUntil = builder.validUntil;
        this.launchGroup = builder.launchGroup;
        this.availabilityZoneGroup = builder.availabilityZoneGroup;
        this.blockDurationMinutes = builder.blockDurationMinutes;
    }

    /**
     * <p>
     * The launch specification.
     * </p>
     * 
     * @return The launch specification.
     */
    public final RequestSpotLaunchSpecification launchSpecification() {
        return launchSpecification;
    }

    /**
     * For responses, this returns true if the service returned a value for the TagSpecifications property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasTagSpecifications() {
        return tagSpecifications != null && !(tagSpecifications instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key-value pair for tagging the Spot Instance request on creation. The value for <code>ResourceType</code>
     * must be <code>spot-instances-request</code>, otherwise the Spot Instance request fails. To tag the Spot Instance
     * request after it has been created, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTagSpecifications} method.
     * </p>
     * 
     * @return The key-value pair for tagging the Spot Instance request on creation. The value for
     *         <code>ResourceType</code> must be <code>spot-instances-request</code>, otherwise the Spot Instance
     *         request fails. To tag the Spot Instance request after it has been created, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
     */
    public final List<TagSpecification> tagSpecifications() {
        return tagSpecifications;
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link InstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
     * @see InstanceInterruptionBehavior
     */
    public final InstanceInterruptionBehavior instanceInterruptionBehavior() {
        return InstanceInterruptionBehavior.fromValue(instanceInterruptionBehavior);
    }

    /**
     * <p>
     * The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInterruptionBehavior} will return {@link InstanceInterruptionBehavior#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #instanceInterruptionBehaviorAsString}.
     * </p>
     * 
     * @return The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
     * @see InstanceInterruptionBehavior
     */
    public final String instanceInterruptionBehaviorAsString() {
        return instanceInterruptionBehavior;
    }

    /**
     * <p>
     * Checks whether you have the required permissions for the action, without actually making the request, and
     * provides an error response. If you have the required permissions, the error response is
     * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     * </p>
     * 
     * @return Checks whether you have the required permissions for the action, without actually making the request, and
     *         provides an error response. If you have the required permissions, the error response is
     *         <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
     */
    public final Boolean dryRun() {
        return dryRun;
    }

    /**
     * <p>
     * The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend using this
     * parameter because it can lead to increased interruptions. If you do not specify this parameter, you will pay the
     * current Spot price.
     * </p>
     * <important>
     * <p>
     * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
     * this parameter.
     * </p>
     * </important>
     * 
     * @return The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend
     *         using this parameter because it can lead to increased interruptions. If you do not specify this
     *         parameter, you will pay the current Spot price.</p> <important>
     *         <p>
     *         If you specify a maximum price, your instances will be interrupted more frequently than if you do not
     *         specify this parameter.
     *         </p>
     */
    public final String spotPrice() {
        return spotPrice;
    }

    /**
     * <p>
     * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">Ensuring idempotency in
     * Amazon EC2 API requests</a> in the <i>Amazon EC2 User Guide</i>.
     * </p>
     * 
     * @return Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">Ensuring
     *         idempotency in Amazon EC2 API requests</a> in the <i>Amazon EC2 User Guide</i>.
     */
    public final String clientToken() {
        return clientToken;
    }

    /**
     * <p>
     * The maximum number of Spot Instances to launch.
     * </p>
     * <p>
     * Default: 1
     * </p>
     * 
     * @return The maximum number of Spot Instances to launch.</p>
     *         <p>
     *         Default: 1
     */
    public final Integer instanceCount() {
        return instanceCount;
    }

    /**
     * <p>
     * The Spot Instance request type.
     * </p>
     * <p>
     * Default: <code>one-time</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SpotInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The Spot Instance request type.</p>
     *         <p>
     *         Default: <code>one-time</code>
     * @see SpotInstanceType
     */
    public final SpotInstanceType type() {
        return SpotInstanceType.fromValue(type);
    }

    /**
     * <p>
     * The Spot Instance request type.
     * </p>
     * <p>
     * Default: <code>one-time</code>
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link SpotInstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The Spot Instance request type.</p>
     *         <p>
     *         Default: <code>one-time</code>
     * @see SpotInstanceType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The start date of the request. If this is a one-time request, the request becomes active at this date and time
     * and remains active until all instances launch, the request expires, or the request is canceled. If the request is
     * persistent, the request becomes active at this date and time and remains active until it expires or is canceled.
     * </p>
     * <p>
     * The specified start date and time cannot be equal to the current date and time. You must specify a start date and
     * time that occurs after the current date and time.
     * </p>
     * 
     * @return The start date of the request. If this is a one-time request, the request becomes active at this date and
     *         time and remains active until all instances launch, the request expires, or the request is canceled. If
     *         the request is persistent, the request becomes active at this date and time and remains active until it
     *         expires or is canceled.</p>
     *         <p>
     *         The specified start date and time cannot be equal to the current date and time. You must specify a start
     *         date and time that occurs after the current date and time.
     */
    public final Instant validFrom() {
        return validFrom;
    }

    /**
     * <p>
     * The end date of the request, in UTC format (<i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
     * </p>
     * <ul>
     * <li>
     * <p>
     * For a persistent request, the request remains active until the <code>ValidUntil</code> date and time is reached.
     * Otherwise, the request remains active until you cancel it.
     * </p>
     * </li>
     * <li>
     * <p>
     * For a one-time request, the request remains active until all instances launch, the request is canceled, or the
     * <code>ValidUntil</code> date and time is reached. By default, the request is valid for 7 days from the date the
     * request was created.
     * </p>
     * </li>
     * </ul>
     * 
     * @return The end date of the request, in UTC format
     *         (<i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).</p>
     *         <ul>
     *         <li>
     *         <p>
     *         For a persistent request, the request remains active until the <code>ValidUntil</code> date and time is
     *         reached. Otherwise, the request remains active until you cancel it.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         For a one-time request, the request remains active until all instances launch, the request is canceled,
     *         or the <code>ValidUntil</code> date and time is reached. By default, the request is valid for 7 days from
     *         the date the request was created.
     *         </p>
     *         </li>
     */
    public final Instant validUntil() {
        return validUntil;
    }

    /**
     * <p>
     * The instance launch group. Launch groups are Spot Instances that launch together and terminate together.
     * </p>
     * <p>
     * Default: Instances are launched and terminated individually
     * </p>
     * 
     * @return The instance launch group. Launch groups are Spot Instances that launch together and terminate
     *         together.</p>
     *         <p>
     *         Default: Instances are launched and terminated individually
     */
    public final String launchGroup() {
        return launchGroup;
    }

    /**
     * <p>
     * The user-specified name for a logical grouping of requests.
     * </p>
     * <p>
     * When you specify an Availability Zone group in a Spot Instance request, all Spot Instances in the request are
     * launched in the same Availability Zone. Instance proximity is maintained with this parameter, but the choice of
     * Availability Zone is not. The group applies only to requests for Spot Instances of the same instance type. Any
     * additional Spot Instance requests that are specified with the same Availability Zone group name are launched in
     * that same Availability Zone, as long as at least one instance from the group is still active.
     * </p>
     * <p>
     * If there is no active instance running in the Availability Zone group that you specify for a new Spot Instance
     * request (all instances are terminated, the request is expired, or the maximum price you specified falls below
     * current Spot price), then Amazon EC2 launches the instance in any Availability Zone where the constraint can be
     * met. Consequently, the subsequent set of Spot Instances could be placed in a different zone from the original
     * request, even if you specified the same Availability Zone group.
     * </p>
     * <p>
     * Default: Instances are launched in any available Availability Zone.
     * </p>
     * 
     * @return The user-specified name for a logical grouping of requests.</p>
     *         <p>
     *         When you specify an Availability Zone group in a Spot Instance request, all Spot Instances in the request
     *         are launched in the same Availability Zone. Instance proximity is maintained with this parameter, but the
     *         choice of Availability Zone is not. The group applies only to requests for Spot Instances of the same
     *         instance type. Any additional Spot Instance requests that are specified with the same Availability Zone
     *         group name are launched in that same Availability Zone, as long as at least one instance from the group
     *         is still active.
     *         </p>
     *         <p>
     *         If there is no active instance running in the Availability Zone group that you specify for a new Spot
     *         Instance request (all instances are terminated, the request is expired, or the maximum price you
     *         specified falls below current Spot price), then Amazon EC2 launches the instance in any Availability Zone
     *         where the constraint can be met. Consequently, the subsequent set of Spot Instances could be placed in a
     *         different zone from the original request, even if you specified the same Availability Zone group.
     *         </p>
     *         <p>
     *         Default: Instances are launched in any available Availability Zone.
     */
    public final String availabilityZoneGroup() {
        return availabilityZoneGroup;
    }

    /**
     * <p>
     * Deprecated.
     * </p>
     * 
     * @return Deprecated.
     */
    public final Integer blockDurationMinutes() {
        return blockDurationMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(launchSpecification());
        hashCode = 31 * hashCode + Objects.hashCode(hasTagSpecifications() ? tagSpecifications() : null);
        hashCode = 31 * hashCode + Objects.hashCode(instanceInterruptionBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(dryRun());
        hashCode = 31 * hashCode + Objects.hashCode(spotPrice());
        hashCode = 31 * hashCode + Objects.hashCode(clientToken());
        hashCode = 31 * hashCode + Objects.hashCode(instanceCount());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(validFrom());
        hashCode = 31 * hashCode + Objects.hashCode(validUntil());
        hashCode = 31 * hashCode + Objects.hashCode(launchGroup());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneGroup());
        hashCode = 31 * hashCode + Objects.hashCode(blockDurationMinutes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RequestSpotInstancesRequest)) {
            return false;
        }
        RequestSpotInstancesRequest other = (RequestSpotInstancesRequest) obj;
        return Objects.equals(launchSpecification(), other.launchSpecification())
                && hasTagSpecifications() == other.hasTagSpecifications()
                && Objects.equals(tagSpecifications(), other.tagSpecifications())
                && Objects.equals(instanceInterruptionBehaviorAsString(), other.instanceInterruptionBehaviorAsString())
                && Objects.equals(dryRun(), other.dryRun()) && Objects.equals(spotPrice(), other.spotPrice())
                && Objects.equals(clientToken(), other.clientToken()) && Objects.equals(instanceCount(), other.instanceCount())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(validFrom(), other.validFrom())
                && Objects.equals(validUntil(), other.validUntil()) && Objects.equals(launchGroup(), other.launchGroup())
                && Objects.equals(availabilityZoneGroup(), other.availabilityZoneGroup())
                && Objects.equals(blockDurationMinutes(), other.blockDurationMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RequestSpotInstancesRequest").add("LaunchSpecification", launchSpecification())
                .add("TagSpecifications", hasTagSpecifications() ? tagSpecifications() : null)
                .add("InstanceInterruptionBehavior", instanceInterruptionBehaviorAsString()).add("DryRun", dryRun())
                .add("SpotPrice", spotPrice()).add("ClientToken", clientToken()).add("InstanceCount", instanceCount())
                .add("Type", typeAsString()).add("ValidFrom", validFrom()).add("ValidUntil", validUntil())
                .add("LaunchGroup", launchGroup()).add("AvailabilityZoneGroup", availabilityZoneGroup())
                .add("BlockDurationMinutes", blockDurationMinutes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "LaunchSpecification":
            return Optional.ofNullable(clazz.cast(launchSpecification()));
        case "TagSpecifications":
            return Optional.ofNullable(clazz.cast(tagSpecifications()));
        case "InstanceInterruptionBehavior":
            return Optional.ofNullable(clazz.cast(instanceInterruptionBehaviorAsString()));
        case "DryRun":
            return Optional.ofNullable(clazz.cast(dryRun()));
        case "SpotPrice":
            return Optional.ofNullable(clazz.cast(spotPrice()));
        case "ClientToken":
            return Optional.ofNullable(clazz.cast(clientToken()));
        case "InstanceCount":
            return Optional.ofNullable(clazz.cast(instanceCount()));
        case "Type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "ValidFrom":
            return Optional.ofNullable(clazz.cast(validFrom()));
        case "ValidUntil":
            return Optional.ofNullable(clazz.cast(validUntil()));
        case "LaunchGroup":
            return Optional.ofNullable(clazz.cast(launchGroup()));
        case "AvailabilityZoneGroup":
            return Optional.ofNullable(clazz.cast(availabilityZoneGroup()));
        case "BlockDurationMinutes":
            return Optional.ofNullable(clazz.cast(blockDurationMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<RequestSpotInstancesRequest, T> g) {
        return obj -> g.apply((RequestSpotInstancesRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends Ec2Request.Builder, SdkPojo, CopyableBuilder<Builder, RequestSpotInstancesRequest> {
        /**
         * <p>
         * The launch specification.
         * </p>
         * 
         * @param launchSpecification
         *        The launch specification.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchSpecification(RequestSpotLaunchSpecification launchSpecification);

        /**
         * <p>
         * The launch specification.
         * </p>
         * This is a convenience method that creates an instance of the {@link RequestSpotLaunchSpecification.Builder}
         * avoiding the need to create one manually via {@link RequestSpotLaunchSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link RequestSpotLaunchSpecification.Builder#build()} is called
         * immediately and its result is passed to {@link #launchSpecification(RequestSpotLaunchSpecification)}.
         * 
         * @param launchSpecification
         *        a consumer that will call methods on {@link RequestSpotLaunchSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #launchSpecification(RequestSpotLaunchSpecification)
         */
        default Builder launchSpecification(Consumer<RequestSpotLaunchSpecification.Builder> launchSpecification) {
            return launchSpecification(RequestSpotLaunchSpecification.builder().applyMutation(launchSpecification).build());
        }

        /**
         * <p>
         * The key-value pair for tagging the Spot Instance request on creation. The value for <code>ResourceType</code>
         * must be <code>spot-instances-request</code>, otherwise the Spot Instance request fails. To tag the Spot
         * Instance request after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * 
         * @param tagSpecifications
         *        The key-value pair for tagging the Spot Instance request on creation. The value for
         *        <code>ResourceType</code> must be <code>spot-instances-request</code>, otherwise the Spot Instance
         *        request fails. To tag the Spot Instance request after it has been created, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(Collection<TagSpecification> tagSpecifications);

        /**
         * <p>
         * The key-value pair for tagging the Spot Instance request on creation. The value for <code>ResourceType</code>
         * must be <code>spot-instances-request</code>, otherwise the Spot Instance request fails. To tag the Spot
         * Instance request after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * 
         * @param tagSpecifications
         *        The key-value pair for tagging the Spot Instance request on creation. The value for
         *        <code>ResourceType</code> must be <code>spot-instances-request</code>, otherwise the Spot Instance
         *        request fails. To tag the Spot Instance request after it has been created, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tagSpecifications(TagSpecification... tagSpecifications);

        /**
         * <p>
         * The key-value pair for tagging the Spot Instance request on creation. The value for <code>ResourceType</code>
         * must be <code>spot-instances-request</code>, otherwise the Spot Instance request fails. To tag the Spot
         * Instance request after it has been created, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/APIReference/API_CreateTags.html">CreateTags</a>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.ec2.model.TagSpecification#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder#build()} is called immediately and
         * its result is passed to {@link #tagSpecifications(List<TagSpecification>)}.
         * 
         * @param tagSpecifications
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ec2.model.TagSpecification.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tagSpecifications(java.util.Collection<TagSpecification>)
         */
        Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications);

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
         * @see InstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(String instanceInterruptionBehavior);

        /**
         * <p>
         * The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
         * </p>
         * 
         * @param instanceInterruptionBehavior
         *        The behavior when a Spot Instance is interrupted. The default is <code>terminate</code>.
         * @see InstanceInterruptionBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceInterruptionBehavior
         */
        Builder instanceInterruptionBehavior(InstanceInterruptionBehavior instanceInterruptionBehavior);

        /**
         * <p>
         * Checks whether you have the required permissions for the action, without actually making the request, and
         * provides an error response. If you have the required permissions, the error response is
         * <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * </p>
         * 
         * @param dryRun
         *        Checks whether you have the required permissions for the action, without actually making the request,
         *        and provides an error response. If you have the required permissions, the error response is
         *        <code>DryRunOperation</code>. Otherwise, it is <code>UnauthorizedOperation</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dryRun(Boolean dryRun);

        /**
         * <p>
         * The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend using
         * this parameter because it can lead to increased interruptions. If you do not specify this parameter, you will
         * pay the current Spot price.
         * </p>
         * <important>
         * <p>
         * If you specify a maximum price, your instances will be interrupted more frequently than if you do not specify
         * this parameter.
         * </p>
         * </important>
         * 
         * @param spotPrice
         *        The maximum price per unit hour that you are willing to pay for a Spot Instance. We do not recommend
         *        using this parameter because it can lead to increased interruptions. If you do not specify this
         *        parameter, you will pay the current Spot price.</p> <important>
         *        <p>
         *        If you specify a maximum price, your instances will be interrupted more frequently than if you do not
         *        specify this parameter.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder spotPrice(String spotPrice);

        /**
         * <p>
         * Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         * information, see <a
         * href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">Ensuring idempotency
         * in Amazon EC2 API requests</a> in the <i>Amazon EC2 User Guide</i>.
         * </p>
         * 
         * @param clientToken
         *        Unique, case-sensitive identifier that you provide to ensure the idempotency of the request. For more
         *        information, see <a
         *        href="https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/Run_Instance_Idempotency.html">Ensuring
         *        idempotency in Amazon EC2 API requests</a> in the <i>Amazon EC2 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientToken(String clientToken);

        /**
         * <p>
         * The maximum number of Spot Instances to launch.
         * </p>
         * <p>
         * Default: 1
         * </p>
         * 
         * @param instanceCount
         *        The maximum number of Spot Instances to launch.</p>
         *        <p>
         *        Default: 1
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceCount(Integer instanceCount);

        /**
         * <p>
         * The Spot Instance request type.
         * </p>
         * <p>
         * Default: <code>one-time</code>
         * </p>
         * 
         * @param type
         *        The Spot Instance request type.</p>
         *        <p>
         *        Default: <code>one-time</code>
         * @see SpotInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceType
         */
        Builder type(String type);

        /**
         * <p>
         * The Spot Instance request type.
         * </p>
         * <p>
         * Default: <code>one-time</code>
         * </p>
         * 
         * @param type
         *        The Spot Instance request type.</p>
         *        <p>
         *        Default: <code>one-time</code>
         * @see SpotInstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SpotInstanceType
         */
        Builder type(SpotInstanceType type);

        /**
         * <p>
         * The start date of the request. If this is a one-time request, the request becomes active at this date and
         * time and remains active until all instances launch, the request expires, or the request is canceled. If the
         * request is persistent, the request becomes active at this date and time and remains active until it expires
         * or is canceled.
         * </p>
         * <p>
         * The specified start date and time cannot be equal to the current date and time. You must specify a start date
         * and time that occurs after the current date and time.
         * </p>
         * 
         * @param validFrom
         *        The start date of the request. If this is a one-time request, the request becomes active at this date
         *        and time and remains active until all instances launch, the request expires, or the request is
         *        canceled. If the request is persistent, the request becomes active at this date and time and remains
         *        active until it expires or is canceled.</p>
         *        <p>
         *        The specified start date and time cannot be equal to the current date and time. You must specify a
         *        start date and time that occurs after the current date and time.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validFrom(Instant validFrom);

        /**
         * <p>
         * The end date of the request, in UTC format (<i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).
         * </p>
         * <ul>
         * <li>
         * <p>
         * For a persistent request, the request remains active until the <code>ValidUntil</code> date and time is
         * reached. Otherwise, the request remains active until you cancel it.
         * </p>
         * </li>
         * <li>
         * <p>
         * For a one-time request, the request remains active until all instances launch, the request is canceled, or
         * the <code>ValidUntil</code> date and time is reached. By default, the request is valid for 7 days from the
         * date the request was created.
         * </p>
         * </li>
         * </ul>
         * 
         * @param validUntil
         *        The end date of the request, in UTC format
         *        (<i>YYYY</i>-<i>MM</i>-<i>DD</i>T<i>HH</i>:<i>MM</i>:<i>SS</i>Z).</p>
         *        <ul>
         *        <li>
         *        <p>
         *        For a persistent request, the request remains active until the <code>ValidUntil</code> date and time
         *        is reached. Otherwise, the request remains active until you cancel it.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        For a one-time request, the request remains active until all instances launch, the request is
         *        canceled, or the <code>ValidUntil</code> date and time is reached. By default, the request is valid
         *        for 7 days from the date the request was created.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder validUntil(Instant validUntil);

        /**
         * <p>
         * The instance launch group. Launch groups are Spot Instances that launch together and terminate together.
         * </p>
         * <p>
         * Default: Instances are launched and terminated individually
         * </p>
         * 
         * @param launchGroup
         *        The instance launch group. Launch groups are Spot Instances that launch together and terminate
         *        together.</p>
         *        <p>
         *        Default: Instances are launched and terminated individually
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder launchGroup(String launchGroup);

        /**
         * <p>
         * The user-specified name for a logical grouping of requests.
         * </p>
         * <p>
         * When you specify an Availability Zone group in a Spot Instance request, all Spot Instances in the request are
         * launched in the same Availability Zone. Instance proximity is maintained with this parameter, but the choice
         * of Availability Zone is not. The group applies only to requests for Spot Instances of the same instance type.
         * Any additional Spot Instance requests that are specified with the same Availability Zone group name are
         * launched in that same Availability Zone, as long as at least one instance from the group is still active.
         * </p>
         * <p>
         * If there is no active instance running in the Availability Zone group that you specify for a new Spot
         * Instance request (all instances are terminated, the request is expired, or the maximum price you specified
         * falls below current Spot price), then Amazon EC2 launches the instance in any Availability Zone where the
         * constraint can be met. Consequently, the subsequent set of Spot Instances could be placed in a different zone
         * from the original request, even if you specified the same Availability Zone group.
         * </p>
         * <p>
         * Default: Instances are launched in any available Availability Zone.
         * </p>
         * 
         * @param availabilityZoneGroup
         *        The user-specified name for a logical grouping of requests.</p>
         *        <p>
         *        When you specify an Availability Zone group in a Spot Instance request, all Spot Instances in the
         *        request are launched in the same Availability Zone. Instance proximity is maintained with this
         *        parameter, but the choice of Availability Zone is not. The group applies only to requests for Spot
         *        Instances of the same instance type. Any additional Spot Instance requests that are specified with the
         *        same Availability Zone group name are launched in that same Availability Zone, as long as at least one
         *        instance from the group is still active.
         *        </p>
         *        <p>
         *        If there is no active instance running in the Availability Zone group that you specify for a new Spot
         *        Instance request (all instances are terminated, the request is expired, or the maximum price you
         *        specified falls below current Spot price), then Amazon EC2 launches the instance in any Availability
         *        Zone where the constraint can be met. Consequently, the subsequent set of Spot Instances could be
         *        placed in a different zone from the original request, even if you specified the same Availability Zone
         *        group.
         *        </p>
         *        <p>
         *        Default: Instances are launched in any available Availability Zone.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneGroup(String availabilityZoneGroup);

        /**
         * <p>
         * Deprecated.
         * </p>
         * 
         * @param blockDurationMinutes
         *        Deprecated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder blockDurationMinutes(Integer blockDurationMinutes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends Ec2Request.BuilderImpl implements Builder {
        private RequestSpotLaunchSpecification launchSpecification;

        private List<TagSpecification> tagSpecifications = DefaultSdkAutoConstructList.getInstance();

        private String instanceInterruptionBehavior;

        private Boolean dryRun;

        private String spotPrice;

        private String clientToken;

        private Integer instanceCount;

        private String type;

        private Instant validFrom;

        private Instant validUntil;

        private String launchGroup;

        private String availabilityZoneGroup;

        private Integer blockDurationMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(RequestSpotInstancesRequest model) {
            super(model);
            launchSpecification(model.launchSpecification);
            tagSpecifications(model.tagSpecifications);
            instanceInterruptionBehavior(model.instanceInterruptionBehavior);
            dryRun(model.dryRun);
            spotPrice(model.spotPrice);
            clientToken(model.clientToken);
            instanceCount(model.instanceCount);
            type(model.type);
            validFrom(model.validFrom);
            validUntil(model.validUntil);
            launchGroup(model.launchGroup);
            availabilityZoneGroup(model.availabilityZoneGroup);
            blockDurationMinutes(model.blockDurationMinutes);
        }

        public final RequestSpotLaunchSpecification.Builder getLaunchSpecification() {
            return launchSpecification != null ? launchSpecification.toBuilder() : null;
        }

        public final void setLaunchSpecification(RequestSpotLaunchSpecification.BuilderImpl launchSpecification) {
            this.launchSpecification = launchSpecification != null ? launchSpecification.build() : null;
        }

        @Override
        public final Builder launchSpecification(RequestSpotLaunchSpecification launchSpecification) {
            this.launchSpecification = launchSpecification;
            return this;
        }

        public final List<TagSpecification.Builder> getTagSpecifications() {
            List<TagSpecification.Builder> result = TagSpecificationListCopier.copyToBuilder(this.tagSpecifications);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTagSpecifications(Collection<TagSpecification.BuilderImpl> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copyFromBuilder(tagSpecifications);
        }

        @Override
        public final Builder tagSpecifications(Collection<TagSpecification> tagSpecifications) {
            this.tagSpecifications = TagSpecificationListCopier.copy(tagSpecifications);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(TagSpecification... tagSpecifications) {
            tagSpecifications(Arrays.asList(tagSpecifications));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tagSpecifications(Consumer<TagSpecification.Builder>... tagSpecifications) {
            tagSpecifications(Stream.of(tagSpecifications).map(c -> TagSpecification.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getInstanceInterruptionBehavior() {
            return instanceInterruptionBehavior;
        }

        public final void setInstanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
        }

        @Override
        public final Builder instanceInterruptionBehavior(String instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior = instanceInterruptionBehavior;
            return this;
        }

        @Override
        public final Builder instanceInterruptionBehavior(InstanceInterruptionBehavior instanceInterruptionBehavior) {
            this.instanceInterruptionBehavior(instanceInterruptionBehavior == null ? null : instanceInterruptionBehavior
                    .toString());
            return this;
        }

        public final Boolean getDryRun() {
            return dryRun;
        }

        public final void setDryRun(Boolean dryRun) {
            this.dryRun = dryRun;
        }

        @Override
        public final Builder dryRun(Boolean dryRun) {
            this.dryRun = dryRun;
            return this;
        }

        public final String getSpotPrice() {
            return spotPrice;
        }

        public final void setSpotPrice(String spotPrice) {
            this.spotPrice = spotPrice;
        }

        @Override
        public final Builder spotPrice(String spotPrice) {
            this.spotPrice = spotPrice;
            return this;
        }

        public final String getClientToken() {
            return clientToken;
        }

        public final void setClientToken(String clientToken) {
            this.clientToken = clientToken;
        }

        @Override
        public final Builder clientToken(String clientToken) {
            this.clientToken = clientToken;
            return this;
        }

        public final Integer getInstanceCount() {
            return instanceCount;
        }

        public final void setInstanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
        }

        @Override
        public final Builder instanceCount(Integer instanceCount) {
            this.instanceCount = instanceCount;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(SpotInstanceType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final Instant getValidFrom() {
            return validFrom;
        }

        public final void setValidFrom(Instant validFrom) {
            this.validFrom = validFrom;
        }

        @Override
        public final Builder validFrom(Instant validFrom) {
            this.validFrom = validFrom;
            return this;
        }

        public final Instant getValidUntil() {
            return validUntil;
        }

        public final void setValidUntil(Instant validUntil) {
            this.validUntil = validUntil;
        }

        @Override
        public final Builder validUntil(Instant validUntil) {
            this.validUntil = validUntil;
            return this;
        }

        public final String getLaunchGroup() {
            return launchGroup;
        }

        public final void setLaunchGroup(String launchGroup) {
            this.launchGroup = launchGroup;
        }

        @Override
        public final Builder launchGroup(String launchGroup) {
            this.launchGroup = launchGroup;
            return this;
        }

        public final String getAvailabilityZoneGroup() {
            return availabilityZoneGroup;
        }

        public final void setAvailabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
        }

        @Override
        public final Builder availabilityZoneGroup(String availabilityZoneGroup) {
            this.availabilityZoneGroup = availabilityZoneGroup;
            return this;
        }

        public final Integer getBlockDurationMinutes() {
            return blockDurationMinutes;
        }

        public final void setBlockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
        }

        @Override
        public final Builder blockDurationMinutes(Integer blockDurationMinutes) {
            this.blockDurationMinutes = blockDurationMinutes;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public RequestSpotInstancesRequest build() {
            return new RequestSpotInstancesRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
