/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * In IPAM, a scope is the highest-level container within IPAM. An IPAM contains two default scopes. Each scope
 * represents the IP space for a single network. The private scope is intended for all private IP address space. The
 * public scope is intended for all public IP address space. Scopes enable you to reuse IP addresses across multiple
 * unconnected networks without causing IP address overlap or conflict.
 * </p>
 * <p>
 * For more information, see <a href="https://docs.aws.amazon.com/vpc/latest/ipam/how-it-works-ipam.html">How IPAM
 * works</a> in the <i>Amazon VPC IPAM User Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamScope implements SdkPojo, Serializable, ToCopyableBuilder<IpamScope.Builder, IpamScope> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamScope::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeId")
            .getter(getter(IpamScope::ipamScopeId))
            .setter(setter(Builder::ipamScopeId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeId")
                    .unmarshallLocationName("ipamScopeId").build()).build();

    private static final SdkField<String> IPAM_SCOPE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeArn")
            .getter(getter(IpamScope::ipamScopeArn))
            .setter(setter(Builder::ipamScopeArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeArn")
                    .unmarshallLocationName("ipamScopeArn").build()).build();

    private static final SdkField<String> IPAM_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamArn")
            .getter(getter(IpamScope::ipamArn))
            .setter(setter(Builder::ipamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamArn")
                    .unmarshallLocationName("ipamArn").build()).build();

    private static final SdkField<String> IPAM_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamRegion")
            .getter(getter(IpamScope::ipamRegion))
            .setter(setter(Builder::ipamRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamRegion")
                    .unmarshallLocationName("ipamRegion").build()).build();

    private static final SdkField<String> IPAM_SCOPE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamScopeType")
            .getter(getter(IpamScope::ipamScopeTypeAsString))
            .setter(setter(Builder::ipamScopeType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamScopeType")
                    .unmarshallLocationName("ipamScopeType").build()).build();

    private static final SdkField<Boolean> IS_DEFAULT_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("IsDefault")
            .getter(getter(IpamScope::isDefault))
            .setter(setter(Builder::isDefault))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IsDefault")
                    .unmarshallLocationName("isDefault").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(IpamScope::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<Integer> POOL_COUNT_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .memberName("PoolCount")
            .getter(getter(IpamScope::poolCount))
            .setter(setter(Builder::poolCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PoolCount")
                    .unmarshallLocationName("poolCount").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamScope::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamScope::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_SCOPE_ID_FIELD, IPAM_SCOPE_ARN_FIELD, IPAM_ARN_FIELD, IPAM_REGION_FIELD, IPAM_SCOPE_TYPE_FIELD,
            IS_DEFAULT_FIELD, DESCRIPTION_FIELD, POOL_COUNT_FIELD, STATE_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("OwnerId", OWNER_ID_FIELD);
                    put("IpamScopeId", IPAM_SCOPE_ID_FIELD);
                    put("IpamScopeArn", IPAM_SCOPE_ARN_FIELD);
                    put("IpamArn", IPAM_ARN_FIELD);
                    put("IpamRegion", IPAM_REGION_FIELD);
                    put("IpamScopeType", IPAM_SCOPE_TYPE_FIELD);
                    put("IsDefault", IS_DEFAULT_FIELD);
                    put("Description", DESCRIPTION_FIELD);
                    put("PoolCount", POOL_COUNT_FIELD);
                    put("State", STATE_FIELD);
                    put("TagSet", TAGS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamScopeId;

    private final String ipamScopeArn;

    private final String ipamArn;

    private final String ipamRegion;

    private final String ipamScopeType;

    private final Boolean isDefault;

    private final String description;

    private final Integer poolCount;

    private final String state;

    private final List<Tag> tags;

    private IpamScope(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamScopeId = builder.ipamScopeId;
        this.ipamScopeArn = builder.ipamScopeArn;
        this.ipamArn = builder.ipamArn;
        this.ipamRegion = builder.ipamRegion;
        this.ipamScopeType = builder.ipamScopeType;
        this.isDefault = builder.isDefault;
        this.description = builder.description;
        this.poolCount = builder.poolCount;
        this.state = builder.state;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The Amazon Web Services account ID of the owner of the scope.
     * </p>
     * 
     * @return The Amazon Web Services account ID of the owner of the scope.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the scope.
     * </p>
     * 
     * @return The ID of the scope.
     */
    public final String ipamScopeId() {
        return ipamScopeId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the scope.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the scope.
     */
    public final String ipamScopeArn() {
        return ipamScopeArn;
    }

    /**
     * <p>
     * The ARN of the IPAM.
     * </p>
     * 
     * @return The ARN of the IPAM.
     */
    public final String ipamArn() {
        return ipamArn;
    }

    /**
     * <p>
     * The Amazon Web Services Region of the IPAM scope.
     * </p>
     * 
     * @return The Amazon Web Services Region of the IPAM scope.
     */
    public final String ipamRegion() {
        return ipamRegion;
    }

    /**
     * <p>
     * The type of the scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipamScopeType}
     * will return {@link IpamScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipamScopeTypeAsString}.
     * </p>
     * 
     * @return The type of the scope.
     * @see IpamScopeType
     */
    public final IpamScopeType ipamScopeType() {
        return IpamScopeType.fromValue(ipamScopeType);
    }

    /**
     * <p>
     * The type of the scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipamScopeType}
     * will return {@link IpamScopeType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipamScopeTypeAsString}.
     * </p>
     * 
     * @return The type of the scope.
     * @see IpamScopeType
     */
    public final String ipamScopeTypeAsString() {
        return ipamScopeType;
    }

    /**
     * <p>
     * Defines if the scope is the default scope or not.
     * </p>
     * 
     * @return Defines if the scope is the default scope or not.
     */
    public final Boolean isDefault() {
        return isDefault;
    }

    /**
     * <p>
     * The description of the scope.
     * </p>
     * 
     * @return The description of the scope.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The number of pools in the scope.
     * </p>
     * 
     * @return The number of pools in the scope.
     */
    public final Integer poolCount() {
        return poolCount;
    }

    /**
     * <p>
     * The state of the IPAM scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamScopeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM scope.
     * @see IpamScopeState
     */
    public final IpamScopeState state() {
        return IpamScopeState.fromValue(state);
    }

    /**
     * <p>
     * The state of the IPAM scope.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamScopeState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The state of the IPAM scope.
     * @see IpamScopeState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag value
     * as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code> and the
     * value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code> for the
     * filter value.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the
     *         tag value as the filter value. For example, to find all resources that have a tag with the key
     *         <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name
     *         and <code>TeamA</code> for the filter value.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamRegion());
        hashCode = 31 * hashCode + Objects.hashCode(ipamScopeTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(isDefault());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(poolCount());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamScope)) {
            return false;
        }
        IpamScope other = (IpamScope) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(ipamScopeId(), other.ipamScopeId())
                && Objects.equals(ipamScopeArn(), other.ipamScopeArn()) && Objects.equals(ipamArn(), other.ipamArn())
                && Objects.equals(ipamRegion(), other.ipamRegion())
                && Objects.equals(ipamScopeTypeAsString(), other.ipamScopeTypeAsString())
                && Objects.equals(isDefault(), other.isDefault()) && Objects.equals(description(), other.description())
                && Objects.equals(poolCount(), other.poolCount()) && Objects.equals(stateAsString(), other.stateAsString())
                && hasTags() == other.hasTags() && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamScope").add("OwnerId", ownerId()).add("IpamScopeId", ipamScopeId())
                .add("IpamScopeArn", ipamScopeArn()).add("IpamArn", ipamArn()).add("IpamRegion", ipamRegion())
                .add("IpamScopeType", ipamScopeTypeAsString()).add("IsDefault", isDefault()).add("Description", description())
                .add("PoolCount", poolCount()).add("State", stateAsString()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamScopeId":
            return Optional.ofNullable(clazz.cast(ipamScopeId()));
        case "IpamScopeArn":
            return Optional.ofNullable(clazz.cast(ipamScopeArn()));
        case "IpamArn":
            return Optional.ofNullable(clazz.cast(ipamArn()));
        case "IpamRegion":
            return Optional.ofNullable(clazz.cast(ipamRegion()));
        case "IpamScopeType":
            return Optional.ofNullable(clazz.cast(ipamScopeTypeAsString()));
        case "IsDefault":
            return Optional.ofNullable(clazz.cast(isDefault()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "PoolCount":
            return Optional.ofNullable(clazz.cast(poolCount()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<IpamScope, T> g) {
        return obj -> g.apply((IpamScope) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamScope> {
        /**
         * <p>
         * The Amazon Web Services account ID of the owner of the scope.
         * </p>
         * 
         * @param ownerId
         *        The Amazon Web Services account ID of the owner of the scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the scope.
         * </p>
         * 
         * @param ipamScopeId
         *        The ID of the scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeId(String ipamScopeId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the scope.
         * </p>
         * 
         * @param ipamScopeArn
         *        The Amazon Resource Name (ARN) of the scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamScopeArn(String ipamScopeArn);

        /**
         * <p>
         * The ARN of the IPAM.
         * </p>
         * 
         * @param ipamArn
         *        The ARN of the IPAM.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamArn(String ipamArn);

        /**
         * <p>
         * The Amazon Web Services Region of the IPAM scope.
         * </p>
         * 
         * @param ipamRegion
         *        The Amazon Web Services Region of the IPAM scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamRegion(String ipamRegion);

        /**
         * <p>
         * The type of the scope.
         * </p>
         * 
         * @param ipamScopeType
         *        The type of the scope.
         * @see IpamScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeType
         */
        Builder ipamScopeType(String ipamScopeType);

        /**
         * <p>
         * The type of the scope.
         * </p>
         * 
         * @param ipamScopeType
         *        The type of the scope.
         * @see IpamScopeType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeType
         */
        Builder ipamScopeType(IpamScopeType ipamScopeType);

        /**
         * <p>
         * Defines if the scope is the default scope or not.
         * </p>
         * 
         * @param isDefault
         *        Defines if the scope is the default scope or not.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder isDefault(Boolean isDefault);

        /**
         * <p>
         * The description of the scope.
         * </p>
         * 
         * @param description
         *        The description of the scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The number of pools in the scope.
         * </p>
         * 
         * @param poolCount
         *        The number of pools in the scope.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder poolCount(Integer poolCount);

        /**
         * <p>
         * The state of the IPAM scope.
         * </p>
         * 
         * @param state
         *        The state of the IPAM scope.
         * @see IpamScopeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeState
         */
        Builder state(String state);

        /**
         * <p>
         * The state of the IPAM scope.
         * </p>
         * 
         * @param state
         *        The state of the IPAM scope.
         * @see IpamScopeState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamScopeState
         */
        Builder state(IpamScopeState state);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * 
         * @param tags
         *        The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and
         *        the tag value as the filter value. For example, to find all resources that have a tag with the key
         *        <code>Owner</code> and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter
         *        name and <code>TeamA</code> for the filter value.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The key/value combination of a tag assigned to the resource. Use the tag key in the filter name and the tag
         * value as the filter value. For example, to find all resources that have a tag with the key <code>Owner</code>
         * and the value <code>TeamA</code>, specify <code>tag:Owner</code> for the filter name and <code>TeamA</code>
         * for the filter value.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamScopeId;

        private String ipamScopeArn;

        private String ipamArn;

        private String ipamRegion;

        private String ipamScopeType;

        private Boolean isDefault;

        private String description;

        private Integer poolCount;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(IpamScope model) {
            ownerId(model.ownerId);
            ipamScopeId(model.ipamScopeId);
            ipamScopeArn(model.ipamScopeArn);
            ipamArn(model.ipamArn);
            ipamRegion(model.ipamRegion);
            ipamScopeType(model.ipamScopeType);
            isDefault(model.isDefault);
            description(model.description);
            poolCount(model.poolCount);
            state(model.state);
            tags(model.tags);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamScopeId() {
            return ipamScopeId;
        }

        public final void setIpamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
        }

        @Override
        public final Builder ipamScopeId(String ipamScopeId) {
            this.ipamScopeId = ipamScopeId;
            return this;
        }

        public final String getIpamScopeArn() {
            return ipamScopeArn;
        }

        public final void setIpamScopeArn(String ipamScopeArn) {
            this.ipamScopeArn = ipamScopeArn;
        }

        @Override
        public final Builder ipamScopeArn(String ipamScopeArn) {
            this.ipamScopeArn = ipamScopeArn;
            return this;
        }

        public final String getIpamArn() {
            return ipamArn;
        }

        public final void setIpamArn(String ipamArn) {
            this.ipamArn = ipamArn;
        }

        @Override
        public final Builder ipamArn(String ipamArn) {
            this.ipamArn = ipamArn;
            return this;
        }

        public final String getIpamRegion() {
            return ipamRegion;
        }

        public final void setIpamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
        }

        @Override
        public final Builder ipamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
            return this;
        }

        public final String getIpamScopeType() {
            return ipamScopeType;
        }

        public final void setIpamScopeType(String ipamScopeType) {
            this.ipamScopeType = ipamScopeType;
        }

        @Override
        public final Builder ipamScopeType(String ipamScopeType) {
            this.ipamScopeType = ipamScopeType;
            return this;
        }

        @Override
        public final Builder ipamScopeType(IpamScopeType ipamScopeType) {
            this.ipamScopeType(ipamScopeType == null ? null : ipamScopeType.toString());
            return this;
        }

        public final Boolean getIsDefault() {
            return isDefault;
        }

        public final void setIsDefault(Boolean isDefault) {
            this.isDefault = isDefault;
        }

        @Override
        public final Builder isDefault(Boolean isDefault) {
            this.isDefault = isDefault;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final Integer getPoolCount() {
            return poolCount;
        }

        public final void setPoolCount(Integer poolCount) {
            this.poolCount = poolCount;
        }

        @Override
        public final Builder poolCount(Integer poolCount) {
            this.poolCount = poolCount;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamScopeState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public IpamScope build() {
            return new IpamScope(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
