/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an IPv6 address range.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Ipv6Range implements SdkPojo, Serializable, ToCopyableBuilder<Ipv6Range.Builder, Ipv6Range> {
    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(Ipv6Range::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<String> CIDR_IPV6_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CidrIpv6")
            .getter(getter(Ipv6Range::cidrIpv6))
            .setter(setter(Builder::cidrIpv6))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CidrIpv6")
                    .unmarshallLocationName("cidrIpv6").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(DESCRIPTION_FIELD,
            CIDR_IPV6_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String description;

    private final String cidrIpv6;

    private Ipv6Range(BuilderImpl builder) {
        this.description = builder.description;
        this.cidrIpv6 = builder.cidrIpv6;
    }

    /**
     * <p>
     * A description for the security group rule that references this IPv6 address range.
     * </p>
     * <p>
     * Constraints: Up to 255 characters in length. Allowed characters are a-z, A-Z, 0-9, spaces, and
     * ._-:/()#,@[]+=&amp;;{}!$*
     * </p>
     * 
     * @return A description for the security group rule that references this IPv6 address range.</p>
     *         <p>
     *         Constraints: Up to 255 characters in length. Allowed characters are a-z, A-Z, 0-9, spaces, and
     *         ._-:/()#,@[]+=&amp;;{}!$*
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The IPv6 address range. You can either specify a CIDR block or a source security group, not both. To specify a
     * single IPv6 address, use the /128 prefix length.
     * </p>
     * <note>
     * <p>
     * Amazon Web Services <a href="https://en.wikipedia.org/wiki/Canonicalization">canonicalizes</a> IPv4 and IPv6
     * CIDRs. For example, if you specify 100.68.0.18/18 for the CIDR block, Amazon Web Services canonicalizes the CIDR
     * block to 100.68.0.0/18. Any subsequent DescribeSecurityGroups and DescribeSecurityGroupRules calls will return
     * the canonicalized form of the CIDR block. Additionally, if you attempt to add another rule with the non-canonical
     * form of the CIDR (such as 100.68.0.18/18) and there is already a rule for the canonicalized form of the CIDR
     * block (such as 100.68.0.0/18), the API throws an duplicate rule error.
     * </p>
     * </note>
     * 
     * @return The IPv6 address range. You can either specify a CIDR block or a source security group, not both. To
     *         specify a single IPv6 address, use the /128 prefix length.</p> <note>
     *         <p>
     *         Amazon Web Services <a href="https://en.wikipedia.org/wiki/Canonicalization">canonicalizes</a> IPv4 and
     *         IPv6 CIDRs. For example, if you specify 100.68.0.18/18 for the CIDR block, Amazon Web Services
     *         canonicalizes the CIDR block to 100.68.0.0/18. Any subsequent DescribeSecurityGroups and
     *         DescribeSecurityGroupRules calls will return the canonicalized form of the CIDR block. Additionally, if
     *         you attempt to add another rule with the non-canonical form of the CIDR (such as 100.68.0.18/18) and
     *         there is already a rule for the canonicalized form of the CIDR block (such as 100.68.0.0/18), the API
     *         throws an duplicate rule error.
     *         </p>
     */
    public final String cidrIpv6() {
        return cidrIpv6;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(cidrIpv6());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Ipv6Range)) {
            return false;
        }
        Ipv6Range other = (Ipv6Range) obj;
        return Objects.equals(description(), other.description()) && Objects.equals(cidrIpv6(), other.cidrIpv6());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Ipv6Range").add("Description", description()).add("CidrIpv6", cidrIpv6()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "CidrIpv6":
            return Optional.ofNullable(clazz.cast(cidrIpv6()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Description", DESCRIPTION_FIELD);
        map.put("CidrIpv6", CIDR_IPV6_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Ipv6Range, T> g) {
        return obj -> g.apply((Ipv6Range) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Ipv6Range> {
        /**
         * <p>
         * A description for the security group rule that references this IPv6 address range.
         * </p>
         * <p>
         * Constraints: Up to 255 characters in length. Allowed characters are a-z, A-Z, 0-9, spaces, and
         * ._-:/()#,@[]+=&amp;;{}!$*
         * </p>
         * 
         * @param description
         *        A description for the security group rule that references this IPv6 address range.</p>
         *        <p>
         *        Constraints: Up to 255 characters in length. Allowed characters are a-z, A-Z, 0-9, spaces, and
         *        ._-:/()#,@[]+=&amp;;{}!$*
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The IPv6 address range. You can either specify a CIDR block or a source security group, not both. To specify
         * a single IPv6 address, use the /128 prefix length.
         * </p>
         * <note>
         * <p>
         * Amazon Web Services <a href="https://en.wikipedia.org/wiki/Canonicalization">canonicalizes</a> IPv4 and IPv6
         * CIDRs. For example, if you specify 100.68.0.18/18 for the CIDR block, Amazon Web Services canonicalizes the
         * CIDR block to 100.68.0.0/18. Any subsequent DescribeSecurityGroups and DescribeSecurityGroupRules calls will
         * return the canonicalized form of the CIDR block. Additionally, if you attempt to add another rule with the
         * non-canonical form of the CIDR (such as 100.68.0.18/18) and there is already a rule for the canonicalized
         * form of the CIDR block (such as 100.68.0.0/18), the API throws an duplicate rule error.
         * </p>
         * </note>
         * 
         * @param cidrIpv6
         *        The IPv6 address range. You can either specify a CIDR block or a source security group, not both. To
         *        specify a single IPv6 address, use the /128 prefix length.</p> <note>
         *        <p>
         *        Amazon Web Services <a href="https://en.wikipedia.org/wiki/Canonicalization">canonicalizes</a> IPv4
         *        and IPv6 CIDRs. For example, if you specify 100.68.0.18/18 for the CIDR block, Amazon Web Services
         *        canonicalizes the CIDR block to 100.68.0.0/18. Any subsequent DescribeSecurityGroups and
         *        DescribeSecurityGroupRules calls will return the canonicalized form of the CIDR block. Additionally,
         *        if you attempt to add another rule with the non-canonical form of the CIDR (such as 100.68.0.18/18)
         *        and there is already a rule for the canonicalized form of the CIDR block (such as 100.68.0.0/18), the
         *        API throws an duplicate rule error.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrIpv6(String cidrIpv6);
    }

    static final class BuilderImpl implements Builder {
        private String description;

        private String cidrIpv6;

        private BuilderImpl() {
        }

        private BuilderImpl(Ipv6Range model) {
            description(model.description);
            cidrIpv6(model.cidrIpv6);
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getCidrIpv6() {
            return cidrIpv6;
        }

        public final void setCidrIpv6(String cidrIpv6) {
            this.cidrIpv6 = cidrIpv6;
        }

        @Override
        public final Builder cidrIpv6(String cidrIpv6) {
            this.cidrIpv6 = cidrIpv6;
            return this;
        }

        @Override
        public Ipv6Range build() {
            return new Ipv6Range(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
