/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a route server endpoint and its properties.
 * </p>
 * <p>
 * A route server endpoint is an Amazon Web Services-managed component inside a subnet that facilitates BGP (Border
 * Gateway Protocol) connections between your route server and your BGP peers. Create two endpoints per subnet for
 * redundancy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class RouteServerEndpoint implements SdkPojo, Serializable,
        ToCopyableBuilder<RouteServerEndpoint.Builder, RouteServerEndpoint> {
    private static final SdkField<String> ROUTE_SERVER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerId")
            .getter(getter(RouteServerEndpoint::routeServerId))
            .setter(setter(Builder::routeServerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerId")
                    .unmarshallLocationName("routeServerId").build()).build();

    private static final SdkField<String> ROUTE_SERVER_ENDPOINT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("RouteServerEndpointId")
            .getter(getter(RouteServerEndpoint::routeServerEndpointId))
            .setter(setter(Builder::routeServerEndpointId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RouteServerEndpointId")
                    .unmarshallLocationName("routeServerEndpointId").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("VpcId")
            .getter(getter(RouteServerEndpoint::vpcId))
            .setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("VpcId")
                    .unmarshallLocationName("vpcId").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(RouteServerEndpoint::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> ENI_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EniId")
            .getter(getter(RouteServerEndpoint::eniId))
            .setter(setter(Builder::eniId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EniId")
                    .unmarshallLocationName("eniId").build()).build();

    private static final SdkField<String> ENI_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("EniAddress")
            .getter(getter(RouteServerEndpoint::eniAddress))
            .setter(setter(Builder::eniAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EniAddress")
                    .unmarshallLocationName("eniAddress").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(RouteServerEndpoint::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<String> FAILURE_REASON_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("FailureReason")
            .getter(getter(RouteServerEndpoint::failureReason))
            .setter(setter(Builder::failureReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("FailureReason")
                    .unmarshallLocationName("failureReason").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(RouteServerEndpoint::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ROUTE_SERVER_ID_FIELD,
            ROUTE_SERVER_ENDPOINT_ID_FIELD, VPC_ID_FIELD, SUBNET_ID_FIELD, ENI_ID_FIELD, ENI_ADDRESS_FIELD, STATE_FIELD,
            FAILURE_REASON_FIELD, TAGS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String routeServerId;

    private final String routeServerEndpointId;

    private final String vpcId;

    private final String subnetId;

    private final String eniId;

    private final String eniAddress;

    private final String state;

    private final String failureReason;

    private final List<Tag> tags;

    private RouteServerEndpoint(BuilderImpl builder) {
        this.routeServerId = builder.routeServerId;
        this.routeServerEndpointId = builder.routeServerEndpointId;
        this.vpcId = builder.vpcId;
        this.subnetId = builder.subnetId;
        this.eniId = builder.eniId;
        this.eniAddress = builder.eniAddress;
        this.state = builder.state;
        this.failureReason = builder.failureReason;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The ID of the route server associated with this endpoint.
     * </p>
     * 
     * @return The ID of the route server associated with this endpoint.
     */
    public final String routeServerId() {
        return routeServerId;
    }

    /**
     * <p>
     * The unique identifier of the route server endpoint.
     * </p>
     * 
     * @return The unique identifier of the route server endpoint.
     */
    public final String routeServerEndpointId() {
        return routeServerEndpointId;
    }

    /**
     * <p>
     * The ID of the VPC containing the endpoint.
     * </p>
     * 
     * @return The ID of the VPC containing the endpoint.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * <p>
     * The ID of the subnet to place the route server endpoint into.
     * </p>
     * 
     * @return The ID of the subnet to place the route server endpoint into.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * The ID of the Elastic network interface for the endpoint.
     * </p>
     * 
     * @return The ID of the Elastic network interface for the endpoint.
     */
    public final String eniId() {
        return eniId;
    }

    /**
     * <p>
     * The IP address of the Elastic network interface for the endpoint.
     * </p>
     * 
     * @return The IP address of the Elastic network interface for the endpoint.
     */
    public final String eniAddress() {
        return eniAddress;
    }

    /**
     * <p>
     * The current state of the route server endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerEndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server endpoint.
     * @see RouteServerEndpointState
     */
    public final RouteServerEndpointState state() {
        return RouteServerEndpointState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the route server endpoint.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link RouteServerEndpointState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the route server endpoint.
     * @see RouteServerEndpointState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * <p>
     * The reason for any failure in endpoint creation or operation.
     * </p>
     * 
     * @return The reason for any failure in endpoint creation or operation.
     */
    public final String failureReason() {
        return failureReason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any tags assigned to the route server endpoint.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return Any tags assigned to the route server endpoint.
     */
    public final List<Tag> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(routeServerId());
        hashCode = 31 * hashCode + Objects.hashCode(routeServerEndpointId());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(eniId());
        hashCode = 31 * hashCode + Objects.hashCode(eniAddress());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(failureReason());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof RouteServerEndpoint)) {
            return false;
        }
        RouteServerEndpoint other = (RouteServerEndpoint) obj;
        return Objects.equals(routeServerId(), other.routeServerId())
                && Objects.equals(routeServerEndpointId(), other.routeServerEndpointId())
                && Objects.equals(vpcId(), other.vpcId()) && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(eniId(), other.eniId()) && Objects.equals(eniAddress(), other.eniAddress())
                && Objects.equals(stateAsString(), other.stateAsString())
                && Objects.equals(failureReason(), other.failureReason()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("RouteServerEndpoint").add("RouteServerId", routeServerId())
                .add("RouteServerEndpointId", routeServerEndpointId()).add("VpcId", vpcId()).add("SubnetId", subnetId())
                .add("EniId", eniId()).add("EniAddress", eniAddress()).add("State", stateAsString())
                .add("FailureReason", failureReason()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "RouteServerId":
            return Optional.ofNullable(clazz.cast(routeServerId()));
        case "RouteServerEndpointId":
            return Optional.ofNullable(clazz.cast(routeServerEndpointId()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "EniId":
            return Optional.ofNullable(clazz.cast(eniId()));
        case "EniAddress":
            return Optional.ofNullable(clazz.cast(eniAddress()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "FailureReason":
            return Optional.ofNullable(clazz.cast(failureReason()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("RouteServerId", ROUTE_SERVER_ID_FIELD);
        map.put("RouteServerEndpointId", ROUTE_SERVER_ENDPOINT_ID_FIELD);
        map.put("VpcId", VPC_ID_FIELD);
        map.put("SubnetId", SUBNET_ID_FIELD);
        map.put("EniId", ENI_ID_FIELD);
        map.put("EniAddress", ENI_ADDRESS_FIELD);
        map.put("State", STATE_FIELD);
        map.put("FailureReason", FAILURE_REASON_FIELD);
        map.put("TagSet", TAGS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<RouteServerEndpoint, T> g) {
        return obj -> g.apply((RouteServerEndpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, RouteServerEndpoint> {
        /**
         * <p>
         * The ID of the route server associated with this endpoint.
         * </p>
         * 
         * @param routeServerId
         *        The ID of the route server associated with this endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerId(String routeServerId);

        /**
         * <p>
         * The unique identifier of the route server endpoint.
         * </p>
         * 
         * @param routeServerEndpointId
         *        The unique identifier of the route server endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder routeServerEndpointId(String routeServerEndpointId);

        /**
         * <p>
         * The ID of the VPC containing the endpoint.
         * </p>
         * 
         * @param vpcId
         *        The ID of the VPC containing the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * The ID of the subnet to place the route server endpoint into.
         * </p>
         * 
         * @param subnetId
         *        The ID of the subnet to place the route server endpoint into.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * The ID of the Elastic network interface for the endpoint.
         * </p>
         * 
         * @param eniId
         *        The ID of the Elastic network interface for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eniId(String eniId);

        /**
         * <p>
         * The IP address of the Elastic network interface for the endpoint.
         * </p>
         * 
         * @param eniAddress
         *        The IP address of the Elastic network interface for the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder eniAddress(String eniAddress);

        /**
         * <p>
         * The current state of the route server endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the route server endpoint.
         * @see RouteServerEndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerEndpointState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the route server endpoint.
         * </p>
         * 
         * @param state
         *        The current state of the route server endpoint.
         * @see RouteServerEndpointState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see RouteServerEndpointState
         */
        Builder state(RouteServerEndpointState state);

        /**
         * <p>
         * The reason for any failure in endpoint creation or operation.
         * </p>
         * 
         * @param failureReason
         *        The reason for any failure in endpoint creation or operation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder failureReason(String failureReason);

        /**
         * <p>
         * Any tags assigned to the route server endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * Any tags assigned to the route server endpoint.
         * </p>
         * 
         * @param tags
         *        Any tags assigned to the route server endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * Any tags assigned to the route server endpoint.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);
    }

    static final class BuilderImpl implements Builder {
        private String routeServerId;

        private String routeServerEndpointId;

        private String vpcId;

        private String subnetId;

        private String eniId;

        private String eniAddress;

        private String state;

        private String failureReason;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(RouteServerEndpoint model) {
            routeServerId(model.routeServerId);
            routeServerEndpointId(model.routeServerEndpointId);
            vpcId(model.vpcId);
            subnetId(model.subnetId);
            eniId(model.eniId);
            eniAddress(model.eniAddress);
            state(model.state);
            failureReason(model.failureReason);
            tags(model.tags);
        }

        public final String getRouteServerId() {
            return routeServerId;
        }

        public final void setRouteServerId(String routeServerId) {
            this.routeServerId = routeServerId;
        }

        @Override
        public final Builder routeServerId(String routeServerId) {
            this.routeServerId = routeServerId;
            return this;
        }

        public final String getRouteServerEndpointId() {
            return routeServerEndpointId;
        }

        public final void setRouteServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
        }

        @Override
        public final Builder routeServerEndpointId(String routeServerEndpointId) {
            this.routeServerEndpointId = routeServerEndpointId;
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getEniId() {
            return eniId;
        }

        public final void setEniId(String eniId) {
            this.eniId = eniId;
        }

        @Override
        public final Builder eniId(String eniId) {
            this.eniId = eniId;
            return this;
        }

        public final String getEniAddress() {
            return eniAddress;
        }

        public final void setEniAddress(String eniAddress) {
            this.eniAddress = eniAddress;
        }

        @Override
        public final Builder eniAddress(String eniAddress) {
            this.eniAddress = eniAddress;
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(RouteServerEndpointState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final String getFailureReason() {
            return failureReason;
        }

        public final void setFailureReason(String failureReason) {
            this.failureReason = failureReason;
        }

        @Override
        public final Builder failureReason(String failureReason) {
            this.failureReason = failureReason;
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public RouteServerEndpoint build() {
            return new RouteServerEndpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
